#!/bin/sh
# maemo-optify-auto-opt
#######################################################
# Copyright: (C) 2010 Nokia
# License:   DO WHAT YOU WANT TO PUBLIC LICENSE
# Authors:   Adam Endrodi, adam.endrodi@blumsoft.eu
#            Leonid Moiseichuk, leonid.moiseichuk@nokia.com
#            Gary Birkett, gary.birkett@collabora.co.uk
#            Denis Karpov, ext-denis.2.karpov@nokia.com
#            Sergey Yakubenko, ext-sergey.yakubenko@nokia.com
#######################################################

# auto-opt script
# ensure a set of folders and files are optified
# run after installation, and each boot to relocate a set of folders

OPT_ROOT="/opt"
OPT_CONF="/etc/maemo-optify-auto-opt.conf"
OPT_LINK=""
#######################################################

set +e

# save stderr descriptor
exec 3>&2

# ...but ALWAYS return with 0 (and restore saved stderr)
trap "exec 2>&3; exit 0" EXIT

# Include test_* functions
. /usr/sbin/maemo-optify-lib.sh

# optify_one: de-optify previousely optified content (file or dir)
# param: $1 path to optify
# return: 0 on success, non-0 on error
unoptify_one()
{
  ITEM_PATH=$1
  [ ! -L "$ITEM_PATH" ] && return 1
  [ ! -e "/opt$ITEM_PATH" ] && return 1

  # check if we have enough space in rootfs for un-optification
  SIZE=`/bin/busybox du -sm "$ITEM_PATH" | /bin/busybox awk '{ print $1 }'`
  dfree "/"
  ROOTFS_SPACE=$?
  if [ $SIZE -gt $ROOTFS_SPACE ]
  then 
    log "unoptify_one: cannot un-optify - not enough space in rootfs ($ROOTFS_SPACE MB, needed $SIZE MB)"
    return 1
  fi

# we are not using "set -e"
  rm -f $ITEM_PATH && mv "/opt$ITEM_PATH" "$ITEM_PATH" && sync

  return 0
}

# get_optified_path: check if entry or any of its parent dirs was already optified
# conditions: the entry is a symlink and points to equivalent path under /opt
# param: $1 entry path
# return: 0 - the entry or it's parent was optified ($OPT_LINK contains path to the 
#             optified parent dir)
#         non-0 - the entry was not optified; 
get_optified()
{
  OPT_LINK=""
  ITEM_PATH=$1

  while [ "$ITEM_PATH" != "/" ]
  do
    if [ "`/bin/busybox readlink "$ITEM_PATH"`" = "/opt$ITEM_PATH" ]
    then
      OPT_LINK="$ITEM_PATH"
      return 0
    fi
    ITEM_PATH=`/bin/busybox dirname $ITEM_PATH`
  done

  return 1
}

# param: $1 path to optify
optify_one () {
  # optify a single item (file or folder) if required
  # first sanity check the file exists and it is not a link
  SRC="$1"
  
  if [ -e "$SRC" ] && [ ! -L "$SRC" ]
  then
    DST="$OPT_ROOT$SRC"
    DSTDIR="`dirname "$OPT_ROOT$SRC"`"

    DST_LINK="`/bin/readlink -f $SRC`"
    if [ "$SRC" = "$DST_LINK" ] 
    then 
      if [ -e "$DST" ]
      then 
        # if DST exists and is not result of previous 
        # optification - clean it up
        log "optify_one: removing pre-existing $DST"
        rm -rf "$DST"
      fi
    else
      if get_optified "$SRC"
      then
	# parent of SRC was optified
        # try to undo the optification of parent, before optifying SRC
        # if not possible to un-optify => bail out 
        log "optify_one: un-optify parent dir $OPT_LINK [$DST]"
        unoptify_one "$OPT_LINK"
        [ $? -eq 1 ] && return
      fi
    fi
      
    # proceed with optification normally
    log "optifying $SRC to $DST"

    mkdir --parent "$DSTDIR" && cp -a "$SRC" "$DSTDIR"
    sync
    rm -rf "$SRC" && ln -s "$DST" "$SRC"
    sync
  fi
}

#######################################################

if [ -e /targets/links/scratchbox.config ]
then
  return 0
fi

if ! test_opt_mount
then
  log "/home/opt is not mount bound to /opt, quitting."
  return 0
fi

if ! test_home_mount
then
 log "/home is not mounted read/write, quitting."  
 return 0
fi

dfree "/home"
if [ $? -lt $HOME_SPACE_LIMIT ]
then
  log "not enough free space in /home (need at least $HOME_SPACE_LIMIT MB free), quitting."
  return 0
fi

log "*** optification session started ***"
while read LINE
do
  case "$LINE" in 
    /*)
      optify_one "$LINE"
    ;;
    -/*)
      LINE="`echo $LINE | /bin/busybox sed -e 's/^-//'`"
      unoptify_one "$LINE"
    ;;
  esac
done < $OPT_CONF

log "*** optification session completed ***"

return 0

