/*
 * Copyright © 2004 Keith Packard
 * Copyright © 2005 Eric Anholt
 * Copyright © 2006 Nokia Corporation
 *
 * Permission to use, copy, modify, distribute and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the names of the authors and/or copyright holders
 * not be used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  The authors and
 * copyright holders make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without any express
 * or implied warranty.
 *
 * THE AUTHORS AND COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: Daniel Stone <daniel.stone@nokia.com>
 * Based on ati_video.c by Eric Anholt, which was based on mach64video.c by
 * Keith Packard.
 */

#ifdef HAVE_KDRIVE_CONFIG_H
#include <kdrive-config.h>
#endif

#include "omap.h"
#include "omapfb.h"
#include <linux/fb.h>
#include <sys/ioctl.h>

#include <X11/extensions/Xv.h>
#include "fourcc.h"

#define OMAP_DEFAULT_CKEY 0x001f
#define OMAP_DEFAULT_BG 0x7f << 5

#define MAKE_ATOM(a) MakeAtom(a, sizeof(a) - 1, TRUE)

#define IMAGE_MAX_WIDTH 800
#define IMAGE_MAX_HEIGHT 480

static KdVideoEncodingRec DummyEncoding[] = {
    { 0, "XV_IMAGE", IMAGE_MAX_WIDTH, IMAGE_MAX_HEIGHT, { 1, 1 } },
};

static KdImageRec Images[] = {
    XVIMAGE_YUY2, /* OMAPFB_COLOR_YUY422 */
    XVIMAGE_UYVY, /* OMAPFB_COLOR_YUV422 */
    XVIMAGE_I420, /* OMAPFB_COLOR_YUV420 */
};

#define NUM_IMAGES (sizeof(Images) / sizeof(Images[0]))

static KdVideoFormatRec Formats[] = {
    { 16, TrueColor },
};

#define NUM_FORMATS (sizeof(Formats) / sizeof(Formats[0]))

static KdAttributeRec Attributes[] = {
    { XvSettable | XvGettable, 0, 0xffff, "XV_COLORKEY" },
    { XvSettable | XvGettable, OMAP_DOWNSCALING_NONE,
      OMAP_DOWNSCALING_BILINEAR, "XV_OMAP_DOWNSCALING" },
    { XvSettable | XvGettable, OMAP_VSYNC_NONE, OMAP_VSYNC_FORCE,
      "XV_OMAP_VSYNC" },
};

#define NUM_ATTRIBUTES (sizeof(Attributes) / sizeof(Attributes[0]))

static Atom xvColorKey, xvScalingAlgorithm, xvVSync;

static _X_INLINE int
omapPlaneIsDirty(OmapPortPriv *pPortPriv, int id, int src_w, int src_h,
                 int dst_x, int dst_y, int dst_w, int dst_h)
{
    if (pPortPriv->id != id || pPortPriv->src_w != src_w ||
        pPortPriv->src_h != src_h || pPortPriv->dst_x != dst_x ||
        pPortPriv->dst_y != dst_y || pPortPriv->dst_w != dst_w ||
        pPortPriv->dst_h != dst_h)
        return 1;
    else
        return 0;
}

static int
omapGetPortAttribute(KdScreenInfo *screen, Atom attribute, int *value,
                     pointer data)
{
    OmapPortPriv *pPortPriv = data;

    ENTER();

    if (attribute == xvColorKey) {
        *value = pPortPriv->colorKey;
        LEAVE();
        return Success;
    }
    else if (attribute == xvScalingAlgorithm) {
        *value = pPortPriv->downscaling;
        LEAVE();
        return Success;
    }
    else if (attribute == xvVSync) {
        *value = pPortPriv->vsync;
        LEAVE();
        return Success;
    }

    LEAVE();

    return BadMatch;
}

static int
omapSetPortAttribute(KdScreenInfo *screen, Atom attribute, int value,
                     pointer data)
{
    OmapPortPriv *pPortPriv = data;

    ENTER();

    if (attribute == xvColorKey) {
        if (pPortPriv->colorKey != value) {
            if (value < 0 || value > 0xffff)
                return BadValue;
            DebugF("omapSetPortAttribute: changing colour key to %d, "
                   "marking port dirty\n", value);
            pPortPriv->colorKey = value;
            REGION_EMPTY(screen->pScreen, &pPortPriv->clip);
            pPortPriv->dirty = TRUE;
        }
        LEAVE();
        return Success;
    }
    else if (attribute == xvScalingAlgorithm) {
        if (pPortPriv->downscaling != value) {
            if (value < OMAP_DOWNSCALING_NONE ||
                value > OMAP_DOWNSCALING_BILINEAR)
                return BadValue;
            pPortPriv->downscaling = value;
            pPortPriv->dirty = TRUE;
        }
        LEAVE();
        return Success;
    }
    else if (attribute == xvVSync) {
        if (value < OMAP_VSYNC_NONE || value > OMAP_VSYNC_FORCE) {
            LEAVE();
            return BadValue;
        }

        if (!pPortPriv->vsyncCapable && value) {
            ErrorF("omapSetPortAttribute: requested vsync on a non-sync "
                   "capable port\n");
            LEAVE();
            return BadValue;
        }

        pPortPriv->vsync = value;
        LEAVE();
        return Success;
    }

    LEAVE();
    return BadMatch;
}

static void
omapQueryBestSize(KdScreenInfo *screen, Bool motion, short vid_w,
                  short vid_h, short dst_w, short dst_h,
                  unsigned int *p_w, unsigned int *p_h, pointer data)
{
    if (dst_w < IMAGE_MAX_WIDTH)
        *p_w = dst_w;
    else
        *p_w = IMAGE_MAX_WIDTH;

    if (dst_h < IMAGE_MAX_HEIGHT)
        *p_h = dst_h;
    else
        *p_h = IMAGE_MAX_HEIGHT;
}

/* Taken from kxv.c, hacked to support software scaling.  Unfortunately,
 * horizontal downscaling will never be pretty, since we need to trash two
 * pixels at a time in order to keep accesses aligned. */
void
omapCopyPackedData(KdScreenInfo *screen, OmapPortPriv *pPortPriv, CARD8 *src,
    CARD8 *dst, int randr, int srcPitch, int dstPitch, const int srcW,
    const int srcH, int top, int left, int h, int w, const int dstW,
    const int dstH)
{
    int i = 0, k = 0, ih = 0, jh = 0, jhn = (srcH - dstH);
    int kh = 0, khn = (srcW - dstW), lh = 0;

    if ((randr & RR_Rotate_All) != RR_Rotate_0) {
        ErrorF("omapCopyPackedData: rotation not supported\n");
        return;
    }

    if (top || left) {
        ErrorF("omapCopyPackedData: partial updates not supported\n");
        return;
    }

    if (srcW & 1 || dstW & 1) {
        ErrorF("omapCopyPackedData: widths must be multiples of two\n");
        return;
    }

    w >>= 1;

    for (i = 0; i < h; i++, src += srcPitch, ih += (srcH - dstH)) {
        if (pPortPriv->vscale && (jh == ih || (jh < ih && jhn > ih))) {
            jh += srcH;
            jhn += srcH;
            continue;
        }

        /* memcpy the whole lot if we can: it's a lot quicker. */
        if (!pPortPriv->hscale) {
            memcpy(dst, src, srcPitch);
        }
        else {
	    CARD32 *s = (CARD32 *)src;
            CARD32 *d = (CARD32 *)dst;

            kh = 0;
            khn = 2 * srcW;
            lh = 2 * (srcW - dstW);
            /* failing this, do 32-bit copies by hand for the pixels we
             * need. */
            for (k = 0; k < w; k++, s++, lh += 2 * (srcW - dstW)) {
                if (k != (w - 1) && (kh == lh || (kh < lh && khn > lh))) {
                    kh += (2 * srcW);
                    khn += (2 * srcW);
                }
                else {
                    *d++ = *s;
                }
            }

        }
        dst += dstPitch;
    }
}

/* FIXME this function should probably not have thirty-five local
 * variables.  (count them if you don't believe me.) */
void
omapCopyPlanarData(KdScreenInfo *screen, OmapPortPriv *pPortPriv, CARD8 *src,
    CARD8 *dstb, int randr, int srcPitch, int srcPitch2, int dstPitch,
    int srcW, int srcH, int height, int top, int left, int h, int w, int id,
    int dstW, int dstH)
{
    CARD8 *src1, *src2, *src3, *dst1;
    int srcDown = srcPitch, srcDown2 = srcPitch2;
    int srcRight = 2, srcRight2 = 1, srcNext = 1;
    int i = 0, k = 0, ih = 0, jh = 0, jhn = 0;
    int kh = 0, khn = 0, lh = 0;

    /* compute source data pointers */
    src1 = src;
    src2 = src1 + height * srcPitch;
    src3 = src2 + (height >> 1) * srcPitch2;
    switch (randr & RR_Rotate_All) {
    case RR_Rotate_0:
	srcDown = srcPitch;
	srcDown2 = srcPitch2;
	srcRight = 2;
	srcRight2 = 1;
	srcNext = 1;
	break;
    case RR_Rotate_90:
	src1 = src1 + srcH - 1;
	src2 = src2 + (srcH >> 1) - 1;
	src3 = src3 + (srcH >> 1) - 1;
	srcDown = -1;
	srcDown2 = -1;
	srcRight = srcPitch * 2;
	srcRight2 = srcPitch2;
	srcNext = srcPitch;
	break;
    case RR_Rotate_180:
	src1 = src1 + srcPitch * (srcH - 1) + (srcW - 1);
	src2 = src2 + srcPitch2 * ((srcH >> 1) - 1) + ((srcW >> 1) - 1);
	src3 = src3 + srcPitch2 * ((srcH >> 1) - 1) + ((srcW >> 1) - 1);
	srcDown = -srcPitch;
	srcDown2 = -srcPitch2;
	srcRight = -2;
	srcRight2 = -1;
	srcNext = -1;
	break;
    case RR_Rotate_270:
	src1 = src1 + srcPitch * (srcW - 1);
	src2 = src2 + srcPitch2 * ((srcW >> 1) - 1);
	src3 = src3 + srcPitch2 * ((srcW >> 1) - 1);
	srcDown = 1;
	srcDown2 = 1;
	srcRight = -srcPitch * 2;
	srcRight2 = -srcPitch2;
	srcNext = -srcPitch;
	break;
    }

    /* adjust for origin */
    src1 += top * srcDown + left * srcNext;
    src2 += (top >> 1) * srcDown2 + (left >> 1) * srcRight2;
    src3 += (top >> 1) * srcDown2 + (left >> 1) * srcRight2;

    if (id == FOURCC_I420) {
	CARD8 *srct = src2;
	src2 = src3;
	src3 = srct;
    }

    dst1 = dstb;

    w >>= 1;
    for (i = 0; i < h; i++, ih += (srcH - dstH)) {
        if (pPortPriv->vscale && (jh == ih || (jh < ih && jhn > ih))) {
            jh += srcH;
            jhn += srcH;
        }
        else {
	    CARD32 *dst = (CARD32 *)dst1;
            CARD8 *s1l = src1;
            CARD8 *s1r = src1 + srcNext;
            CARD8 *s2 = src2;
            CARD8 *s3 = src3;

            kh = 0;
            khn = 2 * srcW;
            lh = srcW - dstW;

            for (k = 0; k < w; k++, lh += 2 * (srcW - dstW)) {
                if (pPortPriv->hscale && (kh == lh || (kh < lh && khn > lh))) {
                    kh += (2 * srcW);
                    khn += (2 * srcW); 
                }
                else {
                    *dst++ = *s1l | (*s1r << 16) | (*s3 << 8) | (*s2 << 24);
                }

                s1l += srcRight;
                s1r += srcRight;
                s2 += srcRight2;
                s3 += srcRight2;
            }

            dst1 += dstPitch;
        }
        src1 += srcDown;
	if (i & 1) {
	    src2 += srcDown2;
	    src3 += srcDown2;
	}
    }
}

static void
_omapStopVideo(OmapPortPriv *pPortPriv, Bool clearclip, Bool clearmem)
{
    struct omapfb_plane_info planeInfo;

    if (pPortPriv->fd) {
        if (ioctl(pPortPriv->fd, OMAPFB_QUERY_PLANE, &planeInfo) == 0) {
            planeInfo.enabled = 0; 
            (void) ioctl(pPortPriv->fd, OMAPFB_SETUP_PLANE, &planeInfo);
            DebugF("omapVideoStop: disabling plane %d\n", pPortPriv->plane);
        }
        else {
            DebugF("omapVideoStop: not disabling plane!\n");
        }
    }

    if (pPortPriv->overlay && pPortPriv->size) {
        if (clearmem)
            bzero(pPortPriv->overlay, pPortPriv->size);
        munmap(pPortPriv->overlay, pPortPriv->size);
        pPortPriv->overlay = NULL;
        pPortPriv->size = 0;
        DebugF("omapVideoStop: unmapping overlay\n");
    }
    else {
        DebugF("omapVideoStop: no region to munmap\n");
    }

    if (pPortPriv->fd) {
        close(pPortPriv->fd);
        pPortPriv->fd = 0;
    }
    else {
        DebugF("omapVideoStop: no fd\n");
    }

    if (clearclip) {
        DamageDamageRegion(pPortPriv->pDrawable, &pPortPriv->clip);
        REGION_EMPTY(pPortPriv->omaps->screen->pScreen, &pPortPriv->clip);
    }
}

static Bool
_omapStartVideo(OmapPortPriv *pPortPriv)
{
    struct fb_var_screeninfo var;
    struct fb_fix_screeninfo fix;
    struct omapfb_plane_info planeInfo;
    struct omapfb_color_key colorKey;

    ENTER();

    pPortPriv->fd = open(pPortPriv->fileName, O_RDWR);
    if (pPortPriv->fd <= 0) {
        ErrorF("omapSetupOverlay: failed to open plane %d\n",
               pPortPriv->plane);
        LEAVE();
        return FALSE;
    }

        if (ioctl(pPortPriv->fd, FBIOGET_VSCREENINFO, &var) != 0) {
            ErrorF("omapSetupOverlay: FBIOGET_VSCREENINFO failed\n");
            LEAVE();
            return FALSE;
        }

        /* hilariously, this will actually trigger upscaling for odd widths, since
           we'll output n-1 pixels, and have the engine scale to n. */
        if (pPortPriv->hscale)
            var.xres = pPortPriv->dst_w & ~1;
        else
            var.xres = pPortPriv->src_w;
        if (pPortPriv->vscale)
            var.yres = pPortPriv->dst_h;
        else
            var.yres = pPortPriv->src_h;

        var.xres_virtual = 0;
        var.yres_virtual = 0;
        var.xoffset = 0;
        var.yoffset = 0;
        var.bits_per_pixel = 0;
        var.rotate = 0;
        var.nonstd = pPortPriv->format;
        var.activate = FB_ACTIVATE_NOW;

        if (ioctl(pPortPriv->fd, FBIOPUT_VSCREENINFO, &var) != 0) {
            ErrorF("omapSetupOverlay: FBIOPUT_VSCREENINFO failed\n");
            LEAVE();
            return FALSE;
        }

        colorKey.channel_out = OMAPFB_CHANNEL_OUT_LCD;
        colorKey.trans_key = pPortPriv->colorKey;
        colorKey.background = OMAP_DEFAULT_BG;
        colorKey.key_type = OMAPFB_COLOR_KEY_GFX_DST;
        if (ioctl(pPortPriv->fd, OMAPFB_SET_COLOR_KEY, &colorKey) != 0) {
            planeInfo.enabled = 0;
            (void) ioctl(pPortPriv->fd, OMAPFB_SETUP_PLANE, &planeInfo);
            ErrorF("omapSetupOverlay: OMAPFB_SET_COLOR_KEY failed\n");
            LEAVE();
            return FALSE;
        }

    if (ioctl(pPortPriv->fd, OMAPFB_QUERY_PLANE, &planeInfo) != 0) {
        ErrorF("omapSetupOverlay: OMAPFB_QUERY_PLANE failed\n");
        LEAVE();
        return FALSE;
    }
    planeInfo.pos_x = pPortPriv->dst_x;
    planeInfo.pos_y = pPortPriv->dst_y;
    planeInfo.out_width = pPortPriv->dst_w;
    planeInfo.out_height = pPortPriv->dst_h;
    planeInfo.channel_out = OMAPFB_CHANNEL_OUT_LCD;
    planeInfo.enabled = 1;
    if (ioctl(pPortPriv->fd, OMAPFB_SETUP_PLANE, &planeInfo) != 0) {
        ErrorF("omapSetupOverlay: OMAPFB_SETUP_PLANE failed\n");
        LEAVE();
        return FALSE;
    }

    if (ioctl(pPortPriv->fd, FBIOGET_FSCREENINFO, &fix) != 0) {
        planeInfo.enabled = 0;
        (void) ioctl(pPortPriv->fd, OMAPFB_SETUP_PLANE, &planeInfo);
        ErrorF("omapSetupOverlay: FBIOGET_FSCREENINFO failed\n");
        LEAVE();
        return FALSE;
    }
    pPortPriv->overlay = mmap(NULL, fix.smem_len, PROT_READ | PROT_WRITE,
                              MAP_SHARED, pPortPriv->fd, 0L);
    if (!pPortPriv->overlay) {
        planeInfo.enabled = 0;
        (void) ioctl(pPortPriv->fd, OMAPFB_SETUP_PLANE, &planeInfo);
        ErrorF("omapSetupOverlay: mmaping overlay failed\n");
        LEAVE();
        return FALSE;
    }
    pPortPriv->size = fix.smem_len;
    pPortPriv->pitch = fix.line_length;

    pPortPriv->dirty = FALSE;
    pPortPriv->active = 1;

    LEAVE();

    return TRUE;
}

void
omapVideoStop(KdScreenInfo *screen, pointer data, Bool exit)
{
    OmapPortPriv *pPortPriv = data, *tmp;
    OmapScreenInfo *omaps = pPortPriv->omaps;
    int i;

    _omapStopVideo(pPortPriv, exit, TRUE);

    tmp = (OmapPortPriv *)
            (&omaps->pAdaptor->pPortPrivates[omaps->numOverlayPorts]);

    /* When we stop one overlay, the others die in sympathy, so we have to
     * restart them ... */
    for (i = 0; i < omaps->numOverlayPorts; i++, tmp++) {
        if (tmp->plane != pPortPriv->plane) {
            _omapStopVideo(tmp, FALSE, FALSE);
            /* ... but not too quickly, otherwise dispc crashes. */
            usleep(5000);
            if (!_omapStartVideo(tmp)) {
                DebugF("couldn't restart overlay %d\n", i);
            }
        }
    }

    pPortPriv->active = 0;

    if (exit) {
        pPortPriv->vsync = pPortPriv->vsyncCapable ? OMAP_VSYNC_TEAR :
                                                     OMAP_VSYNC_NONE;
        pPortPriv->colorKey = OMAP_DEFAULT_CKEY;
        pPortPriv->downscaling = OMAP_DOWNSCALING_POINT_SAMPLE;
    }

    DamageDamageRegion(pPortPriv->pDrawable, &pPortPriv->clip);
    REGION_EMPTY(screen->pScreen, &pPortPriv->clip);
}


static Bool
omapSetupOverlay(KdScreenInfo *screen, OmapPortPriv *pPortPriv, int id,
                 int src_w, int src_h, int dst_x, int dst_y, int dst_w,
                 int dst_h)
{
    int useExt = 0;

    ENTER();

    if (dst_w >= 640 || dst_h >= 420) {
        DebugF("omapSetupOverlay: (%d, %d) candidate for hailstorm\n", dst_w,
               dst_h);
        useExt = 1;
    }

    if (dst_w >= src_w)
        pPortPriv->hscale = FALSE;
    else
        pPortPriv->hscale = TRUE;
    if (dst_h >= src_h)
        pPortPriv->vscale = FALSE;
    else
        pPortPriv->vscale = TRUE;

    if (pPortPriv->hscale || pPortPriv->vscale) {
        if (pPortPriv->downscaling == OMAP_DOWNSCALING_NONE) {
            ErrorF("omapSetupOverlay: downscaling not allowed on plane %d\n",
                   pPortPriv->plane);
            LEAVE();
            return FALSE;
        }
        else if (pPortPriv->downscaling == OMAP_DOWNSCALING_POINT_SAMPLE) {
            DebugF("omapSetupOverlay: enabling point-sampling downscaling "
                   "(h %d, v %d)\n", pPortPriv->hscale, pPortPriv->vscale);
            pPortPriv->scalePacked = omapCopyPackedData;
            pPortPriv->scalePlanar = omapCopyPlanarData;
        }
        else if (pPortPriv->downscaling == OMAP_DOWNSCALING_BILINEAR) {
            ErrorF("omapSetupOverlay: bilinear downscaling not yet "
                   "supported\n");
            LEAVE();
            return FALSE;
        }
        else {
            ErrorF("omapSetupOverlay: unknown downscaling algorithm\n");
            LEAVE();
            return FALSE;
        }
    }

    switch (id) {
    case FOURCC_I420:
    case FOURCC_YUY2:
        pPortPriv->format = OMAPFB_COLOR_YUY422;
        pPortPriv->bpp = 2;
        break;
    case FOURCC_UYVY:
        pPortPriv->format = OMAPFB_COLOR_YUV422;
        pPortPriv->bpp = 2;
        break;
    default:
        ErrorF("omapSetupOverlay: bad FourCC %d!\n", pPortPriv->id);
        LEAVE();
        return FALSE;
    }

    pPortPriv->src_w = src_w;
    pPortPriv->src_h = src_h;
    pPortPriv->dst_w = dst_w;
    pPortPriv->dst_h = dst_h;
    pPortPriv->dst_x = dst_x;
    pPortPriv->dst_y = dst_y;
    pPortPriv->id = id;
    pPortPriv->dirty = TRUE;

    LEAVE();

    return _omapStartVideo(pPortPriv);
}

static void
omapDisplayFrame(KdScreenInfo *screen, OmapPortPriv *pPortPriv,
                 RegionPtr clipBoxes)
{
    struct omapfb_update_window updateWindow;

    /* always update the entire window, since we don't have the smarts
     * to only dirty a smaller region for partial updates. */
    updateWindow.x = 0;
    updateWindow.y = 0;
    updateWindow.width = pPortPriv->dst_w;
    updateWindow.height = pPortPriv->dst_h;
    updateWindow.format = pPortPriv->format;

    if (pPortPriv->vsync == OMAP_VSYNC_TEAR)
        updateWindow.format |= OMAPFB_FORMAT_FLAG_TEARSYNC;
    else if (pPortPriv->vsync == OMAP_VSYNC_FORCE)
        updateWindow.format |= OMAPFB_FORMAT_FLAG_FORCE_VSYNC;

#ifdef PROFILE_ME_HARDER
    pPortPriv->frames++;
#endif

    if (ioctl(pPortPriv->fd, OMAPFB_UPDATE_WINDOW, &updateWindow) != 0) {
        /* If the update mode isn't manual (e.g. disabled for blanked
         * screen), this will fail, so let's just move on with our
         * lives. */
        DebugF("omapDisplayFrame: failed to update window!\n");
    }

    if (!REGION_EQUAL(screen->pScreen, &pPortPriv->clip, clipBoxes)) {
        REGION_COPY(screen->pScreen, &pPortPriv->clip, clipBoxes);
        KXVPaintRegion(pPortPriv->pDrawable, &pPortPriv->clip,
                       pPortPriv->colorKey);
        DamageDamageRegion(pPortPriv->pDrawable, &pPortPriv->clip);
    }
}

/* FIXME do something clever with randr */
static int
omapPutImage(KdScreenInfo *screen, DrawablePtr pDrawable,
             short src_x, short src_y,
             short dst_x, short dst_y,
             short src_w, short src_h,
             short dst_w, short dst_h,
             int id,
             unsigned char *buf,
             short width,
             short height,
             Bool sync,
             RegionPtr clipBoxes,
             pointer data)
{
    OmapPortPriv *pPortPriv = (OmapPortPriv *) data;
    unsigned char *dst;
    unsigned char *src = buf;

    /* the kernel will disallow this anyway, there's no clipping. */
    if (dst_x + dst_w > screen->width || dst_y + dst_h > screen->height) {
        ErrorF("omapPutImage: specified dimensions (%d, %d) at (%d, %d) are "
               "larger than the screen (%d, %d)\n", dst_w, dst_h, dst_x,
               dst_y, screen->width, screen->height);
        return BadValue;
    }

    pPortPriv->pDrawable = pDrawable;

    if (omapPlaneIsDirty(pPortPriv, id, src_w, src_h, dst_x, dst_y, dst_w,
                         dst_h) || pPortPriv->overlay == NULL) {
        if (!omapSetupOverlay(screen, pPortPriv, id, src_w, src_h, dst_x,
                              dst_y, dst_w, dst_h)) {
            ErrorF("omapPutImage: failed to set up overlay: from (%d, %d) "
                   "to (%d, %d) at (%d, %d) on plane %d\n", src_w, src_h,
                   dst_w, dst_h, dst_x, dst_y, pPortPriv->plane);
            return BadAlloc;
        }
    }

    DebugF("omapPutImage: putting image from (%d, %d, %d, %d) to "
           "(%d, %d, %d, %d)\n", src_x, src_y, src_w, src_h, dst_x, dst_y,
           dst_w, dst_h);

    dst = pPortPriv->overlay;

    if (pPortPriv->vsync)
        ioctl(pPortPriv->fd, OMAPFB_SYNC_GFX);

    /* The hardware locks up when we try downscaling, so we do it in software
     * instead. */
    switch (id) {
    case FOURCC_UYVY:
    case FOURCC_YUY2:
        if (pPortPriv->hscale || pPortPriv->vscale)
            (*pPortPriv->scalePacked)(screen, pPortPriv, src, pPortPriv->overlay,
                                      RR_Rotate_0, width << 1, pPortPriv->pitch,
                                      src_w, src_h, src_x, src_y, height,
                                      width, dst_w, dst_h);
        else
            KdXVCopyPackedData(screen, src, pPortPriv->overlay, RR_Rotate_0,
                               width << 1, pPortPriv->pitch, src_w, src_h, src_x,
                               src_y, height, width);
        break;
    case FOURCC_I420:
        if (pPortPriv->hscale || pPortPriv->vscale)
            (*pPortPriv->scalePlanar)(screen, pPortPriv, src, pPortPriv->overlay,
                                      RR_Rotate_0, width, (width >> 1),
                                      pPortPriv->pitch, src_w, src_h, height,
                                      src_x, src_y, height, width, id, dst_w,
                                      dst_h);
        else
            KdXVCopyPlanarData(screen, src, pPortPriv->overlay, RR_Rotate_0,
                               width, (width >> 1), pPortPriv->pitch,
                               src_w, src_h, height, src_x, src_y, height,
                               width, id);
        break;
    }

    omapDisplayFrame(screen, pPortPriv, clipBoxes);

    return Success;    
}

static int
omapReputImage(KdScreenInfo *screen, DrawablePtr pDraw, short dst_x,
               short dst_y, RegionPtr clipBoxes, pointer data)
{
    OmapPortPriv *pPortPriv = (OmapPortPriv *) data;
    BoxPtr pOldExtents = REGION_EXTENTS(pScreen, &pPortPriv->clip);
    BoxPtr pNewExtents = REGION_EXTENTS(pScreen, clipBoxes);

    if (pOldExtents->x1 != pNewExtents->x1 ||
        pOldExtents->x2 != pNewExtents->x2 ||
        pOldExtents->y1 != pNewExtents->y1 ||
        pOldExtents->y2 != pNewExtents->y2)
        return BadMatch;

    if (!_omapStartVideo(pPortPriv)) {
        ErrorF("omapReputImage: failed to set up overlay: from (%d, %d) "
               "to (%d, %d) at (%d, %d) on plane %d\n", pPortPriv->src_w,
               pPortPriv->src_h, pPortPriv->dst_w, pPortPriv->dst_h,
               pPortPriv->dst_x, pPortPriv->dst_y, pPortPriv->plane);
        return BadAlloc;
    }

    omapDisplayFrame(screen, pPortPriv, clipBoxes);

    return Success;
}

static int
omapQueryImageAttributes(KdScreenInfo *screen, int id, unsigned short *w,
                         unsigned short *h, int *pitches, int *offsets)
{
    int size = 0, tmp = 0;

    if (*w > IMAGE_MAX_WIDTH)
        *w = IMAGE_MAX_WIDTH;
    if (*h > IMAGE_MAX_HEIGHT)
        *h = IMAGE_MAX_HEIGHT;

    *w = (*w + 1) & ~1;
    if (offsets)
        offsets[0] = 0;

    switch (id)
    {
    case FOURCC_I420:
        size = *w;
        if (pitches)
            pitches[0] = size;
        size *= *h;
        if (offsets)
            offsets[1] = size;
        tmp = (*w >> 1);
        if (pitches)
            pitches[1] = pitches[2] = tmp;
        tmp *= (*h >> 1);
        size += tmp;
        if (offsets)
            offsets[2] = size;
        size += tmp;
        break;
    case FOURCC_UYVY:
    case FOURCC_YUY2:
    default:
        size = *w << 1;
        if (pitches)
            pitches[0] = size;
        size *= *h;
        break;
    }

    return size;
}

static KdVideoAdaptorPtr
omapVideoOverlaySetup(ScreenPtr pScreen)
{
    KdScreenPriv(pScreen);
    omapScreenInfo(pScreenPriv);
    KdVideoAdaptorPtr adapt;
    OmapPortPriv *pPortPriv;
    int i, fd;
    unsigned long caps;

    /* FIXME uncredible hack */
    omaps->numOverlayPorts = 2;
#if 0
    for (i = 0; i < OMAP_NUM_PLANES; i++) {
        /* We only consider dispc ports here because we use ext as a kind
         * of spare, to pivot in and out of if needed. */
        if (omapc->plane[i]->type == OMAP_PLANE_DISPC_VID &&
            access(omapc->plane->filename, O_RDWR) == 0)
            omaps->numOverlayPorts++;
    }
#endif

    adapt = xcalloc(1, sizeof(KdVideoAdaptorRec) +
                       omaps->numOverlayPorts *
                        (sizeof(OmapPortPriv) + sizeof(DevUnion)));
    if (!adapt)
        return NULL;

    adapt->type = XvWindowMask | XvInputMask | XvImageMask;
    adapt->flags = (VIDEO_CLIP_TO_VIEWPORT | VIDEO_OVERLAID_IMAGES);
    adapt->name = "OMAP Video Overlay";
    adapt->nEncodings = 1;
    adapt->pEncodings = DummyEncoding;
    adapt->nFormats = NUM_FORMATS;
    adapt->pFormats = Formats;
    adapt->nPorts = omaps->numOverlayPorts;
    adapt->pPortPrivates = (DevUnion *)(&adapt[1]);
    
    pPortPriv = (OmapPortPriv *)
                  (&adapt->pPortPrivates[omaps->numOverlayPorts]);

    for (i = 0; i < omaps->numOverlayPorts; i++) {
        pPortPriv[i].colorKey = OMAP_DEFAULT_CKEY;
        pPortPriv[i].downscaling = OMAP_DOWNSCALING_POINT_SAMPLE;
        pPortPriv[i].id = i;
        if (i == 0)
            pPortPriv[i].fileName = "/dev/fb1";
        else
            pPortPriv[i].fileName = "/dev/fb2";

        REGION_INIT(pScreen, &pPortPriv[i].clip, NullBox, 0);
        adapt->pPortPrivates[i].ptr = &pPortPriv[i];

        fd = open(pPortPriv[i].fileName, O_RDWR);
        if (!fd)
            continue;

        if (ioctl(fd, OMAPFB_GET_CAPS, &caps) == 0) {
            if (caps & OMAPFB_CAPS_TEARSYNC)
                pPortPriv[i].vsyncCapable = 1;
        }
        pPortPriv[i].vsync = pPortPriv[i].vsyncCapable ? OMAP_VSYNC_TEAR :
                                                         OMAP_VSYNC_NONE;
        pPortPriv[i].omaps = omaps;

        close(fd);
    }

    adapt->nAttributes = NUM_ATTRIBUTES;
    adapt->pAttributes = Attributes;
    adapt->nImages = NUM_IMAGES;
    adapt->pImages = Images;
    adapt->PutVideo = NULL;
    adapt->PutStill = NULL;
    adapt->GetVideo = NULL;
    adapt->GetStill = NULL;
    adapt->PutImage = omapPutImage;
    adapt->ReputImage = omapReputImage;
    adapt->StopVideo = omapVideoStop;
    adapt->GetPortAttribute = omapGetPortAttribute;
    adapt->SetPortAttribute = omapSetPortAttribute;
    adapt->QueryBestSize = omapQueryBestSize;
    adapt->QueryImageAttributes = omapQueryImageAttributes;

    omaps->pAdaptor = adapt;

    xvColorKey = MAKE_ATOM("XV_COLORKEY");
    xvScalingAlgorithm = MAKE_ATOM("XV_OMAP_DOWNSCALING");
    xvVSync = MAKE_ATOM("XV_OMAP_VSYNC");

    return adapt;
}

Bool
omapVideoInit(ScreenPtr pScreen)
{
    KdScreenPriv(pScreen);
    omapScreenInfo(pScreenPriv);
    KdScreenInfo *screen = pScreenPriv->screen;
    KdVideoAdaptorPtr *adaptors, *newAdaptors = NULL;
    KdVideoAdaptorPtr newAdaptor = NULL;
    int numAdaptors = 0;

    omaps->pAdaptor = NULL;

    numAdaptors = KdXVListGenericAdaptors(screen, &adaptors);
    newAdaptor = omapVideoOverlaySetup(pScreen);

    if (!newAdaptor)
        return FALSE;

    if (!numAdaptors) {
        numAdaptors = 1;
        adaptors = &newAdaptor;
    }
    else {
        newAdaptors = xcalloc(sizeof(KdVideoAdaptorPtr *),
                              numAdaptors + 1);
        if (!newAdaptors)
            return FALSE;

        memcpy(newAdaptors, adaptors,
               numAdaptors * sizeof(KdVideoAdaptorPtr));
        newAdaptors[numAdaptors] = newAdaptor;
        adaptors = newAdaptors;
        numAdaptors++;
    }

    KdXVScreenInit(pScreen, adaptors, numAdaptors);

    if (newAdaptors)
        xfree(newAdaptors);

    return TRUE;
}

void
omapVideoFini(ScreenPtr pScreen)
{
    KdScreenPriv(pScreen);
    omapScreenInfo(pScreenPriv);
    KdVideoAdaptorPtr adapt = omaps->pAdaptor;
    OmapPortPriv *pPortPriv;
    int i;

    if (!adapt)
        return;

    for (i = 0; i < omaps->numOverlayPorts; i++) {
        pPortPriv = (OmapPortPriv *)(&adapt->pPortPrivates[i].ptr);
        REGION_UNINIT(pScreen, &pPortPriv->clip);
    }

    xfree(adapt);
    omaps->pAdaptor = NULL;
}
