/*
 * Copyright © 2004-2006 Nokia Corporation
 *
 * Permission to use, copy, modify, distribute and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the names of the authors and/or copyright holders
 * not be used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  The authors and
 * copyright holders make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without any express
 * or implied warranty.
 *
 * THE AUTHORS AND COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: Daniel Stone <daniel.stone@nokia.com>
 */

#ifdef HAVE_KDRIVE_CONFIG_H
#include <kdrive-config.h>
#endif

#include "omap.h"
#include "omapfb.h"
#include <linux/fb.h>
#include <sys/ioctl.h>

#include <X11/Xmd.h>

#include <linux/fb.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <errno.h>
#include <stdint.h>

#include "scrnintstr.h"
#include "pixmapstr.h"
#include "mistruct.h"
#include "damagestr.h"

#include "miscstruct.h"
#include "region.h"
#include "mi.h"

#ifdef XSP
#include "spext.h"
#include <X11/extensions/xspproto.h>
#include <X11/extensions/xspwire.h>
#endif

static void _X_INLINE omapResetDamage(OmapScreenInfo *omaps)
{
    omaps->dirtyArea.x1 = 30000;
    omaps->dirtyArea.y1 = 30000;
    omaps->dirtyArea.x2 = -30000;
    omaps->dirtyArea.y2 = -30000;
}

static int _X_INLINE omapRegionIsNull(OmapScreenInfo *omaps)
{
    return (omaps->dirtyArea.x1 == 30000 && omaps->dirtyArea.x2 == -30000 &&
            omaps->dirtyArea.y1 == 30000 && omaps->dirtyArea.y2 == -30000);
}

static int _X_INLINE omapBoxesIntersect(BoxRec box1, BoxRec box2)
{
    return !(box2.x2 <= box1.x1 || box2.x1 >= box1.x2 ||
             box2.y2 <= box1.y1 || box2.y1 >= box1.y2);
}

static void _X_INLINE omapAccumulateDamage(OmapScreenInfo *omaps,
                                           RegionPtr region)
{
    int i = 0;
    BoxPtr box = REGION_RECTS(region);

    for (i = 0; i < REGION_NUM_RECTS(region); i++) {
        if (box->x1 < omaps->dirtyArea.x1)
            omaps->dirtyArea.x1 = box->x1;
        if (box->y1 < omaps->dirtyArea.y1)
            omaps->dirtyArea.y1 = box->y1;
        if (box->x2 > omaps->dirtyArea.x2)
            omaps->dirtyArea.x2 = box->x2;
        if (box->y2 > omaps->dirtyArea.y2)
            omaps->dirtyArea.y2 = box->y2;
        box++;
    }
}
#if 0
static void _X_INLINE omapCheckDSPDamage(OmapScreenInfo *omaps)
{
#ifdef XSP
    xspScrPrivPtr pScrPriv = NULL;
    int dsp_fd = 0;
    xXSPDSPStoppedEvent xE;

    if (xspScrPrivateIndex > 0)
        pScrPriv = xspGetScrPriv(omaps->screen->pScreen);

    /* If this falls within a DSP area, stop the DSP. */
    if (pScrPriv && pScrPriv->dsp_enabled && pScrPriv->dsp_client) {
        if (omapBoxesIntersect(omaps->dirtyArea, pScrPriv->dsp_box)) {
            DebugF("omapAddDamage: damaged DSP; stopping\n");

            dsp_fd = open(OMAP_DSP_FILENAME, O_RDWR);
            if (dsp_fd < 0) {
                ErrorF("omapAddDamage: unable to open DSP\n");
            }
            else {
                if (ioctl(dsp_fd, OMAP_DSP_IOCTL_FBDIS) < 0)
                    ErrorF("omapAddDamage: unable to stop DSP\n");
                close(dsp_fd);
            }

            xE.type = XSPEventBase + X_XSPDSPStopped;
            xE.sequenceNumber = pScrPriv->dsp_client->sequence;
            if (!pScrPriv->dsp_client->clientGone)
                WriteEventsToClient(pScrPriv->dsp_client, 1, (xEvent *) &xE);

            pScrPriv->dsp_enabled = 0;
            pScrPriv->dsp_client = NULL;
            
        }
    }
#endif
}

static void omapFlushDamage(OmapScreenInfo *omaps)
{
    struct omapfb_update_window updateWindow = { 0 };

    if (omapRegionIsNull(omaps))
        return;

    omapCheckDSPDamage(omaps);

#if 0
    DebugF("omapFlushDamage: updating (%d, %d) to (%d, %d)\n",
           omaps->dirtyArea.x1, omaps->dirtyArea.y1,
           omaps->dirtyArea.x2, omaps->dirtyArea.y2);
#endif

    updateWindow.format = OMAPFB_COLOR_RGB565;
    if (omaps->pixelDoubled)
        updateWindow.format |= OMAPFB_FORMAT_FLAG_DOUBLE;
    updateWindow.x = omaps->dirtyArea.x1;
    updateWindow.y = omaps->dirtyArea.y1;
    updateWindow.width = omaps->dirtyArea.x2 - omaps->dirtyArea.x1;
    updateWindow.height = omaps->dirtyArea.y2 - omaps->dirtyArea.y1;

    if (ioctl(omaps->omapc->fbdev.fd, OMAPFB_UPDATE_WINDOW, &updateWindow) != 0) {
        DebugF("omapFlushDamage: window update failed (%d, %d) to (%d, %d)\n",
               omaps->dirtyArea.x1, omaps->dirtyArea.y1, omaps->dirtyArea.x2,
               omaps->dirtyArea.y2);
    }

    omapResetDamage(omaps);
}
#endif
#ifdef PROFILE_ME_HARDER
static void omapVideoStats(OmapScreenInfo *omaps, CARD32 time)
{
    OmapPortPriv *pPortPriv = NULL;
    int i = 0;

    pPortPriv = (OmapPortPriv *)
                  (&omaps->pAdaptor->pPortPrivates[omaps->numOverlayPorts]);
    
    for (i = 0; i < 2; i++) {
        if (pPortPriv[i].frames) {
            ErrorF("port %d (%d x %d): %d frames in %.4f sec (%.4f FPS)\n",
                   i, pPortPriv[i].dst_w, pPortPriv[i].dst_h,
                   pPortPriv[i].frames,
                   (float)(time - pPortPriv[i].frames_since) / 1000,
                   pPortPriv[i].frames / ((float)(time - pPortPriv[i].frames_since) / 1000));
            pPortPriv[i].frames = 0;
        }
        pPortPriv[i].frames_since = time;
    }

    omaps->updates = 0;
}
#endif

static CARD32 omapDamageTimer(OsTimerPtr timer, CARD32 time, pointer arg)
{
    OmapScreenInfo *omaps = arg;
    int needUpdate = 0, emptyDamage = 0;
    RegionPtr region = NULL;
#ifdef XSP
    xspScrPrivPtr pScrPriv = NULL;
#endif

#ifdef XSP
    if (xspScrPrivateIndex > 0) {
        pScrPriv = xspGetScrPriv(omaps->screen->pScreen);
        if (pScrPriv && pScrPriv->dsp_enabled)
            needUpdate = 1;
    }
#endif

    if (!omaps->pDamage) {
        omaps->timerActive = 0;
        omaps->emptyUpdates = 0;
        return 0;
    }

#ifdef PROFILE_ME_HARDER
    omaps->updates++;
    if (omaps->updates > (5000 / OMAP_UPDATE_TIME))
        omapVideoStats(omaps, time);
#endif

    region = DamageRegion(omaps->pDamage);
    if (REGION_NOTEMPTY(omaps->screen->pScreen, region)) {
        emptyDamage = 1;
        if (!needUpdate)
        {
          omapAccumulateDamage(omaps, region);
          /* DamageEmpty(omaps->pDamage); */
          needUpdate = 1;
        }
    }

    if (needUpdate) {
        struct omapfb_update_window updateWindow = { 0 };

        omaps->emptyUpdates = 0;

      /*  if (!omapRegionIsNull(omaps))
        {*/
#ifdef XSP
/*            xspScrPrivPtr pScrPriv = NULL;*/
            int dsp_fd = 0, stopped = 0, i;
            xXSPDSPStoppedEvent xE;
/*
            if (xspScrPrivateIndex > 0)
                pScrPriv = xspGetScrPriv(omaps->screen->pScreen);
*/
            /* If this falls within a DSP area, stop the DSP. */
            if (pScrPriv && pScrPriv->dsp_enabled && pScrPriv->dsp_client) {
                
                /* mjv: In a case we have DSP working, we flush each damage
                   area separately. This is because we have many situations
                   when none of the areas individually clip the DSP region, but
                   the bounding box would. We'll end up having multiple
                   ioctls, but it's unlily that we'll have many updates while DSP
                   is on anyway... */
                BoxPtr box = REGION_RECTS(region);
                DebugF("DSP enabled, flushing %d rects separately\n", REGION_NUM_RECTS(region));
            
                for (i = 0; i < REGION_NUM_RECTS(region); i++) {
                    if (!stopped && omapBoxesIntersect(*box, pScrPriv->dsp_box))
                    {
                        DebugF("damaged DSP; stopping");
                        DebugF("dirtyArea: x1=%d y1=%d x2=%d y2=%d\n", 
		                    box->x1, box->y1,
		                    box->x2, box->y2);
                        DebugF("DSP Area: x1=%d y1=%d x2=%d y2=%d\n", 
		                    pScrPriv->dsp_box.x1, pScrPriv->dsp_box.y1,
		                    pScrPriv->dsp_box.x2, pScrPriv->dsp_box.y2);

                        dsp_fd = open(OMAP_DSP_FILENAME, O_RDWR);
                        if (dsp_fd < 0) {
                            ErrorF("omapAddDamage: unable to open DSP\n");
                        }
                        else {
                            if (ioctl(dsp_fd, OMAP_DSP_IOCTL_FBDIS) < 0)
                                ErrorF("omapAddDamage: unable to stop DSP\n");
                            close(dsp_fd);
                        }

                        xE.type = XSPEventBase + X_XSPDSPStopped;
                        xE.sequenceNumber = pScrPriv->dsp_client->sequence;
                        if (!pScrPriv->dsp_client->clientGone)
                            WriteEventsToClient(pScrPriv->dsp_client, 1, (xEvent *) &xE);

                        pScrPriv->dsp_enabled = 0;
                        pScrPriv->dsp_client = NULL;
                        stopped = 1;
                    }

                    updateWindow.format = OMAPFB_COLOR_RGB565;
                    if (omaps->pixelDoubled)
                        updateWindow.format |= OMAPFB_FORMAT_FLAG_DOUBLE;
                    updateWindow.x = box->x1;
                    updateWindow.y = box->y1;
                    updateWindow.width = box->x2 - box->x1;
                    updateWindow.height = box->y2 - box->y1;

                    ioctl(omaps->omapc->fbdev.fd, OMAPFB_UPDATE_WINDOW, &updateWindow);
                    box++;
                }

                DebugF("Flush done. DSP stopped = %d\n", stopped);
            }
            else { /* No DSP, update normally using the bounding box */
                DebugF("No DSP, flushing single rectangle\n");            
                updateWindow.format = OMAPFB_COLOR_RGB565;
                if (omaps->pixelDoubled)
                    updateWindow.format |= OMAPFB_FORMAT_FLAG_DOUBLE;
                updateWindow.x = omaps->dirtyArea.x1;
                updateWindow.y = omaps->dirtyArea.y1;
                updateWindow.width = omaps->dirtyArea.x2 - omaps->dirtyArea.x1;
                updateWindow.height = omaps->dirtyArea.y2 - omaps->dirtyArea.y1;

                if (ioctl(omaps->omapc->fbdev.fd, OMAPFB_UPDATE_WINDOW, &updateWindow) != 0) {
                    DebugF("omapFlushDamage: window update failed (%d, %d) to (%d, %d)\n",
                       omaps->dirtyArea.x1, omaps->dirtyArea.y1, omaps->dirtyArea.x2,
                       omaps->dirtyArea.y2);
                }
            }
#else
#error "This is just a hack. Not implemented without XSP."
#endif
            omapResetDamage(omaps);
        /*}*/
    }
    else {
        omaps->emptyUpdates++;
    }

    if (emptyDamage) {
        DamageEmpty(omaps->pDamage);
    }

    /* Kill the timer if we've gone more than 500ms without an update. */
    if (omaps->emptyUpdates >= 500 / OMAP_UPDATE_TIME) {
        omaps->timerActive = 0;
        omaps->emptyUpdates = 0;
        return 0;
    }
    else {
        return OMAP_UPDATE_TIME;
    }
}

static int omapUpdatePixelDoubling(OmapScreenInfo *omaps, int enable)
{
    DebugF("omapUpdatePixelDoubling: pixel doubling is %d\n", enable);
    omaps->pixelDoubled = enable;
    return Success;
}

#ifdef XSP
int omapXSPEvent(int event, int screen, void *closure)
{
    OmapScreenInfo *omaps = closure;

    switch (event) {
    case XSP_EVENT_PIXEL_DOUBLE_EN:
        return omapUpdatePixelDoubling(omaps, 1);
    case XSP_EVENT_PIXEL_DOUBLE_DIS:
        return omapUpdatePixelDoubling(omaps, 0);
    default:
        return BadMatch;
    }
}
#endif

static void omapDamageReport(DamagePtr pDamage, RegionPtr pRegion,
                             void *closure)
{
    OmapScreenInfo *omaps = closure;

    if (!omaps->timerActive) {
        omaps->timerActive = 1;
        TimerSet(NULL, 0, 1, omapDamageTimer, omaps);
    }
}

Bool omapCreateDrawResources(ScreenPtr pScreen)
{
    KdScreenPriv(pScreen);
    omapScreenInfo(pScreenPriv);
    PixmapPtr pPixmap = NULL;

    ENTER();

    /* Set up our damage listener to update the window by hand. */
    omaps->timerActive = 0;
    omaps->emptyUpdates = 0;
    omaps->pDamage = DamageCreate(omapDamageReport, NULL,
                                  DamageReportNonEmpty, TRUE, pScreen, omaps);
    pPixmap = (*pScreen->GetScreenPixmap) (pScreen);
    omaps->pScreenPixmap = pPixmap;
    DamageRegister(&omaps->pScreenPixmap->drawable, omaps->pDamage);
    omapResetDamage(omaps);

    omaps->pixelDoubled = 0;
#ifdef XSP
    XSPSetEventCallback(omaps->screen->pScreen->myNum, omapXSPEvent, omaps);
#endif

    LEAVE();

    return TRUE;
}

void omapRemoveDrawResources(ScreenPtr pScreen)
{
    KdScreenPriv(pScreen);
    omapScreenInfo(pScreenPriv);

    ENTER();

    if (omaps->pDamage) {
        DamageUnregister(&omaps->pScreenPixmap->drawable, omaps->pDamage);
        omaps->pScreenPixmap = NULL;
        DamageDestroy(omaps->pDamage);
    }

#ifdef XSP
    XSPSetEventCallback(omaps->screen->pScreen->myNum, NULL, NULL);
#endif

    LEAVE();
}
