/**
 * @file osso_fm_moveto.c
 *
 * This file contains routines for copy, move and delete operations
 *
 * Copyright (C) 2005 Nokia Corporation
 */

#include <sys/statvfs.h>
#include <errno.h>
//#include <sys/vfs.h>
#include "osso_fm_generic.h"
#include "osso_fm_controller.h"
#include "osso_fm_callbacks.h"
#include "osso_fm_utils.h"
#include "osso_fm_xfer.h"
#include "hildon-shorten-names-dialog.h"

#define ASYNC_JOB_PRIORITY GNOME_VFS_PRIORITY_DEFAULT
#define GNOMEVFS_RESPONSE_TIMEOUT 5 // seconds
#define FILE_SYSTEM "file:"
extern guint timer_id;
gboolean operation_cancel = FALSE;
gint delayed_delete_timer_cnt;
GTimeVal start_time;

/*
 * gint backup_started;
 * 1 backup started (any writes should be aborted)
 * 0 no back is happening can proced
*/
gint backup_started;
static gboolean check_target_not_found(GnomeVFSXferProgressInfo * progress_info);
/**
 * Function checks whether source and destination are from same location
 * @param src_uri source uri
 * @param tgt_uri target uri
 * @return TRUE if same else FALSE
 *
 */
static gboolean is_both_from_samelocation(GnomeVFSURI *src_uri, GnomeVFSURI *tgt_uri);
typedef enum 
{
        LOCAL_FILE,
	INTERNAL_MMC,
        EXTERNAL_MMC
} Path_location;

typedef enum 
{
	CSNS_OK,
	CSNS_PATH_LONG,
	CSNS_CANCELED
}CHECK_SHORTEN_NAMES_STATUS;

static gboolean check_for_partial_file(GnomeVFSXferProgressInfo * progress_info, 
		XferCallbackData * xfer_data);
static gint phase_completed_tasks(XferCallbackData * cdata);
static gint
xfer_progress_sync_callback(GnomeVFSXferProgressInfo * info, XferCallbackData * xfer_data);

static int
handle_progress_ok(GnomeVFSXferProgressInfo * progress_info, XferCallbackData * xfer_data);

static int
handle_progress_vfs_error(GnomeVFSXferProgressInfo * progress_info, XferCallbackData * xfer_data);

static int
handle_progress_overwrite(GnomeVFSXferProgressInfo * progress_info, XferCallbackData * xfer_data);

static int
handle_progress_duplicate(GnomeVFSXferProgressInfo * progress_info, XferCallbackData * xfer_data);

static gint
xfer_progress_callback(GnomeVFSAsyncHandle * handle,
		       GnomeVFSXferProgressInfo * info, XferCallbackData * xfer_data);


static int handle_xfer_duplicate_errors(GnomeVFSXferProgressInfo * progress_info,
					XferCallbackData * cdata);
static int handle_xfer_copy_errors(GnomeVFSXferProgressInfo * progress_info,
				   XferCallbackData * cdata);
static int handle_xfer_move_errors(GnomeVFSXferProgressInfo * progress_info,
				   XferCallbackData * cdata);
static int handle_xfer_rename_errors(GnomeVFSXferProgressInfo * progress_info,
				     XferCallbackData * cdata);
static int handle_xfer_cutpaste_errors(GnomeVFSXferProgressInfo * progress_info,
				       XferCallbackData * cdata);
static int handle_xfer_delete_errors(GnomeVFSXferProgressInfo * progress_info,
				     XferCallbackData * cdata);
static int handle_xfer_newfolder_errors(GnomeVFSXferProgressInfo * progress_info,
					XferCallbackData * cdata);


GnomeVFSURI *append_to_uri_renaming_safe_folder(GnomeVFSURI * srcuri, GnomeVFSURI * tgtpathuri,
						gchar * actualname)
{
	GtkFilePath *srcpath = NULL;
	gchar *shortname = NULL;
	GnomeVFSURI *tgturi = NULL;

	srcpath = osso_fm_uri_to_path(srcuri);
	if (srcpath == NULL) {
		tgturi = gnome_vfs_uri_append_path(tgtpathuri, actualname);
		return tgturi;
	}
	if (osso_fm_fs_get_path_type(global_ofm->fs,srcpath) 
			== FM_FILE_SYSTEM_MODEL_SAFE_FOLDER_GAMES) {
		shortname = osso_fm_path_get_shortname(srcpath);
		FM_ULOG_INFO("SHORT NAME %s\n", shortname);
		if (shortname == NULL) {
			tgturi = gnome_vfs_uri_append_path(tgtpathuri, actualname);
		} else {
			tgturi = gnome_vfs_uri_append_path(tgtpathuri, shortname);
			g_free(shortname);
		}
	} else {
		tgturi = gnome_vfs_uri_append_path(tgtpathuri, actualname);
	}
	///////////TEMP//////////////
	gchar *text_tgturi = NULL;
	text_tgturi = gnome_vfs_uri_to_string(tgturi, 0);
	//FM_ULOG_INFO("SOURCE URI %s\n",srcpath);
	//FM_ULOG_INFO("TARGET URI %s\n",text_tgturi);
	g_free(text_tgturi);
	///////////TEMP//////////////

	gtk_file_path_free(srcpath);
	return tgturi;

}

/**
 * add_directory_entries_to_list:
 * A helper function for creating recursive lists
 * @param sourceuri GnomeVFSURI
 * @param targeturi GnomeVFSURI
 * @param cdata XferCallbackData
 * @return void
 **/
void
add_directory_entries_to_list(GnomeVFSURI * sourceuri,
			      GnomeVFSURI * targeturi, XferCallbackData * cdata)
{

	GnomeVFSURI *newsource = NULL;
	GnomeVFSURI *newtarget = NULL;
	gchar *targetpathstr = NULL;
	gchar *targetpathstr_unesc = NULL;
	GnomeVFSFileInfo *info = NULL;
	GnomeVFSDirectoryHandle *handle = NULL;

	DEBUG_PRINT_START;

	gnome_vfs_directory_open_from_uri(&handle, sourceuri, GNOME_VFS_FILE_INFO_DEFAULT);
	if (handle == NULL) {
		FM_ULOG_INFO("Couldn't open directory\n");
		return;
	}
	info = gnome_vfs_file_info_new();

	while (gnome_vfs_directory_read_next(handle, info) == GNOME_VFS_OK) {
		if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_TYPE) {
			if (info->name != NULL) {
				if ((info->name[0] == '.' && info->name[1] == '\0') ||
				    (info->name[0] == '.' && info->name[1] == '.'
				     && info->name[2] == '\0')) {
					gnome_vfs_file_info_clear(info);
					continue;
				}
				newsource = gnome_vfs_uri_append_path(sourceuri, info->name);
				newtarget = append_to_uri_renaming_safe_folder(newsource, targeturi,
									       info->name);
				//newtarget = gnome_vfs_uri_append_path (targeturi, 
				//  info->name);

				if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
					cdata->detlists.source_dir_list =
					    g_list_append(cdata->detlists.source_dir_list, newsource);
					cdata->detlists.target_dir_list =
					    g_list_append(cdata->detlists.target_dir_list, newtarget);
					add_directory_entries_to_list(newsource, newtarget, cdata);
				} else {
					cdata->detlists.source_list =
					    g_list_append(cdata->detlists.source_list, newsource);
					cdata->detlists.target_list =
					    g_list_append(cdata->detlists.target_list, newtarget);
					//FM_ULOG_DEBUG1("SOURCE URI1 %s\n",(gchar *)newsource);
				}
				targetpathstr =
				    gnome_vfs_uri_to_string(newtarget,
						    GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD);
				targetpathstr_unesc = gnome_vfs_unescape_string(targetpathstr,NULL);
				FM_ULOG_ERR("\n%s\n",targetpathstr_unesc);

				if (g_utf8_strlen(targetpathstr_unesc, -1) > SHORTEN_DIALOG_LENGTH) {
					cdata->detlists.long_names = g_list_append(cdata->detlists.long_names,
									  gnome_vfs_uri_dup
									  (newtarget));
				}
				g_free(targetpathstr);
				g_free(targetpathstr_unesc);
			} else {
				FM_ULOG_ERR("Error gettign info->name\n");
			}
		} else {
			FM_ULOG_ERR("Error getting info\n");
		}
		if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_SIZE) {
			cdata->source_size += info->size;
		}
		gnome_vfs_file_info_clear(info);
	}
	gnome_vfs_file_info_unref(info);
	gnome_vfs_directory_close(handle);

	DEBUG_PRINT_END;
	return;
}



/**
 * get_file_list:
 * Function to append the list to the detail list in callback data.
 *
 * @param sourceuri uri representing source selection
 * @param targeturi uri representing target path
 * @param cdata callback data to store the detail list
 *
 * @return TRUE on success
 **/
static gboolean get_file_list(GnomeVFSURI * sourceuri,
			      GnomeVFSURI * targeturi,
			      XferCallbackData * cdata)
{
	gchar *targetpathstr = NULL;
	gchar *targetpathstr_unesc = NULL;

	DEBUG_PRINT_START;

	cdata->detlists.source_list = g_list_append(cdata->detlists.source_list,
						gnome_vfs_uri_dup(sourceuri));
	cdata->detlists.target_list = g_list_append(
						cdata->detlists.target_list,
						gnome_vfs_uri_dup(targeturi));
	targetpathstr = gnome_vfs_uri_to_string(targeturi, 
				GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD);

	targetpathstr_unesc = gnome_vfs_unescape_string(targetpathstr, NULL);
	FM_ULOG_ERR("get_file_list: %s\n", targetpathstr_unesc);

	if (g_utf8_strlen(targetpathstr_unesc, -1) > SHORTEN_DIALOG_LENGTH) {
		cdata->detlists.long_names = g_list_append(
						cdata->detlists.long_names,
					  	gnome_vfs_uri_dup(targeturi));
	}
	g_free(targetpathstr);
	g_free(targetpathstr_unesc);

	DEBUG_PRINT_END;
        return TRUE;
}


/**
 * make_recursive_list:
 * The function makes the recursive list of source and target paths.
 *
 * @param source_list List of source selection
 * @param target_list List of target selection
 * @param cdata Structure to store the recursive list created
 *
 * @return void
 */
static void make_recursive_list(GList * source_list, GList * target_list,
				XferCallbackData * cdata)
{
	GList *cursel = NULL;
	GList *targetsel = NULL;
        gboolean no_error = TRUE;

	DEBUG_PRINT_START;
	cdata->detlists.source_list = NULL;
	cdata->detlists.target_list = NULL;
	cdata->detlists.source_dir_list = NULL;
	cdata->detlists.target_dir_list = NULL;

	//TODO : to be updated as per target settings
	cdata->detlists.long_names = NULL;

	for (cursel = source_list, targetsel = target_list;
	     cursel != NULL && no_error;
             cursel = g_list_next(cursel),
             targetsel = g_list_next(targetsel)) {
		no_error = get_file_list(cursel->data, targetsel->data, cdata);
	}
	DEBUG_PRINT_END;
}

/**
* 
*/
void print_glist(GList * list)
{
	GList *cursel = NULL;
	gchar *str = NULL;
	FM_ULOG_DEBUG1("\n\n LIST PRINT \n\n");
	for (cursel = list; cursel != NULL; cursel = g_list_next(cursel)) {
		str = gnome_vfs_uri_to_string(cursel->data, 0);
		FM_ULOG_DEBUG1("LIST : %s\n", str);
		g_free(str);
	}
	return;
}
/**
 * reduce_exceded_length:
 * A Helper function to auto rename long names
 * @param path original path
 * @param length max length
 * @return reduced path
 **/
gchar *reduce_exceded_length (gchar *path, gint length)
{
	gchar *name = NULL;
	gchar *dup_name = NULL;
	GnomeVFSURI *pathuri = NULL;
	gchar *filepath = NULL;
	gchar *extension = NULL;
	gchar *newpath = NULL;
	gint remlen = 0;
	remlen = strlen(path) - length - 7; //TODO 7 should be replaced with method length
	if(remlen > 0)
	{
		pathuri = gnome_vfs_uri_new(path);
		GnomeVFSURI *parent_pathuri = NULL;
		parent_pathuri = gnome_vfs_uri_get_parent(pathuri);
		filepath = gnome_vfs_uri_to_string(parent_pathuri,0);
		gnome_vfs_uri_unref(parent_pathuri);
		//filepath = gnome_vfs_uri_extract_dirname(pathuri);
		name = gnome_vfs_uri_extract_short_name(pathuri);
		dup_name = g_strdup(name);
		extension = osso_fm_path_get_extension(name);
		if(extension != NULL)
		{
			newpath = g_strdup_printf("%s/%.*s%s", filepath, 
				strlen(dup_name) - remlen,dup_name,extension);
		}
		else
		{
			newpath = g_strdup_printf("%s/%.*s", filepath, 
				strlen(dup_name) - remlen,dup_name);
		}
		return newpath;
	}
	g_free(name);//VAL
	g_free(dup_name);//VAL
	g_free(filepath);//VAL
	return g_strdup(path);
}

/**
 * replace_target_uris
 * A function to replace the target names after shorten names dialog
 * @param cdata XferCallbackData
 * @param old GtkListStore
 * @param new GtkListStore
 * @return void
 **/
void replace_target_uris(XferCallbackData * cdata, GtkListStore * old, GtkListStore * new)
{
	GtkTreeIter olditer;
	GtkTreeIter newiter;
	gchar *oldtexturi = NULL;
	gchar *newtexturi = NULL;
	gchar *newtexturi_reduced = NULL;
	gchar *texturi = NULL;
	GList *cur = NULL;
	gtk_tree_model_get_iter_first(GTK_TREE_MODEL(old), &olditer);
	gtk_tree_model_get_iter_first(GTK_TREE_MODEL(new), &newiter);
	//for(cur=cdata->detlists.target_list;cur!=NULL;cur=cur->next)
	//{
	//  texturi=gnome_vfs_uri_to_string(cur->data,0);
	//  FM_ULOG_INFO("SHORT BEFORE LIST %s\n",texturi);
	//  g_free(texturi);
	//}
	do {
		gtk_tree_model_get(GTK_TREE_MODEL(old), &olditer, 0, &oldtexturi, -1);
		gtk_tree_model_get(GTK_TREE_MODEL(new), &newiter, 0, &newtexturi, -1);
		newtexturi_reduced = reduce_exceded_length(newtexturi,SHORTEN_DIALOG_LENGTH);
		//FM_ULOG_INFO("SHORT OLD %s\n",oldtexturi);
		//FM_ULOG_INFO("SHORT NEW %s\n",newtexturi);
		for (cur = cdata->detlists.target_list; cur != NULL; cur = cur->next) {
			texturi = gnome_vfs_uri_to_string((GnomeVFSURI *) cur->data, 0);
			FM_ULOG_INFO("SHORT LIST FILES %s\n", texturi);
			if (strcmp(oldtexturi, texturi) == 0) {
				gnome_vfs_uri_unref(cur->data);
				cur->data = gnome_vfs_uri_new(newtexturi_reduced);
				g_free(texturi);
				break;
			}
			g_free(texturi);
		}
		for (cur = cdata->detlists.target_dir_list; cur != NULL; cur = cur->next) {
			texturi = gnome_vfs_uri_to_string((GnomeVFSURI *) cur->data, 0);
			FM_ULOG_INFO("SHORT LIST DIR %s\n", texturi);
			if (strcmp(oldtexturi, texturi) == 0) {
				gnome_vfs_uri_unref(cur->data);
				cur->data = gnome_vfs_uri_new(newtexturi_reduced);
				g_free(texturi);
				break;
			}
			g_free(texturi);
		}

	} while (gtk_tree_model_iter_next(GTK_TREE_MODEL(old), &olditer) &&
		 gtk_tree_model_iter_next(GTK_TREE_MODEL(new), &newiter));
	//for(cur=cdata->detlists.target_list;cur!=NULL;cur=cur->next)
	//{
	//  texturi=gnome_vfs_uri_to_string(cur->data,0);
	//  FM_ULOG_INFO("SHORT AFTER LIST %s\n",texturi);
	//  g_free(texturi);
	//}

}

/**
 * A function to check if shorten names dialog is required
 * @param mainwindow GtkWindow
 * @param cdata XferCallbackData
 * @return gboolean TRUE if required
 **/
static CHECK_SHORTEN_NAMES_STATUS 
check_shorter_names_dialog(GtkWindow * mainwindow, XferCallbackData * cdata)
{
	GList *cursel = NULL;
	GtkWidget *shortendialog = NULL;
	GtkListStore *store = NULL;
	gchar *textpath = NULL;
	GtkListStore *storeout = NULL;
	gchar *filename = NULL;
	gchar *extension = NULL;
	gint textpath_len = 0;

	GtkTreeIter iter;
	DEBUG_PRINT_START;

	store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);

	for (cursel = cdata->detlists.long_names; cursel != NULL; cursel = g_list_next(cursel)) {
		textpath = gnome_vfs_uri_extract_dirname(cursel->data);
		textpath_len = strlen(textpath);
		filename = gnome_vfs_uri_extract_short_name(cursel->data);
		extension = osso_fm_path_get_extension(filename);
		g_free(filename);
		if(extension != NULL)
		{
			textpath_len+=strlen(extension);
		}
		if (textpath[textpath_len-1]  != '/') {
			textpath_len++;
		}
		if (textpath_len > SHORTEN_DIALOG_LENGTH) {
			FM_ULOG_INFO("check_shorter_names_dialog:PATH ITSELF IS LONG\n");
			return CSNS_PATH_LONG;
		}
		gtk_list_store_append(store, &iter);
		gtk_list_store_set(store, &iter, 0, gnome_vfs_uri_to_string(cursel->data, 0), -1);
	}

	shortendialog = hildon_shorten_names_dialog_new(mainwindow);

	if (shortendialog != NULL) {
		hildon_shorten_names_dialog_set_model((HildonShortenNamesDialog *) shortendialog,
						      store, SHORTEN_DIALOG_LENGTH);
		if (cdata->detlists.long_names != NULL && cdata->detlists.long_names->next == NULL)
			gtk_window_set_title(GTK_WINDOW(shortendialog), 
					dgettext(HILDON_FM_L10N_PACKAGE,"hfil_ti_shorten_name"));
		if (gtk_dialog_run(GTK_DIALOG(shortendialog)) == GTK_RESPONSE_ACCEPT) {
			storeout = (GtkListStore *) hildon_shorten_names_dialog_get_model(
					(HildonShortenNamesDialog *) shortendialog);
			replace_target_uris(cdata, store, storeout);
		} else {
			gtk_widget_destroy(shortendialog);
			return CSNS_CANCELED;
		}
		gtk_widget_destroy(shortendialog);
	}
	DEBUG_PRINT_END;
	return CSNS_OK;
}
static gboolean is_both_from_samelocation(GnomeVFSURI *src_uri, GnomeVFSURI *tgt_uri) {
	DEBUG_PRINT_START
	Path_location src_loc=LOCAL_FILE;
	Path_location tgt_loc=LOCAL_FILE;
       	gchar *src_path=NULL;
       	gchar *tgt_path=NULL;
	gboolean path_in_internal_mmc =FALSE;
	src_path = (gchar *)gnome_vfs_uri_to_string(src_uri, 0);
	tgt_path = (gchar *)gnome_vfs_uri_to_string(tgt_uri, 0);	
	if (src_path ==NULL || tgt_path==NULL)
	{
		FM_ULOG_INFO ("TARGET AND SOURCE PATH IS NULL\n");
		return FALSE;
	}
	if(g_str_has_prefix (src_path, FILE_SYSTEM))
	{
		src_loc = LOCAL_FILE;
		if (g_str_has_prefix (tgt_path, FILE_SYSTEM))
		{

			if (is_uri_in_mmc((GnomeVFSURI *)src_uri, &path_in_internal_mmc))
			{
				if (path_in_internal_mmc) {
					src_loc = INTERNAL_MMC;
				}				
			 	else { 
					src_loc = EXTERNAL_MMC;
				}
			}
			if (is_uri_in_mmc((GnomeVFSURI *)tgt_uri, &path_in_internal_mmc)) {
				if (path_in_internal_mmc) {
					tgt_loc = INTERNAL_MMC;
				} else {
					tgt_loc = EXTERNAL_MMC; 
				}
			}
		}
		

	}
	if (src_path)
		g_free(src_path);
	if (tgt_path)
		g_free(tgt_path);
	if (src_loc == tgt_loc)
		return TRUE;
	else 	
		return FALSE;
	DEBUG_PRINT_END
}

/**
 * A fuction to check avaliable space before operation 
 * @param tgt_uri GnomeVFSURI target uri
 * @param sourcesize size of the source files
 * @return gboolean
 **/
gboolean check_free_space(GnomeVFSURI *tgt_uri, gint64 sourcesize)
{
	struct statvfs info;
	int status = -1;
	gulong avail_space = 0;
	GnomeVFSURI *parent_uri = NULL;
	gchar *unesc_path = NULL;
	gint64 reserved_space=0;
        gboolean internal=FALSE;
        gboolean retval=FALSE;
	parent_uri = gnome_vfs_uri_get_parent(tgt_uri);
	unesc_path = gnome_vfs_unescape_string(gnome_vfs_uri_get_path(parent_uri),NULL);
	retval=is_uri_in_mmc(tgt_uri,&internal);
	if (retval) {
		reserved_space = 0;
	} else {
		reserved_space = RESERVED_SPACE;
	}
	status = statvfs(unesc_path, &info);
	g_free(unesc_path);
	FM_ULOG_INFO("SRC SIZE %lu, BSIZE %lu, BAVAIL %lu\n",
			(gulong)sourcesize,(gulong)info.f_bsize,(gulong)info.f_bavail);
	gnome_vfs_uri_unref(parent_uri);
	if (status == 0)
	{
		avail_space = (info.f_bsize) * (info.f_bavail);
		if(avail_space < reserved_space) {
			avail_space = 0;
		} else {
			avail_space = avail_space - reserved_space;
		}
		FM_ULOG_INFO("\nSuccess : avail_space is %lu\n",avail_space);
		if(sourcesize > avail_space)
			return FALSE;
	} else {
		FM_ULOG_INFO("\nError getting capacity : %s\n",strerror(errno));
	}
	return TRUE;
}
/*
gboolean check_free_space(GnomeVFSURI *tgt_uri, gint64 sourcesize)
{
	GnomeVFSResult res = -1;
	GnomeVFSURI *parent_uri = NULL;
	guint64 free_space = 0;
	parent_uri = gnome_vfs_uri_get_parent(tgt_uri);
	res = free_space = gnome_vfs_get_volume_free_space(parent_uri, &free_space);
	FM_ULOG_INFO("SRC SIZE %lu, AVAIL %lu\n",sourcesize,free_space);
	gnome_vfs_uri_unref(parent_uri);
	if (res == GNOME_VFS_OK)
	{
		if(sourcesize > free_space - RESERVED_SPACE)
		{
			return FALSE;
		}
	} else {
		FM_ULOG_INFO("gnome_vfs_get_volume_free_space:error %d\n",res);
	}
	return TRUE;
}
*/
/**
 * xfer_uri:
 * This function does the specified operation with the selected source list
 * and the selected target location.
 * This uses gnome_vfs_asunc_xfer() and if exceeds 
 * a defined time shows the progress bar 
 * 
 * @param source_uri_list	Pointer to source list
 * @param target_uri_list	Pointer to target uri list
 * @param mainwindow		Pointer to main window if the File Manager.
 * @param xfer_action		Operation to be performed
 * 
 * @return void
 */
void
xfer_uri(GList * source_uri_list,
	 GList * target_uri_list,
	 gpointer mainwindow, 
	 TransferAction xfer_action, gint skippedfiles,
	 gint non_existant, gboolean non_recursive)
{
	XferCallbackData *callback_data = NULL;
	GnomeVFSResult result;
	gboolean names_shortened = FALSE;
	GList *cur_md = NULL;
	CHECK_SHORTEN_NAMES_STATUS csns_ret;

	DEBUG_PRINT_START;

	callback_data = g_new0(XferCallbackData, 1);
	if (!callback_data) {
		FM_ULOG_ERR("xfer_uri:Unable to allocate "
			    "memory for callback_data\n");
		return;
	}
	callback_data->res.final_result_text = NULL;
	callback_data->res.final_result_note_button_text = NULL;
	callback_data->item_counts.errors = 0;
	callback_data->item_counts.in_use = 0;
	callback_data->item_counts.skipped = skippedfiles;
	callback_data->item_counts.non_existant= non_existant;
	callback_data->window = GTK_WIDGET(mainwindow);
	callback_data->item_counts.existing = 0;
	callback_data->replace_query_status = -1;
	callback_data->progress_verb = NULL;
	callback_data->res.button_text1 = NULL;
	callback_data->res.button_text2 = NULL;
	callback_data->detlists.source_list = NULL;
	callback_data->detlists.target_list = NULL;
	callback_data->detlists.source_dir_list = NULL;
	callback_data->detlists.target_dir_list = NULL;
	callback_data->detlists.long_names = NULL;
	callback_data->partial_copied_file = NULL;
	callback_data->locked_files = NULL;
	callback_data->source_size = 0;
	global_ofm->xfer_callback_data = callback_data;
	callback_data->watchdog_timer = g_timer_new();
	if (xfer_action == XFER_DELETE) {
		make_recursive_list(source_uri_list, source_uri_list,
				    callback_data);
	} else {
		make_recursive_list(source_uri_list, target_uri_list,
				    callback_data);
	}

	g_get_current_time(&start_time);
	if ((xfer_action == XFER_DUPLICATE || 
	     xfer_action == XFER_COPY ||
	     xfer_action == XFER_MOVE ||
	     xfer_action == XFER_CUTPASTE)
	     && !is_uri_in_gateway(target_uri_list->data))
	{
		if (xfer_action == XFER_MOVE && 
		    is_both_from_samelocation(source_uri_list->data,
			                      target_uri_list->data)) 
		{
			FM_ULOG_INFO("Xfer move operation and "
				     "src/tgt locations also same\n");
			FM_ULOG_INFO("Free space check is not required\n");
		} else if (!check_free_space(target_uri_list->data,
					     callback_data->source_size))
		{
			fm_infonote(GTK_WINDOW(global_ofm->window),
				dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE, 
					"sfil_ni_not_enough_memory"),
				dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE, 
					"sfil_ni_not_enough_memory_ok"));
			xfer_cleanup(&callback_data);
	                osso_fm_set_operation_state(OPER_NONE);
			return;
		}
	}
	switch (xfer_action) {
	case XFER_DUPLICATE:
		if ((source_uri_list == NULL) ||
		    (target_uri_list == NULL) || (mainwindow == NULL)) 
		{
			if (callback_data != NULL) {
				xfer_cleanup(&callback_data);
			}
			break;
		}
		if (callback_data->detlists.long_names != NULL) {
			csns_ret = check_shorter_names_dialog(mainwindow,
				   	callback_data);	
			if (csns_ret == CSNS_OK) {
				names_shortened = TRUE;
			} else if (csns_ret == CSNS_PATH_LONG) {
				FM_ULOG_ERR("NAMES NOT SHORTENED/CANNOT"
					    " BE SHORTENED\n");
				fm_infonote(GTK_WINDOW(global_ofm->window),
					_("sfil_ni_target_foldername_long"),
					NULL);
				if (callback_data != NULL) {
					xfer_cleanup(&callback_data);
				}
				break;
			}
			else
			{
				if (callback_data != NULL) {
					xfer_cleanup(&callback_data);
				}
				break;
			}
		}
		//Copy operation
		callback_data->action = XFER_DUPLICATE;
		callback_data->status_window = NULL;
		callback_data->progress_verb
			= g_strdup(_("docm_nw_duplicating_file"));
		callback_data->res.button_text1
			= g_strdup(_("docm_bd_duplicating_file"));
		callback_data->res.button_text2
			= g_strdup(_("docm_bd_duplicating_folder"));

		//Initiates callback data
		xfer_init(callback_data, (GtkWindow *) mainwindow);
		gchar *uri = (gchar *)gnome_vfs_uri_to_string (
				      	target_uri_list->data,
				 	GNOME_VFS_URI_HIDE_NONE);
		hildon_file_selection_move_cursor_to_uri (global_ofm->fs, uri);
		g_free(uri);
 
		global_ofm->operation_started = TRUE;
		FM_ULOG_INFO("Duplicating\n");
		if (names_shortened) {
			FM_ULOG_INFO("Using detailed Lists\n");
			if (callback_data->detlists.target_dir_list != NULL) {
				FM_ULOG_INFO("Creating dir structure before"
					" copying name shortened files\n");
				for (cur_md = callback_data->detlists.target_dir_list; cur_md != NULL;
				     cur_md = cur_md->next) {
					gnome_vfs_make_directory_for_uri(cur_md->data, 0775);
				}
			}
			if (callback_data->detlists.source_list != NULL &&
			    callback_data->detlists.target_list != NULL) {
				g_timer_reset(callback_data->watchdog_timer);
				result = gnome_vfs_async_xfer(
					 	&callback_data->async_handle,
						callback_data->detlists.source_list,
						callback_data->detlists.target_list,
						GNOME_VFS_XFER_TARGET_DEFAULT_PERMS,
						GNOME_VFS_XFER_ERROR_MODE_QUERY,
						GNOME_VFS_XFER_OVERWRITE_MODE_QUERY,
						ASYNC_JOB_PRIORITY,
						(GnomeVFSAsyncXferProgressCallback)
						xfer_progress_callback,
						callback_data,
						(GnomeVFSXferProgressCallback)
						xfer_progress_sync_callback,
						callback_data);
			}
		} else {
			g_timer_reset(callback_data->watchdog_timer);
			result = gnome_vfs_async_xfer(
				 	&callback_data->async_handle,
					source_uri_list,
					target_uri_list,
					GNOME_VFS_XFER_TARGET_DEFAULT_PERMS |
					GNOME_VFS_XFER_RECURSIVE,
					GNOME_VFS_XFER_ERROR_MODE_QUERY,
					GNOME_VFS_XFER_OVERWRITE_MODE_QUERY,
					ASYNC_JOB_PRIORITY,
					(GnomeVFSAsyncXferProgressCallback)
					xfer_progress_callback, callback_data,
					(GnomeVFSXferProgressCallback)
					xfer_progress_sync_callback,
					callback_data);
		}
		break;
	case XFER_COPY:
		if ((source_uri_list == NULL) ||
		    (target_uri_list == NULL) || (mainwindow == NULL)) 
		{
			if (callback_data != NULL) {
				xfer_cleanup(&callback_data);
			}
			break;
		}
		if (callback_data->detlists.long_names != NULL) {
			csns_ret = check_shorter_names_dialog(mainwindow,
				   	callback_data);	
			if (csns_ret == CSNS_OK) {
				names_shortened = TRUE;
			} else if (csns_ret == CSNS_PATH_LONG) {
				FM_ULOG_ERR("NAMES NOT SHORTENED/CANNOT"
					    " BE SHORTENED\n");
				fm_infonote(GTK_WINDOW(global_ofm->window),
					_("sfil_ni_target_foldername_long"),
					NULL);
				if (callback_data != NULL) {
					xfer_cleanup(&callback_data);
				}
				break;
			}
			else
			{
				if (callback_data != NULL) {
					xfer_cleanup(&callback_data);
				}
				break;
			}
		}
		//Copy operation is selected
		callback_data->action = XFER_COPY;
		callback_data->status_window = NULL;
		callback_data->progress_verb
			= g_strdup(dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					    "ckct_nw_pasting"));
		callback_data->res.button_text1
			= g_strdup(dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					    "ckct_bd_pasting_cancel"));
		callback_data->res.button_text2
			= g_strdup(_("ckct_bd_pasting_stop"));

		//Initiates callback data
		xfer_init(callback_data, (GtkWindow *) mainwindow);
		callback_data->item_counts.existing
			= osso_fm_get_existing_file_count(target_uri_list);
		global_ofm->operation_started = TRUE;
		FM_ULOG_INFO("Copying\n");
		if (names_shortened || non_recursive) {
			FM_ULOG_INFO("Using detailed Lists\n");
			if (callback_data->detlists.target_dir_list != NULL) {
				FM_ULOG_INFO("Creating dir structure before"
					     " copying name shortened files\n");
				for (cur_md = callback_data->detlists.target_dir_list; cur_md != NULL;
				     cur_md = cur_md->next) {
					gnome_vfs_make_directory_for_uri(cur_md->data, 0775);
				}
			}
			if (callback_data->detlists.source_list != NULL &&
			    callback_data->detlists.target_list != NULL) {
				g_timer_reset(callback_data->watchdog_timer);
				result = gnome_vfs_async_xfer(
					 	&callback_data->async_handle, 
						callback_data->detlists.source_list, 
						callback_data->detlists.target_list, 
						GNOME_VFS_XFER_TARGET_DEFAULT_PERMS,
						GNOME_VFS_XFER_ERROR_MODE_QUERY,
						GNOME_VFS_XFER_OVERWRITE_MODE_QUERY,
						ASYNC_JOB_PRIORITY,
						(GnomeVFSAsyncXferProgressCallback)
						xfer_progress_callback,
						callback_data,
						(GnomeVFSXferProgressCallback)
						xfer_progress_sync_callback,
						callback_data);
			}
		} else {
			g_timer_reset(callback_data->watchdog_timer);
			result = gnome_vfs_async_xfer(
			         	&callback_data->async_handle,
					source_uri_list,
					target_uri_list,
					GNOME_VFS_XFER_TARGET_DEFAULT_PERMS
					| GNOME_VFS_XFER_RECURSIVE,
					GNOME_VFS_XFER_ERROR_MODE_QUERY,
					GNOME_VFS_XFER_OVERWRITE_MODE_QUERY,
					ASYNC_JOB_PRIORITY,
					(GnomeVFSAsyncXferProgressCallback)
					xfer_progress_callback, callback_data,
					(GnomeVFSXferProgressCallback)
					xfer_progress_sync_callback,
					callback_data);
		}
		break;
	case XFER_MOVE:
		if ((source_uri_list == NULL) ||
		    (target_uri_list == NULL) || (mainwindow == NULL)) 
		{
			if (callback_data != NULL) {
				xfer_cleanup(&callback_data);
			}
			break;
		}
		if (callback_data->detlists.long_names != NULL) {
			csns_ret = check_shorter_names_dialog(mainwindow,
				   	callback_data);	
			if (csns_ret == CSNS_OK) {
				names_shortened = TRUE;
			} else if (csns_ret == CSNS_PATH_LONG) {
				FM_ULOG_ERR("NAMES NOT SHORTENED/CANNOT "
					    "BE SHORTENED\n");
				fm_infonote(GTK_WINDOW(global_ofm->window),
					_("sfil_ni_target_foldername_long"),
					NULL);
				if (callback_data != NULL) {
					xfer_cleanup(&callback_data);
				}
				break;
			}
			else
			{
				if (callback_data != NULL) {
					xfer_cleanup(&callback_data);
				}
				break;
			}
		}
		//When a move operation is selected
		callback_data->action = XFER_MOVE;
		callback_data->status_window = NULL;
		callback_data->progress_verb
			= g_strdup(dgettext(HILDON_FM_L10N_PACKAGE,
					    "sfil_nw_moving_file"));
		callback_data->res.button_text1
			= g_strdup(dgettext(HILDON_FM_L10N_PACKAGE,
					    "sfil_bd_moving_file"));
		callback_data->res.button_text2
			= g_strdup(dgettext(HILDON_FM_L10N_PACKAGE,
					    "docm_bd_moving_files"));

		xfer_init(callback_data, (GtkWindow *) mainwindow);
		callback_data->item_counts.existing
			= osso_fm_get_existing_file_count(target_uri_list);

		global_ofm->operation_started = TRUE;
		FM_ULOG_INFO("Moving\n");
		if (names_shortened) {
			FM_ULOG_INFO("Using detailed Lists\n");
			if (callback_data->detlists.target_dir_list != NULL) {
				FM_ULOG_INFO("Creating dir structure before"
					     " copying name shortened files\n");
				for (cur_md = callback_data->detlists.target_dir_list; cur_md != NULL;
				     cur_md = cur_md->next) {
					gnome_vfs_make_directory_for_uri(cur_md->data, 0775);
				}
				if (callback_data->detlists.source_list == NULL &&
				    callback_data->detlists.target_list == NULL) {
					for (cur_md = callback_data->detlists.source_dir_list;
					     cur_md != NULL; cur_md = cur_md->next) {
						gnome_vfs_remove_directory_from_uri(cur_md->data);
					}
				}
			}
			if (callback_data->detlists.source_list != NULL &&
			    callback_data->detlists.target_list != NULL) {
				g_timer_reset(callback_data->watchdog_timer);
				result = gnome_vfs_async_xfer(
					 	&callback_data->async_handle,
						callback_data->detlists.source_list,
						callback_data->detlists.target_list,
						GNOME_VFS_XFER_TARGET_DEFAULT_PERMS |
						GNOME_VFS_XFER_REMOVESOURCE,
						GNOME_VFS_XFER_ERROR_MODE_QUERY,
						GNOME_VFS_XFER_OVERWRITE_MODE_QUERY,
						ASYNC_JOB_PRIORITY,
						(GnomeVFSAsyncXferProgressCallback)
						xfer_progress_callback,
						callback_data,
						(GnomeVFSXferProgressCallback)
						xfer_progress_sync_callback,
						callback_data);
				callback_data->delete_sources = TRUE;
			}
		} else {
			g_timer_reset(callback_data->watchdog_timer);
			result = gnome_vfs_async_xfer(
			         	&callback_data->async_handle,
					source_uri_list,
					target_uri_list,
					GNOME_VFS_XFER_RECURSIVE |
					GNOME_VFS_XFER_REMOVESOURCE |
					GNOME_VFS_XFER_TARGET_DEFAULT_PERMS,
					GNOME_VFS_XFER_ERROR_MODE_QUERY,
					GNOME_VFS_XFER_OVERWRITE_MODE_QUERY,
					ASYNC_JOB_PRIORITY,
					(GnomeVFSAsyncXferProgressCallback)
					xfer_progress_callback,
					callback_data,
					(GnomeVFSXferProgressCallback)
					xfer_progress_sync_callback,
					callback_data);
		}
		break;
	case XFER_RENAME:
		if ((source_uri_list == NULL) ||
		    (target_uri_list == NULL) || (mainwindow == NULL)) {
			if (callback_data != NULL) {
				xfer_cleanup(&callback_data);
			}
			break;
		}
		callback_data->action = XFER_RENAME;
		callback_data->status_window = NULL;
		callback_data->progress_verb = NULL;
		callback_data->res.button_text1 = NULL;
		callback_data->res.button_text2 = NULL;

		xfer_init(callback_data, (GtkWindow *) mainwindow);

		gchar *rename_uri = (gchar *)gnome_vfs_uri_to_string(
						target_uri_list->data,
						GNOME_VFS_URI_HIDE_NONE);
		hildon_file_selection_move_cursor_to_uri(global_ofm->fs,
							 rename_uri);
		g_free(rename_uri);

		FM_ULOG_INFO("Renaming\n");
		global_ofm->operation_started = TRUE;
		g_timer_reset(callback_data->watchdog_timer);
		result = gnome_vfs_async_xfer(&callback_data->async_handle,
					source_uri_list,
					target_uri_list,
					GNOME_VFS_XFER_RECURSIVE |
					GNOME_VFS_XFER_REMOVESOURCE,
					GNOME_VFS_XFER_ERROR_MODE_QUERY,
					GNOME_VFS_XFER_OVERWRITE_MODE_QUERY,
					ASYNC_JOB_PRIORITY,
					(GnomeVFSAsyncXferProgressCallback)
					xfer_progress_callback,
					callback_data,
					(GnomeVFSXferProgressCallback)
					xfer_progress_sync_callback,
					callback_data);
		break;
	case XFER_CUTPASTE:
		if ((source_uri_list == NULL) ||
		    (target_uri_list == NULL) || (mainwindow == NULL)) 
		{
			if (callback_data != NULL) {
				xfer_cleanup(&callback_data);
			}
			break;
		}
		if (callback_data->detlists.long_names != NULL) {
			csns_ret = check_shorter_names_dialog(mainwindow,
							      callback_data);	
			if (csns_ret == CSNS_OK) {
				names_shortened = TRUE;
			} else if (csns_ret == CSNS_PATH_LONG){
				FM_ULOG_ERR("NAMES NOT SHORTENED/CANNOT "
					    "BE SHORTENED\n");
				fm_infonote(GTK_WINDOW(global_ofm->window),
					_("sfil_ni_target_foldername_long"),
					NULL);
				if (callback_data != NULL) {
					xfer_cleanup(&callback_data);
				}
				break;
			}
			else
			{
				if (callback_data != NULL) {
					xfer_cleanup(&callback_data);
				}
				break;
			}
		}
		//When a move operation is selected
		callback_data->action = XFER_CUTPASTE;
		callback_data->status_window = NULL;
		callback_data->progress_verb
			= g_strdup(dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					    "ckct_nw_pasting"));
		callback_data->res.button_text1
			= g_strdup(dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					    "ckct_bd_pasting_cancel"));
		callback_data->res.button_text2
			= g_strdup(_("ckct_bd_pasting_stop"));

		xfer_init(callback_data, (GtkWindow *) mainwindow);
		callback_data->item_counts.existing
			= osso_fm_get_existing_file_count(target_uri_list);

		global_ofm->operation_started = TRUE;
		FM_ULOG_INFO("Moving cp\n");
		if (names_shortened) {
			FM_ULOG_INFO("Using detailed Lists\n");
			g_timer_reset(callback_data->watchdog_timer);
			result = gnome_vfs_async_xfer(
			         	&callback_data->async_handle,
					callback_data->detlists.source_list,
					callback_data->detlists.target_list,
					GNOME_VFS_XFER_TARGET_DEFAULT_PERMS |
					GNOME_VFS_XFER_REMOVESOURCE,
					GNOME_VFS_XFER_ERROR_MODE_QUERY,
					GNOME_VFS_XFER_OVERWRITE_MODE_QUERY,
					ASYNC_JOB_PRIORITY,
					(GnomeVFSAsyncXferProgressCallback)
					xfer_progress_callback, callback_data,
					(GnomeVFSXferProgressCallback)
					xfer_progress_sync_callback,
					callback_data);
		} else {
			g_timer_reset(callback_data->watchdog_timer);
			result = gnome_vfs_async_xfer(
				 	&callback_data->async_handle,
					source_uri_list,
					target_uri_list,
					GNOME_VFS_XFER_RECURSIVE |
					GNOME_VFS_XFER_TARGET_DEFAULT_PERMS |
					GNOME_VFS_XFER_REMOVESOURCE,
					GNOME_VFS_XFER_ERROR_MODE_QUERY,
					GNOME_VFS_XFER_OVERWRITE_MODE_QUERY,
					ASYNC_JOB_PRIORITY,
					(GnomeVFSAsyncXferProgressCallback)
					xfer_progress_callback, callback_data,
					(GnomeVFSXferProgressCallback)
					xfer_progress_sync_callback,
					callback_data);
		}
		break;
	case XFER_DELETE:
		if ((source_uri_list == NULL) || (mainwindow == NULL)) {
			if (callback_data != NULL) {
				//free (callback_data);
				xfer_cleanup(&callback_data);
			}
			break;
		}
		callback_data->action = XFER_DELETE;
		callback_data->status_window = NULL;
		callback_data->progress_verb
			= g_strdup(dgettext(HILDON_FM_L10N_PACKAGE,
					    "docm_nw_deleting_file"));
		callback_data->res.button_text1
			= g_strdup(dgettext(HILDON_FM_L10N_PACKAGE,
					    "docm_bd_deleting_file"));
		callback_data->res.button_text2
			= g_strdup(dgettext(HILDON_FM_L10N_PACKAGE,
					    "docm_bd_deleting_files"));

		xfer_init(callback_data, (GtkWindow *) mainwindow);

		if (source_uri_list == NULL) {
			FM_ULOG_ERR("NO PATH EXISTS TO DELETE\n");
			GtkWidget *error_dialog = NULL;
			error_dialog = hildon_note_new_information((GtkWindow *)
						  callback_data->window,
						  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
							  "sfil_ni_operation_failed"));
			hildon_note_set_button_text((HildonNote *) error_dialog,
						    dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
							    "sfil_ni_operation_failed_ok"));
			gtk_widget_show(error_dialog);
			gtk_dialog_run((GtkDialog *) error_dialog);
			gtk_widget_destroy(error_dialog);
			xfer_cleanup(&callback_data);
			break;
		}

		FM_ULOG_INFO("Deleting\n");
		global_ofm->operation_started = TRUE;
		//deletes the selected list recursively
		g_timer_reset(callback_data->watchdog_timer);
		result = gnome_vfs_async_xfer(
			 	&callback_data->async_handle,
				source_uri_list,
				NULL,
				GNOME_VFS_XFER_DELETE_ITEMS |
				GNOME_VFS_XFER_RECURSIVE,
				GNOME_VFS_XFER_ERROR_MODE_QUERY,
				GNOME_VFS_XFER_OVERWRITE_MODE_QUERY,
				ASYNC_JOB_PRIORITY,
				(GnomeVFSAsyncXferProgressCallback)
				xfer_progress_callback,
				callback_data,
				(GnomeVFSXferProgressCallback)
				xfer_progress_sync_callback,
				callback_data);
		break;
		//case XFER_OPENING progress needs to be added.
	case XFER_NEWFOLDER:
		if ((target_uri_list == NULL) || (mainwindow == NULL)) {
			if (callback_data != NULL) {
				xfer_cleanup(&callback_data);
			}
			xfer_cleanup(&callback_data);
		}
		callback_data->action = XFER_NEWFOLDER;
		callback_data->status_window = NULL;

		xfer_init(callback_data, (GtkWindow *) mainwindow);

		//deletes the selected list recursively

		global_ofm->operation_started = TRUE;
		g_timer_reset(callback_data->watchdog_timer);
		result = gnome_vfs_async_xfer(
			 	&callback_data->async_handle, 
				NULL, 
				target_uri_list, 
				GNOME_VFS_XFER_DELETE_ITEMS |
				GNOME_VFS_XFER_RECURSIVE, 
				GNOME_VFS_XFER_ERROR_MODE_QUERY, 
				GNOME_VFS_XFER_OVERWRITE_MODE_QUERY, 
				ASYNC_JOB_PRIORITY, 
				(GnomeVFSAsyncXferProgressCallback)
				xfer_progress_callback, 
				callback_data, 
				NULL,	// sync callback
				NULL);
		break;
	default:
		break;
	}

	DEBUG_PRINT_END;
	if (source_uri_list != NULL)
		osso_fm_free_gnomevfs_uri_list(source_uri_list);
	if (target_uri_list != NULL)
		osso_fm_free_gnomevfs_uri_list(target_uri_list);
	return;
}


/**
 * xfer_progress_sync_callback:
 * This is callback function for gnome_vfs_xfer
 * 
 * @param Info		pointer to Progress Info structure
 * @param xfer_data	pointer to callback data 
 * 
 * @return 		gint
 */

static gint
xfer_progress_sync_callback(GnomeVFSXferProgressInfo * progress_info, XferCallbackData * cdata)
{
	static gboolean completed_called = FALSE;
	g_timer_reset(cdata->watchdog_timer);
	if(global_ofm->xfer_callback_data == NULL)
	{
		FM_ULOG_ERR("sync_callback:xfer_callback_data == NULL:Must be Watch Dogs work\n");
		return 0;
	}
	if (progress_info->phase == GNOME_VFS_XFER_PHASE_INITIAL ||
	    progress_info->phase == GNOME_VFS_XFER_CHECKING_DESTINATION ||
	    progress_info->phase == GNOME_VFS_XFER_PHASE_COLLECTING) {
		completed_called = FALSE;
	}
	if (progress_info->phase == GNOME_VFS_XFER_PHASE_COMPLETED) {
		FM_ULOG_INFO("Sync GNOME_VFS_XFER_PHASE_COMPLETED\n");
		if (cdata->action == XFER_CUTPASTE ||
		    cdata->action == XFER_COPY) {
			if (cdata->item_counts.non_existant ==
			    progress_info->files_total ||
			    progress_info->files_total == 0) {
				gdk_threads_enter();
				fm_infoprint(GTK_WINDOW(global_ofm->window),
				  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					   "ecoc_ib_edwin_nothing_to_paste"));
				gdk_threads_leave();
			}
		}
		if (completed_called) {
			FM_ULOG_INFO("Sync GNOME_VFS_XFER_PHASE_COMPLETED called again\n");
			return 1;
		}
		completed_called = TRUE;
		gdk_threads_enter();
		phase_completed_tasks(cdata);
		gdk_threads_leave();
	}
	return 1;
}

/**
 * A timer function to delete the partial files after a small time
 * @param data timer callback data
 * @return gboolean
 **/
gboolean delayed_delete_timer(gpointer data)
{
	GList *dat = NULL;
	DEBUG_PRINT_START;
	if (data == NULL) {
		FM_ULOG_ERR("%s DATA IS NULL \n", __FUNCTION__);
		DEBUG_PRINT_END;
		return FALSE;
	}
	dat = (GList *) data;
	if (dat->data == NULL) {
		FM_ULOG_ERR("%s TARGET NAME NULL \n", __FUNCTION__);
		g_list_free(dat);
		DEBUG_PRINT_END;
		return FALSE;
	}
	if (dat->next->data == NULL) {
		FM_ULOG_ERR("%s SOURCE NAME NULL \n", __FUNCTION__);
		g_free(dat->data);
		g_list_free(dat);
		DEBUG_PRINT_END;
		return FALSE;
	}
	if (delayed_delete_timer_cnt < 3) {
		if (is_path_exists(dat->next->data))	//source exists
		{
			if (gnome_vfs_unlink((gchar *) dat->data) == GNOME_VFS_OK) {
				delayed_delete_timer_cnt = 0;
				g_free(dat->data);
				g_free(dat->next->data);
				g_list_free(dat);
				DEBUG_PRINT_END;
				return FALSE;
			}
		} else {
			FM_ULOG_ERR("Error:%s:source is deleted so leaving target\n", __FUNCTION__);
			delayed_delete_timer_cnt = 0;
			g_free(dat->data);
			g_free(dat->next->data);
			g_list_free(dat);
			DEBUG_PRINT_END;
			return FALSE;
		}
	} else {
		delayed_delete_timer_cnt = 0;
		g_free(dat->data);
		g_free(dat->next->data);
		g_list_free(dat);
		DEBUG_PRINT_END;
		return FALSE;
	}
	delayed_delete_timer_cnt++;
	DEBUG_PRINT_END;
	return TRUE;
}

/**
 * check_for_partial_file:
 * A function to check if a file is copied partially
 * @param progress_info GnomeVFSXferProgressInfo
 * @param xfer_data XferCallbackData
 * @return gboolean
 **/
static gboolean check_for_partial_file(GnomeVFSXferProgressInfo * progress_info, 
		XferCallbackData * xfer_data)
{
	if (progress_info->phase != GNOME_VFS_XFER_PHASE_COMPLETED) {
		if (progress_info->target_name != NULL
		    && progress_info->source_name != NULL
		    && g_strcasecmp(progress_info->source_name, progress_info->target_name) != 0
		    && progress_info->bytes_total != progress_info->bytes_copied) {
			xfer_data->partial_copied_file =
			    g_list_append(xfer_data->partial_copied_file,
					  g_strdup(progress_info->target_name));

			xfer_data->partial_copied_file =
			    g_list_append(xfer_data->partial_copied_file,
					  g_strdup(progress_info->source_name));
			return TRUE;
		}
	}
	return FALSE;
}

/**
 * xfer_progress_callback:
 * This callback is called periodically during transfer 
 * updates progress dialog, choose which action shall 
 * be taken in case of overwrites, etc.
 * 
 * @param handle	Pointer to Asynchoronous gnome_xfer_async_xfer
 * @param Info		Pointer to Progress Info structure 
 * @param xfer_data	Pointer to callback data 
 * 
 * @return 		gint
 */
static gint
xfer_progress_callback(GnomeVFSAsyncHandle * handle,
		       GnomeVFSXferProgressInfo * progress_info, XferCallbackData * xfer_data)
{
	gint retval = 0;
	//GList *delete_me=NULL;
	DEBUG_PRINT_START;
	if(global_ofm->xfer_callback_data == NULL)
	{
		FM_ULOG_ERR("xfer_progress_callback:xfer_data == NULL:Must be Watch Dogs work\n");
		return 0;
	}
			
	if (handle == NULL || progress_info == NULL || xfer_data == NULL) {
		FM_ULOG_ERR("Invalid Parameters\n");
		return TRUE;
	}
	if (backup_started == 2) {
		FM_ULOG_ERR("Backup Started returning 0 to stop xfer\n");
		backup_started = 1;
		return 0;
	}

	if (operation_cancel && progress_info->phase != GNOME_VFS_XFER_PHASE_COMPLETED) {
		if (progress_info->target_name != NULL
		    && progress_info->source_name != NULL
		    && g_strcasecmp(progress_info->source_name, progress_info->target_name) != 0
		    && progress_info->bytes_total != progress_info->bytes_copied) {
			xfer_data->partial_copied_file =
			    g_list_append(xfer_data->partial_copied_file,
					  g_strdup(progress_info->target_name));

			xfer_data->partial_copied_file =
			    g_list_append(xfer_data->partial_copied_file,
					  g_strdup(progress_info->source_name));
			//stop_mmc_umount_delay_timer();
		}
		FM_ULOG_INFO("returning 0 from progress_callback to cancel operation\n");
		return 0;
	}
	gdk_threads_enter();
	switch (progress_info->status) {
	case GNOME_VFS_XFER_PROGRESS_STATUS_OK:
		retval = handle_progress_ok(progress_info, xfer_data);
		break;
	case GNOME_VFS_XFER_PROGRESS_STATUS_VFSERROR:
		retval = handle_progress_vfs_error(progress_info, xfer_data);
		break;

	case GNOME_VFS_XFER_PROGRESS_STATUS_OVERWRITE:
		retval = handle_progress_overwrite(progress_info, xfer_data);
		break;

	case GNOME_VFS_XFER_PROGRESS_STATUS_DUPLICATE:
		retval = handle_progress_duplicate(progress_info, xfer_data);
		break;

	default:
		FM_ULOG_ERR("Unknown GnomeVFSXferProgressStatus %d\n", progress_info->status);
		retval = TRUE;
		break;
	}
	gdk_threads_leave();
	//Checking the cancelation flags again to ensure cancellation of operation 
	//for the flags which changed in the mean time.
	//Mostely at the time of replace confirmation dialog
	//and at the time of updating the progress bar.
	if (backup_started == 2) {
		FM_ULOG_INFO("Backup Started returning 0 to stop xfer\n");
		backup_started = 1;
		return 0;
	}
	DEBUG_PRINT_END;
	return retval;
}

/**
 * xfer_watchdog_cleanup_idle:
 * Func to call cleanup when operation is cancelled by watchdog timer
 * param data callback data
 * @return gboolean
 **/
gboolean 
xfer_watchdog_cleanup_idle(gpointer data)
{
	XferCallbackData *temp_cdata = NULL;
	temp_cdata = ((OSSOFileManager *)data)->xfer_callback_data;
	FM_ULOG_INFO("CLEANING UP FROM WATCH DOG\n");
	gdk_threads_enter();
	xfer_cleanup((XferCallbackData **)&temp_cdata);
	gdk_threads_leave();
	return FALSE;
}
/**
 * xfer_watchdog_timer:
 * Watch dog timer function to see if the progress is hung
 * @param data timer callback data
 * @return gboolean
 **/
gboolean 
xfer_watchdog_timer(gpointer data)
{
	XferCallbackData *cdata = NULL;
	DEBUG_PRINT_START;
	cdata = ((OSSOFileManager *)data)->xfer_callback_data;
	if(((OSSOFileManager *)data)->xfer_callback_data == NULL)
	{
		FM_ULOG_INFO("xfer_watchdog:xfer cbdata already freed. i am stoping\n");
		return FALSE;
	}
	if(g_timer_elapsed(cdata->watchdog_timer,NULL) > 
			GNOMEVFS_RESPONSE_TIMEOUT)
	{
		gnome_vfs_async_cancel(cdata->async_handle);
		//Cleanup of the callback data will be done in an idle function since
		//there is a chance that the gnomevfs callbacks will get called even after 
		//calling gnome_vfs_async_cancel if they are already dispatched which will lead
		//to a crash if we free the callback data now without checking if the callbacks
		//are getting called.
		g_idle_add(xfer_watchdog_cleanup_idle,data);
		global_ofm->gnomevfs_watchdog_timer_source = 0;
		return FALSE;
	}
	return TRUE;
}
/**
 * xfer_dialog_on_response_event:
 * Progress of the operation is Cancelled
 * 
 * @param widget	Pointer to progress window
 * @param cdata		Pointer to callback data 
 * 
 * @return 		void
 */
void xfer_dialog_on_response_event(GtkWidget * widget, gint resp,XferCallbackData * data)
{
	g_return_if_fail(data != NULL);

	DEBUG_PRINT_START;
	operation_cancel = TRUE;
	FM_ULOG_INFO("XFER PROGRESS CANCEL PRESSED\n");
	if(global_ofm->gnomevfs_watchdog_timer_source == 0)
	{
		global_ofm->gnomevfs_watchdog_timer_source = 
			g_timeout_add(1000,xfer_watchdog_timer,global_ofm);
	}
	DEBUG_PRINT_END;
	return;
}

/**
 * xfer_init:
 * Creates Progress dialog
 * 
 * @param cdata		Pointer to callback data
 * @param mainwindow	Pointer to Main Window of File Manager Application.
 * 
 * @return 		void
 */
void xfer_init(XferCallbackData * cdata, GtkWindow * mainwindow)
{

	GtkWidget *xfer_progress_dialog = NULL;

	operation_cancel = FALSE;

	DEBUG_PRINT_START;
	cdata->delete_sources = FALSE;
	if (mainwindow == NULL) {
		FM_ULOG_ERR("Error validating input parameters\n");
	}
	if(cdata->action != XFER_RENAME)
	{
		cdata->progress_bar = GTK_PROGRESS_BAR(gtk_progress_bar_new());
		xfer_progress_dialog = (GtkWidget *)
		    hildon_note_new_cancel_with_progress_bar(GTK_WINDOW(mainwindow),
							     cdata->progress_verb,
							     cdata->progress_bar);
		gtk_window_set_transient_for(GTK_WINDOW(xfer_progress_dialog), mainwindow);
		g_get_current_time(&(cdata->prev_upd_time));
		cdata->last_index = 0;
		cdata->prev_progress = 0;
		gtk_window_set_modal(GTK_WINDOW(xfer_progress_dialog), TRUE);

		cdata->status_window = xfer_progress_dialog;
		g_return_if_fail(cdata->status_window);

		g_signal_connect(G_OBJECT(cdata->status_window), "response",
				 G_CALLBACK(xfer_dialog_on_response_event), cdata);
	}

	DEBUG_PRINT_END;
	return;
}




/**
 * xfer_cleanup:
 * Destroy the Progress window and cleans the callback data
 * 
 * @param cdata		Pointer to callback data 
 * 
 * @return 		void
 */
void xfer_cleanup(XferCallbackData ** cdata)
{
	GTimeVal end_time;
	glong diff = 0;
	g_return_if_fail((*cdata) != NULL);

	DEBUG_PRINT_START;

	FM_ULOG_ERR("xfer_cleanup: CLEAN UP STARTED\n");
	if(global_ofm->xfer_callback_data == NULL)
	{
		FM_ULOG_ERR("xfer_cleanup: CLEAN UP ALREADY DONE returning\n");
		return;
	}
	global_ofm->xfer_callback_data = NULL;
	
	g_get_current_time(&end_time);
	diff = timeval_diff(&end_time, &start_time);
	FM_ULOG_INFO("TIME TAKEN FOR OPERATION %ld\n", diff);
	FM_ULOG_INFO("Performing SYNC\n");
	sync();
	FM_ULOG_INFO("Completed SYNC\n");

	if ((*cdata)->status_window != NULL)
		gtk_widget_destroy((*cdata)->status_window);
	(*cdata)->status_window = NULL;

	if ((*cdata)->progress_verb != NULL)
		g_free((*cdata)->progress_verb);
	(*cdata)->progress_verb = NULL;

	if ((*cdata)->res.button_text1 != NULL)
		g_free((*cdata)->res.button_text1);
	(*cdata)->res.button_text1 = NULL;

	if ((*cdata)->res.button_text2 != NULL)
		g_free((*cdata)->res.button_text2);
	(*cdata)->res.button_text2 = NULL;

	if ((*cdata)->res.final_result_text != NULL)
		g_free((*cdata)->res.final_result_text);
	(*cdata)->res.final_result_text = NULL;

	if ((*cdata)->res.final_result_note_button_text != NULL)
		g_free((*cdata)->res.final_result_note_button_text);
	(*cdata)->res.final_result_note_button_text = NULL;

	if ((*cdata)->detlists.source_list != NULL)
		osso_fm_free_gnomevfs_uri_list((*cdata)->detlists.source_list);
	(*cdata)->detlists.source_list = NULL;

	if ((*cdata)->detlists.target_list != NULL)
		osso_fm_free_gnomevfs_uri_list((*cdata)->detlists.target_list);
	(*cdata)->detlists.target_list = NULL;

	if ((*cdata)->detlists.source_dir_list != NULL)
		osso_fm_free_gnomevfs_uri_list((*cdata)->detlists.source_dir_list);
	(*cdata)->detlists.source_dir_list = NULL;

	if ((*cdata)->detlists.target_dir_list != NULL)
		osso_fm_free_gnomevfs_uri_list((*cdata)->detlists.target_dir_list);
	(*cdata)->detlists.target_dir_list = NULL;

	if ((*cdata)->detlists.long_names != NULL)
		osso_fm_free_gnomevfs_uri_list((*cdata)->detlists.long_names);
	(*cdata)->detlists.long_names = NULL;

	if((*cdata)->watchdog_timer != NULL)
		g_timer_destroy((*cdata)->watchdog_timer);
	(*cdata)->watchdog_timer = NULL;

	g_free(*cdata);
	*cdata = NULL;

	DEBUG_PRINT_END;
	return;
}




/**
 * check_source_to_target:
 * Check if the source and target list are same
 * 
 * @param sourceselection	Pointer to selected source list
 * @param targetpath		Pointer to target location 
 * 
 * @return 		int
 * 			0 If copy/move is possible
 * 			-1 can not perform this operation(unknown err)
 * 			-1 if target is in the source
 * 			-2 if target replaces the source
 * 				
 **/
SrcToTgtStatus check_source_to_target(GSList * sourceselection, GtkFilePath * targetpath)
{


	GSList *cursel = NULL;
	GnomeVFSURI *src = NULL;
	GnomeVFSURI *dst = NULL;
	GnomeVFSURI *dstentry = NULL;
	gchar *shortname = NULL;

	//gchar *dirname = NULL;

	DEBUG_PRINT_START;
	if ((sourceselection == NULL) || (targetpath == NULL)) {
		return -1;
	}
	dst = osso_fm_path_to_uri(targetpath);
	if (dst == NULL) {
		FM_ULOG_ERR("Error allocating memory\n");
		return -1;
	}

	for (cursel = sourceselection; cursel != NULL; cursel = cursel->next) {
		shortname = osso_fm_path_get_shortname(cursel->data);
		dstentry = gnome_vfs_uri_append_path(dst, shortname);
		g_free(shortname);

		src = gnome_vfs_uri_new((gchar *) cursel->data);
		if (src == NULL) {
			FM_ULOG_ERR("Error allocating memory\n");
			return -1;
		}
		if (gnome_vfs_uri_is_parent(src, dst, TRUE) || gnome_vfs_uri_equal(src, dst)) {
			gnome_vfs_uri_unref(dst);
			gnome_vfs_uri_unref(src);
			gnome_vfs_uri_unref(dstentry);
			return -2;
		} else if (gnome_vfs_uri_exists(dstentry)) {
			if (gnome_vfs_uri_is_parent(dstentry, src, TRUE)) {
				gnome_vfs_uri_unref(dst);
				gnome_vfs_uri_unref(src);
				gnome_vfs_uri_unref(dstentry);
				return -3;
			}
		}
		gnome_vfs_uri_unref(src);
		gnome_vfs_uri_unref(dstentry);
	}
	gnome_vfs_uri_unref(dst);
	DEBUG_PRINT_END;
	return 0;
}

/**
 * match_parents_path:
 * Check if the source parent and target are same
 * 
 * @param sourceselection	Pointer to selected source list
 * @param targetpath		Pointer to target location 
 * 
 * @return 			gboolean
 **/
gboolean match_parents_path(GSList * sourceselection, GtkFilePath * targetpath)
{

	GnomeVFSURI *sourceuri = NULL;
	GnomeVFSURI *parenturi = NULL;
	gchar *sourceparent = NULL;
	gchar *targetpath_texturi = NULL;

	DEBUG_PRINT_START;

	if ((sourceselection == NULL) || (targetpath == NULL) || (sourceselection->data == NULL)) {
		return FALSE;
	}
	sourceuri = osso_fm_path_to_uri(sourceselection->data);
	if (sourceuri == NULL) {
		return FALSE;
	}
	parenturi = gnome_vfs_uri_get_parent(sourceuri);
	sourceparent = gnome_vfs_uri_to_string(parenturi, GNOME_VFS_URI_HIDE_NONE);
	gnome_vfs_uri_unref(sourceuri);
	if (sourceparent == NULL) {
		gnome_vfs_uri_unref(parenturi);
		return FALSE;
	}
	targetpath_texturi = osso_fm_path_to_texturi(targetpath);
	FM_ULOG_DEBUG1("source path : %s\n", (gchar *) sourceselection->data);
	FM_ULOG_DEBUG1("target path : %s\n", targetpath_texturi);
	FM_ULOG_DEBUG1("source parent: %s\n", sourceparent);
	if (g_strcasecmp(sourceparent, targetpath_texturi) == 0) {
		gnome_vfs_uri_unref(parenturi);
		g_free(sourceparent);
		g_free(targetpath_texturi);
		return TRUE;
	}

	gnome_vfs_uri_unref(parenturi);
	g_free(sourceparent);
	g_free(targetpath_texturi);
	DEBUG_PRINT_END;
	return FALSE;
}

/** 
 * can_progress_update:
 * This is a helper function which tells weather the progressbar needs to
 * be updated as frequent update of progress bar slows down the operations
 * 1) consider time of last update(typical interval 500mS)
 * 2) consider the amount of progress (typical 10%)
 * 3) consider the file inder (typical for every file)
 *
 * @param progress_info : Progress Information
 * 
 * @return gboolean 
 **/
gboolean can_progress_update(GnomeVFSXferProgressInfo * progress_info, XferCallbackData * cdata)
{
	GTimeVal curtime = { 0, 0 };
	glong diff = 0;
	gint progress = 0;
	if (cdata == NULL || progress_info == NULL) {
		FM_ULOG_ERR("can_progress_update : Error in parameters \n");
		return FALSE;
	}
	DEBUG_PRINT_START;
	g_get_current_time(&curtime);
	diff = timeval_diff(&curtime, &(cdata->prev_upd_time));
	if (progress_info->bytes_total == 0) {
		return FALSE;
	}
	progress = (progress_info->total_bytes_copied * 100) / progress_info->bytes_total;
	//FM_ULOG_ERR("TDIF %03lu ms  progress %d  Index %lu\n",
//                diff/1000,progress,progress_info->file_index);
	if ((diff > 500000) ||
	    ((progress - cdata->prev_progress) > 10) ||
	    progress_info->file_index != cdata->last_index) {
		cdata->prev_upd_time.tv_sec = curtime.tv_sec;
		cdata->prev_upd_time.tv_usec = curtime.tv_usec;
		cdata->last_index = progress_info->file_index;
		cdata->prev_progress = progress;
		DEBUG_PRINT_END;
		//FM_ULOG_INFO("UPDATED\n");
		return TRUE;
	}

	DEBUG_PRINT_END;
	//FM_ULOG_INFO("NOT UPDATED\n");
	return FALSE;
}

/**
 * osso_fm_file_operations_progress_update:
 * Updates the progres information dialog with 
 * Progressing info received from the callback
 * 
 * @param progress_info: progress info 
 * @param cdata: pointer to callback data
 * 
 * @return gboolean
 */
gboolean
osso_fm_file_operations_progress_update(GnomeVFSXferProgressInfo * progress_info,
					XferCallbackData * cdata)
{
	gchar *progresslabel = NULL;
	//static gint last_index = -1;
	static gint last_index = 0;
	gdouble fraction = 0;

	DEBUG_PRINT_START;
	if ((progress_info == NULL) || (cdata == NULL) ){
		FM_ULOG_ERR("Invalid parameter ");
		return FALSE;
	}
	if(cdata->status_window == NULL)
	{
		//No progress banner, so no updation
		//This case is for rename where progress banner is removed
		return TRUE;
	}	
	last_index = cdata->last_index;
	if (!can_progress_update(progress_info, cdata))
		return TRUE;
	if (progress_info->bytes_total > 0) {
		fraction = (gdouble)progress_info->total_bytes_copied / (gdouble)progress_info->bytes_total;
		//gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(cdata->progress_bar),
		//			      (progress_info->total_bytes_copied *
		//			       1.0) / progress_info->bytes_total);
		gtk_progress_bar_set_fraction(cdata->progress_bar,fraction);
		if (last_index != progress_info->file_index) {
			if (progress_info->files_total > 1 && cdata->action != XFER_DUPLICATE) {
				progresslabel = g_strdup_printf("%s %ld/%ld",
								cdata->progress_verb,
								progress_info->file_index,
								progress_info->files_total);
				gtk_widget_set_size_request(cdata->status_window,
				  osso_fm_predict_progress_dialog_width_based_on_text(cdata->status_window,progresslabel),
				  -1);
				g_object_set(G_OBJECT(cdata->status_window),
					     "description", progresslabel, NULL);
				g_free(progresslabel);
			}
			last_index = progress_info->file_index;
		}
	}

	DEBUG_PRINT_END;
	return TRUE;
}

/**
 * display_xfer_error:
 * A function to display error messages after operations.
 * @param listptr timer callback data
 * @return gboolean
 **/
static gboolean display_xfer_error(gpointer listptr)
{
	GSList *list = NULL;
	gchar *msg_text = NULL;
	gchar *but_text = NULL;
	GtkWindow *parent = NULL;

	list = (GSList *) listptr;
	if (list == NULL || list->next == NULL) {
		FM_ULOG_ERR("display_xfer_error:Invalid param\n");
		return FALSE;
	}
	msg_text = list->data;
	but_text = list->next->data;
	if (list->next->next != NULL) {
		parent = (GtkWindow *) list->next->next->data;
	}
	
	gdk_threads_enter();
	if(but_text!=NULL)
	{
		if(strlen(but_text) != 0)
		{
			fm_infonote(parent,msg_text,but_text);
		}
		else
		{
			fm_infonote(parent,msg_text,NULL);
		}
	}
	else
	{
		fm_infoprint(parent,msg_text);
	}
	gdk_threads_leave();
	g_free(msg_text);
	g_free(but_text);
	g_slist_free(list);
	return FALSE;
}

/**
 * phase_completed_tasks:
 * A function to perform the clean up and error reporting tasks after operation
 * @param cdata XferCallbackData
 * @return gint
 **/
static gint phase_completed_tasks(XferCallbackData * cdata)
{
	GnomeVFSResult partial_del_ret = 0;
	GSList *errmsglist = NULL;
	GList *cur_tgt = NULL;
	GList *cur_src = NULL;
	GnomeVFSResult result;
	if (backup_started == 1) {
		backup_started = 0;
	}

	if (cdata->partial_copied_file != NULL) {
		if (cdata->partial_copied_file->data == NULL ||
		    cdata->partial_copied_file->next == NULL ||
		    cdata->partial_copied_file->next->data == NULL) {
			FM_ULOG_ERR("Error in partial copied file delete list\n");
		} else {
			if (operation_cancel) {
			
				if (is_path_exists(cdata->partial_copied_file->next->data))
				{
					partial_del_ret = gnome_vfs_unlink( (gchar *) 
							cdata->partial_copied_file->data);
				}
			}
			else
			{
				partial_del_ret = gnome_vfs_unlink((gchar *) 
						cdata->partial_copied_file->data);
			}
			if (partial_del_ret != GNOME_VFS_OK) {
				FM_ULOG_INFO
				    ("Delete the partially copied file Err#%d\n",
				     partial_del_ret);
			}
			g_free(cdata->partial_copied_file->data);
			g_free(cdata->partial_copied_file->next->data);
			g_list_free(cdata->partial_copied_file);
			cdata->partial_copied_file = NULL;
		}
	}
	if (operation_cancel || cdata->delete_sources) {
		if (cdata->action == XFER_MOVE || cdata->action == XFER_CUTPASTE) {
			//delete the source if target exists
			for (cur_tgt = cdata->detlists.target_list, cur_src = cdata->detlists.source_list;
			     cur_tgt != NULL && cur_src != NULL;
			     cur_tgt = cur_tgt->next, cur_src = cur_src->next) {
				FM_ULOG_DEBUG1("CUR CHK FILE %s - %s\n",
					    gnome_vfs_uri_to_string(cur_src->data, 0),
					    gnome_vfs_uri_to_string(cur_tgt->data, 0));
				if (gnome_vfs_uri_exists((GnomeVFSURI *) cur_tgt->data)) {
					if (gnome_vfs_uri_exists((GnomeVFSURI *) cur_src->data)) {
						if (gnome_vfs_unlink_from_uri
						    ((GnomeVFSURI *) cur_src->data) !=
						    GNOME_VFS_OK) {
							FM_ULOG_ERR("Error in deleting src file\n");
						}
					}
				}
			}
			for (cur_tgt = cdata->detlists.target_dir_list, cur_src =
			     cdata->detlists.source_dir_list; cur_tgt != NULL && cur_src != NULL;
			     cur_tgt = cur_tgt->next, cur_src = cur_src->next) {
				FM_ULOG_DEBUG1("CUR CHK FILE %s - %s\n",
					    gnome_vfs_uri_to_string(cur_src->data, 0),
					    gnome_vfs_uri_to_string(cur_tgt->data, 0));
				if (gnome_vfs_uri_exists((GnomeVFSURI *) cur_tgt->data)) {
					if (gnome_vfs_uri_exists((GnomeVFSURI *) cur_src->data)) {
						result =
						    gnome_vfs_remove_directory_from_uri((GnomeVFSURI
											 *)
											cur_src->
											data);
						if (result != GNOME_VFS_OK
						    && result !=
						    GNOME_VFS_ERROR_DIRECTORY_NOT_EMPTY) {
							FM_ULOG_ERR("Error in deleting src file\n");
						}
					}
				}
			}
		}
	}
	if(!operation_cancel && cdata->res.final_result_text == NULL)
	{
		GtkTreeIter iter;
		GtkFilePath *path = NULL;
		HildonFileSystemModel *model = NULL;
		g_object_get(G_OBJECT(global_ofm->fs), "model", &model, NULL);
		
		for (cur_tgt = cdata->detlists.target_list;
		     cur_tgt != NULL;
		     cur_tgt = cur_tgt->next) 
		{
			path = osso_fm_uri_to_path(cur_tgt->data);
			hildon_file_system_model_load_path(model,path,&iter);
			gtk_file_path_free(path);
		}
	}
		
	if (cdata->res.final_result_text == NULL) {
		update_thumbnails(cdata->detlists.source_list, cdata->detlists.target_list, cdata->action);
	}

	if (cdata->res.final_result_text == NULL && cdata->item_counts.skipped > 0) {
		DEBUG_PRINT;
		switch (cdata->action) {
		case XFER_DUPLICATE:
		case XFER_COPY:
		case XFER_RENAME:
		case XFER_NEWFOLDER:
			break;
		case XFER_MOVE:
			if(cdata->item_counts.skipped > cdata->item_counts.non_existant)
			{
				xfer_set_final_error_text(cdata, 1, FALSE,
						  _("sfil_ni_cannot_move_some_items"),
						  _("sfil_bd_cannot_move_some_items_ok"));
			}
			break;
		case XFER_CUTPASTE:	//TODO this case has to be handled before operation
			//  xfer_set_final_error_text(cdata,1,FALSE,
			//    _("sfil_ib_cannot_cut_some_items"));
			break;
		case XFER_DELETE:
			xfer_set_final_error_text(cdata, 1, FALSE,
						  _("sfil_ni_cannot_delete_some_items"),
						  _("sfil_bd_cannot_delete_some_items_ok"));
			break;
		}
	}
	if (cdata->res.final_result_text != NULL) {

		FM_ULOG_INFO(cdata->res.final_result_text);
		errmsglist = g_slist_append(errmsglist, g_strdup(cdata->res.final_result_text));

		if (cdata->res.final_result_show_type == 1) {
			if(cdata->res.final_result_note_button_text != NULL)
			{
				errmsglist = g_slist_append(errmsglist,
						    g_strdup(cdata->res.final_result_note_button_text));
			}
			else
			{
				errmsglist = g_slist_append(errmsglist,g_strdup(""));
			}
		}
		else
		{
			errmsglist=g_slist_append(errmsglist,NULL);
		}
		errmsglist = g_slist_append(errmsglist, cdata->window);
		g_timeout_add(100, display_xfer_error, errmsglist);
	}
	xfer_cleanup(&cdata);
	osso_fm_set_operation_state(OPER_NONE);
	return 1;

}

/**
 * handle_progress_ok:
 * Updates the progres information dialog with 
 * Progressing info received from the callback
 * 
 * @param progress_info	Progress info 
 * @param cdata		Pointer to callback data
 * 
 * @return 		int
 */
static int handle_progress_ok(GnomeVFSXferProgressInfo * progress_info, XferCallbackData * cdata)
{
	gint retval = 0;
	static gboolean completed_called = FALSE;
	DEBUG_PRINT_START;
	switch (progress_info->phase) {
	case GNOME_VFS_XFER_PHASE_INITIAL:
		if (cdata->status_window != NULL) {
			g_object_set(G_OBJECT(cdata->status_window),
				     "description",
				     _("hfil_nw_preparing"), NULL);
			hildon_note_set_button_text(
				HILDON_NOTE(cdata->status_window),
				_("hfil_bd_preparing_cancel"));
			gtk_widget_show(cdata->status_window);

			/* show the dialog now */
			while (gtk_events_pending()) {
				gtk_main_iteration();
			}
		}
		completed_called = FALSE;
		FM_ULOG_DEBUG1("GNOME_VFS_PHASE_INITIAL\n");
		retval = 1;
		break;
	case GNOME_VFS_XFER_CHECKING_DESTINATION:
		completed_called = FALSE;
		FM_ULOG_DEBUG1("GNOME_VFS_PHASE_CHECKING_DESTINATION\n");
		retval = 1;
		break;
	case GNOME_VFS_XFER_PHASE_COLLECTING:
		completed_called = FALSE;
		FM_ULOG_DEBUG1("GNOME_VFS_PHASE_COLLECTING, TARGET"
			       " PATH: %s\n", progress_info->source_name);
		if (cdata->status_window != NULL) {
			gtk_progress_bar_pulse(cdata->progress_bar);
		}
		retval = 1;
		break;
	case GNOME_VFS_XFER_PHASE_OPENSOURCE:
		retval = 1;
		break;
	case GNOME_VFS_XFER_PHASE_OPENTARGET:
		retval = 1;
		break;
	case GNOME_VFS_XFER_PHASE_FILECOMPLETED:
		FM_ULOG_DEBUG1("TARGET PATH : %s\n", progress_info->target_name);
		DEBUG_PRINT;
		retval = 1;
		break;
	case GNOME_VFS_XFER_PHASE_READYTOGO:
		FM_ULOG_INFO("GNOME_VFS_XFER_PHASE_READYTOGO\n");
		if (cdata->status_window != NULL) {
			g_object_set(G_OBJECT(cdata->status_window),
				     "description",
				     cdata->progress_verb, NULL);
			if (progress_info->files_total > 1) {
				hildon_note_set_button_text(
					HILDON_NOTE(cdata->status_window),
					cdata->res.button_text2);
			} else {
				hildon_note_set_button_text(
					HILDON_NOTE(cdata->status_window),
					cdata->res.button_text1);
			}
			gtk_progress_bar_set_fraction(cdata->progress_bar, 0);

			/* show the updated dialog now */
			while (gtk_events_pending()) {
				gtk_main_iteration();
			}
		}
		retval = 1;
		break;

	case GNOME_VFS_XFER_PHASE_DELETESOURCE:
		if (cdata->action == XFER_DELETE) {
			retval = osso_fm_file_operations_progress_update(progress_info, cdata);
		} else {
			retval = 1;
		}
		break;

	case GNOME_VFS_XFER_PHASE_MOVING:
	case GNOME_VFS_XFER_PHASE_COPYING:
		DEBUG_PRINT;
		retval = osso_fm_file_operations_progress_update(progress_info, cdata);
		DEBUG_PRINT;
		break;

	case GNOME_VFS_XFER_PHASE_CLEANUP:
		DEBUG_PRINT;
		FM_ULOG_INFO("GNOME_VFS_XFER_PHASE_CLEANUP\n");

		if (cdata->action == XFER_CUTPASTE) {
			osso_fm_clipboard_clear();
			if (cdata->locked_files != NULL) {
				GList *l;
				FM_ULOG_INFO("restore files that were locked");
				osso_fm_copy_glist_to_clipboard(
					(GtkWidget*)global_ofm->window,
					cdata->locked_files, TRUE);
				for (l = cdata->locked_files; l != NULL;
				     l = g_list_next(l)) {
					g_free(l->data);
				}
				g_list_free(cdata->locked_files);
				cdata->locked_files = NULL;
			}
		}

		retval = 1;
		break;

	case GNOME_VFS_XFER_PHASE_COMPLETED:
		FM_ULOG_INFO("Prog GNOME_VFS_XFER_PHASE_COMPLETED\n");
		if (completed_called) {
			FM_ULOG_ERR("Prog GNOME_VFS_XFER_PHASE_COMPLETED called again\n");
			retval = 1;
			break;
		}
		completed_called = TRUE;
		retval = 1;
	default:
		DEBUG_PRINT;
		retval = 1;
		break;
	}
	DEBUG_PRINT_END;
	return retval;
}

/**
 * A function to the error text 
 * @param cdata XferCallbackData where the error is stored
 * @param show_type note or info print
 * @param reset_error gboolean clear previous error
 * @param err_text error
 * @param button_text text of button for into notes
 * @return void
 **/
void
xfer_set_final_error_text(XferCallbackData * cdata, gint show_type,
			  gboolean reset_error, gchar * err_text, gchar * button_text)
{
	DEBUG_PRINT_START;
	g_print(err_text);
	if (reset_error == TRUE) {
		if (cdata->res.final_result_text != NULL) {
			g_free(cdata->res.final_result_text);
			if (cdata->res.final_result_note_button_text != NULL) {
				g_free(cdata->res.final_result_note_button_text);
			}
			cdata->res.final_result_text = g_strdup(err_text);
			if (button_text != NULL) {
				cdata->res.final_result_note_button_text = g_strdup(button_text);
			}
			cdata->res.final_result_show_type = show_type;
			DEBUG_PRINT_END;
			return;
		}
	}
	if (cdata->res.final_result_text == NULL) {
		cdata->res.final_result_text = g_strdup(err_text);
		if (button_text != NULL) {
			cdata->res.final_result_note_button_text = g_strdup(button_text);
		}
		cdata->res.final_result_show_type = show_type;
	}
	DEBUG_PRINT_END;
	return;
}

static gboolean check_target_not_found(GnomeVFSXferProgressInfo * progress_info)
{
	GtkFilePath *path = NULL;
	GtkFilePath *ppath = NULL;
	gchar *parent_texturi=NULL;
	DEBUG_PRINT_START;
	if (progress_info->phase != GNOME_VFS_XFER_PHASE_WRITETARGET &&
			(progress_info->phase ==GNOME_VFS_XFER_CHECKING_DESTINATION
			|| progress_info->phase == GNOME_VFS_XFER_PHASE_OPENTARGET
			|| progress_info->phase ==GNOME_VFS_XFER_PHASE_WRITETARGET 
			|| progress_info->phase==GNOME_VFS_XFER_PHASE_CLOSETARGET ))
	{
		DEBUG_PRINT_END;
		return TRUE;
	}
	path = osso_fm_texturi_to_path(progress_info->target_name);
	if(path==NULL) {
		DEBUG_PRINT_END;
		return TRUE;
	}
	ppath = osso_fm_get_parent(path);
	if (ppath ==NULL) {
		DEBUG_PRINT_END;
		return TRUE;
	}
	parent_texturi= osso_fm_path_to_texturi(ppath);
	gtk_file_path_free(path);
	gtk_file_path_free(ppath);
	if (progress_info->phase == GNOME_VFS_XFER_PHASE_MOVING && 
			!( is_path_exists (parent_texturi))) 
	{
		if(parent_texturi)
		{
			g_free(parent_texturi);
		}
		DEBUG_PRINT_END;
		return TRUE;
	}
	if(parent_texturi){
		g_free(parent_texturi);
	}
	DEBUG_PRINT_END;
	return FALSE;
	
}

/**
 * handle_xfer_duplicate_errors:
 * Error determination function function
 * @param progress_info
 * @param cdata
 * @return int
 **/
static int
handle_xfer_duplicate_errors(GnomeVFSXferProgressInfo * progress_info, XferCallbackData * cdata)
{
	DEBUG_PRINT_START;
	switch (progress_info->vfs_status) {
	case GNOME_VFS_ERROR_NOT_FOUND:
	case GNOME_VFS_ERROR_BAD_PARAMETERS:
	case GNOME_VFS_ERROR_BAD_FILE:
	case GNOME_VFS_ERROR_INVALID_URI:

		if (selectionstate.no_folders > 0) {
			if (progress_info->phase != GNOME_VFS_XFER_PHASE_WRITETARGET) {
				xfer_set_final_error_text(cdata, 1, FALSE,
							  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
								  "sfil_ni_cannot_continue_target_folder_deleted"),
							  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
								"sfil_ni_cannot_continue_target_folder_deleted_ok"));
			} else {
				xfer_set_final_error_text(cdata, 1, FALSE,
							  _
							  ("sfil_ni_cannot_duplicate_folder_not_found"),
							  _
							  ("sfil_bd_cannot_duplicate_folder_not_found_ok"));
			}
		} else {
			xfer_set_final_error_text(cdata, 1, FALSE,
						  _("sfil_ni_cannot_duplicate_file_not_found"),
						  _("sfil_bd_cannot_duplicate_file_not_found_ok"));
		}
		break;

	case GNOME_VFS_ERROR_TOO_BIG:
	case GNOME_VFS_ERROR_NO_SPACE:
		xfer_set_final_error_text(cdata, 1, FALSE,
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_not_enough_memory"),
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_not_enough_memory_ok"));
		break;
	case GNOME_VFS_ERROR_READ_ONLY:
	case GNOME_VFS_ERROR_NOT_OPEN:
	case GNOME_VFS_ERROR_INVALID_OPEN_MODE:
	case GNOME_VFS_ERROR_ACCESS_DENIED:
		if (selectionstate.no_folders > 0) {
			xfer_set_final_error_text(cdata, 0, FALSE,
//						  (gchar *)						//CHECK
//						  osso_common_error(CERM_CAN_NOT_DUP_FOLDER), NULL);
							_("sfil_ib_cannot_duplicate_folder"),NULL);						  
		} else {
			xfer_set_final_error_text(cdata, 0, FALSE,
						  (gchar *)
//						  osso_common_error(CERM_CAN_NOT_DUP_FILE), NULL);
							_("sfil_ib_cannot_duplicate_file"),NULL);						  
		}
		break;
	case GNOME_VFS_ERROR_NAME_TOO_LONG:
		break;
	case GNOME_VFS_ERROR_GENERIC:
	case GNOME_VFS_ERROR_INTERNAL:
	case GNOME_VFS_ERROR_NOT_SUPPORTED:
	case GNOME_VFS_ERROR_IO:
	case GNOME_VFS_ERROR_CORRUPTED_DATA:
	case GNOME_VFS_ERROR_WRONG_FORMAT:
	case GNOME_VFS_ERROR_TOO_MANY_OPEN_FILES:
	case GNOME_VFS_ERROR_EOF:
	case GNOME_VFS_ERROR_NOT_A_DIRECTORY:
	case GNOME_VFS_ERROR_IN_PROGRESS:
	case GNOME_VFS_ERROR_INTERRUPTED:
	case GNOME_VFS_ERROR_FILE_EXISTS:
	case GNOME_VFS_ERROR_LOOP:
	case GNOME_VFS_ERROR_NOT_PERMITTED:
	case GNOME_VFS_ERROR_IS_DIRECTORY:
	case GNOME_VFS_ERROR_NO_MEMORY:
	case GNOME_VFS_ERROR_HOST_NOT_FOUND:
	case GNOME_VFS_ERROR_INVALID_HOST_NAME:
	case GNOME_VFS_ERROR_HOST_HAS_NO_ADDRESS:
	case GNOME_VFS_ERROR_LOGIN_FAILED:
	case GNOME_VFS_ERROR_CANCELLED:
	case GNOME_VFS_ERROR_DIRECTORY_BUSY:
	case GNOME_VFS_ERROR_DIRECTORY_NOT_EMPTY:
	case GNOME_VFS_ERROR_TOO_MANY_LINKS:
	case GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM:
	case GNOME_VFS_ERROR_NOT_SAME_FILE_SYSTEM:
	case GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE:
	case GNOME_VFS_ERROR_SERVICE_OBSOLETE:
	case GNOME_VFS_ERROR_PROTOCOL_ERROR:
	case GNOME_VFS_ERROR_NO_MASTER_BROWSER:
	case GNOME_VFS_ERROR_NO_DEFAULT:
	case GNOME_VFS_ERROR_NO_HANDLER:
	case GNOME_VFS_ERROR_PARSE:
	case GNOME_VFS_ERROR_LAUNCH:
	case GNOME_VFS_ERROR_TIMEOUT:
	case GNOME_VFS_ERROR_NAMESERVER:
	case GNOME_VFS_ERROR_LOCKED:
	case GNOME_VFS_ERROR_DEPRECATED_FUNCTION:
	case GNOME_VFS_NUM_ERRORS:
		xfer_set_final_error_text(cdata, 1, FALSE,
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_operation_failed"),
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_operation_failed_ok"));
		break;
	case GNOME_VFS_OK:
	default:
		//this case should not araise
		break;
	}
	DEBUG_PRINT_END;
	return TRUE;
}

/**
 * handle_xfer_copy_errors:
 * Error determination function function
 * @param progress_info
 * @param cdata
 * @return int
 **/
static int handle_xfer_copy_errors(GnomeVFSXferProgressInfo * progress_info,
				   XferCallbackData * cdata)
{
	gint retval = 0;
	DEBUG_PRINT_START;
	retval = progress_info->status;

	switch (progress_info->vfs_status) {
	case GNOME_VFS_ERROR_NOT_FOUND:
		cdata->item_counts.non_existant++;
		/* fall through */
	case GNOME_VFS_ERROR_BAD_PARAMETERS:
	case GNOME_VFS_ERROR_BAD_FILE:
	case GNOME_VFS_ERROR_INVALID_URI:
		if (progress_info->phase != GNOME_VFS_XFER_PHASE_WRITETARGET) {
			if (progress_info->files_total > 1) {
				xfer_set_final_error_text(cdata, 0, FALSE,
							  dgettext(HILDON_FM_L10N_PACKAGE,
								  	"sfil_ib_items_not_found"), NULL);
			} else {
				if (selectionstate.no_folders > 0)
					xfer_set_final_error_text(cdata, 0, FALSE,
						  dgettext(HILDON_FM_L10N_PACKAGE, "sfil_ib_folder_not_found"),
								  NULL);
				else
					xfer_set_final_error_text(cdata, 0, FALSE,
						  dgettext(HILDON_FM_L10N_PACKAGE, "sfil_ib_file_not_found"), NULL);
			}
		}
		break;
	case GNOME_VFS_ERROR_TOO_BIG:	//TODO this option has to be checked
	case GNOME_VFS_ERROR_NO_SPACE:
		xfer_set_final_error_text(cdata, 1, FALSE,
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_not_enough_memory"),
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_not_enough_memory_ok"));
		break;
	case GNOME_VFS_ERROR_ACCESS_DENIED:
		if (progress_info->files_total > 1) {
			if (cdata->item_counts.errors < progress_info->files_total ||
			    cdata->item_counts.skipped > 0) {
				xfer_set_final_error_text(cdata, 1, FALSE,
							  _("sfil_ni_cannot_copy_some_items"),
							  _("sfil_bd_cannot_copy_some_items_ok"));
			} else {
				xfer_set_final_error_text(cdata, 0, FALSE,
							  dgettext(HILDON_FM_L10N_PACKAGE, 
								  "sfil_ib_copying_not_allowed"), NULL);
			}
		}
		break;
	case GNOME_VFS_ERROR_READ_ONLY:
	case GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM:
	case GNOME_VFS_ERROR_NOT_OPEN:
	case GNOME_VFS_ERROR_INVALID_OPEN_MODE:
		xfer_set_final_error_text(cdata, 0, FALSE,
//					  (gchar *) osso_common_error(CERM_PATH_RONLY), NULL);
						dgettext(HILDON_FM_L10N_PACKAGE,"sfil_ib_readonly_location"),NULL);					  
		break;
	case GNOME_VFS_ERROR_FILE_EXISTS:
		break;

	case GNOME_VFS_ERROR_IO:
		xfer_set_final_error_text(cdata, 1, FALSE,
					  _("sfil_ni_cannot_paste_no_connection"),
					  _("sfil_bd_cannot_paste_no_connection_ok"));
		check_for_partial_file(progress_info,cdata);
		break;
	case GNOME_VFS_ERROR_GENERIC:
	case GNOME_VFS_ERROR_INTERNAL:
	case GNOME_VFS_ERROR_NOT_SUPPORTED:
	case GNOME_VFS_ERROR_CORRUPTED_DATA:
	case GNOME_VFS_ERROR_WRONG_FORMAT:
	case GNOME_VFS_ERROR_TOO_MANY_OPEN_FILES:
	case GNOME_VFS_ERROR_EOF:
	case GNOME_VFS_ERROR_NOT_A_DIRECTORY:
	case GNOME_VFS_ERROR_IN_PROGRESS:
	case GNOME_VFS_ERROR_INTERRUPTED:
	case GNOME_VFS_ERROR_LOOP:
	case GNOME_VFS_ERROR_NOT_PERMITTED:
	case GNOME_VFS_ERROR_IS_DIRECTORY:
	case GNOME_VFS_ERROR_NO_MEMORY:
	case GNOME_VFS_ERROR_HOST_NOT_FOUND:
	case GNOME_VFS_ERROR_INVALID_HOST_NAME:
	case GNOME_VFS_ERROR_HOST_HAS_NO_ADDRESS:
	case GNOME_VFS_ERROR_LOGIN_FAILED:
	case GNOME_VFS_ERROR_CANCELLED:
	case GNOME_VFS_ERROR_DIRECTORY_BUSY:
	case GNOME_VFS_ERROR_DIRECTORY_NOT_EMPTY:
	case GNOME_VFS_ERROR_TOO_MANY_LINKS:
	case GNOME_VFS_ERROR_NOT_SAME_FILE_SYSTEM:
	case GNOME_VFS_ERROR_NAME_TOO_LONG:
	case GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE:
	case GNOME_VFS_ERROR_SERVICE_OBSOLETE:
	case GNOME_VFS_ERROR_PROTOCOL_ERROR:
	case GNOME_VFS_ERROR_NO_MASTER_BROWSER:
	case GNOME_VFS_ERROR_NO_DEFAULT:
	case GNOME_VFS_ERROR_NO_HANDLER:
	case GNOME_VFS_ERROR_PARSE:
	case GNOME_VFS_ERROR_LAUNCH:
	case GNOME_VFS_ERROR_TIMEOUT:
	case GNOME_VFS_ERROR_NAMESERVER:
	case GNOME_VFS_ERROR_LOCKED:
	case GNOME_VFS_ERROR_DEPRECATED_FUNCTION:
	case GNOME_VFS_NUM_ERRORS:
		//TODOwhat ever may be the error we are displaying some message
		FM_ULOG_INFO("NO INFO MESSAGE TO THESE CASES\n");
		xfer_set_final_error_text(cdata, 1, FALSE,
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_operation_failed"),
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_operation_failed_ok"));
		break;
	case GNOME_VFS_OK:
	default:
		FM_ULOG_INFO("THIS CASE SHOULD NOT ARAISE SHOULD BE ASSERTED");
		//this case should not araise should be asserted
		break;
	}
	DEBUG_PRINT_END;
	return TRUE;
}

/**
 * handle_xfer_move_errors:
 * Error determination function function
 * @param progress_info
 * @param cdata
 * @return int
 **/
static int handle_xfer_move_errors(GnomeVFSXferProgressInfo * progress_info,
				   XferCallbackData * cdata)
{
	DEBUG_PRINT_START;
	switch (progress_info->vfs_status) {
	case GNOME_VFS_ERROR_NOT_FOUND:
	case GNOME_VFS_ERROR_BAD_PARAMETERS:
	case GNOME_VFS_ERROR_BAD_FILE:
	case GNOME_VFS_ERROR_INVALID_URI:
	case GNOME_VFS_ERROR_CORRUPTED_DATA:
		//if (progress_info->phase != GNOME_VFS_XFER_PHASE_WRITETARGET) {
		if (!check_target_not_found(progress_info)) {
			if (progress_info->files_total > 1) {
				if(progress_info->files_total==progress_info->file_index
						&& cdata->item_counts.errors > 0) {
					xfer_set_final_error_text(cdata, 1, FALSE,
						  _("sfil_ni_cannot_move_items_not_found"),
						  _("sfil_bd_cannot_move_items_not_found_ok"));
				}
				else
				{
					 return GNOME_VFS_XFER_ERROR_ACTION_SKIP;
				}
			} else {
				xfer_set_final_error_text(cdata, 1, FALSE,
						  _("sfil_ni_cannot_move_file_not_found"),
						  _("sfil_bd_cannot_move_file_not_found_ok"));
			}
		} else {
			xfer_set_final_error_text(cdata, 1, FALSE,
				  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					  "sfil_ni_cannot_continue_target_folder_deleted"),
				  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					  "sfil_ni_cannot_continue_target_folder_deleted_ok"));
		}
		break;
	case GNOME_VFS_ERROR_TOO_BIG:	//TODO this option has to be checked
	case GNOME_VFS_ERROR_NO_SPACE:
		xfer_set_final_error_text(cdata, 1, FALSE,
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_not_enough_memory"),
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_not_enough_memory_ok"));
		break;

	case GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM:
		//TODO check if the path is memory card and then display
		xfer_set_final_error_text(cdata, 0, FALSE, dgettext(KE_RECV_L10N_PACKAGE,"mmc_ib_mmc_is_readonly"), NULL);
		break;
	case GNOME_VFS_ERROR_READ_ONLY:
	case GNOME_VFS_ERROR_NOT_OPEN:
	case GNOME_VFS_ERROR_INVALID_OPEN_MODE:
	case GNOME_VFS_ERROR_ACCESS_DENIED:
		if (progress_info->files_total > 1) {
			if (cdata->item_counts.errors < progress_info->files_total ||
			    cdata->item_counts.skipped > 0) {
				xfer_set_final_error_text(cdata, 1, FALSE,
							  _("sfil_ni_cannot_move_some_items"),
							  _("sfil_bd_cannot_move_some_items_ok"));
			} else {
				xfer_set_final_error_text(cdata, 0, TRUE,
						dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
							"sfil_ib_unable_to_move_selected_items"),
						NULL);							  
			}
		} else {
			if (selectionstate.no_folders > 0)
				xfer_set_final_error_text(cdata, 0, FALSE,
							dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
							  "sfil_ib_unable_to_move_selected_folder"),
							  NULL);
			else
				xfer_set_final_error_text(cdata, 0, FALSE,
							  dgettext(HILDON_FM_L10N_PACKAGE,
								  "sfil_ib_cannot_move_file"), NULL);
		}
		break;
	case GNOME_VFS_ERROR_DIRECTORY_NOT_EMPTY:
		if(progress_info->source_name !=NULL && progress_info->target_name !=NULL)  {
		if (strcmp(progress_info->source_name, progress_info->target_name)
		    == 0) {
			xfer_set_final_error_text(cdata, 0, FALSE,
//						  (gchar *)
//						  osso_common_error
//						  (CERM_CAN_NOT_MOVE_ORIG_LOCATION), NULL);
						dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
							"ckdg_ib_unable_to_move_to_current_location"),NULL);						  
		} else {
			xfer_set_final_error_text(cdata, 0, FALSE,
//						  (gchar *)
//						  osso_common_error
//						  (CERM_CAN_NOT_MV_FOLDER_INTO_ITSELF), NULL);
						dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
							"ckdg_ib_unable_to_move_folder_into_itself"),NULL);						  
		}
		} else {
			xfer_set_final_error_text(cdata, 1, FALSE,
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_operation_failed"),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_operation_failed_ok"));
		}
		break;
	case GNOME_VFS_ERROR_FILE_EXISTS:
		break;
	case GNOME_VFS_ERROR_IO:
		xfer_set_final_error_text(cdata, 1, FALSE,
					  _("sfil_ni_cannot_move_no_connection"),
					  _("sfil_bd_cannot_move_no_connection_ok"));
		check_for_partial_file(progress_info,cdata);
		break;
	case GNOME_VFS_ERROR_LOCKED:
		cdata->item_counts.in_use++;
		if (progress_info->files_total > 1) {
			if (cdata->item_counts.in_use < progress_info->files_total ||
			    cdata->item_counts.skipped > 0) {
				xfer_set_final_error_text(cdata, 1, TRUE,
						_("sfil_ni_some_selected_items_in_use"),NULL);
			} else {
				xfer_set_final_error_text(cdata, 1, TRUE,
						_("sfil_ni_selected_items_in_use"),NULL);							  
			}
		} else {
			xfer_set_final_error_text(cdata, 1, FALSE,
						  _("sfil_ni_selected_item_in_use"),NULL);
		}
		return GNOME_VFS_XFER_ERROR_ACTION_SKIP;
		break;
	case GNOME_VFS_ERROR_GENERIC:
	case GNOME_VFS_ERROR_INTERNAL:
	case GNOME_VFS_ERROR_NOT_SUPPORTED:
	case GNOME_VFS_ERROR_WRONG_FORMAT:
	case GNOME_VFS_ERROR_TOO_MANY_OPEN_FILES:
	case GNOME_VFS_ERROR_EOF:
	case GNOME_VFS_ERROR_NOT_A_DIRECTORY:
	case GNOME_VFS_ERROR_IN_PROGRESS:
	case GNOME_VFS_ERROR_INTERRUPTED:
	case GNOME_VFS_ERROR_LOOP:
	case GNOME_VFS_ERROR_NOT_PERMITTED:
	case GNOME_VFS_ERROR_IS_DIRECTORY:
	case GNOME_VFS_ERROR_NO_MEMORY:
	case GNOME_VFS_ERROR_HOST_NOT_FOUND:
	case GNOME_VFS_ERROR_INVALID_HOST_NAME:
	case GNOME_VFS_ERROR_HOST_HAS_NO_ADDRESS:
	case GNOME_VFS_ERROR_LOGIN_FAILED:
	case GNOME_VFS_ERROR_CANCELLED:
	case GNOME_VFS_ERROR_DIRECTORY_BUSY:
	case GNOME_VFS_ERROR_TOO_MANY_LINKS:
	case GNOME_VFS_ERROR_NOT_SAME_FILE_SYSTEM:
	case GNOME_VFS_ERROR_NAME_TOO_LONG:
	case GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE:
	case GNOME_VFS_ERROR_SERVICE_OBSOLETE:
	case GNOME_VFS_ERROR_PROTOCOL_ERROR:
	case GNOME_VFS_ERROR_NO_MASTER_BROWSER:
	case GNOME_VFS_ERROR_NO_DEFAULT:
	case GNOME_VFS_ERROR_NO_HANDLER:
	case GNOME_VFS_ERROR_PARSE:
	case GNOME_VFS_ERROR_LAUNCH:
	case GNOME_VFS_ERROR_TIMEOUT:
	case GNOME_VFS_ERROR_NAMESERVER:
	case GNOME_VFS_ERROR_DEPRECATED_FUNCTION:
	case GNOME_VFS_NUM_ERRORS:
		//TODO what ever may be the error we are displaying some message
		FM_ULOG_INFO("NO INFO MESSAGE TO THESE CASES\n");
		xfer_set_final_error_text(cdata, 1, FALSE,
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_operation_failed"),
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_operation_failed_ok"));
		break;
	case GNOME_VFS_OK:
	default:
		FM_ULOG_ERR("THIS CASE SHOULD NOT ARAISE SHOULD BE ASSERTED");
		//this case should not araise should be asserted
		break;
	}
	DEBUG_PRINT_END;
	return FALSE;
}

/**
 * handle_xfer_rename_errors:
 * Error determination function function
 * @param progress_info
 * @param cdata
 * @return int
 **/
static int handle_xfer_rename_errors(GnomeVFSXferProgressInfo * progress_info,
				     XferCallbackData * cdata)
{
	DEBUG_PRINT_START;
	switch (progress_info->vfs_status) {
	case GNOME_VFS_ERROR_NOT_FOUND:
	case GNOME_VFS_ERROR_BAD_PARAMETERS:
	case GNOME_VFS_ERROR_BAD_FILE:
	case GNOME_VFS_ERROR_INVALID_URI:
		if (selectionstate.no_folders > 0)
			xfer_set_final_error_text(cdata, 1, FALSE,
						  _("sfil_ni_cannot_rename_folder_not_found"),
						  _("sfil_bd_cannot_rename_folder_not_found_ok"));
		else
			xfer_set_final_error_text(cdata, 1, FALSE,
					  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						  "sfil_ni_unable_to_rename_file_not_found"),
					  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						  "sfil_ni_unable_to_rename_file_not_found_ok"));
		break;
	case GNOME_VFS_ERROR_READ_ONLY:
	case GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM:
	case GNOME_VFS_ERROR_ACCESS_DENIED:
		xfer_set_final_error_text(cdata, 0, FALSE,
//					  (gchar *)
//				  osso_common_error(CERM_CAN_NOT_RENAME_RONLY_LOCATION),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"ckdg_ib_unable_to_rename"),
					  NULL);
		break;
	case GNOME_VFS_ERROR_FILE_EXISTS:
		if (selectionstate.no_folders > 0)
			xfer_set_final_error_text(cdata, 0, FALSE,
//						  (gchar *)
//						  osso_common_error(CERM_FOLDER_NAME_EXIST), NULL);
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"ckdg_ib_folder_already_exists")
					,NULL);						  
		else
			xfer_set_final_error_text(cdata, 0, FALSE, 
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
							"ckdg_ib_file_exists"), NULL);
		break;
	case GNOME_VFS_ERROR_LOCKED:
		xfer_set_final_error_text(cdata, 1, FALSE,
					  _("sfil_ni_selected_item_in_use"),NULL);
		break;
	case GNOME_VFS_ERROR_NOT_SUPPORTED:
	case GNOME_VFS_ERROR_DIRECTORY_NOT_EMPTY:
	case GNOME_VFS_ERROR_NOT_OPEN:
	case GNOME_VFS_ERROR_INVALID_OPEN_MODE:
	case GNOME_VFS_ERROR_TOO_BIG:
	case GNOME_VFS_ERROR_NO_SPACE:
	case GNOME_VFS_ERROR_GENERIC:
	case GNOME_VFS_ERROR_INTERNAL:
	case GNOME_VFS_ERROR_IO:
	case GNOME_VFS_ERROR_CORRUPTED_DATA:
	case GNOME_VFS_ERROR_WRONG_FORMAT:
	case GNOME_VFS_ERROR_TOO_MANY_OPEN_FILES:
	case GNOME_VFS_ERROR_EOF:
	case GNOME_VFS_ERROR_NOT_A_DIRECTORY:
	case GNOME_VFS_ERROR_IN_PROGRESS:
	case GNOME_VFS_ERROR_INTERRUPTED:
	case GNOME_VFS_ERROR_LOOP:
	case GNOME_VFS_ERROR_NOT_PERMITTED:
	case GNOME_VFS_ERROR_IS_DIRECTORY:
	case GNOME_VFS_ERROR_NO_MEMORY:
	case GNOME_VFS_ERROR_HOST_NOT_FOUND:
	case GNOME_VFS_ERROR_INVALID_HOST_NAME:
	case GNOME_VFS_ERROR_HOST_HAS_NO_ADDRESS:
	case GNOME_VFS_ERROR_LOGIN_FAILED:
	case GNOME_VFS_ERROR_CANCELLED:
	case GNOME_VFS_ERROR_DIRECTORY_BUSY:
	case GNOME_VFS_ERROR_TOO_MANY_LINKS:
	case GNOME_VFS_ERROR_NOT_SAME_FILE_SYSTEM:
	case GNOME_VFS_ERROR_NAME_TOO_LONG:
	case GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE:
	case GNOME_VFS_ERROR_SERVICE_OBSOLETE:
	case GNOME_VFS_ERROR_PROTOCOL_ERROR:
	case GNOME_VFS_ERROR_NO_MASTER_BROWSER:
	case GNOME_VFS_ERROR_NO_DEFAULT:
	case GNOME_VFS_ERROR_NO_HANDLER:
	case GNOME_VFS_ERROR_PARSE:
	case GNOME_VFS_ERROR_LAUNCH:
	case GNOME_VFS_ERROR_TIMEOUT:
	case GNOME_VFS_ERROR_NAMESERVER:
	case GNOME_VFS_ERROR_DEPRECATED_FUNCTION:
	case GNOME_VFS_NUM_ERRORS:
		//TODO what ever may be the error we are displaying some message
		FM_ULOG_INFO("NO INFO MESSAGE TO THESE CASES\n");
		xfer_set_final_error_text(cdata, 1, FALSE,
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_operation_failed"),
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_operation_failed_ok"));
		break;
	case GNOME_VFS_OK:
	default:
		FM_ULOG_ERR("THIS CASE SHOULD NOT ARAISE SHOULD BE ASSERTED");
		//this case should not araise should be asserted
		break;
	}
	DEBUG_PRINT_END;
	return TRUE;
}

/**
 * handle_xfer_cutpaste_errors:
 * Error determination function function
 * @param progress_info
 * @param cdata
 * @return int
 **/
static int handle_xfer_cutpaste_errors(GnomeVFSXferProgressInfo * progress_info,
				       XferCallbackData * cdata)
{
	DEBUG_PRINT_START;
	switch (progress_info->vfs_status) {
	case GNOME_VFS_ERROR_READ_ONLY:
	case GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM:
	case GNOME_VFS_ERROR_ACCESS_DENIED:
		xfer_set_final_error_text(cdata, 0, FALSE,
//					  (gchar *) osso_common_error(CERM_PATH_RONLY), NULL);
						dgettext(HILDON_FM_L10N_PACKAGE,"sfil_ib_readonly_location"),NULL);						
		FM_ULOG_DEBUG1("Gnome Vfs Error Read Only\n\n");
		break;
	case GNOME_VFS_ERROR_NO_SPACE:
		xfer_set_final_error_text(cdata, 1, FALSE,
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_not_enough_memory"),
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_not_enough_memory_ok"));
		break;
	case GNOME_VFS_ERROR_IO:
		xfer_set_final_error_text(cdata, 1, FALSE,
					  _("sfil_ni_cannot_paste_no_connection"),
					  _("sfil_bd_cannot_paste_no_connection_ok"));
		check_for_partial_file(progress_info,cdata);
		break;
	case GNOME_VFS_ERROR_LOCKED:
		cdata->item_counts.in_use++;
		if (progress_info->files_total > 1) {
			if (cdata->item_counts.in_use < progress_info->files_total ||
			    cdata->item_counts.skipped > 0) {
				xfer_set_final_error_text(cdata, 1, TRUE,
						_("sfil_ni_some_selected_items_in_use"),NULL);
			} else {
				xfer_set_final_error_text(cdata, 1, TRUE,
						_("sfil_ni_selected_items_in_use"),NULL);							  
			}
		} else {
			xfer_set_final_error_text(cdata, 1, FALSE,
						  _("sfil_ni_selected_item_in_use"),NULL);
		}

		/* store the URI to a list so that it can be kept on the
		 * clipboard */
		cdata->locked_files = g_list_append(cdata->locked_files,
                                        g_strdup(progress_info->source_name));
		return GNOME_VFS_XFER_ERROR_ACTION_SKIP;
		break;
	case GNOME_VFS_ERROR_NOT_FOUND:
		cdata->item_counts.non_existant++;
		FM_ULOG_ERR("source file %s was removed\n",
			    progress_info->source_name);
		return GNOME_VFS_XFER_ERROR_ACTION_SKIP;
	case GNOME_VFS_ERROR_FILE_EXISTS:
		break;
	case GNOME_VFS_ERROR_NOT_SUPPORTED:
	case GNOME_VFS_ERROR_BAD_PARAMETERS:
	case GNOME_VFS_ERROR_BAD_FILE:
	case GNOME_VFS_ERROR_INVALID_URI:
	case GNOME_VFS_ERROR_DIRECTORY_NOT_EMPTY:
	case GNOME_VFS_ERROR_NOT_OPEN:
	case GNOME_VFS_ERROR_INVALID_OPEN_MODE:
	case GNOME_VFS_ERROR_TOO_BIG:
	case GNOME_VFS_ERROR_GENERIC:
	case GNOME_VFS_ERROR_INTERNAL:
	case GNOME_VFS_ERROR_CORRUPTED_DATA:
	case GNOME_VFS_ERROR_WRONG_FORMAT:
	case GNOME_VFS_ERROR_TOO_MANY_OPEN_FILES:
	case GNOME_VFS_ERROR_EOF:
	case GNOME_VFS_ERROR_NOT_A_DIRECTORY:
	case GNOME_VFS_ERROR_IN_PROGRESS:
	case GNOME_VFS_ERROR_INTERRUPTED:
	case GNOME_VFS_ERROR_LOOP:
	case GNOME_VFS_ERROR_NOT_PERMITTED:
	case GNOME_VFS_ERROR_IS_DIRECTORY:
	case GNOME_VFS_ERROR_NO_MEMORY:
	case GNOME_VFS_ERROR_HOST_NOT_FOUND:
	case GNOME_VFS_ERROR_INVALID_HOST_NAME:
	case GNOME_VFS_ERROR_HOST_HAS_NO_ADDRESS:
	case GNOME_VFS_ERROR_LOGIN_FAILED:
	case GNOME_VFS_ERROR_CANCELLED:
	case GNOME_VFS_ERROR_DIRECTORY_BUSY:
	case GNOME_VFS_ERROR_TOO_MANY_LINKS:
	case GNOME_VFS_ERROR_NOT_SAME_FILE_SYSTEM:
	case GNOME_VFS_ERROR_NAME_TOO_LONG:
	case GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE:
	case GNOME_VFS_ERROR_SERVICE_OBSOLETE:
	case GNOME_VFS_ERROR_PROTOCOL_ERROR:
	case GNOME_VFS_ERROR_NO_MASTER_BROWSER:
	case GNOME_VFS_ERROR_NO_DEFAULT:
	case GNOME_VFS_ERROR_NO_HANDLER:
	case GNOME_VFS_ERROR_PARSE:
	case GNOME_VFS_ERROR_LAUNCH:
	case GNOME_VFS_ERROR_TIMEOUT:
	case GNOME_VFS_ERROR_NAMESERVER:
	case GNOME_VFS_ERROR_DEPRECATED_FUNCTION:
	case GNOME_VFS_NUM_ERRORS:
		FM_ULOG_ERR("%s\n",
                    gnome_vfs_result_to_string(progress_info->vfs_status));
		xfer_set_final_error_text(cdata, 1, FALSE,
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_operation_failed"),
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_operation_failed_ok"));
		break;
	case GNOME_VFS_OK:
	default:
		FM_ULOG_ERR("THIS CASE SHOULD NOT ARAISE SHOULD BE ASSERTED");
		//this case should not araise should be asserted
		break;
	}
	DEBUG_PRINT_END;
	return FALSE;
}

/**
 * handle_xfer_delete_errors:
 * Error determination function function
 * @param progress_info
 * @param cdata
 * @return int
 **/
static int handle_xfer_delete_errors(GnomeVFSXferProgressInfo * progress_info,
				     XferCallbackData * cdata)
{
	GtkFilePath *path = NULL;
	GtkFilePath *ppath = NULL;
	gchar *parentname = NULL;
	gchar *errstr = NULL;
	gchar *trnstr = NULL;
	DEBUG_PRINT_START;
	switch (progress_info->vfs_status) {
	case GNOME_VFS_ERROR_READ_ONLY:
	case GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM:
	case GNOME_VFS_ERROR_ACCESS_DENIED:
		path = osso_fm_texturi_to_path(progress_info->source_name);
		ppath = osso_fm_get_parent(path);
		parentname = osso_fm_path_get_shortname(ppath);
		trnstr = _("sfil_ib_unable_to_continue_read_only_folder");
		errstr = g_strdup_printf(trnstr,parentname);	
		xfer_set_final_error_text(cdata, 1, TRUE,errstr, NULL);
		g_free(errstr);
		g_free(parentname);
		gtk_file_path_free(ppath);
		gtk_file_path_free(path);
		return 0;
		break;
	case GNOME_VFS_ERROR_LOCKED:
		cdata->item_counts.in_use++;
		if (progress_info->files_total > 1) {
			if (cdata->item_counts.in_use < progress_info->files_total ||
			    cdata->item_counts.skipped > 0) {
				xfer_set_final_error_text(cdata, 1, TRUE,
						_("sfil_ni_some_selected_items_in_use"),NULL);
			} else {
				xfer_set_final_error_text(cdata, 1, TRUE,
						_("sfil_ni_selected_items_in_use"),NULL);							  
			}
		} else {
			xfer_set_final_error_text(cdata, 1, FALSE,
						  _("sfil_ni_selected_item_in_use"),NULL);
		}
		return GNOME_VFS_XFER_ERROR_ACTION_SKIP;
		break;
	case GNOME_VFS_ERROR_NOT_FOUND:
		cdata->item_counts.non_existant++;
		FM_ULOG_ERR("source file %s was removed\n",
			    progress_info->source_name);
		return GNOME_VFS_XFER_ERROR_ACTION_SKIP;
	case GNOME_VFS_ERROR_NO_SPACE:
	case GNOME_VFS_ERROR_NOT_SUPPORTED:
	case GNOME_VFS_ERROR_FILE_EXISTS:
	case GNOME_VFS_ERROR_BAD_PARAMETERS:
	case GNOME_VFS_ERROR_BAD_FILE:
	case GNOME_VFS_ERROR_INVALID_URI:
	case GNOME_VFS_ERROR_DIRECTORY_NOT_EMPTY:
	case GNOME_VFS_ERROR_NOT_OPEN:
	case GNOME_VFS_ERROR_INVALID_OPEN_MODE:
	case GNOME_VFS_ERROR_TOO_BIG:
	case GNOME_VFS_ERROR_GENERIC:
	case GNOME_VFS_ERROR_INTERNAL:
	case GNOME_VFS_ERROR_IO:
	case GNOME_VFS_ERROR_CORRUPTED_DATA:
	case GNOME_VFS_ERROR_WRONG_FORMAT:
	case GNOME_VFS_ERROR_TOO_MANY_OPEN_FILES:
	case GNOME_VFS_ERROR_EOF:
	case GNOME_VFS_ERROR_NOT_A_DIRECTORY:
	case GNOME_VFS_ERROR_IN_PROGRESS:
	case GNOME_VFS_ERROR_INTERRUPTED:
	case GNOME_VFS_ERROR_LOOP:
	case GNOME_VFS_ERROR_NOT_PERMITTED:
	case GNOME_VFS_ERROR_IS_DIRECTORY:
	case GNOME_VFS_ERROR_NO_MEMORY:
	case GNOME_VFS_ERROR_HOST_NOT_FOUND:
	case GNOME_VFS_ERROR_INVALID_HOST_NAME:
	case GNOME_VFS_ERROR_HOST_HAS_NO_ADDRESS:
	case GNOME_VFS_ERROR_LOGIN_FAILED:
	case GNOME_VFS_ERROR_CANCELLED:
	case GNOME_VFS_ERROR_DIRECTORY_BUSY:
	case GNOME_VFS_ERROR_TOO_MANY_LINKS:
	case GNOME_VFS_ERROR_NOT_SAME_FILE_SYSTEM:
	case GNOME_VFS_ERROR_NAME_TOO_LONG:
	case GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE:
	case GNOME_VFS_ERROR_SERVICE_OBSOLETE:
	case GNOME_VFS_ERROR_PROTOCOL_ERROR:
	case GNOME_VFS_ERROR_NO_MASTER_BROWSER:
	case GNOME_VFS_ERROR_NO_DEFAULT:
	case GNOME_VFS_ERROR_NO_HANDLER:
	case GNOME_VFS_ERROR_PARSE:
	case GNOME_VFS_ERROR_LAUNCH:
	case GNOME_VFS_ERROR_TIMEOUT:
	case GNOME_VFS_ERROR_NAMESERVER:
	case GNOME_VFS_ERROR_DEPRECATED_FUNCTION:
	case GNOME_VFS_NUM_ERRORS:
		FM_ULOG_INFO("NO INFO MESSAGE TO THESE CASES\n");
		xfer_set_final_error_text(cdata, 1, FALSE,
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_operation_failed"),
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_operation_failed_ok"));
		break;
	case GNOME_VFS_OK:
	default:
		FM_ULOG_ERR("THIS CASE SHOULD NOT ARAISE SHOULD BE ASSERTED");
		break;
	}
	DEBUG_PRINT_END;
	return FALSE;
}

/**
 * handle_xfer_newfolder_errors:
 * Error determination function function
 * @param progress_info
 * @param cdata
 * @return int
 **/
static int
handle_xfer_newfolder_errors(GnomeVFSXferProgressInfo * progress_info, XferCallbackData * cdata)
{
	DEBUG_PRINT_START;
	switch (progress_info->vfs_status) {
	case GNOME_VFS_ERROR_READ_ONLY:
	case GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM:
	case GNOME_VFS_ERROR_ACCESS_DENIED:
		xfer_set_final_error_text(cdata, 0, FALSE,
//					  (gchar *) osso_common_error(CERM_NEWFOLDER_NOT_ALLOWED),
					dgettext(HILDON_FM_L10N_PACKAGE,"sfil_ib_create_folder_not_allowed"),					  					  NULL);
		break;
	case GNOME_VFS_ERROR_FILE_EXISTS:
		xfer_set_final_error_text(cdata, 0, FALSE,
//					  (gchar *) osso_common_error(CERM_FOLDER_NAME_EXIST),
						dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"ckdg_ib_folder_already_exists"),					  
					  NULL);
		break;
	case GNOME_VFS_ERROR_NO_SPACE:
	case GNOME_VFS_ERROR_NOT_SUPPORTED:
	case GNOME_VFS_ERROR_NOT_FOUND:
	case GNOME_VFS_ERROR_BAD_PARAMETERS:
	case GNOME_VFS_ERROR_BAD_FILE:
	case GNOME_VFS_ERROR_INVALID_URI:
	case GNOME_VFS_ERROR_DIRECTORY_NOT_EMPTY:
	case GNOME_VFS_ERROR_NOT_OPEN:
	case GNOME_VFS_ERROR_INVALID_OPEN_MODE:
	case GNOME_VFS_ERROR_TOO_BIG:
	case GNOME_VFS_ERROR_GENERIC:
	case GNOME_VFS_ERROR_INTERNAL:
	case GNOME_VFS_ERROR_IO:
	case GNOME_VFS_ERROR_CORRUPTED_DATA:
	case GNOME_VFS_ERROR_WRONG_FORMAT:
	case GNOME_VFS_ERROR_TOO_MANY_OPEN_FILES:
	case GNOME_VFS_ERROR_EOF:
	case GNOME_VFS_ERROR_NOT_A_DIRECTORY:
	case GNOME_VFS_ERROR_IN_PROGRESS:
	case GNOME_VFS_ERROR_INTERRUPTED:
	case GNOME_VFS_ERROR_LOOP:
	case GNOME_VFS_ERROR_NOT_PERMITTED:
	case GNOME_VFS_ERROR_IS_DIRECTORY:
	case GNOME_VFS_ERROR_NO_MEMORY:
	case GNOME_VFS_ERROR_HOST_NOT_FOUND:
	case GNOME_VFS_ERROR_INVALID_HOST_NAME:
	case GNOME_VFS_ERROR_HOST_HAS_NO_ADDRESS:
	case GNOME_VFS_ERROR_LOGIN_FAILED:
	case GNOME_VFS_ERROR_CANCELLED:
	case GNOME_VFS_ERROR_DIRECTORY_BUSY:
	case GNOME_VFS_ERROR_TOO_MANY_LINKS:
	case GNOME_VFS_ERROR_NOT_SAME_FILE_SYSTEM:
	case GNOME_VFS_ERROR_NAME_TOO_LONG:
	case GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE:
	case GNOME_VFS_ERROR_SERVICE_OBSOLETE:
	case GNOME_VFS_ERROR_PROTOCOL_ERROR:
	case GNOME_VFS_ERROR_NO_MASTER_BROWSER:
	case GNOME_VFS_ERROR_NO_DEFAULT:
	case GNOME_VFS_ERROR_NO_HANDLER:
	case GNOME_VFS_ERROR_PARSE:
	case GNOME_VFS_ERROR_LAUNCH:
	case GNOME_VFS_ERROR_TIMEOUT:
	case GNOME_VFS_ERROR_NAMESERVER:
	case GNOME_VFS_ERROR_LOCKED:
	case GNOME_VFS_ERROR_DEPRECATED_FUNCTION:
	case GNOME_VFS_NUM_ERRORS:
		//TODO what ever may be the error we are displaying some message
		FM_ULOG_INFO("NO INFO MESSAGE TO THESE CASES\n");
		xfer_set_final_error_text(cdata, 1, FALSE,
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_operation_failed"),
			  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ni_operation_failed_ok"));
		break;
	case GNOME_VFS_OK:
	default:
		FM_ULOG_ERR("THIS CASE SHOULD NOT ARAISE SHOULD BE ASSERTED");
		//this case should not araise should be asserted
		break;
	}
	DEBUG_PRINT_END;
	return TRUE;
}

/**
 * handle_progress_vfs_error:
 * Error handling function
 * @param progress_info
 * @param cdata
 * @return int
 **/
static int
handle_progress_vfs_error(GnomeVFSXferProgressInfo * progress_info, XferCallbackData * cdata)
{

	//char *text = NULL;
	//char *detail = NULL;

	DEBUG_PRINT_START;
	FM_ULOG_ERR("PROGRESS ERROR ::: [%d][%s]\n", progress_info->vfs_status,
		    gnome_vfs_result_to_string(progress_info->vfs_status));
	FM_ULOG_ERR("PHASE OF OPERATION %d\n", progress_info->phase);
	cdata->item_counts.errors++;
	switch (cdata->action) {
	case XFER_DUPLICATE:
		DEBUG_PRINT;
		handle_xfer_duplicate_errors(progress_info, cdata);
		break;
	case XFER_COPY:
		handle_xfer_copy_errors(progress_info, cdata);
		DEBUG_PRINT;
		break;
	case XFER_MOVE:
		return handle_xfer_move_errors(progress_info, cdata);
		DEBUG_PRINT;
		break;
	case XFER_RENAME:
		handle_xfer_rename_errors(progress_info, cdata);
		DEBUG_PRINT;
		break;
	case XFER_CUTPASTE:
		return handle_xfer_cutpaste_errors(progress_info, cdata);
		DEBUG_PRINT;
		break;
	case XFER_DELETE:
		return handle_xfer_delete_errors(progress_info, cdata);
		DEBUG_PRINT;
		break;
	case XFER_NEWFOLDER:
		handle_xfer_newfolder_errors(progress_info, cdata);
		DEBUG_PRINT;
		break;
	}

	DEBUG_PRINT_END;
	return FALSE;		//TODO

}




/**
 * handle_progress_overwrite:
 * Handles Overwrites by replacing. 
 * 
 * @param progress_info	Progress info 
 * @param cdata		Pointer to callback data
 * 
 * @return 		int
 */
static int
handle_progress_overwrite(GnomeVFSXferProgressInfo * progress_info, XferCallbackData * cdata)
{
	gchar *text = NULL;
	gchar *formatted_name = NULL;
	gchar *button1_text = NULL;
	gchar *button2_text = NULL;
	gboolean target_is_dir = FALSE;
	gboolean source_is_dir = FALSE;
	GtkWidget *xfer_query_dialog = NULL;
	gint ret = 0;
	GnomeVFSURI *uri = NULL;
	GList *del_uri_list = NULL;
	//HildonFileSelection *fs = NULL;
	//HildonFileSystemModel *model = NULL;

	DEBUG_PRINT_START;
	FM_ULOG_DEBUG1("QUERY CALLBACK ::: [%d][%s]\n", progress_info->vfs_status,
		     gnome_vfs_result_to_string(progress_info->vfs_status));

	formatted_name = osso_fm_texturi_get_shortname(progress_info->target_name);
	target_is_dir = osso_fm_is_texturi_directory(progress_info->target_name);
	source_is_dir = osso_fm_is_texturi_directory(progress_info->source_name);

	switch (cdata->action) {
	case XFER_COPY:
	case XFER_MOVE:
	case XFER_CUTPASTE:
		if (cdata->item_counts.existing > 1) {
			if (cdata->replace_query_status == -1) {
				text = g_strdup_printf(dgettext(HILDON_FM_L10N_PACKAGE,"docm_nc_replace_multiple"));

				button1_text = g_strdup(dgettext(HILDON_FM_L10N_PACKAGE,"docm_bd_replace_multiple_ok"));
				button2_text = g_strdup(dgettext(HILDON_FM_L10N_PACKAGE,"docm_bd_replace_multiple_cancel"));
			} else if (cdata->replace_query_status == 0) {
				DEBUG_PRINT_END;
				return GNOME_VFS_XFER_OVERWRITE_ACTION_ABORT;
			} else if (cdata->replace_query_status == 1) {
				DEBUG_PRINT_END;
				return GNOME_VFS_XFER_OVERWRITE_ACTION_REPLACE;
			}

		} else {
			if (target_is_dir) {
				text = g_strdup_printf("%s\n%s",
						       _("docm_nc_replace_folder"), formatted_name);
				button1_text = g_strdup(_("docm_bd_replace_folder_ok"));
				button2_text = g_strdup(_("docm_bd_replace_folder_cancel"));
			} else {
				text = g_strdup_printf("%s\n%s",
						       dgettext(HILDON_FM_L10N_PACKAGE,"docm_nc_replace_file"), formatted_name);
				button1_text = g_strdup(dgettext(HILDON_FM_L10N_PACKAGE,"docm_bd_replace_file_ok"));
				button2_text = g_strdup(dgettext(HILDON_FM_L10N_PACKAGE,"docm_bd_replace_file_cancel"));
			}
		}
		break;
	case XFER_DUPLICATE:
	case XFER_RENAME:
	case XFER_DELETE:
	case XFER_NEWFOLDER:
		break;
	default:
		break;
		//TODO case should not come here should be asserted

	}
	if (text != NULL && button1_text != NULL) {
		xfer_query_dialog = (GtkWidget *)
		    hildon_note_new_confirmation_add_buttons((GtkWindow *) cdata->window, text,
							     button1_text, GTK_RESPONSE_YES,
							     button2_text, GTK_RESPONSE_NO, NULL);

		ret = gtk_dialog_run(GTK_DIALOG(xfer_query_dialog));
		FM_ULOG_DEBUG1("RETURN FROM DIALOG %d\n", ret);
		gtk_widget_destroy((GtkWidget *) xfer_query_dialog);
	}

	if (formatted_name != NULL)
		g_free(formatted_name);
	if (text != NULL)
		g_free(text);
	if (button1_text != NULL)
		g_free(button1_text);
	if (button2_text != NULL)
		g_free(button2_text);

	if (ret == GTK_RESPONSE_YES) {
		if (target_is_dir && source_is_dir) {
			uri = gnome_vfs_uri_new(progress_info->target_name);
			del_uri_list = g_list_append(del_uri_list, uri);
			/*result = */ gnome_vfs_xfer_delete_list( del_uri_list,
							GNOME_VFS_XFER_ERROR_MODE_ABORT,
							GNOME_VFS_XFER_DELETE_ITEMS
							| GNOME_VFS_XFER_RECURSIVE,
							NULL, NULL);
			gnome_vfs_uri_unref(uri);
			g_list_free(del_uri_list);
		}
		cdata->replace_query_status = 1;
		DEBUG_PRINT_END;
		return GNOME_VFS_XFER_OVERWRITE_ACTION_REPLACE;
	} else {
		cdata->replace_query_status = 0;
		DEBUG_PRINT_END;
		return GNOME_VFS_XFER_OVERWRITE_ACTION_ABORT;
	}
	DEBUG_PRINT_END;
	return 0;
}

/**
 * handle_progress_duplicate:
 * Handles duplication
 * 
 * @param progress_info	Progress info 
 * @param cdata		Pointer to callback data
 * 
 * @return 		int
 */
static int
handle_progress_duplicate(GnomeVFSXferProgressInfo * progress_info, XferCallbackData * cdata)
{
	DEBUG_PRINT_START;
	switch (cdata->action) {
	case XFER_COPY:
		//
		//TODO progress_info->duplicate_name = auto_rename 
		//  (progress_info->duplicate_name,
		//   progress_info->duplicate_count);
		break;
	default:
		break;
	}

	DEBUG_PRINT_END;
	return GNOME_VFS_XFER_ERROR_ACTION_SKIP;
}
