/**
 * @file  osso_fm_utils.h
 *
 * This file contains the utility functions for File Manager application.
 *
 * Copyright (C) 2005 Nokia Corporation
 */


#ifndef __UTILS_H__
#define __UTILS_H__

#include <stdio.h>
#include "osso_fm_fmstate.h"

#include <libgnomevfs/gnome-vfs.h>
// TODO : to be changed to HILDON specific return type 
#define FM_FILE_SYSTEM_MODEL_SAFE_FOLDER_GAMES 20

#define THUMBNAIL_PARSE 1<<6
#define ASCENDING_PARSE 1<<5

#define CLIPBOARD_FILES "x-hildon/files_copied"
#define PHONE_CONNECTION_DIALOG_STATUS_GROUP "PHONE_CONNECTION_DIALOG_GROUP"
#define PHONE_CONNECTION_DIALOG_STATUS "PHONE_CONNECTION_DIALOG_STATUS"

#define TOOLBAR_STATUS "TOOLBAR_STATUS_GROUP"
#define NORMAL_VIEW_TOOLBAR "NORMAL_VIEW_TOOLBAR"
#define FULLSCREEN_TOOLBAR "FULLSCREEN_TOOLBAR"


typedef enum {
	XFER_COPY,
	XFER_MOVE,
	XFER_DELETE,
	XFER_DUPLICATE,
	XFER_NEWFOLDER,
	XFER_CUTPASTE,
	XFER_RENAME
} TransferAction;

typedef enum {
	ERROR_READ_ONLY,
	ERROR_NOT_READABLE,
	ERROR_NOT_WRITABLE,
	ERROR_NOT_ENOUGH_PERMISSIONS,
	ERROR_NO_SPACE,
	ERROR_SOURCE_IN_TARGET,
	ERROR_OTHER
} FMFileOperationsErrorKind;

typedef enum {
	ERROR_LOCATION_UNKNOWN,
	ERROR_LOCATION_SOURCE,
	ERROR_LOCATION_SOURCE_PARENT,
	ERROR_LOCATION_SOURCE_OR_PARENT,
	ERROR_LOCATION_TARGET
} FMFileOperationsErrorLocation;

typedef enum {
	OSSO_FM_INSUFFICIENT_MEMORY = -1,
	OSSO_FM_MEMORY_ALLOCATION_FAILURE = -2
} OSSOFMErrors;

extern gchar *homedir;

gchar *osso_fm_path_to_texturi(const GtkFilePath *path);

GtkFilePath *osso_fm_texturi_to_path(const gchar *uri);

GnomeVFSURI *osso_fm_path_to_uri(const GtkFilePath *path);

GtkFilePath *osso_fm_uri_to_path(const GnomeVFSURI *uri);

GtkFilePath *osso_fm_make_path(const GtkFilePath *path,const gchar *name);

GtkFilePath *osso_fm_get_parent(const GtkFilePath *path);

gchar *osso_fm_texturi_get_parent(const gchar *path_texturi);

/*! Method to autorename a file / folder path if the specified path exists..
\param reference of file path. The path is modified to the new name as per 
the specification.
\return void
*/
void autorename(GtkFilePath **filepath, HildonFileSystemModel * model);

void autorename_texturi(gchar **filepath, HildonFileSystemModel * model);

/*! Method to check if the path specified is readonly.
*param path path of file/folder
*param checkparent checks for parent path if TRUE.
*return TRUE if path is readonly
*/
gboolean is_path_readonly(const GtkFilePath * path, const gboolean checkparent);

/**
* Method to check if the path_texturi specified is readonly.
* @param path of file/folder
* @param checkparent checks for parent path if TRUE.
* @return gboolean
*/
gboolean is_texturi_readonly(const gchar * path_texturi, const gboolean checkparent);

/*! Make GList source and target paths.
\param spurceselection source list as GSList
\param sourcelist GList of source slection
\param targetpath target path to make the target list
\param targetlist GList of target paths. For each path in source list, 
an item in targetlist is appended.
\return void
*/
/*gboolean 
make_uri_list (const GSList *sourceselection, GList **sourcelist, 
		const GtkFilePath *targetpath, GList **targetlist,
		GtkWidget *window);
*/
void
make_uri_list(const GSList * sourceselection, GList ** sourcelist,
	      const GtkFilePath * targetpath, GList ** targetlist,
	      HildonFileSystemModel * model, gboolean canautorename);

/*! Method to check if the path specified exist.
\param path path of file/folder
\return true if file exist 
*/
gboolean is_path_exists(const gchar * path);


/*! Method to get the type of the specified path
 * \param path Path of file/folder
 * \return File/Folder type.
 */
gint osso_fm_get_path_type(const gchar * path);


/**
 * This method returns the display name for the current selection.
 *
 * @param fs HildonFileSelection
 *
 * @return gchar* representing the display name for the current path
 */
gchar *osso_fm_path_get_display_name(const GtkFilePath * path);

gchar *osso_fm_texturi_get_display_name(const gchar * path_texturi);


/*! Method to extract short name of the entire file path specified.
 * \param path Path of file/folder.
 * \param shortname Short name of the file/folder specified in param 1.
 */
gchar *osso_fm_path_get_shortname(const GtkFilePath * path);

gchar *osso_fm_texturi_get_shortname(const gchar* path_texturi);


/*! Method to extract the parent directory path ofthe file path specified.
 * \param path path of file/folder.
 * \param dirname directory path of the full path specified in param 1.
 */
gchar *osso_fm_path_get_dirname(const gchar * path);


/*! Method to extract the extension from the file name specified.
 * \param path File name from which extension to be extracted.
 * \param extension Extension of the file name specified.
 */
gchar *osso_fm_path_get_extension(const gchar * path);



/**
* Method to whether a texturi is that of a directory
* 
* @param uri is uri of an item
* 
* @return gboolean
*/
gboolean osso_fm_is_path_directory(const GtkFilePath *path);


/**
* Method to whether a texturi is that of a directory
* 
* @param uri is uri of an item
* 
* @return gboolean
*/
gboolean osso_fm_is_texturi_directory(const char *path_texturi);


/**
 * Method checks the path is located at home with the specified shortname
 *
 * @param path path to be verified
 * @param shortname the name against which the path is to be checked.
 *
 * @return TRUE if the shortname matches with path
 * shortname and location is home
 */
gboolean is_path_from_home(const gchar * path, const gchar * shortname);


/**
 * Checks if the given uri is in Gateway
 *
 * @param fs HildonFileSelection Widget
 *
 * @return TRUE if uri is in gateway else FALSE
 */
gboolean is_uri_in_gateway(const GnomeVFSURI* path_uri);

/**
 * Checks if the given path is in Gateway
 *
 * @param fs HildonFileSelection Widget
 *
 * @return TRUE if path is in gateway else FALSE
 */
gboolean is_path_in_gateway(const GtkFilePath * path);

gboolean is_texturi_in_gateway(const gchar * path_texturi);


/**
 * Check if the selected location is in Gateway
 *
 * @param fs HildonFileSelection Widget
 *
 * @return TRUE if path is in MMC else FALSE
 */
gboolean selected_location_in_gateway(HildonFileSelection * fileselection);

/**
 * Check if the selected location is in MMC
 *
 * @param fs HildonFileSelection Widget
 *
 * @return TRUE if path is in MMC else FALSE
 */
gboolean selected_location_in_mmc(HildonFileSelection * fs);

gboolean selected_location_in_internal_mmc(HildonFileSelection * fs);


gboolean is_texturi_in_media_server(gchar *path_texturi);

gboolean is_path_in_media_server(GtkFilePath *path);

gboolean selected_location_in_media_server(HildonFileSelection * fileselection);

/*! Method to check for the path to be a safe folder or device folder.
 * \param path File/folder path to be checked
 * \return HildonFileSystemModelItemType
 */
//HildonFileSystemModelItemType //TODO: wrong hack ... 
//to be change the return type to HildonFileSystemModelType

HildonFileSystemModelItemType osso_fm_fs_get_path_type(
		HildonFileSelection *fs,const GtkFilePath *path);



/**
 * is_uri_in_mmc:
 * Check if the given uri is in MMC
 * @param uri GnomeVFSURI
 * @return TRUE if path is in MMC else FALSE
 */
gboolean is_uri_in_mmc(GnomeVFSURI * uri, gboolean *internal);

/**
 * is_path_in_mmc:
 * Check if the given path is in MMC
 * @param path GtkFilePath
 * @return TRUE if path is in MMC else FALSE
 */
gboolean is_path_in_mmc(GtkFilePath* path, gboolean *internal);

/**
 * is_texturi_in_mmc:
 * Check if the given gchar texturi is in MMC
 * @param path_texturi gchar *
 * @return TRUE if path is in MMC else FALSE
 */
gboolean is_texturi_in_mmc(gchar* path_texturi, gboolean *internal);



/**
 * Method will return the MD5 digest for the path specified
 *
 * @param pathuri pathuri representing the absolute path of a file/folder
 *
 * @return gchar* MD5 digest of the path specified.
 */
//gchar*
//osso_fm_get_MD5_digest (const gchar *pathuri);


/**
 * Method to initialize the key file for persistence of folder settings
 */
void osso_fm_key_file_init();

/**
* Method to read gconf value for the connection dialog.
* @return 1 if wizad has to be displayed
* 0 if it should not be displayed.
*/
gint osso_fm_get_gateway_phone_connection_status();

/**
* Method to read gconf value for the connection dialog.
* @param status status of phone connection dialog
* @return void.
*/
void osso_fm_set_gateway_phone_connection_status(gint status);


/**
* Method to read gconf value for the toolbar status.
* @return 1 successful 
* 0 failed.
*/
gboolean osso_fm_get_toolbar_status(gint * normal, gint * fullscreen);

/**
* Method to read gconf value for the connection dialog.
* @param status status of phone connection dialog
* @return void.
*/
void osso_fm_set_toolbar_status(gint normal, gint fullscreen);

/**
 * Method to read gconf value for the path specified.
 *
 * @param path gchar* representing the uri
 * @param showthumbnails TRUE if thumbnails view
 * @param key represents the column on which sort is to be performed.
 * @param order sorting type ascending / descending
 *
 * @return 0 on success.
 */

gint
osso_fm_get_config_value_for_path(const GtkFilePath * pathuri,
				  gboolean * showthumbnails,
				  HildonFileSelectionSortKey * key, GtkSortType * order);

gint
osso_fm_get_config_value_for_texturi(const gchar * path_texturi,
				  gboolean * showthumbnails,
				  HildonFileSelectionSortKey * key, GtkSortType * order);

/**
 * Method to write gconf value for the path specified.
 *
 * @param path gchar* representing the text_uri
 * @param showthumbnails TRUE if thumbnails view
 * @param key represents the column on which sort is to be performed.
 * @param order sorting type ascending / descending
 *
 * @return 0 on success
 */
gint
osso_fm_set_config_value_for_texturi(const gchar * path_texturi,
				  const gboolean showthumbnails,
				  const HildonFileSelectionSortKey key, const GtkSortType order);
/**
 * Method to write gconf value for the path specified.
 *
 * @param path GtkFilePath
 * @param showthumbnails TRUE if thumbnails view
 * @param key represents the column on which sort is to be performed.
 * @param order sorting type ascending / descending
 *
 * @return 0 on success
 */
gint
osso_fm_set_config_value_for_path(const GtkFilePath * pathuri,
				  const gboolean showthumbnails,
				  const HildonFileSelectionSortKey key, const GtkSortType order);


/**
 * Method to save the preferences file
 */
void osso_fm_save_key_file();
void osso_fm_close_key_file();


void osso_fm_clipboard_init();


void osso_fm_copy_to_clipboard(GtkWidget * fs, GSList * selection, gboolean cut);
void osso_fm_copy_glist_to_clipboard(GtkWidget *app, GList *list, gboolean cut);


void osso_fm_get_list_from_clipboard(GtkWidget * fs, GSList ** selection, gboolean * cut);

void osso_fm_clipboard_clear();

gint
remove_safe_folders_from_gslist(HildonFileSelection * selection,
				GSList ** sourceselection, gint *nonexistant, gboolean menucheck);

/**
   To Select The Newly Created Folder using g_timeout_add .
   @param data  It is a Single LinkList, where the First  List Contains
   Address of HildonFileSelection & second List contains the address of
   newly created path.
   @return gboolean
*/
gboolean on_select_path_timer(gpointer data);

//vivek
void fm_infoprint(const GtkWindow * w, const gchar *data);

void fm_infonote(const GtkWindow * window, const gchar *text,const gchar *but_text);
gint fm_confnote(const GtkWindow * window, const gchar *title, const gchar *text,
		const gchar *but1_text,const gint but1_res,
		const gchar *but2_text,const gint but2_res,const gboolean sensitive);

void register_selection_change_sensitive_dialog(OSSOFileManager *ofm,GtkWidget *dialog);

void unregister_selection_change_sensitive_dialog(OSSOFileManager *ofm,GtkWidget *dialog);

void close_selection_change_sensitivity_open_dialogs(OSSOFileManager *ofm);


void update_thumbnails(GList * src_list, GList * tgt_list, TransferAction xfer_action);


/**
 * This is a helper function to show a confirmation note
 * @param text 	Text to be displayed in the confirmmation note
 * @parma button_text Label to be shown on the button
 * @return void
**/
void osso_fm_display_confirmation_note(gchar * text, gchar * button_text);


/**
 * The function returns rename dialog title depending on the file type
 *
 * @param oldpath original path
 *
 * @return title for rename dialog
 */
gchar *get_rename_file_title(const gchar * oldpath);


/**
 * The function returns the already existing file error message.
 * 
 * @param newname New path to verify
 *
 * @return Info message to be displayed.
 */
gchar *get_rename_file_existing_error(const gchar * newname);

/** This function checks the given list of files/folders for 
* existance and returns the count 
* of existing files/folders
* @param list		Pointer to the list of files/folders
* @return gint		No of existing files/folders
**/

gint osso_fm_get_existing_file_count(GList * list);

/** This function checks the given list of files/folders for 
* existance and returns the count 
* of existing files/folders
* @param list		Pointer to the list of files/folders
* @return gint		No of existing files/folders
**/

gint osso_fm_get_existing_file_count_from_gslist(GSList * list, gint min_req);

gboolean valid_mmc_name (gchar * mmc_name, gchar *illegals, gint max_illegals);

//extern gint mmc_umount_timer_status;
//void start_mmc_umount_delay_timer();
//void stop_mmc_umount_delay_timer();

glong timeval_diff(GTimeVal * end, GTimeVal * start);

void osso_fm_select_texturi_with_time(const gchar * text_uri, gboolean select);

void osso_fm_select_path_with_time(const GtkFilePath * path, gboolean select);

gboolean is_device_in_flight_mode();

void osso_fm_copy_uri_view_settings(GnomeVFSURI * src_uri, GnomeVFSURI * dst_uri);

void osso_fm_copy_texturi_view_settings(const gchar * dst, const gchar * src);

gint osso_fm_predict_dialog_width_based_on_title(GtkWidget * dialog, const gchar * title);

gint osso_fm_predict_progress_dialog_width_based_on_text(GtkWidget * dialog, const gchar * title);

void osso_fm_set_operation_state(MenuOperation oper);

void osso_fm_free_gnomevfs_uri_list(GList * list);
void osso_fm_free_string_list(GList * list);
void osso_fm_free_string_slist(GSList * list);


//BEG: Validation functions
/**
 * Initalize the selection state structure
 */
void init_selection_state();

/**
 * Prepares the SelectionType as per the selection state structure
 */
SelectionType get_selection_type();

SelectionState *get_selection_state_for_selection(GSList * selection);

/**
 Validates the selection in file selection widget.
 @param fs Reference of file selection widget.
 @return selection state
*/
SelectionType validate_selection(HildonFileSelection * fs);

/**!Updates the popup menu items as per the selection
 * @param app Reference of application.
 * @return void
 */
void refresh_popup_menu_items(OSSOFileManager *ofm, gboolean contentpane);

void refresh_file_menu_items(GtkMenuItem * w, gpointer data);

void refresh_edit_menu_items(GtkMenuItem * w, gpointer data);

void refresh_toolbar_buttons(gpointer data);

void refresh_view_menu_items(GtkMenuItem * w, gpointer data);

void refresh_edit_menu_select_items(GtkWidget * w, gpointer data);
gboolean is_clipboard_contents_valid(GSList *sourceselection);

//END: Validation functions

typedef gboolean (*OSSOFMDirVisitCallback) (GnomeVFSFileInfo *info,
						gboolean *recursive,
						gpointer user_data);

typedef struct {
	GnomeVFSDirectoryHandle *dir_handle;
	GtkFilePath *path;
}DirVisitHandle;

typedef struct {
	GQueue *handle_stack; //stack of DirVisitHandle
	OSSOFMDirVisitCallback callback;
	gpointer callback_data;
}OSSOFMDirVisitHandle; 

OSSOFMDirVisitHandle * osso_fm_dir_visit_init(GtkFilePath *path,
					OSSOFMDirVisitCallback callback,
					gpointer callback_data);

gboolean osso_fm_dir_visit_fetch_next(OSSOFMDirVisitHandle *self,gint no_of_items);

void osso_fm_dir_visit_close(OSSOFMDirVisitHandle *self);

void osso_fm_unknown_file_open(OSSOFileManager *ofm,GSList *filelist);

gboolean osso_fm_path_type_mmc(HildonFileSystemModelItemType type);

gboolean check_for_media_items(MenuOperation oper,OSSOFileManager *ofm, gboolean menucheck);

gboolean is_path_virtual(GtkFilePath *path);
gboolean is_texturi_virtual(gchar *path_texturi);
gboolean is_uri_virtual(GnomeVFSURI *path_uri);
gboolean is_xfer_in_progress();
gboolean is_path_readable(const gchar * path);
#endif
