/**
 * @file  osso_fm_utils.c
 *
 * This file contains implementation for utility functions for File Manager application.
 *
 * Copyright (C) 2005 Nokia Corporation
 */


#include <hildon/hildon-note.h>
#include <libosso.h>

#include "osso_fm_generic.h"
#include "osso_fm_controller.h"
#include "osso_fm_utils.h"
#include "osso_fm_fmstate.h"
#include "osso_fm_callbacks.h"
#include "osso_fm_operations.h"
#include "osso_fm_details.h"
#include <hildon-mime.h>
#include <dbus/dbus.h>

#include "mce/dbus-names.h"
#include "mce/mode-names.h"
#define BUFFSIZE 8192

static const GtkTargetEntry clipboard_targets[] = {
	{CLIPBOARD_FILES, 0, 0},
};

static GdkAtom copied_files_atom;

/**
* This is Global variable used to store the path of HOME DIRECTORY
* and is initializing in function 'osso_fm_key_file_init'
*/
gchar *homedir = NULL;

static void refresh_edit_menu_paste_items(GtkWidget * widget, gpointer data);
static gboolean refresh_timer_toolbar_buttons(gpointer data);
static SelectionType update_current_selection_state(HildonFileSelection * fileselection);

/**
 * osso_fm_path_to_texturi:
 * This is a helper function which converts the GtkFilePath to the texturi
 * @param path a pointer to GtkFilePath
 * @return gchar * newly allocated texturi or NULL if something fails
 **/ 
gchar *osso_fm_path_to_texturi(const GtkFilePath *path)
{
	return gtk_file_system_path_to_uri(global_ofm->gtkfilesystem,path);
}

/**
 * osso_fm_texturi_to_path:
 * This is a helper function which converts the texturi to GtkFilePath  
 * @param uri a gchar pointer to text uri to be converted
 * @return GtkFilePath* newly allocated GtkFilePath or NULL if something fails
 **/ 
GtkFilePath *osso_fm_texturi_to_path(const gchar *uri)
{
	return gtk_file_system_uri_to_path(global_ofm->gtkfilesystem,uri);
}

/**
 * osso_fm_path_to_uri:
 * This is a helper function which converts the GtkFilePath to the GnomeVFSURI
 * @param path a pointer to GtkFilePath
 * @return GnomeVFSURI * newly allocated GnomeVFSURI or NULL if something fails
 **/ 
GnomeVFSURI *osso_fm_path_to_uri(const GtkFilePath *path)
{
	gchar *texturi = NULL;
	GnomeVFSURI *uri = NULL;
	texturi =  osso_fm_path_to_texturi(path);
	if(texturi != NULL)
	{
		uri = gnome_vfs_uri_new(texturi);
		g_free(texturi);
	}
	return uri;
}

/**
 * osso_fm_uri_to_path:
 * This is a helper function which converts the GnomeVFSURI to GtkFilePath
 * @param path a pointer to GnomeVFSURI
 * @return GtkFilePath * newly allocated GtkFilePath or NULL if something fails
 **/ 
GtkFilePath *osso_fm_uri_to_path(const GnomeVFSURI *uri)
{
	gchar *texturi = NULL;
	GtkFilePath *path = NULL;
	texturi = gnome_vfs_uri_to_string(uri,GNOME_VFS_URI_HIDE_NONE);
	if(texturi != NULL)
	{
		path = osso_fm_texturi_to_path(texturi);
		g_free(texturi);
	}
	return path;
}
/**
 * osso_fm_make_path:
 * This function creates a new path which points to the file/folder <name> in the directory 
 * given by <path> (prefix path to file name)
 * @param path a pointer to GtkFilePath
 * @param name a pointer to the name of the file/folder
 * @return gchar * newly allocated GtkFilePath or NULL if something fails
 **/ 
GtkFilePath *osso_fm_make_path(const GtkFilePath *path,const gchar *name)
{
	return gtk_file_system_make_path(global_ofm->gtkfilesystem,path,name,NULL);
}

/**
 * osso_fm_get_parent:
 * This function get the parent folder of the given path by calling the gtk_file_system_get_parent
 * @param path a pointer to GtkFilePath
 * @return GtkFilePath * parent folder of the <path>
 **/
GtkFilePath *osso_fm_get_parent(const GtkFilePath *path)
{
	GtkFilePath *parent_path = NULL;
	gtk_file_system_get_parent(global_ofm->gtkfilesystem,path,&parent_path,NULL);
	return parent_path;
}

/**
 * osso_fm_texturi_get_parent:
 * This function get the parent folder of the given path by calling the gtk_file_system_get_parent
 * @param path a gchar pointer 
 * @return gchar* parent folder of the <path>
 **/
gchar *osso_fm_texturi_get_parent(const gchar *path_texturi)
{
	GtkFilePath *path = NULL;
	GtkFilePath *parent_path = NULL;
	gchar *parent_path_texturi = NULL;
	path=osso_fm_texturi_to_path(path_texturi);
	parent_path = osso_fm_get_parent(path);
	parent_path_texturi = osso_fm_path_to_texturi(parent_path);
	gtk_file_path_free(path);
	gtk_file_path_free(parent_path);
	return parent_path_texturi;
}

/**
 * autorename:
 * same as autorename_texturi but takes GtkFilePath
 * @param filepath a double pointer to GtkFilePath (in/out param)
 * @param model a pointer to HildonFileSystemModel
 * @return void
*/
void autorename(GtkFilePath **filepath, HildonFileSystemModel * model)
{
	gchar *filepath_texturi = NULL;

	DEBUG_PRINT_START;
	if ((*filepath == NULL) || (model == NULL)) {
		FM_ULOG_ERR("Error validating input parameters\n");
		return;
	}

	filepath_texturi = osso_fm_path_to_texturi(*filepath);
	autorename_texturi(&filepath_texturi,model);
	
	gtk_file_path_free(*filepath);
	*filepath = osso_fm_texturi_to_path(filepath_texturi);
	g_free(filepath_texturi);

	DEBUG_PRINT_END;
	return;
}

/**
 * autorename_texturi:
 * Method to autorename a file / folder path if the specified path already exists or
 * otherwise returns the same path
 * @param filepath_texturi a double pointer to gchar (in/out param)
 * @param model a pointer to HildonFileSystemModel
 * @return void
 **/
void autorename_texturi(gchar **filepath_texturi, HildonFileSystemModel * model)
{
	gchar *newname_texturi = NULL;
	GError *error = NULL;

	DEBUG_PRINT_START;
	if ((*filepath_texturi == NULL) || (model == NULL)) {
		FM_ULOG_ERR("Error validating input parameters\n");
		return;
	}

	newname_texturi = hildon_file_system_model_autoname_uri(model, *filepath_texturi, &error);
	if (error != NULL) {
		FM_ULOG_ERR("hildon_file_system_model_autoname_uri:Error:%s\n", error->message);
		g_error_free(error);
		DEBUG_PRINT_END;
		return;
	}
	if (newname_texturi == NULL) {
		FM_ULOG_ERR("hildon_file_system_model_autoname_uri:returned NULL");
		DEBUG_PRINT_END;
		return;
	}
	FM_ULOG_INFO("OLD NAME %s  NEW NAME %s\n", *filepath_texturi, newname_texturi);

	if (newname_texturi != NULL) {
		g_free(*filepath_texturi);
		*filepath_texturi = newname_texturi;
	}
	DEBUG_PRINT_END;
	return;
}

/**
 * is_texturi_readonly:
 * Method to check if the path specified is readonly.
 * @param path path of file/folder
 * @param checkparent checks for parent path if TRUE.
 * @return gboolean
 **/
gboolean is_texturi_readonly(const gchar * path_texturi, const gboolean checkparent)
{
	GnomeVFSFileInfo *info = NULL;
	GnomeVFSResult result;
	GnomeVFSURI *uripath = NULL;
	gboolean retval = FALSE;
	gchar *dirpath = NULL;

	DEBUG_PRINT_START;

	if (path_texturi == NULL) {
		FM_ULOG_ERR("Invalid parameter : path (null)");
		return TRUE;
	}

	/* optimisation for UPnP/AV paths: they are always read-only */
	if (g_str_has_prefix(path_texturi, UPNP_PATH)) {
		return TRUE;
	}

	uripath = gnome_vfs_uri_new(path_texturi);

	if (uripath == NULL) {
		return TRUE;
	}

	if(!checkparent)
	{
		if (!gnome_vfs_uri_exists(uripath)) {
			if (uripath != NULL) {
				gnome_vfs_uri_unref(uripath);
			}
			return TRUE;
		}
	}

	info = gnome_vfs_file_info_new();
	if (info == NULL) {
		if (uripath != NULL) {
			gnome_vfs_uri_unref(uripath);
		}
		return TRUE;
	}

	if (checkparent) {
		GnomeVFSURI *parent_uri;

		parent_uri = gnome_vfs_uri_get_parent (uripath);

		if (parent_uri) {
			dirpath = gnome_vfs_uri_to_string (parent_uri, GNOME_VFS_URI_HIDE_NONE);
		}

		if (dirpath == NULL) {
			if (uripath != NULL) {
				gnome_vfs_uri_unref(uripath);
			}
			if (info != NULL) {
				gnome_vfs_file_info_unref(info);
			}
			return TRUE;
		}
		result = gnome_vfs_get_file_info(dirpath,
						 info,
						 (GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS |
						  GNOME_VFS_FILE_INFO_DEFAULT |
						  GNOME_VFS_FILE_INFO_FOLLOW_LINKS));
	} else {
		result = gnome_vfs_get_file_info(path_texturi,
						 info,
						 (GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS |
						  GNOME_VFS_FILE_INFO_DEFAULT |
						  GNOME_VFS_FILE_INFO_FOLLOW_LINKS));
	}

	if (result == GNOME_VFS_OK) {
		if (!(info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_ACCESS)) {
			/* could not get access permission from uri, treat as readwrite */
			retval = FALSE;
		} else {
			//check permission of path to be read and not write
			if ((info->permissions & GNOME_VFS_PERM_USER_READ) &&
			    !(info->permissions & GNOME_VFS_PERM_USER_WRITE)) {
				retval = TRUE;
			} else {
				retval = FALSE;
			}
		}
	} else {
		FM_ULOG_ERR("Unable to determine readonly status\n");
		retval = TRUE;
	}
	if (dirpath != NULL) {
		g_free(dirpath);
	}
	if (uripath != NULL) {
		gnome_vfs_uri_unref(uripath);
	}

	if (info != NULL) {
		gnome_vfs_file_info_unref(info);
	}

	DEBUG_PRINT_END;
	return retval;
}

/**
 * is_path_readonly:
 * GtkFilePath varient of is_texturi_readonly
 * @param path GtkFilePath of file/folder
 * @param checkparent checks for parent path if TRUE.
 * @return gboolean
 **/
gboolean is_path_readonly(const GtkFilePath * path, const gboolean checkparent)
{
	gchar *path_texturi = NULL;
	gboolean retval = FALSE;

	DEBUG_PRINT_START;

	path_texturi = osso_fm_path_to_texturi(path);
	retval = is_texturi_readonly(path_texturi, checkparent);
	if (path_texturi != NULL){
		g_free(path_texturi);
	}

	DEBUG_PRINT_END;
	return retval;
}

/**
 * autoname_remove_autonumber:
 * Removes the autorename number from the file name and returns a newly allocated file name without
 * number
 * @param name File name to remove autonumber
 * @return gchar * newly allocated filename with out number
 **/
gchar *autoname_remove_autonumber(gchar *name, gint *number)
{
	gint len = 0,startbrace = 0;
	len = strlen(name);
	if(name[len - 1] != ')')
	{
		return g_strdup(name);
	}
	for(startbrace = len - 2;startbrace > 0;startbrace--)
	{
		if(name[startbrace] == '(' || name[startbrace] < '0' || name[startbrace] > '9')
			break;
	}
	if(name[startbrace] == '(' && name[startbrace-1] == ' ')
	{
		if(number != NULL)
		{
			*number = (gint)g_ascii_strtoull(&name[startbrace + 1],NULL,10);
		}
		return g_strndup(name,startbrace-1);
	}
	return g_strdup(name);
}

	
/**
 * autorename_if_exists_in_list:
 * This fuction does the auto renaming of <targetfile> if it is existing in the 
 * list of paths present in GList <tgtlist>
 * @param targetfile double pointer to gchar having the file to be autorenamed
 * @param tgtlist GList of paths to be checked
 * @return gboolean TRUE if renamed else FALSE
 **/
gboolean autorename_if_exists_in_list(gchar ** targetfile, GList * tgtlist)
{
	GList *cur = NULL;
	gchar *filepath = NULL;
	GnomeVFSURI *filepathuri = NULL;
	GnomeVFSURI *newfilepathuri = NULL;
	gchar *parentpath = NULL;
	gchar *filename = NULL;
	gchar *name = NULL;
	gchar *tmp_extn = NULL;
	gchar *extn = NULL;
	gchar *newfilename = NULL;
	gchar *newname = NULL;
	gchar *newname_tmp = NULL;
	gchar *partial_newname = NULL;
	gint autonum = 0;

	for (cur = tgtlist; cur != NULL; cur = cur->next) {
		filepath = gnome_vfs_uri_to_string(cur->data, 0);
		if (g_strcasecmp(*targetfile, filepath) == 0)
			break;
		g_free(filepath);
		filepath = NULL;
	}
	//if the above loop has reached end without breaking then filename
	//doesn't exist in the list so return FALSE
	if (cur == NULL) {
		return FALSE;
	}
	filepathuri = gnome_vfs_uri_new(filepath);
	g_free(filepath);
	filepath = NULL;
	parentpath = gnome_vfs_uri_extract_dirname(filepathuri);
	filename = gnome_vfs_uri_extract_short_name(filepathuri);
	tmp_extn = osso_fm_path_get_extension(filename);
	name = g_strdup(filename);
	if (tmp_extn != NULL) {
		*(name + (tmp_extn - filename)) = '\0';
		extn = g_strdup(tmp_extn);
	}
	//partial_newname = g_new0(gchar, strlen(name));
	FM_ULOG_INFO("name without autonumber %s\n",autoname_remove_autonumber(name,NULL));
	partial_newname = autoname_remove_autonumber(name, &autonum);
	//sscanf(name, "%s (%d)", partial_newname, &autonum);
	do {
		if (newname_tmp != NULL)
			g_free(newname_tmp);
		if (extn != NULL) {
			newname_tmp =
			    g_strdup_printf("%s (%d)%s", partial_newname, ++autonum, extn);
		} else {
			newname_tmp = g_strdup_printf("%s (%d)", partial_newname, ++autonum);
		}
		newname = gnome_vfs_escape_path_string(newname_tmp);
		if (g_strncasecmp(*targetfile, "obex", 4) == 0)
			newfilename =
			    g_strdup_printf("%.26s%s/%s", *targetfile, parentpath, newname);
		else
			newfilename =
			    g_strdup_printf("%.7s%s/%s", *targetfile, parentpath, newname);
		g_free(newname);
		if (newfilepathuri != NULL) {
			gnome_vfs_uri_unref(newfilepathuri);
		}
		newfilepathuri = gnome_vfs_uri_new(newfilename);
	} while (gnome_vfs_uri_exists(newfilepathuri));
	if (newfilepathuri != NULL) {
		gnome_vfs_uri_unref(newfilepathuri);
	}
	g_free(*targetfile);
	*targetfile = newfilename;
	FM_ULOG_INFO("NEW NAME %s\n", newfilename);
	for (cur = tgtlist; cur != NULL; cur = cur->next) {
		filepath = gnome_vfs_uri_to_string(cur->data, 0);
		if (g_strcasecmp(*targetfile, filepath) == 0) {
			if (filepath != NULL) {
				g_free(filepath);
				filepath = NULL;
			}
			break;
		}
		if (filepath != NULL) {
			g_free(filepath);
			filepath = NULL;
		}
	}
	if (cur != NULL) {
		autorename_if_exists_in_list(targetfile, tgtlist);
	}
	return TRUE;
}

/**
 * make_uri_list:
 * Make GList source and target paths.
 * The lists created by using this function will be pased to the xfer_uri
 * to perform the required operation.
 * @param sourceselection  source list as GSList
 * @param sourcelist GList ** to store the output source list
 * @param targetpath target path to make the target list
 * @param targetlist GList ** to store the output target list
 * @param model for to pass to autorename function
 * @return void
 **/
void
make_uri_list(const GSList * sourceselection,
	      GList ** sourcelist,
	      const GtkFilePath * targetpath,
	      GList ** targetlist, HildonFileSystemModel * model, gboolean canautorename)
	       //GtkWidget *window)
{
	GSList *cursel = NULL;
	GSList *cur = NULL;
	gchar *shortname = NULL;
	gchar *shortname_tmp = NULL;
	gchar *targetpath_texturi = NULL;
	gchar *spacechar = NULL;
	gchar *char_pos = NULL;
	GnomeVFSURI *sourceuri = NULL;
	GnomeVFSURI *targeturi = NULL;
	gint tempcount = 0;

	//do not initialize targetfile to NULL. g_strdup_printf is not working.
	gchar *targetfile;

	DEBUG_PRINT_START;

	if (sourceselection == NULL) {
		FM_ULOG_ERR("Invalid parameters : sourceselection");
		return;
	}
	if (targetpath == NULL) {
		FM_ULOG_ERR("Invalid parameters : targetpath");
		return;
	}
	targetpath_texturi = osso_fm_path_to_texturi(targetpath);
	if(targetpath_texturi == NULL)
	{
		FM_ULOG_ERR("make_uri_list:osso_fm_path_to_texturi returns NULL\n");
		return;
	}
	for (cur = (GSList *) sourceselection; cur != NULL; cur = cur->next) {
		FM_ULOG_DEBUG1("DIRENT %s\n", (gchar *) cur->data);
	}
	for (cursel = (GSList *) sourceselection; cursel != NULL; cursel = cursel->next) {
		if (shortname != NULL) {
			g_free(shortname);
			shortname = NULL;
		}
		shortname = osso_fm_path_get_shortname(cursel->data);
		if (shortname == NULL) {
			break;
		}
		sourceuri = osso_fm_path_to_uri(cursel->data);

		if (sourceuri) {

			//if dir name is device.. check for the safe folder path..
			//TODO : to be changed once the actual api 
			//from file selection widget is available to 
			//check the display name
			tempcount = 0;
			FM_ULOG_DEBUG1("Target Path : %s\n", targetpath_texturi);
			if (is_path_from_home(targetpath_texturi, "MyDocs") == TRUE) {
				if ((g_strcasecmp(shortname, dgettext(HILDON_FM_L10N_PACKAGE,"sfil_li_folder_documents")) == 0)
				    || (g_strcasecmp(shortname, dgettext(HILDON_FM_L10N_PACKAGE, "sfil_li_folder_sound_clips")) ==
					0)
				    || (g_strcasecmp(shortname, dgettext(HILDON_FM_L10N_PACKAGE,"sfil_li_folder_video_clips")) ==
					0)
				    || (g_strcasecmp(shortname, dgettext(HILDON_FM_L10N_PACKAGE,"sfil_li_folder_games")) == 0)
				    || (g_strcasecmp(shortname, dgettext(HILDON_FM_L10N_PACKAGE,"sfil_li_folder_images")) == 0)) {
					tempcount = 1;
				}
			}

			char_pos = shortname;

			/* replace all '/' with '-' */
			while ((char_pos = strchr (char_pos, '/')) != NULL)
				*char_pos = '-';

			shortname_tmp = gnome_vfs_escape_string(shortname);
			if (tempcount != 0) {
				spacechar = gnome_vfs_escape_path_string(" ");
				targetfile = g_strdup_printf("%s/%s%s(%d)",
							     targetpath_texturi, 
							     shortname_tmp,
							     spacechar,
							     tempcount);
				g_free(spacechar);
			} else {
				targetfile = g_strdup_printf("%s/%s",
							     targetpath_texturi, 
							     shortname_tmp);
			}
			g_free(shortname_tmp);

			if (canautorename) {
				autorename_texturi(&targetfile, model);
				autorename_if_exists_in_list(&targetfile, *targetlist);
			}
			if (targetfile != NULL) {
				targeturi = gnome_vfs_uri_new(targetfile);
				g_free(targetfile);
			}

			if ((targeturi != NULL) && (sourceuri != NULL)) {
				*sourcelist = g_list_append(*sourcelist, sourceuri);
				*targetlist = g_list_append(*targetlist, targeturi);
				FM_ULOG_DEBUG1("SRC PATH %s TGT PATH %s\n",
					     gnome_vfs_uri_get_path(sourceuri),
					     gnome_vfs_uri_get_path(targeturi));
			}
		}
	}
	if (shortname != NULL) {
		g_free(shortname);
		shortname = NULL;
	}
	g_free(targetpath_texturi);
	DEBUG_PRINT_END;
	return;
}

/**
 * Method to check if the path is readable.
 * @param path texturi of file/folder to check for access 
 * @return TRUE if accessible
 **/
gboolean is_path_readable(const gchar * path)
{
	GnomeVFSDirectoryHandle *handle = NULL;
	GnomeVFSResult result;
	result = gnome_vfs_directory_open(&handle, path, GNOME_VFS_FILE_INFO_DEFAULT);
	if (handle == NULL && result == GNOME_VFS_ERROR_ACCESS_DENIED) {
		FM_ULOG_ERR("Dir is not readable \n");
		return FALSE;
	}
	if(handle!=NULL)
	gnome_vfs_directory_close(handle);
	return TRUE;
}
/**
 * Method to check if the path specified exist.
 * @param path texturi of file/folder to check for existance
 * @return TRUE if exists 
 **/
gboolean is_path_exists(const gchar * path)
{
	GnomeVFSURI *uri = NULL;
	gboolean retval = FALSE;

	DEBUG_PRINT_START;

	if (path == NULL) {
		FM_ULOG_ERR("Invalid parameter : path (null)");
		return FALSE;
	}

	uri = gnome_vfs_uri_new(path);
	if (uri == NULL) {
		FM_ULOG_ERR("Error allocating memeory");
		return FALSE;
	}

	retval = gnome_vfs_uri_exists(uri);
	FM_ULOG_DEBUG1("retval : %d\n", retval);

	if (uri != NULL) {
		gnome_vfs_uri_unref(uri);
	}

	DEBUG_PRINT_END;
	return retval;

}

/**
 * Method to get the path type
 * @param path File/folder path
 * @return specifies the path type
 **/
gint osso_fm_get_path_type(const gchar * path)
{
	GnomeVFSFileInfo *info = NULL;
	GnomeVFSResult result;
	gint type = GNOME_VFS_FILE_TYPE_UNKNOWN;

	DEBUG_PRINT_START;


	if (path == NULL) {
		FM_ULOG_ERR("Invalid parameter : path (null)");
		DEBUG_PRINT_END;
		return type;
	}

	info = gnome_vfs_file_info_new();
	if (info == NULL) {
		DEBUG_PRINT_END;
		return OSSO_FM_MEMORY_ALLOCATION_FAILURE;
	}

	result = gnome_vfs_get_file_info(path,
					 info,
					 (GNOME_VFS_FILE_INFO_GET_MIME_TYPE
					  | GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS
					  | GNOME_VFS_FILE_INFO_FOLLOW_LINKS));

	if (result == GNOME_VFS_OK) {
		type = info->type;
	}

	gnome_vfs_file_info_unref(info);

	DEBUG_PRINT_END;
	return type;
}


static gchar * get_display_name (HildonFileSystemModel *fsmodel,
		const GtkFilePath *path, 
		const gchar *path_texturi)
{
	GnomeVFSURI *pathuri = NULL;
	GtkTreeIter iter;
	gchar *displayname = NULL;
	DEBUG_PRINT_START;
	if (!hildon_file_system_model_search_path (fsmodel, path, &iter, NULL, TRUE)) {
		FM_ULOG_ERR("Unable to find path in the model %s\n",path_texturi);
		pathuri = gnome_vfs_uri_new(path_texturi);
		displayname = gnome_vfs_uri_extract_short_name(pathuri);
		gnome_vfs_uri_unref(pathuri);
	}
	else
	{
		gtk_tree_model_get((GtkTreeModel *) fsmodel,
				   &iter, HILDON_FILE_SYSTEM_MODEL_COLUMN_DISPLAY_NAME, 
				   &displayname, -1);
	}
	DEBUG_PRINT_END;
	return displayname;
}
/**
 * osso_fm_path_get_display_name:
 * This method returns the display name for the given path.
 * GtkFilePath varient of osso_fm_texturi_get_display_name
 * @param path GtkFilePath of the file/folder whose display name has be obtained
 * @return gchar* display name for the given path
 **/
gchar *osso_fm_path_get_display_name(const GtkFilePath * path)
{
	HildonFileSystemModel *fsmodel = NULL;
	gchar * path_texturi;
	gchar *displayname = NULL;

	DEBUG_PRINT_START;
	if (path == NULL) {
		FM_ULOG_ERR("osso_fm_path_get_display_name: path == NULL\n");
		return NULL;
	}
	g_object_get(G_OBJECT(global_ofm->fs), "model", &fsmodel, NULL);
	if (fsmodel == NULL) {
		FM_ULOG_ERR("osso_fm_path_get_display_name:Unable to get Model\n");
		return NULL;
	}
	path_texturi = osso_fm_path_to_texturi(path);
	if(path_texturi == NULL)
	{
		FM_ULOG_ERR("osso_fm_path_get_display_name:path_texturi == NULL\n");
		return NULL;
	}
	displayname = get_display_name(fsmodel,path,path_texturi);

	g_free(path_texturi);

	DEBUG_PRINT_END;
	return displayname;
}

/**
 * osso_fm_texturi_get_display_name:
 * This method returns the display name for the given path.
 * @param path_texturi, text uri of the file/folder whose display name has be obtained
 * @return gchar* display name for the given path
 **/
gchar *osso_fm_texturi_get_display_name(const gchar * path_texturi)
{
	HildonFileSystemModel *fsmodel = NULL;
	GtkFilePath *path = NULL;
	gchar *displayname = NULL;

	DEBUG_PRINT_START;
	if (path_texturi == NULL) {
		FM_ULOG_ERR("osso_fm_texturi_get_display_name: path_texturi == NULL\n");
		return NULL;
	}
	g_object_get(G_OBJECT(global_ofm->fs), "model", &fsmodel, NULL);
	if (fsmodel == NULL) {
		FM_ULOG_ERR("osso_fm_texturi_get_display_name:Unable to get Model\n");
		return NULL;
	}
	path = osso_fm_texturi_to_path(path_texturi);
	if(path == NULL)
	{
		FM_ULOG_ERR("osso_fm_texturi_get_display_name:path == NULL\n");
		return NULL;
	}
	displayname = get_display_name(fsmodel,path,path_texturi);

	gtk_file_path_free(path);

	DEBUG_PRINT_END;
	return displayname;
}

/**
 * osso_fm_texturi_get_shortname:
 * Method to get the shortname of a path_texturi
 * @param path File's/folder's text uri
 * @return shortname if the path_texturi.
 */
gchar *osso_fm_texturi_get_shortname(const gchar* path_texturi)
{
	GnomeVFSURI *pathuri = NULL;
	GtkFilePath *path = NULL;
	gchar *shortname = NULL;

	DEBUG_PRINT_START;

	if (path_texturi == NULL) {
		FM_ULOG_ERR("Invalid parameter : path (null)");
		return NULL;
	}
	path = osso_fm_texturi_to_path(path_texturi);

	switch (osso_fm_fs_get_path_type(global_ofm->fs,path)) {
	case HILDON_FILE_SYSTEM_MODEL_LOCAL_DEVICE:
	case HILDON_FILE_SYSTEM_MODEL_GATEWAY:
	case HILDON_FILE_SYSTEM_MODEL_MMC:
		shortname = osso_fm_path_get_display_name(path);
		break;
	case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_DOCUMENTS:
		shortname = g_strdup_printf("%s", dgettext(HILDON_FM_L10N_PACKAGE,
					"sfil_li_folder_documents"));
		break;
	case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_IMAGES:
		shortname = g_strdup_printf("%s", dgettext(HILDON_FM_L10N_PACKAGE,
					"sfil_li_folder_images"));
		break;
	case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_SOUNDS:
		shortname = g_strdup_printf("%s", dgettext(HILDON_FM_L10N_PACKAGE,
					"sfil_li_folder_sound_clips"));
		break;
	case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_VIDEOS:
		shortname = g_strdup_printf("%s", dgettext(HILDON_FM_L10N_PACKAGE,
					"sfil_li_folder_video_clips"));
		break;
	case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_GAMES:
		shortname = g_strdup_printf("%s", dgettext(HILDON_FM_L10N_PACKAGE,
					"sfil_li_folder_games"));
		break;
//	case HILDON_FILE_SYSTEM_MODEL_MMC:
//		shortname = g_strdup_printf("%s", dgettext(HILDON_FM_L10N_PACKAGE,
//					"sfil_li_mmc_localdevice"));
//		break;
	default:
		pathuri = gnome_vfs_uri_new (path_texturi);

		if (pathuri != NULL) {
			shortname = gnome_vfs_uri_extract_short_name(pathuri);
		} else {
			FM_ULOG_ERR("Error allocating Memory");
			gtk_file_path_free(path);
			return NULL;
		}
		gnome_vfs_uri_unref(pathuri);
	}

	gtk_file_path_free(path);
	DEBUG_PRINT_END;
	return shortname;
}
/**
 * osso_fm_path_get_shortname:
 * Method to get the shortname of a path
 * @param path GtkFilePath to the path
 * @return shortname if the given path.
 **/
gchar *osso_fm_path_get_shortname(const GtkFilePath * path)
{
	GnomeVFSURI *pathuri = NULL;
	gchar *shortname = NULL;
	gchar *shortpathname = NULL;

	DEBUG_PRINT_START;

	if (path == NULL) {
		FM_ULOG_ERR("Invalid parameter : path (null)");
		return NULL;
	}


	switch (osso_fm_fs_get_path_type(global_ofm->fs,path)) {
	case HILDON_FILE_SYSTEM_MODEL_LOCAL_DEVICE:
	case HILDON_FILE_SYSTEM_MODEL_GATEWAY:	//Added to Fix Bug 18758
	case HILDON_FILE_SYSTEM_MODEL_MMC:
		shortname = osso_fm_path_get_display_name(path);
		break;
	case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_DOCUMENTS:
		shortname = g_strdup_printf("%s", dgettext(HILDON_FM_L10N_PACKAGE,
					"sfil_li_folder_documents"));
		break;
	case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_IMAGES:
		shortname = g_strdup_printf("%s", dgettext(HILDON_FM_L10N_PACKAGE,
					"sfil_li_folder_images"));
		break;
	case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_SOUNDS:
		shortname = g_strdup_printf("%s", dgettext(HILDON_FM_L10N_PACKAGE,
					"sfil_li_folder_sound_clips"));
		break;
	case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_VIDEOS:
		shortname = g_strdup_printf("%s", dgettext(HILDON_FM_L10N_PACKAGE,
					"sfil_li_folder_video_clips"));
		break;
	case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_GAMES:
		shortname = g_strdup_printf("%s", dgettext(HILDON_FM_L10N_PACKAGE,
					"sfil_li_folder_games"));
		break;
//	case HILDON_FILE_SYSTEM_MODEL_MMC:
//		shortname = g_strdup_printf("%s", dgettext(HILDON_FM_L10N_PACKAGE,
//					"sfil_li_mmc_localdevice"));
//		break;
	default:
		pathuri = osso_fm_path_to_uri(path);

		if (pathuri != NULL) {
			shortpathname = gnome_vfs_uri_extract_short_path_name(pathuri);
			shortname = gnome_vfs_unescape_string (shortpathname, NULL);
			g_free (shortpathname);
		} else {
			FM_ULOG_ERR("Error allocating Memory");
			return NULL;
		}
		gnome_vfs_uri_unref(pathuri);
	}

	DEBUG_PRINT_END;
	return shortname;
}


/**
 * osso_fm_path_get_dirname:
 * Gets the directory path of the text uri
 * @param path text uri for which the directory path has to be retrived
 * @return gchar * the dirpath text uri.
 **/
gchar *osso_fm_path_get_dirname(const gchar * path)
{
	gchar *dirname = NULL;
	GnomeVFSURI *pathuri = NULL;
	GnomeVFSURI *parenturi = NULL;
	DEBUG_PRINT_START;

	if (path == NULL) {
		FM_ULOG_ERR("Path specified NULL");
		return NULL;
	}
	pathuri = gnome_vfs_uri_new(path);
	if (pathuri == NULL) {
		FM_ULOG_ERR("Error allocating memory\n");
		return NULL;
	}
	parenturi = gnome_vfs_uri_get_parent(pathuri);
	gnome_vfs_uri_unref(pathuri);

	if (parenturi == NULL) {
		FM_ULOG_ERR("Error allocating memory\n");
		return NULL;
	}
	dirname = gnome_vfs_uri_to_string(parenturi, GNOME_VFS_URI_HIDE_NONE);

	gnome_vfs_uri_unref(parenturi);


	DEBUG_PRINT_END;
	return dirname;
}



typedef struct {
	gchar *extension;
	gchar *mime;
} MimeType;

/**
 * mime_list_insert:
 * A local helper function in getting the extension of file name
 * @param a gconstpointer
 * @param b gconstpointer
 * @return gint
 **/
static gint mime_list_insert(gconstpointer a, gconstpointer b)
{
	return strlen(((MimeType *) b)->extension) - strlen(((MimeType *) a)->extension);
}

/**
 * osso_fm_path_get_extension:
 * A Method to get the extension of a file by looking at the mime database
 * @param name gchar * file name
 * @return gchar * pointers to the '.' from were the extension starts in a given name. 
 **/
gchar *osso_fm_path_get_extension(const gchar * name)
{
	static GSList *types = NULL;
	MimeType *type;
	GSList *iter;
	gchar *mime = NULL;
	gchar *candidate = NULL;
	gint len;
	if (name == NULL) {
		FM_ULOG_ERR("osso_fm_path_get_extension:Error in params\n");
		return NULL;
	}
	mime = gnome_vfs_get_mime_type(name);
	/* Unrecognized types are not touched */
	if (mime && g_ascii_strcasecmp(mime, "application/octet-stream") == 0)
		return NULL;

	/* Initialize suffix hash table from /usr/share/mime/globs */
	if (!types) {
		FILE *f;
		gchar line[256];
		gchar *sep;

		f = fopen("/usr/share/mime/globs", "rt");
		if (f) {
			while (fgets(line, sizeof(line), f)) {
				if (line[0] == 0 || line[0] == '#')
					continue;
				/* fgets leaves newline into buffer */
				len = strlen(line);
				if (line[len - 1] == '\n')
					line[len - 1] = 0;
				sep = strstr(line, ":*.");
				if (sep == NULL)
					continue;
				*sep = 0;	/* Clear colon */

				type = g_new(MimeType, 1);
				type->extension = g_strdup(sep + 2);
				type->mime = g_strdup(line);
				types = g_slist_insert_sorted(types, type, mime_list_insert);
			}

			fclose(f);

/*      for (iter = types; iter; iter = iter->next)
      {
        type = iter->data;
        fprintf(stderr, "%s: %s\n", type->extension, type->mime);
      }*/
		}
	}

	/* Now we must search possible extensions from the list that match
	   suffix of the given name. If mime type is given, it also has to match */
	len = strlen(name);
	for (iter = types; iter; iter = iter->next) {
		type = iter->data;

		if (!mime || g_ascii_strcasecmp(mime, type->mime) == 0) {
			candidate = (gchar *) name + len - strlen(type->extension);

			if (name <= candidate &&
			    g_ascii_strcasecmp(candidate, type->extension) == 0)
				return candidate;
		}
	}

	///* If we didn't find a match (but type was recognized), then we just
	//   return the part after last dot. If we didn't have type information,
	//   then we do not touch the name. */
	//return mime ? g_strrstr(name, ".") : NULL;
	return g_strrstr(name, ".");
}



/**
 * osso_fm_is_path_directory:
 * Method to whether a texturi is that of a directory
 * @param uri is uri of an item
 * @return gboolean
 */
gboolean osso_fm_is_path_directory(const GtkFilePath *path)
{
	gboolean retval = FALSE;
	gchar *path_texturi = NULL;
	DEBUG_PRINT_START;
	path_texturi = osso_fm_path_to_texturi(path);
	retval = osso_fm_is_texturi_directory(path_texturi);
	g_free(path_texturi);
	DEBUG_PRINT_END;
	return retval;
}

/**
 * osso_fm_is_texturi_directory:
 * Method to whether a texturi is that of a directory
 * @param path_texturi is uri of an item
 * @return gboolean
 **/
gboolean osso_fm_is_texturi_directory(const char *path_texturi)
{
	GnomeVFSFileInfo *info = NULL;
	GnomeVFSResult result;
	gboolean is_dir = FALSE;

	DEBUG_PRINT_START;

	if (path_texturi == NULL) {
		FM_ULOG_ERR("Invalid parameter : uri ");
		return is_dir;
	}

	info = gnome_vfs_file_info_new();
	if (info == NULL) {
		return FALSE;
	}
	result = gnome_vfs_get_file_info(path_texturi, info, GNOME_VFS_FILE_INFO_DEFAULT);

	if (result == GNOME_VFS_OK && info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_TYPE) {
		is_dir = (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY);
	}

	gnome_vfs_file_info_unref(info);

	DEBUG_PRINT_END;
	return is_dir;
}

/**
 * is_path_from_home:
 * Method checks the path is located at home with the specified shortname
 * @param path path to be verified
 * @param shortname the name against which the path is to be checked.
 * @return gboolean TRUE if the shortname matches with path  shortname 		    
 * and location is home
 **/
gboolean is_path_from_home(const gchar * path, const gchar * shortname)
{

	gchar *checkpath = NULL;
	gchar *homepath = NULL;
	GnomeVFSURI *temp_uri = NULL;
	gboolean retval = FALSE;

	DEBUG_PRINT_START;
	if ((path == NULL) || (shortname == NULL)) {
		FM_ULOG_ERR("Error validating input parameters\n");
		return FALSE;
	}
	//homedir is set in the initialization
	temp_uri = gnome_vfs_uri_new(homedir);
	if (temp_uri == NULL) {
		FM_ULOG_ERR("Error allocating memory");
		return FALSE;
	}
	homepath = gnome_vfs_uri_to_string(temp_uri, 0);

	checkpath = g_strdup_printf("%s/%s", homepath, shortname);

	if (checkpath == NULL) {
		FM_ULOG_ERR("Error allocating memory\n");
		if (homepath != NULL) {
			g_free(homepath);
		}
		gnome_vfs_uri_unref(temp_uri);
		return FALSE;
	}
	/* Changed ,strcmp with g_strcasecmp  */
	retval = (g_utf8_collate(path, checkpath) == 0);




	g_free(checkpath);
	if (homepath != NULL) {
		g_free(homepath);
	}
	gnome_vfs_uri_unref(temp_uri);
	DEBUG_PRINT_END;
	return retval;
}

/**
 * is_uri_in_gateway:
 * Checks if the given uri is in Gateway
 * @param fs HildonFileSelection Widget
 * @return TRUE if path is in gateway else FALSE
 **/
gboolean is_uri_in_gateway(const GnomeVFSURI* path_uri)
{
	gboolean retval = FALSE;
	gchar * path_texturi = NULL;
	DEBUG_PRINT_START;
	if (path_uri == NULL) {
		FM_ULOG_ERR("is_uri_in_gateway : path_uri == NULL\n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	path_texturi = gnome_vfs_uri_to_string(path_uri,GNOME_VFS_URI_HIDE_NONE);
	retval = is_texturi_in_gateway(path_texturi);
	g_free(path_texturi);
	return retval;
}

/**
 * is_path_in_gateway:
 * Checks if the given path is in Gateway
 * @param path GtkFilePath which has to be checked
 * @return TRUE if path is in gateway else FALSE
 */
gboolean is_path_in_gateway(const GtkFilePath * path)
{
	gboolean retval = FALSE;
	gchar * path_texturi = NULL;
	DEBUG_PRINT_START;
	if (path == NULL) {
		FM_ULOG_ERR("is_path_in_gateway : path == NULL\n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	path_texturi = osso_fm_path_to_texturi(path);
	retval = is_texturi_in_gateway(path_texturi);
	g_free(path_texturi);
	return retval;
}

/**
 * is_texturi_in_gateway:
 * Checks if the given path is in Gateway
 * @param path gchar * text uri which has to be checked
 * @return TRUE if text uri is in gateway else FALSE
 */
gboolean is_texturi_in_gateway(const gchar * path_texturi)
{
	DEBUG_PRINT_START;
	if (path_texturi == NULL) {
		FM_ULOG_ERR("is_texturi_in_gateway:path_texturi == NULL\n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	if (g_strncasecmp((gchar *) path_texturi, "obex", 4) == 0) {
		DEBUG_PRINT_END;
		return TRUE;
	} else {
		DEBUG_PRINT_END;
		return FALSE;
	}
	return FALSE;
}

/**
 * selected_location_in_gateway:
 * Checks if the selected location is in Gateway
 * @param fs HildonFileSelection Widget
 * @return TRUE if path is in gateway else FALSE
 */
gboolean selected_location_in_gateway(HildonFileSelection * fileselection)
{
	GtkFilePath *path = NULL;
	gboolean retval = FALSE;
	DEBUG_PRINT_START;
	if (fileselection == NULL) {
		FM_ULOG_ERR("Invalid Input Parameters\n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	path = hildon_file_selection_get_current_folder(fileselection);
	if (path == NULL) {
		FM_ULOG_ERR("Error getting current selected path\n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	retval = is_path_in_gateway(path);
	gtk_file_path_free(path);
	DEBUG_PRINT_END;
	return retval;
}

/**
 * selected_location_in_mmc:
 * Check if the selected location is in MMC
 * @param fs HildonFileSelection Widget
 * @return TRUE if path is in MMC else FALSE
 **/
gboolean selected_location_in_mmc(HildonFileSelection * fileselection)
{
	GtkFilePath *path = NULL;
	gboolean internal = FALSE;
	gboolean retval = FALSE;
	DEBUG_PRINT_START;
	if (fileselection == NULL) {
		FM_ULOG_ERR("Invalid Input Parameters\n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	path = hildon_file_selection_get_current_folder(fileselection);
	if (path == NULL) {
		FM_ULOG_ERR("selected_location_in_mmc:Error getting current path\n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	retval = is_path_in_mmc(path,&internal);
	gtk_file_path_free(path);
	if(internal)
	{
		DEBUG_PRINT_END;
		return FALSE;
	}
	return retval;
}

/**
 * selected_location_in_internal_mmc:
 * Check if the selected location is in MMC
 * @param fs HildonFileSelection Widget
 * @return TRUE if path is in MMC else FALSE
 **/
gboolean selected_location_in_internal_mmc(HildonFileSelection * fileselection)
{
	GtkFilePath *path = NULL;
	gboolean internal = FALSE;
	gboolean retval = FALSE;
	DEBUG_PRINT_START;
	if (fileselection == NULL) {
		FM_ULOG_ERR("Invalid Input Parameters\n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	path = hildon_file_selection_get_current_folder(fileselection);
	if (path == NULL) {
		FM_ULOG_ERR("selected_location_in_internal_mmc:Error getting current path\n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	retval = is_path_in_mmc(path,&internal);
	gtk_file_path_free(path);
	if(!internal)
	{
		DEBUG_PRINT_END;
		return FALSE;
	}
	return retval;
}

/**
 * is_texturi_in_media_server:
 * Checks if the given path is in Media Server
 * @param path gchar * text uri which has to be checked
 * @return TRUE if text uri is in Media Server else FALSE
 **/
gboolean is_texturi_in_media_server(gchar *path_texturi)
{
	DEBUG_PRINT_START;
	if(path_texturi == NULL)
	{
		FM_ULOG_ERR("is_texturi_in_media_server:path_texturi = NULL\n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	if (g_strncasecmp(path_texturi, UPNP_PATH, strlen(UPNP_PATH)) == 0) {
		DEBUG_PRINT_END;
		return TRUE;
	} else {
		DEBUG_PRINT_END;
		return FALSE;
	}
}

/**
 * is_path_in_media_server:
 * Checks if the given path is in Media Server
 * @param path GtkFilePath path which has to be checked
 * @return TRUE if path is in Media Server else FALSE
 */
gboolean is_path_in_media_server(GtkFilePath *path)
{
	gchar *path_texturi = NULL;
	gboolean retval = FALSE;
	if (path == NULL) {
		FM_ULOG_ERR("is_path_in_media_server:path = NULL\n");
		DEBUG_PRINT_END;
		return retval;
	}
	path_texturi = osso_fm_path_to_texturi(path);
	retval = is_texturi_in_media_server(path_texturi);
	if(path_texturi != NULL) {
		g_free(path_texturi);
	}
	DEBUG_PRINT_END;
	return retval;
}
/**
 * selected_location_in_media_server:
 * Check if the selected location is in Media Server
 * @param fs HildonFileSelection Widget
 * @return TRUE if path is in Media Server else FALSE
 */
gboolean selected_location_in_media_server(HildonFileSelection * fileselection)
{
	GtkFilePath *path = NULL;
	gboolean retval = FALSE;
	DEBUG_PRINT_START;
	if (fileselection == NULL) {
		FM_ULOG_ERR("Invalid Input Parameters\n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	path = hildon_file_selection_get_current_folder(fileselection);
	retval = is_path_in_media_server(path);
	if(path != NULL) {
		gtk_file_path_free(path);
	}
	DEBUG_PRINT_END;
	return retval;
}

/**
 * is_texturi_in_mmc:
 * Check if the given gchar texturi is in MMC
 * @param path_texturi gchar *
 * @return TRUE if path is in MMC else FALSE
 */
gboolean is_texturi_in_mmc(gchar * path_texturi, gboolean *internal)
{
	gchar *total_mmc_path = NULL;
	gchar *total_internal_mmc_path = NULL;
	DEBUG_PRINT_START;
	if (path_texturi == NULL) {
		FM_ULOG_ERR("is_texturi_in_mmc Invalid Input Parameters\n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	total_mmc_path = g_strdup_printf("file://%s", MMC_PATH);
	if (g_strncasecmp(path_texturi, total_mmc_path, strlen(total_mmc_path)) == 0) {
		g_free(total_mmc_path);
		if(internal)
			*internal = FALSE;
		DEBUG_PRINT_END;
		return TRUE;
	}
	g_free(total_mmc_path);
	total_internal_mmc_path = g_strdup_printf("file://%s", INTERNAL_MMC_PATH);
	if (g_strncasecmp(path_texturi, total_internal_mmc_path, strlen(total_internal_mmc_path)) 
			== 0) {
		g_free(total_internal_mmc_path);
		if(internal)
			*internal = TRUE;
		DEBUG_PRINT_END;
		return TRUE;
	}
	g_free(total_internal_mmc_path);
	DEBUG_PRINT_END;
	return FALSE;

}
/**
 * is_path_in_mmc:
 * Check if the given path is in MMC
 * @param path GtkFilePath
 * @return TRUE if path is in MMC else FALSE
 */
gboolean is_path_in_mmc(GtkFilePath * path, gboolean *internal)
{
	gboolean retval = FALSE;
	gchar *path_texturi = NULL;
	DEBUG_PRINT_START;
	if (path == NULL) {
		FM_ULOG_ERR("is_path_in_mmc Invalid Input Parameters\n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	path_texturi = osso_fm_path_to_texturi(path);
	retval = is_texturi_in_mmc(path_texturi,internal);
	g_free(path_texturi);
	
	DEBUG_PRINT_END;
	return retval;

}
/**
 * is_uri_in_mmc:
 * Check if the given uri is in MMC
 * @param uri GnomeVFSURI
 * @return TRUE if path is in MMC else FALSE
 */
gboolean is_uri_in_mmc(GnomeVFSURI * uri, gboolean *internal)
{
	gboolean retval = FALSE;
	gchar *path_texturi = NULL;
	DEBUG_PRINT_START;
	if (uri == NULL) {
		FM_ULOG_ERR("is_uri_in_mmc Invalid Input Parameters\n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	path_texturi = gnome_vfs_uri_to_string(uri, 0);
	if (path_texturi == NULL) {
		FM_ULOG_ERR("is_uri_in_mmc : gnome_vfs_uri_to_string failed\n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	retval = is_texturi_in_mmc(path_texturi,internal);
	g_free(path_texturi);
	
	DEBUG_PRINT_END;
	return retval;
}

/**
 * osso_fm_fs_get_path_type:
 * This function get the path type of from hildon file system model
 * @param fs HildonFileSelection to get the model
 * @param path GtkFilePath path for which type has to be determined.
 * @return HildonFileSystemModelItemType type obtained from model of 
 * HILDON_FILE_SYSTEM_MODEL_UNKNOWN in case of any failure
 **/
HildonFileSystemModelItemType osso_fm_fs_get_path_type(
		HildonFileSelection *fs,const GtkFilePath *path)
{
	HildonFileSystemModelItemType type = HILDON_FILE_SYSTEM_MODEL_UNKNOWN;
	HildonFileSystemModel *fsmodel = NULL;
	GtkTreeIter iter;
	if(fs == NULL)
	{
		FM_ULOG_ERR("osso_fm_fs_get_path_type:fs = NULL\n");
		return HILDON_FILE_SYSTEM_MODEL_UNKNOWN;
	}
	if(path == NULL)
	{
		FM_ULOG_ERR("osso_fm_fs_get_path_type:path = NULL\n");
		return HILDON_FILE_SYSTEM_MODEL_UNKNOWN;
	}
	g_object_get(G_OBJECT(fs), "model", &fsmodel, NULL);
	if (fsmodel == NULL) {
		FM_ULOG_ERR("osso_fm_fs_get_path_type:Unable to get Model\n");
		return HILDON_FILE_SYSTEM_MODEL_UNKNOWN;
	}
	if (!hildon_file_system_model_search_path (fsmodel, path, &iter, NULL, TRUE)) {
		FM_ULOG_ERR("Unable to find path in the model %s\n",(gchar *)path);
                /* handle the error later */
		return HILDON_FILE_SYSTEM_MODEL_FILE;
	}
	gtk_tree_model_get((GtkTreeModel *) fsmodel,
			   &iter, HILDON_FILE_SYSTEM_MODEL_COLUMN_TYPE, &type, -1);
	return type;
}






/** 
 * set_default_paths_to_key_file:
 * helper Method which will set the inital folder settings
 * @return void
 **/
static void set_default_paths_to_key_file()
{
	gchar *imagepath = NULL;
	gchar *audiopath = NULL;
	gchar *gamespath = NULL;

	gchar *imageuri = NULL;
	gchar *audiouri = NULL;
	gchar *gamesuri = NULL;

	DEBUG_PRINT_START;

	imagepath = gnome_vfs_expand_initial_tilde("~/MyDocs/.images");
	audiopath = gnome_vfs_expand_initial_tilde("~/MyDocs/.sounds");
	gamespath = gnome_vfs_expand_initial_tilde("~/MyDocs/.games");

	imageuri = gnome_vfs_get_uri_from_local_path(imagepath);
	audiouri = gnome_vfs_get_uri_from_local_path(audiopath);
	gamesuri = gnome_vfs_get_uri_from_local_path(gamespath);

	osso_fm_set_config_value_for_texturi(imageuri, TRUE, 
					HILDON_FILE_SELECTION_SORT_MODIFIED, GTK_SORT_DESCENDING);
	osso_fm_set_config_value_for_texturi(audiouri, TRUE,
					HILDON_FILE_SELECTION_SORT_NAME, GTK_SORT_ASCENDING);
	osso_fm_set_config_value_for_texturi(gamesuri, FALSE,
					HILDON_FILE_SELECTION_SORT_MODIFIED, GTK_SORT_DESCENDING);

	if (imagepath != NULL) {
		g_free(imagepath);
	}
	if (audiopath != NULL) {
		g_free(audiopath);
	}
	if (imageuri != NULL) {
		g_free(imageuri);
	}
	if (audiouri != NULL) {
		g_free(audiouri);
	}
	if (imagepath != NULL) {
		g_free(gamespath);
	}
	if (imageuri != NULL) {
		g_free(gamesuri);
	}
	DEBUG_PRINT_END;
}

/**
 * ensure_preference_file_path_exists:
 * This function ensures the path of the preference file exists. 
 * creates the path if it doesn't exist
 * @return gboolean TRUE if exists or created and FALSE if failed to create or some other error
 **/
static gboolean ensure_preference_file_path_exists()
{
	gchar *keyfilepath = NULL;
	GnomeVFSURI *keyfilepathuri = NULL;
	GnomeVFSResult result;
	DEBUG_PRINT_START;
	keyfilepath = gnome_vfs_expand_initial_tilde(PREFERENCES_FILE_PATH);
	if (keyfilepath == NULL) {
		FM_ULOG_ERR("ensure_preference_file_path_exists:keyfilepath is NULL\n");
		return FALSE;
	}
	keyfilepathuri = gnome_vfs_uri_new(keyfilepath);
	g_free(keyfilepath);
	if (keyfilepathuri == NULL) {
		FM_ULOG_ERR("ensure_preference_file_path_exists:keyfilepathuri is NULL\n");
		return FALSE;
	}

	if (!gnome_vfs_uri_exists(keyfilepathuri)) {
		result = gnome_vfs_make_directory_for_uri(keyfilepathuri, 0755);
		if (result != GNOME_VFS_OK) {
			FM_ULOG_ERR("Error Creating Preference Directory [Err %d]\n", result);
			gnome_vfs_uri_unref(keyfilepathuri);
			return FALSE;
		}
	}
	gnome_vfs_uri_unref(keyfilepathuri);
	return TRUE;
}

/**
 * osso_fm_key_file_init:
 * Method to initialize the key file for persistence of folder settings
 * @return void
 **/
void osso_fm_key_file_init()
{

	gchar *keypath = NULL;
	GnomeVFSURI *uri = NULL;
	//GError *error = NULL;

	DEBUG_PRINT_START;

	homedir = (gchar *) g_get_home_dir();

	if (homedir == NULL) {
		FM_ULOG_ERR("Error allocating memory for variable homedir\n");
		return;
	}
	if (!ensure_preference_file_path_exists()) {
		FM_ULOG_ERR("Error Lookingup/creating keyfilepath\n");
	}
	keypath = gnome_vfs_expand_initial_tilde(PREFERENCES_FILE);
	FM_ULOG_DEBUG1("keypath === %s\n", keypath);

	if (global_ofm->keyfile == NULL) {
		global_ofm->keyfile = g_key_file_new();
	}

	if (global_ofm->keyfile != NULL) {
		g_key_file_load_from_file(global_ofm->keyfile, keypath, 0, NULL);	//&error);
	} else {
		FM_ULOG_ERR("Error creating key file\n");
	}

	FM_ULOG_DEBUG1("keypath === %s\n", keypath);
	uri = gnome_vfs_uri_new(keypath);
	if (uri != NULL) {
		if (!gnome_vfs_uri_exists(uri)) {
			set_default_paths_to_key_file();
		}
		gnome_vfs_uri_unref(uri);
	} else {
		FM_ULOG_ERR("Error creating uri\n");
	}

	g_free(keypath);

	DEBUG_PRINT_END;
	return;
}

/**
 * osso_fm_get_gateway_phone_connection_status:
 * Method to read gconf value for the connection dialog.
 * @return 1 if wizad has to be displayed
 * 0 if it should not be displayed.
 **/
gint osso_fm_get_gateway_phone_connection_status()
{
	gint value = 0;
	GError *error = NULL;
	DEBUG_PRINT_START;
	value = g_key_file_get_integer(global_ofm->keyfile,
				       PHONE_CONNECTION_DIALOG_STATUS_GROUP,
				       PHONE_CONNECTION_DIALOG_STATUS, &error);

	if (error != NULL) {
		if (error->code == G_KEY_FILE_ERROR_KEY_NOT_FOUND) {
			FM_ULOG_DEBUG1("Error reading phone connection value : %s\n", error->message);
		} else {
			FM_ULOG_ERR("Error reading phone connection value : %s\n", error->message);
		}
		return -1;
	}
	FM_ULOG_DEBUG1("Phone Connection Status %d\n", value);
	DEBUG_PRINT_END;
	return value;
}

/**
 * osso_fm_set_gateway_phone_connection_status:
 * Method to read gconf value for the connection dialog.
 * @param status status of phone connection dialog
 * @return void.
 */
void osso_fm_set_gateway_phone_connection_status(gint status)
{
	DEBUG_PRINT_START;
	g_key_file_set_integer(global_ofm->keyfile,
			       PHONE_CONNECTION_DIALOG_STATUS_GROUP,
			       PHONE_CONNECTION_DIALOG_STATUS, status);
	DEBUG_PRINT_END;
	return;
}

/**
 * osso_fm_get_toolbar_status:
 * Method to get the tool bar status from preferences file.
 * @param normal out param for normal screen toolbar status
 * @param fullscreen out param for fullscreen mode toolbar status
 * @return 1 successful 0 failed.
 */
gboolean osso_fm_get_toolbar_status(gint * normal, gint * fullscreen)
{
	//gint value = 0;
	GError *error_nor = NULL;
	GError *error_ful = NULL;
	//value = 0;
	DEBUG_PRINT_START;
	if (normal == NULL || fullscreen == NULL) {
		FM_ULOG_ERR("Invalid parameters \n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	*normal = g_key_file_get_integer(global_ofm->keyfile,
					 TOOLBAR_STATUS, NORMAL_VIEW_TOOLBAR, &error_nor);

	//if(error->code == G_KEY_FILE_ERROR_KEY_NOT_FOUND || 
	//    error->code == G_KEY_FILE_ERROR_INVALID_VALUE)
	if (error_nor != NULL) {
		*normal = -1;
		if (error_nor->code == G_KEY_FILE_ERROR_KEY_NOT_FOUND ||
		    error_nor->code == G_KEY_FILE_ERROR_GROUP_NOT_FOUND) {
			FM_ULOG_DEBUG1("Error reading normal mode toolbar value : %s\n",
				     error_nor->message);
		} else {
			FM_ULOG_ERR("Error reading normal mode toolbar value : %s\n",
				    error_nor->message);
		}
		g_key_file_set_integer(global_ofm->keyfile,
				       TOOLBAR_STATUS, NORMAL_VIEW_TOOLBAR, 1);
	}
	*fullscreen = g_key_file_get_integer(global_ofm->keyfile,
					     TOOLBAR_STATUS, FULLSCREEN_TOOLBAR, &error_ful);
	//if(error->code == G_KEY_FILE_ERROR_KEY_NOT_FOUND || 
	//    error->code == G_KEY_FILE_ERROR_INVALID_VALUE)
	if (error_ful != NULL) {
		*fullscreen = -1;
		if (error_ful->code == G_KEY_FILE_ERROR_KEY_NOT_FOUND ||
		    error_ful->code == G_KEY_FILE_ERROR_GROUP_NOT_FOUND) {
			FM_ULOG_DEBUG1("Error reading fullscreen mode value : %s\n",
				     error_ful->message);
		} else {
			FM_ULOG_ERR("Error reading fullscreen mode value : %s\n",
				    error_ful->message);
		}
		g_key_file_set_integer(global_ofm->keyfile,
				       TOOLBAR_STATUS, FULLSCREEN_TOOLBAR, 1);
	}
	//FM_ULOG_DEBUG1("Phone Connection Status %d\n",value);
	if (error_nor != NULL && error_ful != NULL) {
		g_error_free(error_ful);
		g_error_free(error_nor);
		return FALSE;
	}
	if (error_ful != NULL) {
		g_error_free(error_ful);
	}
	if (error_nor != NULL) {
		g_error_free(error_nor);
	}
	DEBUG_PRINT_END;
	return TRUE;
}

/**
 * osso_fm_set_toolbar_status:
 * Method to set the tool bar status to the preferences file.
 * @param normal in param for normal screen toolbar status
 * @param fullscreen in param for fullscreen mode toolbar status
 * @return void.
 */
void osso_fm_set_toolbar_status(gint normal, gint fullscreen)
{
	DEBUG_PRINT_START;
	if (normal != -1) {
		g_key_file_set_integer(global_ofm->keyfile,
				       TOOLBAR_STATUS, NORMAL_VIEW_TOOLBAR, normal);
	}
	if (fullscreen != -1) {
		g_key_file_set_integer(global_ofm->keyfile,
				       TOOLBAR_STATUS, FULLSCREEN_TOOLBAR, fullscreen);
	}
	DEBUG_PRINT_END;
	return;
}

/**
 * osso_fm_get_config_value_for_path:
 * Method to read preferences key file to get the values for the path specified.
 * GtkFilePath varient of osso_fm_get_config_value_for_texturi
 * @param path GtkFilePath 
 * @param showthumbnails TRUE if thumbnails view
 * @param key represents the column on which sort is to be performed.
 * @param order sorting type ascending / descending
 * @return 0 on success.
 */
gint
osso_fm_get_config_value_for_path(const GtkFilePath * path,
				  gboolean * showthumbnails,
				  HildonFileSelectionSortKey * key, GtkSortType * order)
{
	gchar *path_texturi = NULL;
	gint retval;

	DEBUG_PRINT_START;

	path_texturi = osso_fm_path_to_texturi(path);
	retval = osso_fm_get_config_value_for_texturi(path_texturi,
							showthumbnails,
							key,
							order);
	g_free(path_texturi);

	DEBUG_PRINT_END;
	return 0;
}

/**
 * osso_fm_get_config_value_for_texturi:
 * Method to read preferences key file to get the values for the path specified.
 * @param path gchar * text uri of the path
 * @param showthumbnails TRUE if thumbnails view
 * @param key represents the column on which sort is to be performed.
 * @param order sorting type ascending / descending
 * @return 0 on success.
 */
gint
osso_fm_get_config_value_for_texturi(const gchar * path_texturi,
				  gboolean * showthumbnails,
				  HildonFileSelectionSortKey * key, GtkSortType * order)
{
	gint value = 0;

	DEBUG_PRINT_START;
	FM_ULOG_DEBUG1("Reading settings for %s\n", path_texturi);

	value = g_key_file_get_integer(global_ofm->keyfile, FOLDER_STATE_GROUP, path_texturi, NULL);

	FM_ULOG_DEBUG1("VALUE : %d\n", value);

	if (value & (THUMBNAIL_PARSE)) {
		*showthumbnails = TRUE;
	}
	if (value & (ASCENDING_PARSE)) {
		*order = 1;
	}

	*key = value & 0x0F;

	FM_ULOG_DEBUG1("GET SORT KEY : %d,  GET SORT_ORDER : %d\n", *key, *order);

	DEBUG_PRINT_END;
	return 0;
}

/**
 * osso_fm_set_config_value_for_texturi:
 * Method to set preferences key file with the values for the path specified.
 * @param path gchar * text uri of the path
 * @param showthumbnails TRUE if thumbnails view
 * @param key represents the column on which sort is to be performed.
 * @param order sorting type ascending / descending
 * @return 0 on success.
 */
gint
osso_fm_set_config_value_for_texturi(const gchar * path_texturi,
				  const gboolean showthumbnails,
				  const HildonFileSelectionSortKey key, const GtkSortType order)
{
	gint value = 0;
	DEBUG_PRINT_START;

	if (showthumbnails == TRUE) {
		value = (THUMBNAIL_PARSE);
	}
	if (order == 1) {
		value = value + (ASCENDING_PARSE);
	}
	value = value + (gint) key;

	FM_ULOG_DEBUG1("SORT KEY : %d,  SORT_ORDER : %d\n", key, order);
	if (value > 0) {
		FM_ULOG_DEBUG1("Write settings for %s [%4X]\n", path_texturi, value);
		g_key_file_set_integer(global_ofm->keyfile,
				       FOLDER_STATE_GROUP, path_texturi, value);
	} else {
		FM_ULOG_DEBUG1("Delete settings for %s [%4X]\n", path_texturi, value);
		g_key_file_remove_key(global_ofm->keyfile, FOLDER_STATE_GROUP, path_texturi, NULL);
	}
	DEBUG_PRINT_END;
	return 0;
}
/**
 * osso_fm_set_config_value_for_path:
 * Method to set preferences key file with the values for the path specified.
 * GtkFilePath varient 
 * @param path GtkFilePath
 * @param showthumbnails TRUE if thumbnails view
 * @param key represents the column on which sort is to be performed.
 * @param order sorting type ascending / descending
 * @return 0 on success.
 */
gint
osso_fm_set_config_value_for_path(const GtkFilePath * path,
				  const gboolean showthumbnails,
				  const HildonFileSelectionSortKey key, const GtkSortType order)
{
	gint retval = 0;
	gchar *path_texturi = NULL;
	DEBUG_PRINT_START;

	if(path == NULL)
	{
		FM_ULOG_ERR("osso_fm_set_config_value_for_path: path == NULL\n");
		return -1;
	}
	path_texturi = osso_fm_path_to_texturi(path);
	retval = osso_fm_set_config_value_for_texturi(path_texturi,showthumbnails,key,order);
	g_free(path_texturi);

	DEBUG_PRINT_END;
	return 0;
}


/**
 * osso_fm_save_key_file:
 * Method to save the preferences file
 * @return void
 **/
void osso_fm_save_key_file()
{
	gchar *keypath = NULL;
	GnomeVFSHandle *handle = NULL;
	GnomeVFSResult result = 0;
	gint length = 0;
	GnomeVFSFileSize length_written = 0;
	gchar *keydata = NULL;
	//GError *error = NULL;
	DEBUG_PRINT_START;

	if (global_ofm->keyfile == NULL) {
		FM_ULOG_ERR("osso_fm_save_key_file:keyfile == NULL");
		return;
	}
	if (!ensure_preference_file_path_exists()) {
		FM_ULOG_ERR("Error Lookingup/creating keyfilepath\n");
		return;
	}
	keypath = gnome_vfs_expand_initial_tilde(PREFERENCES_FILE);
	FM_ULOG_DEBUG1("keypath === %s\n", keypath);
	result = gnome_vfs_create(&handle, keypath,
				  GNOME_VFS_OPEN_WRITE | GNOME_VFS_OPEN_RANDOM, FALSE, 0644);
	FM_ULOG_DEBUG1("Gnome RETURN : %s\n", gnome_vfs_result_to_string(result));

	if (result == GNOME_VFS_OK) {

		FM_ULOG_DEBUG1("file creation succesful\n");
		keydata = g_key_file_to_data(global_ofm->keyfile, &length, NULL);	//&error);
		FM_ULOG_DEBUG1("Keydata \n%s\n", keydata);
		gnome_vfs_write(handle, keydata, length, &length_written);
		gnome_vfs_close(handle);
	} else {
		FM_ULOG_DEBUG1("file creation failed\n");
	}
	g_free(keydata);
	g_free(keypath);
	DEBUG_PRINT_END;
	return;
}

/**
 * osso_fm_close_key_file:
 * Method to close the key file
 * @return void
 **/
void osso_fm_close_key_file()
{
	if (global_ofm->keyfile != NULL) {
		g_key_file_free(global_ofm->keyfile);
		global_ofm->keyfile = NULL;
	}
}

/**
 * osso_fm_clipboard_init:
 * Method to initialize the clipboard
 * @return void
 */
void osso_fm_clipboard_init()
{
	DEBUG_PRINT_START;
	copied_files_atom = gdk_atom_intern(CLIPBOARD_FILES, FALSE);
	DEBUG_PRINT_END;
	return;
}

/**
 * osso_fm_selection_to_string:
 * Method to convert selection to string
 * @param selection 
 * @param cut
 * @return gchar 
 **/
static gchar *osso_fm_selection_to_string(GSList ** selection, gboolean cut)
{
	GString *selectionpaths = NULL;
	gchar *result = NULL;
	GSList *cursel = NULL;
	gchar *path_texturi = NULL;
	DEBUG_PRINT_START;

	selectionpaths = g_string_new(cut ? "cut" : "copy");

	for (cursel = *selection; cursel != NULL; cursel = g_slist_next(cursel)) {
		if (cursel->data != NULL) {
			g_string_append_c(selectionpaths, '\n');
			path_texturi = osso_fm_path_to_texturi(cursel->data);
			g_string_append(selectionpaths, path_texturi);
			g_free(path_texturi);
		}
	}
	result = selectionpaths->str;
	g_string_free(selectionpaths, FALSE);

	DEBUG_PRINT_END;
	return result;
}

static gchar *osso_fm_glist_to_string(GList *list, gboolean cut)
{
	GString *s;
	gchar *result;

	s = g_string_new(cut ? "cut" : "copy");

	for (; list != NULL; list = g_list_next(list)) {
		if (list->data != NULL) {
			g_string_append_c(s, '\n');
			g_string_append(s, (gchar*)list->data);
		}
	}
	result = s->str;
	g_string_free(s, FALSE);

	return result;
}

/**
 * osso_fm_get_clipboard_cb:
 * callback function which will be called when clipboard data is required by someone
 * @param clipboard
 * @param selectiondata
 * @param info
 * @param user_data
 * @return void
 **/
static void
osso_fm_get_clipboard_cb(GtkClipboard * clipboard,
			 GtkSelectionData * selectiondata, guint info, gpointer user_data)
{
	gchar *data = NULL;
	DEBUG_PRINT_START;
	if ((selectiondata == NULL) || (copied_files_atom == NULL) || (user_data == NULL)) {
		FM_ULOG_ERR("Error validation input parameters\n");
		return;
	}
	data = (gchar *) user_data;
	gtk_selection_data_set(selectiondata, copied_files_atom, 8,	//TODO : no of bits per unit had to be checked.
			       data, g_utf8_strlen(data, -1));
	DEBUG_PRINT_END;
}
/**
 * osso_fm_clear_clipboard_cb:
 * Callback function which will be called when clipboad contents are no longer required
 * @param clipboard
 * @param user_data
 * @return void
 */

static void osso_fm_clear_clipboard_cb(GtkClipboard * clipboard, gpointer user_data)
{
	DEBUG_PRINT_START;
	g_free((gchar *) user_data);
	//    user_data = NULL;
	DEBUG_PRINT_END;
}

/**
 * osso_fm_get_clipboard:
 * Method to get the default clipboard
 * @param widget
 * @return GtkClipboard
 **/
static GtkClipboard *osso_fm_get_clipboard(GtkWidget * widget)
{
	FM_ULOG_DEBUG1("DISPLAY : %s\n", gdk_display_get_name(gdk_display_get_default()));
	return gtk_clipboard_get(GDK_NONE);	//SAI

}

void osso_fm_clipboard_clear()
{
	gtk_clipboard_clear(osso_fm_get_clipboard(GTK_WIDGET(global_ofm->window)));
}
/**
 * osso_fm_copy_to_clipboard
 * Method to copy contents to clipboard
 * @param app
 * @param selection
 * @param cut
 * @return void
 **/
void osso_fm_copy_to_clipboard(GtkWidget * app, GSList * selection, gboolean cut)
{
	GtkClipboard *clipboard;
	gchar *selectionlist = NULL;
	DEBUG_PRINT_START;
	selectionlist = osso_fm_selection_to_string(&selection, cut);
	if (selectionlist != NULL) {
		clipboard = osso_fm_get_clipboard(GTK_WIDGET(app));

		gtk_clipboard_set_with_data(clipboard,
					    clipboard_targets,
					    G_N_ELEMENTS(clipboard_targets),
					    osso_fm_get_clipboard_cb,
					    osso_fm_clear_clipboard_cb, selectionlist);

		gtk_clipboard_set_can_store (clipboard,
					     clipboard_targets,
					     G_N_ELEMENTS (clipboard_targets));
		gtk_clipboard_store (clipboard);
	}
	DEBUG_PRINT_END;
}

void osso_fm_copy_glist_to_clipboard(GtkWidget *app, GList *list, gboolean cut)
{
	GtkClipboard *clipboard;
	gchar *selectionlist;
	selectionlist = osso_fm_glist_to_string(list, cut);

	if (selectionlist != NULL) {
		clipboard = osso_fm_get_clipboard(GTK_WIDGET(app));

		gtk_clipboard_set_with_data(
			clipboard,
			clipboard_targets,
			G_N_ELEMENTS(clipboard_targets),
			osso_fm_get_clipboard_cb,
			osso_fm_clear_clipboard_cb, selectionlist);

		gtk_clipboard_set_can_store (clipboard,
					     clipboard_targets,
					     G_N_ELEMENTS (clipboard_targets));
		gtk_clipboard_store (clipboard);
	}
}

/**
 * osso_fm_get_selection_from_string:
 * Method to convert string to a GSList 
 * @param string
 * @param cut
 * @return GSList
 **/
static GSList *osso_fm_get_selection_from_string(gchar * string, gboolean * cut)
{
	GSList *selection = NULL;
	gchar **paths = NULL;
	gint i = 0;
	DEBUG_PRINT_START;

	if (string == NULL) {
		FM_ULOG_ERR("Error validating input params\n");
		return NULL;
	}

	paths = g_strsplit(string, "\n", 0);
	if (paths[0] == NULL) {
		return NULL;
	}
	if (g_strcasecmp(paths[0], "cut") == 0) {
		*cut = TRUE;
	} else if (g_strcasecmp(paths[0], "copy") == 0) {
		*cut = FALSE;
	} else {
		return NULL;
	}

	for (i = 1; paths[i] != NULL; i++) {
		selection = g_slist_append(selection, osso_fm_texturi_to_path(paths[i]));
	}

	g_strfreev(paths);

	DEBUG_PRINT_END;
	return selection;
}

/**
 * osso_fm_get_list_from_clipboard:
 * Method to get the contents of clipboard for paste
 * @param app
 * @param selection
 * @param cut
 * @return void
 **/
void osso_fm_get_list_from_clipboard(GtkWidget * app, GSList ** selection, gboolean * cut)
{
	GtkSelectionData *selectiondata = NULL;
	DEBUG_PRINT_START;
	selectiondata = gtk_clipboard_wait_for_contents(osso_fm_get_clipboard(GTK_WIDGET(app)),
							copied_files_atom);
	if (selectiondata == NULL) {
		FM_ULOG_DEBUG1("selectiondata is null\n");
		*selection = NULL;
		DEBUG_PRINT_END;
		return;
	}
	if ((selectiondata->type != copied_files_atom) || (selectiondata->length <= 0)) {
		*selection = NULL;
	} else {
		selectiondata->data[selectiondata->length] = '\0';
		*selection = osso_fm_get_selection_from_string(selectiondata->data, cut);
	}
	//    if (erase == TRUE){
	//            osso_fm_copy_to_clipboard (fs, NULL, *cut);
	//    }
	gtk_selection_data_free(selectiondata);
	DEBUG_PRINT_END;
}

/**
 * remove_safe_folders_from_gslist:
 * This method removes the safe folders from the GSlist
 * @param fileselection HildonFileSelection
 * @param selection GSList 
 * @param nonexistant gint * to store the count of non existant items
 * @param menucheck gboolean states if the selection has to be removed or not
 * @return gint count of removed items
 **/
gint
remove_safe_folders_from_gslist(HildonFileSelection * fileselection,
				GSList ** selection, gint *nonexistant, gboolean menucheck)
{
	gint error_items = 0;
	gint total_items = 0;
	GSList *cursel = NULL;
	GSList *nextsel = NULL;
	//gint foldtype = 0;
	DEBUG_PRINT_START;
	if (*selection == NULL) {
		FM_ULOG_ERR("Invalid Parameters\n");
		return 0;
	}

	for (cursel = *selection; cursel != NULL; cursel = nextsel) {
		total_items++;
		nextsel = cursel->next;
		switch (osso_fm_fs_get_path_type(fileselection, cursel->data)) {
		case HILDON_FILE_SYSTEM_MODEL_UNKNOWN:
		case HILDON_FILE_SYSTEM_MODEL_LOCAL_DEVICE:
		case HILDON_FILE_SYSTEM_MODEL_MMC:
		case HILDON_FILE_SYSTEM_MODEL_GATEWAY:
		case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_DOCUMENTS:
		case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_IMAGES:
		case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_SOUNDS:
		case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_VIDEOS:
		case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_GAMES:
			error_items++;
			if (menucheck == FALSE) {
				if (fileselection != NULL) {
					hildon_file_selection_unselect_path(fileselection,
									    (GtkFilePath *) cursel->
									    data);
				}
				*selection = g_slist_remove(*selection, cursel->data);
			}
			break;
		case HILDON_FILE_SYSTEM_MODEL_FOLDER:
			if(is_path_readonly(cursel->data,FALSE))
			{
				error_items++;
				if (menucheck == FALSE) {
					if (fileselection != NULL) {
						hildon_file_selection_unselect_path(fileselection,
										    (GtkFilePath *)
										    cursel->data);
					}
					*selection = g_slist_remove(*selection, cursel->data);
				}
				break;
			}
		case HILDON_FILE_SYSTEM_MODEL_FILE:
			break;
		}
		if (menucheck == TRUE) {
			if (total_items > 0 && error_items != total_items) {
				break;
			}
		}
	}

	DEBUG_PRINT_END;
	return error_items;
}


/**
 * on_select_path_timer:
 * To Select The Newly Created Folder using g_timeout_add .
 * @param data  It is a Single LinkList, where the First  List Contains
 * Address of HildonFileSelection & second List contains the address of
 * newly created path.
 * @return gboolean
 **/
gboolean on_select_path_timer(gpointer data)
{
	static int cnt = 0;
	GSList *List = NULL;
	GSList *List_Tmp = NULL;
	//GSList *new_sel = NULL;
	gchar *parentpath = NULL;
	GError *error = NULL;
	HildonFileSystemModel *fsmodel = NULL;
	GtkTreeIter iter;

	DEBUG_PRINT_START;
	if (data == NULL) {
		FM_ULOG_ERR("%s DATA IS NULL \n", __FUNCTION__);
		DEBUG_PRINT_END;
		return FALSE;
	}
	List = (GSList *) data;

	if ((List == NULL) ||
	    (List->data == NULL) ||
	    (List->next == NULL) ||
	    (List->next->data == NULL) ||
	    (List->next->next == NULL) || (List->next->next->data == NULL)) {
		FM_ULOG_ERR(" INVALID PARAMETERS \n");
		DEBUG_PRINT_END;
		cnt = 0;
		return FALSE;
	}
	
	g_object_get(List->data, "model", &fsmodel, NULL);

	if(hildon_file_system_model_load_path(fsmodel,List->next->data,&iter)) {
		if (g_strcasecmp(List->next->next->data, "CONTENT_PANE") == 0) {
			gdk_threads_enter();
			List_Tmp = hildon_file_selection_get_selected_paths(
					(HildonFileSelection *) List->data);

			hildon_file_selection_select_path((HildonFileSelection *) List->data,
							  (GtkFilePath *) List->next->data, &error);
			gdk_threads_leave();
			if (error == NULL) {
				cnt = 0;
				if ((List_Tmp != NULL) && (List_Tmp->data != NULL)) {
					gdk_threads_enter();
					hildon_file_selection_unselect_path(
							(HildonFileSelection *) List->data,
							(GtkFilePath *) List_Tmp->data);
					gdk_threads_leave();
				}
				g_free(List->next->data);
				g_free(List->next->next->data);
				g_slist_free(List);
				//TODO check and uncomment g_slist_free(List_Tmp);
				cnt = 0;
				DEBUG_PRINT_END;
				return FALSE;
			}
			FM_ULOG_DEBUG1("PATH TO SELECT ERROR %s\n", error->message);
			g_error_free(error);
			if (parentpath != NULL) {
				g_free(parentpath);
			}
		} else			//SELECTION SHOULD DONE IN NAV PANE
		{
			gdk_threads_enter();
			hildon_file_selection_set_current_folder(
						(HildonFileSelection *) List->data,
						 (GtkFilePath *) List->next->data, &error);
			gdk_threads_leave();
			if (error == NULL) {
				g_free(List->next->data);
				g_free(List->next->next->data);
				g_slist_free(List);
				cnt = 0;
				DEBUG_PRINT_END;
				return FALSE;
			}
			FM_ULOG_DEBUG1("PATH TO SET CURRENT ERROR %s\n", error->message);
			g_error_free(error);

		}

	} 
	cnt++;
	if (cnt >= 5) {
		g_free(List->next->data);
		g_slist_free(List);
		cnt = 0;
		DEBUG_PRINT_END;
		return FALSE;
	} else {
		DEBUG_PRINT_END;
		return TRUE;
	}
	DEBUG_PRINT_END;
}

gboolean print_info(gpointer str)
{
	fm_infoprint(GTK_WINDOW(global_ofm->window), str);
	return FALSE;
}

/**
 * fm_infoprint:
 * This is a wrapper function for gtk_infoprint
 * @param window pointer to GtkWindow
 * @param text gchar * text for info print
 * @return void
 **/
void fm_infoprint(const GtkWindow * window, const gchar *text)
{
  if (text == NULL)
    {
      FM_ULOG_ERR("fm_infoprint:text = NULL\n");
      return;
    }

  if(window == NULL)
    window = GTK_WINDOW (global_ofm->window);

  /* XXX - We always use NULL as the parent for info banners.
           Otherwise, they might get obscured by menus.
  */

  hildon_banner_show_information (NULL, NULL, text);
}

/**
 * fm_infonote:
 * This function displays hildon_note_new_information with given text and the button text
 * @param window pointer to the parent window
 * @param text dialog text to be displayed
 * @param but_text text to be displayed on button
 * @return void
 **/
void fm_infonote(const GtkWindow * window, const gchar *text,const gchar *but_text)
{
	GtkWidget *infnote = NULL;
	if(text == NULL)
	{
		FM_ULOG_ERR("fm_infonote:text = NULL\n");
		return;
	}
	FM_ULOG_INFO("fm_infonote:Info Note with %s\n",text);
	if(window == NULL) {
		infnote = hildon_note_new_information(GTK_WINDOW (global_ofm->window),text);
	} else {
		infnote = hildon_note_new_information(GTK_WINDOW(window),text);
	}
	
	if(but_text != NULL) {
		hildon_note_set_button_text((HildonNote *) infnote,but_text);
	}
	if(infnote == NULL) {
		FM_ULOG_ERR("fm_infonote:Unable to create dialog\n");
		return;
	}
	gtk_widget_show(infnote);
	gtk_dialog_run((GtkDialog *) infnote);
	gtk_widget_destroy(infnote);
	return;
}

/**
 * fm_confnote:
 * Displays an conf note with the following info
 * @param window : parent window (can be null)
 * @param text : confirmation text
 * @param but1_text : text to be displayed on first button
 * @param but1_text : response to be emitted for first button
 * @param but2_text : text to be displayed on second button
 * @param but2_text : response to be emitted for second button
 * @param sensitive : is the conf note sensitive to selection change
 * 			TRUE if the dialog has to close on selection change
 * @return gint response of the conf note
 */ 
gint fm_confnote(const GtkWindow * window, const gchar *title, const gchar *text,
		const gchar *but1_text,const gint but1_res,
		const gchar *but2_text,const gint but2_res,const gboolean sensitive)
{
	GtkWidget *confirmnote = NULL;
	gint response = 0;
	confirmnote = hildon_note_new_confirmation_add_buttons(GTK_WINDOW(global_ofm->window),text ,
								but1_text, but1_res,
								but2_text,but2_res,
								NULL);
	if(sensitive){
		register_selection_change_sensitive_dialog(global_ofm,confirmnote);
	}
	response = gtk_dialog_run(GTK_DIALOG(confirmnote));
	if(sensitive){
		unregister_selection_change_sensitive_dialog(global_ofm,confirmnote);
	}
	gtk_widget_destroy(confirmnote);
	return response;
}
/**
 * register_selection_change_sensitive_dialog:
 * @param ofm pointer to OSSOFileManager structure
 * @param dialog GtkWidget dialog to register for closer on selection change
 * @return void
 **/
void register_selection_change_sensitive_dialog(OSSOFileManager *ofm,GtkWidget *dialog)
{
	ofm->open_dialog_ptr = dialog;
	ofm->open_dialog_sel_change_sensitive = TRUE;
	ofm->open_dialog_sensitive_on_selection = gtk_file_paths_copy(ofm->si->current_selection);
}
/**
 * unregister_selection_change_sensitive_dialog:
 * @param ofm pointer to OSSOFileManager structure
 * @param dialog GtkWidget dialog to unregister for closer on selection change
 * @return void
 **/
void unregister_selection_change_sensitive_dialog(OSSOFileManager *ofm,GtkWidget *dialog)
{
	if(ofm->open_dialog_ptr != dialog)
	{
		FM_ULOG_ERR("UNREGISTERING a non REGISTERED DIALOG\n");
	}
	ofm->open_dialog_ptr = NULL;
	ofm->open_dialog_sel_change_sensitive = FALSE;
	gtk_file_paths_free(ofm->open_dialog_sensitive_on_selection);
}

/**
 * close_selection_change_sensitivity_open_dialogs:
 * Closes the registered dialogs. called when selection changes
 * @param ofm pointer to OSSOFileManager structure
 * @param dialog GtkWidget dialog to unregister for closer on selection change
 * @return void
 **/
void close_selection_change_sensitivity_open_dialogs(OSSOFileManager *ofm)
{
	GSList *cursel = NULL;
	GSList *prev = NULL, *cur = NULL;
	if(ofm->open_dialog_ptr != NULL && ofm->open_dialog_sel_change_sensitive)
	{
		if ((hildon_file_selection_get_active_pane(ofm->fs) ==
		     HILDON_FILE_SELECTION_PANE_CONTENT)) {
			cursel = hildon_file_selection_get_selected_paths(ofm->fs);
		}
		else
		{
			cursel = g_slist_append(cursel, 
					hildon_file_selection_get_current_folder(ofm->fs));
		}
		for(prev = ofm->open_dialog_sensitive_on_selection, cur = cursel;
				cur != NULL && prev != NULL;
				cur=cur->next, prev = prev->next)
		{
			if(gtk_file_path_compare(cur->data,prev->data) != 0)
				break;
		}
		if(cur != NULL || prev != NULL)
		{
			FM_ULOG_INFO("CLOSING SELECTION SENSITIVE DIALOG\n");
			gtk_dialog_response(GTK_DIALOG(ofm->open_dialog_ptr),GTK_RESPONSE_NONE);
		}
		else
		{
			FM_ULOG_INFO("SELECTION NOT CHANGED TO CLOSE THE DIALOG\n");
		}
	}
}

/**
 * update_thumbnails:
 * This fuction copies/moves/deletes thumbnails based on the parameters passed.
 * this will be called when files are copied/moved/deleted.
 * @param src_list list of source files
 * @param tgt_list list of target files
 * @param xfer_action operation performed
 * @return void
 **/
void update_thumbnails(GList * src_list, GList * tgt_list, TransferAction xfer_action)
{
	GList *cur_src = NULL;
	GList *cur_tgt = NULL;
	gchar *src_txturi = NULL;
	gchar *tgt_txturi = NULL;
	DEBUG_PRINT_START;
	for (cur_src = src_list, cur_tgt = tgt_list;
	     cur_src != NULL && cur_tgt != NULL;
	     cur_src = g_list_next(cur_src), cur_tgt = g_list_next(cur_tgt)) {
		if (cur_src->data != NULL)
			src_txturi = gnome_vfs_uri_to_string(cur_src->data, 0);
		else
			continue;

		if (cur_tgt->data != NULL)
			tgt_txturi = gnome_vfs_uri_to_string(cur_tgt->data, 0);
		else if (xfer_action != XFER_DELETE) {

			if (src_txturi != NULL) {
				g_free(src_txturi);
				src_txturi = NULL;
			}
			continue;
		}

		switch (xfer_action) {
		case XFER_DUPLICATE:
		case XFER_COPY:
			if (gnome_vfs_uri_exists(cur_tgt->data)) {	//copy thumbnails
				hildon_thumbnail_factory_copy(src_txturi, tgt_txturi);
			}
			break;
		case XFER_RENAME:
		case XFER_MOVE:
		case XFER_CUTPASTE:
			if (gnome_vfs_uri_exists(cur_tgt->data)) {	//move thumbnails
				hildon_thumbnail_factory_move(src_txturi, tgt_txturi);
			}
			break;
		case XFER_DELETE:
			if (!gnome_vfs_uri_exists(cur_src->data)) {	//delete thumbnails
				hildon_thumbnail_factory_remove(src_txturi);
			}
			break;
		default:
			break;
		}
		if (src_txturi != NULL) {
			g_free(src_txturi);
			src_txturi = NULL;
		}
		if (tgt_txturi != NULL) {
			g_free(tgt_txturi);
			tgt_txturi = NULL;
		}
	}
	DEBUG_PRINT_END;
}


/**
 * get_rename_file_title:
 * The function returns rename dialog title depending on the file type
 * @param oldpath original path
 * @return title for rename dialog
 **/
gchar *get_rename_file_title(const gchar * oldpath)
{
	gchar *mimetype = NULL;
	HildonMimeCategory filetype;
	gchar *title = NULL;
	DEBUG_PRINT_START;

	mimetype = (gchar *) gnome_vfs_get_mime_type(oldpath);
	FM_ULOG_DEBUG1("MIME TYPE is %s\n", mimetype);

	filetype = hildon_mime_get_category_for_mime_type(mimetype);
	FM_ULOG_DEBUG1("Catagory for mime type is %d\n", filetype);

	switch (filetype) {
	case HILDON_MIME_CATEGORY_BOOKMARKS:
		title = g_strdup(dgettext(HILDON_LIBS_L10N_PACKAGE,"ckdg_ti_rename_bookmark"));
		break;
	case HILDON_MIME_CATEGORY_DOCUMENTS:
		title = g_strdup(dgettext(HILDON_LIBS_L10N_PACKAGE,"ckdg_ti_rename_document"));
		break;
	case HILDON_MIME_CATEGORY_EMAILS:
		title = g_strdup(dgettext(HILDON_LIBS_L10N_PACKAGE,"ckdg_ti_rename_message"));
		break;
	case HILDON_MIME_CATEGORY_IMAGES:
		title = g_strdup(dgettext(HILDON_LIBS_L10N_PACKAGE,"ckdg_ti_rename_image"));
		break;
	case HILDON_MIME_CATEGORY_AUDIO:
		title = g_strdup(dgettext(HILDON_LIBS_L10N_PACKAGE,"ckdg_ti_rename_sound"));
		break;
	case HILDON_MIME_CATEGORY_VIDEO:
		title = g_strdup(dgettext(HILDON_LIBS_L10N_PACKAGE,"ckdg_ti_rename_video"));
		break;
	case HILDON_MIME_CATEGORY_OTHER:
	case HILDON_MIME_CATEGORY_CONTACTS:
	case HILDON_MIME_CATEGORY_ALL:
	default:
		title = g_strdup(dgettext(HILDON_LIBS_L10N_PACKAGE,"ckdg_ti_rename_default"));
		break;
	}

	DEBUG_PRINT_END;

	if(mimetype != NULL) {
		g_free(mimetype);
	}
	return title;

}


/**
 * get_rename_file_existing_error:
 * The function returns the already existing file error message.
 * @param newname New path to verify
 * @return Info message to be displayed.
 */
gchar *get_rename_file_existing_error(const gchar * newname)
{
	gchar *mimetype = NULL;
	HildonMimeCategory filetype;
	gchar *title = NULL;
	DEBUG_PRINT_START;

	mimetype = (gchar *) gnome_vfs_get_mime_type(newname);
	FM_ULOG_DEBUG1("MIME TYPE is %s\n", mimetype);

	filetype = hildon_mime_get_category_for_mime_type(mimetype);
	FM_ULOG_DEBUG1("Catagory for mime type is %d\n", filetype);

	switch (filetype) {
	case HILDON_MIME_CATEGORY_BOOKMARKS:
		title = g_strdup(dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"ckdg_ib_bookmark_exists"));
		break;
	case HILDON_MIME_CATEGORY_DOCUMENTS:
		title = g_strdup(dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"ckdg_ib_document_exists"));
		break;
	case HILDON_MIME_CATEGORY_EMAILS:
		title = g_strdup(dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"ckdg_ib_message_exists"));
		break;
	case HILDON_MIME_CATEGORY_IMAGES:
		title = g_strdup(dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"ckdg_ib_image_exists"));
		break;
	case HILDON_MIME_CATEGORY_AUDIO:
		title = g_strdup(dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"ckdg_ib_sound_exists"));
		break;
	case HILDON_MIME_CATEGORY_VIDEO:
		title = g_strdup(dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"ckdg_ib_video_exists"));
		break;
	case HILDON_MIME_CATEGORY_OTHER:
	case HILDON_MIME_CATEGORY_CONTACTS:
	case HILDON_MIME_CATEGORY_ALL:
	default:
		title = g_strdup(dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"ckdg_ib_file_exists"));
		break;
	}

	DEBUG_PRINT_END;

	return title;
}

/** 
 * osso_fm_get_existing_file_count:
 * This function checks the given list of files/folders for 
 * existance and returns the count 
 * of existing files/folders
 * @param list		Pointer to the list of files/folders
 * @return gint		No of existing files/folders
 **/

gint osso_fm_get_existing_file_count(GList * list)
{
	int count = 0;
	GList *cur;

	DEBUG_PRINT_START;

	for (cur = list, count = 0; cur != 0; cur = cur->next) {
		if (gnome_vfs_uri_exists(cur->data))
			count++;
	}
	DEBUG_PRINT_END;
	return count;
}

/** 
 * osso_fm_get_existing_file_count_from_gslist:
 * This function checks the given list of files/folders for 
 * existance and returns the count 
 * of existing files/folders
 * @param list		Pointer to the list of files/folders
 * @return gint		No of existing files/folders
 **/

gint osso_fm_get_existing_file_count_from_gslist(GSList * list, gint min_req)
{
	int count = 0;
	GnomeVFSURI *uri = NULL;
	GSList *cur;

	DEBUG_PRINT_START;
	if (list == NULL || min_req == 0) {
		FM_ULOG_ERR("osso_fm_get_existing_file_count_from_gslist:Error in param\n");
		if (min_req == 0) {
			FM_ULOG_ERR("0 minimum existing files\n");
		}
		return 0;
	}
	for (cur = list, count = 0; cur != 0; cur = cur->next) {
		uri = osso_fm_path_to_uri(cur->data);
		if (gnome_vfs_uri_exists(uri))
			count++;
		gnome_vfs_uri_unref(uri);
		if (min_req != -1 && count >= min_req) {
			return count;
		}
	}
	DEBUG_PRINT_END;
	return count;
}

/**
 * valid_mmc_name:
 * check if the name given is valid for renaming mmc or not
 * @param mmc_name name to be checked for validity
 * @parm illegals An array to be filled with the illegal characters found
 *                Will be null-terminated.
 * @parm max_illegals size of the array Must be greater than one.
 * @return gboolean TRUE if valid
 **/
gboolean
valid_mmc_name(gchar * mmc_name,
	       gchar *illegals, gint max_illegals)
{
  gint i, j;

  illegals[0] = '\0';
  for (i = 0, j = 0; mmc_name[i] && j < max_illegals - 1; i++)
    {
      /* This is not the best thing to do, but we keep it for
	 compatability: mmc_name is only partly uppercased.
      */
      mmc_name[i] = g_ascii_toupper(mmc_name[i]);

      if (mmc_name[i] != ' '
	  && !g_ascii_isalnum(mmc_name[i])
	  && !strchr (illegals, mmc_name[i]))
	{
	  illegals[j++] = mmc_name[i];
	  illegals[j] = '\0';
	}
    }

  return (j == 0);
}


glong timeval_diff(GTimeVal * end, GTimeVal * start)
{
	glong diff = 0;
	if (end == NULL || start == NULL) {
		FM_ULOG_ERR("osso_fm_timeval_diff : Error validating parameters\n");
		return 0;
	}
	if (end->tv_sec - start->tv_sec > 1000) {
		diff = (1 << 30);
	} else if (end->tv_sec - start->tv_sec < -1000) {
		diff = -(1 << 30);
	} else {
		if (end->tv_usec < start->tv_usec) {
			diff = (end->tv_sec - start->tv_sec - 1) * 1000000ul;
			diff += (1000000ul + end->tv_usec - start->tv_usec);
		} else {
			diff = (end->tv_sec - start->tv_sec) * 1000000ul;
			diff += (end->tv_usec - start->tv_usec);
		}
	}
	return diff;
}

/**
 * osso_fm_select_texturi_with_time:
 * text uri varient of osso_fm_select_path_with_time
 * @param text_uri
 * @param select specifies content pane of navigarion pane
 * @return void
 **/
void osso_fm_select_texturi_with_time(const gchar * text_uri, gboolean select)
{
	GtkFilePath *path = NULL;
	DEBUG_PRINT_START;
	path = osso_fm_texturi_to_path(text_uri);
	osso_fm_select_path_with_time(path,select);
	gtk_file_path_free(path);
	DEBUG_PRINT_END;
}

/**
 * osso_fm_select_path_with_time:
 * selects a path after some time.
 * @param path GtkFilePath to select
 * @param select specifies content pane of navigarion pane
 * @return void
 **/
void osso_fm_select_path_with_time(const GtkFilePath * path, gboolean select)
{
	GSList *List = NULL;

	DEBUG_PRINT_START;
	if (path == NULL) {
		FM_ULOG_ERR("Error validating input parameters\n");
		return;
	}
	List = g_slist_append(List, (gpointer) global_ofm->fs);
	List = g_slist_append(List, (gpointer) gtk_file_path_copy(path));
	if (select)
		List = g_slist_append(List, g_strdup("CONTENT_PANE"));
	else
		List = g_slist_append(List, g_strdup("NAVIGATION_PANE"));
	
	g_timeout_add(G_TIMEOUT, on_select_path_timer, List);
	DEBUG_PRINT_END;
}

/**
 * osso_fm_copy_uri_view_settings:
 * This function copies the view settings of a path to another path in the preference file
 * @param dst_uri
 * @param src_uri
 * @return void
 **/
void osso_fm_copy_uri_view_settings(GnomeVFSURI * dst_uri, GnomeVFSURI * src_uri)
{
	char *src = NULL, *dst = NULL;
	DEBUG_PRINT_START;
	if (src_uri == NULL || dst_uri == NULL) {
		FM_ULOG_ERR("osso_fm_copy_uri_view_settings:Error:Invalid params\n");
		return;
	}
	src = gnome_vfs_uri_to_string(src_uri, GNOME_VFS_URI_HIDE_NONE);
	if (src == NULL) {
		FM_ULOG_ERR("osso_fm_copy_uri_view_settings:Error Allocating Memory\n");
		return;
	}
	dst = gnome_vfs_uri_to_string(dst_uri, GNOME_VFS_URI_HIDE_NONE);
	if (dst == NULL) {
		FM_ULOG_ERR("osso_fm_copy_uri_view_settings:Error Allocating Memory\n");
		g_free(src);
		return;
	}
	osso_fm_copy_texturi_view_settings(dst, src);
	g_free(src);
	g_free(dst);
	DEBUG_PRINT_END;
}

/**
 * osso_fm_copy_uri_view_settings:
 * This function copies the view settings of a path to another path in the preference file
 * @param dst_uri
 * @param src_uri
 * @return void
 **/
void osso_fm_copy_texturi_view_settings(const gchar * dst, const gchar * src)
{
	gboolean showthumbnails = FALSE;
	HildonFileSelectionSortKey key = HILDON_FILE_SELECTION_SORT_NAME;
	GtkSortType order = 0;
	DEBUG_PRINT_START;
	if (src == NULL || dst == NULL) {
		FM_ULOG_ERR("osso_fm_path_uri_view_settings:Error:Invalid params\n");
		return;
	}
	osso_fm_get_config_value_for_texturi(src, &showthumbnails, &key, &order);
	osso_fm_set_config_value_for_texturi(dst, showthumbnails, key, order);
	DEBUG_PRINT_END;


}


/**
 * osso_fm_predict_dialog_width_based_on_title:
 * This is a helper function to predict a given dialog's width based on the
 * dialog title string. Note that this width is rough estimate only. This
 * function assumes that dialog title font size is double that of normal text
 * and margin on either side of title is 50 pixels.
 * 
 * @param  dialog  Pointer to dialog widget
 * @param  title   Title for the dialog
 * @return gint    Predicted width for the dialog.
 **/
gint osso_fm_predict_dialog_width_based_on_title(GtkWidget * dialog, const gchar * title)
{
	gint width = -1, height = -1;
	PangoLayout *layout = NULL;

	layout = gtk_widget_create_pango_layout(dialog, title);
	pango_layout_get_pixel_size(layout, &width, &height);
	/*Double the width and add the margin of around 50 pixels on either side */
	width = 2 * (width + 50);
	FM_ULOG_DEBUG1("Predicted dialog title width = %d\n", width);
	g_object_unref(G_OBJECT(layout));
	return width;
}

/**
 * osso_fm_predict_progress_dialog_width_based_on_text:
 * This is a helper function to predict a given dialog's width based on the
 * dialog content string. Note that this width is rough estimate only. 
 * 
 * @param  dialog  Pointer to dialog widget
 * @param  title   text in the dialog
 * @return gint    Predicted width for the dialog.
 **/
gint osso_fm_predict_progress_dialog_width_based_on_text(GtkWidget * dialog, const gchar * title)
{
	gint width = -1, height = -1;
	PangoLayout *layout = NULL;

	layout = gtk_widget_create_pango_layout(dialog, title);
	pango_layout_get_pixel_size(layout, &width, &height);
	width = width + 50;
	FM_ULOG_DEBUG1("Predicted dialog text width = %d\n", width);
	g_object_unref(G_OBJECT(layout));
	return width;
}

/**
 * osso_fm_set_operation_state:
 * Sets the current operation to the global_ofm structure
 * @param open current operation
 * @return void
 **/
void osso_fm_set_operation_state(MenuOperation oper)
{
	if (oper == OPER_NONE &&
	    global_ofm->cur_operation != OPER_DETAILS && appstate.application_untopped) {
		FM_ULOG_INFO("SET KILLABLE\n");
		hildon_program_set_can_hibernate (global_ofm->prog, TRUE);
	}
	global_ofm->cur_operation = oper;
}

/**
 * osso_fm_free_gnomevfs_uri_list:
 * A Small helper function to free list of GnomeVFSURI
 * @param list GList to be freed
 * @return void
 **/
void osso_fm_free_gnomevfs_uri_list(GList * list)
{
	GList *cur;
	for (cur = list; cur != NULL; cur = cur->next) {
		if (cur->data != NULL) {
			gnome_vfs_uri_unref(cur->data);
		}
	}
}

/**
 * osso_fm_free_string_list:
 * A Small helper function to free list of strings 
 * @param list GList to be freed
 * @return void
 **/
void osso_fm_free_string_list(GList * list)
{
	GList *cur;
	for (cur = list; cur != NULL; cur = cur->next) {
		if (cur->data != NULL) {
			g_free(cur->data);
		}
	}
}


/**
 * osso_fm_free_string_slist:
 * A Small helper function to free singly linked list of strings 
 * @param list GSList to be freed
 * @return void
 **/
void osso_fm_free_string_slist(GSList * list)
{
	GSList *cur;
	for (cur = list; cur != NULL; cur = cur->next) {
		if (cur->data != NULL) {
			g_free(cur->data);
		}
	}
}




//BEG: Validation functions

//extern HildonFileSystemModel *fsmodel;

/**
 * init_selection_state:
 * Initalize the selection state structure
 * @return void
 **/
void init_selection_state()
{
	selectionstate.selection = FALSE;
	selectionstate.ronly_parent = FALSE;
	selectionstate.no_files = 0;
	selectionstate.no_folders = 0;
	selectionstate.no_ronly_files = 0;
	selectionstate.no_ronly_folders = 0;
	selectionstate.no_safe_folders = 0;
	selectionstate.no_devices = 0;
	selectionstate.no_mmc = 0;
	selectionstate.error = 0;
}

void print_selection_state()
{
	DEBUG_PRINT_START;
	FM_ULOG_DEBUG1
	    ("selectionstate.selection = %d \n selectionstate.ronly_parent = %d \n selectionstate.no_files = %d \n selectionstate.no_folders = %d \n selectionstate.no_ronly_files = %d \n selectionstate.no_ronly_folders = %d \n selectionstate.no_safe_folders = %d \n selectionstate.no_devices = %d\n selectionstate.no_mmc = %d\n selectionstate.error  = %d\n",
	     (gint) selectionstate.selection, (gint) selectionstate.ronly_parent,
	     (gint) selectionstate.no_files, (gint) selectionstate.no_folders,
	     (gint) selectionstate.no_ronly_files, (gint) selectionstate.no_ronly_folders,
	     (gint) selectionstate.no_safe_folders, (gint) selectionstate.no_devices,
	     (gint) selectionstate.no_mmc, (gint) selectionstate.error);
	DEBUG_PRINT_END;

}


/**
 * get_selection_type:
 * Prepares the SelectionType as per the selection state structure
 * @return SelectionType
 **/
SelectionType get_selection_type()
{
	if (selectionstate.error == ERROR_ALLOCATING_MEMORY) {
		return ERROR_ALLOCATING_MEMORY;
	}
	if ((selectionstate.no_files == 0) &&
	    (selectionstate.no_folders == 0) && (selectionstate.error == ERR_READING_FILE_INFO)) {
		return ERROR_READING_FILE_INFO;
	}
	if (selectionstate.no_devices > 0) {
		if (selectionstate.no_ronly_folders > 0) {
			return RONLY_DEVICE_SELECTED;
		}
		return DEVICE_SELECTED;
	}
	if (selectionstate.no_folders + selectionstate.no_files == 1) {
		if (selectionstate.no_mmc > 0) {
			if (selectionstate.no_ronly_folders > 0) {
				return RONLY_MMC_SELECTED;
			}
			return MMC_SELECTED;
		}
	}
	if (selectionstate.no_safe_folders > 0) {
		if ((selectionstate.no_ronly_files == 1) && (selectionstate.no_safe_folders == 1)) {
			return SINGLE_RONLY_FOLDER_SELECTED;
		} else if ((selectionstate.no_ronly_files > 1) &&
			   (selectionstate.no_safe_folders > 1)) {
			return MULTIPLE_RONLY_FOLDERS_SELECTED;
		}
		return SAFE_FOLDER_SELECTED;
	}
	if ((selectionstate.no_files == 1) && (selectionstate.no_folders == 0)) {
		if (selectionstate.ronly_parent == TRUE) {
			return SINGLE_PRONLY_FILE_SELECTED;
		} else if (selectionstate.no_ronly_files == 1) {
			return SINGLE_RONLY_FILE_SELECTED;
		}
		return SINGLE_FILE_SELECTED;
	}
	if ((selectionstate.no_files == 0) && (selectionstate.no_folders == 1)) {
		if (selectionstate.ronly_parent == TRUE) {
			return SINGLE_PRONLY_FOLDER_SELECTED;
		} else if (selectionstate.no_ronly_folders == 1) {
			return SINGLE_RONLY_FOLDER_SELECTED;
		}
		return SINGLE_FOLDER_SELECTED;
	}
	if ((selectionstate.no_files > 1) && (selectionstate.no_folders == 0)) {
		if (selectionstate.ronly_parent == TRUE) {
			return MULTIPLE_PRONLY_FILES_SELECTED;
		} else if (selectionstate.no_ronly_files == selectionstate.no_files) {
			return MULTIPLE_RONLY_FILES_SELECTED;
		} else if (selectionstate.no_ronly_files > 0) {
			return MULTIPLE_FILES_WITH_RONLY_SELECTED;
		}
		return MULTIPLE_FILES_SELECTED;
	}
	if ((selectionstate.no_files == 0) && (selectionstate.no_folders > 1)) {
		if (selectionstate.ronly_parent == TRUE) {
			return MULTIPLE_PRONLY_FOLDERS_SELECTED;
		} else if (selectionstate.no_ronly_folders == selectionstate.no_folders) {
			return MULTIPLE_RONLY_FOLDERS_SELECTED;
		} else if (selectionstate.no_ronly_folders > 0) {
			return MULTIPLE_FOLDERS_WITH_RONLY_SELECTED;
		}
		return MULTIPLE_FOLDERS_SELECTED;
	}
	if ((selectionstate.no_files == 1) && (selectionstate.no_folders == 1)) {
		if (selectionstate.ronly_parent == TRUE) {
			return SINGLE_FILE_SINGLE_FOLDER_WITH_PRONLY_SELECTED;
		} else if (selectionstate.no_ronly_folders > 0) {
			return SINGLE_FILE_SINGLE_FOLDER_WITH_RONLY_SELECTED;
		}
		return SINGLE_FILE_SINGLE_FOLDER_SELECTED;
	}
	if ((selectionstate.no_files == 1) && (selectionstate.no_folders > 1)) {
		if (selectionstate.ronly_parent == TRUE) {
			return SINGLE_FILE_MULTIPLE_FOLDERS_WITH_PRONLY_SELECTED;
		} else if (selectionstate.no_ronly_folders > 0) {
			return SINGLE_FILE_MULTIPLE_FOLDERS_WITH_RONLY_SELECTED;
		}
		return SINGLE_FILE_MULTIPLE_FOLDERS_SELECTED;
	}
	if ((selectionstate.no_files > 1) && (selectionstate.no_folders == 1)) {
		if (selectionstate.ronly_parent == TRUE) {
			return MULTIPLE_FILES_SINGLE_FOLDER_WITH_PRONLY_SELECTED;
		} else if (selectionstate.no_ronly_folders > 0) {
			return MULTIPLE_FILES_SINGLE_FOLDER_WITH_RONLY_SELECTED;
		}
		return MULTIPLE_FILES_SINGLE_FOLDER_SELECTED;
	}
	if ((selectionstate.no_files > 1) && (selectionstate.no_folders > 1)) {
		if (selectionstate.ronly_parent == TRUE) {
			return MULTIPLE_FILES_MULTIPLE_FOLDERS_WITH_PRONLY_SELECTED;
		} else if (selectionstate.no_ronly_folders > 0) {
			return MULTIPLE_FILES_MULTIPLE_FOLDERS_WITH_RONLY_SELECTED;
		}
		return MULTIPLE_FILES_MULTIPLE_FOLDERS_SELECTED;
	}
	return NO_ITEM_SELECTED;
}



/**
 * validate_selection:
 * Validates the selection in file selection widget.
 * @param fileselection Reference of file selection widget.
 * @return selection state
 **/
SelectionType validate_selection(HildonFileSelection * fileselection)
{
	DEBUG_PRINT_START;
	appstate.mainwindow.selection_type = NO_ITEM_SELECTED;

	appstate.mainwindow.selection_type = update_current_selection_state(fileselection);
	if (appstate.mainwindow.selection_type == NO_ITEM_SELECTED) {
		appstate.mainwindow.selection_type = get_selection_type();
	}
	DEBUG_PRINT_END;
	return appstate.mainwindow.selection_type;
}



/**
 * update_current_selection_state:
 * Updates selection state in file selection widget.
 * @param fileselection Reference of file selection widget.
 * @return SelectionType of the current selection
 **/
static SelectionType update_current_selection_state(HildonFileSelection * fileselection)
{
	GSList *cursel = NULL;
	GSList *selection = NULL;
	GtkFilePath *path = NULL;
	GnomeVFSFileInfo *info = NULL;
	GnomeVFSResult result = GNOME_VFS_ERROR_NOT_FOUND;
	SelectionType retval = NO_ITEM_SELECTED;
	gboolean gatewaypath = FALSE;
	HildonFileSystemModel *fsmodel = NULL;
	GtkTreeIter iter;
	HildonFileSystemModelItemType type;
	GTimeVal time_before1;
	GTimeVal time_after1;
	glong diff = 0;

	DEBUG_PRINT_START;

	if (appstate.mainwindow.checkselection == TRUE) {
		return appstate.mainwindow.selection_type;
	} else {
		//A monitor can be installed to know any background changes to the files 
		//using gnome_vfs_monitor_add 
		appstate.mainwindow.checkselection = TRUE;
	}

	if (fileselection == NULL) {
		FM_ULOG_ERR("Error validating input parameters");
		return ERROR_INPUT_PARAMETERS;
	};

	init_selection_state();

	g_stpcpy(appstate.mainwindow.currentselection, "");
	osso_fm_set_currentselection("");

	if ((hildon_file_selection_get_active_pane(fileselection) ==
	     HILDON_FILE_SELECTION_PANE_CONTENT) || (global_ofm->si->selected_all == TRUE)) {
		selection = hildon_file_selection_get_selected_paths(fileselection);
	}
	path = hildon_file_selection_get_current_folder(fileselection);

	if ((selection == NULL) && (path != NULL)) {
		selectionstate.selection = TRUE;
		FM_ULOG_DEBUG1("Selection is NULL \n");
		selection = g_slist_append(selection, gtk_file_path_copy(path));
	}

	gatewaypath = selected_location_in_gateway(fileselection);

	//checking if parent is readonly
	if ((selection != NULL) && (selection->data != NULL)) {
		if (!gatewaypath) {
			if (is_path_readonly(selection->data, TRUE)) {
				selectionstate.ronly_parent = TRUE;
			}
		}
	} else {
		FM_ULOG_ERR("update_current_selection_state:No items selected");
		gtk_file_path_free(path);
		return NO_ITEM_SELECTED;
	}

	if (gatewaypath) {
		g_object_get(G_OBJECT(fileselection), "model", &fsmodel, NULL);
		if (fsmodel == NULL) {
			FM_ULOG_ERR("Unable to get Model\n");
			return NO_ITEM_SELECTED;
		}
	}
	for (cursel = selection; cursel; cursel = g_slist_next(cursel)) {
		FM_ULOG_DEBUG1("update_current_selection_state "
			       "CURSEL PATH: %s\n", (char *) cursel->data);
		if (cursel->data &&
		    g_str_has_prefix(cursel->data, "drive://")) {
			FM_ULOG_DEBUG1("update_current_selection_state: "
				       "GnomeVFS drive\n");
			/* it's a GnomeVFS drive
			 * (magic string "drive:" comes from hildon-fm) */
			selectionstate.no_folders++;
			selectionstate.no_mmc++;
			continue;
		}

		if (gatewaypath) {
			if (!hildon_file_system_model_search_path
			    (fsmodel, cursel->data, &iter, NULL, TRUE)) {
				FM_ULOG_ERR("Unable to get iter\n");
				continue;
			}
			gtk_tree_model_get((GtkTreeModel *) fsmodel,
					   &iter, HILDON_FILE_SYSTEM_MODEL_COLUMN_TYPE, &type, -1);
			if (type == HILDON_FILE_SYSTEM_MODEL_FOLDER) {
				selectionstate.no_folders++;
			} else if (type == HILDON_FILE_SYSTEM_MODEL_FILE) {
				selectionstate.no_files++;
			} else if (type == HILDON_FILE_SYSTEM_MODEL_GATEWAY) {
				selectionstate.no_folders++;
				selectionstate.no_devices++;
			} else {
				FM_ULOG_ERR
				    ("Invalid type in HildonFileSystemModelItemType in gw %d\n",
				     type);
			}
		} else {
			if (info != NULL) {
				gnome_vfs_file_info_unref(info);
				info = NULL;
			}
			info = gnome_vfs_file_info_new();
			if (info == NULL) {
				FM_ULOG_ERR("Error allocating memory");
				selectionstate.error = ERR_ALLOCATING_MEMORY;
				//g_slist_free (selection);
				gtk_file_paths_free(selection);
				gtk_file_path_free(path);
				return ERROR_ALLOCATING_MEMORY;
			}
			result = GNOME_VFS_ERROR_NOT_FOUND;

			g_get_current_time(&time_before1);
			result = gnome_vfs_get_file_info(cursel->data, info,
							 (GNOME_VFS_FILE_INFO_FIELDS_ACCESS |
							  GNOME_VFS_FILE_INFO_DEFAULT |
							  GNOME_VFS_FILE_INFO_FOLLOW_LINKS));
			g_get_current_time(&time_after1);
			diff = timeval_diff(&time_after1, &time_before1);
			//FM_ULOG_ERR("gnome_vfs_get_file_info time %ld:%06ld\n",diff/1000000,diff%1000000);

			if (result != GNOME_VFS_OK) {	//info reading succeeds
				FM_ULOG_ERR
				    ("update_current_selection_state::Error reading file info:%s\n",
				     gnome_vfs_result_to_string(result));
				selectionstate.error = ERROR_READING_FILE_INFO;
				continue;
			}
			if (info->type == GNOME_VFS_FILE_TYPE_REGULAR) {
				selectionstate.no_files++;
				// if not gateway path
				//if (!gatewaypath){
				if ((info->permissions & GNOME_VFS_PERM_USER_READ) &&
				    !(info->permissions & GNOME_VFS_PERM_USER_WRITE)) {
					selectionstate.no_ronly_files++;
				}
				//}

			}	//else checking for a directory type
			else if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
				selectionstate.no_folders++;
				if (!gatewaypath) {
					if ((info->permissions & GNOME_VFS_PERM_USER_READ) &&
					    !(info->permissions & GNOME_VFS_PERM_USER_WRITE)) {
						selectionstate.no_ronly_folders++;
					}
				}
				switch (osso_fm_fs_get_path_type(fileselection, cursel->data)) {
				case HILDON_FILE_SYSTEM_MODEL_LOCAL_DEVICE:
				case HILDON_FILE_SYSTEM_MODEL_GATEWAY:
					selectionstate.no_devices++;
					break;
				case HILDON_FILE_SYSTEM_MODEL_MMC:
					selectionstate.no_mmc++;
					break;
				case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_DOCUMENTS:
				case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_IMAGES:
				case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_SOUNDS:
				case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_VIDEOS:
				case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_GAMES:
					selectionstate.no_safe_folders++;
					break;
				default:
					break;
				}	//end of switch
			}	//end if else if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY)
		}		//end of else of gatewaypath
	}			//end of for

	if ((selectionstate.no_folders + selectionstate.no_files) == 1) {
		FM_ULOG_DEBUG1("SELECTION : %s\n", osso_fm_path_to_texturi(selection->data));
		osso_fm_set_currentselection(osso_fm_path_to_texturi(selection->data));
	} else {
		FM_ULOG_DEBUG1("SELECTION : %s\n", osso_fm_path_to_texturi(path));
		osso_fm_set_currentselection(osso_fm_path_to_texturi(path));
	}

	//    retval = get_selection_type ();
	if (info != NULL) {
		gnome_vfs_file_info_unref(info);
		info = NULL;
	}

	if (selection != NULL) {
		gtk_file_paths_free(selection);
	}
	if (path != NULL) {
		gtk_file_path_free(path);
	}

	DEBUG_PRINT_END;
	return retval;
}

/**
 * get_selection_state_for_selection:
 * Gives the SelectionState for a given selection
 * @param selection
 * @return SelectionState 
 **/
SelectionState *get_selection_state_for_selection(GSList * selection)
{
	GSList *cursel = NULL;
	SelectionState *sel_state = NULL;
	gboolean gatewaypath = FALSE;
	GnomeVFSFileInfo *info = NULL;
	GnomeVFSResult result = GNOME_VFS_ERROR_NOT_FOUND;

	if (selection == NULL) {
		FM_ULOG_ERR("get_selection_state_for_selection:Error in input params\n");
		return NULL;
	}
	sel_state = g_new0(SelectionState, 1);
	gatewaypath = is_path_in_gateway(selection->data);
	if(!gatewaypath)
	{
		sel_state->ronly_parent = is_path_readonly(selection->data,TRUE);
	}
	for (cursel = selection; cursel; cursel = g_slist_next(cursel)) {
		FM_ULOG_DEBUG1("get_selection_state_for_selection "
			       "CURSEL PATH: %s\n", (char *) cursel->data);
		if (info != NULL) {
			gnome_vfs_file_info_unref(info);
			info = NULL;
		}
		info = gnome_vfs_file_info_new();
		if (info == NULL) {
			FM_ULOG_ERR("Error allocating memory");
			sel_state->error = ERR_ALLOCATING_MEMORY;
			//g_slist_free (selection);
			g_free(sel_state);
			return NULL;
		}
		result = GNOME_VFS_ERROR_NOT_FOUND;

		result = gnome_vfs_get_file_info(cursel->data, info,
						 (GNOME_VFS_FILE_INFO_FIELDS_ACCESS |
						  GNOME_VFS_FILE_INFO_DEFAULT |
						  GNOME_VFS_FILE_INFO_FOLLOW_LINKS));

		if (result != GNOME_VFS_OK) {	//info reading succeeds
			FM_ULOG_ERR("get_selection_state_for_selection:Error reading file info:%s",
				    gnome_vfs_result_to_string(result));
			sel_state->error = ERROR_READING_FILE_INFO;
			FM_ULOG_INFO("sel_state->error %d\n", sel_state->error);

			continue;
		}
		if (info->type == GNOME_VFS_FILE_TYPE_REGULAR) {
			sel_state->no_files++;
			// if not gateway path
			//if (!gatewaypath){
			if ((info->permissions & GNOME_VFS_PERM_USER_READ) &&
			    !(info->permissions & GNOME_VFS_PERM_USER_WRITE)) {
				sel_state->no_ronly_files++;
			}
			//}

		}		//else checking for a directory type
		else if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
			sel_state->no_folders++;
			if (!gatewaypath) {
				if ((info->permissions & GNOME_VFS_PERM_USER_READ) &&
				    !(info->permissions & GNOME_VFS_PERM_USER_WRITE)) {
					sel_state->no_ronly_folders++;
				}
			}
			switch (osso_fm_fs_get_path_type(global_ofm->fs,cursel->data)) {
			case HILDON_FILE_SYSTEM_MODEL_LOCAL_DEVICE:
			case HILDON_FILE_SYSTEM_MODEL_GATEWAY:
				sel_state->no_devices++;
				break;
			case HILDON_FILE_SYSTEM_MODEL_MMC:
				sel_state->no_mmc++;
				break;
			case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_DOCUMENTS:
			case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_IMAGES:
			case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_SOUNDS:
			case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_VIDEOS:
			case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_GAMES:
				sel_state->no_safe_folders++;
				break;
			default:
				break;
			}
		}
	}

	if (info != NULL) {
		gnome_vfs_file_info_unref(info);
		info = NULL;
	}

	DEBUG_PRINT_END;
	return sel_state;
}

/**
 * refresh_file_menu_items:
 * Function to enable/disable menu items in file menu based on the current selection
 * @param w GtkMenuItem
 * @param data gpointer
 * @return void
 **/
void refresh_file_menu_items(GtkMenuItem * w, gpointer data)
{

	FMMenuFile *file_m = NULL;
	FMMenuTools *file_t = NULL;

	DEBUG_PRINT_START;
	file_m = ((OSSOFileManager *)data)->fmmmenu->file;
	file_t = ((OSSOFileManager *)data)->fmmmenu->tools;
	update_current_selection_state(((OSSOFileManager *)data)->fs);

	if (selectionstate.error != ERR_NO_ERROR) {
		gtk_widget_set_sensitive((GtkWidget *) file_m->new_folder, FALSE);
		gtk_widget_set_sensitive((GtkWidget *) file_m->open, FALSE);
		gtk_widget_set_sensitive((GtkWidget *) file_m->rename, FALSE);
		gtk_widget_set_sensitive((GtkWidget *) file_m->delete, FALSE);
		gtk_widget_set_sensitive((GtkWidget *) file_m->send_as_email, FALSE);
		gtk_widget_set_sensitive((GtkWidget *) file_m->details, FALSE);
		gtk_widget_set_sensitive((GtkWidget *) file_t->format, FALSE);
		return;
	}
	if (osso_fm_new_folder((OSSOFileManager *)data, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) file_m->new_folder, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) file_m->new_folder, TRUE);
	}

	gtk_widget_set_sensitive((GtkWidget *) file_m->open, TRUE); //open always enabled

	if (osso_fm_rename_activate((OSSOFileManager *)data, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) file_m->rename, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) file_m->rename, TRUE);
	}

	if (osso_fm_delete_activate((gpointer) data, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) file_m->delete, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) file_m->delete, TRUE);
	}
	if (osso_send_as_email((OSSOFileManager *)data, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) file_m->send_as_email, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) file_m->send_as_email, TRUE);
	}
	if (osso_fm_send_via_bluetooth((OSSOFileManager *)data, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) file_m->send_via_bluetooth, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) file_m->send_via_bluetooth, TRUE);
	}
	if (osso_fm_details_activate(data, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) file_m->details, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) file_m->details, TRUE);
	}

	if (osso_fm_format_activate(data, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) file_t->format, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) file_t->format, TRUE);
	}

	DEBUG_PRINT_END;
	return;
}


/**
 * refresh_timer_edit_menu_items:
 * Function to enable/disable menu items in edit menu based on the current selection
 * @param w GtkMenuItem
 * @param data gpointer
 * @return void
 **/
static gboolean refresh_timer_edit_menu_items(gpointer data)
{
	FMMenuEdit *edit_m;
	
	DEBUG_PRINT_START;
	edit_m = ((OSSOFileManager *)data)->fmmmenu->edit;

	update_current_selection_state(((OSSOFileManager *)data)->fs);

	if (osso_fm_moveto (data, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) edit_m->move_to, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) edit_m->move_to, TRUE);
	}
	if (on_duplicate_activate((OSSOFileManager *)data, TRUE) 
			== FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) edit_m->duplicate, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) edit_m->duplicate, TRUE);
	}

	if (on_mark_all_activate(((OSSOFileManager *)data)->fs, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) edit_m->select_all, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) edit_m->select_all, TRUE);
	}

	refresh_edit_menu_paste_items(NULL, data);

	DEBUG_PRINT_END;

	return FALSE;
}

/**
 * refresh_timer_edit_menu_items:
 * Function to enable/disable menu items in edit menu based on the current selection
 * @param w GtkMenuItem
 * @param data gpointer
 * @return void
 **/
void refresh_edit_menu_items(GtkMenuItem * w, gpointer data)
{
	DEBUG_PRINT_START;
	refresh_timer_edit_menu_items(data);
	DEBUG_PRINT_END;
	return;
}

/**
 * refresh_edit_menu_paste_items:
 * Function to enable/disable menu items in edit menu based on the current selection
 * @param widget GtkMenuItem
 * @param data gpointer
 * @return void
 **/
static void refresh_edit_menu_paste_items(GtkWidget * widget, gpointer data)
{
	FMMenuEdit *edit_m;

	DEBUG_PRINT_START;

	edit_m = ((OSSOFileManager *)data)->fmmmenu->edit;

	if (on_paste_activate((OSSOFileManager *)data, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) edit_m->paste, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) edit_m->paste, TRUE);
	}

	if (on_cut_activate((OSSOFileManager *)data, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) edit_m->cut, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) edit_m->cut, TRUE);
	}

	if (on_copy_activate((OSSOFileManager *)data, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) edit_m->copy, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) edit_m->copy, TRUE);
	}

	DEBUG_PRINT_END;
	return;
}

/**
 * refresh_timer_toolbar_buttons:
 * Function to enable/disable toolbar buttons based on the current selection
 * @param data gpointer user data
 * @return void
 **/
static gboolean refresh_timer_toolbar_buttons(gpointer data)
{
	FMToolbar *fmtoolbar = NULL;

	DEBUG_PRINT_START;
	fmtoolbar = ((OSSOFileManager *)data)->fmtoolbar;

	update_current_selection_state(((OSSOFileManager *)data)->fs);

	if (osso_fm_delete_activate((gpointer) data, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) fmtoolbar->delete_button, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) fmtoolbar->delete_button, TRUE);
	}
	if (osso_fm_moveto(data, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) fmtoolbar->moveto_button, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) fmtoolbar->moveto_button, TRUE);
	}
	if (osso_fm_new_folder((OSSOFileManager *) data, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) fmtoolbar->new_button, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) fmtoolbar->new_button, TRUE);
	}
	if (osso_fm_details_activate((OSSOFileManager *) data, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) fmtoolbar->details_button, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) fmtoolbar->details_button, TRUE);
	}
	DEBUG_PRINT_END;
	return FALSE;
}

/**
 * refresh_toolbar_buttons:
 * Function to enable/disable toolbar buttons based on the current selection
 * @param data gpointer user data
 * @return void
 **/
void refresh_toolbar_buttons(gpointer data)
{
	DEBUG_PRINT_START;
	refresh_timer_toolbar_buttons(data);
	DEBUG_PRINT_END;
	return;
}

/**
 * refresh_edit_menu_select_items:
 * Function to enable/disable edit->select menu items based on the current selection
 * @param data gpointer user data
 * @return void
 **/
void refresh_edit_menu_select_items(GtkWidget * w, gpointer data)
{
	FMMenuEdit *edit_m = NULL;

	DEBUG_PRINT_START;
	
	edit_m = ((OSSOFileManager *)data)->fmmmenu->edit;
	
	if (on_mark_all_activate(((OSSOFileManager *)data)->fs, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) edit_m->select_all, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) edit_m->select_all, TRUE);
	}

	DEBUG_PRINT_END;
	return;
}


/**
 * refresh_view_menu_items:
 * Function to enable/disable view menu items based on the current selection
 * @param w GtkMenuItem
 * @param data gpointer user data
 * @return void
 **/
void refresh_view_menu_items(GtkMenuItem * w, gpointer data)
{
	FMMenuView *view_m;
	
	gint toolvisible;
	gboolean thumbnails = FALSE;

	DEBUG_PRINT_START;
	view_m = ((OSSOFileManager *)data)->fmmmenu->view;
	update_current_selection_state(((OSSOFileManager *)data)->fs);

	//Setting Thumbnail/List view Menu Item
	thumbnails = hildon_file_selection_get_mode(((OSSOFileManager *)data)->fs);

	g_signal_handlers_block_by_func(G_OBJECT(view_m->thumbnail),
					G_CALLBACK(on_menu_view_thumbnail_activate),
					data);
	g_signal_handlers_block_by_func(G_OBJECT(view_m->list),
					G_CALLBACK(on_menu_view_list_activate),
					data);

	if (thumbnails == TRUE) {
		gtk_check_menu_item_set_active((GtkCheckMenuItem *) view_m->thumbnail, TRUE);
	} else {
		gtk_check_menu_item_set_active((GtkCheckMenuItem *) view_m->list, TRUE);
	}

	g_signal_handlers_unblock_by_func(G_OBJECT(view_m->thumbnail),
					  G_CALLBACK(on_menu_view_thumbnail_activate),
					  data);
	g_signal_handlers_unblock_by_func(G_OBJECT(view_m->list),
					  G_CALLBACK(on_menu_view_list_activate),
					  data);

	if (selectionstate.error != ERR_NO_ERROR) {
		gtk_widget_set_sensitive((GtkWidget *) view_m->list, FALSE);
		gtk_widget_set_sensitive((GtkWidget *) view_m->thumbnail, FALSE);
		gtk_widget_set_sensitive((GtkWidget *) view_m->full_screen, FALSE);
		return;
	}


	if (on_sort_activate(((OSSOFileManager *)data)->fs, TRUE) == FALSE) {
		gtk_widget_set_sensitive((GtkWidget *) view_m->sort, FALSE);
	} else {
		gtk_widget_set_sensitive((GtkWidget *) view_m->sort, TRUE);
	}


	g_object_get(G_OBJECT(((OSSOFileManager *)data)->fmtoolbar->toolbar), 
			"visible", &toolvisible, NULL);

	FM_ULOG_DEBUG1("toolbar visibility:::::%d\n", toolvisible);

#if 0
	g_signal_handlers_block_by_func(G_OBJECT(view_m->full_screen),
					G_CALLBACK(on_menu_full_screen_activate), 
					(gpointer) ((OSSOFileManager *)data)->window);

	if (hildon_appview_get_fullscreen(((OSSOFileManager *)data)->appview)) {
		gtk_check_menu_item_set_active((GtkCheckMenuItem *) view_m->full_screen, TRUE);
	} else {
		gtk_check_menu_item_set_active((GtkCheckMenuItem *) view_m->full_screen, FALSE);
	}

	g_signal_handlers_unblock_by_func(G_OBJECT(view_m->full_screen),
					  G_CALLBACK(on_menu_full_screen_activate), 
					  (gpointer) ((OSSOFileManager *)data)->window);
#endif

	DEBUG_PRINT_END;
	return;
}

/**
 * refresh_popup_menu_items:
 * Method to update the popup menu items conditions.
 * @param app is HildonApp.
 * @param contentpane
 * @return void
 **/

static void
check_sensitive (OSSOFileManager *ofm, GtkWidget *w,
		 gboolean (*menu_check) (OSSOFileManager *, gboolean))
{
  if (menu_check (ofm, TRUE) == FALSE)
    gtk_widget_set_sensitive (w, FALSE);
  else
    gtk_widget_set_sensitive (w, TRUE);
}

static void
show_items (GtkWidget *w, ...)
{
  va_list args;

  va_start (args, w);
  while (w)
    {
      gtk_widget_show (w);
      w = va_arg (args, GtkWidget *);
    }
  va_end (args);
}

static void
hide_items (GtkWidget *w, ...)
{
  va_list args;

  va_start (args, w);
  while (w)
    {
      gtk_widget_hide (w);
      w = va_arg (args, GtkWidget *);
    }
  va_end (args);
}

static void
sync_popup_view_menu (OSSOFileManager *ofm, FMPMenuView *m)
{
  gboolean thumbnails = FALSE;
  HildonFileSelectionSortKey key = 0;
  GtkSortType order = 0;
   
  hildon_file_selection_get_sort_key (ofm->fs, &key, &order);
  thumbnails = hildon_file_selection_get_mode (ofm->fs);

  g_signal_handlers_block_by_func(G_OBJECT(m->list),
				  G_CALLBACK(on_menu_view_list_activate),
				  ofm);
  g_signal_handlers_block_by_func(G_OBJECT(m->thumbnails),
				  G_CALLBACK(on_menu_view_thumbnail_activate),
				  ofm);

  if (thumbnails == TRUE) {
    gtk_check_menu_item_set_active((GtkCheckMenuItem *) m->thumbnails,
				   TRUE);
  } else {
    gtk_check_menu_item_set_active((GtkCheckMenuItem *) m->list, TRUE);
  }
  g_signal_handlers_unblock_by_func(G_OBJECT(m->list),
				    G_CALLBACK(on_menu_view_list_activate),
				    ofm);
  g_signal_handlers_unblock_by_func(G_OBJECT(m->thumbnails),
				    G_CALLBACK(on_menu_view_thumbnail_activate),
				    ofm);

  g_signal_handlers_block_by_func(G_OBJECT(m->sort_by_name),
				  G_CALLBACK(osso_fm_sort_by_name), (gpointer) ofm->fs);
  g_signal_handlers_block_by_func(G_OBJECT(m->sort_by_type),
				  G_CALLBACK(osso_fm_sort_by_type), (gpointer) ofm->fs);
  g_signal_handlers_block_by_func(G_OBJECT(m->sort_by_date),
				  G_CALLBACK(osso_fm_sort_by_modified),
				  (gpointer) ofm->fs);
  g_signal_handlers_block_by_func(G_OBJECT(m->sort_by_size),
				  G_CALLBACK(osso_fm_sort_by_size), (gpointer) ofm->fs);


  switch (key) {
  case HILDON_FILE_SELECTION_SORT_NAME:
    gtk_check_menu_item_set_active((GtkCheckMenuItem *) m->sort_by_name, TRUE);
    break;
  case HILDON_FILE_SELECTION_SORT_TYPE:
    gtk_check_menu_item_set_active((GtkCheckMenuItem *) m->sort_by_type, TRUE);
    break;
  case HILDON_FILE_SELECTION_SORT_MODIFIED:
    gtk_check_menu_item_set_active((GtkCheckMenuItem *) m->sort_by_date, TRUE);
    break;
  case HILDON_FILE_SELECTION_SORT_SIZE:
    gtk_check_menu_item_set_active((GtkCheckMenuItem *) m->sort_by_size, TRUE);
    break;
  default:
    gtk_check_menu_item_set_active((GtkCheckMenuItem *) m->sort_by_name, TRUE);
    break;
  }
  g_signal_handlers_unblock_by_func(G_OBJECT(m->sort_by_name),
				    G_CALLBACK(osso_fm_sort_by_name),
				    (gpointer) ofm->fs);
  g_signal_handlers_unblock_by_func(G_OBJECT(m->sort_by_type),
				    G_CALLBACK(osso_fm_sort_by_type),
				    (gpointer) ofm->fs);
  g_signal_handlers_unblock_by_func(G_OBJECT(m->sort_by_date),
				    G_CALLBACK(osso_fm_sort_by_modified),
				    (gpointer) ofm->fs);
  g_signal_handlers_unblock_by_func(G_OBJECT(m->sort_by_size),
				    G_CALLBACK(osso_fm_sort_by_size),
				    (gpointer) ofm->fs);


}

void refresh_popup_menu_items(OSSOFileManager *ofm, gboolean contentpane)
{
  /* There are a number of different popup menus depending on what is
     selected.  These menus are created by hiding/showing menu items
     in a big 'union' menu that contains everything that might ever be
     shown.

     This is the union menu.  The concrete menus are described below
     as deltas to it.

       Open
       ------
       Copy
       Paste
       Edit >
         Cut
         Copy
         Paste
         Move To
         Duplicate
         -----
         Rename
         Delete         
       Select All
       Rename
       Format memory card
       ------
       Send >
         Send as email
         Send via Bluetooth
       Details
       ------
       New Folder
       View >
         .. (doesn't change) ..
       Help

    multiple items, regardless of type:

      Hidden: Copy, Paste, Edit > Duplicate, Edit > Rename, Rename,
              Format memory card, Details

    single file:

      Hidden: Copy, Paste, Rename, Format

    single folder:

      Hidden: Open, Copy, Paste, Rename, Format memory card, Send

    single storage root folder:

      Hidden: Open, Edit, Rename, Format memory card, Send
    
    single memory card root folder:

      Hidden: Open, Edit, Send

    single network root folder (UPnP root, SMB root, Bluetooth root):

      Hidden: Open, Copy, Paste, Edit, Select All, Rename, Format
              memory card, Send, Details, New Folder

    single bluetooth device:

      Hidden: Copy, Paste, Edit, Rename, Format memory card, Send,
              Details

    when some error happens in determing which menu to show:

      Nothing is hidden, nothing is insensitive.
  */

  FMPopupMenu *m = ofm->fmpmenu;
  const char *sel_uri = NULL;
  
  update_current_selection_state (ofm->fs);

  /* Make everything visible that might have been hidden.
   */

  show_items (m->file_open,
	      m->sep_1,
	      m->copy,
	      m->paste,
	      m->edit->self,
	      m->edit->duplicate,
	      m->edit->rename,
	      m->newfolder,
	      m->select_all,
	      m->sep_3,
	      m->rename,
	      m->format,
	      m->sep_2,
	      m->send,
	      m->details,	      
	      m->sep_4,
	      NULL);

  /* Determine sensitivity of items.
   */
  
  check_sensitive (ofm, m->copy, on_copy_activate);
  check_sensitive (ofm, m->paste, on_paste_activate);
  check_sensitive (ofm, m->newfolder, osso_fm_new_folder);
  check_sensitive (ofm, m->send_as_email, osso_send_as_email);
  check_sensitive (ofm, m->send_via_bluetooth, osso_fm_send_via_bluetooth);
  check_sensitive (ofm, m->details, osso_fm_details_activate);

  check_sensitive (ofm, m->edit->cut, on_cut_activate);
  check_sensitive (ofm, m->edit->copy, on_copy_activate);
  check_sensitive (ofm, m->edit->paste, on_paste_activate);
  check_sensitive (ofm, m->edit->move_to, osso_fm_moveto);
  check_sensitive (ofm, m->edit->delete, osso_fm_delete_activate);
  check_sensitive (ofm, m->edit->rename, osso_fm_rename_activate);
  check_sensitive (ofm, m->edit->duplicate, on_duplicate_activate);

  sync_popup_view_menu (ofm, m->view);

  /* Determine visibility of items.
   */

  if (ofm->si->current_selection)
    sel_uri = gtk_file_path_get_string
      ((GtkFilePath *) (ofm->si->current_selection->data));
  else
    sel_uri = "";

  /* XXX - get rid of the hardcoded of URIs here.  Use type as
           reported by hildon-fm.
   */

  if (ofm->si->current_selection
      && ofm->si->current_selection->next)
    {
      /* Multiple selection
       */
      hide_items (m->copy,
		  m->paste,
		  m->edit->duplicate,
		  m->edit->rename,
		  m->rename,
		  m->format,
		  m->sep_2,
		  m->details,
		  NULL);
    }
  else if (ofm->si->no_mmc > 0
	   && !g_str_has_prefix (sel_uri, "file:///media/usb/"))
    {
      /* Single memory card
       */
      hide_items (m->file_open,
		  m->sep_1,
		  m->edit->self,
		  m->send,
		  NULL);
    }
  else if (ofm->si->no_folders > 0
	   && ofm->si->no_devices == 0
	   && ofm->si->no_mmc == 0)
    {
      /* Single folder
       */
      hide_items (m->file_open,
		  m->sep_1,
		  m->copy,
		  m->paste,
		  m->rename,
		  m->format,
		  m->sep_2,
		  m->send,
		  NULL);
    }
  else if (ofm->si->no_files > 0)
    {
      /* Single file
       */
      hide_items (m->copy,
		  m->paste,
		  m->rename,
		  m->format,
		  m->sep_2,
		  NULL);
    }
  else if (g_str_has_prefix (sel_uri, "file://"))
    {
      /* Single storage root folder
      */
      hide_items (m->file_open,
		  m->sep_1,
		  m->edit->self,
		  m->rename,
		  m->format,
		  m->sep_2,
		  m->send,
		  NULL);
    }
  else if (g_str_has_prefix (sel_uri, "obex://[")
	   && sel_uri[8])
    {
      /* Single Bluetooth device
       */
      hide_items (m->copy,
		  m->paste,
		  m->edit->self,
		  m->rename,
		  m->format,
		  m->sep_2,
		  m->send,
		  m->details,
		  m->sep_4,
		  NULL);
    }
  else if (g_str_has_prefix (sel_uri, "upnpav://")
	   || g_str_has_prefix (sel_uri, "smb://")
	   || g_str_has_prefix (sel_uri, "obex://"))
    {
      /* Single network root folder (UPnP root, SMB root, Bluetooth root):
       */
      hide_items (m->file_open,
		  m->sep_1,
		  m->copy,
		  m->paste,
		  m->edit->self,
		  m->select_all,
		  m->sep_3,
		  m->rename,
		  m->format,
		  m->sep_2,
		  m->send,
		  m->details,
		  m->sep_4,
		  m->newfolder,
		  NULL);
    }
  else
    {
      /* Something went wrong.  Hide nothing.
       */
    }
}

//END: Validation functions

/**
 * is_clipboard_contents_valid:
 * This function checks if the contents of the clipboard w.r.t gateway. i.e. if the 
 * copied files are from gateway then it checks if the same gateway pairing exists or not
 * @param sourceselection clipboard contents
 * @return gboolean TRUE if the contents are valid
 **/
gboolean is_clipboard_contents_valid(GSList *sourceselection)
{
	gchar *gateway_addr = NULL;
	gchar **paths = NULL;
	GSList *list = NULL, *cur = NULL;
	//return TRUE;// this statement has been added for multiple BT support
	if(is_path_in_gateway(sourceselection->data))
	{
		list = get_bt_addresses();
		if(list == NULL)
		{
			return TRUE;
		}
		paths = g_strsplit(sourceselection->data, "/", 0);
		if(paths == NULL || paths[0] == NULL || paths[1] == NULL)
		{
			return TRUE;
		}
		for(cur = list;cur != NULL;cur=cur->next)
		{
			if(gateway_addr != NULL)
			{
				g_free(gateway_addr);
				gateway_addr = NULL;
			}
			gateway_addr = g_strconcat("[",cur->data,"]",NULL);
			if(g_ascii_strcasecmp(gateway_addr,paths[2]) == 0) {
				g_free(gateway_addr);
				break;
			}
		}
		g_strfreev(paths);
		g_slist_foreach(list, (GFunc)g_free,NULL);
		g_slist_free(list);
		if(cur == NULL)
		{
			return FALSE;
		}
		else
		{
			return TRUE;
		}
	}
	return TRUE;
}



/**
 * osso_fm_dir_visit_init:
 * Function which initiates the dirrecursive visit
 * returns OSSOFMDirVisitHandle which should be used with other functions
 * @param path GtkFilePath to visit recursively
 * @param callback function to be called with each of the item in the directory
 * @param callback_data data to be passed to callback
 * @return OSSOFMDirVisitHandle handle to the operation or NULL
 **/
OSSOFMDirVisitHandle * osso_fm_dir_visit_init(GtkFilePath *path,
					OSSOFMDirVisitCallback callback,
					gpointer callback_data)
{
	DirVisitHandle *dv_handle = NULL;
	OSSOFMDirVisitHandle *self = NULL;
	gchar *path_texturi = NULL;
	GnomeVFSResult result;

	self = g_new0(OSSOFMDirVisitHandle,1);
	dv_handle = g_new0(DirVisitHandle,1);
	
	path_texturi = osso_fm_path_to_texturi(path);
	result = gnome_vfs_directory_open(&dv_handle->dir_handle,
					path_texturi,
					GNOME_VFS_FILE_INFO_DEFAULT);
	g_free(path_texturi);
	if(result != GNOME_VFS_OK)
	{
		FM_ULOG_ERR("osso_fm_dir_visit_init:directory_open err: %d\n",result);
		g_free(self);
		return NULL;
	}
	dv_handle->path = gtk_file_path_copy(path);
	self->handle_stack = g_queue_new();
	g_queue_push_head(self->handle_stack, dv_handle);
	self->callback = callback;
	self->callback_data = callback_data;
	return self;
}

/**
 * osso_fm_dir_visit_fetch_next:
 * Fetches the next <no_of_items> items and calls the callback
 * @param self OSSOFMDirVisitHandle 
 * @param no_of_items 
 * @return TRUE if successful
 **/
gboolean osso_fm_dir_visit_fetch_next(OSSOFMDirVisitHandle *self,gint no_of_items)
{
	DirVisitHandle *dv_handle = NULL;
	DirVisitHandle *tmp_dv_handle = NULL;
	GnomeVFSFileInfo *info = NULL;
	gboolean recursive = TRUE;
	GtkFilePath * childpath = NULL;
	gchar *path_texturi = NULL;
	int cur_item_count = 0;
	GnomeVFSResult result;

	if(self == NULL)
		return FALSE;

	dv_handle = g_queue_peek_head (self->handle_stack);
	info = gnome_vfs_file_info_new();
	if(info == NULL)
		return FALSE;
	cur_item_count = 0; 
	while(cur_item_count < no_of_items || no_of_items == -1)
	{
		dv_handle = g_queue_peek_head (self->handle_stack);
		if(gnome_vfs_directory_read_next(dv_handle->dir_handle, info) != GNOME_VFS_OK) {
			tmp_dv_handle = g_queue_pop_head(self->handle_stack);
			dv_handle = g_queue_peek_head (self->handle_stack);
			gnome_vfs_directory_close(tmp_dv_handle->dir_handle);
			gtk_file_path_free(tmp_dv_handle->path);
			g_free(tmp_dv_handle);
			if(dv_handle == NULL)
				return FALSE;
			continue;//take care of item count
		}
		if (info->name != NULL) {
			if ((info->name[0] == '.' && info->name[1] == '\0') ||
			    (info->name[0] == '.' && info->name[1] == '.' && info->name[2] == '\0'))
			{
				continue;
			}
		}
		cur_item_count++;
		self->callback(info,&recursive,self->callback_data);
		if (info->type != GNOME_VFS_FILE_TYPE_DIRECTORY) {
			continue;
		}
		if(recursive)
		{
			tmp_dv_handle = g_new0(DirVisitHandle,1);
			childpath = osso_fm_make_path(dv_handle->path,info->name);
			tmp_dv_handle->path = gtk_file_path_copy(childpath);
			path_texturi = osso_fm_path_to_texturi(childpath);
			result = gnome_vfs_directory_open(&tmp_dv_handle->dir_handle,
							path_texturi,
							GNOME_VFS_FILE_INFO_DEFAULT);
			g_free(path_texturi);
			if(result != GNOME_VFS_OK)
			{
				FM_ULOG_ERR("osso_fm_dir_visit_fetch_next:directory_open err: %d\n",
						result);
				g_free(tmp_dv_handle);
				gtk_file_path_free(childpath);
				continue;	
			}
			g_queue_push_head(self->handle_stack, tmp_dv_handle);
		} if(childpath != NULL) {
		gtk_file_path_free(childpath);	//VAL
		}
	}
	return TRUE;
}

/*
 * osso_fm_dir_visit_close:
 * Closes the OSSOFMDirVisitHandle and frees the memories
 * @param self OSSOFMDirVisitHandle to be closed
 * @return void
 **/
void osso_fm_dir_visit_close(OSSOFMDirVisitHandle *self)
{
	DirVisitHandle *dv_handle = NULL;

	if (self ==NULL)
		return;

	while((dv_handle = g_queue_pop_head(self->handle_stack)) != NULL)
	{
		gnome_vfs_directory_close(dv_handle->dir_handle);
		gtk_file_path_free(dv_handle->path);
		g_free(dv_handle);
	}
	g_queue_free(self->handle_stack);
	g_free(self);
}
typedef struct 
{
	GtkDialog *dialog;
	GtkTreeView *treeview;
	GtkTreeModel *model;
	GtkTreeModel *sort_model;
}OSSOFMUnknownFileOpenDialog;

typedef struct 
{
	gchar *app_name;
	gchar *text_domain;
	gchar *osso_service;
}AppListItem;

/**
 * application_listing_cb:
 * small helper function used as a callback
 * @param info GnomeVFSFileInfo *
 * @param recursive gboolean *
 * @param user_data gpointer
 * @return gboolean
 **/
static gboolean 
application_listing_cb(GnomeVFSFileInfo *info,gboolean *recursive,gpointer user_data)
{
	AppListItem *applistitem;
	GKeyFile *key_file = NULL;
	gchar *group = NULL;
	gchar *filepath = NULL;
	gchar *app_name = NULL,*osso_service = NULL,*text_domain = NULL;
	GSList **list = NULL;
	FM_ULOG_ERR("application_listing_cb called\n");
	if(g_strrstr(info->name,".desktop") != NULL)
	{
		list = (GSList **)user_data;
		key_file = g_key_file_new();
		filepath = g_strdup_printf("/usr/share/applications/hildon/%s",info->name);
		FM_ULOG_ERR("%s\n",filepath);
		g_key_file_load_from_file(key_file,filepath,G_KEY_FILE_NONE,NULL);
		group = g_key_file_get_start_group (key_file);

		app_name = g_key_file_get_string (key_file, 
						group, 
						"Name", 
						NULL);
		osso_service = g_key_file_get_string (key_file, 
						group, 
						"X-Osso-Service", 
						NULL);
		text_domain = g_key_file_get_string (key_file, 
						group, 
						"X-Text-Domain", 
						NULL);
		if(app_name != NULL && osso_service != NULL)
		{
			applistitem = g_new0(AppListItem,1);
			applistitem->app_name = app_name;
			applistitem->osso_service = osso_service;
			applistitem->text_domain = text_domain;
			*list = g_slist_append(*list,applistitem);
		}
		FM_ULOG_ERR("DATA from DESKTOP FILE %s %s\n",app_name, osso_service);
	}
	return TRUE;
}
/**
 * get_application_list:
 * Helper function to get the list of applications
 * @return GSList list of applications
 **/
static GSList * get_application_list()
{
	GSList *list = NULL;
	GSList *cur = NULL;
	GtkFilePath *path = NULL;
	OSSOFMDirVisitHandle *handle;
	FM_ULOG_ERR("get_application_list\n");
	path = osso_fm_texturi_to_path("file:///usr/share/applications/hildon/");
	handle = osso_fm_dir_visit_init(path,application_listing_cb,(gpointer)&list);
	osso_fm_dir_visit_fetch_next(handle,-1);
	osso_fm_dir_visit_close(handle);
	FM_ULOG_ERR("PRINTING LIST\n");
	for(cur = list; cur != NULL; cur = cur->next)
	{
		FM_ULOG_ERR("NAME %s DOMAIN %s SERVICE %s \n",
				((AppListItem *)cur->data)->app_name,
				((AppListItem *)cur->data)->text_domain,
				((AppListItem *)cur->data)->osso_service);
	}
	return list;
}

/**
 * mime_launch_add_arg:
 * Code taken from libossomime 
 * @param uri gchar *
 * @param iter *DBusMessageIter
 * @return void
 **/
static void mime_launch_add_arg (const gchar     *uri, 
				 DBusMessageIter *iter)
{
	if (!g_utf8_validate (uri, -1, NULL)) {
		g_warning ("Invalid UTF-8 passed to osso_mime_open\n");
		return;
	}

	FM_ULOG_ERR ("  %s\n", uri);
	
	dbus_message_iter_append_basic (iter, DBUS_TYPE_STRING, &uri);
}

/**
 * mime_launch:
 * Launches a application of given service name
 * @param service_name service name of the application.
 * @param list GSList of the paramentes to be passed to the application
 * @con DBusConnection
 * @return void
 **/
static void mime_launch (const gchar *service_name, GSList *list, DBusConnection *con)
{
	DBusMessage     *msg;
	DBusMessageIter  iter;
	gchar           *service;
	gchar           *object_path;
	gchar           *interface;

	/* If the service name has a '.', treat it as a full name, otherwise
	 * prepend com.nokia. */
	if (strchr (service_name, '.')) {
		service = g_strdup (service_name);
		object_path = g_strdup_printf ("/%s", service_name);
		interface = g_strdup (service_name);

		g_strdelimit (object_path, ".", '/');
	} else {
		service = g_strdup_printf ("com.nokia.%s", service_name);
		object_path = g_strdup_printf ("/com/nokia/%s", service_name);
		interface = g_strdup (service);
	}
	
	FM_ULOG_ERR ("Activating service: %s\n", service);

	msg = dbus_message_new_method_call (service, object_path,
					    interface, "mime_open");
	if (msg) {
		dbus_message_set_no_reply (msg, TRUE);
		
		dbus_message_iter_init_append (msg, &iter);
		
		g_slist_foreach (list, (GFunc) mime_launch_add_arg, &iter);
		
		dbus_connection_send (con, msg, NULL);
		dbus_connection_flush (con);
		
		dbus_message_unref (msg);
		
		/* From osso-rpc.c */
		/* Inform the task navigator that we are launching the service */
		msg = dbus_message_new_method_call (TASK_NAV_SERVICE,
						    APP_LAUNCH_BANNER_METHOD_PATH,
						    APP_LAUNCH_BANNER_METHOD_INTERFACE,
						    APP_LAUNCH_BANNER_METHOD);
		
		if (msg) {
			if (dbus_message_append_args (msg,
						      DBUS_TYPE_STRING, &service,
						      DBUS_TYPE_INVALID)) {
				dbus_connection_send (con, msg, NULL);
				dbus_connection_flush (con);
			} else {
				FM_ULOG_ERR ("Couldn't add service: %s\n", service);
			}
			dbus_message_unref (msg);
		} else {
			FM_ULOG_ERR ("Couldn't create msg to: %s\n", service);
		}
	}

	g_free (service);
	g_free (object_path);
	g_free (interface);
}

/**
 * create_unknown_file_open_dialog:
 * Function to create dialog to select an application while opening file of unknown type
 * @param file_path GtkFilePath to display the name of the file in the dialog
 * @return OSSOFMUnknownFileOpenDialog a structure representing the dialog
 **/ 
OSSOFMUnknownFileOpenDialog *create_unknown_file_open_dialog(GtkFilePath *file_path)
{
	OSSOFMUnknownFileOpenDialog *fod = NULL;
	GtkSizeGroup *sizegrp;
	GtkWidget *vbox;
	GtkWidget *label;
	GtkScrolledWindow *scrolled_win;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	gchar *trans_str = NULL;
	gchar *trans_str_with_params = NULL;
	gchar *filename = NULL;
	GdkGeometry hints;
	

	fod = g_new0(OSSOFMUnknownFileOpenDialog,1);
	
	fod->dialog = (GtkDialog *)gtk_dialog_new_with_buttons(_("hfil_ti_select_application"),
						(GtkWindow *)global_ofm->window,
						GTK_DIALOG_NO_SEPARATOR | GTK_DIALOG_MODAL,
						_("hfil_bd_select_application_ok"), 
						GTK_RESPONSE_OK,
						_("hfil_bd_select_application_cancel"),
						GTK_RESPONSE_CANCEL,
						NULL);
	gtk_window_set_resizable(GTK_WINDOW(fod->dialog), FALSE);
	hints.min_width = 432;
	hints.min_height = 300; // -1
	hints.max_width = 432;
	hints.max_height = 300; // -1
	gtk_window_set_geometry_hints(GTK_WINDOW(fod->dialog), GTK_WIDGET(fod->dialog),
			 &hints, GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE);
	sizegrp = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);
	vbox = gtk_vbox_new(FALSE, 0);
	trans_str = _("hfil_ia_select_application_information");
	filename = osso_fm_path_get_shortname(file_path);
	trans_str_with_params = g_strdup_printf(trans_str,filename);
	label = gtk_label_new(trans_str_with_params);
	g_free(filename);
	g_free(trans_str_with_params);
	gtk_box_pack_start(GTK_BOX(vbox), (GtkWidget *) label, FALSE, FALSE, 0);

	fod->model = (GtkTreeModel *)gtk_list_store_new(2,G_TYPE_STRING,G_TYPE_STRING);
	fod->sort_model = gtk_tree_model_sort_new_with_model(fod->model);
	gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(fod->sort_model),
			0,
			GTK_SORT_ASCENDING);

	fod->treeview = (GtkTreeView *)gtk_tree_view_new_with_model(fod->sort_model);
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes("Application", renderer,
							"text", 0, NULL);
	gtk_tree_view_append_column(fod->treeview,column);
	scrolled_win = (GtkScrolledWindow *)gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(scrolled_win,GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type(scrolled_win,GTK_SHADOW_NONE);
	gtk_container_add(GTK_CONTAINER(scrolled_win),(GtkWidget *)fod->treeview);
	gtk_box_pack_start(GTK_BOX(vbox), (GtkWidget *) scrolled_win, TRUE, TRUE, 0);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(fod->dialog)->vbox),
			  GTK_WIDGET(vbox));
	gtk_widget_show(GTK_WIDGET(fod->treeview));
	gtk_widget_show(GTK_WIDGET(scrolled_win));
	gtk_widget_show_all(vbox);
	return fod;
}

/**
 * osso_fm_unknown_file_open:
 * Function to open File of Unknown file type
 * @param ofm OSSOFileManager
 * @param filelist GSList of paths to be opened.
 * @return void
 **/
void osso_fm_unknown_file_open(OSSOFileManager *ofm,GSList *filelist)
{
	GtkTreeIter iter;
	GtkTreeIter sort_iter;
	GSList *applist = NULL,*cur = NULL;
	OSSOFMUnknownFileOpenDialog *fod;
	GtkTreeSelection *selection;
	gchar *service_name = NULL;
	gchar *conftext = _("hfil_ai_unrecognized_filetype_information");
	gchar *conftext_with_filename = _("hfil_ai_unrecognized_filetype_information");
	applist = get_application_list();

	conftext_with_filename = g_strdup_printf(conftext,
					osso_fm_path_get_shortname(filelist->data));
	if(fm_confnote(GTK_WINDOW(ofm->window),
			"",
			conftext_with_filename,
			_("hfil_bd_unrecognized_filetype_yes"),GTK_RESPONSE_OK,
			_("hfil_bd_unrecognized_filetype_no"),GTK_RESPONSE_CANCEL,
			TRUE) != GTK_RESPONSE_OK)
	{
		g_free(conftext_with_filename);
		return;
	}
	g_free(conftext_with_filename);

	fod = create_unknown_file_open_dialog(filelist->data);
	for(cur = applist;cur != NULL; cur = cur->next)
	{
		gtk_list_store_append (GTK_LIST_STORE(fod->model), &iter);
		if(((AppListItem *)cur->data)->text_domain == NULL)
		{
			gtk_list_store_set (GTK_LIST_STORE(fod->model), &iter,
					0, dgettext("maemo-af-desktop",
						((AppListItem *)cur->data)->app_name),
					1,((AppListItem *)cur->data)->osso_service,-1);
		}
		else
		{
			gtk_list_store_set (GTK_LIST_STORE(fod->model), &iter,
					0, dgettext(((AppListItem *)cur->data)->text_domain,
						((AppListItem *)cur->data)->app_name),
					1,((AppListItem *)cur->data)->osso_service,-1);
		}
	}
	gtk_widget_show_all(GTK_WIDGET(fod->dialog));
	if(gtk_dialog_run(fod->dialog) == GTK_RESPONSE_OK)
	{
		selection = gtk_tree_view_get_selection(fod->treeview);
		gtk_tree_selection_get_selected(selection, NULL, &sort_iter); //may return false
		gtk_tree_model_sort_convert_iter_to_child_iter(GTK_TREE_MODEL_SORT(fod->sort_model),
				&iter,&sort_iter);
		gtk_tree_model_get(fod->model, &iter,
				1,&service_name,-1);
		mime_launch(service_name,filelist,osso_fm_get_dbus_connection(fm_app_ctx));
	}
	gtk_widget_destroy(GTK_WIDGET(fod->dialog));
}

/**
 * osso_fm_path_type_mmc:
 * Checks if the given type is MMC
 * @param type
 * @return TRUE if type is HILDON_FILE_SYSTEM_MODEL_MMC
 **/
gboolean osso_fm_path_type_mmc(HildonFileSystemModelItemType type)
{
	if(type == HILDON_FILE_SYSTEM_MODEL_MMC)
	{
		return TRUE;
	}
	return FALSE;
}

/**
 * media_items_get_rename_error:
 * Function which returns the error string based on the selection type
 * @param seltype
 * @return gchar * error string
 **/
static gchar *media_items_get_rename_error(SelectionInfoType seltype)
{
	switch (seltype)
	{
		case SIT_SINGLE_FOLDER_SELECTED:
		case SIT_SINGLE_FILE_SELECTED:
			return dgettext(HILDON_FM_L10N_PACKAGE,"sfil_ib_renaming_not_supported");//_("FIL-INF050");
			break;
		case SIT_DEVICE_SELECTED:
			if(is_path_virtual(global_ofm->si->current_selection->data)) {
				return _("sfil_ib_unable_rename_shared_media");//_("FIL-INF098");
			} else {
				return dgettext(HILDON_FM_L10N_PACKAGE,"sfil_ib_renaming_not_supported");
			}
			break;
		case SIT_MULTIPLE_FILES_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
		case SIT_MMC_SELECTED:
		case SIT_SINGLE_SAFE_FOLDER_SELECTED:
		case SIT_MULTIPLE_SAFE_FOLDERS_SELECTED:
			FM_ULOG_ERR("media_items_get_rename_error:This case should not arraise\n");
			break;
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			break;
	}
	return NULL;
}
/**
 * media_items_get_moveto_error:
 * Function which returns the error string based on the selection type
 * @param seltype
 * @return gchar * error string
 **/
static gchar *media_items_get_moveto_error(SelectionInfoType seltype)
{
	switch (seltype)
	{
		case SIT_SINGLE_FOLDER_SELECTED:
			return dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"sfil_ib_unable_to_move_selected_folder");//_("WID-INF309");
			break;
		case SIT_SINGLE_FILE_SELECTED:
		case SIT_MULTIPLE_FILES_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
			return dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"sfil_ib_unable_to_move_selected_items");//_("WID-INF308");
			break;
		case SIT_DEVICE_SELECTED:
			if (global_ofm->si->selection_in_gateway) {
				return _("sfil_ib_unable_move_bluetooth");
			} else if (global_ofm->si->selection_in_media_server) {
				return _("sfil_ib_unable_move_shared_media");
			} else {
				return _("sfil_ib_cannot_move_device");
			}
			break;
		case SIT_MMC_SELECTED:
		case SIT_SINGLE_SAFE_FOLDER_SELECTED:
		case SIT_MULTIPLE_SAFE_FOLDERS_SELECTED:
			FM_ULOG_ERR("media_items_get_cut_error:This case should not arraise\n");
			break;
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			break;
	}
	return NULL;
}
/**
 * media_items_get_cut_error:
 * Function which returns the error string based on the selection type
 * @param seltype
 * @return gchar * error string
 **/
static gchar *media_items_get_cut_error(SelectionInfoType seltype)
{
	switch (seltype)
	{
		case SIT_SINGLE_FOLDER_SELECTED:
			return dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"sfil_ib_unable_to_cut_selected_folder");//_("WID-INF305");
			break;
		case SIT_SINGLE_FILE_SELECTED:
		case SIT_MULTIPLE_FILES_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
			return dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"sfil_ib_unable_to_cut_selected_items");//_("WID-INF304");
			break;
		case SIT_DEVICE_SELECTED:
			if(is_path_virtual(global_ofm->si->current_selection->data)) {
				return _("sfil_ib_unable_cut_shared_media");//_("FIL-INF095");
			} else {
				return _("sfil_ib_cannot_cut_device");
			}
			break;
		case SIT_MMC_SELECTED:
		case SIT_SINGLE_SAFE_FOLDER_SELECTED:
		case SIT_MULTIPLE_SAFE_FOLDERS_SELECTED:
			FM_ULOG_ERR("media_items_get_cut_error:This case should not arraise\n");
			break;
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			break;
	}
	return NULL;
}
/**
 * media_items_get_delete_error:
 * Function which returns the error string based on the selection type
 * @param seltype
 * @return gchar * error string
 **/
static gchar *media_items_get_delete_error(SelectionInfoType seltype)
{
	switch (seltype)
	{
		case SIT_SINGLE_FOLDER_SELECTED:
			return dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"sfil_ib_unable_to_delete_selected_folder");//_("WID-INF307");
			break;
		case SIT_SINGLE_FILE_SELECTED:
			return dgettext(HILDON_FM_L10N_PACKAGE,
					"sfil_ib_cannot_delete_file");//_("FIL-INF060");
			break;
		case SIT_MULTIPLE_FILES_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
			return dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"sfil_ib_unable_to_delete_selected_items");//_("WID-INF306");
			break;
		case SIT_DEVICE_SELECTED:
			if(is_path_virtual(global_ofm->si->current_selection->data)) {
				return _("sfil_ib_unable_delete_shared_media");//_("FIL-INF099");
			} else {
				return _("sfil_ib_cannot_delete_device");
			}
			break;
		case SIT_MMC_SELECTED:
		case SIT_SINGLE_SAFE_FOLDER_SELECTED:
		case SIT_MULTIPLE_SAFE_FOLDERS_SELECTED:
			FM_ULOG_ERR("media_items_get_cut_error:This case should not arraise\n");
			break;
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			break;
	}
	return NULL;
}
	
/**
 * check_for_media_items:
 * Function to check if an operation can be performed on the given items 
 * if they are in media server
 * @oper MenuOperation operation performed
 * @param ofm OSSOFileManager *
 * @param menucheck 
 * @return TRUE if the operation can be performed else false
 **/
gboolean check_for_media_items(MenuOperation oper,OSSOFileManager *ofm, gboolean menucheck)
{	
	gchar *msg = NULL;
	SelectionInfoType seltype;
	if(!ofm->si->selection_in_media_server)
	{
		return FALSE;
	}
	seltype = selection_info_get_selection_type(ofm->si);
		
	switch(oper)
	{
		case OPER_CUT :
			msg = media_items_get_cut_error(seltype);
			break;
		case OPER_COPY :
			if(seltype == SIT_DEVICE_SELECTED)
			{
				if(is_path_virtual(ofm->si->current_selection->data)) { 
					msg = _("sfil_ib_unable_copy_shared_media");//_("FIL-INF096");
				} else {
					msg = _("sfil_ib_cannot_copy_device");
				}
			}
			break;
		case OPER_MOVE :
			msg = media_items_get_moveto_error(seltype);
			break;
		case OPER_RENAME :
			msg = media_items_get_rename_error(seltype);
			break;
		case OPER_DELETE :
			msg = media_items_get_delete_error(seltype);
			break;
		case OPER_NEWFOLDER:
			 msg = dgettext(HILDON_FM_L10N_PACKAGE,"sfil_ib_readonly_location");
			 break;
		default :
			break;
	}
	if(msg == NULL)
	{
		return FALSE;
	}
	else
	{
		if(!menucheck)
		{
			fm_infoprint(GTK_WINDOW(ofm->window),msg);
		}
		return TRUE;
	}
}

gboolean is_path_virtual(GtkFilePath *path)
{
	gchar *path_texturi = NULL;
	gboolean retval = FALSE;
	if(path == NULL)
	{
		FM_ULOG_ERR("is_path_virtual:path = NULL\n");
		return FALSE;
	}
	path_texturi = osso_fm_path_to_texturi (path);
	retval = is_texturi_virtual(path_texturi);
	g_free(path_texturi);
	return retval;
}

gboolean is_texturi_virtual(gchar *path_texturi)
{
	if(path_texturi == NULL)
	{
		FM_ULOG_ERR("is_texturi_virtual:path = NULL\n");
		return FALSE;
	}

	return (g_str_has_suffix (path_texturi, "://") ||
		g_str_has_suffix (path_texturi, ":///"));
}

gboolean is_xfer_in_progress()
{
	if(global_ofm->xfer_callback_data != NULL)
		return TRUE;
	return FALSE;
}

