/**
 * @file osso_fm_operations.c
 *
 * This file contains routines for copy, move and delete operations
 *
 * Copyright (C) 2005 Nokia Corporation
 */

#include <strings.h>
#include <hildon/hildon-file-selection.h>
#include <hildon/hildon-file-chooser-dialog.h>
#include "osso_fm_generic.h"
#include "osso_fm_controller.h"
#include "osso_fm_callbacks.h"
#include "osso_fm_utils.h"
#include "osso_fm_fmstate.h"
#include "osso_fm_operations.h"
#include "osso_fm_xfer.h"


//BEG: New Folder related functions
typedef enum {
	NFCS_ERROR_IN_PARAMETERS = -1,
	NFCS_ERROR_ALREADY_EXISTS = 0,
	NFCS_CREATION_SUCCESSFUL = 1
}NewFolderCreationStatus; 

/**
 * newfolder_okbutton_clicked:
 * This function is called when the user click on ok after entering 
 * the folder name in the new folder dialog. This creates the new folder
 * in the given location if possible.
 * 
 * @param window	Pointer to the application's main window
 * @param newpath	A string having the path of the new folder
 * 
 * @return NewFolderCreationStatus
 */
static NewFolderCreationStatus 
newfolder_okbutton_clicked(OSSOFileManager *ofm, const gchar * newpath)
{
	GnomeVFSURI *newpathuri;
	GnomeVFSURI *newpath_parenturi;
	GnomeVFSResult result;

	DEBUG_PRINT_START;
	if (newpath == NULL) {
		FM_ULOG_ERR("Error validating input parameter");
		return NFCS_ERROR_IN_PARAMETERS;
	}
	FM_ULOG_INFO("Checking if new folder name already exists\n");
	if (is_path_exists(newpath)) {
		FM_ULOG_ERR("FOLDER EXISTS %s\n", newpath);
		fm_infoprint(GTK_WINDOW(ofm->window),
			     dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
				      "ckdg_ib_folder_already_exists"));
		DEBUG_PRINT_END;
		return NFCS_ERROR_ALREADY_EXISTS;
	}
	//New folder creation
	FM_ULOG_DEBUG1("New folder path before make directory : %s\n", newpath);
	result = gnome_vfs_make_directory(newpath, 0775);
	if (result == GNOME_VFS_OK) {
		FM_ULOG_INFO("Directory Creation succesful");
		newpathuri = gnome_vfs_uri_new(newpath);
		if (newpathuri != NULL) {
			newpath_parenturi = gnome_vfs_uri_get_parent(newpathuri);
			if (newpath_parenturi != NULL) {
				osso_fm_copy_uri_view_settings(newpathuri, newpath_parenturi);
				gnome_vfs_uri_unref(newpath_parenturi);
			}
			gnome_vfs_uri_unref(newpathuri);
		}
	} else if (result == GNOME_VFS_ERROR_FILE_EXISTS){
		fm_infoprint(GTK_WINDOW(ofm->window),dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"ckdg_ib_folder_already_exists"));

		FM_ULOG_ERR("Error creating folder : %d", result);
	} else if(result == GNOME_VFS_ERROR_NOT_SUPPORTED){
		fm_infoprint(GTK_WINDOW(global_ofm->window),
			dgettext(HILDON_FM_L10N_PACKAGE,
				"sfil_ib_create_folder_not_allowed"));
	} else {
		fm_infonote(GTK_WINDOW(global_ofm->window), 
			dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
				"sfil_ni_operation_failed"), 
			dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
				"sfil_ni_operation_failed_ok"));
	}

	DEBUG_PRINT_END;
	return NFCS_CREATION_SUCCESSFUL;
}
/**
 * create_new_folder:
 * Creates a New Folder in the path specified.
 * 
 * @param window 	Pointer to the application main window.
 * @param path 	Represents the path where the New Folder is created.
 * 
 * @return void
 */
void create_new_folder(OSSOFileManager *ofm, GtkFilePath * path)
{
	GtkWidget *NewFolder = NULL;
	HildonFileSystemModel *fsmodel = NULL;
	gint response = 0;
	gchar *newpath = NULL;
	NewFolderCreationStatus retval = NFCS_ERROR_IN_PARAMETERS;
	gchar *path_texturi = NULL;

	DEBUG_PRINT_START;
	if (path == NULL) {
		FM_ULOG_ERR("Error : Invalid parameter");
		return;
	}
	path_texturi = osso_fm_path_to_texturi(path);
	if(path_texturi == NULL)
	{
		FM_ULOG_ERR("create_new_folder: path_texturi == NULL\n");
		return;
	}
	g_object_get(ofm->fs, "model", &fsmodel, NULL);
	if (fsmodel != NULL) {
		NewFolder = hildon_file_chooser_dialog_new_with_properties((GtkWindow *) ofm->window,
							   "action",
							   GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER,
							   "file-system-model",
							   fsmodel, 
							   "max-name-length", 255, NULL);
	} else {
		FM_ULOG_DEBUG1("Unable to get fsmodel\n");
		NewFolder = hildon_file_chooser_dialog_new((GtkWindow *) ofm->window,
							   GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER);
	}

	gtk_window_set_modal(GTK_WINDOW(NewFolder), TRUE);

	if (gtk_file_chooser_set_current_folder_uri(GTK_FILE_CHOOSER(NewFolder),
						path_texturi) == FALSE) {
		FM_ULOG_ERR("Set current Folder failed\n");
	} else {
		do {
			if(newpath != NULL){
				g_free(newpath);
				newpath = NULL;
			}
			g_signal_emit_by_name(NewFolder, "show");
			response = gtk_dialog_run(GTK_DIALOG(NewFolder));
			newpath = gtk_file_chooser_get_uri((GtkFileChooser *) NewFolder);
			if (response == GTK_RESPONSE_OK) {
				retval = newfolder_okbutton_clicked(ofm, newpath);
			} else {
				break;
			}
		} while (retval == NFCS_ERROR_ALREADY_EXISTS);

		//The following lines are commented for not selecting the newfolder after creation
		//if (retval == NFCS_CREATION_SUCCESSFUL) {
		//	osso_fm_select_texturi_with_time(newpath, TRUE);
		//}
		gtk_widget_destroy(NewFolder);
	}
	if (fsmodel != NULL) {
		g_object_unref(fsmodel);
	}
	if(newpath != NULL){
		g_free(newpath);
	}
	g_free(path_texturi);
	DEBUG_PRINT_END;
	return;
}

/**
 * osso_fm_new_folder:
 * Invoked when a new folder options is selected from File Manager menu.
 * 
 * @param window 	Pointer to the application's main window.
 * 
 * @return void
 */
gboolean osso_fm_new_folder(OSSOFileManager* ofm, gboolean menucheck)
{
	gboolean retval = FALSE;
	gboolean readonly_path = FALSE;

	DEBUG_PRINT_START;
	
	selection_info_update(ofm,!menucheck);
		
	if(check_for_media_items(OPER_NEWFOLDER,ofm,menucheck))
	{
		return FALSE;
	}
	if(is_path_virtual(ofm->si->current_folder))
	{
		if(!menucheck)
		{
			FM_ULOG_ERR("Virtual path new folder not allowed\n");
			fm_infoprint(GTK_WINDOW(global_ofm->window),
				dgettext(HILDON_FM_L10N_PACKAGE, 
					"sfil_ib_create_folder_not_allowed"));
		}
		return FALSE;
	}

	switch(selection_info_get_selection_type(ofm->si))
	{
		case SIT_SINGLE_FOLDER_SELECTED:
		case SIT_SINGLE_SAFE_FOLDER_SELECTED:
		case SIT_DEVICE_SELECTED:
		case SIT_MMC_SELECTED:
			retval = ! (is_path_readonly(ofm->si->current_selection->data,TRUE) &&
				    !ofm->si->selection_in_gateway);
			if (menucheck) {
				break;
			}
			if(is_path_readonly(ofm->si->current_selection->data,TRUE) && 
					!ofm->si->selection_in_gateway)
			{
				fm_infoprint(GTK_WINDOW(ofm->window), 
//						osso_common_error(CERM_PATH_RONLY));
						dgettext(HILDON_FM_L10N_PACKAGE,"sfil_ib_readonly_location"));
			} else {
				create_new_folder(ofm, ofm->si->current_selection->data);
			}
			break;
		case SIT_SINGLE_FILE_SELECTED:
		case SIT_MULTIPLE_SAFE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_FILES_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
			retval = ! (is_path_readonly(ofm->si->current_folder,FALSE) &&
				    !ofm->si->selection_in_gateway);
			if (menucheck) {
				break;
			}
			if(is_path_readonly(ofm->si->current_folder,FALSE) &&
					!ofm->si->selection_in_gateway)
			{
				fm_infoprint(GTK_WINDOW(ofm->window), 
//						osso_common_error(CERM_PATH_RONLY));
						dgettext(HILDON_FM_L10N_PACKAGE,"sfil_ib_readonly_location"));
			} else {
				create_new_folder(ofm, ofm->si->current_folder);
			}
			break;
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			retval = FALSE;
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			retval = FALSE;
			break;
	}

	DEBUG_PRINT_END;
	return retval;
}

//END: New Folder related functions

//BEG: Rename related functions

/** 
 * osso_fm_rename_activate:
 * Method invoked when a rename menu command is actiavated.
 * 
 * @param window Main file manager window
 * @return void
 */
gboolean osso_fm_rename_activate(OSSOFileManager *ofm, gboolean menucheck)
{
	HildonFileSelection *fs = NULL;
	gboolean retval = FALSE;
	SelectionInfoType seltype;
	GtkWidget *window = GTK_WIDGET (ofm->window);
	DEBUG_PRINT_START;
	gboolean readonly_path = FALSE;

	if (window == NULL) {
		if (menucheck == FALSE) {
			FM_ULOG_ERR("Invalid parameter : window specified");
		}
		DEBUG_PRINT_END;
		return FALSE;
	}

	if(!ofm->si->selection_in_gateway)
	{
		readonly_path = is_path_readonly(ofm->si->current_selection->data,TRUE);
	}
	fs = global_ofm->fs;
	selection_info_update(global_ofm,!menucheck);
	seltype = selection_info_get_selection_type(global_ofm->si);

	if(global_ofm->si->selection_in_gateway)
	{
		if (menucheck == FALSE) {
			if (seltype == SIT_DEVICE_SELECTED && 
					is_path_virtual(global_ofm->si->current_selection->data)) {
			    fm_infoprint(GTK_WINDOW(window), _("sfil_ib_unable_rename_bluetooth"));
			}
			else {
			fm_infoprint(GTK_WINDOW(window), dgettext(HILDON_FM_L10N_PACKAGE,
						"sfil_ib_renaming_not_supported"));
			}
		}
		DEBUG_PRINT_END;
		return FALSE;
	}

	if(check_for_media_items(OPER_RENAME,global_ofm,menucheck))
	{
		return FALSE;
	}
	switch(seltype)
	{
		case SIT_SINGLE_FILE_SELECTED:
		case SIT_SINGLE_FOLDER_SELECTED:
			if(!is_path_exists(global_ofm->si->current_selection->data))
			{
				if (!menucheck) {
					if(global_ofm->si->no_files > 0)
					{
						fm_infonote(GTK_WINDOW(window),
						  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						    "sfil_ni_unable_to_rename_file_not_found"),
						  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						    "sfil_ni_unable_to_rename_file_not_found_ok"));
					}
					else
					{
						fm_infonote(GTK_WINDOW(window),
						  _("sfil_ni_cannot_rename_folder_not_found"),
						  _("sfil_bd_cannot_rename_folder_not_found_ok"));
					}
				}
			}
			else if(readonly_path)
			{
				if (!menucheck) {
					fm_infoprint(GTK_WINDOW(window),
						      dgettext(HILDON_FM_L10N_PACKAGE,
							      "sfil_ib_renaming_not_allowed"));
				}
			} else {
				if (!menucheck) {
					osso_create_rename_dialog(window,
						osso_fm_path_to_texturi(
							global_ofm->si->current_selection->data));
				}
				retval = TRUE;
			}
			break;
		case SIT_MULTIPLE_SAFE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_FILES_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(window),  
						dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
							"ckct_ib_select_single_item"));
			}
			retval = FALSE;
			break;
		case SIT_DEVICE_SELECTED:
			if (menucheck == FALSE) {
				if(global_ofm->si->selection_in_gateway)
				{
					fm_infoprint(GTK_WINDOW(window), 
						dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ib_unable_to_rename_selected_folder"));
				}
				else
				{
					fm_infoprint(GTK_WINDOW(window), 
						dgettext(HILDON_FM_L10N_PACKAGE,
						"sfil_ib_renaming_not_supported"));
				}
			}
			retval = FALSE;
			break;
		case SIT_MMC_SELECTED:
		  {
		    OSSOFMFileInfo *info =
		      global_ofm->si->current_selection_withinfo->data;
		    const char *uri = osso_fm_path_to_texturi (info->path);
		    const char *device = info->extra;
		    
		    if (g_str_has_prefix (uri, "file:///media/usb/"))
		      {
			if (menucheck == FALSE)
			  fm_infoprint(GTK_WINDOW(window), 
				       dgettext(HILDON_FM_L10N_PACKAGE,
						"sfil_ib_renaming_not_supported"));
			retval = FALSE;
		      }
		    else
		      {
			if (menucheck == FALSE)
			  {
			    fprintf (stderr, "RENAMING %s %s\n", uri, device);
			    osso_fm_mmc_rename (global_ofm, uri, device);
			  }
			retval = TRUE;
		      }
		  }
		  break;
		case SIT_SINGLE_SAFE_FOLDER_SELECTED:
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(window),
					       dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						       "sfil_ib_unable_to_rename_selected_folder"));
			}
			retval = FALSE;
			break;
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			retval = FALSE;
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			retval = FALSE;
			break;
	}
	DEBUG_PRINT_END;
	return retval;
}

/** 
 * osso_fm_format_activate:
 * Method invoked when a format menu command is actiavated.
 * 
 * @param window Main file manager window
 * @return void
 */
gboolean osso_fm_format_activate (OSSOFileManager *ofm, gboolean menucheck)
{
	HildonFileSelection *fs = NULL;
	gboolean retval = FALSE;
	SelectionInfoType seltype;
	GtkWidget *window = GTK_WIDGET (ofm->window);
	DEBUG_PRINT_START;

	if (window == NULL) {
		if (menucheck == FALSE) {
			FM_ULOG_ERR("Invalid parameter : window specified");
		}
		DEBUG_PRINT_END;
		return FALSE;
	}

	fs = global_ofm->fs;
	selection_info_update(global_ofm,!menucheck);
	seltype = selection_info_get_selection_type(global_ofm->si);

	if (seltype == SIT_MMC_SELECTED)
	  {
	    OSSOFMFileInfo *info =
	      global_ofm->si->current_selection_withinfo->data;
	    const char *uri = osso_fm_path_to_texturi (info->path);
	    const char *device = info->extra;
	    
	    if (g_str_has_prefix (uri, "file:///media/usb/"))
	      retval = FALSE;
	    else
	      {
		if (menucheck == FALSE)
		  {
		    fprintf (stderr, "FORMATTING %s %s\n", uri, device);
		    osso_fm_mmc_format (device);
		  }
		retval = TRUE;
	      }
	  }
	else
	  retval = FALSE;

	if (!menucheck && retval == FALSE)
	  fm_infoprint(GTK_WINDOW(window), _("sfil_ib_select_memory_card"));

	DEBUG_PRINT_END;
	return retval;
}

static gchar*
translate_special_name (const gchar *name)
{
  if (name && name[0] == '~' && name[1] != '\0')
    {
      char *trans = dgettext (HILDON_FM_L10N_PACKAGE, name + 1);

      if (trans != name + 1)
	name = g_strdup (trans);
    }

  return g_strdup (name);
}

/**
 * osso_create_rename_dialog:
 * Method to create the Rename dialog
 * 
 * @param mainwindow Main window pointer
 * @param oldname oldname that needs to be renamed
 **/
void osso_create_rename_dialog(GtkWidget * mainwindow, gchar * oldname)
{
	GtkWidget *renamedialog = NULL;
	HildonFileSystemModel *fsmodel = NULL;
	gchar *parenturi = NULL;
	gchar *newpath = NULL;
	gchar *shortname = NULL;
	gchar *trans = NULL;
	gint pathtype = 0;
	gchar *rename_dialog_title = NULL;
	gint response = 0;
	gboolean dialog_return = FALSE;

	DEBUG_PRINT_START;

	if ((mainwindow == NULL) || (oldname == NULL)) {
		FM_ULOG_ERR("Error validating input parameters\n");
		return;
	}

	parenturi = osso_fm_path_get_dirname(oldname);
	shortname = osso_fm_texturi_get_shortname (oldname);

	if (parenturi == NULL) {
		FM_ULOG_ERR("Error creating parent uri\n");
		g_free(oldname);
		return;
	}
	if (shortname == NULL) {
		g_free(parenturi);
		FM_ULOG_ERR("Error creating shortname\n");
		g_free(oldname);
		return;
	}
	g_object_get(global_ofm->fs, "model", &fsmodel, NULL);
	if (fsmodel != NULL) {
		renamedialog =
		    hildon_file_chooser_dialog_new_with_properties(GTK_WINDOW(mainwindow), "action",
								   GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER,
								   "file-system-model", fsmodel,
								   "select-multiple", FALSE,
								   "show-folder-button", FALSE,
								   "show-location", FALSE,
								   "max-name-length", 255,
								   "autonaming", FALSE, NULL);
	} else {
		renamedialog =
		    hildon_file_chooser_dialog_new_with_properties(GTK_WINDOW(mainwindow), "action",
								   GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER,
								   "select-multiple", FALSE,
								   "show-folder-button", FALSE,
								   "show-location", FALSE,
								   "max-name-length", 255,
								   "autonaming", FALSE, NULL);
	}

	pathtype = osso_fm_get_path_type(oldname);

	switch (pathtype) {
	case GNOME_VFS_FILE_TYPE_REGULAR:
		rename_dialog_title = get_rename_file_title(oldname);
		trans = translate_special_name (shortname);
		gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(renamedialog), trans);
		break;
	case GNOME_VFS_FILE_TYPE_DIRECTORY:
		rename_dialog_title = g_strdup_printf("%s", 
				dgettext(HILDON_LIBS_L10N_PACKAGE,"ckdg_ti_rename_folder"));
		trans = translate_special_name (shortname);
		gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(renamedialog), trans);
		break;
	case OSSO_FM_MEMORY_ALLOCATION_FAILURE:
		fm_infoprint(GTK_WINDOW(mainwindow), 
				dgettext(KE_RECV_L10N_PACKAGE, "memr_ib_operation_disabled"));	//NO CHANGE
		FM_ULOG_ERR("Memory allocation error\n");
		g_object_unref(fsmodel);
		g_free(parenturi);
		g_free(shortname);
		g_free(oldname);
		return;
		break;
	default:
		gtk_widget_destroy(renamedialog);
		FM_ULOG_ERR("Unknown File type %d\n", pathtype);
		g_free(parenturi);
		g_free(shortname);
		g_free(oldname);
		return;
		break;
	}

	if (rename_dialog_title == NULL) {
		FM_ULOG_ERR("Error getting rename dialog title\n");
		g_object_unref(fsmodel); 
		g_free(parenturi);
		g_free(shortname);
		g_free(oldname);
		g_free(trans);
		return;
	}

	gtk_window_set_title(GTK_WINDOW(renamedialog), rename_dialog_title);

	gtk_window_set_modal(GTK_WINDOW(renamedialog), TRUE);
	if (gtk_file_chooser_set_current_folder_uri(GTK_FILE_CHOOSER(renamedialog), parenturi) ==
	    FALSE) {
		FM_ULOG_ERR("Error setting current folder\n");
		g_object_unref(fsmodel); 
		g_free(parenturi);
		g_free(shortname);
		g_free(oldname);
		g_free(trans);
		return;
	}
	gtk_window_set_default_size(GTK_WINDOW(renamedialog),
			 osso_fm_predict_dialog_width_based_on_title(renamedialog,
			 rename_dialog_title),
			 -1);
	dialog_return = FALSE;

	while (dialog_return == FALSE) {
		g_signal_emit_by_name(renamedialog, "show");
		response = gtk_dialog_run(GTK_DIALOG(renamedialog));

		if (response == GTK_RESPONSE_OK) {
			newpath = gtk_file_chooser_get_uri((GtkFileChooser *) renamedialog);
			FM_ULOG_INFO("New Path [%s]\n", newpath);

			if (!is_path_exists(oldname)) {
				if (pathtype == GNOME_VFS_FILE_TYPE_REGULAR) {
					fm_infoprint(GTK_WINDOW(mainwindow),
						dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
							"sfil_ni_unable_to_rename_file_not_found"));
				} else if (pathtype == GNOME_VFS_FILE_TYPE_DIRECTORY) {
					fm_infoprint(GTK_WINDOW(mainwindow),
							_("sfil_ni_cannot_rename_folder_not_found"));
				}
			} else {
				FM_ULOG_INFO("\n\nNew Path from file chooser[%s]\n\n", newpath);
				gchar *newshortname = NULL;
				gchar *newshortname_unesc = NULL;
				gchar *path_for_newname = NULL;
				newshortname_unesc = osso_fm_texturi_get_shortname(newpath);
				newshortname = gnome_vfs_escape_string(newshortname_unesc);
				g_free(newshortname_unesc);
				path_for_newname = osso_fm_path_get_dirname(oldname);
				g_free(newpath);
				newpath = g_strdup_printf("%s/%s", path_for_newname, newshortname);
				g_free(path_for_newname);
				g_free(newshortname);
				FM_ULOG_INFO("\n\nNew Path [%s]\n\n", newpath);
				dialog_return = on_rename_dialog_okbutton_clicked(mainwindow,
										  oldname, newpath);
			}
			FM_ULOG_DEBUG1("New path : %s\n", newpath);
		} else {
			break;
		}
	}			//While loop

	gtk_widget_destroy(renamedialog);
	g_object_unref(fsmodel);
	g_free(rename_dialog_title);
	g_free(parenturi);
	g_free(shortname);
	g_free(newpath);
	g_free(oldname);
	g_free(trans);
	DEBUG_PRINT_END;
	return;
}

gchar *get_file_type(gchar * path)
{
	gchar *mime_type = NULL;
	gchar *file_type = NULL;
	gchar *tmp = NULL;
	gchar *info_string = NULL;
	int i = 0, j = 0;


	DEBUG_PRINT_START;

	mime_type = gnome_vfs_get_mime_type(path);

	tmp = g_utf8_strrchr(mime_type, MAX_PATH_LENGTH, '/');

	if (tmp != NULL) {
		file_type = g_malloc(g_utf8_strlen(tmp, -1));
		for (i = 1, j = 0; tmp[i] != '\0'; j++, i++) {
			file_type[j] = tmp[i];
		}
		file_type[j] = '\0';
	}

	info_string = g_strdup_printf(" %s name already in use", file_type);

	if (mime_type != NULL) {
		g_free(mime_type);
	}
	if (file_type != NULL) {
		g_free(file_type);
	}

	DEBUG_PRINT_END;

	return info_string;
}

/**
 * on_rename_dialog_okbutton_clicked:
 * Rename operation is done.
 * 
 * @param mainwindow : pointer to Main Window
 * @param oldname  Old name which is to be renamed
 * @param newname  The New name 
 * @return void
 **/
gboolean
on_rename_dialog_okbutton_clicked(GtkWidget * mainwindow,
				  const gchar * oldname, const gchar * newname)
{
	gchar *shortname = NULL;
	gchar *parentpath = NULL;
	gboolean returnVal = FALSE;
	gchar *return_string = NULL;
	GnomeVFSURI *tmp_uri;
	gchar *tmp_newpath;
	DEBUG_PRINT_START;
	if ((mainwindow == NULL) || (oldname == NULL) || (newname == NULL)) {
		FM_ULOG_ERR("Error validating input parameters");
		return FALSE;
	}

	FM_ULOG_DEBUG1("NEW NAME : %s\n", newname);
	tmp_uri = gnome_vfs_uri_new(newname);
	tmp_newpath = gnome_vfs_unescape_string((const char *) gnome_vfs_uri_get_path(tmp_uri),
						NULL);
	gnome_vfs_uri_unref(tmp_uri);
	g_free(tmp_newpath);

	if (is_path_exists(newname) == TRUE && strcasecmp(oldname,newname) != 0){// || strcmp(oldname,newname) == 0)) {
		switch (osso_fm_get_path_type(newname)) {
		case GNOME_VFS_FILE_TYPE_REGULAR:
			return_string = get_rename_file_existing_error(newname);
			fm_infoprint(GTK_WINDOW(mainwindow), return_string);
			if (return_string != NULL) {
				g_free(return_string);
			}
			return FALSE;
			break;
		case GNOME_VFS_FILE_TYPE_DIRECTORY:
			fm_infoprint(GTK_WINDOW(mainwindow),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"ckdg_ib_folder_already_exists"));				      
			return FALSE;
			break;
		case OSSO_FM_MEMORY_ALLOCATION_FAILURE:
			fm_infoprint(GTK_WINDOW(mainwindow), 
					dgettext(KE_RECV_L10N_PACKAGE, "memr_ib_operation_disabled"));
			FM_ULOG_ERR("Memory allocation error\n");
			return FALSE;
			break;
		default:
			fm_infoprint(GTK_WINDOW(mainwindow),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"ckdg_ib_folder_already_exists"));				      
			return FALSE;
			break;
		}
	}

	shortname = osso_fm_texturi_get_shortname(newname);
	parentpath = osso_fm_path_get_dirname(newname);
	FM_ULOG_DEBUG1("Short name : %s\n", shortname);
	FM_ULOG_DEBUG1("Parent name : %s\n", parentpath);

	if (is_path_from_home((const gchar *) parentpath, "MyDocs") == TRUE) {
		FM_ULOG_INFO("\n\nshortname in rename::[%s]\n\n", shortname);
		if (shortname != NULL) {
			if ((g_utf8_collate(shortname, dgettext(HILDON_FM_L10N_PACKAGE,"sfil_li_folder_documents")) == 0) ||
			    (g_utf8_collate(shortname, dgettext(HILDON_FM_L10N_PACKAGE,"sfil_li_folder_images")) == 0) ||
			    (g_utf8_collate(shortname, dgettext(HILDON_FM_L10N_PACKAGE,"sfil_li_folder_sound_clips")) == 0) ||
			    (g_utf8_collate(shortname, dgettext(HILDON_FM_L10N_PACKAGE,"sfil_li_folder_games")) == 0) ||
			    (g_utf8_collate(shortname, dgettext(HILDON_FM_L10N_PACKAGE,"sfil_li_folder_video_clips")) == 0)) {
				DEBUG_PRINT;
				FM_ULOG_ERR("\n\n\n\nRename Error\n\n\n\n");
				fm_infoprint(GTK_WINDOW(mainwindow),
						dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"ckdg_ib_folder_already_exists"));					      
				g_free(shortname);
				g_free(parentpath);
				return FALSE;
			}
		}
	}
	DEBUG_PRINT;
	returnVal = osso_fm_rename_path(mainwindow, oldname, newname);

	DEBUG_PRINT_END;
	g_free(shortname);
	g_free(parentpath);
	return returnVal;
}



/**
 * osso_fm_rename_path:
 * Method to rename source path to target path..
 * 
 * @param mainwindow Main file manager window
 * @param sourcepath source path
 * @param targetpath target path
 * 
 * @return void
 **/
gboolean
osso_fm_rename_path(GtkWidget * mainwindow, const gchar * sourcepath, const gchar * targetpath)
{

	GList *sourcelist = NULL;
	GList *targetlist = NULL;
	GnomeVFSURI *sourceuri = NULL;
	GnomeVFSURI *targeturi = NULL;

	DEBUG_PRINT_START;

	if ((mainwindow == NULL) || (sourcepath == NULL) || (targetpath == NULL)) {
		FM_ULOG_ERR("Error validating input parameters\n");
		return FALSE;
	}
	if (strcmp(sourcepath, targetpath)) {
		FM_ULOG_DEBUG1("sourcepath : %s", sourcepath);
		FM_ULOG_DEBUG1("targetpath : %s", targetpath);

		sourceuri = gnome_vfs_uri_new(sourcepath);
		targeturi = gnome_vfs_uri_new(targetpath);

		if ((sourceuri != NULL) && (targeturi != NULL)) {
			sourcelist = g_list_append(NULL, sourceuri);
			targetlist = g_list_append(NULL, targeturi);

			if ((sourcelist != NULL) && (targetlist != NULL)) {
				xfer_uri(sourcelist, targetlist, mainwindow, XFER_RENAME, 0, 0, FALSE);
			} else {
				FM_ULOG_ERR("Error creating lists");
			}
		}
	} else {
		FM_ULOG_DEBUG1("renamed to same");
	}

	DEBUG_PRINT_END;
	return TRUE;
}
//END: Rename related functions

//BEG: Delete related functions



/**
 * delete_dialog_deletebutton_clicked:
 * Validates the selection and deletes the selected list files and folders
 * @param ofm pointer to OSSOFileManger
 * @return void
 */
static void delete_dialog_deletebutton_clicked(OSSOFileManager *ofm)
{
	GSList *cursel = NULL;
	GList *urilist = NULL;
	GnomeVFSURI *sourceuri = NULL;
	gint error_items = 0;
	DEBUG_PRINT_START;


	for (cursel = ofm->si->current_selection_withinfo; cursel != NULL; cursel = cursel->next) {
		switch (((OSSOFMFileInfo *) cursel->data)->type) {
		case HILDON_FILE_SYSTEM_MODEL_UNKNOWN:
		case HILDON_FILE_SYSTEM_MODEL_LOCAL_DEVICE:
		case HILDON_FILE_SYSTEM_MODEL_MMC:
		case HILDON_FILE_SYSTEM_MODEL_GATEWAY:
		case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_DOCUMENTS:
		case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_IMAGES:
		case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_SOUNDS:
		case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_VIDEOS:
		case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_GAMES:
			error_items++;
			break;
		case HILDON_FILE_SYSTEM_MODEL_FOLDER:
		case HILDON_FILE_SYSTEM_MODEL_FILE:
			sourceuri = osso_fm_path_to_uri(
					((OSSOFMFileInfo *) cursel->data)->path);
			FM_ULOG_DEBUG1("cursel data  : %s\n", 
					osso_fm_path_to_texturi (
						((OSSOFMFileInfo *) cursel->data)->path));
			if (sourceuri != NULL) {
				urilist = g_list_append(urilist, sourceuri);
			} else {
				FM_ULOG_ERR("Err getting vfs_uri for path %s\nTrying to continue\n",
						osso_fm_path_to_texturi(
							((OSSOFMFileInfo *) cursel->data)->path));
				error_items ++;
			}
			break;
		}
	}

	if (urilist != NULL) {
		xfer_uri(urilist, NULL, ofm->window, XFER_DELETE, error_items, 0,FALSE);
	}
	else {
		fm_infoprint(GTK_WINDOW(ofm->window),
			       dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
				       "sfil_ib_unable_to_delete_selected_items"));
	}
	DEBUG_PRINT_END;
	return;
}


/**
 * get_del_conf_note:
 * A Helper function to get the conf note string for deleted operation
 * @param seltype info about selection
 * @param si pointer to si structure
 * @return gchar * conf note string
 **/
static gchar *get_del_conf_note(SelectionInfoType seltype,SelectionInfo *si)
{
	gchar *note = NULL;
	gchar *display_name = NULL;

	DEBUG_PRINT_START;
	switch(seltype)
	{
		case SIT_SINGLE_FOLDER_SELECTED:
			display_name = osso_fm_path_get_display_name (si->current_selection->data);
			note = g_strdup_printf("%s\n%s",_("docm_nc_delete_folder"),display_name);
			break;
		case SIT_SINGLE_FILE_SELECTED:
			display_name = osso_fm_path_get_display_name (si->current_selection->data);
			if(si->no_ronly_files > 0)
				note = g_strdup_printf("%s\n%s",_("sfil_nc_delete_read_only_file"),
						display_name);
			else
				note = g_strdup_printf("%s\n%s",_("docm_nc_delete_file"),
						display_name);
			break;
		case SIT_MULTIPLE_FILES_SELECTED:
			if(si->no_ronly_files == 0)
				note =  g_strdup(_("docm_nc_delete_files"));
			else if(si->no_ronly_files == si->no_files)
				note = g_strdup(_("sfil_nc_delete_read_only_files"));
			else
				note = g_strdup(_("sfil_nc_delete_partially_read_only_items"));
			break;
		case SIT_MULTIPLE_FOLDERS_SELECTED:
			note = g_strdup(_("docm_nc_delete_folders"));
			break;
		case SIT_MULTIPLE_ITEMS_SELECTED:
			if(si->no_ronly_files + si->no_ronly_folders > 0)
				note = g_strdup(_("sfil_nc_delete_partially_read_only_items"));
			else if(si->no_files == 1)
			{
				if(si->no_folders == 1)
					note = g_strdup(_("docm_nc_delete_folder_and_file"));
				else
					note = g_strdup(_("docm_nc_delete_folders_and_file"));
			}
			else
			{
				if(si->no_folders == 1)
					note = g_strdup(_("docm_nc_delete_folder_and_files"));
				else
					note = g_strdup(_("docm_nc_delete_folders_and_files"));
			}
			break;		
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			break;
		default:
			FM_ULOG_ERR("get_del_conf_note:default NOT EXPECTED\n");
			break;
	}
	DEBUG_PRINT_END;
	if(display_name != NULL){
		g_free(display_name);
	}
	return note;
}


/**
 * get_del_err_for_preadonly:
 * A Helper function to get the error string for deleted operation
 * @param seltype info about selection
 * @return gchar * error string
 **/
static gchar *get_del_err_for_preadonly(SelectionInfoType seltype)
{
	switch(seltype)
	{
		case SIT_SINGLE_FILE_SELECTED:
			return dgettext(HILDON_FM_L10N_PACKAGE,"sfil_ib_cannot_delete_file");
			break;
		case SIT_SINGLE_FOLDER_SELECTED:
			return  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"sfil_ib_unable_to_delete_selected_folder");
			break;
		case SIT_MULTIPLE_FILES_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
			return  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"sfil_ib_unable_to_delete_selected_items");
			break;
		default:
			FM_ULOG_ERR("get_del_err_for_preadonly: Unexpected value %d\n",seltype);
			return  dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"sfil_ib_unable_to_delete_selected_items");
	}
}
	
/**
 * osso_fm_delete_activate:
 * Validates the selection type  and creates
 * the delete confirmation dialog
 * @param ofm Pointer to OSSOFileManager
 * @param menucheck
 * @return gboolean 
 **/
gboolean osso_fm_delete_activate(OSSOFileManager *ofm, gboolean menucheck)
{
	gchar *note = NULL;
	gboolean retval = FALSE;
	gboolean readonly_path = FALSE;
	SelectionInfoType seltype;// = SelectionInfoType:0;
	DEBUG_PRINT_START;
	
	selection_info_update(ofm,!menucheck);

	if(!ofm->si->selection_in_gateway)
	{
		readonly_path = is_path_readonly(ofm->si->current_selection->data,TRUE);
	}
	if(check_for_media_items(OPER_DELETE,ofm,menucheck))
	{
		return FALSE;
	}
	seltype = selection_info_get_selection_type(ofm->si);
	switch(seltype)
	{
		case SIT_SINGLE_SAFE_FOLDER_SELECTED:
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window),
					       dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						       "sfil_ib_unable_to_delete_selected_folder"));
			}
			retval = FALSE;
			break;
		case SIT_MULTIPLE_SAFE_FOLDERS_SELECTED:
			
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window),
					       dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						       "sfil_ib_unable_to_delete_selected_items"));
			}
			retval = FALSE;
			break;
		case SIT_DEVICE_SELECTED:
			if(ofm->si->selection_in_gateway)
			{
				retval=FALSE;
				if (!menucheck) {
					if(is_path_virtual(ofm->si->current_selection->data)) {
						fm_infoprint(GTK_WINDOW(ofm->window), _("sfil_ib_unable_delete_bluetooth"));
					} else {
						fm_infoprint(GTK_WINDOW(ofm->window), 
							_("sfil_ib_cannot_delete_device"));
					}
				}
				break;
			}
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window), 
						_("sfil_ib_cannot_delete_device"));
			}
			break;
			
		case SIT_MMC_SELECTED:
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window), 
						_("sfil_ib_cannot_delete_mmc"));
			}
			retval = FALSE;
			break;
			
		case SIT_SINGLE_FOLDER_SELECTED:
		case SIT_SINGLE_FILE_SELECTED:
		case SIT_MULTIPLE_FILES_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
			retval = !(readonly_path && !ofm->si->selection_in_gateway);
			if (menucheck) break;
			if(readonly_path && !ofm->si->selection_in_gateway)
			{
				fm_infoprint(GTK_WINDOW(ofm->window),
						get_del_err_for_preadonly(seltype)); 
				retval = FALSE;
				break;
			}
			note = get_del_conf_note(seltype,ofm->si);
			
			if(fm_confnote(GTK_WINDOW(ofm->window), NULL,(const gchar *) note,
					       _("docm_bd_delete_ok"),
					       GTK_RESPONSE_OK,
					       _("docm_bd_delete_cancel"),
					       GTK_RESPONSE_CANCEL,TRUE) == GTK_RESPONSE_OK)
			{
				delete_dialog_deletebutton_clicked(ofm);
			}
			g_free(note);
			break;
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			retval = FALSE;
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			retval = FALSE;
			break;
	}
	return retval;
}

//END: Delete related functions

//BEG Sort Dialog related functions


/** 
 * sort_dialog:
 * Method to invoke the sortdialog.
 * @param parent Main file manager window
 * @return void
 **/
void sort_dialog(GtkWindow * parent)
{
	HildonSortDialog *sort_dialog = NULL;
	gint response_id = 0;

	HildonFileSelectionSortKey key = 0;

	GtkSortType order = 0;

	DEBUG_PRINT_START;

	if (parent == NULL) {
		FM_ULOG_ERR("Invalid parameter : window");
		return;
	}

	sort_dialog = HILDON_SORT_DIALOG(hildon_sort_dialog_new(parent));

	g_return_if_fail(sort_dialog);

	hildon_sort_dialog_add_sort_key(sort_dialog, dgettext(HILDON_LIBS_L10N_PACKAGE,"ckdg_va_sort_name"));
	hildon_sort_dialog_add_sort_key(sort_dialog, _("docm_li_sort_by_type"));
	hildon_sort_dialog_add_sort_key(sort_dialog, dgettext(HILDON_LIBS_L10N_PACKAGE,"ckdg_va_sort_date"));
	hildon_sort_dialog_add_sort_key(sort_dialog, dgettext(HILDON_LIBS_L10N_PACKAGE,"ckdg_va_sort_size"));


	hildon_file_selection_get_sort_key(global_ofm->fs, &key, &order);

	FM_ULOG_DEBUG1("SORT KEY : %d,  SORT_ORDER : %d\n", key, order);

	hildon_sort_dialog_set_sort_key(sort_dialog, key);

	hildon_sort_dialog_set_sort_order(sort_dialog, order);

	response_id = gtk_dialog_run(GTK_DIALOG(sort_dialog));

	if (response_id == GTK_RESPONSE_OK) {
		key = hildon_sort_dialog_get_sort_key(sort_dialog);
		order = hildon_sort_dialog_get_sort_order(sort_dialog);

		FM_ULOG_DEBUG1("SORT KEY : %d,  SORT_ORDER : %d\n", key, order);

		osso_fm_sort(global_ofm->fs, key, order);

	}

	gtk_widget_destroy(GTK_WIDGET(sort_dialog));

	DEBUG_PRINT_END;
	return;
}

//END: Sort Dialog related functions

enum MOVETO_OK_STATUS {	MTOK_UNKNOWN_ERR = -1,
			MTOK_ERROR_REDISPLAY = 0,
			MTOK_ERROR_NOTSUPPORTED_IN_BT = 1,
			MTOK_OPERATION_CANCELED = 2,
			MTOK_SUCCESSFUL = 3};
/**
 * on_move_dialog_okbutton_clicked:
 * Validates the selected source list and target list
 * If valid the moves the source list to the specfied target location.
 * 
 * @param move_dialog pointer to Moveto dialog 
 * @param sourcefs Pointer to the File Selection widget
 * 
 * @return gboolean
 */
static gint on_move_dialog_okbutton_clicked(GtkWidget * move_dialog, gpointer sourcefs)
{
	gchar *targetpath_texturi = NULL;
	GtkFilePath *targetpath = NULL;
	GtkFilePath *sourcepath = NULL;
	GtkWidget *mainwindow = NULL;
	GtkWidget *window = NULL;
	GSList *sourceselection = NULL;
	GList *sourcelist = NULL;
	GList *targetlist = NULL;
	HildonFileSystemModel *fsmodel = NULL;
	gint removed_items = 0;
	gint nonexistant_items = 0;
	SrcToTgtStatus sttsres;

	DEBUG_PRINT_START;

	if ((sourcefs == NULL) || (move_dialog == NULL)) {
		FM_ULOG_ERR("Error validating input parameters");
		return MTOK_UNKNOWN_ERR;
	}
	g_object_get(G_OBJECT(sourcefs), "model", &fsmodel, NULL);
	if (fsmodel == NULL) {
		FM_ULOG_ERR("Error getting model\n");
		return MTOK_UNKNOWN_ERR;
	}

	targetpath_texturi = gtk_file_chooser_get_uri(GTK_FILE_CHOOSER(move_dialog));
	if (targetpath_texturi == NULL) {
		FM_ULOG_ERR("Error obtaining target path");
		return MTOK_UNKNOWN_ERR;
	}
	if(is_texturi_virtual(targetpath_texturi))
	{
		fm_infonote(GTK_WINDOW(global_ofm->window), 
			dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
				"sfil_ni_operation_failed"), 
			dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
				"sfil_ni_operation_failed_ok"));
		return MTOK_UNKNOWN_ERR;
	}
	if(selected_location_in_gateway((HildonFileSelection *) sourcefs) && 
			is_texturi_in_gateway(targetpath_texturi))
	{
		FM_ULOG_INFO("Source and target are in gateway\n");
		return MTOK_ERROR_NOTSUPPORTED_IN_BT;
	}

	if (is_texturi_readonly(targetpath_texturi, FALSE)) {
		fm_infoprint(GTK_WINDOW(window),
			dgettext(HILDON_FM_L10N_PACKAGE,"sfil_ib_readonly_location"));
		if (targetpath_texturi != NULL) {
			g_free(targetpath_texturi);
		}


		return MTOK_ERROR_REDISPLAY;
	}

	FM_ULOG_DEBUG1("Target path : %s\n", targetpath_texturi);

	if ((hildon_file_selection_get_active_pane(sourcefs) ==
	     HILDON_FILE_SELECTION_PANE_CONTENT) || (global_ofm->si->selected_all == TRUE)) {
		sourceselection = hildon_file_selection_get_selected_paths(sourcefs);
	}
	if (sourceselection == NULL) {
		sourcepath = hildon_file_selection_get_current_folder(sourcefs);
		if (sourcepath != NULL) {
			sourceselection = g_slist_append(sourceselection,
					gtk_file_path_copy(sourcepath));
		}
	}
	//remove safe items from list

	removed_items = remove_safe_folders_from_gslist(sourcefs, &sourceselection, 
			&nonexistant_items,FALSE);
	FM_ULOG_INFO("removed_items %d nonexistant_items %d\n",removed_items,nonexistant_items);
	if (sourceselection == NULL) {
		if(selectionstate.no_folders == 1 && selectionstate.no_files == 0)
		{
			if(removed_items == nonexistant_items)
			{
				fm_infonote(GTK_WINDOW(global_ofm->window),
						_("sfil_ni_cannot_move_items_not_found"),
						_("sfil_bd_cannot_move_items_not_found_ok"));
			} else
			{
				fm_infoprint(GTK_WINDOW(global_ofm->window),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"sfil_ib_unable_to_move_selected_folder"));
			}
				
	        } else if(selectionstate.no_folders == 0 && selectionstate.no_files == 1) {
			if(removed_items == nonexistant_items)
			{
				fm_infonote(GTK_WINDOW(global_ofm->window),
						_("sfil_ni_cannot_move_file_not_found"),
						_("sfil_bd_cannot_move_file_not_found_ok"));
			} else
			{
				fm_infoprint(GTK_WINDOW(global_ofm->window),
					dgettext(HILDON_FM_L10N_PACKAGE,
						"sfil_ib_cannot_move_file"));
			}
                } else {
			if(removed_items == nonexistant_items)
			{
				fm_infonote(GTK_WINDOW(global_ofm->window),
						_("sfil_ni_cannot_move_items_not_found"),
						_("sfil_bd_cannot_move_items_not_found_ok"));
			} else
			{
				fm_infoprint(GTK_WINDOW(global_ofm->window),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ib_unable_to_move_selected_items"));
			}
		}
		return MTOK_UNKNOWN_ERR;
	}

	targetpath = osso_fm_texturi_to_path(targetpath_texturi);
	sttsres = check_source_to_target(sourceselection, targetpath);
	gtk_file_path_free(targetpath);
	switch (sttsres) {
	case STTS_MEMORY_ERR:
	case STTS_TGT_IN_SRC:
		fm_infoprint(GTK_WINDOW(window),
//			     osso_common_error(CERM_CAN_NOT_MV_FOLDER_INTO_ITSELF));
				dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"ckdg_ib_unable_to_move_folder_into_itself"));			     
		if (targetpath_texturi != NULL) {
			g_free(targetpath_texturi);
		}
		if (sourceselection != NULL) {
			g_slist_free(sourceselection);
			sourceselection = NULL;
		}
		if (sourcepath != NULL) {
			gtk_file_path_free(sourcepath);
		}
		return MTOK_ERROR_REDISPLAY;
		break;
	case STTS_TGT_REPLACES_SRC:
		fm_infoprint(GTK_WINDOW(window),
		      dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
			      "sfil_ib_unable_to_copy_or_move_a_subfolder_within_original_folder"));
		if (targetpath_texturi != NULL) {
			g_free(targetpath_texturi);
		}
		if (sourceselection != NULL) {
			g_slist_free(sourceselection);
			sourceselection = NULL;
		}
		if (sourcepath != NULL) {
			gtk_file_path_free(sourcepath);
		}
		return MTOK_ERROR_REDISPLAY;
		break;
	case STTS_OK:
		break;
	}
	targetpath = osso_fm_texturi_to_path(targetpath_texturi);
	if (match_parents_path(sourceselection, targetpath)) {
		fm_infoprint(GTK_WINDOW(window),
//			     osso_common_error(CERM_CAN_NOT_MOVE_ORIG_LOCATION));
				dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"ckdg_ib_unable_to_move_to_current_location"));			     

		if (sourceselection != NULL) {
			g_slist_free(sourceselection);
			sourceselection = NULL;
		}
		if (targetpath_texturi != NULL) {
			g_free(targetpath_texturi);
			targetpath_texturi = NULL;
		}
		gtk_file_path_free(targetpath);
		return MTOK_ERROR_REDISPLAY;
	}
	gtk_file_path_free(targetpath);
	targetpath = osso_fm_texturi_to_path(targetpath_texturi);
	make_uri_list(sourceselection, &sourcelist,
		      targetpath, &targetlist, fsmodel, FALSE);
	gtk_file_path_free(targetpath);
	if ((sourcelist != NULL) && (targetlist != NULL)) {
		mainwindow = gtk_widget_get_ancestor(sourcefs, GTK_TYPE_WINDOW);
		/* Move the selected source list to the selected target location */
		xfer_uri(sourcelist, targetlist, mainwindow, XFER_MOVE, removed_items, nonexistant_items, FALSE);
	}
	if (targetpath_texturi != NULL) {
		g_free(targetpath_texturi);
	}
	if (sourceselection != NULL) {
		g_slist_free(sourceselection);
		sourceselection = NULL;
	}
	if (sourcepath != NULL) {
		gtk_file_path_free(sourcepath);
	}
	DEBUG_PRINT_END;
	return MTOK_SUCCESSFUL;
}
		
/**
 * create_move_to_dialog:
 * This functions creates the moveto dialog.
 * 
 * @param window	Pointer to Main window of File Manager Application
 * @param sourcefs	Selection widget of source files
 * 
 * @return void
 **/
void create_move_to_dialog(GtkWindow * window, HildonFileSelection * sourcefs)
{

	GtkWidget *MoveDialog = NULL;
	HildonFileSystemModel *fsmodel = NULL;
	gchar *cur_folder_texturi = NULL;
	GtkFilePath *cur_folder_path = NULL;
	gint response = 0;
	gint retval = 0;

	DEBUG_PRINT_START;
	if ((sourcefs == NULL) || (window == NULL)) {
		FM_ULOG_ERR("Invalid parameters : sourcefs/window (null)");
		return;
	}
	g_object_get(sourcefs, "model", &fsmodel, NULL);
	if (fsmodel != NULL) {
		MoveDialog = hildon_file_chooser_dialog_new_with_properties((GtkWindow *) window,
									    "file-system-model",
									    fsmodel, 
									    "action",
									    GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
									    NULL);
	} else {
		FM_ULOG_DEBUG1("Unable to get fsmodel\n");
		MoveDialog = hildon_file_chooser_dialog_new(window,
							    GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
	}
	
	if (MoveDialog == NULL) {
		FM_ULOG_ERR("Unable to Allocate memory for MoveTo dlg\n");
		return;
	}
	hildon_file_chooser_dialog_set_show_upnp(HILDON_FILE_CHOOSER_DIALOG(MoveDialog),FALSE);
	
	cur_folder_path = hildon_file_selection_get_current_folder(sourcefs);
	if (cur_folder_path == NULL) {
		FM_ULOG_ERR("Error getting current folder\n");
		return;
	}
	cur_folder_texturi = osso_fm_path_to_texturi(cur_folder_path);
	gtk_file_chooser_set_current_folder_uri(GTK_FILE_CHOOSER(MoveDialog), cur_folder_texturi);
	gtk_window_set_title(GTK_WINDOW(MoveDialog), dgettext(HILDON_FM_L10N_PACKAGE,
				"ckdg_ti_move_to_folder"));
	gtk_window_set_modal(GTK_WINDOW(MoveDialog), TRUE);
	gtk_window_set_resizable(GTK_WINDOW(MoveDialog), FALSE);
	register_selection_change_sensitive_dialog(global_ofm,MoveDialog);
	do {
		DEBUG_PRINT;
		response = gtk_dialog_run(GTK_DIALOG(MoveDialog));
		if (response == GTK_RESPONSE_OK) {
			retval = on_move_dialog_okbutton_clicked(MoveDialog, sourcefs);
		} else {
			retval = MTOK_OPERATION_CANCELED;
		}
	} while (retval == MTOK_ERROR_REDISPLAY);
	unregister_selection_change_sensitive_dialog(global_ofm,MoveDialog);

	gtk_widget_destroy(MoveDialog);
	if(retval == MTOK_ERROR_NOTSUPPORTED_IN_BT) {
		fm_infonote(window,_("sfil_ni_move_not_supported_bluetooth"),NULL);
	}
		
	gtk_file_path_free(cur_folder_path);
	g_free(cur_folder_texturi);
	DEBUG_PRINT_END;
	return;
}



