/**
 * @file  osso_fm_fmstate.c
 *
 * This file contains the implemenation for the state management of
 * file manager application.
 *
 * Copyright (C) 2005 Nokia Corporation
 */


#include "osso_fm_generic.h"
#include "osso_fm_controller.h"
#include "osso_fm_fmstate.h"
#include "osso_fm_callbacks.h"
#include "osso_fm_utils.h"

gchar *filedata;

void load_default_state()
{
	//strcpy (appstate.mainwindow.currentselection,   "");    
	osso_fm_set_currentselection("");
	//appstate.mainwindow.viewnormal = TRUE;  //TRUE if view mode is normal
	appstate.mainwindow.fullscreen = FALSE;	//TRUE if view mode is fullscreen
	appstate.mainwindow.opendialog = OPEN_DIALOG_NONE;
}


/**
 * load_fm_state:
 * Loads File manager state from the file
 * @param void
 * @return void
 */
void load_fm_state()
{
  osso_return_t ret;
  osso_state_t state;
  
  gchar **paths = NULL;
  gint i = 0;
  GSList *selection = NULL;
  gboolean nav_pane_selected = FALSE;
  
  DEBUG_PRINT_START;
	
  state.state_size = 0;
  state.state_data = NULL;
  ret = osso_state_read (fm_app_ctx, &state);

  if (ret != OSSO_OK)
    {
      load_default_state ();
      return;
    }
	  
  if (state.state_size < 15) 
    {
      DEBUG_PRINT load_default_state();
    } 
  else
    {
      FM_ULOG_DEBUG1("\n********* FM state data load *********\n");
      FM_ULOG_DEBUG1("%s", (char *)state.state_data);
      FM_ULOG_DEBUG1("\n********* FM state data load end *****\n");
      
      paths = g_strsplit((char *)state.state_data, "\n", 0);
      if (paths[0] == NULL) {
	return;
      }
      if (g_strcasecmp(paths[0], "Fullscreen") == 0) {
	appstate.mainwindow.fullscreen = TRUE;
      } else {
	appstate.mainwindow.fullscreen = FALSE;
      }

      if (g_strcasecmp(paths[1], "Details") == 0) {
	appstate.mainwindow.opendialog = OPEN_DIALOG_DETAILS;
      } else {
	appstate.mainwindow.opendialog = OPEN_DIALOG_NONE;
      }
      if (g_strcasecmp(paths[2], "ContentPane") == 0) {
	nav_pane_selected = FALSE;
      } else {
	nav_pane_selected = TRUE;
      }
      if (paths[3] == NULL) {
	return;
      }
      //strcpy (appstate.mainwindow.currentselection, paths[3]);
      osso_fm_set_currentselection(paths[3]);
      for (i = 4; paths[i] != NULL; i++) {
	if (paths[i] != NULL) {
	  selection = g_slist_append(selection, g_strdup(paths[i]));
	} else {
	  break;
	}
      }
      appstate.mainwindow.selection = selection;
	  
      g_strfreev(paths);
      set_appliaction_state(nav_pane_selected);
    }

  g_free(state.state_data);

  DEBUG_PRINT_END;
}

/**
 * save_fm_state:
 * Saves File manager state to the file
 * @param void
 * @return void
 */
void save_fm_state()
{
  GSList *cursel = NULL;
  GString *statedata = NULL;
  GSList *selection = NULL;
  GtkFilePath *path = NULL;
  gchar *path_texturi = NULL;

  osso_return_t ret;
  osso_state_t state;

  DEBUG_PRINT_START;

  if (global_ofm->cur_operation == OPER_CLOSE) {
    clear_fm_state();
    return;
  }

  statedata = g_string_new(appstate.mainwindow.fullscreen ? "Fullscreen" : "Normal");
	
  g_string_append_c(statedata, '\n');
  if (appstate.mainwindow.opendialog == OPEN_DIALOG_DETAILS) {
    g_string_append(statedata, g_strdup("Details"));
  } else {
    g_string_append(statedata, g_strdup("None"));
  }
  g_string_append_c(statedata, '\n');
  if (hildon_file_selection_get_active_pane(global_ofm->fs) ==
      HILDON_FILE_SELECTION_PANE_CONTENT) {
    g_string_append(statedata, g_strdup("ContentPane"));
  } else {
    g_string_append(statedata, g_strdup("NavigationPane"));
  }
  g_string_append_c(statedata, '\n');
  path = hildon_file_selection_get_current_folder(global_ofm->fs);
  g_string_append(statedata, osso_fm_path_to_texturi(path));
  gtk_file_path_free(path);
	
  selection = hildon_file_selection_get_selected_paths(global_ofm->fs);
  for (cursel = selection; cursel != NULL; cursel = g_slist_next(cursel)) {
    if (cursel->data != NULL) {
      g_string_append_c(statedata, '\n');
      path_texturi = osso_fm_path_to_texturi(cursel->data);
      g_string_append(statedata, path_texturi);
      g_free(path_texturi);
    }
  }
  gtk_file_paths_free(selection);
  g_string_append_c(statedata, '\n');
  g_string_append_c(statedata, '\0');
	
  FM_ULOG_DEBUG1("\n********* FM state data save *********\n");
  FM_ULOG_DEBUG1("%s", statedata->str);
  FM_ULOG_DEBUG1("\n********* FM state data save end *****\n");

  state.state_data = statedata->str;
  state.state_size = strlen (state.state_data);
  ret = osso_state_write (fm_app_ctx, &state);
  if (ret != OSSO_OK)
    FM_ULOG_ERR("Error writing state\n");

  g_string_free (statedata, 1);

  DEBUG_PRINT_END;
}


/**
 * clear_fm_state:
 * Saves default file manager state to the file
 * @param void
 * @return void
 */
void clear_fm_state()
{
  osso_return_t ret;
  osso_state_t state;

  if(fm_app_ctx == NULL)
    {
      FM_ULOG_ERR("clear_fm_state: fm_app_ctx == NULL");
      return;
    }

  state.state_data = "Closed";
  state.state_size = strlen (state.state_data);
  ret = osso_state_write (fm_app_ctx, &state);
  if (ret != OSSO_OK)
    FM_ULOG_ERR("Error writing state\n");
}


/**
 * set_appliaction_state:
 * Sets the File manager application state to the loaded state
 * @param app 	Pointer to the HildonApp of the Application
 * @return void
 */
void set_appliaction_state(gboolean nav_pane_active)
{
	GSList *cursel = NULL;
	HildonWindow *window = NULL;
	GError *error = NULL;

	DEBUG_PRINT_START;
	window = global_ofm->window;
	if (window == NULL) {
		FM_ULOG_ERR("Error getting view \n");
		return;
	}
	hildon_file_selection_set_current_folder(global_ofm->fs,
						 (GtkFilePath *) appstate.mainwindow.
						 currentselection, &error);
	if (appstate.mainwindow.fullscreen) {
	  // on_menu_full_screen_activate(NULL, window);
	}

	for (cursel = appstate.mainwindow.selection; cursel != NULL; cursel = g_slist_next(cursel)) 
	{
		hildon_file_selection_select_path(global_ofm->fs, cursel->data, NULL);
	}
	if (nav_pane_active) {
		g_object_set(global_ofm->fs, 
				"active-pane", 
				HILDON_FILE_SELECTION_PANE_NAVIGATION, 
				NULL);
	}
	if (appstate.mainwindow.opendialog == OPEN_DIALOG_DETAILS) {
		g_signal_emit_by_name(global_ofm->fmmmenu->file->details, 
				"activate", 
				global_ofm->window);

	}
	DEBUG_PRINT_END;
	return;
}

/**
 * osso_fm_set_currentselection:
 * Sets the currentselection by freeing the previous selection
 * @param newcursel
 * @return void
 */
void osso_fm_set_currentselection(const gchar * newcursel)
{
	if (appstate.mainwindow.currentselection != NULL) {
		g_free(appstate.mainwindow.currentselection);
	}
	appstate.mainwindow.currentselection = g_strdup(newcursel);
}

/**
 * selection_info_init:
 * Initialises the SelectionInfo structure
 * @param si SelectionInfo structure to be initialised
 * @return void
 **/
void selection_info_init(SelectionInfo *si)
{
	si->selection = FALSE;
	si->ronly_parent = FALSE;
	si->no_files = 0;
	si->no_folders = 0;
	si->no_ronly_files = 0;
	si->no_ronly_folders = 0;
	si->no_safe_folders = 0;
	si->no_devices = 0;
	si->no_mmc = 0;
	si->needs_update = TRUE;
	si->current_folder = NULL;
	si->current_selection = NULL;
	si->current_selection_withinfo = NULL;
	si->ro_status_needs_update = TRUE;
	si->selected_all = FALSE;
	si->just_selected_all = FALSE;
	si->selection_in_gateway = FALSE;
	si->selection_in_mmc = FALSE;
}
/**
 * selection_info_members_free
 * Frees the SelectionInfo structure
 * @param si SelectionInfo structure to be freed
 * @return void
 **/
static void
free_file_info (OSSOFMFileInfo *info)
{
  g_free (info->extra);
  g_free (info);
}

void selection_info_members_free(SelectionInfo *si)
{
	DEBUG_PRINT_START;
	if(si->current_folder != NULL)
	{
		gtk_file_path_free(si->current_folder);
		si->current_folder = NULL;
	}
	if(si->current_selection != NULL)
	{
		gtk_file_paths_free(si->current_selection);
		si->current_selection = NULL;
	}
	if(si->current_selection_withinfo != NULL)
	{
		g_slist_foreach(si->current_selection_withinfo,
				(GFunc)free_file_info, NULL);
		si->current_selection_withinfo = NULL;
	}
	DEBUG_PRINT_END;
}
/**
 * selection_info_reset:
 * Resets the SelectionInfo structure
 * @param si SelectionInfo structure to be reset
 * @return void
 **/
void selection_info_reset(SelectionInfo *si)
{
	DEBUG_PRINT_START;
	si->selection = FALSE;
	si->ronly_parent = FALSE;
	si->no_files = 0;
	si->no_folders = 0;
	si->no_ronly_files = 0;
	si->no_ronly_folders = 0;
	si->no_safe_folders = 0;
	si->no_devices = 0;
	si->no_mmc = 0;
	si->needs_update = TRUE;
	si->ro_status_needs_update = TRUE;
	DEBUG_PRINT_END;
}

/**
 * selection_info_set_current_folder:
 * sets the current folder in SelectionInfo struct
 * @param si SelectionInfo
 * @param cur_folder new value
 * @return void
 **/
void selection_info_set_current_folder(SelectionInfo *si,GtkFilePath *cur_folder)
{
	DEBUG_PRINT_START;
	if(si->current_folder != NULL)
		gtk_file_path_free(si->current_folder);
	si->current_folder = cur_folder;
	DEBUG_PRINT_END;
}
/**
 * selection_info_set_current_selection:
 * sets the current selection in SelectionInfo struct
 * @param si SelectionInfo
 * @param cur_sel new list
 * @return void
 **/
void selection_info_set_current_selection(SelectionInfo *si,GSList *cur_sel)
{
	DEBUG_PRINT_START;
	if(si->current_selection != NULL)
		gtk_file_paths_free(si->current_selection);
	si->current_selection = cur_sel;
	g_slist_foreach(si->current_selection_withinfo,
			(GFunc)free_file_info,NULL);
	si->current_selection_withinfo = NULL;
	DEBUG_PRINT_END;
}

/**
 * si_update_path_info:
 * A Helper function to update the si struture
 * @param si SelectionInfo
 * @param path GtkFilePath
 * @param type HildonFileSystemModelItemType
 * @param real gboolean
 * @return void
 **/
static void si_update_path_info(SelectionInfo *si,GtkFilePath *path,HildonFileSystemModelItemType type,gboolean real, char *extra)
{
	OSSOFMFileInfo *info = NULL;
	DEBUG_PRINT_START;
	info = g_new0(OSSOFMFileInfo,1);
	info->path = path;
	info->extra = extra;
	info->type = type;
	info->readonly = FALSE;
	switch(type)
	{
		case HILDON_FILE_SYSTEM_MODEL_UNKNOWN:
			FM_ULOG_ERR("Undefined type of item in model\n");
			break;
		case HILDON_FILE_SYSTEM_MODEL_FILE:
			si->no_files++;
			if(real && !si->selection_in_gateway)
			{
				info->readonly = is_path_readonly(path,FALSE);
				if(info->readonly)
					si->no_ronly_files++;
			}
			break;
		case HILDON_FILE_SYSTEM_MODEL_FOLDER:
			si->no_folders++;
			if(real && !si->selection_in_gateway)
			{
				info->readonly = is_path_readonly(path,FALSE);
				if(info->readonly)
					si->no_ronly_folders++;
			}
			break;
		case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_IMAGES:
		case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_VIDEOS:
		case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_SOUNDS:
		case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_DOCUMENTS:
		case HILDON_FILE_SYSTEM_MODEL_SAFE_FOLDER_GAMES:
			si->no_folders++;
			si->no_safe_folders++;
			if(real && !si->selection_in_gateway)
			{
				info->readonly = is_path_readonly(path,FALSE);
				if(info->readonly)
					si->no_ronly_folders++;
			}
			break;
		case HILDON_FILE_SYSTEM_MODEL_MMC:
			si->no_folders++;
			si->no_mmc++;
			break;
		case HILDON_FILE_SYSTEM_MODEL_GATEWAY:
		case HILDON_FILE_SYSTEM_MODEL_LOCAL_DEVICE:
			si->no_folders++;
			si->no_devices++;
			break;
	}//end of switch
	si->current_selection_withinfo = 
		g_slist_append(si->current_selection_withinfo,info);
	DEBUG_PRINT_END;
	
}

/**
 * selection_info_update:
 * Updates the selection info structure with the current selection
 * @param ofm ptr to OSSOFileManager
 * @param real gboolean
 * @return gboolean
 **/
gboolean selection_info_update(OSSOFileManager *ofm,gboolean real)
{
	GSList *cursel = NULL;
	GSList *selection = NULL;
	GtkFilePath *path = NULL;
	char *extra = NULL;
	HildonFileSystemModel *fsmodel = NULL;
	GtkTreeIter iter;
	HildonFileSystemModelItemType type;
	SelectionInfo *si = ofm->si;

	DEBUG_PRINT_START;

	if (!si->needs_update && !real) {
		return TRUE;
	}
	if (!si->ro_status_needs_update && real) {
		return TRUE;
	}
	selection_info_reset(si);

	if ((hildon_file_selection_get_active_pane(ofm->fs) ==
	     HILDON_FILE_SELECTION_PANE_CONTENT) || (si->selected_all == TRUE)) {
		selection = hildon_file_selection_get_selected_paths(ofm->fs);
	}
	path = hildon_file_selection_get_current_folder(ofm->fs);

	if ((selection == NULL) && (path != NULL)) {
		FM_ULOG_DEBUG1("Selection taken from content pane\n");
		si->selection = TRUE;
		selection = g_slist_append(selection, gtk_file_path_copy(path));
	}
	selection_info_set_current_folder(si,path);
	selection_info_set_current_selection(si,selection);

	si->selection_in_gateway = selected_location_in_gateway(ofm->fs);
	si->selection_in_mmc = selected_location_in_mmc(ofm->fs);
	si->selection_in_media_server = selected_location_in_media_server(ofm->fs);

	g_object_get(G_OBJECT(ofm->fs), "model", &fsmodel, NULL);
	if (fsmodel == NULL) {
		FM_ULOG_ERR("Unable to get Model\n");
		return FALSE;
	}
	for (cursel = selection; cursel; cursel = g_slist_next(cursel)) {
		FM_ULOG_DEBUG1("selection_info_update CURSEL PATH: %s\n",
			       (char *) cursel->data);
		if (!hildon_file_system_model_search_path
		    (fsmodel, cursel->data, &iter, NULL, TRUE)) {
			FM_ULOG_ERR("Unable to find path in the model %s\n",(gchar *)cursel->data);
			continue;
		}
		gtk_tree_model_get
		  ((GtkTreeModel *) fsmodel, &iter,
		   HILDON_FILE_SYSTEM_MODEL_COLUMN_TYPE, &type,
		   HILDON_FILE_SYSTEM_MODEL_COLUMN_EXTRA_INFO, &extra,
		   -1);
		si_update_path_info(ofm->si,cursel->data,type,real, extra);
	}//end of for

	if(real && !si->selection_in_gateway )
	{
		si->ronly_parent = is_path_readonly(selection->data,TRUE);
		si->ro_status_needs_update = FALSE;
	}

	DEBUG_PRINT_END;
	si->needs_update = FALSE;
	return TRUE;
}

/**
 * get_selection_type_for_nofiles
 * A Helper function for updating SelectionInfo
 * @param si SelectionInfo
 * @return SelectionInfoType
 **/
static SelectionInfoType get_selection_type_for_nofiles(SelectionInfo *si)
{
	if(si->no_folders == 0)
	{
		return SIT_NO_ITEM_SELECTED;
	}
	else if(si->no_folders == 1)
	{
		if(si->no_safe_folders == 1)
		{
			return SIT_SINGLE_SAFE_FOLDER_SELECTED;
		}
		else if(si->no_devices == 1)
		{
			return SIT_DEVICE_SELECTED;
		}
		else if(si->no_mmc == 1)
		{
			return SIT_MMC_SELECTED;
		}
		return SIT_SINGLE_FOLDER_SELECTED;
	}
	if(si->no_folders == si->no_safe_folders)
	{
		return SIT_MULTIPLE_SAFE_FOLDERS_SELECTED;
	}
	return SIT_MULTIPLE_FOLDERS_SELECTED;
}
/**
 * selection_info_get_selection_type:
 * Gets the selection type for a given selection 
 * @param si SelectionInfo
 * @return SelectionInfoType
 **/
SelectionInfoType selection_info_get_selection_type(SelectionInfo *si)
{
	DEBUG_PRINT_START;
	if(si->needs_update) 
	{
		return SIT_ERROR_IN_GETTING_INFO;
	}
	else if(si->no_files == 0)
	{
		return get_selection_type_for_nofiles(si);
	}
	else if(si->no_files == 1)
	{
		if(si->no_folders == 0)
		{
			return SIT_SINGLE_FILE_SELECTED;
		}
		return SIT_MULTIPLE_ITEMS_SELECTED;
	}
	if(si->no_folders == 0)
	{
		return SIT_MULTIPLE_FILES_SELECTED;
	}
	return SIT_MULTIPLE_ITEMS_SELECTED;
	DEBUG_PRINT_END;
}

