/**
 * @file osso_fm_details.c
 *
 * This file contains the Folder Details dialog info formatting modules
 *
 * Copyright (C) 2005-2007 Nokia Corporation
 *
 */
#include <gtk/gtk.h>
#include <time.h>
#include <hildon/hildon-file-details-dialog.h>
#include <hildon/hildon-file-system-storage-dialog.h>
#include <hildon/hildon-caption.h>
#include <sys/vfs.h>

#include "osso_fm_generic.h"
#include "osso_fm_controller.h"
#include "osso_fm_details.h"
#include "osso_fm_callbacks.h"
#include "osso_fm_fmstate.h"
#include "errno.h"
#include "math.h"

/**
* Represent each row in device details dialog
*/
enum {
	TOTAL_USED_ROW = 0,
	TOTAL_FREE_ROW,
	SEPARATOR_ROW,
	EMAILS_ROW,
	IMAGES_ROW,
	VIDEOS_ROW,
	SOUNDS_ROW,
	NOTES_ROW,
	WEBPAGES_ROW,
	DOCUMENTS_ROW,
	CONTACTS_ROW,
	APPLICATIONS_ROW,
	OTHERS_ROW
};

void on_folder_details_response(GtkWidget * button, gpointer uri);

typedef struct {
	gboolean readonly;	/* Access permission */
	gboolean locationreadonly;
	gchar *label_size;	/* Label text for size */
	gchar *label_content;	/* Label text to display the content */
	gchar *label_name;	/* Label text to display directory name */
	gchar *label_location;	/* Location of the folder(device/gateway) */
	gchar *modified_date;	/* Date of modification */
	gchar *modified_time;	/* Time of modification */
} FormatFolderDetailsData;

/* Structure to have Directory property  */
typedef struct {
	gchar *folder_name;
	gchar *location_name;
	GdkPixbuf *location_icon;
	gchar *device_name;
	GdkPixbuf *device_icon;
	gchar *content_str;
	gchar *date;
	gchar *time;
	gchar *size_str;
	gboolean readonly;
	gboolean locationreadonly;
} FormatedFolderDetailsData;

typedef struct {
	GtkDialog *dialog;
        GtkWidget *scrolled_window;
	GtkWidget *name_label;
	GtkWidget *location_label;
	GtkImage  *location_icon_image;
	GtkWidget *device_label;
	GtkImage  *device_icon_image;
	GtkWidget *contents_label;
	GtkWidget *date_label;
	GtkWidget *time_label;
	GtkWidget *size_label;
	GtkWidget *readonly_checkbox;
	GtkFilePath *path;
	OSSOFMDirVisitHandle *dv_handle;
	gint rec_fetch_tid;
	FormatedFolderDetailsData *ffdd;
	FolderDetailsCounts *fd_counts;
	GTimeVal starttime;
	gboolean banner_shown;
}FolderDetailsDialog;


/**
 * get_parent_details:
 * A helper function to get the details of the parents
 * @param ofm ptr to structure OSSOFileManager
 * @param location_icon out param to store the location icon
 * @param device_icon out param to store the device icon
 * @param location_name out param to store the name of the parent folder
 * @param device_name out param to store the name of the device
 * @return gboolean TRUE if successful
 **/
static gboolean
get_parent_details(OSSOFileManager *ofm, 
		GdkPixbuf **location_icon, 
		GdkPixbuf **device_icon, 
		gchar **location_name, 
		gchar ** device_name)
{
	GtkTreeIter iter, parent_iter, temp_iter;
	GtkTreeModel *model = NULL;
	gint type;
	DEBUG_PRINT_START;

	hildon_file_selection_get_current_folder_iter((HildonFileSelection *) global_ofm->fs, 
			&iter);

	g_object_get(G_OBJECT(global_ofm->fs), "model", &model, NULL);
	if (model == NULL) {
		FM_ULOG_DEBUG1("Error getting model\n");
		DEBUG_PRINT_END;
		return FALSE;
	}
	if (hildon_file_selection_get_active_pane(global_ofm->fs) == 
			HILDON_FILE_SELECTION_PANE_CONTENT) {
		parent_iter = iter;
	} else {
		if(!gtk_tree_model_iter_parent(model, &parent_iter, &iter))
		{
			FM_ULOG_ERR("Error getting parent iter\n");
			return FALSE;
		}
		
	}

	gtk_tree_model_get(model, &parent_iter,
			   HILDON_FILE_SYSTEM_MODEL_COLUMN_DISPLAY_NAME, location_name,
			   HILDON_FILE_SYSTEM_MODEL_COLUMN_ICON, location_icon, -1);

	// Go upwards in model until we find a device node 
	while (TRUE) {
		gtk_tree_model_get(model, &parent_iter,
				   HILDON_FILE_SYSTEM_MODEL_COLUMN_TYPE, &type, -1);

		if (type >= HILDON_FILE_SYSTEM_MODEL_MMC)
			break;

		if (gtk_tree_model_iter_parent(model, &temp_iter, &parent_iter))
			parent_iter = temp_iter;
		else
			break;
	}

	gtk_tree_model_get(model, &parent_iter,
			   HILDON_FILE_SYSTEM_MODEL_COLUMN_DISPLAY_NAME, device_name,
			   HILDON_FILE_SYSTEM_MODEL_COLUMN_ICON, device_icon, -1);
	g_object_unref(model);
	DEBUG_PRINT_END;
	return TRUE;
}

/**
 * fd_get_general_data:
 * A helper function to get some general data for folder details dialog
 * @param ofm OSSOFileManager *
 * @param path GtkFilePath of the folder
 * @param ffdd FormatedFolderDetailsData to store the data.
 * @return void
 **/
static void fd_get_general_data(OSSOFileManager *ofm,const GtkFilePath *path,FormatedFolderDetailsData *ffdd)
{
	GnomeVFSFileInfo *info = NULL;
	struct tm *tmptr = NULL;
	GnomeVFSResult result = 0;
	gchar tmpdate[DATE_STRING_LENGTH] = "";
	gchar tmptime[TIME_STRING_LENGTH] = "";
	gchar tmpdatefmt[] = "%x",tmptimefmt[] = "%X";
	gchar *path_texturi = NULL;

	path_texturi = osso_fm_path_to_texturi(path);
	ffdd->folder_name = osso_fm_path_get_display_name (path);
	ffdd->readonly = is_path_readonly(path,FALSE);
	ffdd->locationreadonly = is_path_readonly(path,TRUE);
	info = gnome_vfs_file_info_new();
	if (info == NULL) {
		FM_ULOG_ERR("Error in gnome_vfs_file_info_new\n");
		DEBUG_PRINT_END;;
		return;
	}
	result = gnome_vfs_get_file_info(path_texturi, info, (GNOME_VFS_FILE_INFO_DEFAULT |
								GNOME_VFS_FILE_INFO_FOLLOW_LINKS));
	if (result == GNOME_VFS_OK && info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_MTIME) {
		tmptr = localtime(&info->mtime);
		if (tmptr == NULL) {
			FM_ULOG_ERR("Unable to convert time field to tm\n");
			ffdd->time = g_strdup("");
			ffdd->date = g_strdup("");
		} else {
			if ((strftime(tmptime, TIME_STRING_LENGTH, tmptimefmt,tmptr)) == 0) {
				FM_ULOG_ERR("strftime Error\n");
			}
			if ((strftime(tmpdate, DATE_STRING_LENGTH, tmpdatefmt,tmptr)) == 0) {
				FM_ULOG_ERR("strftime Error\n");
			}
			ffdd->time = g_strdup(tmptime);
			ffdd->date = g_strdup(tmpdate);
		}
	} if(path_texturi != NULL) {
		g_free(path_texturi);
	}
	if(!get_parent_details(ofm,&ffdd->location_icon,
				&ffdd->device_icon,
				&ffdd->location_name,
				&ffdd->device_name))
	{
		FM_ULOG_ERR("unable to get parent details\n");
		return;
	}
}

/**
 * fd_info_callback:
 * A callback function to get the count of items
 * @param info GnomeVFSFileInfo
 * @param recursive ptr to gboolean
 * @param user_data callback data
 * @return gboolean
 **/
static gboolean fd_info_callback(GnomeVFSFileInfo *info, gboolean *recursive, gpointer user_data)
{
	DEBUG_PRINT_START;
	if (info->type == GNOME_VFS_FILE_TYPE_REGULAR) {
		((FolderDetailsCounts *)user_data)->total_files++;
		((FolderDetailsCounts *)user_data)->total_size += info->size;
	} else if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
		((FolderDetailsCounts *)user_data)->total_folders++;
		((FolderDetailsCounts *)user_data)->total_size += info->size;
	}
	*recursive = TRUE;
	DEBUG_PRINT_END;
	return TRUE;
}

/**
 * fd_get_contents_string:
 * Function to get the translated string for folder contents
 * @param filecount no of files
 * @param foldercount no of folders
 * @return gchar * translated string
 **/
static gchar *fd_get_contents_string(gint filecount,gint foldercount)
{
	gchar *content = NULL;
	gchar *retval = NULL;
	gchar **tokens;

	DEBUG_PRINT_START;
	if ((filecount == 1) && (foldercount == 0)) {
		content = g_strdup_printf(_("sfil_fi_folder_details_contents_01"));
	} else if ((filecount == 2) && (foldercount == 0)) {
		content = g_strdup_printf(_("sfil_fi_folder_details_contents_02"));
	} else if ((filecount > 2) && (foldercount == 0)) {
		content = g_strdup_printf(_("sfil_fi_folder_details_contents_0%d"), filecount);
	} else if ((filecount == 0) && (foldercount == 0)) {
		content =  g_strdup_printf(_("sfil_fi_folder_details_contents_00"));
	} else if ((filecount == 0) && (foldercount == 1)) {
        content = g_strdup_printf(_("sfil_fi_folder_details_contents_10"));
	} else if ((filecount == 0) && (foldercount == 2)) {
        content = g_strdup_printf(_("sfil_fi_folder_details_contents_20"));
	} else if ((filecount == 0) && (foldercount >= 2)) {
        content = g_strdup_printf(_("sfil_fi_folder_details_contents_%d0"), foldercount);
	} else if ((filecount == 1) && (foldercount > 1)) {
        content = g_strdup_printf(_("sfil_fi_folder_details_contents_%d1"), foldercount);
	} else if ((filecount == 2) && (foldercount > 1)) {
        content = g_strdup_printf(_("sfil_fi_folder_details_contents_%d2"), foldercount);
	} else if ((filecount > 1) && (foldercount == 1)) {
        content = g_strdup_printf(_("sfil_fi_folder_details_contents_1%d"), filecount);
	} else if ((filecount > 1) && (foldercount == 2)) {
        content = g_strdup_printf(_("sfil_fi_folder_details_contents_2%d"), filecount);
	} else if ((filecount == 1) && (foldercount == 1)) {
        content = g_strdup_printf(_("sfil_fi_folder_details_contents_11"));
	} else {
        content = g_strdup_printf(_("sfil_fi_folder_details_contents_%d%d"),
				    foldercount, filecount);
	}
	
	tokens = g_strsplit_set (content, "\n", -1);
	retval = g_strjoinv (" ", tokens);
	g_free (content);
	g_strfreev (tokens);

	DEBUG_PRINT_END;

	return retval;
}


/**
 * get_size_string:
 * To get the Translated string for size of folder
 * @param size size in gint64
 * @return size in gchar * translated
 **/
static gchar *get_size_string(gint64 size)
{
	gchar *str = NULL;
	if(0 <= size && size < 1024) 
	{
		str = dgettext(HILDON_FM_L10N_PACKAGE,"ckdg_va_properties_size_kb");
		return g_strdup_printf(str,(gint) ceil(size/1024.0));
	}
	else if(1024 <= size && size < (100 * 1024))
	{
		str = dgettext(HILDON_FM_L10N_PACKAGE,"ckdg_va_properties_size_1kb_99kb");
		return g_strdup_printf(str, (gint)ceil(size/1024.0));
	}
	else if((100 * 1024) <= size && size < (1024 * 1024))
	{
		str = dgettext(HILDON_FM_L10N_PACKAGE,"ckdg_va_properties_size_100kb_1mb");
		return g_strdup_printf(str, (gint)ceil(size/1024.0));
	}
	else if((1024 * 1024) <= size && size < (10 * 1024 * 1024))
	{
		str = dgettext(HILDON_FM_L10N_PACKAGE,"ckdg_va_properties_size_1mb_10mb");
		return g_strdup_printf(str, (size/(1024.0 * 1024.0)));
	}
	else if((10 * 1024 * 1024) <= size && size < (1024 * 1024 *1024))
	{
		str = dgettext(HILDON_FM_L10N_PACKAGE,"ckdg_va_properties_size_10mb_1gb");
		return g_strdup_printf(str, (size/(1024.0 * 1024.0)));
	}
	else
	{
		str = dgettext(HILDON_FM_L10N_PACKAGE,"ckdg_va_properties_size_1gb_or_greater");
		return g_strdup_printf(str, (size/(1024.0 * 1024.0 * 1024.0)));
	}
}

/**
 * fd_recursive_fetch:
 * A timer function to fetch folder details 
 * @param data user data
 * @return gboolean 
 **/
static gboolean fd_recursive_fetch(gpointer data)
{
	GTimeVal curtime;
	gchar *buf = NULL;
	gboolean retval = FALSE;

	if(!((FolderDetailsDialog *)data)->banner_shown)
	{
		g_get_current_time(&curtime);
		if(timeval_diff(&curtime,&((FolderDetailsDialog *)data)->starttime) > 2000000)
		{
			((FolderDetailsDialog *)data)->banner_shown = TRUE;
		}
	}
	retval = osso_fm_dir_visit_fetch_next(((FolderDetailsDialog *)data)->dv_handle,10);
	gdk_threads_enter();
	buf = fd_get_contents_string(((FolderDetailsDialog *)data)->fd_counts->total_files,
			((FolderDetailsDialog *)data)->fd_counts->total_folders);
	gtk_label_set_text(GTK_LABEL(((FolderDetailsDialog *)data)->contents_label),buf);
	g_free(buf);
	buf = get_size_string(((FolderDetailsDialog *)data)->fd_counts->total_size);
	gtk_label_set_text(GTK_LABEL(((FolderDetailsDialog *)data)->size_label),buf);
	g_free(buf);
	if(!retval){
		((FolderDetailsDialog *)data)->banner_shown = FALSE; 
	}
	gdk_threads_leave();
	return retval;
}


/*	
	This function is a workaround to be able to show long labels
	on the dialog without using horizontal scrolling.  This
	function is called after all the widgets are realized. so that
	their actual size is known. Sizes are differnt in different
	locales.
*/

static void
setup_label_wrapping (GtkWidget *label, int width)
{
  gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
  gtk_label_set_line_wrap_mode (GTK_LABEL (label), PANGO_WRAP_CHAR);
  gtk_widget_set_size_request (label, width, -1);
}

static void
do_line_wrapping (GtkWidget *w, gpointer data)
{
  FolderDetailsDialog *fdd = (FolderDetailsDialog *)data;

  GtkWidget *label;
  GtkAllocation label_allocation;
  GtkAllocation window_allocation;
  
  gint ideal_width;

  label = fdd->name_label;
  label_allocation = label->allocation;
  window_allocation = fdd->scrolled_window->allocation;

  /* GtkWidgets are not aware of their parents size, so we have to explicitly
     tell to the labels where their visibility ends --> wrapping point. 
     Scrolled window is actually aware of it's size on screen so we adjust
     the label widths according to it 
  */
  ideal_width = (window_allocation.width) - (label_allocation.x);

  if (gtk_scrolled_window_get_vscrollbar 
      (GTK_SCROLLED_WINDOW(fdd->scrolled_window)))
    ideal_width -= 40; //Scroll bar width 

  setup_label_wrapping (fdd->name_label, ideal_width);
  setup_label_wrapping (fdd->location_label, ideal_width - 30);
  setup_label_wrapping (fdd->device_label, ideal_width - 30);
  setup_label_wrapping (fdd->contents_label, ideal_width);
  setup_label_wrapping (fdd->date_label, ideal_width);
  setup_label_wrapping (fdd->time_label, ideal_width);
  setup_label_wrapping (fdd->size_label, ideal_width);
}

/** 
 * create_fd_dialog:
 * A Helper function to create the folder details dialog
 * @param uripath
 * @return struct FolderDetailsDialog
 **/
static FolderDetailsDialog *create_fd_dialog(const GtkFilePath *uripath)
{
	FolderDetailsDialog *fdd;
	GtkSizeGroup *sizegrp;
	GtkWidget *sw;

	/* Captions and HBOXes for icon-label combos */
	GtkWidget *caption_name, *caption_location, *caption_device;
	GtkWidget *caption_date, *caption_time, *caption_size,
		  *caption_readonly;
	GtkWidget *caption_content;
	GtkWidget *hbox_location, *hbox_device;

	/* The main VBOX */
	GtkWidget *main_box;

	/* Dialog buttons */
	GtkWidget *button_ok, *button_cancel;

	DEBUG_PRINT_START;
	fdd = g_new0(FolderDetailsDialog,1);
	if(fdd == NULL)
	{
		FM_ULOG_ERR("create_fd_dialog:Error allocationg fdd\n");
		return NULL;
	}

	sw = gtk_scrolled_window_new (NULL, NULL);
	fdd->scrolled_window = sw;
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_NEVER,
					GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
					     GTK_SHADOW_NONE);

	sizegrp = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);
	main_box = gtk_vbox_new(FALSE, 0);

	fdd->location_label = gtk_label_new("");
	fdd->name_label = gtk_label_new("");
	fdd->device_label = gtk_label_new("");
	fdd->contents_label = gtk_label_new("");
	fdd->date_label = gtk_label_new("");
	fdd->time_label = gtk_label_new("");
	fdd->size_label = gtk_label_new("");
	fdd->readonly_checkbox = gtk_check_button_new();

	fdd->location_icon_image = GTK_IMAGE(gtk_image_new());
	fdd->device_icon_image = GTK_IMAGE(gtk_image_new());

	gtk_misc_set_alignment((GtkMisc *) fdd->name_label, 0, 0.5);
	gtk_misc_set_alignment((GtkMisc *) fdd->location_label, 0, 0.5);
	gtk_misc_set_alignment((GtkMisc *) fdd->device_label, 0, 0.5);
	gtk_misc_set_alignment((GtkMisc *) fdd->contents_label, 0, 0.5);
	gtk_misc_set_alignment((GtkMisc *) fdd->date_label, 0, 0.5);
	gtk_misc_set_alignment((GtkMisc *) fdd->time_label, 0, 0.5);
	gtk_misc_set_alignment((GtkMisc *) fdd->size_label, 0, 0.5);

	/* Packing icons for Device and Location */
	hbox_location = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
	hbox_device = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);

	gtk_box_pack_start (GTK_BOX (hbox_location), 
			GTK_WIDGET (fdd->location_icon_image), FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_location),
			GTK_WIDGET (fdd->location_label), TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (hbox_device),
			GTK_WIDGET (fdd->device_icon_image), FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_device),
			GTK_WIDGET (fdd->device_label), TRUE, TRUE, 0);	

	/* Create captions */
	caption_name = hildon_caption_new(sizegrp, 
					_("hfil_fi_folder_details_name"), 
					fdd->name_label,
					NULL, 
					HILDON_CAPTION_OPTIONAL);

	caption_location = hildon_caption_new(sizegrp,
					_("hfil_fi_folder_details_location"), 
					hbox_location,
					NULL, 
					HILDON_CAPTION_OPTIONAL);
	
	caption_device = hildon_caption_new(sizegrp,
					_("hfil_fi_folder_details_device"), 
					hbox_device,
					NULL, 
					HILDON_CAPTION_OPTIONAL);

	caption_content = hildon_caption_new(sizegrp,
					_("hfil_fi_folder_details_contents"),
					fdd->contents_label,
					NULL, 
					HILDON_CAPTION_OPTIONAL);

	caption_date = hildon_caption_new(sizegrp,
					_("hfil_fi_folder_details_date"),
					fdd->date_label,
					NULL, 
					HILDON_CAPTION_OPTIONAL);

	caption_time = hildon_caption_new(sizegrp,
					_("hfil_fi_folder_details_time"), 
					fdd->time_label,
					NULL, 
					HILDON_CAPTION_OPTIONAL);

	caption_size = hildon_caption_new(sizegrp,
					_("hfil_fi_folder_details_size"), 
					fdd->size_label,
					NULL, 
					HILDON_CAPTION_OPTIONAL);

	caption_readonly = hildon_caption_new(sizegrp,
					 _("sfil_fi_folder_details_readonly"),
					 fdd->readonly_checkbox, 
					 NULL, 
					 HILDON_CAPTION_OPTIONAL);

	/* Packing the captions to main box */
	gtk_box_pack_start(GTK_BOX(main_box), caption_name, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(main_box), caption_location, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(main_box), caption_device, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(main_box), caption_content, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(main_box), caption_date, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(main_box), caption_time, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(main_box), caption_size, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(main_box), caption_readonly, FALSE, FALSE, 0);

	/* Create the Dialog */
	fdd->dialog = GTK_DIALOG (gtk_dialog_new ());
	gtk_window_set_modal (GTK_WINDOW (fdd->dialog), TRUE);
	gtk_window_set_title (GTK_WINDOW (fdd->dialog),
			      _("hfil_ti_folder_details"));
	g_object_set (G_OBJECT (fdd->dialog), "has-separator", FALSE, NULL);
	
	button_ok = gtk_button_new_with_label (_("hfil_bd_folder_details_dialog_ok"));
	button_cancel = gtk_button_new_with_label (_("sfil_bd_folder_details_dialog_cancel"));
	gtk_widget_show (button_ok);
	gtk_widget_show (button_cancel);

	gtk_dialog_add_action_widget (GTK_DIALOG (fdd->dialog), button_ok, GTK_RESPONSE_OK);
	gtk_dialog_add_action_widget (GTK_DIALOG (fdd->dialog), button_cancel, GTK_RESPONSE_CANCEL);
	gtk_widget_grab_focus (button_ok);	

	gtk_window_set_resizable(GTK_WINDOW(fdd->dialog), FALSE);

	/* Add the main box to the dialog */
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (sw),
					       main_box);
	gtk_widget_show_all (sw);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(fdd->dialog)->vbox),
			  GTK_WIDGET(sw));

	/* Drop the initial reference to the GtkSizegroup */
	g_object_unref (sizegrp);

	g_signal_connect (G_OBJECT (fdd->dialog), "show",
			  G_CALLBACK (do_line_wrapping), fdd);
	
	DEBUG_PRINT_END;
	return fdd;
}
typedef struct 
{
	
}FolderDetailsData; 

/**
 * set_fd_data:
 * A Helper function to set the values in folder details dialog
 * @param fdd FolderDetailsDialog
 * @return void
 **/
static void set_fd_data(FolderDetailsDialog *fdd)
{
	
	gtk_label_set_text(GTK_LABEL(fdd->name_label),fdd->ffdd->folder_name);
	gtk_label_set_text(GTK_LABEL(fdd->location_label),fdd->ffdd->location_name);
	gtk_image_set_from_pixbuf(fdd->location_icon_image,fdd->ffdd->location_icon);
	gtk_label_set_text(GTK_LABEL(fdd->device_label),fdd->ffdd->device_name);
	gtk_image_set_from_pixbuf(fdd->device_icon_image,fdd->ffdd->device_icon);
	gtk_label_set_text(GTK_LABEL(fdd->date_label),fdd->ffdd->date);
	gtk_label_set_text(GTK_LABEL(fdd->time_label),fdd->ffdd->time);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(fdd->readonly_checkbox),fdd->ffdd->readonly);
	if(fdd->ffdd->locationreadonly)
	{
		gtk_widget_set_sensitive(fdd->readonly_checkbox, FALSE);
	}
}

/**
 * A function to handle to response of folder details dialog
 * @param dia, GtkDialog
 * @param res gint response code
 * @param data callback data
 * @return void
 **/
void fd_response(GtkDialog *dia,gint res,gpointer data)
{
	GnomeVFSFileInfo *info = NULL;
	gchar *uri;

	g_source_remove(((FolderDetailsDialog *)data)->rec_fetch_tid);
	osso_fm_dir_visit_close(((FolderDetailsDialog *)data)->dv_handle);
	if(((FolderDetailsDialog *)data)->banner_shown)
	{
		((FolderDetailsDialog *)data)->banner_shown = FALSE;
	}

	//TODO FREE ALL DATA
	if(res != GTK_RESPONSE_OK)
	{
		return;
	}
	info = gnome_vfs_file_info_new();
	if (info == NULL) {
		FM_ULOG_ERR("Error creating file info\n");
		return;
	}
	uri = osso_fm_path_to_texturi(((FolderDetailsDialog *)data)->path);
	if (uri == NULL) {
		FM_ULOG_ERR("osso_fm_path_to_texturi returned NULL\n");
		return;
	}
	
	gnome_vfs_get_file_info(uri, info, GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS);
	FM_ULOG_DEBUG1("Initial Permissions : %X\n", info->permissions);
	if (gtk_toggle_button_get_active(
			(GtkToggleButton *) ((FolderDetailsDialog *)data)->readonly_checkbox)) {
		FM_ULOG_DEBUG1("Checkbox enabled \n");
		info->permissions &=
		    (0xFFFF ^ (GNOME_VFS_PERM_USER_WRITE | GNOME_VFS_PERM_GROUP_WRITE));
	} else {
		//set writable permission
		FM_ULOG_DEBUG1("Checkbox disabled\n");
		info->permissions |= (GNOME_VFS_PERM_USER_WRITE | GNOME_VFS_PERM_GROUP_WRITE);
	}
	FM_ULOG_DEBUG1("Current Permissions : %X\n", info->permissions);
	gnome_vfs_set_file_info(uri, info, GNOME_VFS_SET_FILE_INFO_PERMISSIONS);

	gnome_vfs_file_info_unref(info);
	g_free(uri);
}

/**
 * create_FolderDetailsDialog:
 * Formatts the Folder details information and creates Folder details dialog.
 * @param path Represents selected file or folder path
 * @param window GtkWindow parent window
 * @return Pointer to folder details dialog
 **/
GtkWidget *create_FolderDetailsDialog(const GtkFilePath *path, GtkWindow * window)
{
	FolderDetailsDialog *fdd;
	GdkGeometry hints;

	DEBUG_PRINT_START;
	if (path == NULL) {
		FM_ULOG_ERR("Invalid parameter : path");
		return NULL;
	}

	fdd = create_fd_dialog(path);

	fdd->path = gtk_file_path_copy(path);
	fdd->fd_counts = g_new0(FolderDetailsCounts,1);
	fdd->dv_handle = osso_fm_dir_visit_init((GtkFilePath *)path, 
						fd_info_callback,
						fdd->fd_counts);
	fdd->ffdd = g_new0(FormatedFolderDetailsData,1);
	fd_get_general_data(global_ofm,path,fdd->ffdd);
	set_fd_data(fdd);
	
	g_signal_connect(fdd->dialog, "response", G_CALLBACK(fd_response), (gpointer) fdd);
	
	fdd->rec_fetch_tid = g_timeout_add(100,fd_recursive_fetch,fdd);
	g_get_current_time(&fdd->starttime);
	
	hints.min_width = 430;
	hints.max_width = 430;
 	hints.min_height = 305;	 
 	hints.max_height = 305;	// -1 

	gtk_window_set_geometry_hints(GTK_WINDOW(fdd->dialog), GTK_WIDGET(fdd->dialog),
				      &hints, GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE);

	DEBUG_PRINT_END;
	return GTK_WIDGET(fdd->dialog);
}

/**
 * free_formatdata_items:
 * It frees the items of structure for folder detail info
 * 
 * @param formatdata  Pointer to FolderDetailsCounts structure, 
 * contains Folder details dialog information
 * @param fdata Structure having dialog info
 * 
 * @return void
 **/
void free_formatdata_items(FormatFolderDetailsData * formatdata, FolderDetailsCounts * fdata)
{
	DEBUG_PRINT_START;

	if (formatdata == NULL) {
		return;
	}
	g_free(formatdata->label_name);
	g_free(formatdata->label_location);
	g_free(formatdata->label_size);
	g_free(fdata);
	g_free(formatdata->label_content);
	DEBUG_PRINT_END;
	return;
}


/**
 * set_time_detail:
 * Updates the folder detail structure
 * 
 * @param tmptr is pointer to time struct
 * @param formatdata is structure that contains Folder details dialog information
 * 
 * @return gboolean TRUE if the time details is succesful
 */
gboolean set_time_detail(struct tm * tmptr, FormatFolderDetailsData * formatdata)
{
	gchar *txt = NULL;
	gchar *text = NULL;
	gchar *dateformat = NULL;

	DEBUG_PRINT_START;

	if ((tmptr == NULL) || (formatdata == NULL)) {
		FM_ULOG_ERR("Invalid parameter : path (null)");
		if (formatdata != NULL) {
			formatdata->modified_time = g_strdup("");
			formatdata->modified_date = g_strdup("");
		}
		return FALSE;
	}
	text = g_new0(gchar, TIME_STRING_LENGTH);
	//Check the Memory requirement for Time
	/* Time formatting */
	if (text != NULL) {

		if ((strftime(text, TIME_STRING_LENGTH, "%X", tmptr)) == 0) {
			g_free(text);
			FM_ULOG_ERR("strftime Error\n");
			return FALSE;
		}
		formatdata->modified_time = g_strdup_printf("%s", text);
	} else {
		FM_ULOG_ERR("Error allocating memory");
		return FALSE;
	}
	g_free(text);
	txt = g_new0(gchar, DATE_STRING_LENGTH);
	/* Date formatting */
	if (txt != NULL) {
		dateformat = g_strdup("%x");
		if (dateformat == NULL) {
			FM_ULOG_ERR("Error alocating Memory \n");
			return FALSE;
		}
		if ((strftime(txt, DATE_STRING_LENGTH, dateformat, tmptr)) == 0) {
			g_free(txt);
			FM_ULOG_ERR("strftime Error");
			return FALSE;
		}
		g_free(dateformat);
		formatdata->modified_date = g_strdup_printf("%s", txt);
	} else {
		FM_ULOG_ERR("Error allocating memory");
		return FALSE;
	}
	g_free(txt);

	DEBUG_PRINT_END;
	return TRUE;
}



/**
 * update_progress_banner:
 * Show/hide the 'updating' banner when details dialog  takes a long time to display
 * @param status
 * @return void
 **/
static void update_progress_banner(const gint status)
{
	static gboolean banner = FALSE;
	static GTimeVal starttime;
	GTimeVal curtime;
	if(status == 0)
	{
		g_get_current_time(&starttime);
		banner = FALSE;
	}
	else if (status == 1)
	{
		if(!banner)
		{
			g_get_current_time(&curtime);
			if(timeval_diff(&curtime,&starttime) > 2000000)
			{
				banner = TRUE;
			}
		}
	}
	else 
	{
		banner = FALSE;
	}
	
}



/**
 * directory_visit_callback_for_mem:
 * This function is called to recursively visit the directory 
 * to get various memory info of device.
 * 
 * @param relpath Variable represents selected folder path
 * @param info Folder info
 * @param recursing_will_loop TRUE to count recursively
 * @param data Memory details struct info to be passed back
 * 
 * @return TRUE or FALSE .
 **/
gboolean
directory_visit_callback_for_mem(const gchar * rel_path,
				 GnomeVFSFileInfo * info,
				 gboolean recursing_will_loop, gpointer data, gboolean * recurse)
{
	DEBUG_PRINT_START;
	if (rel_path == NULL || data == NULL || info == NULL) {
		FM_ULOG_ERR("Invalid input parameter");
		return FALSE;
	}

	fileMemData *dat = (fileMemData *) data;
	update_progress_banner(1);

	if (info->name[0] != '.' || (info->name[1] != '.' && info->name[1] != 0)
	    || info->name[2] != 0) {
		if (recursing_will_loop) {
			FM_ULOG_ERR("Loop detected");
			return FALSE;
		}
		*recurse = TRUE;
	} else {
		*recurse = FALSE;
	}
	if (info->type == GNOME_VFS_FILE_TYPE_REGULAR) {
		dat->fileno++;
		FM_ULOG_DEBUG1("************ opening dev detail %s\n", rel_path);
		FM_ULOG_DEBUG1("apptype is %s\n", info->mime_type);
		if (info->mime_type == NULL) {
			return FALSE;
		}
		dat->files_usedsize += info->size;

		if (g_strncasecmp(info->mime_type, "image", 5) == 0) {
			dat->imagesize += info->size;
		} else if (g_strncasecmp(info->mime_type, "audio", 5) == 0) {
			dat->audiosize += info->size;
		} else if (g_strncasecmp(info->mime_type, "video", 5) == 0) {
			dat->videosize += info->size;
		} else if (g_strcasecmp(info->mime_type, "sketch/png") == 0) {
			dat->imagesize += info->size;
		} else if (g_strncasecmp(info->mime_type, "text", 4) == 0) {
			if ((g_strcasecmp(info->mime_type, "text/xml") == 0)
			    || (g_strcasecmp(info->mime_type, "text/html") == 0)) {
				dat->htmlsize += info->size;
			} else if ((g_strcasecmp(info->mime_type, "text/plain") == 0)
				   || (g_strcasecmp(info->mime_type, "text/x-notes") == 0)
				   || (g_strncasecmp(info->mime_type, "text/note", 9) == 0)
				   || (g_strcasecmp(info->mime_type, "text/richtext") == 0)) {
				dat->docsize += info->size;
			} else {
				dat->othersize += info->size;
			}
		} else if (g_strncasecmp(info->mime_type, "application", 11) == 0) {
			if ((g_strcasecmp(info->mime_type, "application/pdf") == 0)
			    || (g_strcasecmp(info->mime_type, "application/rss+xml") == 0)) {
				dat->docsize += info->size;
			} else if (g_strcasecmp(info->mime_type, "application/x-sketch-png") == 0) {
				dat->imagesize += info->size;
			} else {
				dat->othersize += info->size;
			}
		} else {
			dat->othersize += info->size;
		}
	} else if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
		dat->folderno++;
	}
	DEBUG_PRINT_END;
	return TRUE;
}

/**
 * get_drive_detail:
 * This function is called to get the details of various memory info of device.
 * @param filememdata: memory details struct info to be passed back
 * @param path variable represents folder path from which
 * device detail is enquired
 * @return gint.
 * 
 **/
GnomeVFSResult get_drive_detail(fileMemData * filememdata, const gchar * path)
{
	GnomeVFSResult result = GNOME_VFS_OK;
	GnomeVFSURI *uri;
	struct statfs info;	/* Structure to keep the file statistics */
	int status = -1;	/* returned by statfs function */
	gint64 reserved_space=0;
	gboolean internal=FALSE;
	gboolean retval=FALSE;
	

	DEBUG_PRINT_START;

	if (filememdata == NULL || path == NULL) {
		FM_ULOG_ERR("Invalid input parameter");
		return result;
	}
	uri = gnome_vfs_uri_new(path);
	g_print("obtained path is %s \n%s\n", path, gnome_vfs_uri_get_path(uri));
	retval=is_uri_in_mmc(uri,&internal);
	if (retval) {
		reserved_space = 0;
	} else {
		reserved_space = RESERVED_SPACE;
	}
	status = statfs(gnome_vfs_uri_get_path(uri), &info);
	gnome_vfs_uri_unref(uri);
	if (status == 0) {
		filememdata->totalsize = ((gint64) info.f_bsize) * ((gint64) info.f_blocks);
		filememdata->freesize = ((gint64) info.f_bsize) * ((gint64) info.f_bavail);
		if (filememdata->freesize > 0) {
			if (filememdata->freesize < reserved_space)
				filememdata->freesize = 0;
			else
				filememdata->freesize = filememdata->freesize - reserved_space;
		}
		filememdata->usedsize = filememdata->totalsize - filememdata->freesize;
	} else {
		filememdata->totalsize = 0;
		filememdata->freesize = 0;
		filememdata->usedsize = 0;
		filememdata->files_usedsize = 0;
		g_print("Getting capacity for %s failed.", path);
		g_print("Error is %s\n", strerror(errno));
	}
	if (filememdata->usedsize != 0) {
		result = gnome_vfs_directory_visit(path,
						   (GNOME_VFS_FILE_INFO_FOLLOW_LINKS |
						    GNOME_VFS_FILE_INFO_GET_MIME_TYPE),
						   GNOME_VFS_DIRECTORY_VISIT_LOOPCHECK,
						   directory_visit_callback_for_mem, filememdata);
	}

	DEBUG_PRINT_END;
	return result;
}

/**
 * directory_visit_callback_for_email:
 * This function is called to recursively visit the directory to get
 * email memory info in device.
 * @param relpath: variable represents selected folder path
 * @param info:  folder info
 * @param recursing_will_loop: to count recursively
 * @param data:  memory details struct info to be passed back
 * @return TRUE or FALSE .
 **/
gboolean
directory_visit_callback_for_email(const gchar * rel_path,
				   GnomeVFSFileInfo * info,
				   gboolean recursing_will_loop, gpointer data, gboolean * recurse)
{
	DEBUG_PRINT_START;
	if ((rel_path == NULL) || (data == NULL) || (info == NULL)) {
		return FALSE;
	}

	fileMemData *dat = (fileMemData *) data;

	if (info->name[0] != '.' || (info->name[1] != '.' && info->name[1] != 0)
	    || info->name[2] != 0) {
		if (recursing_will_loop) {
			FM_ULOG_ERR("Loop detected");
			return FALSE;
		}
		*recurse = TRUE;
	} else {
		*recurse = FALSE;
	}
	dat->emailsize += info->size;
	dat->files_usedsize += info->size;

	DEBUG_PRINT_END;
	return TRUE;
}

GtkWidget *
create_DeviceDetailsDialog (const GtkFilePath *path, GtkWidget *window)
{
  char *uri;
  GtkWidget *dialog;

  uri = osso_fm_path_to_texturi (path);
  dialog = hildon_file_system_storage_dialog_new (GTK_WINDOW(window), uri);
  g_free (uri);

  return dialog;
}

/**
 * osso_fm_details_activate:
 * Creates and shows details dialog for the selection and also used for validation
 * @param ofm: pointer to OSSOFileManager
 * @param menucheck: gboolean to state if the call has been made for validation
 *
 * @return gboolean returns TRUE if successful/can be successful
 **/
gboolean osso_fm_details_activate(OSSOFileManager * ofm, gboolean menucheck)
{

	GtkWidget *DetailsDialog = NULL;
	GnomeVFSURI *details_uri = NULL;
	HildonFileSystemModel *model = NULL;
	gboolean retval = FALSE;
	GtkTreeIter iter;

	DEBUG_PRINT_START;
	selection_info_update(ofm,FALSE);
	switch(selection_info_get_selection_type(ofm->si))
	{
		case SIT_SINGLE_FILE_SELECTED:
			retval = TRUE;
			if (menucheck) {
				break;
			}
			g_object_get(G_OBJECT(ofm->fs), "model", &model, NULL);
			if (model == NULL) {
				FM_ULOG_ERR("Error abtaining HildonFileSystemModel widget");
				return FALSE;
			}
			details_uri = osso_fm_path_to_uri(ofm->si->current_selection->data);
			if (!hildon_file_selection_get_current_content_iter(
				(HildonFileSelection *) ofm->fs, &iter)) {
				FM_ULOG_ERR("hildon_file_selection_get_"
					"current_content_iter failed\n");
				return FALSE;
			}
			appstate.mainwindow.opendialog = OPEN_DIALOG_DETAILS;
			DetailsDialog = (GtkWidget *) hildon_file_details_dialog_new_with_model(
								(GtkWindow *) ofm->window, 
								model);
			hildon_file_details_dialog_set_file_iter((HildonFileDetailsDialog *)
								 DetailsDialog, &iter);
			gtk_dialog_run(GTK_DIALOG(DetailsDialog));
			gtk_widget_destroy(DetailsDialog);
			gnome_vfs_uri_unref(details_uri);
			appstate.mainwindow.opendialog = OPEN_DIALOG_NONE;
			break;
		case SIT_DEVICE_SELECTED:
			if(ofm->si->selection_in_gateway ||
					ofm->si->selection_in_media_server)
			{
				retval = FALSE;
				if (!menucheck) {
					fm_infoprint(GTK_WINDOW(ofm->window), _("sfil_ib_unable_details_bluetooth"));
				}
				break;
			}
			retval = TRUE;
			if (menucheck) {
				break;
			}
			appstate.mainwindow.opendialog = OPEN_DIALOG_DETAILS;
			DetailsDialog = create_DeviceDetailsDialog(ofm->si->current_selection->data,
								   (GtkWidget *) ofm->window);
			break;
		case SIT_MMC_SELECTED:
			retval = TRUE;
			if (menucheck) {
				break;
			}
			appstate.mainwindow.opendialog = OPEN_DIALOG_DETAILS;
			DetailsDialog = create_DeviceDetailsDialog(ofm->si->current_selection->data,
								(GtkWidget *) ofm->window);
			break;
		case SIT_SINGLE_FOLDER_SELECTED:
		case SIT_SINGLE_SAFE_FOLDER_SELECTED:
			retval = TRUE;
			if (menucheck) {
				break;
			}
			appstate.mainwindow.opendialog = OPEN_DIALOG_DETAILS;
			DetailsDialog = create_FolderDetailsDialog(ofm->si->current_selection->data,
								(GtkWindow *) ofm->window);
			break;
		case SIT_MULTIPLE_SAFE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_FILES_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
			retval = FALSE;
			if (!menucheck) {
				fm_infoprint(GTK_WINDOW(ofm->window),
						dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
							"ckct_ib_select_single_item"));
			}
			break;
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			retval = FALSE;
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			retval = FALSE;
			break;
	}
	if (GTK_IS_WIDGET(DetailsDialog) && GTK_IS_DIALOG(DetailsDialog)) {
		register_selection_change_sensitive_dialog(global_ofm,DetailsDialog);
		gtk_dialog_run(GTK_DIALOG(DetailsDialog));
		unregister_selection_change_sensitive_dialog(global_ofm,DetailsDialog);
		gtk_widget_destroy(GTK_WIDGET(DetailsDialog));
		g_signal_emit_by_name(global_ofm->fs,"selection-changed");
	}
	appstate.mainwindow.opendialog = OPEN_DIALOG_NONE;
	
	DEBUG_PRINT_END;
	return retval;
}

