/**
 * @file  osso_fm_controller.h
 *
 * This is the interface module between afw and the Filemanager.
 *
 * Copyright (C) 2005 Nokia Corporation
 */

#ifndef _OSSO_FM_CONTROLLER_H_
#define _OSSO_FM_CONTROLLER_H_

#include <libosso.h>
#include <gtk/gtk.h>
#include "osso_fm_generic.h"

#define FM_ULOG_OPEN(x) ULOG_OPEN(x)
#define FM_LOG_CLOSE LOG_CLOSE

#define FM_ENABLE_LOG_ERR 
#define FM_ENABLE_LOG_WARN 
#define FM_ENABLE_LOG_INFO
#define FM_ENABLE_LOG_DEBUG1


#ifdef FM_ENABLE_LOG_ERR
#define FM_ULOG_ERR(...) osso_log (LOG_ERR,__VA_ARGS__);g_print(__VA_ARGS__)
#else
#define FM_ULOG_ERR(...) ;
#endif

#ifdef FM_ENABLE_LOG_WARN
#define FM_ULOG_WARN(...) osso_log (LOG_WARNING, __VA_ARGS__);g_print(__VA_ARGS__)
#else
#define FM_ULOG_WARN(...) ;
#endif

#ifdef FM_ENABLE_LOG_INFO
#define FM_ULOG_INFO(...) osso_log (LOG_INFO, __VA_ARGS__);g_print(__VA_ARGS__)
#else
#define FM_ULOG_INFO(...) ;
#endif

#ifdef FM_ENABLE_LOG_DEBUG1
#define FM_ULOG_DEBUG1(...) osso_log (LOG_INFO, __VA_ARGS__);g_print(__VA_ARGS__)
#else
#define FM_ULOG_DEBUG1(...) ;
#endif

#define MMC_FORMAT_CANCEL_METHOD "mmc_formatting_cancelled"
#define OPEN_FOLDER_METHOD "open_folder"

osso_context_t *fm_app_ctx;

extern GtkFilePath *path_to_select_on_open;

/**
 * This function initializes the library and connects to the message bus,
 * and integrates with the GLib main loop, and initializes the library for
 * use. #osso_initialize should only be called once for each application.
 * @param application The name that this application is known as. This is
 * not the name that a user sees, but rather the name that other
 * applications use as an address to communicate with this program.
 * @param version The version string of the application. It mus be simply
 * comparable with strcmp().
 * @param activation If this is TRUE, then the library assumes that the
 * application binary is executed by the D-BUS daemon.
 * @param context The GLib main-loop context to connect to, use NULL for
 * the default context.
 * @return A context to use in later calls to this library. NULL is
 * returned if there is an error.
 */
osso_context_t *osso_fm_initialize(const gchar * application,
				   const gchar * version,
				   gboolean activation, GMainContext * context);


/**
 * This function will close the message bus connection and free all memory
 * allocated by the library. This function renders the library in an
 * unusable state, #osso_initialize has to be called again to reinitialize
 * the library.
 * @param osso The library reference as returned by #osso_initialize.
 */
void osso_fm_deinitialize(osso_context_t * osso);


/**
 * This function will et all the required callbacks to the afw.
 */
void osso_fm_set_afw_callbacks();


osso_return_t
osso_fm_application_top(osso_context_t * osso, const gchar * application, const gchar * arguments);

/**
 * This is the type for the top callback function.
 * @param arguments This string lists some extra parameters for the
 * application. It can be used to give the names of the files to open, etc.
 * @param data The data that was set with the #osso_application_set_top_cb
 * function.
 */
void osso_fm_application_top_cb_f(const gchar * arguments, gpointer data);


/**
 * This function registers a top-event handler.
 * @param osso The library context as returned by #osso_initialize.
 * @param cb The callback function.
 * @param data An arbitrary pointer to some application specific data.
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
 * #OSSO_INVALID if some parameter is invalid.
 */
osso_return_t
osso_fm_application_set_top_cb(osso_context_t * osso,
			       osso_application_top_cb_f * cb, gpointer data);

/**
 * This function emits a signal to exit the application
 * @param die_now TRUE to exit application immediately
 * @param data callback specific data
 */
void osso_fm_application_exit_cb(gboolean die_now, gpointer data);



/** This function adds an application window to the tasknavs application list
 *      if it does not previously exist.
 *      If the id and application name combination
 *      already exists in the navigator,
 *      the window is moved to the top of the window
 *      list of the application in question.
 *
 *      An application should call this function when ever it tops a
 *      window. This way the task navigator can keep track of the
 *      accessing order of the windows.
 *
 *      TODO: If window id already exists the title and arguments
 *                    should be updated.
 *
 *
 *
 *      @param osso The osso library context returned by #osso_initialize
 *
 *      @param id   Unique id for the window, Can be
 *                              generated by the application but must
 *                              be unique inside the application.
 *
 *      @param title Title of the window to be displayed on the navigator.
 *
 *      @param args  TaskNavigator will use this string as an argument for
 *                             the 'top_application' message when
 *                             user taps this window in the popup menu.
 *                             Contents specified by the application.
 *
 *      @return #OSSO_OK on success, #OSSO_ERROR in case of an error,
 *                     #OSSO_INVALID if a paramter is invalid.
 */
osso_return_t
osso_fm_tasknav_window_update(osso_context_t * osso,
			      guint id, const gchar * title, const gchar * args);


/** Method to tell the task navigator that program opens a modal dialog
 *
 *    @param osso Osso context returned by #osso_initialize
 *
 *    @param id Id of the window as given to the #osso_tasknav_window_update
 *
 *    @param dialog_name Name of the dialog to be displayed in an infoprint.
 */
osso_return_t
osso_fm_tasknav_window_open_dialog(osso_context_t * osso, guint id, const gchar * dialog_name);


/** Method to tell the task navigator that program closes a modal dialog
 *
 *    @param osso Osso context returned by #osso_initialize
 *
 *    @param id Id of the window as given to the #osso_tasknav_window_update
 */
osso_return_t osso_fm_tasknav_window_close_dialog(osso_context_t * osso, guint id);


/** This function removes an application window from the task navigator.
 *     Application should call this when a window is closed.
 *
 *     @param osso The osso library context returned by #osso_initialize
 *     @param id  The id of the window given to #osso_tasknav_window_update
 *     @return #OSSO_OK on success, #OSSO_ERROR in case of an error,
 *                     #OSSO_INVALID if a paramter is invalid.
 */
osso_return_t osso_fm_tasknav_window_remove(osso_context_t * osso, guint id);


/**
 * This is a wrapper to osso_email_files
 * @param osso The library context as returned by #osso_initialize.
 * @param list A GSList of strings -- URI's in GnomeVFS format.
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
 * #OSSO_INVALID if some parameter is invalid.
 */
osso_return_t osso_fm_email_files(osso_context_t * osso, GSList * list);


/** Returns dbus session bus connection used by osso
 *  @param osso osso context as returned by #osso_initialize
 *
 *  @return pointer to DBusConnection on a gpointer
 */
gpointer osso_fm_get_dbus_connection(osso_context_t * context);


void osso_fm_hw_cb_f(osso_hw_state_t * state, gpointer data);

osso_return_t
osso_fm_rpc_callback_handler(const gchar * interface,
			     const gchar * method,
			     GArray * arguments, gpointer data, osso_rpc_t * retval);

/*static osso_return_t
osso_fm_mmc_formatting_cancel ( osso_context_t *osso );
*/
osso_return_t osso_fm_open_folder(osso_context_t * osso, const gchar * folder);

//BEG: File open related functions

/**
   Method that is invoked when a file open operation selected from menu item.
   
   @param window Main window
   @return void
*/
gboolean osso_fm_file_open(OSSOFileManager *ofm, gboolean menucheck);

//END: File open related functions


//BEG: Send as email related functions 
/**
* Method to send a file as email.
* 
* @param window Pointer to the file manager window
* @return void
*/
gboolean osso_send_as_email(OSSOFileManager* ofm, gboolean menucheck);

//END: Send as email related functions 

gboolean osso_fm_send_via_bluetooth(OSSOFileManager *ofm, gboolean menucheck);

//BEG: Global search related functions

void osso_fm_invoke_global_search();

//END: Global search related functions

GSList *get_bt_addresses();

#endif
