/**
 * @file  osso_fm_controller.c
 *
 * This file contains implementation of interface module between afw and the Filemanager.
 *
 * Copyright (C) 2005 Nokia Corporation
 */

#include "osso_fm_controller.h"
#include "osso_fm_generic.h"
#include "osso_fm_fmstate.h"
//#include "osso_fm_validation.h"
#include "osso_fm_utils.h"
#include <ossoemailinterface.h>
#include <libgnomevfs/gnome-vfs-mime-info.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <hildon-mime.h>

//LOCAL FUNCTIONS PROTOTYPES

/**
 * osso_fm_initialize:
 * This function initializes the library and connects to the message bus,
 * and integrates with the GLib main loop, and initializes the library for
 * use. #osso_initialize should only be called once for each application.
 * @param application The name that this application is known as. This is
 * not the name that a user sees, but rather the name that other
 * applications use as an address to communicate with this program.
 * @param version The version string of the application. It mus be simply
 * comparable with strcmp().
 * @param activation If this is TRUE, then the library assumes that the
 * application binary is executed by the D-BUS daemon.
 * @param context The GLib main-loop context to connect to, use NULL for
 * the default context.
 * @return A context to use in later calls to this library. NULL is
 * returned if there is an error.
 */
osso_context_t *osso_fm_initialize(const gchar * application,
				   const gchar * version,
				   gboolean activation, GMainContext * context)
{
	if (!gnome_vfs_init()) {
         	FM_ULOG_ERR("gnome_vfs_init() failed");
        }
	DEBUG_PRINT_START return osso_initialize(application, version, activation, context);
}


/**
 * osso_fm_deinitialize:
 * This function will close the message bus connection and free all memory
 * allocated by the library. This function renders the library in an  
 * unusable state, #osso_initialize has to be called again to reinitialize
 * the library.
 * @param osso The library reference as returned by #osso_initialize.
 */
void osso_fm_deinitialize(osso_context_t * osso)
{
	DEBUG_PRINT_START;
	if(osso == NULL)
	{
		FM_ULOG_ERR("osso_fm_deinitialize: ctx == NULL");
		return;
	}
       	osso_deinitialize(osso);
	DEBUG_PRINT_END;
}


/**
 * osso_fm_set_afw_callbacks:
 * @param data user data
 * This function will set all the required callbacks to the afw.
 */
void osso_fm_set_afw_callbacks(gpointer data)
{
  DEBUG_PRINT_START
  // set application top callback 
  osso_fm_application_set_top_cb(fm_app_ctx,
				 osso_fm_application_top_cb_f, data);
  DEBUG_PRINT_END;
}


/**
 * osso_fm_application_top:
 * This is a callback function which will be called when application is topped
 * @param osso osso_context_t
 * @param application gchar * name of the application
 * @param arguments gchar * command line arguments
 * @return osso_return_t
 **/
osso_return_t
osso_fm_application_top(osso_context_t * osso, const gchar * application, const gchar * arguments)
{
	DEBUG_PRINT_START return osso_application_top(osso, application, arguments);
}


/**
 * osso_fm_application_top_cb_f:
 * This is the type for the top callback function.
 * @param arguments This string lists some extra parameters for the
 * application. It can be used to give the names of the files to open, etc.
 * @param data The data that was set with the #osso_application_set_top_cb
 * function.
 * @return void
 */
void osso_fm_application_top_cb_f(const gchar * arguments, gpointer prog)
{
  DEBUG_PRINT_START
#if 0
    gtk_window_present(GTK_WINDOW(window));
#endif
	hildon_program_set_can_hibernate (HILDON_PROGRAM (prog), FALSE);
	//load_fm_state();
	//set_appliaction_state (app);
DEBUG_PRINT_END}


/**
 * osso_fm_application_set_top_cb:
 * This function registers a top-event handler.
 * @param osso The library context as returned by #osso_initialize.
 * @param cb The callback function.
 * @param data An arbitrary pointer to some application specific data.
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
 * #OSSO_INVALID if some parameter is invalid.
 */
osso_return_t
osso_fm_application_set_top_cb(osso_context_t * osso, osso_application_top_cb_f * cb, gpointer data)
{
	DEBUG_PRINT_START return osso_application_set_top_cb(osso, cb, data);
}


/**
 * osso_fm_email_files:
 * This is a wrapper to osso_email_files
 * @param osso The library context as returned by #osso_initialize.
 * @param list A GSList of strings -- URI's in GnomeVFS format.
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
 * #OSSO_INVALID if some parameter is invalid.
 */
osso_return_t osso_fm_email_files(osso_context_t * osso, GSList * list)
{
	//DEBUG_PRINT_START
	return osso_email_files_email(osso, list);
	//return OSSO_OK;
}


/** 
 * osso_fm_get_dbus_connection:
 * Returns dbus session bus connection used by osso
 * @param osso osso context as returned by #osso_initialize
 * @return pointer to DBusConnection on a gpointer
 */
gpointer osso_fm_get_dbus_connection(osso_context_t * context)
{
	//* FIXME Have to returen gpointer now since due to errors
	//*      in dbus.h it can not be included to every software.
	return osso_get_dbus_connection(context);
}

GtkFilePath *path_to_select_on_open = NULL;

/**
 * osso_fm_rpc_callback_handler
 * This function is a callback function for rpc handler.
 * This function gets called whenever Dbus message is received from other
 * applications
 * @param  interface  contains interface name
 * @param  method  contains corresponding method to execute
 * @param  arguments contains arguments in GAarry struct
 * @param  data    generic data
 * @param  retval  contains the return value after executing this method
 * @return OSSO_OK if the application can be launched, OSSO_ERROR if not.
 **/

osso_return_t
osso_fm_rpc_callback_handler(const gchar * interface,
			     const gchar * method,
			     GArray * arguments, gpointer data, osso_rpc_t * retval)
{
	osso_return_t rv = OSSO_OK;
	osso_log(LOG_DEBUG, " osso_fm_rpc_callback_handler starts\n");
	g_print(" osso_fm_rpc_callback_handler starts\n");
	FM_ULOG_INFO(" osso_fm_rpc_callback_handler starts\n");

	if (method == NULL) {
		if (arguments == NULL) {
			g_print("method and arguments is NULL\n");
		}
	} else if (!strcmp(method, OPEN_FOLDER_METHOD)) {
		path_to_select_on_open = strdup(g_array_index(arguments,
                                                osso_rpc_t, 0).value.s);
	} else {
		g_print("osso_fm_rpc_callback_handler: \
		    received unknown dbus message - \
		    interface: %s, method:%s, \n", interface, method);
	}
	return rv;
}



/**
 * osso_fm_open_folder:
 * This is a function which will be called via dbus to select a folder.
 * @param osso osso_context_t
 * @param folder gchar * path to select
 * @return osso_return_t
 **/
osso_return_t osso_fm_open_folder(osso_context_t * osso, const gchar * folder)
{
	osso_return_t rv = OSSO_OK;

	if (global_ofm->fs) {
		gtk_window_present((GtkWindow *) global_ofm->window);
		if (appstate.mainwindow.opendialog == OPEN_DIALOG_NONE) {
			osso_fm_select_texturi_with_time(g_strdup(folder), FALSE);
		}
	} else {
		path_to_select_on_open = osso_fm_texturi_to_path(folder);
	}

	g_print("osso_fm_open_folder: %s\n", folder);
	return rv;
}


//BEG: File open related functions

#define NO_ASSOCIATED_APPS_ERR  -6	// -6 is a random number < 1

/**
 * invoke_application
 * Method to invoke an application corresponding to the user selection.
 * @param window Main file manager window
 * @param selection handle to the list of selected items.
 * @return void
 */
static void invoke_application(GtkWidget * window, GSList * selection)
{
	char *mime_type = NULL;
	GSList *OpenList = NULL;
	GSList *cursel = NULL;
	gchar *path_texturi = NULL;
	GnomeVFSFileInfo *info;
	GnomeVFSMimeApplication *first_application = NULL;
	GnomeVFSMimeApplication *temp_application = NULL;
	gint openresult = 10;
	DEBUG_PRINT_START;
	if ((window == NULL) || (selection == NULL)) {
		FM_ULOG_ERR("Error validating invalid parameters");
		return;
	}
	info = gnome_vfs_file_info_new();
	gnome_vfs_file_info_clear(info);
	if (selection != NULL) {
		for (cursel = selection;cursel != NULL; cursel = g_slist_next(cursel)) {
			path_texturi = osso_fm_path_to_texturi(cursel->data);
			if (osso_fm_is_texturi_directory(path_texturi)) {
				g_free(path_texturi);
				continue;
			}
			mime_type = gnome_vfs_get_mime_type(path_texturi);
			if(mime_type == NULL)
			{
				FM_ULOG_ERR("mime_type is NULL for %s\n",path_texturi);
				fm_infoprint (GTK_WINDOW (window),
					dgettext(HILDON_FM_L10N_PACKAGE,
						"sfil_ib_opening_not_allowed"));	
				g_free(path_texturi);
				return;
			}
			temp_application = gnome_vfs_mime_get_default_application(mime_type);
			if ((temp_application != NULL) && (first_application == NULL)) {
				first_application =
				    gnome_vfs_mime_application_copy(temp_application);
			}
			if ((first_application != NULL) && (temp_application != NULL) &&
			    (g_utf8_collate(first_application->name, temp_application->name) ==
			     0)) {
				OpenList = g_slist_append(OpenList, g_strdup(path_texturi));
			}
			if (mime_type != NULL) {
				g_free(mime_type);
			}
			if (temp_application != NULL) {
				gnome_vfs_mime_application_free(temp_application);
				temp_application = NULL;
			}
			g_free(path_texturi);
		}
	}

	if (first_application != NULL) {
		FM_ULOG_DEBUG1("PATH for osso_mime_open_file_list\n");
		for (cursel = OpenList; cursel != NULL; cursel = g_slist_next(cursel)) {
			if (cursel->data != NULL) {
				FM_ULOG_DEBUG1("%s\n", (gchar *) cursel->data);
			}
		}
		openresult =
		    hildon_mime_open_file_list(osso_fm_get_dbus_connection(fm_app_ctx), OpenList);

		g_slist_free(OpenList);
		if (first_application != NULL) {
			gnome_vfs_mime_application_free(first_application);
		}
	} else {
		openresult = NO_ASSOCIATED_APPS_ERR;
	}

	switch (openresult) {
	case 1:
		//  case 0:
		//Succesful
		//gtk_infoprint (GTK_WINDOW (window), executionpath);
		//          gtk_infoprint (GTK_WINDOW (window), 
		//                          first_application->name);
		break;
	case -1:
		//Some error

		fm_infoprint(GTK_WINDOW(window),
				dgettext(HILDON_FM_L10N_PACKAGE,
					    "sfil_ib_opening_not_allowed"));
		FM_ULOG_ERR("Some error");
		break;
	case -2:
		//invalid parameters

	  fm_infoprint(GTK_WINDOW(window),
		       dgettext(HILDON_FM_L10N_PACKAGE,
					    "sfil_ib_opening_not_allowed"));
		FM_ULOG_ERR("Invalid parameters");
		break;
	case -3:
		//Error for short cut creation // not for this operation.
		// break;
	case NO_ASSOCIATED_APPS_ERR:
		osso_fm_unknown_file_open(global_ofm,selection);
		FM_ULOG_ERR("NO APPLICATION\n\n");
		break;
	default:
		osso_fm_unknown_file_open(global_ofm,selection);
		FM_ULOG_ERR("Unknown Error %d\n", openresult);
		break;
	}

	gnome_vfs_file_info_unref(info);
	DEBUG_PRINT_END;
	return;
}

/**
 * osso_fm_file_open:
 * Method that is invoked when a file open operation selected from menu item.
 * @param window Main window
 * @return void
 **/

gboolean osso_fm_file_open(OSSOFileManager *ofm, gboolean menucheck)
{
	gboolean retval = FALSE;
	gboolean valid_iter  = FALSE;
	GtkTreeIter iter;
        GtkFilePath *uri=NULL;
        GtkTreeModel *model = NULL;
        GSList *cur = NULL;
        GSList *selection = NULL;

	DEBUG_PRINT_START;
	if(!menucheck)
	{
		FM_ULOG_INFO("Open Called\n");
	}
	selection_info_update(ofm,FALSE);
	if(check_for_media_items(OPER_OPEN,ofm,menucheck))
	{
		return FALSE;
	}
	switch(selection_info_get_selection_type(ofm->si))
	{
		case SIT_SINGLE_FOLDER_SELECTED:
		case SIT_SINGLE_SAFE_FOLDER_SELECTED:
			retval = TRUE;
			if(!is_path_exists(ofm->si->current_selection->data))
			{
				fm_infoprint(GTK_WINDOW(global_ofm->window), 
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ib_unable_to_open_folder_not_found")); 
				break;
			}
		case SIT_MMC_SELECTED:
		case SIT_DEVICE_SELECTED:
			retval = TRUE;
			if (menucheck) {
				break;
			}
			if (hildon_file_selection_get_active_pane(ofm->fs) ==
			    HILDON_FILE_SELECTION_PANE_CONTENT) {
				hildon_file_selection_set_current_folder(
						ofm->fs,
						ofm->si->current_selection->data,
						NULL);
			}
			break;	
		case SIT_MULTIPLE_SAFE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
			retval = TRUE;
			if(!is_path_exists(ofm->si->current_selection->data))
			{
				fm_infoprint(GTK_WINDOW(global_ofm->window), 
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ib_unable_to_open_folder_not_found")); 
				break;
			}
			hildon_file_selection_set_current_folder(
					ofm->fs,
					ofm->si->current_selection->data,
					NULL);
			break;	
		case SIT_SINGLE_FILE_SELECTED:
			if(!is_path_exists(ofm->si->current_selection->data))
			{
				fm_infonote(GTK_WINDOW(global_ofm->window), 
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ni_unable_to_open_file_not_found"), 
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ni_unable_to_open_file_not_found_ok"));
				break;
			}
		case SIT_MULTIPLE_FILES_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
			selection = hildon_file_selection_get_selected_paths(ofm->fs);
			g_object_get (G_OBJECT (ofm->fs), "model", &model, NULL);
			valid_iter = hildon_file_selection_get_active_content_iter(ofm->fs,&iter);
			if (!valid_iter) {
				invoke_application(GTK_WIDGET(ofm->window),ofm->si->current_selection);
			} 
			else {
			gtk_tree_model_get ( model, &iter,
					 HILDON_FILE_SYSTEM_MODEL_COLUMN_GTK_PATH_INTERNAL, &uri,
					 -1);
			if(gtk_file_path_compare(uri,selection->data)!=0) {
				for(cur=selection;cur!=NULL;cur=cur->next)
				{
				   if(gtk_file_path_compare(uri,cur->data)==0)
				   {
					 selection=g_slist_remove_link(selection,cur);
					 selection=g_slist_insert(selection,cur->data,0);
					 g_slist_free_1(cur);
					 break;
				    }
			         }
			}
			invoke_application(GTK_WIDGET(ofm->window),selection);
			}
			if (selection != NULL) {
				gtk_file_paths_free(selection);
				selection = NULL;
			}
			break;
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			retval = FALSE;
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			fm_infoprint(GTK_WINDOW(ofm->window),
					dgettext(HILDON_FM_L10N_PACKAGE,
						"sfil_ib_opening_not_allowed"));	
			retval = FALSE;
			break;
	}
	return retval;
}

//END: File open related functions


//BEG: Send as email related functions

/**
 * osso_send_as_email:
 * Method to send a file as email.
 * 
 * @param window Pointer to the file manager window
 * @return void
 **/
gboolean osso_send_as_email(OSSOFileManager *ofm, gboolean menucheck)
{
	gint return_val = 0;
	gboolean retval = FALSE;

	DEBUG_PRINT_START;
	if(!menucheck)
	{
		FM_ULOG_INFO("Send as email called\n");
	}
	selection_info_update(ofm,FALSE);
	if(check_for_media_items(OPER_MAIL,ofm,menucheck))
	{
		return FALSE;
	}
	switch(selection_info_get_selection_type(ofm->si))
	{
		case SIT_SINGLE_FILE_SELECTED:
		case SIT_MULTIPLE_FILES_SELECTED:
			retval = TRUE;
			if (menucheck) {
				break;
			}
			return_val = osso_fm_email_files(fm_app_ctx, ofm->si->current_selection);
			switch (return_val) {
			case OSSO_OK:
				FM_ULOG_INFO("Emailing Succesful");
				break;
			case OSSO_ERROR:

				fm_infoprint(GTK_WINDOW(ofm->window),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ib_unable_to_send"));	
				FM_ULOG_ERR("Error sending mail");
				break;
			case OSSO_INVALID:

				fm_infoprint(GTK_WINDOW(ofm->window),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ib_unable_to_send"));	
				FM_ULOG_ERR("Invalid parameters in sending mail");
				break;
			default:

				fm_infoprint(GTK_WINDOW(ofm->window),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ib_unable_to_send"));	
				FM_ULOG_ERR("Unknown error in sending mail");
				break;
			}
			retval = TRUE;
			break;
		case SIT_SINGLE_FOLDER_SELECTED:
		case SIT_SINGLE_SAFE_FOLDER_SELECTED:
		case SIT_MULTIPLE_SAFE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
			if (menucheck == FALSE) {
				retval = FALSE;
				fm_infoprint(GTK_WINDOW(ofm->window),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ib_unable_to_send_folders"));	
			}
			retval = FALSE;
			break;
		case SIT_DEVICE_SELECTED:
			if(ofm->si->selection_in_gateway)
			{
				retval=FALSE;
				if (!menucheck) {
					fm_infoprint(GTK_WINDOW(ofm->window), _("sfil_ib_unable_send_bluetooth"));
				}
				break;
			}
			if(ofm->si->selection_in_media_server) 
			{
				retval=FALSE;
				if (!menucheck) {
					fm_infoprint(GTK_WINDOW(ofm->window), _("sfil_ib_unable_send_shared_media"));
				}
				break;
			}
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ib_unable_to_send_folders"));	
			}
			retval = FALSE;
			break;
		case SIT_MMC_SELECTED:
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window), _("sfil_ib_cannot_send_mmc"));
			}
			retval = FALSE;
			break;
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			retval = FALSE;
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			retval = FALSE;
			break;
	}
	
	DEBUG_PRINT_END;
	return retval;
}

//END: Send as email related functions

//BEG: Global search related functions

/**
 * osso_fm_invoke_global_search:
 * Function to invole global search
 * @return void
 **/
void osso_fm_invoke_global_search()
{
	osso_return_t retval = OSSO_ERROR;
	DEBUG_PRINT_START;
	retval = osso_rpc_run_with_defaults(fm_app_ctx,
					    GLOBAL_SEARCH_APP,
					    SEARCH_FILES, NULL, DBUS_TYPE_INVALID);

	switch (retval) {
	case OSSO_OK:
		//success
		break;
	case OSSO_INVALID:
	case OSSO_ERROR_NAME:
		FM_ULOG_ERR("Error Invalid parameters\n");
		break;
	case OSSO_RPC_ERROR:
	case OSSO_ERROR:
	case OSSO_ERROR_NO_STATE:
	case OSSO_ERROR_STATE_SIZE:
		FM_ULOG_ERR("Error invoking search application\n");
		break;
	}
	DEBUG_PRINT_END;
	return;
}

//END: Global search related functions

/*
 * send_via_bt_call_rpc_method:
 * Function to send a dbus message for sending files via BT
 * @param ctx osso_context_t
 * @paran files array of file paths
 * @return gboolean TRUE if successful
 **/
static gboolean
send_via_bt_call_rpc_method(osso_context_t * ctx, gchar **files)
{
	DBusGProxy *proxy = NULL;
	DBusGConnection *sys_conn = NULL;
	GError *error = NULL;

	sys_conn = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
	
	if(sys_conn == NULL)
	{
		return FALSE;
	}
	
	FM_ULOG_INFO("sent_via_bt_call_rpc_method: calling dbus_g_proxy_new_for_name()\n");
	/* Open connection for btdialogs service */
     	proxy = dbus_g_proxy_new_for_name(sys_conn,
                               CONBTDIALOGS_DBUS_SERVICE,
                               CONBTDIALOGS_DBUS_PATH,
                               CONBTDIALOGS_DBUS_INTERFACE);
     	
	FM_ULOG_INFO("sent_via_bt_call_rpc_method: calling dbus_g_proxy_call()\n");
	/* Send send file request to btdialogs service */
     	if (!dbus_g_proxy_call(proxy, CONBTDIALOGS_SEND_FILE_REQ,
                                 &error, G_TYPE_STRV, files, 
				 G_TYPE_INVALID, G_TYPE_INVALID))
     	{
            	g_print("Error: %s\n", error->message);
            	g_clear_error(&error);
	    	g_object_unref(G_OBJECT(proxy));
		dbus_g_connection_unref(sys_conn);
	    return FALSE;
     	}
	FM_ULOG_INFO("sent_via_bt_call_rpc_method: calling g_object_unref()\n");
	                                                                                                         
     	g_object_unref(G_OBJECT(proxy));
	dbus_g_connection_unref(sys_conn);
	return TRUE;
}	       

/**
 * convert_paths_slist_to_array:
 * Function to convert a selection GSList to a string array
 * @param list list of GtkFilePaths
 * @return gchar ** array of paths
 **/
gchar ** convert_paths_slist_to_array(GSList *list)
{
	GSList *cur = NULL;
	gchar **strs = NULL;
	gint count = 0;
	int i = 0;
	for(cur = list ; cur != NULL; cur = cur->next,count++);
	strs = g_new0(gchar*,count+1);
	for(i = 0,cur = list ; cur != NULL; cur = cur->next,i++)
	{
		strs[i] = osso_fm_path_to_texturi(cur->data);
	}
	strs[count]=NULL;
	for(i = 0 ; i < count ; i++)
	{
		FM_ULOG_INFO("PATHS %s\n",strs[i]);
	}
	return strs;
}


/*
 * osso_fm_paths_send_via_bluetooth:
 * Function to send files via BT
 * @param ctx osso_context_t
 * @param list GSList of file paths
 * @return osso_return_t
 **/
osso_return_t osso_fm_paths_send_via_bluetooth(osso_context_t * osso, GSList * list)
{
	gchar **args = NULL;
	gboolean ret = FALSE;

	FM_ULOG_DEBUG1("send via bluetooth\n");

	if ((osso == NULL) || (list == NULL)) {
		return OSSO_INVALID;
	}

	if (osso_get_dbus_connection(osso) == NULL) {
		FM_ULOG_ERR("error: no D-BUS connection\n");
		return OSSO_INVALID;
	}

	args = convert_paths_slist_to_array(list);

	ret = send_via_bt_call_rpc_method(osso, args);

	FM_ULOG_INFO("osso_fm_paths_send_via_bluetooth: got back from send_via_bt_call_rpc_method\n");
	/* TODO: args should be freed */
	if(ret)
		return OSSO_OK;
	return OSSO_INVALID;
}
/**
 * osso_fm_send_via_bluetooth:
* Method to send a file via BT.
* @param ofm Pointer to the OSSOFileManager struct
* @param menucheck gboolean 
* @return void
*/
gboolean osso_fm_send_via_bluetooth(OSSOFileManager *ofm, gboolean menucheck)
{
	gint return_val = 0;
	gboolean retval = FALSE;

	DEBUG_PRINT_START;
	if(!menucheck)
	{
		FM_ULOG_INFO("Send via bluetooth called\n");
	}
	selection_info_update(ofm,FALSE);
	if(check_for_media_items(OPER_SENDVIABT,ofm,menucheck))
	{
		return FALSE;
	}
	switch(selection_info_get_selection_type(ofm->si))
	{
		case SIT_SINGLE_FILE_SELECTED:
		case SIT_MULTIPLE_FILES_SELECTED:
			retval = TRUE;
			if (menucheck) {
				break;
			}
			return_val = osso_fm_paths_send_via_bluetooth(fm_app_ctx, 
					ofm->si->current_selection);
			switch (return_val) {
			case OSSO_OK:
				FM_ULOG_INFO("Sending via bluetooth Succesful\n");
				break;
			case OSSO_ERROR:

				fm_infoprint(GTK_WINDOW(ofm->window),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ib_unable_to_send"));	
				FM_ULOG_ERR("Error sending via bluetooth\n");
				break;
			case OSSO_INVALID:

				fm_infoprint(GTK_WINDOW(ofm->window),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ib_unable_to_send"));	
				FM_ULOG_ERR("Invalid parameters in sending via bluetooth\n");
				break;
			default:

				fm_infoprint(GTK_WINDOW(ofm->window),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ib_unable_to_send"));	
				FM_ULOG_ERR("Unknown error in sending via bluetooth\n");
				break;
			}
			retval = TRUE;
			break;
		case SIT_SINGLE_FOLDER_SELECTED:
		case SIT_SINGLE_SAFE_FOLDER_SELECTED:
		case SIT_MULTIPLE_SAFE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
			if (menucheck == FALSE) {
				retval = FALSE;
				fm_infoprint(GTK_WINDOW(ofm->window),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ib_unable_to_send_folders"));	
			}
			retval = FALSE;
			break;
		case SIT_DEVICE_SELECTED:
			if(ofm->si->selection_in_gateway)
			{
				retval=FALSE;
				if (!menucheck) {
					fm_infoprint(GTK_WINDOW(ofm->window), _("sfil_ib_unable_send_bluetooth"));
				}
				break;
			}
			if(ofm->si->selection_in_media_server) 
			{
				retval=FALSE;
				if (!menucheck) {
					fm_infoprint(GTK_WINDOW(ofm->window), _("sfil_ib_unable_send_shared_media"));
				}
				break;
			}
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ib_unable_to_send_folders"));	
			}
			retval = FALSE;
			break;
		case SIT_MMC_SELECTED:
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window), _("sfil_ib_cannot_send_mmc"));
			}
			retval = FALSE;
			break;
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			retval = FALSE;
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			retval = FALSE;
			break;
	}
	
	DEBUG_PRINT_END;
	return retval;
}

/**
 * get_bt_addresses:
 * Function to the list paired BT devices
 * @return GSList of BT addresses
 **/
GSList *get_bt_addresses()
{
	DBusConnection *sys_conn;
	DBusMessage *m = NULL, *reply = NULL;
	//dbus_bool_t ret = FALSE;
	DBusError err;
	DBusError err1;
	DBusMessageIter iter;
	gchar *s = NULL;
	int count = 0,i=0;
	GSList *list = NULL;
	DEBUG_PRINT_START;
	m = dbus_message_new_method_call("org.bluez",
					"/org/bluez/hci0", 
					"org.bluez.Adapter",
					"ListBondings");
	sys_conn = (DBusConnection *) osso_get_sys_dbus_connection(fm_app_ctx);
	dbus_error_init(&err);
	reply = dbus_connection_send_with_reply_and_block(sys_conn, m, 20000, &err);
	FM_ULOG_DEBUG1("get_bt_addresses : sent message to bluez\n");
	if (reply == NULL) {
		FM_ULOG_ERR("get_bt_addresses:reply was null, error: %s", err.message);
		dbus_error_free(&err);
		return NULL;
	} else {
		dbus_message_iter_init(reply, &iter);
		dbus_error_init(&err1);
		if(!dbus_message_get_args(reply,
					&err1,
					DBUS_TYPE_ARRAY,
					DBUS_TYPE_STRING,
					&s,
					&count,
					DBUS_TYPE_INVALID))
		{
			FM_ULOG_ERR("Error getting arguments from dbus msg: %s\n",err1.message);
			dbus_error_free(&err1);
			return NULL;
		}
		else 
		{
			for(i=0;i<count;i++)
			{
				list = g_slist_append(list,((char **)s)[i]);
				FM_ULOG_DEBUG1("BT ADDRS(%d) %s\n",i,((char **)s)[i]);
			}
			return list;
		}
	}
	DEBUG_PRINT_END;
	return NULL;
}

