/**
 * @file osso_fm_callbacks.c
 *
 * This file contains implementation of all the callback routines.
 *
 * Copyright (C) 2005 Nokia Corporation
 */


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include <gtk/gtk.h>

#include "osso_fm_generic.h"
#include "osso_fm_callbacks.h"
#include "osso_fm_operations.h"
#include "osso_fm_xfer.h"
#include "osso_fm_utils.h"
#include "osso_fm_fmstate.h"
#include "osso_fm_details.h"
#include <hildon/hildon-help.h>
#include "osso_fm_controller.h"


extern clock_t start;
extern clock_t end;
GtkToolbar * CreateToolbar(HildonProgram *prog);
gboolean create_file_selection(OSSOFileManager *ofm);
void osso_fm_set_fs_view_mode(OSSOFileManager* ofm, HildonFileSelectionMode mode);
FMMainMenu *osso_fm_create_main_menu(HildonProgram *prog,
				     HildonWindow *window,
				     HildonFileSelection *fs);
FMPopupMenu *osso_fm_create_popup_menu(OSSOFileManager *ofm);

/**
 * on_fileselection_row_changed:
 *
 *  
 * @param treemodel - Pointer to the GtkTreeModel
 * @param changed_path - Pointer to the GtkTreePath
 * @param changed_iter - Pointer to the GtkTreeIter
 * @param view - Pointer to the appview
 * 
 * @returns void
 **/
void
on_fileselection_row_changed(GtkTreeModel * treemodel,
			     GtkTreePath * changed_path, GtkTreeIter * changed_iter, gpointer window)
{
	GtkTreeIter current_iter;
	GtkTreePath *current_path = NULL;
	gchar *title = NULL;

	if (!hildon_file_selection_get_current_folder_iter((HildonFileSelection *) global_ofm->fs,
							   &current_iter)) {
		FM_ULOG_ERR("on_fileselection_row_changed:Error getting cur fol Iter\n");
		return;
	}
	current_path = gtk_tree_model_get_path(treemodel, &current_iter);
	if (gtk_tree_path_compare(current_path, changed_path) == 0) {
		gtk_tree_model_get(treemodel, changed_iter,
				   HILDON_FILE_SYSTEM_MODEL_COLUMN_DISPLAY_NAME, &title, -1);
		if (title != NULL) {
			gtk_window_set_title (window, title);
			FM_ULOG_DEBUG1("Title is set\n");
			g_free(title);
		} else {
			FM_ULOG_ERR("Title is NULL\n");
		}
	}
	gtk_tree_path_free(current_path);
}

/**
 * update_fm_title:
 * This function updates the title bar of the application.
 * 
 * @param file_sel_widget is file selection widget.
 * @param view is pointer to the HildonAppView
 *
 * @return void
 **/
void update_fm_title(GtkWidget * file_sel_widget, HildonWindow *window)
{
	GtkFilePath *path = NULL;
	gboolean showthumbnails = FALSE;
	GtkSortType order = 0;
	HildonFileSelectionSortKey key = 0;
	HildonFileSystemModel *fsmodel = NULL;
	GtkTreeIter iter;	// = NULL;

	gchar *title = NULL;

	DEBUG_PRINT_START;

	if (file_sel_widget == NULL || window == NULL) {
		FM_ULOG_ERR("Invalid parameter : (null)");
		return;
	}

	path = hildon_file_selection_get_current_folder((HildonFileSelection *) file_sel_widget);
	if (path == NULL) {
		FM_ULOG_ERR("Invalid path parameter : (null)");
		return;
	}
	FM_ULOG_DEBUG1("%s", osso_fm_path_to_texturi(path));
	if (osso_fm_get_config_value_for_path(path,
					      &showthumbnails, &key, &order) == 0) {

		//osso_fm_set_mode((HildonFileSelection *) file_sel_widget, showthumbnails);
		hildon_file_selection_set_sort_key((HildonFileSelection *) file_sel_widget,
						   key, order);
		osso_fm_set_fs_view_mode(global_ofm, 
				showthumbnails?
					HILDON_FILE_SELECTION_MODE_THUMBNAILS:
					HILDON_FILE_SELECTION_MODE_LIST);
	}

	if (hildon_file_selection_get_current_folder_iter((HildonFileSelection *) file_sel_widget,
							  &iter)) {
		g_object_get(G_OBJECT(file_sel_widget), "model", &fsmodel, NULL);
		if (fsmodel == NULL) {
			FM_ULOG_ERR("Error getting model\n");
			gtk_file_path_free(path);
			return;
		}
		gtk_tree_model_get((GtkTreeModel *) fsmodel, &iter,
				   HILDON_FILE_SYSTEM_MODEL_COLUMN_DISPLAY_NAME, &title, -1);
	} else {
		title = osso_fm_path_get_display_name (path);
	}

	if (title != NULL) {
		gtk_window_set_title (GTK_WINDOW (window), title);
		FM_ULOG_DEBUG1("Title is set\n");
	} else {
		FM_ULOG_ERR("Title is NULL\n");
	}
	g_free(title);
	gtk_file_path_free(path);
	appstate.mainwindow.checkselection = FALSE;

	DEBUG_PRINT_END;
	return;
}


/**
 * on_menu_file_open_activate:
 *
 * This function gets called when open menu option is clicked
 *
 * @param menuitem Pointer to GtkMenuItem
 * @param window Pointer to OSSOFileManager
 *
 * @return void
 **/
void on_menu_file_open_activate(GtkMenuItem * menuitem, gpointer data)
{
	DEBUG_PRINT_START;

	if (data == NULL) {
		FM_ULOG_ERR("Err:on_menu_file_open_activate:data = NULL\n");
		return;
	}
	osso_fm_set_operation_state(OPER_OPEN);
	osso_fm_file_open((OSSOFileManager *)data, FALSE);
	osso_fm_set_operation_state(OPER_NONE);

	DEBUG_PRINT_END;
	return;
}

/**
 * on_pane_changed:
 *
 * This function gets called when pane changed
 * 
 * @param obj pointer to the GObject
 * @param param pointer to the GParamSpec
 * @param app pointer to the HildonApp 
 *
 * @returns void
 **/
void on_pane_changed(GObject * obj, GParamSpec * param, gpointer app)
{
	DEBUG_PRINT_START;
	appstate.mainwindow.checkselection = FALSE;
	g_signal_emit_by_name(global_ofm->fs,"selection-changed");
	//refresh_toolbar_buttons(global_ofm);
	DEBUG_PRINT_END;
	return;
}

void on_file_loading_finished( HildonFileSystemModel *fsmodel)
{
	DEBUG_PRINT_START;
	fm_loading=FALSE;
	DEBUG_PRINT_END;
}

/**
 * on_selection_changed:
 *
 * This Method called when selection changed
 *
 * @param menuitem Pointer to GtkMenuItem
 * @param window Pointer to app
 *
 * @return void
 **/
static void on_selection_changed(GtkWidget * widget, gpointer app)
{
	GTimeVal time_before;
	GTimeVal time_after;

	DEBUG_PRINT_START;
	global_ofm->si->needs_update = TRUE;
	if (global_ofm->si->just_selected_all) {
		global_ofm->si->just_selected_all = FALSE;
	} else {
		global_ofm->si->selected_all = FALSE;
	}
	appstate.mainwindow.checkselection = FALSE;

	g_get_current_time(&time_before);
        /* this calls update_current_selection_state() */
	refresh_toolbar_buttons(global_ofm);
	g_get_current_time(&time_after);

	if(global_ofm->fmpmenu) {
		if (GTK_WIDGET_VISIBLE(global_ofm->fmpmenu->menu)) {
			refresh_popup_menu_items((OSSOFileManager *) global_ofm, TRUE);
		}
	}

	FM_ULOG_INFO("Sel Changed update took %ld usec\n",
			timeval_diff(&time_after, &time_before));
	close_selection_change_sensitivity_open_dialogs(global_ofm);
	DEBUG_PRINT_END;
	return;
}

/**
 * get_count_of_listed_items:
 *
 * Helper function counts the no.of listed items in the current folder
 * @param num_items_dimmed pointer to gint or NULL.
 * 
 * @return gint -  no.of listed items
 **/
static gint get_count_of_listed_items(gint *num_items_dimmed)
{
	HildonFileSystemModel *fsmodel = NULL;
	GtkTreeIter parent_iter, iter;
	gint count = 0;
	gint count_dimmed = 0;
	gboolean itemhidden = FALSE;
	gboolean available = FALSE;
	DEBUG_PRINT_START;

	g_object_get(global_ofm->fs, "model", &fsmodel, NULL);
	if (fsmodel == NULL) {
		FM_ULOG_ERR("get_count_of_listed_items:fsmodel == NULL\n");
		DEBUG_PRINT_END;
		return -1;
	}
	if (!hildon_file_selection_get_current_folder_iter(global_ofm->fs,
							   &parent_iter)) {
		FM_ULOG_ERR("hildon_file_selection_get_current"
			    "_folder_iter failed\n");
		DEBUG_PRINT_END;
		return -1;
	}
	if (!gtk_tree_model_iter_children((GtkTreeModel *) fsmodel, &iter, &parent_iter)) {
		FM_ULOG_INFO("No Items in the current folder\n");
		DEBUG_PRINT_END;
		return 0;
	}
	/*Count the number of children*/
	do {
		gtk_tree_model_get((GtkTreeModel *) fsmodel, &iter,
				   HILDON_FILE_SYSTEM_MODEL_COLUMN_IS_HIDDEN, &itemhidden,
					HILDON_FILE_SYSTEM_MODEL_COLUMN_IS_AVAILABLE, &available, -1);
		if (!itemhidden)
			count++;
		if (!available) {
			count_dimmed++;
		}
	} while (gtk_tree_model_iter_next((GtkTreeModel *) fsmodel, &iter));
	DEBUG_PRINT_END;

	if (num_items_dimmed != NULL)
		*num_items_dimmed = count_dimmed;

	return count;
}

/**
 * on_current_folder_changed:
 *
 * This function gets called when current folder changed
 *
 * @param widget Pointer to GtkMenuItem
 * @param data  pointer to the app
 *
 * @return void
 **/
void 
on_current_folder_changed (GtkWidget *widget, gpointer data)
{
        hildon_file_selection_undim_all (global_ofm->fs);

	if(get_count_of_listed_items(NULL) == 0)
	{
		g_signal_emit_by_name(global_ofm->fs,"selection-changed");
		FM_ULOG_ERR("Emitting 'selection-changed'\n");
	}
	return;
}
/**
 * on_newfolder_activate:
 * SP+
 */
void on_newfolder_activate(gpointer data)
{
	DEBUG_PRINT_START;

	if (data == NULL) {
		FM_ULOG_ERR("Err:on_newfolder_activate:data = NULL");
		return;
	}
	osso_fm_set_operation_state(OPER_NEWFOLDER);
	global_ofm->operation_started = FALSE;
	osso_fm_new_folder((OSSOFileManager *) data, FALSE);
	if (!global_ofm->operation_started) {
		osso_fm_set_operation_state(OPER_NONE);
	}
	DEBUG_PRINT_END;
	return;
}

/**
 * on_menu_file_new_folder_activate:
 *
 * Function gets called on clicking new folder menu item
 * @param menuitem pointer to the GtkMenuItem
 * @param data pointer to the app
 * 
 * @return void
 **/
void on_menu_file_new_folder_activate(GtkMenuItem * menuitem, gpointer data)
{
	DEBUG_PRINT_START;
	on_newfolder_activate(data);
	DEBUG_PRINT_END;
	return;
}


/**
 * on_menu_file_rename_activate:
 * 
 * This function created rename dialog
 * @param menuitem GtkMenuItem from which the event is triggered
 * @param window HildonApp pointer
 * @return void
 **/
void on_menu_file_rename_activate(GtkMenuItem * menuitem, gpointer window)
{
	DEBUG_PRINT_START;

	if (window == NULL) {
		FM_ULOG_ERR("Invalid parameter : window (null)");
		return;
	}
	osso_fm_set_operation_state(OPER_RENAME);
	global_ofm->operation_started = FALSE;
	osso_fm_rename_activate(global_ofm, FALSE);
	if (!global_ofm->operation_started) {
		osso_fm_set_operation_state(OPER_NONE);
	}
	DEBUG_PRINT_END;
	return;
}

/**
 * on_menu_tools_format_activate:
 * Callback function for menu find activate
 */
void on_menu_tools_format_activate(GtkMenuItem * menuitem, gpointer window)
{
	DEBUG_PRINT_START;
	osso_fm_format_activate (global_ofm, FALSE);
	DEBUG_PRINT_END;
}

/**
 * on_delete_activate:
 * 
 * This function creates delete dialog 
 * @param data pointer to the app
 *
 * @return void
 **/
void on_delete_activate(gpointer data)
{
	DEBUG_PRINT_START;
	if (data == NULL) {
		FM_ULOG_ERR("Err:on_delete_activate:data = NULL\n");
		return;
	}
	if(global_ofm->cur_operation != OPER_NONE) {
		FM_ULOG_ERR("Delete operation\n");
		return;
	} 
	osso_fm_set_operation_state(OPER_DELETE);
	global_ofm->operation_started = FALSE;
	osso_fm_delete_activate(data, FALSE);
	if (!global_ofm->operation_started) {
		osso_fm_set_operation_state(OPER_NONE);
	}

	DEBUG_PRINT_END;
	return;
}

/**
 * on_menu_file_delete_activate:
 *
 * This function gets called on clicking the delete menu item
 * 
 * @param menuitem pointer to the GtkMenuItem
 * @param data pointer to the app
 *
 * @return void
 **/
void on_menu_file_delete_activate(GtkMenuItem * menuitem, gpointer data)
{
	DEBUG_PRINT_START;
	on_delete_activate(data);
	DEBUG_PRINT_END;
}

/**
 * on_menu_send_as_email_activate:
 *
 * Function sends selected files as an attachment to email.
 *
 * @param menuitem GtkMenuItem from which the event is triggered
 * @param window HildonApp pointer
 *
 * @return void
 **/
void on_menu_send_as_email_activate(GtkMenuItem * menuitem, gpointer data)
{
	DEBUG_PRINT_START;

	if (data == NULL) {
		FM_ULOG_ERR("on_menu_send_as_email_activate:Invalid parameter:data = null");
		return;
	}

	osso_fm_set_operation_state(OPER_MAIL);
	osso_send_as_email(data, FALSE);
	osso_fm_set_operation_state(OPER_NONE);

	DEBUG_PRINT_END;
	return;
}

/**
 * on_menu_send_via_bluetooth_activate:
 *
 * Function sends selected files via bluetooth
 *
 * @param menuitem GtkMenuItem from which the event is triggered
 * @param window HildonApp pointer
 *
 * @return void
 **/
void on_menu_send_via_bluetooth_activate(GtkMenuItem * menuitem, gpointer data)
{
	DEBUG_PRINT_START;
	osso_fm_send_via_bluetooth(global_ofm,FALSE);
	DEBUG_PRINT_END;
}

//NEW CODE ENDS

/**
 * on_details_activate:
 *
 * Function displays details of selected item
 *
 * @param data pointer to the app
 *
 * @return void
 **/
void on_details_activate(gpointer data)
{
	DEBUG_PRINT_START;
	if (data == NULL) {
		FM_ULOG_ERR("Err:on_details_activate:data = NULL\n");
		return;
	}
	osso_fm_set_operation_state(OPER_DETAILS);
	osso_fm_details_activate(data, FALSE);
	osso_fm_set_operation_state(OPER_NONE);

	DEBUG_PRINT_END;
	return;
}

/**
 * on_menu_details_activate:
 * 
 * Function gets called on clicking the details menu button, 
 * shows the details of particuler item selected 
 *
 * @param menuitem pointer to the GtkMenuItem
 * @param data pointer to the app
 *
 * @return void
 **/
void on_menu_details_activate(GtkMenuItem * menuitem, gpointer data)
{
	DEBUG_PRINT_START;
	on_details_activate(data);
	DEBUG_PRINT_END;
}
	

/**
 * on_menu_edit_cut_activate:
 *
 * Cut the selected file path to the clipboard.
 *
 * @param menuitem GtkMenuItem from which the event is triggered
 * @param window HildonApp pointer
 *
 * @return void
 **/
void on_menu_edit_cut_activate(GtkMenuItem * menuitem, gpointer data)
{
	DEBUG_PRINT_START;

	if (data == NULL) {
		FM_ULOG_ERR("on_menu_edit_cut_activate: data = NULL");
		return;
	}
	osso_fm_set_operation_state(OPER_CUT);
	on_cut_activate(data, FALSE);
	osso_fm_set_operation_state(OPER_NONE);

	DEBUG_PRINT_END;
}

/** 
 * get_item_list_to_cut:
 *
 * Function gets the list of items that are cut and skips if the items are readonly
 * 
 * @param si constant pointer to the SelectionInfo
 * @param skipped_items integer pointer, selected items are skipped when the items are readonly
 *
 * @retun GSList of items
 */
static GSList * get_item_list_to_cut(const SelectionInfo *si, gint *skipped_items)
{
	GSList *cur = NULL, *final_list = NULL;
	DEBUG_PRINT_START;
	if(skipped_items != NULL)
		*skipped_items = 0;
	for(cur=si->current_selection_withinfo ; cur != NULL ; cur = cur->next)
	{
		if(((OSSOFMFileInfo *)cur->data)->type == HILDON_FILE_SYSTEM_MODEL_FILE)
		{
			final_list = g_slist_append(final_list,
					gtk_file_path_copy(((OSSOFMFileInfo *)cur->data)->path));	
		}
		else if(((OSSOFMFileInfo *)cur->data)->type == HILDON_FILE_SYSTEM_MODEL_FOLDER)
		{
			if(((OSSOFMFileInfo *)cur->data)->readonly)
			{
				if(skipped_items != NULL)
					(*skipped_items)++;
			}
			else
			{
				final_list = g_slist_append(final_list,
					gtk_file_path_copy(((OSSOFMFileInfo *)cur->data)->path));
			}
		}
		else
		{
			if(skipped_items != NULL)
				(*skipped_items)++;
		}

	}
	DEBUG_PRINT_END;
	return final_list;
}

/** 
 * dim_selection:
 * 
 * 
 * @param ofm constant pointer to the OSSOFileManager
 * @param sel GSList of selected items 
 *
 * @return void
 **/
static void dim_selection(const OSSOFileManager *ofm,GSList *sel)
{
	HildonFileSystemModel *model = NULL;
	GtkTreeIter iter;
	GSList *cur;
	DEBUG_PRINT_START;
	hildon_file_selection_undim_all(ofm->fs);
	
	g_object_get(G_OBJECT(ofm->fs), "model", &model, NULL);

	for(cur=sel;cur!=NULL;cur=cur->next)
	{
		if(hildon_file_system_model_search_path(model,
					cur->data,&iter,NULL,TRUE))
		{
			hildon_file_system_model_iter_available(model, &iter,
							FALSE);
		}
		else
		{
			FM_ULOG_ERR("Unable to find iter %s\n",(gchar *)cur->data);
			return;
		}
	}
	DEBUG_PRINT_END;
}
/**
 * on_cut_activate:
 *
 * Function  Cut the selected file path to the clipboard.
 *
 * @param menuitem GtkMenuItem from which the event is triggered
 * @param window HildonApp pointer
 * @param menucheck  gboolean 
 *
 * @return TRUE if operation is succesful else FALSE 
 **/
gboolean on_cut_activate(OSSOFileManager *ofm, gboolean menucheck)
{
	gboolean retval = FALSE;
	GSList *final_sel = NULL;
	gint skipped_items = 0;
	gboolean readonly_path = FALSE;

	DEBUG_PRINT_START;

	selection_info_update(ofm,!menucheck);

	if(!ofm->si->selection_in_gateway)
	{
		readonly_path = is_path_readonly(ofm->si->current_selection->data,TRUE);
	}
	if(check_for_media_items(OPER_CUT,ofm,menucheck))
	{
		return FALSE;
	}
	switch(selection_info_get_selection_type(ofm->si))
	{
		case SIT_SINGLE_SAFE_FOLDER_SELECTED:
			if (menucheck == FALSE) {
				fm_infoprint (GTK_WINDOW(ofm->window),
					      dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE, 
						      	"sfil_ib_unable_to_cut_selected_folder"));
			}
			retval = FALSE;
			break;
		case SIT_MULTIPLE_SAFE_FOLDERS_SELECTED:
			
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window),
//						     osso_common_error
//						     (CERM_CAN_NOT_CUT_SELECTED_ITEMS));
							dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ib_unable_to_cut_selected_items"));						     
			}
			retval = FALSE;
			break;
		case SIT_DEVICE_SELECTED:
			if(ofm->si->selection_in_gateway)
			{
				retval=FALSE;
				if (!menucheck) {
					if(is_path_virtual(ofm->si->current_selection->data)) { 
						fm_infoprint(GTK_WINDOW(ofm->window), _("sfil_ib_unable_cut_bluetooth"));
					} else  {
						fm_infoprint(GTK_WINDOW(ofm->window), 
							_("sfil_ib_cannot_cut_device"));						
					}
				}
				break;
			}
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window), 
						_("sfil_ib_cannot_cut_device"));						
			}
			break;
		case SIT_MMC_SELECTED:
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window), 
//						osso_common_error(CERM_CAN_NOT_CUT_MMC));
						_("sfil_ib_cannot_cut_mmc"));						
			}
			retval = FALSE;
			break;
			
		case SIT_SINGLE_FOLDER_SELECTED:
		case SIT_SINGLE_FILE_SELECTED:
		case SIT_MULTIPLE_FILES_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
			final_sel = get_item_list_to_cut(ofm->si,&skipped_items);
			if(final_sel == NULL)
			{
				if (menucheck == FALSE) {
					if(ofm->si->no_folders == 1 && ofm->si->no_files == 0)
					{
						fm_infoprint (GTK_WINDOW(ofm->window),
						      dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
							      "sfil_ib_unable_to_cut_selected_folder"));
					}
					else
					{
						fm_infoprint(GTK_WINDOW(ofm->window),
//							     osso_common_error
//							     (CERM_CAN_NOT_CUT_SELECTED_ITEMS));
							dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,					     
								"sfil_ib_unable_to_cut_selected_items"));							     
					}
				}
				retval = FALSE;
				break;
			}
			retval = !(readonly_path && !ofm->si->selection_in_gateway);
			if (menucheck) break;
			if(skipped_items != 0)
			{
				fm_infoprint(GTK_WINDOW(ofm->window),
						dgettext(HILDON_FM_L10N_PACKAGE, 
							"sfil_ib_cannot_cut_some_items"));
			}
			if(readonly_path && !ofm->si->selection_in_gateway)
			{
				fm_infoprint(GTK_WINDOW(ofm->window), 
						dgettext(HILDON_FM_L10N_PACKAGE, 
							"sfil_ib_readonly_location")); 
				retval = FALSE;
				break;
			}
			osso_fm_copy_to_clipboard((GtkWidget *)ofm->window, final_sel, TRUE);
			dim_selection(ofm,final_sel);
			break;
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			retval = FALSE;
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			retval = FALSE;
			break;
	}
	if (final_sel != NULL) {
		gtk_file_paths_free(final_sel);
	}

	DEBUG_PRINT_END;
	return retval;
}


void on_menu_edit_copy_activate(GtkMenuItem * menuitem, gpointer data)
{
	DEBUG_PRINT_START;

	if (data == NULL) {
		FM_ULOG_ERR("on_menu_edit_copy_activate:data = NULL");
		return;
	}
	osso_fm_set_operation_state(OPER_COPY);
	on_copy_activate((OSSOFileManager *)data, FALSE);
	osso_fm_set_operation_state(OPER_NONE);

	DEBUG_PRINT_END;
}



/**
 * on_copy_activate:
 *
 * Function copies the selected paths to clipboard
 *
 * @param menuitem GtkMenuItem from which the event is triggered
 * @param window HildonApp pointer
 *
 * @return void
 **/
gboolean on_copy_activate(OSSOFileManager *ofm, gboolean menucheck)
{
	gboolean retval = FALSE;
	DEBUG_PRINT_START;

	selection_info_update(ofm,FALSE);
	if(check_for_media_items(OPER_COPY,ofm,menucheck))
	{
		return FALSE;
	}
	switch(selection_info_get_selection_type(ofm->si))
	{
		case SIT_DEVICE_SELECTED:
			if(ofm->si->selection_in_gateway)
			{
				retval=FALSE;
				if (!menucheck) {
					if(is_path_virtual(ofm->si->current_selection->data)) {
						fm_infoprint(GTK_WINDOW(ofm->window), _("sfil_ib_unable_copy_bluetooth"));
					} else {
						fm_infoprint(GTK_WINDOW(ofm->window), 
							_("sfil_ib_cannot_copy_device"));						
					}
				}
				break;
			}
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window), 
						_("sfil_ib_cannot_copy_device"));						
			}
			break;
			
		case SIT_MMC_SELECTED:
		case SIT_SINGLE_FOLDER_SELECTED:
		case SIT_SINGLE_FILE_SELECTED:
		case SIT_SINGLE_SAFE_FOLDER_SELECTED:
		case SIT_MULTIPLE_FILES_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_SAFE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
			retval = TRUE;
			if (menucheck) break;
			osso_fm_copy_to_clipboard((GtkWidget *)ofm->window, 
					ofm->si->current_selection, FALSE);
			hildon_file_selection_undim_all(ofm->fs);
			break;
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			retval = FALSE;
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			retval = FALSE;
			break;
	}
	DEBUG_PRINT_END;
	return retval;
}


/**
 * on_menu_edit_paste_activate:
 *
 * Function pastes the clipboard path to the current selected path
 *
 * @param menuitem GtkMenuItem from which the event is triggered
 * @param fileselection Pointer to HildonFileSelection
 *
 * @return void
 **/
void on_menu_edit_paste_activate(GtkMenuItem * menuitem, gpointer data)
{
	DEBUG_PRINT_START;

	if (data == NULL) {
		FM_ULOG_ERR("on_menu_edit_paste_activate:data = NULL\n");
		return;
	}
	osso_fm_set_operation_state(OPER_PASTE);
	global_ofm->operation_started = FALSE;
	on_paste_activate(data, FALSE);
	if (!global_ofm->operation_started) {
		osso_fm_set_operation_state(OPER_NONE);
	}
	DEBUG_PRINT_END;
}



/**
 * on_paste_activate:
 * Function pastes the clipboard path to the current selected path
 *
 * @param menuitem GtkMenuItem from which the event is triggered
 * @param fileselection Pointer to HildonFileSelection
 * @param menucheck TRUE to run opeartion in scilent mode. (used for validating menu items)
 *
 * @return TRUE if operation succeed 
 **/
gboolean on_paste_activate(OSSOFileManager *ofm, gboolean menucheck)
{
	GtkFilePath *targetpath = NULL;
	GSList *sourceselection = NULL;
	GList *sourcelist = NULL;
	GList *targetlist = NULL;
	gboolean cut_operation = FALSE;
	HildonFileSystemModel *fsmodel = NULL;
	gboolean retval = FALSE;
	gboolean autorenameflag = FALSE;
	SrcToTgtStatus sttsres;
	gboolean readonly_path = FALSE;

	DEBUG_PRINT_START;

	if(!ofm->si->selection_in_gateway)
	{
		readonly_path = is_path_readonly(ofm->si->current_selection->data,TRUE);
	}
	g_object_get(G_OBJECT(global_ofm->fs), "model", &fsmodel, NULL);
	if (fsmodel == NULL) {
		FM_ULOG_ERR("Error getting model\n");
		return FALSE;
	}
	if(is_xfer_in_progress())
	{
		FM_ULOG_ERR("on_paste_activate:xfer in progress\n");
		return FALSE;
	}
	osso_fm_get_list_from_clipboard((GtkWidget *)ofm->window, &sourceselection, &cut_operation);

	if (sourceselection == NULL) {
		if (menucheck == FALSE) {
			fm_infoprint(GTK_WINDOW(ofm->window),
				dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"ecoc_ib_edwin_nothing_to_paste"));				     
		}
		DEBUG_PRINT_END;
		return FALSE;
	} 
	if(menucheck == FALSE) {
		if(!is_clipboard_contents_valid(sourceselection)) {
			FM_ULOG_INFO("Clipboard contents are of different gateway");
			fm_infoprint(GTK_WINDOW(ofm->window),
				dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"ecoc_ib_edwin_nothing_to_paste"));					
			return TRUE;
		}
	}

	if(!menucheck && 
		ofm->si->selection_in_gateway && is_texturi_in_gateway(sourceselection->data))
	{
		if(cut_operation)
		{
			fm_infonote(GTK_WINDOW(ofm->window),
					_("sfil_ni_move_not_supported_bluetooth"),NULL);
		}
		else
		{
			fm_infonote(GTK_WINDOW(ofm->window),
					_("sfil_ni_copy_not_supported_bluetooth"),NULL);
		}
		osso_fm_free_string_slist(sourceselection);
		g_slist_free(sourceselection);
		DEBUG_PRINT_END;
		return FALSE;
	}
	if(ofm->si->selection_in_media_server)
	{
		if(!menucheck)
		{
			fm_infoprint(GTK_WINDOW(ofm->window),
				_("sfil_ib_unable_paste_shared_media"));
		}
		osso_fm_free_string_slist(sourceselection);
		g_slist_free(sourceselection);
		DEBUG_PRINT_END;
		return FALSE;
	}
	selection_info_update(ofm,FALSE);
	if(check_for_media_items(OPER_DUPLICATE,ofm,menucheck))
	{
		return FALSE;
	}
	switch(selection_info_get_selection_type(ofm->si))
	{
		case SIT_SINGLE_FILE_SELECTED:
		case SIT_MULTIPLE_FILES_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_SAFE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
			targetpath = gtk_file_path_copy(ofm->si->current_folder);
			break;
		case SIT_DEVICE_SELECTED:
			 if(ofm->si->selection_in_gateway && 
					 is_path_virtual(ofm->si->current_selection->data)) {
				 retval=FALSE;
				 if (!menucheck) {
						fm_infoprint(GTK_WINDOW(ofm->window),_("sfil_ib_unable_paste_bluetooth"));
				 }
				 break;
			 }
		case SIT_MMC_SELECTED:
		case SIT_SINGLE_FOLDER_SELECTED:
		case SIT_SINGLE_SAFE_FOLDER_SELECTED:
			targetpath = gtk_file_path_copy(ofm->si->current_selection->data);
			break;
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			retval = FALSE;
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			retval = FALSE;
			break;
	}
	if (targetpath == NULL) {
		return retval;
	}
	if(!ofm->si->selection_in_gateway)
	{
		if(is_path_readonly(targetpath,FALSE))
		{
			if (menucheck)
				return FALSE;

			fm_infoprint(GTK_WINDOW(ofm->window),
					dgettext(HILDON_FM_L10N_PACKAGE,"sfil_ib_readonly_location"));					
			if (sourceselection != NULL) {
				osso_fm_free_string_slist(sourceselection);
				g_slist_free(sourceselection);
				sourceselection = NULL;
			}
			if (targetpath != NULL) {
				gtk_file_path_free(targetpath);
			}
			DEBUG_PRINT_END;
			return FALSE;
		}
	}
	sttsres = check_source_to_target(sourceselection, targetpath);
	switch (sttsres) {
	case STTS_MEMORY_ERR:
	case STTS_TGT_IN_SRC:
		if (menucheck == FALSE) {
			if (cut_operation == TRUE) {
				fm_infoprint(GTK_WINDOW(ofm->window),
						dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"ckdg_ib_unable_to_move_folder_into_itself"));					     
			} else {
				fm_infoprint(GTK_WINDOW(ofm->window),
						dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"ckdg_ib_unable_to_copy_folder_into_itself"));					     
			}
		}
		if (sourceselection != NULL) {
			osso_fm_free_string_slist(sourceselection);
			g_slist_free(sourceselection);
			sourceselection = NULL;
		}
		if (targetpath != NULL) {
			gtk_file_path_free(targetpath);
			targetpath = NULL;
		}
		return FALSE;
		break;
	case STTS_TGT_REPLACES_SRC:
		if (menucheck == FALSE) {
			fm_infoprint(GTK_WINDOW(ofm->window),
			dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
				"sfil_ib_unable_to_copy_or_move_a_subfolder_within_original_folder"));
		}
		if (sourceselection != NULL) {
			osso_fm_free_string_slist(sourceselection);
			g_slist_free(sourceselection);
			sourceselection = NULL;
		}
		if (targetpath != NULL) {
			gtk_file_path_free(targetpath);
			targetpath = NULL;
		}
		return FALSE;
		break;
	case STTS_OK:
		break;
	}
	if (match_parents_path(sourceselection, targetpath)) {
		if (cut_operation == TRUE) {

			if(menucheck == FALSE)
			{
				fm_infoprint(GTK_WINDOW(ofm->window),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"ckdg_ib_unable_to_paste_to_same_location"));					     
			}
			if (sourceselection != NULL) {
				osso_fm_free_string_slist(sourceselection);
				g_slist_free(sourceselection);
				sourceselection = NULL;
			}
			if (targetpath != NULL) {
				gtk_file_path_free(targetpath);
				targetpath = NULL;
			}
			return FALSE;
		} else {
			autorenameflag = TRUE;
		}
	}
	if (menucheck) {
		if (sourceselection != NULL) {
			osso_fm_free_string_slist(sourceselection);
			g_slist_free(sourceselection);
			sourceselection = NULL;
		}
		if (targetpath != NULL) {
			gtk_file_path_free(targetpath);
			targetpath = NULL;
		}
		DEBUG_PRINT_END;
		return TRUE;
	}
	make_uri_list(sourceselection, &sourcelist,
		      targetpath, &targetlist, fsmodel, autorenameflag);

	if ((sourcelist != NULL) && (targetlist != NULL)) {
		if (cut_operation == TRUE) {
			xfer_uri(sourcelist, targetlist, ofm->window,
				 XFER_CUTPASTE, 0, 0, FALSE);
		} else {
			xfer_uri(sourcelist, targetlist, ofm->window,
				 XFER_COPY, 0, 0, FALSE);
		}
		hildon_file_selection_undim_all(global_ofm->fs);
		retval = TRUE;
	}

	if (sourceselection != NULL) {
		osso_fm_free_string_slist(sourceselection);
		g_slist_free(sourceselection);
		sourceselection = NULL;
	}
	if (targetpath != NULL) {
		gtk_file_path_free(targetpath);
		targetpath = NULL;
	}
	DEBUG_PRINT_END;
	return retval;
}

#if 0
/**
 * on_edit_menu_select_items:
 *
 * Method to select unselect folder items in content pane.
 *
 * @param menuitem GtkMenuItem from which the event is triggered
 * @param filesection Pointer to HildonFileSelection
 *
 * @return void
 **/
void on_edit_menu_select_items(GtkMenuItem * menuitem, gpointer fileselection)
{
	DEBUG_PRINT_START;
	if (on_edit_menu_select_activate(fileselection, FALSE) == TRUE) {
		refresh_edit_menu_select_items((GtkWidget *) menuitem,global_ofm);
	}
	DEBUG_PRINT_END;
	return;
}


/**
 * on_edit_menu_select_activate:
 *
 * Method to validate the selection menu item
 *
 * @param fileselection Pointer to HIldonFileSelection
 * @param menucheck TRUE to execute function for vlidating menu items
 * 
 * @ return TRUE if operation suceeds else FALSE;
 **/
gboolean on_edit_menu_select_activate(HildonFileSelection * fileselection, gboolean menucheck)
{
	GSList *selection = NULL;
	GnomeVFSDirectoryHandle *handle = NULL;
	GnomeVFSFileInfo *info = NULL;
	GtkFilePath *curpath = NULL;
	gchar *curpath_texturi = NULL;
	gboolean items_exist = FALSE;

	DEBUG_PRINT_START;
	if (fileselection == NULL) {
		FM_ULOG_ERR("Error validating input params\n");
		return FALSE;
	}

	curpath = hildon_file_selection_get_current_folder(fileselection);
	if(curpath == NULL)
	{
		FM_ULOG_ERR("on_edit_menu_select_activate: curpath == NULL\n");
		return FALSE;
	}
	curpath_texturi = osso_fm_path_to_texturi(curpath);
	gtk_file_path_free(curpath);

	gnome_vfs_directory_open(&handle, curpath_texturi, GNOME_VFS_FILE_INFO_DEFAULT);
	g_free(curpath_texturi);
	if (handle == NULL) {
		FM_ULOG_ERR("Unable to open dir \n");
		return FALSE;
	}
	info = gnome_vfs_file_info_new();
	if (info == NULL) {
		FM_ULOG_ERR("on_edit_menu_select_activate:Error allocating memory\n");
		return FALSE;
	}
	while (gnome_vfs_directory_read_next(handle, info) == GNOME_VFS_OK) {
		if (info->name != NULL) {
			if ((info->name[0] == '.' && info->name[1] == '\0') ||
			    (info->name[0] == '.' && info->name[1] == '.' && info->name[2] == '\0'))
			{
				continue;
			}
			items_exist = TRUE;
			break;
		}
	}
	gnome_vfs_file_info_unref(info);
	gnome_vfs_directory_close(handle);
	selection = hildon_file_selection_get_selected_paths(fileselection);

	if (selection == NULL) {
		if (menucheck == FALSE) {
			fm_infoprint(GTK_WINDOW(global_ofm->window),
				     dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE, "ckct_ib_nothing_to_select"));
		}
		if (items_exist) {
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(global_ofm->window),
					     dgettext(HILDON_FM_L10N_PACKAGE, "sfil_ib_cannot_select_items"));
			}
		}
		DEBUG_PRINT_END;
		if (!menucheck)
		  return FALSE;
	} else {
		gtk_file_paths_free(selection);
		DEBUG_PRINT_END;
		return TRUE;
	}
	DEBUG_PRINT_END;
	return TRUE;
}
#endif

/**
 * on_mark_all_activate:
 *
 * Function Marks all the selected file or olders on the list pane
 *
 * @param fileselection Pointer to HildonFileSelection widget
 * @param menucheck gboolean representing menu checking
 *
 * @return gboolean.
 **/
gboolean on_mark_all_activate(gpointer fileselection, gboolean menucheck)
{
	gboolean retval = FALSE;
	gint num_items_dimmed = 0;
	GSList *list = NULL, *cursel = NULL;

	int count = 0;
	int physical_count = 0;
	DEBUG_PRINT_START;

	if (fileselection == NULL) {
		FM_ULOG_ERR("Invalid parameter : fileselection (null)");
		return FALSE;
	}

	retval = TRUE;
	list = hildon_file_selection_get_selected_paths((HildonFileSelection *) fileselection);

	for (cursel = list; cursel != NULL; cursel = g_slist_next(cursel)) {
		FM_ULOG_DEBUG1("PATH %s\n", osso_fm_path_to_texturi( cursel->data));
		count++;
	}
	gtk_file_paths_free(list);
	physical_count = get_count_of_listed_items(&num_items_dimmed);

	if (num_items_dimmed != 0) {
		if (menucheck)
			return FALSE;

		if (num_items_dimmed == 1)
			fm_infoprint(GTK_WINDOW(global_ofm->window), 
							_("sfil_ib_unable_to_select_item"));
		else
			fm_infoprint(GTK_WINDOW(global_ofm->window), 
							_("sfil_ib_unable_to_select_some_items"));

		return FALSE;
	}

	if (count == 0 && physical_count == 0) {
		if (menucheck == FALSE) {
			fm_infoprint(GTK_WINDOW(global_ofm->window),
				     dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					     "ckct_ib_nothing_to_select"));
		}
		retval = FALSE;
	} else if (count == physical_count) {
		if (menucheck == FALSE) {
			fm_infoprint(GTK_WINDOW(global_ofm->window),
				     dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE, "cdkg_ib_all_items_marked"));
		}
		retval = FALSE;
	} else {
		if (menucheck == FALSE) {
			hildon_file_selection_select_all((HildonFileSelection *) fileselection);
			global_ofm->si->selected_all = TRUE;
			global_ofm->si->just_selected_all = TRUE;

			list = hildon_file_selection_get_selected_paths((HildonFileSelection *)
									fileselection);
			count = 0;
			for (cursel = list; cursel != NULL; cursel = g_slist_next(cursel)) {
				FM_ULOG_DEBUG1("PATH %s\n", osso_fm_path_to_texturi( cursel->data));
				count++;
			}
			gtk_file_paths_free(list);

			if (count == 0) {
				fm_infoprint(GTK_WINDOW(global_ofm->window),
					     dgettext(HILDON_FM_L10N_PACKAGE, "sfil_ib_cannot_select_items"));
			}
		}
		retval = TRUE;
	}

	DEBUG_PRINT_END;
	return retval;
}



/**
 * on_menu_mark_all_activate:
 *
 * Function marks all the selected file or olders on the list pane
 *
 * @param menuitem GtkMenuItem from which the event is triggered
 * @param fileselection Pointer to HildonFileSelection widget
 *
 * @return void.
 **/
void on_menu_mark_all_activate(GtkMenuItem * menuitem, gpointer fileselection)
{
	DEBUG_PRINT_START;

	if (fileselection == NULL) {
		FM_ULOG_ERR("Invalid parameter : fileselection (null)");
		return;
	}
	//g_signal_disconnect(G_OBJECT(fileselection), "selection-changed",
	//              G_CALLBACK (on_selection_changed), NULL);

	on_mark_all_activate(fileselection, FALSE);

	//g_signal_connect(G_OBJECT(fileselection), "selection-changed",
	//                G_CALLBACK (on_selection_changed), NULL);

	DEBUG_PRINT_END;
	return;
}


/**
 * on_moveto_activate:
 * 
 * Function creates move to dialog
 *
 * @param data pointer to app
 * @return void
 **/
void on_moveto_activate(gpointer data)
{
	DEBUG_PRINT_START;
	if(data == NULL)
	{
		FM_ULOG_ERR("Err:on_moveto_activate:data = NULL\n");
		return;
	}
	osso_fm_moveto(data,FALSE);
	DEBUG_PRINT_END;
}

/**
 * on_menu_move_to_activate:
 * 
 * Function creates move to dialog on clicking the move to menu button
 *
 * @param menuitem GtkMenuItem from which the event is triggered
 * @param data pointer to app
 * 
 * @return void
 **/
void on_menu_move_to_activate(GtkWidget * menuitem, gpointer data)
{
	DEBUG_PRINT_START;
	osso_fm_set_operation_state(OPER_MOVE);
	global_ofm->operation_started = FALSE;
	on_moveto_activate(data);
	if (!global_ofm->operation_started) {
		osso_fm_set_operation_state(OPER_NONE);
	}
	DEBUG_PRINT_END;
}



/**
 *  osso_fm_moveto:
 *  
 *  For a valid selection creates Moveto dialog  
 *  otherwise pops up an error message window
 *
 *  @param window HildonApp pointer
 *  @param menucheck TRUE to run operation in silent mode. used for validating menu items.
 *
 *  @return TRUE if operation is sussesful
 **/
gboolean osso_fm_moveto (OSSOFileManager *ofm, gboolean menucheck)
{
	gboolean retval = FALSE;
	gboolean readonly_path = FALSE;

	DEBUG_PRINT_START;
	selection_info_update(ofm,!menucheck);

	if(!ofm->si->selection_in_gateway)
	{
		readonly_path = is_path_readonly(ofm->si->current_selection->data,TRUE);
	}
	selection_info_update(ofm,FALSE);
	if(check_for_media_items(OPER_MOVE,ofm,menucheck))
	{
		return FALSE;
	}
	switch(selection_info_get_selection_type(ofm->si))
	{
		case SIT_SINGLE_SAFE_FOLDER_SELECTED:
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window),
							dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						      "sfil_ib_unable_to_move_selected_folder"));
			}
			retval = FALSE;
			break;
		case SIT_MULTIPLE_SAFE_FOLDERS_SELECTED:
			
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window),
//						     osso_common_error
//						     (CERM_CAN_NOT_MOVE_SELECTED_ITEMS));
						dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ib_unable_to_move_selected_items"));						     
			}
			retval = FALSE;
			break;
		case SIT_DEVICE_SELECTED:
			if(ofm->si->selection_in_gateway)
			{
				retval=FALSE;
				if (!menucheck) {
					if(is_path_virtual(ofm->si->current_selection->data))
					{
					 	fm_infoprint(GTK_WINDOW(ofm->window), _("sfil_ib_unable_move_bluetooth"));
					}
					else {
						fm_infoprint(GTK_WINDOW(ofm->window),
								_("sfil_ib_cannot_move_device"));
					}
				}
				break;
			}
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window), 
//						osso_common_error(CERM_CAN_NOT_MOVEICE));
						_("sfil_ib_cannot_move_device"));						
			}
			break;
		case SIT_MMC_SELECTED:
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window), 
//						osso_common_error(CERM_CAN_NOT_MOVE_MMC));
						_("sfil_ib_cannot_move_mmc"));						
			}
			retval = FALSE;
			break;
		case SIT_SINGLE_FILE_SELECTED:
			if(readonly_path && !ofm->si->selection_in_gateway)
			{
				if (menucheck == FALSE) {
					fm_infoprint(GTK_WINDOW(ofm->window),
					     dgettext(HILDON_FM_L10N_PACKAGE,"sfil_ib_cannot_move_file"));
				}
			} else {
				if (menucheck == FALSE) {
					create_move_to_dialog((GtkWindow *)ofm->window, ofm->fs);
				}
				retval = TRUE;
			}
			break;
		case SIT_SINGLE_FOLDER_SELECTED:
			if(readonly_path && !ofm->si->selection_in_gateway)
			{
				if (menucheck == FALSE) {
					fm_infoprint(GTK_WINDOW(ofm->window),
							dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						     "sfil_ib_unable_to_move_selected_folder"));
				}
			} else {
				if (menucheck == FALSE) {
					create_move_to_dialog((GtkWindow *)ofm->window, ofm->fs);
				}
				retval = TRUE;
			}
			break;
		case SIT_MULTIPLE_FILES_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
			if(readonly_path && !ofm->si->selection_in_gateway)
			{
				if (menucheck == FALSE) {
					fm_infoprint(GTK_WINDOW(ofm->window),
//					     osso_common_error(CERM_CAN_NOT_MOVE_SELECTED_ITEMS));
						dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"sfil_ib_unable_to_move_selected_items"));					     
				}
			} else {
				if (menucheck == FALSE) {
					create_move_to_dialog((GtkWindow *)ofm->window, ofm->fs);
				}
				retval = TRUE;
			}
			break;
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			retval = FALSE;
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			retval = FALSE;
			break;
	}
	DEBUG_PRINT_END;
	return retval;
}
/**
 * on_menu_duplicate_activate:
 * Function duplicates the selected files and folders
 *
 * @param menuitem GtkMenuItem from which the event is triggered
 * @param window HildonWindow pointer
 *
 * @return void
 **/
void on_menu_duplicate_activate(GtkMenuItem * menuitem, gpointer data)
{
	DEBUG_PRINT_START;

	if (data == NULL) {
		FM_ULOG_ERR("on_menu_duplicate_activate:Err: data = NULL\n");
		return;
	}
	osso_fm_set_operation_state(OPER_DUPLICATE);
	global_ofm->operation_started = FALSE;
	on_duplicate_activate(data, FALSE);
	if (!global_ofm->operation_started) {
		osso_fm_set_operation_state(OPER_NONE);
	}

	DEBUG_PRINT_END;
}

/**
 * osso_fm_duplicate:
 *
 * Function duplicates the selected file/folder
 *
 * @param ofm pointer to the OSSOFileManager
 *
 * returns true if success else false in case of error
 **/
gboolean osso_fm_duplicate(OSSOFileManager *ofm)
{
	GtkFilePath *path = NULL, *parent_path = NULL;
	HildonFileSystemModelItemType type;
	HildonFileSystemModel *fsmodel = NULL;
	GList *sourcelist = NULL;
	GList *targetlist = NULL;

	if(!ofm->si->selection_in_gateway)
	{
		gboolean readonly_path = FALSE;
		readonly_path = is_path_readonly(ofm->si->current_selection->data,TRUE);
		if (readonly_path)
		  return FALSE;
	}
	g_object_get(G_OBJECT(ofm->fs), "model", &fsmodel, NULL);
	if (fsmodel == NULL) {
		FM_ULOG_ERR("Error getting model\n");
		return FALSE;
	}
	path = ofm->si->current_selection->data;
	type = ((OSSOFMFileInfo *)ofm->si->current_selection_withinfo->data)->type;

	parent_path = osso_fm_get_parent(path);
	if (parent_path == NULL) {
		FM_ULOG_ERR("Unable to get parent path \n");
		return FALSE;
	}
	make_uri_list(ofm->si->current_selection, &sourcelist,
			      parent_path, &targetlist,
			      fsmodel, TRUE);
	if ((sourcelist != NULL) || (targetlist != NULL)) {
		/* Move the selected fils and folers to the target folder */
		xfer_uri(sourcelist, targetlist, (GtkWidget *)ofm->window, XFER_DUPLICATE, 0, 0,FALSE);
	}
	gtk_file_path_free(parent_path);
	return TRUE;
}
/**
 * on_duplicate_activate:
 * Duplicates the selected files and folders
 *
 * @param menuitem GtkMenuItem from which the event is triggered
 * @param window HildonApp pointer
 *
 * @return void
 **/
gboolean on_duplicate_activate(OSSOFileManager *ofm, gboolean menucheck)
{
	HildonFileSystemModel *fsmodel = NULL;
	gboolean retval = FALSE;
	gboolean readonly_path = FALSE;
	SelectionInfoType sel_type;

	if(!ofm->si->selection_in_gateway)
	{
		readonly_path = is_path_readonly(ofm->si->current_selection->data,TRUE);
	}
	DEBUG_PRINT_START;

	g_object_get(G_OBJECT(ofm->fs), "model", &fsmodel, NULL);
	if (fsmodel == NULL) {
		if (menucheck == FALSE) {
			FM_ULOG_ERR("Error getting model\n");
		}
		return FALSE;
	}

	selection_info_update(ofm,FALSE);
	if(ofm->si->selection_in_media_server &&
			selection_info_get_selection_type(ofm->si) != SIT_DEVICE_SELECTED)
	{
		if(menucheck == FALSE)
		{
				fm_infonote(GTK_WINDOW(ofm->window),
						_("sfil_ni_duplicate_not_supported_bluetooth"), 
						NULL);
		}
		return FALSE;
	}
		
	if(ofm->si->selection_in_gateway && 
			selection_info_get_selection_type(ofm->si) != SIT_DEVICE_SELECTED )
	{
		if(menucheck == FALSE)
		{
				fm_infonote(GTK_WINDOW(ofm->window),
						_("sfil_ni_duplicate_not_supported_bluetooth"), 
						NULL);
		}
		return FALSE;
	}
	sel_type = selection_info_get_selection_type(ofm->si);
	switch(sel_type)
	{
		case SIT_MULTIPLE_FILES_SELECTED:
		case SIT_MULTIPLE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_SAFE_FOLDERS_SELECTED:
		case SIT_MULTIPLE_ITEMS_SELECTED:
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window),
//						osso_common_error(CERM_SELECT_ONE_ITEM_ONLY));
						dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
								"ckct_ib_select_single_item"));					     
			}
			retval = FALSE;
			break;
		case SIT_SINGLE_FILE_SELECTED:
		case SIT_SINGLE_FOLDER_SELECTED:
		case SIT_SINGLE_SAFE_FOLDER_SELECTED:
			retval = !(readonly_path || !is_path_readable (ofm->si->current_selection->data));
			if(menucheck)
				break;
			if(readonly_path || !is_path_readable(ofm->si->current_selection->data))
			{
				if(sel_type == SIT_SINGLE_FILE_SELECTED) {
					fm_infoprint(GTK_WINDOW(ofm->window),
//						osso_common_error(CERM_CAN_NOT_DUP_FILE));
						_("sfil_ib_cannot_duplicate_file"));						
				} else {
					fm_infoprint(GTK_WINDOW(ofm->window),
//						osso_common_error(CERM_CAN_NOT_DUP_FOLDER));
						_("sfil_ib_cannot_duplicate_folder"));						
				}
				
			} else {
				retval = osso_fm_duplicate(ofm);
			}
			break;

		case SIT_DEVICE_SELECTED:
			if(ofm->si->selection_in_gateway)
			{
				retval=FALSE;
				if (!menucheck) {
					if(is_path_virtual(ofm->si->current_selection->data)) {
						fm_infoprint(GTK_WINDOW(ofm->window), _("sfil_ib_unable_duplicate_bluetooth"));
					} else {
						fm_infoprint(GTK_WINDOW(ofm->window), 
							_("sfil_ib_cannot_duplicate_device"));						
					}
				}
				break;
			}
			if(ofm->si->selection_in_media_server) 
			{
				retval=FALSE;
				if (!menucheck) {
					if(is_path_virtual(ofm->si->current_selection->data)) {
						fm_infoprint(GTK_WINDOW(ofm->window),
							_("sfil_ib_unable_duplicate_shared_media"));	
					} else {
						fm_infoprint(GTK_WINDOW(ofm->window),
								 _("sfil_ib_cannot_duplicate_device"));
					}					
				}
				break;
			}
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window), 
						_("sfil_ib_cannot_duplicate_device"));						
			}
			retval = FALSE;
			break;
		case SIT_MMC_SELECTED:
			if (menucheck == FALSE) {
				fm_infoprint(GTK_WINDOW(ofm->window), 
//						osso_common_error(CERM_CAN_NOT_DUP_MMC));
						_("sfil_ib_cannot_duplicate_mmc"));						
			}
			retval = FALSE;
			break;
		case SIT_NO_ITEM_SELECTED:
			FM_ULOG_ERR("NO ITEMS SELECTED\n");
			retval = FALSE;
			break;
		case SIT_ERROR_IN_GETTING_INFO:
			FM_ULOG_ERR("SIT_ERROR_IN_GETTING_INFO\n");
			retval = FALSE;
			break;
	}
	
	DEBUG_PRINT_END;
	return retval;
}
/**
 * osso_fm_save_path_preferences:
 *
 * Funtion will save the current path preferences for future reference
 *
 * @param fileselection HildonFileSelection 
 * @return void
 **/
void osso_fm_save_path_preferences(HildonFileSelection * fileselection)
{
	GtkFilePath *path = NULL;
	gboolean thumbnails = FALSE;
	HildonFileSelectionSortKey key = 0;
	GtkSortType order = 0;

	DEBUG_PRINT_START;
	if (fileselection == NULL) {

		FM_ULOG_ERR("Error validation input params\n");
		return;
	}
	path = hildon_file_selection_get_current_folder(fileselection);
	if (path == NULL) {
		FM_ULOG_ERR("Error retriving path\n");
		return;
	}
	thumbnails = hildon_file_selection_get_mode(fileselection);
	hildon_file_selection_get_sort_key(fileselection, &key, &order);
	osso_fm_set_config_value_for_path(path, thumbnails, key, order);
	gtk_file_path_free(path);
	DEBUG_PRINT_END;
	return;
}
/**
 * osso_fm_set_fs_view_mode:
 * Method to set the mode for file selection widget
 *
 * @param ofm pointer to the OSSOFileManager
 * @param mode HildonFileSelectionMode
 *
 * @return void
 **/
void osso_fm_set_fs_view_mode(OSSOFileManager* ofm, HildonFileSelectionMode mode)
{
	DEBUG_PRINT_START;
	
	g_signal_handlers_block_by_func(G_OBJECT(ofm->fmtoolbar->thumbview_button),
					G_CALLBACK(on_toolbar_thumbview_clicked), ofm);
	g_signal_handlers_block_by_func(G_OBJECT(ofm->fmtoolbar->listview_button), 
					G_CALLBACK(on_toolbar_listview_clicked), ofm);
	
	hildon_file_selection_set_mode(ofm->fs, mode);
	if (mode == HILDON_FILE_SELECTION_MODE_THUMBNAILS) {
		gtk_toggle_tool_button_set_active((GtkToggleToolButton *)
						ofm->fmtoolbar->listview_button, FALSE);
		gtk_toggle_tool_button_set_active((GtkToggleToolButton *)
						ofm->fmtoolbar->thumbview_button, TRUE);
	} else {
		gtk_toggle_tool_button_set_active((GtkToggleToolButton *)
						ofm->fmtoolbar->listview_button, TRUE);
		gtk_toggle_tool_button_set_active((GtkToggleToolButton *)
						ofm->fmtoolbar->thumbview_button, FALSE);
	}

	g_signal_handlers_unblock_by_func(G_OBJECT(ofm->fmtoolbar->thumbview_button),
					G_CALLBACK(on_toolbar_thumbview_clicked), ofm);
	g_signal_handlers_unblock_by_func(G_OBJECT(ofm->fmtoolbar->listview_button), 
					G_CALLBACK(on_toolbar_listview_clicked), ofm);

	osso_fm_save_path_preferences(ofm->fs);
	DEBUG_PRINT_END;
	return;
}


/**
 * on_listview_activate:
 * 
 * Function Changes the contents of the current folder to be shown in list view.
 *
 * @param data pointer to the OSSOFileManager
 * @return void
 **/
void on_listview_activate(gpointer data)
{
	DEBUG_PRINT_START;
	if (data == NULL) {
		FM_ULOG_ERR("Err:on_listview_activate:data = NULL\n");
		return;
	}
	osso_fm_set_fs_view_mode(data, HILDON_FILE_SELECTION_MODE_LIST);
	DEBUG_PRINT_END;
	return;
}

/**
 * on_thumbview_activate: 
 *
 * Function changes the contents of the current folder to be shown in thumbnail view.
 *
 * @param data pointer to the OSSOFileManager
 * @return void
 **/
void on_thumbview_activate(gpointer data)
{
	DEBUG_PRINT_START;
	if (data == NULL) {
		FM_ULOG_ERR("Err:on_thumbview_activate:data = NULL\n");
		return;
	}
	osso_fm_set_fs_view_mode(data, HILDON_FILE_SELECTION_MODE_THUMBNAILS);
	DEBUG_PRINT_END;
	return;
}

/**
 * on_listview_activate:
 * 
 * Function Changes the contents of the current folder to be shown in list view.
 *
 * @param menuitem pointer to the GtkMenuItem
 * @param data pointer to the OSSOFileManager
 * @return void
 **/
void on_menu_view_list_activate(GtkMenuItem * menuitem, gpointer data)
{
	DEBUG_PRINT_START;
	on_listview_activate(data);
	DEBUG_PRINT_END;
}

/**
 * on_menu_view_thumbnail_activate:
 * Function changes the contents of the current folder to be shown in thumbnail view.
 *
 * @param menuitem pointer to the GtkMenuItem
 * @param data pointer to the OSSOFileManager
 * @return void
 **/
void on_menu_view_thumbnail_activate(GtkMenuItem * menuitem, gpointer data)
{
	DEBUG_PRINT_START;
	on_thumbview_activate(data);
	DEBUG_PRINT_END;
	return;
}

/**
 * on_sort_activate:
 *
 * Function Sorts the item
 *
 * @param menuitem GtkMenuItem from which the event is triggered
 * @param fileselection Pointer to HildonFileSelection widget
 * @param menucheck boolean representing menucheck status
 *
 * @return void
 **/
gboolean on_sort_activate(gpointer fileselection, gboolean menucheck)
{
	GSList *list = NULL;
	gboolean retval = FALSE;
	DEBUG_PRINT_START;

	if (fileselection == NULL) {
		FM_ULOG_ERR("Invalid parameter : fileselection (null)\n");
		return FALSE;
	}

	list = hildon_file_selection_get_selected_paths((HildonFileSelection *) fileselection);
	if (list == NULL) {
		if (menucheck == FALSE) {
			FM_ULOG_DEBUG1("\n\nSELECTION IS NULL \n\n");
			fm_infoprint(GTK_WINDOW(global_ofm->window)
				     , dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE, "ckdg_ib_nothing_to_sort"));
		}
		retval = FALSE;
	} else {
		if (menucheck == FALSE) {
			sort_dialog((GtkWindow *) global_ofm->window);
		}
		retval = TRUE;
		gtk_file_paths_free(list);
	}

	DEBUG_PRINT_END;
	return retval;
}

/**
 * on_menu_sort_activate:
 * 
 * Function Sorts the item on clicking the sort menu item
 * @param menuitem GtkMenuItem from which the event is triggered
 * @param fileselection Pointer to HildonFileSelection widget
 *
 * @return void
 **/
void on_menu_sort_activate(GtkMenuItem * menuitem, gpointer fileselection)
{
	DEBUG_PRINT_START;

	if (fileselection == NULL) {
		FM_ULOG_ERR("Invalid parameter : fileselection (null)\n");
		return;
	}
	osso_fm_set_operation_state(OPER_SORT);

	on_sort_activate(fileselection, FALSE);

	osso_fm_set_operation_state(OPER_NONE);

	DEBUG_PRINT_END;
	return;
}

/**
 * osso_fm_sort:
 * 
 * Method to set sort for file selection widget
 * @param fs HildonFileSelectionWidget
 * @param key Represents the column by which sort has to be performed
 * @param order sorting order ascending or descending
 *
 * @return void
 **/
void osso_fm_sort(HildonFileSelection * fs, HildonFileSelectionSortKey key, GtkSortType order)
{
	DEBUG_PRINT_START;
	if (fs == NULL) {
		FM_ULOG_ERR("Error validating input parameters\n");
		return;
	}

	FM_ULOG_DEBUG1("SORT KEY : %d,  SORT_ORDER : %d\n", key, order);
	hildon_file_selection_set_sort_key(fs, key, order);
	osso_fm_save_path_preferences(fs);
	DEBUG_PRINT_END;
	return;

}

/**
 * osso_fm_sort_by_name:
 * Method to sort by name
 *
 * @param widget Source widget from which the event is raised. 
 * @param fs HildonFileSelection widget
 *
 * @return void
 **/
void osso_fm_sort_by_name(GtkWidget * widget, HildonFileSelection * fs)
{
	osso_fm_sort(fs, HILDON_FILE_SELECTION_SORT_NAME, GTK_SORT_ASCENDING);
}


/**
* Method to sort by type
*
* @param widget Source widget from which the event is raised. 
* @param fs HildonFileSelection widget
*/
void osso_fm_sort_by_type(GtkWidget * widget, HildonFileSelection * fs)
{
	osso_fm_sort(fs, HILDON_FILE_SELECTION_SORT_TYPE, GTK_SORT_ASCENDING);
}


/**
 * osso_fm_sort_by_modified:
 * Method to sort by modified time
 *
 * @param widget Source widget from which the event is raised. 
 * @param fs HildonFileSelection widget
 * @return void
 **/
void osso_fm_sort_by_modified(GtkWidget * widget, HildonFileSelection * fs)
{
	osso_fm_sort(fs, HILDON_FILE_SELECTION_SORT_MODIFIED, GTK_SORT_ASCENDING);
}


/**
 * osso_fm_sort_by_size:
 * Method to sort by size
 *
 * @param widget Source widget from which the event is raised. 
 * @param fs HildonFileSelection widget
 * @return void
 **/
void osso_fm_sort_by_size(GtkWidget * widget, HildonFileSelection * fs)
{
	osso_fm_sort(fs, HILDON_FILE_SELECTION_SORT_SIZE, GTK_SORT_ASCENDING);
}


/**
 * backup_signal_cb:
 * Function does the backup of selected files 
 * @param connection pointer to the DBusConnection
 * @param user_data
 * @return DBusHandlerResult
 **/
DBusHandlerResult
backup_signal_cb(DBusConnection * connection, DBusMessage * message, void *user_data)
{
	if (dbus_message_is_signal(message, "com.nokia.backup", "backup_start")) {
		backup_started = 2;
		FM_ULOG_INFO("Backup START signal received\n");
	} else if (dbus_message_is_signal(message, "com.nokia.backup", "backup_finish")) {
		backup_started = 0;
		FM_ULOG_INFO("Backup STOP signal received\n");
	}
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

//BEG: MMC RELATED FUNCTIONS

/**
 * osso_fm_mmc_format:
 * Function formats mmc
 * @param menuitem pointer to the GtkMenuItem
 * @param user_data pointer to the app
 * @return void
 **/
void osso_fm_mmc_format (const char *device)
{
	osso_return_t retval = OSSO_ERROR;
	osso_rpc_t ret = { 0 };
//	GtkWidget *infnote;
	GtkWidget *confirmnote = NULL;

	DEBUG_PRINT_START;
	osso_fm_set_operation_state(OPER_MMCFORMAT);
	confirmnote =
	    hildon_note_new_confirmation_add_buttons(GTK_WINDOW(global_ofm->window),
						     (const gchar *) _("hfil_nc_format_mmc"),
						     _("hfil_bd_format_mmc_ok"), GTK_RESPONSE_OK,
						     _("hfil_bd_format_mmc_cancel"),
						     GTK_RESPONSE_CANCEL, NULL);
	if (gtk_dialog_run(GTK_DIALOG(confirmnote)) != GTK_RESPONSE_OK) {
		gtk_widget_destroy(confirmnote);
		return;
		g_print("got ok in for mmc format: sending message to ke-recv\n");
	}
	gtk_widget_destroy(confirmnote);
	retval = osso_rpc_run (fm_app_ctx,
			       MMC_FORMAT_SERVICE,
			       MMC_FORMAT_OBJECT_PATH,
			       MMC_FORMAT_SERVICE,
			       MMC_FORMAT_METHOD, 
			       &ret,
			       DBUS_TYPE_STRING, device,
			       DBUS_TYPE_STRING, "", /* label */
			       DBUS_TYPE_INVALID);

	g_print("Entering switch (formatting) \n");
	switch (retval) {
	case OSSO_OK:
		//success
		break;
	case OSSO_INVALID:
	case OSSO_ERROR_NAME:
		FM_ULOG_ERR("Error Invalid parameters\n");
		break;
	case OSSO_RPC_ERROR:
		FM_ULOG_ERR("Error: %s\n", ret.value.s);
		g_print("Error: %s\n", ret.value.s);
#if 0
		if (g_strcasecmp(ret.value.s, "canceled") == 0) {
			infnote =
			    hildon_note_new_information(GTK_WINDOW(global_ofm->app),
							_("hfil_nc_cancel_format_mmc"));
			hildon_note_set_button_text((HildonNote *) infnote,
						    _("hfil_bd_cancel_format_mmc_ok"));
			gtk_widget_show(infnote);
			gtk_dialog_run((GtkDialog *) infnote);
			gtk_widget_destroy(infnote);
		}
#endif
		if (g_strcasecmp(ret.value.s, "in_use") == 0) {
			fm_infonote(GTK_WINDOW(global_ofm->window),
					_("sfil_ni_mmc_format_mmc_in_use"),
					NULL);
		}
		break;
		//TODO check for the 'canceled' string and display the infoprint(format cancelled)
	case OSSO_ERROR:
	case OSSO_ERROR_NO_STATE:
	case OSSO_ERROR_STATE_SIZE:
		FM_ULOG_ERR("Error: %s\n", ret.value.s);
		g_print("Error: %s\n", ret.value.s);

		break;
	}

	g_print("Done with formatting \n");

	osso_fm_set_operation_state(OPER_NONE);
	DEBUG_PRINT_END;
	return;
}

/**
 * response_handler:
 * helper function - Small help utilities for response_handler
 * @param widget pointer to the GtkWidget
 * @param arg1 int value
 * @param data
 * @return void
 **/
static void response_handler(GtkWidget * widget, gint arg1, gpointer data)
{
	//gchar *entry_text = NULL;
	gchar *texturi = NULL;
	switch (arg1) {
		case GTK_RESPONSE_OK:
			texturi = gtk_file_chooser_get_uri((GtkFileChooser *) widget);
			FM_ULOG_INFO("RENMMC1 %s",texturi);
			//entry_text = osso_fm_texturi_get_shortname(texturi);
			//entry_text = g_strdup(gtk_entry_get_text(GTK_ENTRY(priv->entry_name)));
			if(texturi == NULL)	
			{
				g_signal_stop_emission_by_name(widget, "response");
				fm_infoprint((GtkWindow *)global_ofm->window, 
						dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
							"ckdg_ib_enter_name"));
				g_signal_emit_by_name(widget, "show");
			}
			else if(strlen(texturi) <= strlen("file:///home/user/MyDocs/"))
			{
				g_signal_stop_emission_by_name(widget, "response");
				fm_infoprint((GtkWindow *)global_ofm->window, 
						dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
							"ckdg_ib_enter_name"));
				g_signal_emit_by_name(widget, "show");
			}
			break;
			default:
				break;
	}

	g_free (texturi);
}

static char *
osso_fm_get_mmc_rename_string(const char *uri)
{
	char *the_name = NULL;
	GnomeVFSVolumeMonitor *vol_monitor = NULL;

	if ((vol_monitor = gnome_vfs_get_volume_monitor()) != NULL)
	{
		GnomeVFSVolume *vol = NULL;

		if (strlen(uri) > 7)
			if (!strncmp(uri, "file://", 7))
				uri = &uri[7];

		if ((vol = gnome_vfs_volume_monitor_get_volume_for_path(vol_monitor, uri)) != NULL)
		{
			char *vol_name = NULL;

			if ((vol_name = gnome_vfs_volume_get_display_name(vol)) != NULL)
			{
				if (strlen(vol_name) >= 18)
					if (!strncmp(vol_name, "mmc-undefined-name", 18))
						the_name = g_strdup("");
				g_free(vol_name);
			}

			gnome_vfs_volume_unref(vol);
		}
	}

	if (!the_name)
		the_name = osso_fm_texturi_get_display_name(uri);

	g_debug("osso_fm_get_mmc_name: In the end, the mmc name will be \"%s\"\n", the_name);

	return the_name;
}

/**
 * osso_fm_mmc_rename:
 * Function renames mmc
 * @param ofm pointer to the OSSOFileManager
 * @param internal gboolean variable
 *
 * @return void
 **/
void osso_fm_mmc_rename(OSSOFileManager* ofm,
			const char *uri, const char *device)
{
	GtkWidget *renamedialog = NULL;
	/* GtkTreeModel has to used as in osso_fm_rename.c, hence commented */
	HildonFileSystemModel *fsmodel = NULL;
	gint response = 0;

	osso_return_t retval = OSSO_ERROR;
	gchar *inputname_path = NULL;
	gchar *inputname = NULL;
	osso_rpc_t ret = { 0 };
	gboolean dialog_return = FALSE;
	gchar *currentname = NULL;
	char *mount_point;
	gint dialog_width;

	gchar illegals[10];

	DEBUG_PRINT_START;

	g_object_get(G_OBJECT(ofm->fs), "model", &fsmodel, NULL);

	osso_fm_set_operation_state(OPER_MMCRENAME);
	if (fsmodel != NULL) {
		renamedialog = hildon_file_chooser_dialog_new_with_properties(
					GTK_WINDOW(ofm->window),
					"action",
					GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER,
					"file-system-model",
					fsmodel,
					"select-multiple",
					FALSE,
					"show-folder-button",
					FALSE,
					"show-location",
					FALSE, "autonaming",
					FALSE,
					"max-name-length",
					MAX_MMC_NAME_LENGTH,
					NULL);
	} else {
		renamedialog = hildon_file_chooser_dialog_new_with_properties(
					GTK_WINDOW(ofm->window),
					"action",
					GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER,
					"select-multiple",
					FALSE,
					"show-folder-button",
					FALSE,
					"show-location",
					FALSE, "autonaming",
					FALSE,
					"max-name-length",
					MAX_MMC_NAME_LENGTH,
					NULL);
	}
	guint signal_id = 0;
	gulong handler_id = 0;
	signal_id  = g_signal_lookup("response", GTK_TYPE_DIALOG);
	handler_id = g_signal_handler_find(renamedialog, G_SIGNAL_MATCH_ID,
			signal_id, 0, NULL, NULL, NULL);
	g_signal_handler_block(renamedialog, handler_id);
	g_signal_connect(renamedialog, "response", G_CALLBACK(response_handler), NULL);
	gtk_window_set_title(GTK_WINDOW(renamedialog), _("sfil_ti_rename_mmc"));
	gtk_window_set_modal(GTK_WINDOW(renamedialog), TRUE);
	currentname = osso_fm_get_mmc_rename_string(uri);
	gtk_file_chooser_set_current_folder_uri(GTK_FILE_CHOOSER(renamedialog),
			"file:///home/user/MyDocs/");
	if(strlen(currentname) > MAX_MMC_NAME_LENGTH)
	{
		currentname[MAX_MMC_NAME_LENGTH] = '\0';
	}
	g_strchomp(currentname);
	gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(renamedialog), currentname);
	g_free(currentname);

	dialog_width = osso_fm_predict_dialog_width_based_on_title
	  (renamedialog,
	   _("sfil_ti_rename_mmc"));
	dialog_width = dialog_width - 230; /* minimum size with header
					      still fully visible
					   */

	gtk_window_set_default_size(GTK_WINDOW(renamedialog),
				    dialog_width, -1);
	gtk_widget_show(renamedialog);

	mount_point = gnome_vfs_get_local_path_from_uri(uri);

	while (dialog_return == FALSE) {
		response = gtk_dialog_run(GTK_DIALOG(renamedialog));
		if (response == GTK_RESPONSE_OK) {
			gchar *escaped_name, *str;
			GnomeVFSURI *uri;
			char buf[256];

			str = gtk_file_chooser_get_uri (GTK_FILE_CHOOSER (renamedialog));
			uri = gnome_vfs_uri_new (str);
			g_free (str);

			escaped_name = gnome_vfs_uri_extract_short_path_name (uri);
			inputname = gnome_vfs_unescape_string (escaped_name, NULL);
			g_free (escaped_name);
			gnome_vfs_uri_unref (uri);

			if (mount_point != NULL) {
				strcpy(buf, mount_point);
				strcat(buf, "/");
				strncat(buf, inputname,
					255 - strlen(mount_point));
				if (g_file_test(buf,
					    	G_FILE_TEST_EXISTS)) {
					fm_infoprint(
						GTK_WINDOW(ofm->window),
						_("sfil_ib_card_name_in_use_for_file"));
					g_signal_emit_by_name(
						renamedialog, "show");
					continue;
				}
			}

			g_signal_emit_by_name(renamedialog, "show");
			if (!valid_mmc_name(inputname,
					    illegals,
					    sizeof(illegals))) {
				gchar *msg =
					g_strdup_printf 
					(dgettext (HILDON_COMMON_STRINGS_L10N_PACKAGE,
						   "ckdg_ib_illegal_characters_entered"),
					 illegals);
				fm_infoprint(GTK_WINDOW(ofm->window), msg);
				g_free (msg);
				g_signal_emit_by_name(renamedialog, "show");
			} else {
				gtk_widget_destroy(renamedialog);
				dialog_return = TRUE;
			}
		} else {
			gtk_widget_destroy(renamedialog);
			dialog_return = TRUE;
			osso_fm_set_operation_state(OPER_NONE);
			return;
		}
	}

	g_print("inputname is %s, calling rename on ke-recv\n", inputname);
	//inputname = osso_fm_path_get_shortname(inputname_path);

	retval = osso_rpc_run(fm_app_ctx, 
			      MMC_FORMAT_SERVICE, 
			      MMC_RENAME_OBJECT_PATH, 
			      MMC_FORMAT_SERVICE, 
			      MMC_RENAME_METHOD, 
			      &ret, 
			      DBUS_TYPE_STRING, device,
			      DBUS_TYPE_STRING, inputname,
			      DBUS_TYPE_INVALID);
	
	switch (retval) {
	case OSSO_OK:
		//success
		break;
	case OSSO_INVALID:
	case OSSO_ERROR_NAME:
		FM_ULOG_ERR("Error Invalid parameters\n");
		break;
	case OSSO_ERROR:
	case OSSO_RPC_ERROR:
	case OSSO_ERROR_NO_STATE:
	case OSSO_ERROR_STATE_SIZE:
		FM_ULOG_ERR("Renaming MMC Error: %s\n", ret.value.s);
		g_print("Renaming MMC Error: %s\n", ret.value.s);
		if (g_strcasecmp(ret.value.s, "in_use") == 0) {
			fm_infonote(GTK_WINDOW(global_ofm->window),
					_("sfil_ni_mmc_rename_mmc_in_use"),
					NULL);
		}
		break;
	}
	osso_fm_set_operation_state(OPER_NONE);
	g_free(inputname_path);
	g_free(inputname);
	g_free(mount_point);
	DEBUG_PRINT_END;
	return;
}


/**
 * on_find_activate:
 * Helper function for invoking global search
 * @return void
 */
void on_find_activate()
{
	DEBUG_PRINT_START;
	osso_fm_set_operation_state(OPER_SEARCH);
	osso_fm_invoke_global_search();
	osso_fm_set_operation_state(OPER_NONE);
	DEBUG_PRINT_END;
	return;
}

/**
 * on_menu_tools_find_activate:
 * Callback function for menu find activate
 */
void on_menu_tools_find_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	DEBUG_PRINT_START;
	on_find_activate();
	DEBUG_PRINT_END;
}

//#define HELP_APP "osso_help"
//#define HELP_ACTIVATE "activate"

/**
 * on_menu_tools_help_activate:
 * Help menu item callback function
 * @param menuitem Activated GtkMenuItem
 * @param window user data
 * @return void
 **/
void on_menu_tools_help_activate(GtkMenuItem * menuitem, gpointer window)
{
	osso_return_t retval = OSSO_ERROR;
	DEBUG_PRINT_START;

	osso_fm_set_operation_state(OPER_HELP);

	retval = hildon_help_show(fm_app_ctx,
				  "applications_filemanager_main", 0);

	switch (retval) {
	case OSSO_OK:
		FM_ULOG_DEBUG1("Help invocation succesful\n");
		//success
		break;
	case OSSO_INVALID:
	case OSSO_ERROR_NAME:
		FM_ULOG_ERR("Error Invalid parameters\n");
		//break;
	case OSSO_RPC_ERROR:
	case OSSO_ERROR:
	case OSSO_ERROR_NO_STATE:
	case OSSO_ERROR_STATE_SIZE:
		//  if (window != NULL){
		//    fm_infoprint (GTK_WINDOW(window), 
		//      _("sfil_li_unable_to_open_help_dev"));
		//  }
		FM_ULOG_ERR("Error invoking help application\n");
		break;
	}

	osso_fm_set_operation_state(OPER_NONE);

	DEBUG_PRINT_END;
	return;
}

//#define BACKUP_APP "backup"
//#define BACKUP_METHOD_ACTIVATE      "activate"

/**
 * on_menu_tools_back_up_restore_activate:
 * Backup and Restore menu item callback function
 * @param menuitem Activated GtkMenuItem
 * @param user_data user data
 * @return void
 **/
void on_menu_tools_back_up_restore_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	osso_return_t retval = OSSO_ERROR;
	DEBUG_PRINT_START;

	osso_fm_set_operation_state(OPER_BACKUP);

	retval = osso_rpc_run_with_defaults(fm_app_ctx,
					    BACKUP_APP,
					    BACKUP_METHOD_ACTIVATE, NULL, DBUS_TYPE_INVALID);

	switch (retval) {
	case OSSO_OK:
		//success
		break;
	case OSSO_INVALID:
	case OSSO_ERROR_NAME:
		FM_ULOG_ERR("Error Invalid parameters\n");
		break;
	case OSSO_RPC_ERROR:
	case OSSO_ERROR:
	case OSSO_ERROR_NO_STATE:
	case OSSO_ERROR_STATE_SIZE:
		FM_ULOG_ERR("Error invoking backup application\n");
		break;
	}

	osso_fm_set_operation_state(OPER_NONE);

	DEBUG_PRINT_END;
	return;
}



/**
 * on_menu_close_fm_activate:
 * Close menu item callback function
 * @param menuitem Activated GtkMenuItem
 * @param user_data user data
 * @return void
 **/
void on_menu_close_fm_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	DEBUG_PRINT_START;

	if (user_data == NULL) {
		FM_ULOG_ERR("Invalid parameter : user_data");
		return;
	}

	osso_fm_set_operation_state(OPER_CLOSE);

	fm_application_destroy_cb((GtkWidget *) user_data, NULL, user_data);
	DEBUG_PRINT_END;
	return;
}


/**
 * on_navigation_popup_menu_activate:
 * Tap and hold callback on navigation pane
 * @param view GtkWidget on which tap and hold has been done
 * @param data user data
 * @return void
 **/
void on_navigation_popup_menu_activate(GtkWidget * view, gpointer data)
{
	on_popup_menu_activate(data, FALSE);
}

/**
 * on_content_popup_menu_activate:
 * Tap and hold callback on content pane
 * @param view GtkWidget on which tap and hold has been done
 * @param data user data
 * @return void
 **/
void on_content_popup_menu_activate(GtkWidget * view, gpointer data)
{
	on_popup_menu_activate(data, TRUE);
}

/**
 * on_popup_menu_activate:
 * Helper function to invoke popup menu. This function will be called 
 * when tap and hold happens on any of the panes
 * @param ofm pointer to structure OSSOFileManager
 * @contentpane gboolean to indicate the location of tap and hold. TRUE if content pane
 * @return void
 **/
void on_popup_menu_activate(OSSOFileManager *ofm, gboolean contentpane)
{

	DEBUG_PRINT_START;
	if (ofm->fmpmenu == NULL) {
		FM_ULOG_INFO("Creating POPUP menu\n");
		ofm->fmpmenu = osso_fm_create_popup_menu(ofm);
		
	}
	if (ofm->fmpmenu == NULL) {
		FM_ULOG_ERR("Unable to get popup_menu");
		return;
	}
	validate_selection((HildonFileSelection *) ofm->fs);
	refresh_popup_menu_items((OSSOFileManager *) ofm, contentpane);
	gtk_menu_popup(GTK_MENU(ofm->fmpmenu->menu), NULL, NULL, NULL, NULL,
		       0, gtk_get_current_event_time());
	DEBUG_PRINT_END;
	return;
}


/**
 * on_file_selection_items_dropped:
 * Method called to when user drags and drops the selection
 * 
 * @param fs HildonFileSelection where the selected items are dropped.
 * @param destination GtkFilePath represention the path where the 
 * selected file/folder items are dropped
 * @param sources GSList representing the selection
 * @return void
 **/
void
on_file_selection_items_dropped(HildonFileSelection * fs,
				GtkFilePath * destination, GSList * sources)
{
	SelectionState *sel_state;
	SrcToTgtStatus sttsres;
	gint removed_items = 0;
	gint nonexistant_items = 0;
	GSList *sources_tmp = NULL;
	GList *sourcelist = NULL;
	GList *targetlist = NULL;
	gchar *display_name = NULL;
	GtkWidget *confirmnote = NULL;
	gint response = GTK_RESPONSE_CANCEL;
	gchar *note = NULL;
	gchar *button1_text = NULL;
	gchar *button2_text = NULL;
	HildonFileSystemModel *fsmodel = NULL;

	if ((fs == NULL) || (sources == NULL) || (destination == NULL)) {
		FM_ULOG_ERR("Error validing input parameters");
		return;
	}
	sel_state = get_selection_state_for_selection(sources);
	if (sel_state == NULL) {
		FM_ULOG_ERR("Error getting selection state");
		return;
	}
	if (sel_state->error != ERR_NO_ERROR) {
		if (is_path_in_gateway(sources->data)) {
			fm_infoprint(GTK_WINDOW(global_ofm->window),
//				     osso_common_error(CERM_CAN_NOT_MOVEICE));
					_("sfil_ib_cannot_move_device"));				     
		}
		FM_ULOG_ERR("selection state error %d\n", sel_state->error);
		g_free(sel_state);
		return;
	}

	if (sel_state->no_devices > 0) {
		if(is_path_in_media_server(sources->data))
		{
			fm_infoprint(GTK_WINDOW(global_ofm->window),
				_("sfil_ib_unable_move_shared_media"));			     
		}
		else if(is_path_virtual(sources->data) && is_path_in_gateway(sources->data))
		{
			fm_infoprint(GTK_WINDOW(global_ofm->window), _("sfil_ib_unable_move_bluetooth"));
		}
		else
		{

			fm_infoprint(GTK_WINDOW(global_ofm->window),
				_("sfil_ib_cannot_move_device"));			     
		}
		g_free(sel_state);
		return;
	}
	if(is_path_in_gateway(destination) && is_path_in_gateway(sources->data))
	{
		fm_infonote(GTK_WINDOW(global_ofm->window),
				_("sfil_ni_move_not_supported_bluetooth"),NULL);
		FM_ULOG_INFO("Source and target are in gateway\n");

		g_free(sel_state);
		return;
	}

	if(is_path_in_media_server(destination))
	{
		fm_infoprint(GTK_WINDOW(global_ofm->window),
				_("sfil_ib_not_allowed"));
		g_free(sel_state);
		return;
	}
	if(is_path_virtual(destination))
	{
		fm_infoprint(GTK_WINDOW(global_ofm->window),
				_("sfil_ib_not_allowed"));

		g_free(sel_state);
		return;
	}
	if (is_path_readonly(destination, FALSE) && !is_path_in_gateway(destination)) {
		fm_infoprint(GTK_WINDOW(global_ofm->window),
				dgettext(HILDON_FM_L10N_PACKAGE,"sfil_ib_readonly_location"));

		g_free(sel_state);
		return;
	}

	if (selection_info_get_selection_type (global_ofm->si) == SIT_MMC_SELECTED) 
	  {
	    fm_infoprint(GTK_WINDOW(global_ofm->window), _("sfil_ib_cannot_move_mmc"));

		g_free(sel_state);
	    return;
	  }
	
	if (sel_state->ronly_parent) { 
		if(sel_state->no_folders == 1 && sel_state->no_files == 0)
		{
			fm_infoprint(GTK_WINDOW(global_ofm->window),
				dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
				"sfil_ib_unable_to_move_selected_folder"));
		}
		else if(sel_state->no_folders == 0 && sel_state->no_files == 1)
		{
			fm_infoprint(GTK_WINDOW(global_ofm->window),
				dgettext(HILDON_FM_L10N_PACKAGE,
				"sfil_ib_cannot_move_file"));
		}
		else if(sel_state->no_folders + sel_state->no_files > 1)
		{
			fm_infoprint(GTK_WINDOW(global_ofm->window),
				dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
				"sfil_ib_unable_to_move_selected_items"));
		}
		g_free(sel_state);
		return;
	}
	sttsres = check_source_to_target(sources, destination);
	switch (sttsres) {
	case STTS_MEMORY_ERR:
	case STTS_TGT_IN_SRC:
		fm_infoprint(GTK_WINDOW(global_ofm->window),
//			     osso_common_error(CERM_CAN_NOT_MV_FOLDER_INTO_ITSELF));
				dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,"ckdg_ib_unable_to_move_folder_into_itself"));			     
		g_free(sel_state);
		return;
		break;
	case STTS_TGT_REPLACES_SRC:
		fm_infoprint(GTK_WINDOW(global_ofm->window),
			     dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
				     "sfil_ib_unable_to_copy_or_move_a_subfolder_within_original_folder")
		    );
		g_free(sel_state);
		return;
		break;
	case STTS_OK:
		break;
	}

	if (match_parents_path(sources, destination)) {
		fm_infoprint(GTK_WINDOW(global_ofm->window),
//			     osso_common_error(CERM_CAN_NOT_MOVE_ORIG_LOCATION));
				dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"ckdg_ib_unable_to_move_to_current_location"));			     
		g_free(sel_state);
		return;
	}
	g_object_get(G_OBJECT(fs), "model", &fsmodel, NULL);
	if (fsmodel == NULL) {
		FM_ULOG_ERR("Error getting model\n");
		g_free(sel_state);
		return;
	}
	sources_tmp = g_slist_copy(sources);
	removed_items = remove_safe_folders_from_gslist(fs, &sources_tmp, 
			&nonexistant_items,TRUE);


	if ((removed_items == sel_state->no_folders + sel_state->no_files) && (removed_items > 0)) {
		if(selectionstate.no_folders == 1 && selectionstate.no_files == 0)
		{
			if(removed_items == nonexistant_items)
			{
				fm_infonote(GTK_WINDOW(global_ofm->window),
						_("sfil_ni_cannot_move_items_not_found"),
						_("sfil_bd_cannot_move_items_not_found_ok"));
			} else
			{
				fm_infoprint(GTK_WINDOW(global_ofm->window),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
					"sfil_ib_unable_to_move_selected_folder"));
			}
				
	        } else if(selectionstate.no_folders == 0 && selectionstate.no_files == 1){
			if(removed_items == nonexistant_items)
			{
				fm_infonote(GTK_WINDOW(global_ofm->window),
						_("sfil_ni_cannot_move_file_not_found"),
						_("sfil_bd_cannot_move_file_not_found_ok"));
			} else
			{
				fm_infoprint(GTK_WINDOW(global_ofm->window),
					dgettext(HILDON_FM_L10N_PACKAGE,
						"sfil_ib_cannot_move_file"));
			}
                } else {
			if(removed_items == nonexistant_items)
			{
				fm_infonote(GTK_WINDOW(global_ofm->window),
						_("sfil_ni_cannot_move_items_not_found"),
						_("sfil_bd_cannot_move_items_not_found_ok"));
			} else
			{
				fm_infoprint(GTK_WINDOW(global_ofm->window),
					dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
						"sfil_ib_unable_to_move_selected_items"));
			}
		}
		g_free(sel_state);
		return;
	}
	nonexistant_items = 0;
	removed_items = remove_safe_folders_from_gslist(fs, &sources_tmp, 
			&nonexistant_items,FALSE);

	display_name = osso_fm_path_get_display_name(destination);

	if ((sel_state->no_files == 1) && (sel_state->no_folders == 0)) {
		note = g_strdup_printf("%s\n%s", _("sfil_nc_move_file"), display_name);
		button1_text = g_strdup(_("sfil_bd_move_file_ok"));
		button2_text = g_strdup(_("sfil_bd_move_file_cancel"));
	} else if ((sel_state->no_files == 0) && (sel_state->no_folders == 1)) {
		note = g_strdup_printf("%s\n%s", _("sfil_nc_move_folder"), display_name);
		button1_text = g_strdup(_("sfil_bd_move_folder_ok"));
		button2_text = g_strdup(_("sfil_bd_move_folder_cancel"));
	} else {
		note = g_strdup_printf("%s\n%s", _("sfil_nc_move_items"), display_name);
		button1_text = g_strdup(_("sfil_bd_move_items_ok"));
		button2_text = g_strdup(_("sfil_bd_move_items_cancel"));
	}
	g_free(display_name);
	osso_fm_set_operation_state(OPER_MOVE);
	if (note != NULL) {
		confirmnote =
		    hildon_note_new_confirmation_add_buttons(GTK_WINDOW
							     (global_ofm->window),
							     (const gchar *) note, button1_text,
							     GTK_RESPONSE_OK, button2_text,
							     GTK_RESPONSE_CANCEL, NULL);
		if (confirmnote == NULL) {
			FM_ULOG_ERR("Error allocation memory");
			g_free(note);
			g_free(button1_text);
			g_free(button2_text);
			osso_fm_set_operation_state(OPER_NONE);
			g_free(sel_state);
			return;
		}
		response = gtk_dialog_run(GTK_DIALOG(confirmnote));
		g_free(note);
		g_free(button1_text);
		g_free(button2_text);
		gtk_widget_destroy(confirmnote);
		if (response != GTK_RESPONSE_OK) {
			osso_fm_set_operation_state(OPER_NONE);
			g_free(sel_state);
			return;
		}
	}

	make_uri_list(sources_tmp, &sourcelist, destination, &targetlist, fsmodel, FALSE);


	if ((sourcelist != NULL) && (targetlist != NULL)) {
		xfer_uri(sourcelist, targetlist,
			 GTK_WINDOW(global_ofm->window),
			 XFER_MOVE, removed_items, nonexistant_items,FALSE);
	} else {
		osso_fm_set_operation_state(OPER_NONE);
	}
	//select destination
	hildon_file_selection_set_current_folder(fs, destination, NULL);
	g_free(sel_state);
}

void
on_file_selection_uris_dropped (HildonFileSelection *fs,
				const char *dest_uri,
				GSList *src_uris)
{
  /* We convert the URIs back to GtkFilePath:s because the whole file
     manager is written in terms of GtkFilePath:s.
  */
  
  GtkFileSystem *gfs = global_ofm->gtkfilesystem;
  GtkFilePath *destination;
  GSList *sources;

  destination = gtk_file_system_uri_to_path (gfs, dest_uri);
  sources = NULL;
  while (src_uris)
    {
      const char *uri = (const char *)src_uris->data;
      sources = g_slist_prepend (sources,
				 gtk_file_system_uri_to_path (gfs, uri));
      src_uris = src_uris->next;
    }
  sources = g_slist_reverse (sources);

  on_file_selection_items_dropped (fs, destination, sources);

  gtk_file_path_free (destination);
  gtk_file_paths_free (sources);
}

/**
 * on_operation_progress_timeout:
 * For a specified time lapse, progress window is shown.
 * @param progresswindow pointer to progresswindow
 * @return gint
 **/
gint on_operation_progress_timeout(gpointer progresswindow)
{
	DEBUG_PRINT_START;

	if (progresswindow != NULL) {
		gtk_widget_show_all(progresswindow);
	}

	DEBUG_PRINT_END;
	return FALSE;
}


static void
topmost_status_changed (GObject *settings, GParamSpec *param, gpointer data)
{
  gboolean is_topmost;

  g_object_get (settings, "is-topmost", &is_topmost, NULL);
  
  if (is_topmost)
    fm_application_topped_cb (data);
  else
    fm_application_untopped_cb (data);
}

/**
 * fm_application_topped_cb:
 * Application Tapped
 * @param prog HildonProgram for the application
 * @return void
 **/
void fm_application_topped_cb (gpointer prog)
{
	DEBUG_PRINT_START;
	if (prog == NULL) {
		FM_ULOG_ERR("Invalid parameter : prog (null)");
		return;
	}
	FM_ULOG_DEBUG1("FM Window topped\n");
	FM_ULOG_ERR("FM Window topped\n");
	appstate.application_untopped = FALSE;
	//fm_infoprint (GTK_WINDOW (app), "application topped");
	//gtk_window_present(GTK_WINDOW(window));
	hildon_program_set_can_hibernate (prog, FALSE);
	osso_fm_key_file_init();
	//load_fm_state ();

	DEBUG_PRINT_END;
	return;
}


/**
 * fm_application_untopped_cb:
 * Application un topped
 * @param prog HildonProgram for the application
 * @return void
 **/
void fm_application_untopped_cb (gpointer prog)
{
	GTimeVal curtime = { 0, 0 };
	gboolean ready;
	HildonFileSystemModel *fsmodel = NULL;
	GtkTreeIter iter;

	DEBUG_PRINT_START;

	if (prog == NULL) {
		FM_ULOG_ERR("Invalid parameter : prog (null)");
		return;
	}
	FM_ULOG_DEBUG1("FM App untopped\n");
	FM_ULOG_ERR("FM App untopped\n");
	appstate.application_untopped = TRUE;
	save_fm_state();
	g_object_get(G_OBJECT(global_ofm->fs), "model", &fsmodel, NULL);
	if (fsmodel == NULL) {
		FM_ULOG_ERR("fm_application_untopped_cb:Error getting model\n");
		return;
	}
	if (hildon_file_selection_get_current_folder_iter(global_ofm->fs, &iter)) {
		gtk_tree_model_get((GtkTreeModel *) fsmodel, &iter,
				   HILDON_FILE_SYSTEM_MODEL_COLUMN_LOAD_READY, &ready, -1);
	}
	FM_ULOG_INFO("IS UPDATED %d\n", ready);
	if (!ready) {
		if (!selected_location_in_gateway(global_ofm->fs)) {
			ready = TRUE;
		}
	}

	if ((global_ofm->cur_operation == OPER_NONE ||
	     global_ofm->cur_operation == OPER_DETAILS) && ready) {
		FM_ULOG_INFO("SET KILLABLE\n");
		hildon_program_set_can_hibernate (prog, TRUE);
	} else {
		FM_ULOG_INFO("NOT SET KILLABLE\n");
	}

	//Cleanup of thumbnails 
	g_get_current_time(&curtime);
	hildon_thumbnail_factory_clean_cache 
	  (-1, curtime.tv_sec - THUMBNAIL_CLEAN_TIME);

	osso_fm_save_key_file();
	//  fm_infoprint (GTK_WINDOW (app), "application untopped");

	DEBUG_PRINT_END;
	return;
}


/**
 * fm_application_destroy_cb:
 * Application destroy event callback
 * @param widget Application window
 * @param userdata NULL
 * @return void 
 **/
gboolean
fm_application_destroy_cb (GtkWidget *window, GdkEvent *ev,
			   gpointer userdata)
{
  DEBUG_PRINT_START;
  fm_closing = TRUE;

  clear_fm_state();

  {
    gint pos = -1;
    g_object_get (global_ofm->fs,
		  "pane-position", &pos,
		  NULL);
    g_key_file_set_integer (global_ofm->keyfile, "ui", "pane_position", pos);
  }
    
  FM_ULOG_ERR("saving keyfile\n");
  osso_fm_save_key_file();

  FM_ULOG_ERR("closing keyfile\n");
  osso_fm_close_key_file();

  gtk_widget_destroy (window);
  FM_ULOG_ERR("app destroyed\n");

  osso_fm_deinitialize(fm_app_ctx);
  LOG_CLOSE();

  exit(0);
}

/**
 * on_fm_exit_signal:
 * This is a helper function to close UI application in normal way.
 * @param  signal  Integer value of the signal received
 * @return void
 **/
void on_fm_exit_signal(int signal)
{
  gtk_widget_destroy (GTK_WIDGET (global_ofm->window));
}


/**
 * on_file_selection_key_release:
 * Event invoked on hard key signal
 * @param widget Widget from which the event is generated.
 * @param event Event key
 * @param window Application main window
 * @return FALSE
 **/
gboolean on_file_selection_key_release(GtkWidget * widget, GdkEvent * event, gpointer window)
{
	DEBUG_PRINT_START;
	if ((event != NULL) && (event->key.keyval != 0) && (window != NULL)) {
		FM_ULOG_DEBUG1("\nGDK KEYVALUE : %d\n\n", event->key.keyval);

		switch (event->key.keyval) {
		case GDK_Return:
			//on_menu_file_open_activate (NULL, global_ofm);

			break;
			//        case GDK_Escape:
		case GDK_Left:
			if (hildon_file_selection_get_active_pane(global_ofm->fs) ==
			    HILDON_FILE_SELECTION_PANE_NAVIGATION) {
				g_object_set(global_ofm->fs, "active-pane",
					     HILDON_FILE_SELECTION_PANE_NAVIGATION, NULL);
			}


		case GDK_Right:
			if (hildon_file_selection_get_active_pane(global_ofm->fs) ==
			    HILDON_FILE_SELECTION_PANE_CONTENT) {
				g_object_set(global_ofm->fs, "active-pane", 
						HILDON_FILE_SELECTION_PANE_CONTENT, NULL);
			}

			break;
		}
	}
	DEBUG_PRINT_END;
	return FALSE;
}


/**
 * on_file_selection_key_press:
 * Event invoked on hard key signal
 * @param widget Widget from which the event is generated.
 * @param event Event key
 * @param app callback data which has Application main window pointer
 * @return FALSE
 */
gboolean on_file_selection_key_press(GtkWidget * widget, GdkEvent * event, gpointer app)
{
	DEBUG_PRINT_START;
	if ((event != NULL) && (event->key.keyval != 0) && (app != NULL)) {
		FM_ULOG_DEBUG1("\nGDK KEYVALUE : %d\n\n", event->key.keyval);

		switch (event->key.keyval) 
		{
		  /* No keys handled here...
		   */
		}
	}
	DEBUG_PRINT_END;
	return FALSE;
}



/**
 * on_fm_timer:
 * Invoked on application startup timer event
 * @param app HildonApp
 * @return TRUE to continue timer event, FALSE to abort
 */
gboolean on_fm_timer(gpointer data)
{
	static gint idle_count;
	HildonProgram *prog = NULL;
	HildonWindow *window = NULL;
	HildonFileSelection *fs = NULL;
	gint normal_toolbar = 0, fullscreen_toolbar = 0;
	GtkFilePath *temp_path;
	HildonFileSystemModel *fsmodel = NULL;
	OSSOFileManager *ofm;
	//temp start
	GTimeVal time_before;
	GTimeVal time_after;
	//temp end

	DEBUG_PRINT_START;
	gdk_threads_enter();
	if(fm_closing)
	{
		gdk_threads_leave();
		return FALSE;
	}
	gdk_threads_leave();
	ofm = (OSSOFileManager* )data;
	prog = ofm->prog;
	idle_count++;
	fs = ofm->fs;
	FM_ULOG_INFO("Ideal Count %d\n", idle_count);
	switch (idle_count) {
	case 1:
		/* Initialize GnomeVFS! */
		if (!gnome_vfs_init()) {
			FM_ULOG_ERR("%s: Failed to initialize VFS", APP_NAME);
			return 1;
		}
		osso_fm_key_file_init();
		g_get_current_time(&time_before);
		gdk_threads_enter();
		create_file_selection(ofm);
		gdk_threads_leave();
		if (global_ofm->fs == NULL) {
			FM_ULOG_ERR("Error accessing Global FileSelection\n");
			return FALSE;
		}
		{
		  GError *error = NULL;
		  gint pos = g_key_file_get_integer (global_ofm->keyfile,
						     "ui", "pane_position",
						     &error);
		  if (error)
		    {
		      g_error_free (error);
		      pos = 250;
		    }
		  g_object_set (global_ofm->fs,
				"pane-position", pos,
				NULL);
		}
		g_get_current_time(&time_after);
		break;
	case 2:
		gdk_threads_enter();
		/*CCP Initialising clipboard values */
		osso_fm_clipboard_init();

		window = ofm->window;
		if ((window != NULL) && (fs != NULL)) {
			temp_path = hildon_file_selection_get_current_folder(fs);
			hildon_file_selection_set_select_multiple(fs, TRUE);
			if(temp_path != NULL)
			{
				hildon_file_selection_set_current_folder(fs, temp_path, NULL);
				gtk_file_path_free(temp_path);
			}

			ofm->fmmmenu = osso_fm_create_main_menu(ofm->prog,ofm->window,ofm->fs);
			GTK_WIDGET_SET_FLAGS(ofm->fmmmenu->self, GTK_CAN_FOCUS);
			gtk_widget_show_all((GtkWidget *) ofm->fmmmenu->self);
			osso_fm_set_currentselection("");
			appstate.mainwindow.fullscreen = FALSE;
			osso_fm_setup_fullscreening_signals ();
		}
		gdk_threads_leave();
		break;
	case 3:
		if (fm_app_ctx == NULL) {
			fm_app_ctx = osso_fm_initialize(APP_NAME, APP_VER, TRUE, NULL);
			if (fm_app_ctx == NULL) {
				FM_ULOG_ERR("%s:Failed initialize context\n", APP_NAME);
				return FALSE;
			}
			if (osso_rpc_set_default_cb_f(fm_app_ctx,
						      osso_fm_rpc_callback_handler,
						      NULL) != OSSO_OK) {
				osso_log(LOG_ERR, " File Manager rpc cb register failed ..");
				g_print(" File Manager rpc cb register failed ..\n");
				osso_deinitialize(fm_app_ctx);
			}
			g_print(" File Manager rpc cb register succeeded ..\n");
		}
		break;
	case 4:
		gdk_threads_enter();
		window = ofm->window;
		osso_fm_set_afw_callbacks (window);

		GTK_WIDGET_SET_FLAGS(window, GTK_CAN_FOCUS);
		GTK_WIDGET_SET_FLAGS(fs, GTK_CAN_FOCUS);
		GTK_WIDGET_UNSET_FLAGS(&(fs->parent), GTK_CAN_FOCUS);

		g_signal_connect(G_OBJECT(window), "key-release-event",
				 G_CALLBACK(on_file_selection_key_release), prog);
		g_signal_connect(G_OBJECT(window), "key-press-event",
				 G_CALLBACK(on_file_selection_key_press), prog);
		g_signal_connect (G_OBJECT(window),
				  "notify::is-topmost",
				  G_CALLBACK(topmost_status_changed),
				  prog);

		g_signal_connect(G_OBJECT(fs), "navigation-pane-context-menu",
				 G_CALLBACK(on_navigation_popup_menu_activate), global_ofm);
		g_signal_connect(G_OBJECT(fs), "content-pane-context-menu",
				 G_CALLBACK(on_content_popup_menu_activate), global_ofm);
		g_signal_connect(G_OBJECT(fs), "current-folder-changed",
				 G_CALLBACK(update_fm_title), window);
		g_object_get(G_OBJECT(fs), "model", &fsmodel, NULL);
		if (fsmodel != NULL) {
			g_signal_connect(G_OBJECT(fsmodel), "row-changed",
					 G_CALLBACK(on_fileselection_row_changed), window);
		} else {
			FM_ULOG_ERR("Unable to obtail fsmodel\n");
		}

		g_signal_connect(G_OBJECT(fs), "file-activated",
				 G_CALLBACK(on_menu_file_open_activate), global_ofm);

		g_signal_connect(G_OBJECT(fs), "selection-changed",
				 G_CALLBACK(on_selection_changed), window);
		g_signal_connect(G_OBJECT(fs), "current-folder-changed",
				G_CALLBACK (on_current_folder_changed), window);
		g_signal_connect(G_OBJECT(fs), "notify::active-pane",
				 G_CALLBACK(on_pane_changed), window);

		g_signal_connect(G_OBJECT(fs), "uris-dropped",
				 G_CALLBACK(on_file_selection_uris_dropped), window);
		if (fsmodel != NULL) {
			g_signal_connect(G_OBJECT(fsmodel), "finished-loading",
					G_CALLBACK(on_file_loading_finished), NULL);
		} else {
			FM_ULOG_ERR("Unable to obtail fsmodel\n");
		}
		gtk_widget_show_all (GTK_WIDGET (window));
		update_fm_title((GtkWidget *)fs, window);
		gdk_threads_leave();
		break;
	case 5:
		gdk_threads_enter();
		if (osso_fm_get_toolbar_status(&normal_toolbar, &fullscreen_toolbar)) {
			DEBUG_PRINT;
			if (normal_toolbar == 0) {
				DEBUG_PRINT;
				gtk_check_menu_item_set_active( (GtkCheckMenuItem *) 
						ofm->fmmmenu->view->normal_mode,
						FALSE);

			}
			if (fullscreen_toolbar == 0) {
				DEBUG_PRINT;
				gtk_check_menu_item_set_active((GtkCheckMenuItem *) 
						ofm->fmmmenu->view->fullscreen_mode,
						FALSE);
			}

			DEBUG_PRINT;
		}
		DEBUG_PRINT;
		gdk_threads_leave();
		break;
	case 6:
		gdk_threads_enter();
		hildon_program_set_can_hibernate (prog, FALSE);
		load_fm_state();
		if (path_to_select_on_open != NULL
		    && appstate.mainwindow.opendialog == OPEN_DIALOG_NONE) {
                        gboolean ret;
			ret = hildon_file_selection_set_current_folder_uri(fs,
				 path_to_select_on_open, NULL);
                        ULOG_DEBUG_F("hildon_file_selection_set_"
                                     "current_folder_uri returned %d", ret);
		}
		ULOG_DEBUG_F("path_to_select_on_open = '%s'",
                             path_to_select_on_open);
		gdk_threads_leave();

		FM_ULOG_OPEN("File Manager");
		FM_ULOG_DEBUG1("FM STARTUP TIME : %ld(ticks), %ld (count)\n\n",
			     (long) (end - start), (long) CLOCKS_PER_SEC);

		return FALSE;
		break;
	}

	DEBUG_PRINT_END;
	return TRUE;
}



//BEG: Toolbar buttons callback functions

/**
 * on_toolbar_find_clicked:
 * Callback function for Find button on toolbar
 * @param button GtkButton pointer which is clicked
 * @param data callback data
 * @return void
 **/

void on_toolbar_find_clicked (GtkButton * button, gpointer data)
{
	DEBUG_PRINT_START;
	on_find_activate();
	DEBUG_PRINT_END;
	return;
}

/**
 * on_toolbar_newfolder_clicked:
 * Callback function for New Folder button on toolbar
 * @param button GtkButton pointer which is clicked
 * @param data callback data
 * @return void
 **/
void on_toolbar_newfolder_clicked (GtkButton * button, gpointer data)
{
	DEBUG_PRINT_START;
	on_newfolder_activate(data);
	DEBUG_PRINT_END;
	return;
}

/**
 * on_toolbar_moveto_clicked:
 * Callback function for Moveto button on toolbar
 * @param button GtkButton pointer which is clicked
 * @param data callback data
 * @return void
 **/
void on_toolbar_moveto_clicked (GtkButton * button, gpointer data)
{
	DEBUG_PRINT_START;
	on_moveto_activate(data);
	DEBUG_PRINT_END;
	return;
}

/**
 * on_toolbar_detail_clicked:
 * Callback function for Details button on toolbar
 * @param button GtkButton pointer which is clicked
 * @param data callback data
 * @return void
 **/
void on_toolbar_detail_clicked (GtkButton * button, gpointer data)
{
	DEBUG_PRINT_START;
	on_details_activate(data);
	DEBUG_PRINT_END;
	return;
}

/**
 * on_toolbar_delete_clicked:
 * Callback function for Delete button on toolbar
 * @param button GtkButton pointer which is clicked
 * @param data callback data
 * @return void
 **/
void on_toolbar_delete_clicked (GtkButton * button, gpointer data)
{
	DEBUG_PRINT_START;
	on_delete_activate(data);
	DEBUG_PRINT_END;
	return;
}

/**
 * on_toolbar_listview_clicked:
 * Callback function for List View button on toolbar
 * @param button GtkButton pointer which is clicked
 * @param data callback data
 * @return void
 **/
void on_toolbar_listview_clicked (GtkButton * button, gpointer data)
{
	DEBUG_PRINT_START;
	on_listview_activate(data);
	DEBUG_PRINT_END;
	return;
}

/**
 * on_toolbar_thumbview_clicked:
 * Callback function for Thumbnail View button on toolbar
 * @param button GtkButton pointer which is clicked
 * @param data callback data
 * @return void
 **/
void on_toolbar_thumbview_clicked (GtkButton * button, gpointer data)
{
	DEBUG_PRINT_START;
	on_thumbview_activate(data);
	DEBUG_PRINT_END;
	return;
}
//END: Toolbar buttons callback functions

/* Full-screen handling
 */

static void
set_current_toolbar_visibility (gboolean f)
{
  if (global_ofm->fmtoolbar->toolbar)
    {
      if (f)
	gtk_widget_show (GTK_WIDGET (global_ofm->fmtoolbar->toolbar));
      else
	gtk_widget_hide (GTK_WIDGET (global_ofm->fmtoolbar->toolbar));
    }
}

void
set_fullscreen (gboolean f)
{
  if (f)
    gtk_window_fullscreen (GTK_WINDOW (global_ofm->window));
  else
    gtk_window_unfullscreen (GTK_WINDOW (global_ofm->window));
}

void
toggle_fullscreen ()
{
  set_fullscreen (!appstate.mainwindow.fullscreen);
}

void
set_toolbar_visibility (gboolean fullscreen, gboolean visibility)
{
  /* XXX - toolbar visibility is not saved in state.
   */

  if (fullscreen)
    {
      if (appstate.mainwindow.fullscreen)
	set_current_toolbar_visibility (visibility);
    }
  else
    {
      if (!appstate.mainwindow.fullscreen)
	set_current_toolbar_visibility (visibility);
    }
}

static gboolean
window_state_event (GtkWidget *widget, GdkEventWindowState *event,
		    gpointer unused)
{
  gboolean f = (event->new_window_state & GDK_WINDOW_STATE_FULLSCREEN);

  if (appstate.mainwindow.fullscreen != f)
    {
      appstate.mainwindow.fullscreen = f;
      gtk_check_menu_item_set_active 
	(GTK_CHECK_MENU_ITEM (global_ofm->fmmmenu->view->full_screen), f);
      if (appstate.mainwindow.fullscreen)
	{
	  // gtk_container_set_border_width (GTK_CONTAINER (widget), 10);
	  set_current_toolbar_visibility 
	    (gtk_check_menu_item_get_active
	     (GTK_CHECK_MENU_ITEM 
	      (global_ofm->fmmmenu->view->fullscreen_mode)));
	}
      else
	{
	  // gtk_container_set_border_width (GTK_CONTAINER (widget), 0);
	  set_current_toolbar_visibility 
	    (gtk_check_menu_item_get_active
	     (GTK_CHECK_MENU_ITEM 
	      (global_ofm->fmmmenu->view->normal_mode)));
	}
    }

  return FALSE;
}

static gboolean
key_event (GtkWidget *widget,
	   GdkEventKey *event,
	   gpointer data)
{
  static gboolean fullscreen_key_repeating = FALSE;

  if (event->type == GDK_KEY_PRESS &&
      event->keyval == HILDON_HARDKEY_FULLSCREEN &&
      !fullscreen_key_repeating)
    {
      toggle_fullscreen ();
      fullscreen_key_repeating = TRUE;
      return TRUE;
    }

  if (event->type == GDK_KEY_RELEASE &&
      event->keyval == HILDON_HARDKEY_FULLSCREEN)
    {
      fullscreen_key_repeating = FALSE;
      return TRUE;
    }
      
  return FALSE;
}

static void
fullscreen_activated (GtkWidget *item)
{
  gboolean active =
    gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (item));
  set_fullscreen (active);
}

static void
normal_toolbar_activated (GtkWidget *item)
{
  gboolean active =
    gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (item));
  set_toolbar_visibility (FALSE, active);
}

static void
fullscreen_toolbar_activated (GtkWidget *item)
{
  gboolean active =
    gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (item));
  set_toolbar_visibility (TRUE, active);
}

void
osso_fm_setup_fullscreening_signals ()
{
  GtkWindow *window = GTK_WINDOW (global_ofm->window);

  g_signal_connect (window, "window_state_event",
		    G_CALLBACK (window_state_event), NULL);
  g_signal_connect (window, "key_press_event",
		    G_CALLBACK (key_event), NULL);
  g_signal_connect (window, "key_release_event",
		    G_CALLBACK (key_event), NULL);

  g_signal_connect (global_ofm->fmmmenu->view->full_screen, "activate",
		    G_CALLBACK (fullscreen_activated), NULL);
  g_signal_connect (global_ofm->fmmmenu->view->normal_mode, "activate",
		    G_CALLBACK (normal_toolbar_activated), NULL);
  g_signal_connect (global_ofm->fmmmenu->view->fullscreen_mode, "activate",
		    G_CALLBACK (fullscreen_toolbar_activated), NULL);
}
