/**
 * @file hildon-shorten-names-dialog.c
 *
 * This file contains the API implementation for shorten file names dialog.
 *
 * Copyright (C) 2005 Nokia Corporation
 */

/* HILDON DOC
* @shortdesc: HildonShortenNamesDialog is a dialog for truncating file
* names so that they can be moved to a target platform.
* @longdesc: HildonShortenNamesDialog is a dialog which lets the
* application to truncate file names for target platform with limited
* filename length. The dialog contains a column listbox with file names,
* a checkbox and 3 buttons (ok, rename, cancel). The part of the file
* name that is too long is shown in gray color. Too long file names are
* indicated with an exclamation mark icon. Manual shortening is done via
* a separate renaming dialog which opens when 'rename' button is pressed.
* If 'cut remaining name automatically' check button is checked then the
* application will do the shortening.
*/

#include <glib.h>
#include <gtk/gtk.h>
#include <errno.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <stdio.h>

#include <hildon-shorten-names-dialog.h>
#include <hildon/hildon-caption.h>
#include "osso_fm_generic.h"
#include "osso_fm_controller.h"
#include "osso_fm_utils.h"
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <libintl.h>
#define _(String) dgettext(PACKAGE, String)

static GtkDialogClass *parent_class;

typedef struct _HildonShortenNamesDialogPrivate
 HildonShortenNamesDialogPrivate;
struct _HildonShortenNamesDialogPrivate {
	GtkButton *okButton;
	GtkButton *renameButton;
	GtkButton *cancelButton;

	GtkCheckButton *is_automatic_button;
	GtkTreeView *treeview;
	GtkListStore *store;

	GdkPixbuf *icon_no;
	GdkPixbuf *icon_yes;
	gint max_width;

	GtkDialog *renameDialog;

	gint num_items;
};

#define SHORTEN_NAMES_READYICON_COLUMN 0
#define SHORTEN_NAMES_MIMEICON_COLUMN 1
#define SHORTEN_NAMES_NAME_COLUMN 2
#define SHORTEN_NAMES_FULLPATH_COLUMN 3

#define HILDON_SHORTEN_NAMES_DIALOG_GET_PRIVATE(o)  \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), HILDON_TYPE_SHORTEN_NAMES_DIALOG,\
HildonShortenNamesDialogPrivate))

static void hildon_shorten_names_dialog_class_init(HildonShortenNamesDialogClass * class);
static void hildon_shorten_names_dialog_init(HildonShortenNamesDialog * widget);
static void open_rename(GtkWidget * widget, HildonShortenNamesDialog * dialog);
static void automatic_toggled(GtkToggleButton * button, gpointer data);
static void exclamation_data_func(GtkTreeViewColumn * col,
				  GtkCellRenderer * renderer,
				  GtkTreeModel * model, GtkTreeIter * iter, gpointer data);
static void dimming_cell_data_func(GtkTreeViewColumn * col,
				   GtkCellRenderer * renderer,
				   GtkTreeModel * model, GtkTreeIter * iter, gpointer data);

void hildon_shorten_names_dialog_set_model(HildonShortenNamesDialog *
					   dialog, GtkListStore * store, gint max_length);
gboolean hildon_shorten_names_dialog_get_automatic(HildonShortenNamesDialog * dialog);

static void automatic_toggled(GtkToggleButton * button, gpointer data)
{
	HildonShortenNamesDialog *dialog = HILDON_SHORTEN_NAMES_DIALOG(data);
	HildonShortenNamesDialogPrivate *priv = HILDON_SHORTEN_NAMES_DIALOG_GET_PRIVATE(dialog);

	if (gtk_toggle_button_get_active(button)) {
		gtk_widget_set_sensitive(GTK_WIDGET(priv->okButton), TRUE);
	}

	else {
		GtkTreeIter iter;
		GtkTreeModel *model;
		gchar *name;
		gboolean ok = TRUE;

		model = gtk_tree_view_get_model(GTK_TREE_VIEW(priv->treeview));
		gtk_tree_model_get_iter_first(GTK_TREE_MODEL(model), &iter);
		do {
			gtk_tree_model_get(model, &iter, 1, &name, -1);
			if (strlen(name) > priv->max_width)
				ok = FALSE;
		}
		while (gtk_tree_model_iter_next(GTK_TREE_MODEL(model), &iter));
		if (!ok)
			gtk_widget_set_sensitive(GTK_WIDGET(priv->okButton), FALSE);
	}
	gtk_widget_queue_draw(GTK_WIDGET(dialog));
}

static void exclamation_data_func(GtkTreeViewColumn * col,
				  GtkCellRenderer * renderer,
				  GtkTreeModel * model, GtkTreeIter * iter, gpointer data)
{
	gchar *name;
	HildonShortenNamesDialog *dialog;
	HildonShortenNamesDialogPrivate *priv;

	dialog = HILDON_SHORTEN_NAMES_DIALOG(data);
	priv = HILDON_SHORTEN_NAMES_DIALOG_GET_PRIVATE(dialog);

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(priv->is_automatic_button))) {
		g_object_set(renderer, "stock-id", NULL, NULL);
		return;
	}

	gtk_tree_model_get(model, iter, 1, &name, -1);
	if (strlen(name) <= priv->max_width)
		g_object_set(renderer, "stock-id", NULL, NULL);
	else
		g_object_set(renderer, "stock-id", "hildon-shorten-warning", NULL);
	g_free(name);
}

static void dimming_cell_data_func(GtkTreeViewColumn * col,
				   GtkCellRenderer * renderer,
				   GtkTreeModel * model, GtkTreeIter * iter, gpointer data)
{
	gchar *name = NULL;
	gchar *extension = NULL;
	gchar *disp_name = NULL;
	gchar *path = NULL;
	gchar *filepath = NULL;
	gchar *filepath_unesc = NULL;
	GnomeVFSURI *pathuri = NULL;
	GnomeVFSURI *parent_pathuri = NULL;
	gchar *buf = NULL;
	gchar *format = NULL;
	gint ok_len = 0;
	HildonShortenNamesDialog *dialog = NULL;
	HildonShortenNamesDialogPrivate *priv = NULL;

	dialog = HILDON_SHORTEN_NAMES_DIALOG(data);
	priv = HILDON_SHORTEN_NAMES_DIALOG_GET_PRIVATE(dialog);

	gtk_tree_model_get(model, iter, 1, &path, -1);

	pathuri = gnome_vfs_uri_new(path);
	parent_pathuri = gnome_vfs_uri_get_parent(pathuri);

	//filepath = gnome_vfs_uri_extract_dirname(pathuri);
	filepath = gnome_vfs_uri_to_string(parent_pathuri,GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD);
	filepath_unesc = gnome_vfs_unescape_string(filepath,NULL);
	name = gnome_vfs_uri_extract_short_name(pathuri);
	extension = osso_fm_path_get_extension(name);
	disp_name = g_strdup(name);
	if (extension == NULL) {
		//-1 is for the slash between the path and the filename
		ok_len = priv->max_width - strlen(filepath_unesc) - 1;
	} else {
		*(disp_name + (extension - name)) = '\0';
		//-1 is for the slash between the path and the filename
		ok_len = priv->max_width - strlen(filepath_unesc) - 1 - strlen(extension);
	}

	//FM_ULOG_ERR("NAME %s\n",name);
	//FM_ULOG_ERR("DISP NAME %s\n",disp_name);
	gnome_vfs_uri_unref(pathuri);
	gnome_vfs_uri_unref(parent_pathuri);
	g_free(filepath);
	g_free(filepath_unesc);

	if (strlen(path) <= priv->max_width) {
		g_object_set(renderer, "markup", disp_name, NULL);
		g_free(path);
		g_free(name);
		g_free(disp_name);
		return;
	}

	/* 30 is the amount of pango code */
	buf = g_malloc(strlen(name) + 300 * sizeof(gchar *));

	format = g_malloc(sizeof(gchar) * 40);
	sprintf(format, "%c.%ds<span color=\"Gray\">%cs</span>", '%',
		ok_len /*priv->max_width */ , '%');
	sprintf(buf, format, disp_name, (disp_name + ok_len /*priv->max_width */ ));
	FM_ULOG_ERR("%s\n", buf);
	g_object_set(renderer, "markup", buf, NULL);

	g_free(buf);
	g_free(format);
	g_free(path);
	g_free(name);
	g_free(disp_name);
}

/*static void open_rename_activated (GtkTreeView *view,
GtkTreePath *path,
GtkTreeViewColumn *col,
gpointer data)
{
open_rename(GTK_WIDGET(view), HILDON_SHORTEN_NAMES_DIALOG(data));
}*/
static gint 
get_path_length(gchar *path_texturi)
{
	GnomeVFSURI *path_uri = NULL;
	gint len =0;
	path_uri = gnome_vfs_uri_new(path_texturi);
	len = strlen(path_texturi);
	len -= strlen(path_uri->method_string);
	len -= strlen("://");
	gnome_vfs_uri_unref(path_uri);
	return len;
}

static void open_rename(GtkWidget * widget, HildonShortenNamesDialog * dialog)
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	//GtkListStore *store;
	GtkTreeSelection *selection;
	gchar *name = NULL;
	gchar *path = NULL;
	gchar *newpath = NULL;
	gchar *newfilename = NULL;
	gchar *newname = NULL;
	gchar *filepath = NULL;
	GnomeVFSURI *pathuri = NULL;
	GnomeVFSURI *newpathuri = NULL;
	gint response;
	HildonShortenNamesDialogPrivate *priv;
	GtkTreeIter iter2;
	gboolean ok = TRUE;

	priv = HILDON_SHORTEN_NAMES_DIALOG_GET_PRIVATE(dialog);
	priv->renameDialog =
	    (GtkDialog *) hildon_file_chooser_dialog_new_with_properties(GTK_WINDOW(dialog),
									 "action",
									 GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER,
									 "select-multiple", FALSE,
									 "show-folder-button",
									 FALSE, "show-location",
									 FALSE, "autonaming", FALSE,
									 NULL);
	g_return_if_fail(GTK_IS_DIALOG(priv->renameDialog));
	gtk_window_set_title(GTK_WINDOW(priv->renameDialog), dgettext(HILDON_LIBS_L10N_PACKAGE,
				"ckdg_ti_rename_default"));

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(priv->treeview));
	gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);
	if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
		gtk_tree_model_get(model, &iter, 1, &path, -1);
		//FM_ULOG_ERR("SEL PATH %s\n",path);
		pathuri = gnome_vfs_uri_new(path);
		name = gnome_vfs_uri_extract_short_name(pathuri);
		gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(priv->renameDialog), name);
		
		GnomeVFSURI *parent_pathuri = NULL;
		parent_pathuri = gnome_vfs_uri_get_parent(pathuri);
		filepath = gnome_vfs_uri_to_string(parent_pathuri,0);
		gnome_vfs_uri_unref(parent_pathuri);
		//filepath = gnome_vfs_uri_extract_dirname(pathuri);
		response = gtk_dialog_run(priv->renameDialog);
		if (response == GTK_RESPONSE_OK) {
			path = gtk_file_chooser_get_uri(GTK_FILE_CHOOSER(priv->renameDialog));
			newpathuri = gnome_vfs_uri_new(path);
			newfilename = gnome_vfs_uri_extract_short_name(newpathuri);
			newpath = g_strdup_printf("%s/%s", filepath, newfilename);
			//gtk_tree_model_set(model, &iter, 1, &path, -1);
			gtk_list_store_set(GTK_LIST_STORE(model), &iter, 1, newpath, -1);
			//FM_ULOG_ERR("%s\n",path);
		}
		gtk_widget_destroy((GtkWidget *) priv->renameDialog);
	}
	// <check> whether okButton should be sensitive or not 
	gtk_tree_model_get_iter_first(GTK_TREE_MODEL(model), &iter2);
	do {
		gtk_tree_model_get(model, &iter2, 1, &newname, -1);
		//if (strlen(newname) > priv->max_width)
		if (get_path_length(newname) > priv->max_width)
			ok = FALSE;
	}
	while (gtk_tree_model_iter_next(GTK_TREE_MODEL(model), &iter2));
	if (ok)
		gtk_widget_set_sensitive(GTK_WIDGET(priv->okButton), TRUE);
	// </check> 
}

static void hildon_shorten_names_dialog_class_init(HildonShortenNamesDialogClass * class)
{
	parent_class = g_type_class_peek_parent(class);
	g_type_class_add_private(class, sizeof(HildonShortenNamesDialogPrivate));
}

static void hildon_shorten_names_dialog_init(HildonShortenNamesDialog * dialog)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	GtkLabel *note;
	GtkScrolledWindow *scrollwindow;
	HildonShortenNamesDialogPrivate *priv = HILDON_SHORTEN_NAMES_DIALOG_GET_PRIVATE(dialog);

	gtk_dialog_set_has_separator(GTK_DIALOG(dialog), FALSE);

	if (priv->num_items > 0)
		gtk_window_set_title(GTK_WINDOW(dialog), dgettext("hildon-fm","hfil_ti_shorten_names"));
	else
		gtk_window_set_title(GTK_WINDOW(dialog), dgettext("hildon-fm","hfil_ti_shorten_name"));


	priv->okButton =
	    GTK_BUTTON(gtk_dialog_add_button
		       (GTK_DIALOG(dialog), dgettext("hildon-fm","hfil_bd_shorten_names_ok"),
			GTK_RESPONSE_ACCEPT));
	gtk_widget_set_sensitive(GTK_WIDGET(priv->okButton), FALSE);

	priv->renameButton =
	    GTK_BUTTON(gtk_button_new_with_label(dgettext("hildon-fm",
					    "hfil_bd_shorten_names_rename")));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
			   GTK_WIDGET(priv->renameButton), FALSE, TRUE, 0);
	g_signal_connect(priv->renameButton, "clicked", G_CALLBACK(open_rename), dialog);

	priv->cancelButton =
	    GTK_BUTTON(gtk_dialog_add_button
		       (GTK_DIALOG(dialog), dgettext("hildon-fm","hfil_bd_shorten_names_cancel"),
			GTK_RESPONSE_CANCEL));

	if (priv->num_items > 1)
		note = (GtkLabel *) gtk_label_new(dgettext(HILDON_FM_L10N_PACKAGE,"hfil_ia_shorten_names"));
	else 
		note = (GtkLabel *) gtk_label_new(dgettext(HILDON_FM_L10N_PACKAGE,"hfil_ia_shorten_name"));

	gtk_label_set_line_wrap(note, TRUE);

	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), GTK_WIDGET(note), FALSE, FALSE, 0);

	priv->store =
	    GTK_LIST_STORE(gtk_list_store_new(3, GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_STRING));
	priv->treeview = (GtkTreeView *) gtk_tree_view_new();
	renderer = gtk_cell_renderer_pixbuf_new();
	col = GTK_TREE_VIEW_COLUMN(gtk_tree_view_column_new_with_attributes(NULL, renderer, NULL));
	gtk_tree_view_column_set_cell_data_func(col, renderer, exclamation_data_func, dialog, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(priv->treeview), col);
	renderer = gtk_cell_renderer_text_new();
	col = GTK_TREE_VIEW_COLUMN(gtk_tree_view_column_new_with_attributes(NULL, renderer, NULL));
	gtk_tree_view_append_column(GTK_TREE_VIEW(priv->treeview), col);
	gtk_tree_view_column_set_cell_data_func(col, renderer,
						dimming_cell_data_func, dialog, NULL);
	gtk_tree_view_set_model(GTK_TREE_VIEW(priv->treeview), GTK_TREE_MODEL(priv->store));
	/*    g_signal_connect(priv->treeview, "row-activated",
	   G_CALLBACK(open_rename_activated), dialog); */

	priv->max_width = 256;
	/*
	   priv->renameDialog = 
	   (GtkDialog *)hildon_file_chooser_dialog_new_with_properties (
	   GTK_WINDOW (dialog),
	   "action",GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER,
	   "select-multiple", FALSE,
	   "show-folder-button", FALSE,
	   "show-location", FALSE,
	   "autonaming", FALSE,
	   NULL);
	   gtk_window_set_title (GTK_WINDOW (priv->renameDialog),
	   _("ckdg_ti_rename_default"));
	 */
	/*
	   priv->renameDialog =
	   (GtkDialog *) gtk_dialog_new_with_buttons(_("sfil_bd_rename_file_dev"), 
	   NULL,
	   GTK_DIALOG_MODAL, "sfil_bd_rename_names_ok_dev",
	   GTK_RESPONSE_ACCEPT,
	   "sfil_bd_shorten_names_cancel_dev",
	   GTK_RESPONSE_CANCEL,
	   NULL);
	   gtk_dialog_set_has_separator(priv->renameDialog, FALSE);
	 */
	scrollwindow = (GtkScrolledWindow *) gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW
					      (scrollwindow), GTK_WIDGET(priv->treeview));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
			   GTK_WIDGET(scrollwindow), FALSE, FALSE, 0);

	priv->is_automatic_button = (GtkCheckButton *)
	    gtk_check_button_new_with_label(dgettext(HILDON_FM_L10N_PACKAGE,
				    "hfil_fi_shorten_names_readonly"));
	g_signal_connect(GTK_TOGGLE_BUTTON(priv->is_automatic_button),
			 "toggled", G_CALLBACK(automatic_toggled), dialog);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
			   GTK_WIDGET(priv->is_automatic_button), FALSE, FALSE, 0);

	gtk_widget_show_all(GTK_DIALOG(dialog)->vbox);
	gtk_widget_show_all(GTK_DIALOG(dialog)->action_area);

}

GType hildon_shorten_names_dialog_get_type(void)
{
	static GType dialog_type = 0;

	if (!dialog_type) {
		static const GTypeInfo dialog_info = {
			sizeof(HildonShortenNamesDialogClass),
			NULL,	/* base_init */
			NULL,	/* base_finalize */
			(GClassInitFunc) hildon_shorten_names_dialog_class_init,
			NULL,	/* class_finalize */
			NULL,	/* class_data */
			sizeof(HildonShortenNamesDialog),
			0,	/* n_preallocs */
			(GInstanceInitFunc) hildon_shorten_names_dialog_init
		};
		dialog_type =
		    g_type_register_static(GTK_TYPE_DIALOG,
					   "HildonShortenNamesDialog", &dialog_info, 0);
	}

	return dialog_type;
}

/**
* hildon_shorten_names_dialog_new:
* @parent: the parent window of the dialog.
*
* Creates a new #HildonShortenNamesDialog widget with Ok, Rename and
* Cancel buttons.
*
* Return value: the newly created #HildonShortenNamesDialog
*/
GtkWidget *hildon_shorten_names_dialog_new(GtkWindow * parent)
{
	HildonShortenNamesDialog *self;

	self = HILDON_SHORTEN_NAMES_DIALOG(g_object_new(HILDON_TYPE_SHORTEN_NAMES_DIALOG, NULL));

	if (parent)
		gtk_window_set_transient_for(GTK_WINDOW(self), parent);

	return GTK_WIDGET(self);
}

/**
* hildon_shorten_names_dialog_set_model:
* @dialog: the shorten names dialog to modify.
* @store: pointer to a GtkListStore with the list of long filenames of
* the type G_TYPE_STRING.
* @max_length: the maximum file length allow by the target platform.
* 
* Sets the model with long file names and the maximum length. The model
* should be contain as the first column the names. Eg. a model could be
* created with:
* 
*  store = (GtkListStore *) gtk_list_store_new(1, G_TYPE_STRING);
*/

void hildon_shorten_names_dialog_set_model(HildonShortenNamesDialog *
					   dialog, GtkListStore * store, gint max_length)
{
	GtkTreeIter iter, iter2;
	gchar *name;
	HildonShortenNamesDialogPrivate *priv = HILDON_SHORTEN_NAMES_DIALOG_GET_PRIVATE(dialog);

	priv->store = gtk_list_store_new(2, GDK_TYPE_PIXBUF, G_TYPE_STRING);
	priv->num_items = 0;

	gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), &iter);
	do {
		gtk_list_store_append(GTK_LIST_STORE(priv->store), &iter2);
		gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, 0, &name, -1);
		//TODO NAME TO BE UNESCAPED
		gtk_list_store_set(GTK_LIST_STORE(priv->store), &iter2, 1, name, -1);
		priv->num_items++;
	}
	while (gtk_tree_model_iter_next(GTK_TREE_MODEL(store), &iter));

	gtk_tree_view_set_model(GTK_TREE_VIEW(priv->treeview), GTK_TREE_MODEL(priv->store));

	priv->max_width = max_length;
	gtk_widget_queue_draw(GTK_WIDGET(priv->treeview));
	gtk_widget_queue_draw(GTK_WIDGET(GTK_DIALOG(dialog)->vbox));
	gtk_widget_show_all(GTK_DIALOG(dialog)->vbox);
	gtk_widget_show_all(GTK_DIALOG(dialog)->action_area);

}

/**
* hildon_shorten_names_dialog_get_model:
* @dialog: the shorten names dialog.
* 
* Return value: The GtkListStore object which contains the shortened file names.
* If the automatic truncation toggle button was checked then some of the
* file names can still be too long and it is the applications'
* responsibility to cut them. Check whether this is the case by calling
* hildon_shorten_names_dialog_get_automatic().
*/

GtkTreeModel * hildon_shorten_names_dialog_get_model(HildonShortenNamesDialog * dialog)
{
	GtkListStore *store;
	GtkTreeIter iter, iter2;
	gchar *name;
	HildonShortenNamesDialogPrivate *priv = HILDON_SHORTEN_NAMES_DIALOG_GET_PRIVATE(dialog);

	store = gtk_list_store_new(1, G_TYPE_STRING);
	gtk_tree_model_get_iter_first(GTK_TREE_MODEL(priv->store), &iter);
	do {
		gtk_list_store_append(GTK_LIST_STORE(store), &iter2);
		gtk_tree_model_get(GTK_TREE_MODEL(priv->store), &iter, 1, &name, -1);
		gtk_list_store_set(GTK_LIST_STORE(store), &iter2, 0, name, -1);
	}
	while (gtk_tree_model_iter_next(GTK_TREE_MODEL(priv->store), &iter));
	g_free(name);
	return (GtkTreeModel *) store;
}

/**
* hildon_shorten_names_dialog_get_automatic:
* @dialog: the shorten names dialog.
*
* Return value: A gboolean stating whether toggle button is active.
*/

gboolean hildon_shorten_names_dialog_get_automatic(HildonShortenNamesDialog * dialog)
{
	HildonShortenNamesDialogPrivate *priv = HILDON_SHORTEN_NAMES_DIALOG_GET_PRIVATE(dialog);
	return gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(priv->is_automatic_button));
}
