#!/usr/bin/python

import unittest
import sys
import time
from setup import *
from dbus import DBusException

# Test cases for hss, to be run on the PC. Requirements:
# 1) ssh configured to connect to device as 'device' without password
# 2) dbus forwarded (using e.g. sockfwd and dbus-forward.sh from multimedia
#    svn) and the dbus environment variables appropriately set
# 3) this script run using fakeroot
#
# Alternatively it should be possible to run the script directly on the
# device if python is installed, but that hasn't been tried

# If run without arguments, a simple API test will be run
# If run as 'test.py full', includes tests that require user interaction.

# TODO: listening tests (to check that audio goes where it's supposed to)
# and headset button tests not done yet

# NOTE: Currently all tests that access enable_det are failing because
# it's suddenly gone from the kernel again

class HssBasicAPITest(unittest.TestCase):
    """Standalone API tests that can always be run"""

    def test00_LoudspeakerOn(self):
        loudspeaker.force_loudspeaker_on()

    def test01_LoudspeakerOff(self):
        loudspeaker.force_loudspeaker_off()

    def test02_LoudspeakerOnDouble(self):
        loudspeaker.force_loudspeaker_on()
        loudspeaker.force_loudspeaker_on()

    def test03_LoudspeakerOffTriple(self):
        loudspeaker.force_loudspeaker_off()
        loudspeaker.force_loudspeaker_off()
        loudspeaker.force_loudspeaker_off()

    def test10_ListenButton(self):
        headset.listen_button()

    def test11_IgnoreButton(self):
        headset.ignore_button()

    def test12_ListenButtonDouble(self):
        headset.listen_button()
        headset.listen_button()

    def test13_IgnoreButtonTriple(self):
        headset.ignore_button()
        headset.ignore_button()
        headset.ignore_button()

    def test12_RequestButton(self):
        button.request()
        button.release(0)

    def test13_ReleaseButtonNotHeld(self):
        self.assertRaises(DBusException, button.release, 0)

    def test14_RequestButtonTwice(self):
        button.request()
        button.request()
        button.release(0)
        button.release(0)
        self.assertRaises(DBusException, button.release, 0)

    def test20_RequestRecord(self):
        record.request()
        record.release(0)

    def test21_ReleaseRecordNotHeld(self):
        self.assertRaises(DBusException, record.release, 0)

    def test22_RequestRecordTwice(self):
        record.request()
        record.request()
        record.release(0)
        record.release(0)
        self.assertRaises(DBusException, record.release, 0)

    def test23_RequestRecordMute(self):
        assert test_capture_switch() == False
        record.request()
        assert test_capture_switch() == True
        record.release(0)
        assert test_capture_switch() == False

class HssConnectHeadphoneTest(unittest.TestCase):
    """Tests that require connecting the headphones"""
    message = 'connect HEADPHONES now'
    limit = HOOKDET_POLL_LIMIT * HOOKDET_DETECT_INTERVAL / 1000

    def test00_Connect(self):
        if headphone_state() == "connected":
            sys.stderr.write('\n[disconnect plug]')
            wait_headphones("disconnected")
        assert set_master_volume(100) == 100
        sys.stderr.write('\n[%s]\n' % self.message)
        wait_headphones("connected")

    def test01_PollingEnabled(self):
        assert int(get_sysfs(RETU_ENABLE)) == 1

    def test02_EnableDet(self):
        try:
            enable_det = int(get_sysfs(RETU_ENABLE_DET))
        except:
            enable_det = 0
        assert enable_det == 0

    def test10_VolumeLimited(self):
        assert get_master_volume() == VOLUME_HEADPHONE

    def test11_ForceLoudspeakerOn(self):
        volume = VOLUME_HEADPHONE + 1
        assert set_master_volume(volume) == volume
        loudspeaker.force_loudspeaker_on()
        assert get_master_volume() > VOLUME_HEADPHONE

    def test12_ForceLoudspeakerOff(self):
        assert get_master_volume() > VOLUME_HEADPHONE
        loudspeaker.force_loudspeaker_off()
        assert get_master_volume() == VOLUME_HEADPHONE

    def test20_WaitDetectionEnd(self):
        for i in range(self.limit):
            enable = int(get_sysfs(RETU_ENABLE))
            if enable == 0: break
            time.sleep(1)
        assert enable == 0

class HssConnectHeadsetTest(HssConnectHeadphoneTest):
    """Tests that require connecting the headset"""
    message = 'connect HEADSET now'
    limit = 1

    # FIXME: enable could already be off in test01, OR enable_det might
    # not be on yet if plug is inserted slowly. Change the tests and/or
    # their order
    def test02_EnableDet(self):
        enable_det = int(get_sysfs(RETU_ENABLE_DET))
        assert enable_det == 1

    def test21_EnableDet(self):
        enable_det = int(get_sysfs(RETU_ENABLE_DET))
        assert enable_det == 1

class HssDisconnectHeadsetTest(unittest.TestCase):
    """Tests that require disconnecting the headset"""

    def test10_EnableRecord(self):
        record.request()
        assert int(get_sysfs(RETU_ENABLE)) == 1
        assert test_capture_switch() == 1

    def test20_Disconnect(self):
        assert headphone_state() == "connected"
        sys.stderr.write('\n[disconnect plug]\n')
        wait_headphones("disconnected")
        assert int(get_sysfs(RETU_ENABLE)) == 0

    def test21_CaptureStillOn(self):
        assert test_capture_switch() == 1

    def test22_NoEnableDet(self):
        assert int(get_sysfs(RETU_ENABLE_DET)) == 0

    def test30_DisableRecord(self):
        record.release(0)
        assert int(get_sysfs(RETU_ENABLE)) == 0
        assert test_capture_switch() == 0

class HssHeadphoneTest(unittest.TestCase):
    """Tests that require headphones connected"""

    def test00_Check(self):
        assert headphone_state() == "connected"
        assert int(get_sysfs(RETU_ENABLE_DET)) == 0

    def test10_RequestRecord(self):
        assert test_capture_switch() == False
        record.request()
        assert test_capture_switch() == True
        assert int(get_sysfs(RETU_ENABLE)) == 0

    def test11_ReleaseRecord(self):
        assert test_capture_switch() == True
        record.release(0)
        assert test_capture_switch() == False

    def test20_ButtonInterface(self):
        headset.listen_button()
        headset.ignore_button()

class HssHeadsetTest(unittest.TestCase):
    """Tests that require headset connected"""

    def test00_Check(self):
        assert headphone_state() == "connected"
        assert int(get_sysfs(RETU_ENABLE_DET)) == 1

    def test10_RequestRecord(self):
        assert test_capture_switch() == False
        record.request()
        assert test_capture_switch() == True
        assert int(get_sysfs(RETU_ENABLE)) == 1

    def test11_ReleaseRecord(self):
        assert test_capture_switch() == True
        record.release(0)
        assert test_capture_switch() == False
        assert int(get_sysfs(RETU_ENABLE)) == 0

    def test20_RequestButton(self):
        assert int(get_sysfs(RETU_ENABLE)) == 0
        headset.listen_button()
        assert int(get_sysfs(RETU_ENABLE)) == 1

    def test21_CheckEnableDet(self):
        assert int(get_sysfs(RETU_ENABLE_DET)) == 1

    def test22_ReleaseButton(self):
        assert int(get_sysfs(RETU_ENABLE)) == 1
        headset.ignore_button()
        assert int(get_sysfs(RETU_ENABLE)) == 0

    def test21_CheckEnableDet(self):
        assert int(get_sysfs(RETU_ENABLE_DET)) == 1


class HssHeadsetButtonTest(unittest.TestCase):
    """Tests that require pressing the button"""
    # not done yet
    pass


if __name__ == '__main__':
    loader = unittest.defaultTestLoader

    suites = [ loader.loadTestsFromTestCase(HssBasicAPITest) ]

    if len(sys.argv) > 1 and sys.argv[1] == 'full':
        suites += [ loader.loadTestsFromTestCase(HssConnectHeadsetTest),
                    loader.loadTestsFromTestCase(HssHeadsetTest),
                    loader.loadTestsFromTestCase(HssHeadsetButtonTest),
                    loader.loadTestsFromTestCase(HssDisconnectHeadsetTest),
                    loader.loadTestsFromTestCase(HssConnectHeadphoneTest),
                    loader.loadTestsFromTestCase(HssHeadphoneTest) ]

    runner = unittest.TextTestRunner()

    result = runner.run(unittest.TestSuite(suites))

    sys.exit(not result.wasSuccessful())
