/**
 * Copyright (C) 2005-2007 Nokia Corporation
 * Contact: Makoto Sugano <makoto.sugano@nokia.com>
 *
 */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <signal.h>
#include <unistd.h>
#include <gconf/gconf-client.h>

#if HAVE_DSP_INTERFACE_COMMON_H
# include <dsp/audiopp_dsptask.h>
#endif

#include "mvi.h"

#include "common.h"
#include "route.h"
#include "hss.h"

enum accessory_e {
  ACCESSORY_NONE,
  ACCESSORY_HEADSET,
  ACCESSORY_HEADPHONES
};

enum headset_e {
  HEADSET_ERROR,
  HEADSET_STD,
  HEADSET_HS48,
  HEADSET_HS48_TRANSITION,
  HEADSET_HS48_BUTTON,
  HEADSET_NONE,
  HEADSET_INVALID
};

typedef struct {
  enum headset_e	last_result;
  int			confidence;
  int			total_polls;
} headset_detect_t;

static gboolean 	_update_audio_route	(void);
static gboolean 	_set_audio_route	(int		route);
static gboolean 	_set_retuhs_enable	(gboolean	state);
static gboolean 	_update_retuhs		(void);
static void		_set_enable_det		(gboolean	state);
static void		_update_mic_power	(void);
static void		_set_accessory_type	(enum accessory_e accessory);
static enum headset_e	_read_hookdet		(void);
static gboolean		_poll_hookdet		(headset_detect_t *headset);
static void		_set_polling		(int		interval, gboolean	reset);
static void		_update_button_polling	(void);
static void		_set_button_state	(gboolean	pressed);
static gboolean		_write_sysfs		(const gchar	*file, const gchar	*value);
static void		_gconf_master_volume_cb	(GConfClient	*client, guint		cnxn_id,
						 GConfEntry	*entry, gpointer	user_data);
#if HAVE_DSP
static dsp_info_t	*dsp = NULL;
#endif
static int		current_audio_route = -1;
static enum accessory_e	current_accessory = -1;
static gboolean		force_loudspeaker = FALSE;
static gboolean		hp_poll_button = FALSE;
static gboolean		mic_powersave = TRUE;
static gboolean		polling = FALSE;
static guint		volume_notify_id = 0;

#define HAVE_770

#define VOLUME_MAX	100
#define VOLUME_MIN	-101

#define SYSFS_RETUHS_ENABLE		"/sys/devices/platform/retu-headset/enable"
#define SYSFS_RETUHS_ENABLE_DET		"/sys/devices/platform/retu-headset/enable_det"
#define SYSFS_RETUHS_HOOKDET		"/sys/devices/platform/retu-headset/hookdet"

#define HOOKDET_DETECT_INTERVAL		50
#define HOOKDET_BUTTON_INTERVAL		100
#define HOOKDET_POLL_LIMIT		100
#define HOOKDET_CONFIDENCE_THRESHOLD	10

#define VOLUME_HEADPHONE	85
#define VOLUME_LOUDSPEAKER	100

#undef ULOG_CRIT
#define ULOG_CRIT g_debug

/**
 * route_initialize:
 * Return: TRUE on success
 */
gboolean 
route_initialize	(dsp_info_t	*dspinfo)
{
  GConfClient		*client; 
  GError		*err = NULL;

#if HAVE_DSP
  assert (dspinfo != NULL);

  dsp = dspinfo;
#endif

  if (MVI_open () < 0) {
    ULOG_CRIT ("failed to open MVI");
    return FALSE;
  }

  client = gconf_client_get_default ();
  g_return_val_if_fail (client != NULL, FALSE);
  gconf_client_add_dir (client, ROUTE_GCONF_MASTER_VOLUME_PATH, GCONF_CLIENT_PRELOAD_NONE, NULL);
  volume_notify_id = gconf_client_notify_add (client, ROUTE_GCONF_MASTER_VOLUME_PATH, _gconf_master_volume_cb, NULL, NULL, &err);
  if (err != NULL) {
    ULOG_CRIT ("Could not register gconf volume update-cb: %s", err->message);
    g_error_free (err);
    return FALSE;
  }
  g_object_unref (client);

  return TRUE;
}

/**
 * route_deinitialize:
 */
void 
route_deinitialize	(void)
{
  _set_retuhs_enable	(FALSE);
  _set_enable_det	(FALSE);

  MVI_close ();
}

#ifdef HAVE_770

#define STATUS_ERROR -1
#define STATUS_OK 0

static int hp_speaker_set_audio_route(int audio_route)
{
#if 0
  int result = -1;

  g_debug ("hp_speaker_set_audio_route");

  if (current_audio_route == audio_route) {
    return STATUS_OK;
  }

  else if (audio_route == HEADPHONE) {
    result = MVI_setState(HEADPHONE);
    g_debug ("hp_speaker_set_audio_route - HEADPHONE");
  }
  else  if (audio_route == LOUDSPEAKER) {
    result = MVI_setState(LOUDSPEAKER);
    g_debug ("hp_speaker_set_audio_route - LOUDSPEAKER");
  }
  else {
    ULOG_DEBUG("HSS: failed to set audio route");
    return STATUS_ERROR;
  }

  if (result != 0) {
    return STATUS_ERROR;
  }

  current_audio_route = audio_route;
#endif

  return STATUS_OK;
}
#endif

/**
 * route_set_hp_plugged:
 *
 * @state: TRUE if the plug was connected, FALSE if disconnected.
 *
 * Changes the hp/speaker state accordingly (updates the accessory
 * type, starts or stops headset detection).
 */
void 
route_set_hp_plugged	(gboolean plugged)
{
  static int is_plugged = -1;

  if (plugged != is_plugged) {
    ULOG_DEBUG("route_set_hp_plugged (%d)", plugged);

    is_plugged = plugged;

    if (is_plugged) {
      _set_accessory_type (ACCESSORY_HEADPHONES);
#ifdef HAVE_770
      hp_speaker_set_audio_route (HEADPHONE);
#else
      _set_polling (HOOKDET_DETECT_INTERVAL, TRUE);
#endif
    } else {
      _set_accessory_type (ACCESSORY_NONE);
#ifdef HAVE_770
      hp_speaker_set_audio_route (LOUDSPEAKER);
#else
      _set_polling (0, TRUE);
#endif
    }
  }
}

/**
 * mic_set_powersave:
 *
 * @state: TRUE to enable or FALSE to disable power saving.
 *
 * Updates external mic bias and/or input path muting.
 */
void 
mic_set_powersave	(gboolean powersave)
{
  if (powersave != mic_powersave) {
    ULOG_DEBUG("route_set_mic_powersave(%d)", powersave);

    mic_powersave = powersave;
#ifndef HAVE_770
    _update_mic_power();
#endif
  }
}

static void 
_update_mic_power	(void)
{
  ULOG_DEBUG("route_update_mic_power(): powersave=%d", mic_powersave);

#ifndef HAVE_770
  _update_retuhs ();
#endif
  if (mic_powersave || (!mic_powersave && route_get_gconf_volume() > 0))
    MVI_setMicMute (mic_powersave);
}

gboolean 
route_force_loudspeaker		(gboolean force)
{
  if (force_loudspeaker == force)
    return TRUE;

  ULOG_DEBUG("route_force_loudspeaker(%d)", force);
  force_loudspeaker = force;

  return _update_audio_route ();
}

gboolean 
route_force_bt			(gboolean force)
{
  gboolean	rv;

  rv = MVI_setBT (force);
#if HAVE_DSP
  if (force && rv == 0) {
    dsp_set_mode (dsp, DSP_MODE_BT);
  }
#endif
  if (force == FALSE) {
    current_audio_route = -1; /* *crap* to force update_audio_route.. */
    _update_audio_route();
  }
  return rv;
}

/**
 * hp_poll_button:
 *
 * @state: TRUE to enable or FALSE to disable button polling
 *
 * When enabled, hookdet is polled for button events also after headset
 * type has been determined.
 */
void 
hp_set_poll_button	(gboolean poll_button)
{
  if (poll_button != hp_poll_button) {
    hp_poll_button = poll_button;
#ifndef HAVE_770
    _update_button_polling ();
#endif
  }
}

static void 
_update_button_polling	(void)
{
  int interval;

  if (current_accessory == ACCESSORY_HEADSET) {
    interval = (hp_poll_button ? HOOKDET_BUTTON_INTERVAL : 0);
    _set_polling (interval, FALSE);
  }
}

/**
 * sets the currently connected accessory 
 */
static void 
_set_accessory_type		(enum accessory_e type)
{
  if (type == current_accessory) 
    return;

  ULOG_DEBUG("route_set_accessory_type(%d)", type);

  current_accessory = type;

#ifndef HAVE_770
  /* FIXME: move this somewhere else */
  _set_enable_det (type == ACCESSORY_HEADSET);
#endif
  _update_audio_route ();
}
    
/**
 * _update_audio_route:
 *
 * Main audio policy logic. Sets audio route according to the current
 * force status and the type of the connected accessory (if any)
 */
static gboolean 
_update_audio_route		(void)
{
  int route;

  if (force_loudspeaker) {
    route = LOUDSPEAKER;
  } else if (current_accessory == ACCESSORY_HEADSET) {
    route = HEADSET;
  } else if (current_accessory == ACCESSORY_HEADPHONES) {
    route = HEADPHONE;
  } else {
    route = LOUDSPEAKER;
  }

  ULOG_DEBUG("route_update_audio_route(): %d %d %d",
	     force_loudspeaker, current_accessory, route);

  return _set_audio_route(route);
}

/**
 * _set_audio_route:
 *
 * @audio_route: the audio route (one of HEADPHONE, HEADSET or LOUDSPEAKER,
 * as defined in mvilib.h)
 *
 * Sets all audio route related parameters.
 *
 * Return value: returns TRUE on success and FALSE on error.
 */
static gboolean 
_set_audio_route	(int audio_route)
{
  int			volume;
#if HAVE_DSP
  enum e_dsp_mode	dsp_mode = -1;
#endif

  if (current_audio_route == audio_route) return TRUE;
  ULOG_DEBUG("route_set_audio_route(%d)", audio_route);

  volume = route_get_gconf_volume();
  g_debug ("*********** JV - volume: %d", volume);
  /* If switching from non-headphone to headphone (or headset), volume
   * needs to be limited to a safe range. Note that this is now done also
   * when returning from loudspeaker force mode.
   */
  if ((audio_route == HEADPHONE || audio_route == HEADSET) &&
      (current_audio_route != HEADPHONE && current_audio_route != HEADSET)) {

    if (volume != -101) {  // special case: -101 = muted zero
      int multiplier = volume < 0 ? -1 : 1;

      if (abs (volume) > VOLUME_HEADPHONE) {
	volume = VOLUME_HEADPHONE * multiplier;
      }
    }
  }

#if HAVE_DSP
  if (audio_route == HEADSET) {
    dsp_mode = DSP_MODE_HS;
  }
  else if (audio_route == HEADPHONE) {
    dsp_mode = DSP_MODE_HP;
  }
  else if (audio_route == LOUDSPEAKER) {
    dsp_mode = DSP_MODE_IHF;
  }
  else {
    ULOG_ERR("unsupported audio route");
    return FALSE;
  }
#endif	/* HAVE_DSP */

  if (MVI_setState(audio_route) != 0) {
    ULOG_ERR("MVI_setState failed");
    return FALSE;
  }

  /* Must always set volume after MVI_setState (even if it wasn't changed
   * by limiting) because the volume table is different for the new state */
  if (MVI_setVolume(volume) == 0) {
    ULOG_DEBUG("  set volume: %d", volume);
    route_set_gconf_volume(volume);
  } else {
    ULOG_ERR("MVI_setVolume failed");
  }

#if HAVE_DSP
  dsp_set_mode(dsp, dsp_mode);
#endif

  current_audio_route = audio_route;
  _update_mic_power();

  return TRUE;
}

/**
 * _read_hookdet:
 *
 * Reads and interprets the value of hookdet.
 *
 * Return value: HEADSET_STD, HEADSET_HS48, HEADSET_HS48_BUTTON,
 * HEADSET_HS48_TRANSITION, HEADSET_INVALID, or HEADSET_ERROR.
 */
static enum headset_e
_read_hookdet	(void)
{
  enum headset_e type = HEADSET_ERROR;
  char buf[10] = { 0, };
  int val;
  int fd;

  fd = open(SYSFS_RETUHS_HOOKDET, O_RDONLY);
  if (fd >= 0) {
    read(fd, buf, sizeof(buf)-1);
    close(fd);
  }

  if (sscanf(buf, "%d", &val) == 1) {
    /* NOTE: Levels are hardware dependent */
    if (val <= 53) {
      type = HEADSET_STD;
    } else if (56 <= val && val <= 75) {
      type = HEADSET_HS48_BUTTON;
    } else if (76 <= val && val <= 499) {
      type = HEADSET_HS48_TRANSITION;
    } else if (500 <= val && val <= 820) {
      type = HEADSET_HS48;
    } else if (900 <= val) {
      type = HEADSET_NONE;
    } else {
      type = HEADSET_INVALID;
    }
  } else {
    ULOG_ERR("Could not read a valid value from %s", SYSFS_RETUHS_HOOKDET);
  }

  return type;
}

/**
 * route_poll_hookdet
 *
 * @headset: pointer to the headset detection data
 *
 * g_timeout callback that polls the hookdet value and interprets it to
 * determine headset type (standard/HS-48) and possible button events.
 *
 * Return value: always TRUE (run until removed by _set_polling())
 */
static gboolean 
_poll_hookdet		(headset_detect_t *headset)
{
  enum headset_e poll;
  gboolean detecting;

  poll = _read_hookdet();
  if (poll == HEADSET_ERROR) {
    return FALSE;
  }

  detecting = (headset->confidence < HOOKDET_CONFIDENCE_THRESHOLD);

  if (detecting) {
    if (poll == HEADSET_HS48) {
      headset->confidence++;
      if (headset->confidence == HOOKDET_CONFIDENCE_THRESHOLD) {
	ULOG_INFO("headset is HS-48");
	_set_accessory_type (ACCESSORY_HEADSET);
	_update_button_polling ();
      }
    }
    else if (poll == HEADSET_NONE) {
      headset->confidence++;
      if (headset->confidence == HOOKDET_CONFIDENCE_THRESHOLD) {
        ULOG_INFO("headset is none");
_set_accessory_type (ACCESSORY_HEADSET);
/*_update_button_polling ();*/
      }
    }
    else {
      headset->confidence = 0;
    }
  }
  
  if (current_accessory == ACCESSORY_HEADSET && !detecting) {
    if (headset->last_result == HEADSET_HS48_BUTTON &&
	(poll == HEADSET_HS48_BUTTON || poll == HEADSET_HS48)) {
      _set_button_state(TRUE);
      /* Last-minute fix for NB#43322: in addition to redetecting on
       * invalid values (see below), button down events are delayed
       * by one poll cycle, and will only be sent if the next poll is
       * valid. [For short clicks the down and up events may now be
       * sent on the same poll, but this shouldn't be a problem] */
    }
    if (poll == HEADSET_HS48_BUTTON) {
      /* already handled */
    } else if (poll == HEADSET_HS48) {
      _set_button_state(FALSE);
    } else if (poll == HEADSET_HS48_TRANSITION &&
	       headset->last_result != HEADSET_HS48_TRANSITION) {
      /* Values between the 'button up' and 'button down' states are
       * sometimes seen when the button is pressed. When it happens,
       * we try again once on the next poll before failing */
    } else {
      /* Unplugging the headset may cause random values (NB#43322). To
       * prevent bogus button events, polling is disabled for a while */
      ULOG_INFO("invalid hookdet value, forcing redetection");
      _set_button_state(FALSE);
      headset->confidence = 0;
      /* The poll interval could be reset to the original 50 ms here
       * (instead of 100 ms which is used for button polling), but
       * being slower on redetection is probably good because it
       * allows more time for plug disconnect */
    }
  }

  if (current_accessory == ACCESSORY_HEADPHONES ||
      (current_accessory == ACCESSORY_HEADSET && poll == HEADSET_NONE)) {
    if (headset->total_polls >= HOOKDET_POLL_LIMIT) {
      /* Give up if headset not detected within a certain time.
       * (TODO: should this also affect redetection?) */
      ULOG_INFO("poll limit reached");
      _set_polling(0, FALSE);
    } else {
      headset->total_polls++;
    }
  }

  headset->last_result = poll;
  return TRUE;
}


/**
 * _set_polling
 *
 * @interval: time between two polls in milliseconds, or 0 to stop polling
 * @reset: TRUE to reset headset detection
 *
 * Starts or stops polling hookdet.
 */
static void 
_set_polling		(int interval, gboolean reset)
{
  static guint source = 0;
  static int current_interval = 0;
  static headset_detect_t headset;

  if (reset == TRUE) {
    headset.last_result = 0;
    headset.confidence = 0;
    headset.total_polls = 0;
  }

  if (interval == 0) {
    /* forget last value when polling is stopped (quick fix to avoid
     * a delayed button down event when restarted) */
    headset.last_result = 0;
  }

  if (interval == current_interval) return;
  ULOG_DEBUG("route_set_polling (%d,%d)", interval, reset);

  if (current_interval != 0) {
    g_source_remove(source);
    source = 0;
  }
  if (interval > 0) {
    source = g_timeout_add(interval,
			   (GSourceFunc) _poll_hookdet,
			   &headset);
    if (source <= 0) {
      ULOG_ERR("g_timeout_add() failed");
    }
  }

  if (source > 0) {
    polling = TRUE;
    current_interval = interval;
  } else {
    polling = FALSE;
    current_interval = 0;
  }
#ifndef HAVE_770
  _update_retuhs (); // check success?
#endif
  if (!polling) _set_button_state(FALSE);
}

/**
 * _set_button_state:
 * @pressed: the current button state
 *
 * Signals a button event when pressed or released.
 */
static void 
_set_button_state(gboolean	pressed)
{
  static gboolean button_state = FALSE;

  if (pressed != button_state) {
    button_state = pressed;
    pressed ? hss_signal_button ("button_pressed") : hss_signal_button ("button_released");
  }
}

static void 
_set_enable_det	(gboolean	state)
{
  static int current_state = -1;

  if (state == current_state) return;
  ULOG_DEBUG("route_set_enable_det(%d)", state);

  if (_write_sysfs(SYSFS_RETUHS_ENABLE_DET, state ? "1" : "0"))
    current_state = state;
}

static gboolean 
_set_retuhs_enable		(gboolean	state)
{
  static int current_state = -1;

  if (state == current_state) 
    return TRUE;

  ULOG_DEBUG("_set_retuhs_enable(%d)", state);

  if (_write_sysfs (SYSFS_RETUHS_ENABLE, state ? "1" : "0")) {
    current_state = state;
    return TRUE;
  } else {
    return FALSE;
  }
}

static gboolean 
_update_retuhs		(void)
{
  gboolean state;

  state = (polling || (current_audio_route == HEADSET && !mic_powersave));

  return _set_retuhs_enable	(state);
}

static gboolean 
_write_sysfs	(const gchar	*file, 
		 const gchar	*value)
{
  int fd, len, n;

  fd = open (file, O_WRONLY);
  if (fd >= 0) {
    len = strlen (value);
    n = write (fd, value, len);
    close (fd);
  }
  if (fd < 0) {
    ULOG_ERR("error opening %s", file);
  } else if (n != len) {
    ULOG_ERR("error writing %s", file);
  } else {
    return TRUE;
  }
  return FALSE;
}

int 
route_get_gconf_volume		(void)
{
  GError *error = NULL;
  GConfClient *client;
  gint volume = 80;

  /* volume will be 0 if get_int fails */
  /* fixme: no way to make exception in this function */

  client = gconf_client_get_default ();
  g_return_val_if_fail (client != NULL, 0);

  volume = gconf_client_get_int (client, ROUTE_GCONF_MASTER_VOLUME_PATH, &error);
  g_object_unref (client);

  if (error != NULL) {
    ULOG_ERR ("cannot get volume: %s", error->message);
    g_error_free (error);
    return volume;
  }

  volume = CLAMP (volume, VOLUME_MIN, VOLUME_MAX);
  
  return volume;
}

void 
route_set_gconf_volume		(int volume)
{
  GError *error = NULL;
  GConfClient *client; 

  if (volume > 100)
    volume = 100;

  client = gconf_client_get_default ();
  g_return_if_fail (client != NULL);

  gconf_client_set_int (client, ROUTE_GCONF_MASTER_VOLUME_PATH, volume, &error);
  if (error != NULL) {
    ULOG_ERR ("cannot set volume: %s", error->message);
    g_error_free (error);
    error = NULL;
  }

  g_object_unref (client);
}

static void		
_gconf_master_volume_cb		(GConfClient	*client, guint		cnxn_id,
				 GConfEntry	*entry, gpointer	user_data)
{
  int		volume;

  volume = route_get_gconf_volume ();
  if (MVI_getBT () == FALSE) {
    MVI_setVolume (volume);
    if (mic_powersave || (!mic_powersave && volume > 0))
      MVI_setMicMute (mic_powersave);
  }
}
