/* resource.c
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved
 *
 * Contact: Makoto Sugano <makoto.sugano@nokia.com>
 *
 *
 * This file implements a resource manager in accordance with the
 * Multimedia Resource Management (RM) Architecture Proposal.
 *
 * The implementation is based on resource handlers which can be added
 * with add_managed_resource(name, path, signature, handler).
 *
 * When a managed resource is requested or released, its handler is
 * called as handler(action, resource, result, args), where:
 *
 *   action    is RESOURCE_ACTION_REQUEST or RESOURCE_ACTION_RELEASE
 *   resource  is the resource name (D-Bus object path)
 *   result    is a pointer to a ResourceResult struct
 *   args      is a pointer to the list of parameters passed to request()
 *
 * The handler must return TRUE if it can handle the named resource (this
 * is for registering handlers for object hierarchies). For a REQUEST
 * action, the handler must also fill in the ResourceResult fields as
 * specified in the RM Architecture Proposal. (The handler must return
 * TRUE even if resource allocation fails but the request was for a valid
 * resource.)
 *
 * Currently result->reason is expected to be a static string (not freed
 * by the resource manager).
 *
 * The args list is currently always NULL (request method arguments are
 * not implemented yet). The idea is that, when implemented, the resource
 * manager can parse message args into an array of void pointers (based
 * on the signature specified when the handler was added).
 *
 * For a RELEASE action, the handler gets the resource handle in
 * result->handle [yes, this is kind of ugly]. The handler is not expected
 * to do anything with the other result fields, and releasing a resource
 * must always succeed (the handler is also called if the resource holder
 * dies, in which case there wouldn't be anyone to handle a return status
 * anyway). A handler is only called to release a resource which it has
 * succesfully granted before.
 *
 * Additionally, the action RESOURCE_ACTION_UNREGISTER is defined and will
 * be used if the resource is removed from the manager. Currently there is
 * no remove_managed_resource() so resources are never removed.
 *
 * (Defining a RESOURCE_ACTION_INIT might also be necessary: Although
 * handler initialization can be postponed until the first request, some
 * types of resources might be slow enough to initialize that it would be
 * better done immediately)
 *
 * Current shortcomings and limitations:
 *
 *  - Service names are not honored (any managed resource can be requested
 *    from any bus name held by HSS)
 *
 *  - Resource request arguments are not supported (to be implemented when
 *    actually needed). Currently request() signatures are not enforced
 *    at all (the method should really fail if it's called with args)
 *
 *  - Resource hierarchies can't be registered (supported in principle but
 *    not enabled yet as nothing is using them)
 *
 *  - It's possible to try adding a handler for the same resource twice
 *    (what happens then is undefined)
 *
 *  - No locking is done (check if it's needed - can there be D-Bus calls
 *    be handled by multiple threads?)
 *
 *  - Generally there could be more sanity checks done. (Needs to be improved
 *    at least if RM develops toward a plugin architecture)
 *
 */
#include <glib.h>
#include <dbus/dbus.h>

#include "common.h"
#include "resource.h"
#include "route.h"

/* not sure yet if method calls may come from non-unique bus names */
#define CHECK_NAME_UNIQUENESS

#define RESOURCE_INTERFACE     "com.nokia.osso_resource_manager"
#define ERROR_INVALID_SENDER   RESOURCE_INTERFACE ".InvalidSender"
#define ERROR_NOT_HOLDING      RESOURCE_INTERFACE ".NotHoldingResource"

#define MATCH_RULE_NAMEOWNER   "type='signal',"	\
  "sender='org.freedesktop.DBus',"		\
  "interface='org.freedesktop.DBus',"		\
  "member='NameOwnerChanged',"			\
  "arg2=''"

typedef struct {
  GQuark           resource;
  gint             handle;
  ResourceHandler  handler;
} AllocatedResource;

static gboolean          request_name        (const gchar       *name);

static void              resource_unregister (DBusConnection    *connection,
                                              void              *user_data);

static DBusHandlerResult resource_handler    (DBusConnection    *connection,
                                              DBusMessage       *message,
                                              void              *user_data);

static DBusMessage      *resource_request    (DBusMessage       *message,
                                              ResourceHandler    handler);

static DBusMessage      *resource_release    (DBusMessage       *message);

static DBusHandlerResult nameowner_handler   (DBusConnection    *connection,
                                              DBusMessage       *message,
                                              void              *user_data);

static gboolean          request             (const gchar       *sender,
                                              const gchar       *resource,
                                              ResourceHandler    handler,
                                              ResourceResult    *result,
                                              gpointer          *args);

static void              release             (AllocatedResource *r);

static gboolean          release_one         (const gchar       *sender,
                                              const gchar       *resource,
                                              gint               handle);

static void              release_all         (const gchar       *sender);

static gboolean		add_nameowner_handler (DBusConnection *bus);

static GHashTable      *service_names;
static GHashTable      *resource_holders;
static DBusConnection  *resource_bus;

static const DBusObjectPathVTable vtable = {
  resource_unregister,
  resource_handler,
};

gboolean 
button_resource_handler		(ResourceAction		action,
				 const gchar *		resource,
				 ResourceResult *	result,
				 gpointer *		args)
{
  static guint count = 0;

  if (action == RESOURCE_ACTION_REQUEST) {
    if (++count == 1) {
      ULOG_DEBUG("button polling active");
      hp_set_poll_button(TRUE);
    }
  } else if (action == RESOURCE_ACTION_RELEASE && count > 0) {
    if (--count == 0) {
      ULOG_DEBUG("button polling inactive");
      hp_set_poll_button(FALSE);
    }
  }
  return TRUE;
}

gboolean 
record_resource_handler		(ResourceAction  action,
				 const gchar    *resource,
				 ResourceResult *result,
				 gpointer       *args)
{
  static int count = 0;
  if (action == RESOURCE_ACTION_REQUEST) {
    count++;
    ULOG_DEBUG("request record -> %d", count);
    if (count == 1) mic_set_powersave(FALSE);
  } else if (action == RESOURCE_ACTION_RELEASE) {
    count--;
    ULOG_DEBUG("release record -> %d", count);
    if (count == 0) mic_set_powersave(TRUE);
  }
  return TRUE;

}

/**
 * resource_init:
 * @bus: The D-Bus connection to use for resource management.
 *
 * Initializes the resource manager. Must be called prior to adding
 * any resources.
 *
 * Returns: TRUE if succesful.
 */
gboolean
resource_init (DBusConnection *bus)
{
  service_names     = g_hash_table_new (g_str_hash, g_str_equal);
  resource_holders  = g_hash_table_new (g_str_hash, g_str_equal);
  resource_bus      = bus;

  resource_manage ("com.nokia.osso_audio_pm", "/com/nokia/osso/pm/audio/record", "", record_resource_handler);
  resource_manage ("com.nokia.osso_hp_ls_controller", "/com/nokia/osso/headset/button", "", button_resource_handler);

  return add_nameowner_handler (bus);
}

/**
 * resource_manage:
 * @name: The service name associated with this type of resources
 * @path: The resource to manage (a D-Bus object path)
 * @signature: Message parameters expected by this resource
 * @handler: The handler function for requesting or releasing the resource
 *
 * Installs a handler for a new resource. (There is no remove_managed_resource;
 * all added resources are currently managed until multimediad exits)
 *
 * Returns: TRUE if succesful.
 */
gboolean
resource_manage		(const gchar     *name,
			 const gchar     *path,
			 const gchar     *signature,
			 ResourceHandler  handler)
{
  gboolean success = FALSE;

  if (signature && *signature != '\0') {
    ULOG_ERR ("Resource signatures currently not supported");
  }
  else if (!request_name(name)) {
    ULOG_ERR ("Failed to acquire name %s", name);
  }
  /* (Should work but untested)
     else if (g_str_has_suffix(path, "/")) {
     success = dbus_connection_register_fallback (resource_bus, path,
     &vtable, handler);
     }
  */
  else {
    g_quark_from_static_string (path);
    success = dbus_connection_register_object_path (resource_bus, path,
						    &vtable, handler);
    if (success) {
      ULOG_DEBUG ("Adding a handler for %s %s", path, "succeeded");
    } else {
      ULOG_ERR ("Adding a handler for %s %s", path, "failed");
    }
  }

  /* TODO: Would be nice if the handler could be given a user_data
   * pointer. This can be added together with support for signatures
   * and service name enforcing (then the message handler will need
   * to be passed more than one value anyway)
   */

  return success;
}

/*
  gboolean
  add_managed_refcount_resource (const gchar      *name,
  const gchar      *path,
  CallbackFunction  allocate,
  CallbackFunction  release,
  gpointer          user_data)
  {
  // create a handler that maintains a refcount of the managed resource
  // (probably with the default signature, and always using resource
  // handle 0) and calls allocate() when count becomes nonzero, release()
  // when zero
  }
*/

/**
 * resource_request_wrapper:
 * @resource: The resource to request
 * @sender: The D-Bus name to request the resource for
 * @handler: The handler function associated with the resource
 *
 * Gives access to the resource request internals. This function exists
 * purely to implement old HSS functionality in terms of resource management
 * and shouldn't be used for anything else.
 *
 * Returns: TRUE if the resource was granted.
 */
gboolean
resource_request_wrapper (const gchar     *resource,
                          const gchar     *sender,
                          ResourceHandler  handler)
{
  gboolean        was_handled;
  gboolean        success = FALSE;
  ResourceResult  result  = { 0, };

  was_handled = request (sender, resource, handler, &result, NULL);
    
  if (!was_handled) {
    ULOG_CRIT ("resource_request_wrapper: %s not handled", resource);
  }
  else if (result.status != RESOURCE_STATUS_OK) {
    ULOG_DEBUG ("resource_request_wrapper failed: %s", result.reason);
  }
  else if (result.handle != 0) {
    ULOG_CRIT ("resource_request_wrapper: got handle != 0 for %s. This is a bug", resource);
    release_one (sender, resource, result.handle);
  }
  else {
    success = TRUE;
  }
  /* g_free (result.reason); ? */

  return success;
}

/**
 * resource_release_wrapper:
 * @resource: The resource to release
 * @sender: The D-Bus name to release the resource for
 *
 * Gives access to the resource release internals. This function exists
 * purely to implement old HSS functionality in terms of resource management
 * and shouldn't be used for anything else.
 *
 * Returns: TRUE if the sender was holding the resource.
 */
gboolean
resource_release_wrapper (const gchar *resource,
                          const gchar *sender)
{
  gboolean was_held = release_one (sender, resource, 0);
  if (!was_held) {
    ULOG_DEBUG ("resource_release_wrapper: %s not held by %s",
		resource, sender);
  }
  return was_held;
}

/**
 * add_nameowner_handler:
 * @bus: The D-Bus connection to add the handler for.
 *
 * Adds a handler which releases all resources held by a name owner when
 * the owner disappears.
 *
 * This function is public mainly to support using resource_request_wrapper()
 * and resource_release_wrapper() for calls from another (i.e. session) bus.
 */
static gboolean
add_nameowner_handler	(DBusConnection *bus)
{
  gboolean  success = FALSE;
  DBusError error;

  dbus_error_init (&error);
  dbus_bus_add_match (bus, MATCH_RULE_NAMEOWNER, &error);

  if (dbus_error_is_set (&error)) {
    dbus_error_free (&error);
  } else {
    success = dbus_connection_add_filter (bus, nameowner_handler,
					  NULL, NULL);
    if (!success) {
      dbus_bus_remove_match (bus, MATCH_RULE_NAMEOWNER, NULL);
    }
  }

  return success;
}
                     

/*-------------------------- D-BUS ----------------------------*/

/* Requests a bus name, returns TRUE if succesful.
 */
static gboolean
request_name (const gchar *name)
{
  guint count = GPOINTER_TO_UINT (g_hash_table_lookup(service_names, name));

  /* Refcounting is (currently) basically just to optimize bus traffic,
   * as we don't release the names anyway
   */

  if (count == 0) {
    /* FIXME: weird stuff might happen if the name was already owned
     * and the queue/replacement flags were different. (Would really
     * need to GetNameOwner to see if we own it.) Resource handlers
     * should probably not be added with names that are being used
     * for something else
     */
    guint reply = dbus_bus_request_name (resource_bus,
					 name,
					 DBUS_NAME_FLAG_DO_NOT_QUEUE,
					 NULL);

    if (reply == DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
      count = 1;
    } else if (reply == DBUS_REQUEST_NAME_REPLY_ALREADY_OWNER) {
      count = 2;
    }
  } else {
    count++;
  }
    
  if (count > 0) {
    g_hash_table_insert (service_names,
			 (gpointer) name,
			 GUINT_TO_POINTER(count));
  }

  return (count > 0);
}

/* Called if a resource object is unregistered.
 */
static void
resource_unregister (DBusConnection  *connection,
                     void            *user_data)
{
  ResourceHandler handler = user_data;
  handler (RESOURCE_ACTION_UNREGISTER, NULL, NULL, NULL);
}

/* Called to handle resource object methods.
 */
static DBusHandlerResult
resource_handler (DBusConnection  *connection,
                  DBusMessage     *message,
                  void            *user_data)
{
  DBusMessage     *reply   = NULL;
  ResourceHandler  handler = user_data;

  if (dbus_message_get_type (message) == DBUS_MESSAGE_TYPE_METHOD_CALL &&
      dbus_message_has_interface (message, RESOURCE_INTERFACE)) {

    if (dbus_message_has_member (message, "request")) {
      /* check signature? */
      reply = resource_request (message, handler);
    }
    else if (dbus_message_has_member (message, "release") &&
	     dbus_message_has_signature (message, "i")) {
      reply = resource_release (message);
    }
  }

  if (reply) {
    dbus_connection_send (connection, reply, NULL);
    dbus_message_unref (reply);
    return DBUS_HANDLER_RESULT_HANDLED;
  } else {
    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
  }
}

/* Handles the request() method and returns a reply message, or NULL if the
 * request was not handled.
 */
static DBusMessage *
resource_request (DBusMessage     *message,
                  ResourceHandler  handler)
{
  DBusMessage    *reply  = NULL;
  ResourceResult  result = { 0, };

  const gchar *path   = dbus_message_get_path (message);
  const gchar *sender = dbus_message_get_sender (message);

  /* parse message args here if handlers with signatures are implemented */
  gpointer *args = NULL;

  ULOG_DEBUG ("request %s", path);

#ifdef CHECK_NAME_UNIQUENESS
  if (*sender != ':') {
    /* not sure if this can happen (if it can, we need to GetNameOwner) */
    ULOG_ERR ("Got a non-unique name: %s", sender);
    reply = dbus_message_new_error (message,
				    ERROR_INVALID_SENDER,
				    "Can only handle requests from unique connection names");
  } else
#endif
    if (request (sender, path, handler, &result, args)) {

      if (result.reason == NULL) {
	result.reason = "";
      }

      reply = dbus_message_new_method_return (message);

      dbus_message_append_args (reply,
				DBUS_TYPE_INT32,  &result.status,
				DBUS_TYPE_INT32,  &result.handle,
				DBUS_TYPE_STRING, &result.reason,
				DBUS_TYPE_INVALID);
      /* if (got_it_from_handler) g_free (result.reason); ? */
    }
  return reply;
}

/* Handles the release() method and returns a reply message.
 */
static DBusMessage *
resource_release (DBusMessage *message)
{
  DBusMessage  *reply;
  gint          handle;

  const gchar *path   = dbus_message_get_path (message);
  const gchar *sender = dbus_message_get_sender (message);

  ULOG_DEBUG ("release %s", path);

  dbus_message_get_args (message, NULL,
			 DBUS_TYPE_INT32, &handle,
			 DBUS_TYPE_INVALID);

  if (release_one (sender, path, handle)) {
    reply = dbus_message_new_method_return (message);
  } else {
    reply = dbus_message_new_error_printf (message,
					   ERROR_NOT_HOLDING,
					   "Not holding handle %d to resource %s",
					   handle, path);
  }
  return reply;
}

/* Releases resources held by a name owner if the name disappears.
 */
static DBusHandlerResult
nameowner_handler (DBusConnection  *connection,
                   DBusMessage     *message,
                   void            *user_data)
{
  const gchar *name;
  const gchar *old_owner;
  const gchar *new_owner;

  if (dbus_message_is_signal (message,
			      "org.freedesktop.DBus",
			      "NameOwnerChanged")

      && dbus_message_get_args (message, NULL,
				DBUS_TYPE_STRING, &name,
				DBUS_TYPE_STRING, &old_owner,
				DBUS_TYPE_STRING, &new_owner,
				DBUS_TYPE_INVALID)
      && (*name == ':')
      && (*new_owner == '\0')
      && g_hash_table_lookup (resource_holders, name)) {

    ULOG_DEBUG ("Owner of name %s disappeared", name);
    release_all (name);
  }

  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}


/*---------------- lowest-level resource handling  ---------------*/

/* Requests a resource from a handler, and updates the resource_holders
 * hash and the requester's array of held resources if succesful.
 * Returns TRUE if the request was handled.
 */
static gboolean
request (const gchar     *sender,
         const gchar     *resource,
         ResourceHandler  handler,
         ResourceResult  *result,
         gpointer        *args)
{
  gboolean           was_handled;
  GArray            *held_resources;
  AllocatedResource  r;

  was_handled = handler (RESOURCE_ACTION_REQUEST, resource, result, args);

  if (was_handled && result->status == RESOURCE_STATUS_OK) {

    held_resources = g_hash_table_lookup (resource_holders, sender);

    if (!held_resources) {
      held_resources = g_array_new(TRUE, TRUE, sizeof(AllocatedResource));
      g_hash_table_insert (resource_holders, g_strdup (sender),
			   held_resources);
    }
    r.resource = g_quark_from_string (resource);
    r.handle   = result->handle;
    r.handler  = handler;

    g_array_append_val (held_resources, r);
  }

  return was_handled;
}

/* Calls a handler to release a resource.
 */
static void
release (AllocatedResource *r)
{
  const gchar     *resource;
  ResourceHandler  handler;
  ResourceResult   result = { 0, };

  resource      = g_quark_to_string (r->resource);
  handler       = r->handler;
  result.handle = r->handle;

  handler (RESOURCE_ACTION_RELEASE, resource, &result, NULL);
}

/* Releases a resource, updating the requester's array of held resources.
 * Returns TRUE if the resource was held.
 */
static gboolean
release_one		(const gchar *sender,
			 const gchar *resource,
			 gint         handle)
{
  gboolean           	 was_held = FALSE;
  GArray             	*held_resources;
  AllocatedResource  	*r;
  int                	 i;
  GQuark             	 resource_q;

  held_resources = g_hash_table_lookup (resource_holders, sender);
  resource_q     = g_quark_try_string (resource);

  if (held_resources && resource_q) {
    for (i = 0; i < held_resources->len; i++) {
      r = &g_array_index (held_resources, AllocatedResource, i);
      if (r->resource == resource_q &&
	  r->handle   == handle) {
	release (r);
	g_array_remove_index_fast (held_resources, i);
	was_held = TRUE;
	break;
      }
    }
  }
  return was_held;
}

/* Releases all resources held by a sender, and removes the name from the
 * resource_holders hash.
 */
static void
release_all		(const gchar *sender)
{
  GArray             	*held_resources;
  gchar              	*orig_sender;
  AllocatedResource  	*r;
  int                	 i;

  g_hash_table_lookup_extended (resource_holders,
				sender,
				(gpointer) &orig_sender,
				(gpointer) &held_resources);
  if (held_resources) {
    for (i = 0; i < held_resources->len; i++) {
      r = &g_array_index (held_resources, AllocatedResource, i);
      release (r);
    }
    g_array_free (held_resources, TRUE);
    g_hash_table_remove (resource_holders, sender);
    g_free (orig_sender);
  }
}
