/*
** Playback manager
** (c) Nokia MM - Makoto Sugano, 2007
** FIXME: review dbus errors
** FIXME: assert | g_return ?
*/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif /* CONFIG_H_ */

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>

#ifdef HAVE_LIBPLAYBACK
# include <libplayback/playback.h>
#endif /* HAVE_LIBPLAYBACK */

#include "common.h"
#include "pb_manager.h"

#define DBUS_PB_MANAGER_PATH "/com/osso/playback/manager"

/*
** Warning! the manager does not take care of ref counting, beware!
** Warning: don't even consider calling these functions in a MT context.
*/
struct pb_manager_s {
  DBusConnection	*connection;
  GHashTable		*playback_proxy;		/* ref all the remote playback by sender-path */
};

typedef struct pb_playback_proxy_s		pb_playback_proxy_t;
typedef struct pb_playback_proxy_s		pb_playback_proxy_query_t;

struct pb_playback_proxy_s {
  pb_manager_t		*pb_manager;
  char			*connection_name;
  char			*path;
  enum pb_class_e	class;
  enum pb_state_e	state;
  enum pb_state_e	requested_state;
  DBusMessage		*requested_state_msg;
  int			allowed_state[PB_STATE_LAST];
};

typedef struct pb_pending_introspect_s {
  char			*sender;
  pb_manager_t		*manager;
} pb_pending_introspect_t;

typedef struct pb_pending_get_s {
  char			*property;
  pb_playback_proxy_t	*proxy;
} pb_pending_get_t;

typedef struct pb_pending_set_string_s {
  char			*property;
  char			*value;
  pb_playback_proxy_t	*proxy;
} pb_pending_set_string_t;

typedef struct pb_select_s {
  GSList		*selection;
  GHRFunc		predicate;
  gpointer		user_data;
} pb_select_t;

static pb_playback_proxy_t *	pb_playback_proxy_new 		(pb_manager_t		*manager,
						      		 const char		*connection_name,
						      		 const char		*path);
static void			pb_playback_proxy_ask_state	(pb_playback_proxy_t	*proxy,
								 enum pb_state_e	state);
static void			pb_playback_proxy_set_req_state	(pb_playback_proxy_t	*proxy,
								 enum pb_state_e	state,
								 DBusMessage		*msg);
static void			pb_playback_proxy_set_allowed_state(pb_playback_proxy_t	*proxy);
static void			pb_playback_proxy_grant_req_state(pb_playback_proxy_t	*proxy);
static void			pb_playback_proxy_deny_req_state(pb_playback_proxy_t	*proxy);
static void			pb_playback_proxy_update	(pb_playback_proxy_t	*proxy);
static void			pb_playback_proxy_manage	(pb_playback_proxy_t	*proxy);
static void			pb_playback_proxy_free		(pb_playback_proxy_t	*proxy);
#if DEBUG
static void			pb_playback_proxy_dump		(pb_playback_proxy_t	*proxy);
#endif

static pb_pending_introspect_t *pb_pending_introspect_new	(pb_manager_t		*manager,
								 const char		*sender);
static void			pb_pending_introspect_free	(pb_pending_introspect_t *pending);

static pb_pending_get_t *	pb_pending_get_new		(pb_playback_proxy_t	*proxy,
								 const char		*property);
static void			pb_pending_get_free		(pb_pending_get_t	*pending);

static pb_pending_set_string_t *pb_pending_set_string_new	(pb_playback_proxy_t	*proxy,
								 const char		*property,
								 const char		*value);
static void			pb_pending_set_string_free	(pb_pending_set_string_t *pending);

static pb_playback_proxy_t *	pb_manager_lookup_proxy		(pb_manager_t		*pb_manager,
								 const gchar		*sender,
								 const gchar		*path);
static GSList *			pb_manager_select		(pb_manager_t		*pb_manager,
								 GHRFunc		predicate,
								 gpointer		user_data);

static DBusHandlerResult
_message_func	(DBusConnection *	conn,
	     	 DBusMessage *		msg,
	     	 gpointer		user_data);

static const DBusObjectPathVTable manager_vtable = {
  NULL,
  _message_func,
};


static inline DBusHandlerResult
_send_message_and_unref		(DBusConnection	*conn,
				 DBusMessage	*msg)
{
  if (msg == NULL) {
    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
  }

  dbus_connection_send (conn, msg, NULL);
  dbus_message_unref (msg);

  return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
_error_reply		(DBusConnection		*conn,
			 DBusMessage		*msg,
			 const char		*name,
			 const char		*descr)
{
  DBusMessage *derr;

  if (!conn || !msg)
    return DBUS_HANDLER_RESULT_HANDLED;

  derr = dbus_message_new_error (msg, name, descr);
  if (derr) {
    dbus_connection_send (conn, derr, NULL);
    return DBUS_HANDLER_RESULT_HANDLED;
  } else {
    PB_LOG ("Unable to allocate new error return");
    return DBUS_HANDLER_RESULT_NEED_MEMORY;
  }
}

static pb_pending_introspect_t *
pb_pending_introspect_new	(pb_manager_t	*manager,
				 const char	*sender)
{
  pb_pending_introspect_t *rv;

  assert (manager != NULL && sender != NULL);
  rv = g_new (pb_pending_introspect_t, 1);
  if (rv == NULL) {
    ULOG_CRIT ("error while allocating a pending intro...");
    return NULL;
  }

  rv->sender = g_strdup (sender);
  rv->manager = manager;

  return rv;
}

static void
pb_pending_introspect_free	(pb_pending_introspect_t *pending)
{
  assert (pending != NULL);

  g_free (pending->sender);
  g_free (pending);
}

static pb_pending_get_t *
pb_pending_get_new		(pb_playback_proxy_t	*proxy,
				 const char		*property)
{
  pb_pending_get_t *rv;

  assert (proxy != NULL && property != NULL);
  rv = g_new (pb_pending_get_t, 1);
  if (rv == NULL) {
    ULOG_CRIT ("error while allocating a pending intro...");
    return NULL;
  }

  rv->property = g_strdup (property);
  rv->proxy = proxy;

  return rv;
}

static void
pb_pending_get_free		(pb_pending_get_t *pending)
{
  assert (pending != NULL);

  g_free (pending->property);
  g_free (pending);
}


static pb_pending_set_string_t *
pb_pending_set_string_new	(pb_playback_proxy_t	*proxy,
				 const char		*property,
				 const char		*value)
{
  pb_pending_set_string_t *rv;

  assert (proxy != NULL && property != NULL);
  rv = g_new (pb_pending_set_string_t, 1);
  if (rv == NULL) {
    ULOG_CRIT ("error while allocating a pending intro...");
    return NULL;
  }

  rv->property = g_strdup (property);
  rv->value = g_strdup (value);
  rv->proxy = proxy;

  return rv;
}

static void
pb_pending_set_string_free	(pb_pending_set_string_t *pending)
{
  assert (pending != NULL);

  g_free (pending->property);
  g_free (pending->value);
  g_free (pending);
}

static pb_playback_proxy_t *
pb_playback_proxy_new (pb_manager_t	*manager,
		       const char	*connection_name,
		       const char	*path)
{
  pb_playback_proxy_t	*proxy;
  int			i;

  assert (manager != NULL && connection_name != NULL && path != NULL);

  proxy = g_new0 (pb_playback_proxy_t, 1);

  if (proxy == NULL) {
    ULOG_CRIT ("cannot allocate a playback proxy!");
    return NULL;
  }

  proxy->pb_manager = manager;
  proxy->connection_name = g_strdup (connection_name);
  proxy->path = g_strdup (path);

  for (i = 0; i < PB_STATE_LAST; ++i) {
    proxy->allowed_state[i] = TRUE;
  }

  return proxy;
}

static void
_proxy_refresh_allowed_states	(pb_playback_proxy_t	*proxy,
				 const char		**allowed_state_list,
				 int			n_allowed_state)
{
  int				i;

  assert (proxy != NULL && allowed_state_list != NULL);

  for (i = 0; i < PB_STATE_LAST; ++i) {
    proxy->allowed_state[i] = FALSE;
  }

  for (i = 0; i < n_allowed_state; ++i) {
    proxy->allowed_state[pb_string_to_state (allowed_state_list[i])] = TRUE;
  }
}

static void
_get_return		(DBusPendingCall	*pending,
			 void			*user_data)
{
  pb_pending_get_t	*pending_get = user_data;
  pb_playback_proxy_t	*proxy;
  DBusMessage		*reply;
  DBusError		error;
  const char		*value;
  const char		**allowed_state_list;
  int			n_allowed_state;

  assert (pending_get != NULL);
  proxy = pending_get->proxy;

  dbus_error_init (&error);
  reply = dbus_pending_call_steal_reply (pending);
  if (reply == NULL) {
    dbus_pending_call_unref (pending);
    ULOG_CRIT("no reply...");
    return;
  }

  if (dbus_set_error_from_message (&error, reply)) {
    goto error;
  }

  if (g_str_equal (pending_get->property, "Class")) {
    dbus_message_get_args (reply, &error,
			   DBUS_TYPE_STRING, &value,
			   DBUS_TYPE_INVALID);
    if (dbus_error_is_set (&error)) {
      goto error;
    }

    /* Change Class */
    proxy->class = pb_string_to_class (value);
    pb_playback_proxy_manage (proxy);
  } else if (g_str_equal (pending_get->property, "State")) {
    dbus_message_get_args (reply, &error,
			   DBUS_TYPE_STRING, &value,
			   DBUS_TYPE_INVALID);
    if (dbus_error_is_set (&error)) {
      goto error;
    }

    /* Change State */
    proxy->state = pb_string_to_state (value);
    pb_playback_proxy_manage (proxy);
  } else if (g_str_equal (pending_get->property, "AllowedState")) {
    dbus_message_get_args (reply, &error,
			   DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, &allowed_state_list, &n_allowed_state,
			   DBUS_TYPE_INVALID);
    if (dbus_error_is_set (&error)) {
      goto error;
    }

    /* Change AllowedState */
    _proxy_refresh_allowed_states (proxy, allowed_state_list, n_allowed_state);
  }

  dbus_message_unref (reply);
  dbus_pending_call_unref (pending);

#ifdef DEBUG
  fprintf (stderr, "update\n");
  pb_playback_proxy_dump (proxy);
#endif
  return;

 error:
  ULOG_CRIT (error.message);
  dbus_error_free (&error);
  dbus_message_unref (reply);
  dbus_pending_call_unref (pending);
}

static void
pb_playback_proxy_get		(pb_playback_proxy_t	*proxy,
				 const char		*property)
{
  DBusMessage			*call;
  DBusPendingCall		*pending;
  DBusError			error;
  const char			*pb_iface = "com.osso.Playback";
  pb_pending_get_t		*pending_get;

  g_return_if_fail (proxy != NULL && property != NULL);

  pending_get = pb_pending_get_new (proxy, property);
  g_return_if_fail (pending_get != NULL);

  dbus_error_init (&error);
  call = dbus_message_new_method_call (proxy->connection_name, proxy->path,
				       DBUS_INTERFACE_PROPERTIES, "Get");
  dbus_message_append_args (call,
			    DBUS_TYPE_STRING, &pb_iface,
			    DBUS_TYPE_STRING, &property,
			    DBUS_TYPE_INVALID);

  dbus_connection_send_with_reply (proxy->pb_manager->connection, call, &pending, -1);
  if (pending != NULL) {
    dbus_pending_call_set_notify (pending, _get_return, pending_get, (DBusFreeFunction)pb_pending_get_free);
  } else {
    ULOG_CRIT ("no pending call allocation!");
  }
  dbus_message_unref (call);
}

static gboolean
_match_proxy                          (gchar				*key,
				       pb_playback_proxy_t		*proxy,
				       pb_playback_proxy_query_t	*query)
{
  assert (key != NULL && proxy != NULL && query != NULL);

  if (query->state != PB_STATE_NONE &&
      query->state != proxy->state)
    return FALSE;

  if (query->class != PB_CLASS_NONE &&
      query->class != proxy->class)
    return FALSE;

  if (query->connection_name != NULL &&
      !g_str_equal (query->connection_name, proxy->connection_name))
    return FALSE;

  return TRUE;
}

#ifdef DEBUG
static void
pb_playback_proxy_dump			(pb_playback_proxy_t	*proxy)
{
  assert (proxy != NULL);

  fprintf(stderr, "name: %s, path: %s, %s, state: %s, req: %s\n",
	  proxy->connection_name, proxy->path,
	  pb_class_to_string (proxy->class),
	  pb_state_to_string (proxy->state),
	  pb_state_to_string (proxy->requested_state));
}
#endif

static void
pb_playback_proxy_manage		(pb_playback_proxy_t	*proxy)
{
  GSList				*selection, *it;
  pb_playback_proxy_query_t		query = { 0, };
  pb_playback_proxy_t			*pbproxy;

  assert (proxy != NULL);

#ifdef DEBUG
  fprintf(stderr, "manage\n");
  pb_playback_proxy_dump (proxy);
#endif

  /* 1) Check the allowed state consistency - if VoIP ongoing, forbid PLAY state */
  if (proxy->class != PB_CLASS_VOIP) {
    query.class = PB_CLASS_VOIP;
    query.state = PB_STATE_PLAY;
    selection = pb_manager_select (proxy->pb_manager, (GHRFunc)_match_proxy, &query);
    if (g_slist_length (selection) > 0 && proxy->allowed_state[PB_STATE_PLAY]) {
      proxy->allowed_state[PB_STATE_PLAY] = FALSE;
      /* notify, FIXME: should take list as arg and get async */
      pb_playback_proxy_set_allowed_state (proxy);
    }
    g_slist_free (selection);
  }

  /* 2) Manage the request */
  if (proxy->requested_state == PB_STATE_NONE || proxy->class == PB_CLASS_NONE)
    return;

  if (proxy->requested_state < PB_STATE_PLAY) {
    if (proxy->class == PB_CLASS_VOIP && proxy->state == PB_STATE_PLAY) {
      /* we can allow state play now */
      query.class = PB_CLASS_MEDIA;
      query.state = PB_STATE_NONE; /* any */
      selection = pb_manager_select (proxy->pb_manager, (GHRFunc)_match_proxy, &query);
      for (it = selection; it != NULL; it = it->next) {
	pbproxy = (pb_playback_proxy_t*)it->data;
	pbproxy->allowed_state[PB_STATE_PLAY] = TRUE;
	pb_playback_proxy_set_allowed_state (pbproxy); /* notify, FIXME: should take list as arg and get async */
      }
      g_slist_free (selection);
    }
    /* Lower state: Granted all the time */
    goto grant;
  }

  query.class = PB_CLASS_VOIP;
  query.state = PB_STATE_PLAY;
  selection = pb_manager_select (proxy->pb_manager, (GHRFunc)_match_proxy, &query);
  if (g_slist_length (selection) > 0) {
    /* VoIP is playing: Denied */
#ifdef DEBUG
    fprintf(stderr, "Deny\n");
    pb_playback_proxy_dump (proxy);
#endif
    pb_playback_proxy_deny_req_state (proxy);
    g_slist_free (selection);
    return;
  }
  g_slist_free (selection);

  if (proxy->class == PB_CLASS_VOIP || proxy->class == PB_CLASS_MEDIA) {
    /* Stop Media if VoIP or Media wants to play */
    query.state = PB_STATE_PLAY;
    query.class = PB_CLASS_MEDIA;
    selection = pb_manager_select (proxy->pb_manager, (GHRFunc)_match_proxy, &query);
    for (it = selection; it != NULL; it = it->next) {
      pbproxy = (pb_playback_proxy_t*)it->data;
      pb_playback_proxy_ask_state (pbproxy, PB_STATE_STOP);
#ifdef DEBUG
      fprintf(stderr, "ask stop to playing ""Media""\n");
      pb_playback_proxy_dump (pbproxy);
#endif
    }
    g_slist_free (selection);
  }

  if (proxy->class == PB_CLASS_VOIP) {
    /* Stop background only if VoiP wants to play */
    query.state = PB_STATE_PLAY;
    query.class = PB_CLASS_BACKGROUND;
    selection = pb_manager_select (proxy->pb_manager, (GHRFunc)_match_proxy, &query);
    for (it = selection; it != NULL; it = it->next) {
      pbproxy = (pb_playback_proxy_t*)it->data;
      pb_playback_proxy_ask_state (pbproxy, PB_STATE_STOP);
#ifdef DEBUG
      fprintf(stderr, "ask stop to playing ""Background""\n");
      pb_playback_proxy_dump (pbproxy);
#endif
    }
    g_slist_free (selection);

    query.class = PB_CLASS_MEDIA;
    query.state = PB_STATE_NONE; /* any */
    selection = pb_manager_select (proxy->pb_manager, (GHRFunc)_match_proxy, &query);
    for (it = selection; it != NULL; it = it->next) {
      pbproxy = (pb_playback_proxy_t*)it->data;
      if (pbproxy == proxy)
	continue;
      pbproxy->allowed_state[PB_STATE_PLAY] = FALSE;
      pb_playback_proxy_set_allowed_state (pbproxy); /* notify, FIXME: should take list as arg and get async */
    }
    g_slist_free (selection);
  }

 grant:
  /* Granted */
#ifdef DEBUG
      fprintf(stderr, "Granted!\n");
      pb_playback_proxy_dump (proxy);
#endif
  pb_playback_proxy_grant_req_state (proxy);
  return;
}

static void
pb_playback_proxy_set_req_state	(pb_playback_proxy_t	*proxy,
				 enum pb_state_e	state,
				 DBusMessage		*msg)
{

  assert (proxy != NULL && state != PB_STATE_LAST);
  assert (proxy->pb_manager != NULL);

  /* ~~if there is a pending request, reply now with an excepetion */
  if (proxy->requested_state_msg != NULL) {
    _error_reply (proxy->pb_manager->connection, proxy->requested_state_msg, "com.osso.Aborted", "req aborted");
    dbus_message_unref (proxy->requested_state_msg);
    proxy->requested_state_msg = NULL;
  }

  proxy->requested_state = state;
  if (msg != NULL) {
    proxy->requested_state_msg = dbus_message_ref (msg);
  }
}

static void
pb_playback_proxy_set_allowed_state		(pb_playback_proxy_t	*proxy)
{
  DBusError		error;
  DBusMessage		*call;
  const char		*iface = "com.osso.Playback";
  const char		*set = "AllowedState";
  const char		**value;
  int			i, j, n_allowed_state;

  assert (proxy != NULL);

  dbus_error_init (&error);
  call = dbus_message_new_method_call (proxy->connection_name,
				       proxy->path,
				       DBUS_INTERFACE_PROPERTIES,
				       "Set");
  if (call == NULL) {
    ULOG_CRIT ("cannot allocate a set call");
    return;
  }

  dbus_message_append_args (call,
			    DBUS_TYPE_STRING, &iface,
			    DBUS_TYPE_STRING, &set,
			    DBUS_TYPE_INVALID);

  n_allowed_state = 0;
  for (i = 0; i < PB_STATE_LAST; ++i) {
    if (proxy->allowed_state[i] == TRUE)
      n_allowed_state++;
  }

  value = g_newa (const char *, n_allowed_state);
  for (i = 0, j = 0; i < PB_STATE_LAST; ++i) {
    if (proxy->allowed_state[i] == TRUE)
      value[j++] = pb_state_to_string (i);
  }

  dbus_message_append_args (call,
			    DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, &value, n_allowed_state,
			    DBUS_TYPE_INVALID);
  dbus_connection_send (proxy->pb_manager->connection, call, NULL);

  dbus_message_unref (call);
}

static void
pb_playback_proxy_grant_req_state		(pb_playback_proxy_t	*proxy)
{
  DBusMessage		*reply;
  const char		*state;

  assert (proxy != NULL);

  if (proxy->requested_state_msg != NULL) {
    reply = dbus_message_new_method_return (proxy->requested_state_msg);

    if (reply == NULL)
      ULOG_CRIT ("no more memory for method return?");
    else {
      state = pb_state_to_string (proxy->requested_state);
      dbus_message_append_args (reply,
				DBUS_TYPE_STRING, &state,
				DBUS_TYPE_INVALID);

      _send_message_and_unref (proxy->pb_manager->connection, reply);
    }

    dbus_message_unref (proxy->requested_state_msg);
    proxy->requested_state_msg = NULL;
  }

  /* clear requested state */
  pb_playback_proxy_set_req_state (proxy, PB_STATE_NONE, NULL);
}

static void
pb_playback_proxy_deny_req_state		(pb_playback_proxy_t	*proxy)
{
  assert (proxy != NULL);

  if (proxy->requested_state_msg != NULL) {
    _error_reply (proxy->pb_manager->connection, proxy->requested_state_msg, "com.osso.Aborted", "State request denied");
    dbus_message_unref (proxy->requested_state_msg);
    proxy->requested_state_msg = NULL;
  }

  /* clear requested state */
  pb_playback_proxy_set_req_state (proxy, PB_STATE_NONE, NULL);
}

static void
_set_string_return	(DBusPendingCall	*pending,
			 void			*user_data)
{
  pb_pending_set_string_t	*pending_set_string = user_data;
  pb_playback_proxy_t		*proxy;
  DBusMessage			*reply;
  DBusError			error;

  assert (pending_set_string != NULL);
  proxy = pending_set_string->proxy;

  dbus_error_init (&error);
  reply = dbus_pending_call_steal_reply (pending);
  if (reply == NULL) {
    dbus_pending_call_unref (pending);
    ULOG_CRIT("no reply...");
    return;
  }

  if (g_str_equal (pending_set_string->property, "State")) {
    proxy->state = pb_string_to_state (pending_set_string->value);
  }

  dbus_message_unref (reply);
  dbus_pending_call_unref (pending);
}

static void
pb_playback_proxy_set_string	(pb_playback_proxy_t	*proxy,
				 const char		*property,
				 const char		*value)
{
  DBusMessage			*call;
  DBusPendingCall		*pending;
  DBusError			error;
  const char			*pb_iface = "com.osso.Playback";
  pb_pending_set_string_t	*pending_set_string;

  g_return_if_fail (proxy != NULL && property != NULL);

  pending_set_string = pb_pending_set_string_new (proxy, property, value);
  g_return_if_fail (pending_set_string != NULL);

  dbus_error_init (&error);
  call = dbus_message_new_method_call (proxy->connection_name, proxy->path,
				       DBUS_INTERFACE_PROPERTIES, "Set");
  dbus_message_append_args (call,
			    DBUS_TYPE_STRING, &pb_iface,
			    DBUS_TYPE_STRING, &property,
			    DBUS_TYPE_STRING, &value,
			    DBUS_TYPE_INVALID);

  dbus_connection_send_with_reply (proxy->pb_manager->connection, call, &pending, -1);
  if (pending != NULL) {
    dbus_pending_call_set_notify (pending, _set_string_return, pending_set_string, (DBusFreeFunction)pb_pending_set_string_free);
  } else {
    ULOG_CRIT ("no pending call allocation!");
  }

  dbus_message_unref (call);
}

static void
pb_playback_proxy_ask_state	(pb_playback_proxy_t	*proxy,
				 enum pb_state_e	state)
{
  assert (proxy != NULL && state != PB_STATE_LAST && state != PB_STATE_NONE);
  assert (proxy->pb_manager != NULL);

  if (proxy->state == state) {
    return;
  }

  pb_playback_proxy_set_string (proxy, "State", pb_state_to_string (state));
}

static void
pb_playback_proxy_update	(pb_playback_proxy_t	*proxy)
{
  assert (proxy != NULL);
  assert (proxy->pb_manager != NULL && proxy->path != NULL && proxy->connection_name != NULL);

  pb_playback_proxy_get (proxy, "Class");
  pb_playback_proxy_get (proxy, "State");
  pb_playback_proxy_get (proxy, "AllowedState");
}

static void
pb_playback_proxy_free (pb_playback_proxy_t	*proxy)
{
  assert (proxy != NULL);

  g_free (proxy->connection_name);
  g_free (proxy->path);
  g_free (proxy);
}

static DBusHandlerResult
_manager_introspect	(pb_manager_t		*manager,
			 DBusMessage		*msg)
{
  DBusMessage	*reply;
  const char	*path;
  const char	introspect[] = \
    "<!DOCTYPE node PUBLIC \"-//freedesktop//DTD D-BUS Object Introspection 1.0//EN\""
    " \"http://www.freedesktop.org/standards/dbus/1.0/introspect.dtd\">\n"
    "<node name=""/com/osso/playback/manager"">\n"
    "<interface name=\"com.osso.Playback.Manager\">\n"
    " <method name=""RequestState"">\n"
    "  <arg name=""pb"" type=""o"" direction=""in""/>\n"
    "  <arg name=""state"" type=""s"" direction=""in""/>\n"
    "  <arg name=""granted_state"" type=""s"" direction=""out""/>\n"
    " </method>\n"
    " <method name=""GetAllowedState"">\n"
    "  <arg name=""pb"" type=""o"" direction=""in""/>\n"
    "  <arg name=""allowed_state"" type=""as"" direction=""out""/>\n"
    " </method>\n"
    " <signal name=""AllowedState"">\n"
    "  <arg name=""class"" type=""s""/>\n"
    "  <arg name=""allowed_state"" type=""as""/>\n"
    " </signal>\n"
    "</interface>\n"
    "</node>";
  const char	*ptr = introspect;

  path = dbus_message_get_path (msg);

  ULOG_CRIT ("Introspect path: %s", path);

  if (!dbus_message_has_signature (msg, DBUS_TYPE_INVALID_AS_STRING)) {
    ULOG_CRIT ("Unexpected signature to introspect call");
    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
  }

  reply = dbus_message_new_method_return (msg);

  if (reply == NULL)
    return DBUS_HANDLER_RESULT_NEED_MEMORY;

  /* FIXME: add introspection xml string */
  dbus_message_append_args (reply,
			    DBUS_TYPE_STRING, &ptr,
			    DBUS_TYPE_INVALID);

  return _send_message_and_unref (manager->connection, reply);
}

static DBusHandlerResult
_manager_request_state	(pb_manager_t		*manager,
			 DBusMessage		*msg)
{
  DBusError		error;
  pb_playback_proxy_t	*proxy;
  const char		*sender, *path, *state;
  enum pb_state_e	req_state;

  dbus_error_init (&error);
  sender = dbus_message_get_sender (msg);

  if (sender == NULL)
    return DBUS_HANDLER_RESULT_NEED_MEMORY;

  dbus_message_get_args (msg, &error,
			 DBUS_TYPE_STRING, &state,
			 DBUS_TYPE_STRING, &path,
			 DBUS_TYPE_INVALID);

  if (dbus_error_is_set (&error)) {
    ULOG_CRIT (error.message);
    _error_reply (manager->connection, msg, "com.osso.InvalidArguments", error.message);
    dbus_error_free (&error);
    return DBUS_HANDLER_RESULT_HANDLED;
  }

  req_state = pb_string_to_state (state);
  if (req_state == PB_STATE_LAST || req_state == PB_STATE_NONE) {
    return _error_reply (manager->connection, msg, "com.osso.InvalidArguments", "Invalid state");
  }

  proxy = pb_manager_lookup_proxy (manager, sender, path);
  if (proxy == NULL) {
    return _error_reply (manager->connection, msg, "com.osso.InvalidArguments", "Unregistered sender/path playback object");
  }

  pb_playback_proxy_set_req_state (proxy, req_state, msg);
  pb_playback_proxy_manage (proxy);

  return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
_manager_get_allowed_state	(pb_manager_t		*manager,
				 DBusMessage		*msg)
{
  const char		**allowed_state_list;
  const char		*sender, *path;
  DBusMessage		*reply;
  DBusError		error;
  int			n_allowed_state, i, j;
  pb_playback_proxy_t	*proxy;

  assert (manager != NULL);

  dbus_error_init (&error);
  sender = dbus_message_get_sender (msg);

  if (sender == NULL)
    return DBUS_HANDLER_RESULT_NEED_MEMORY;

  dbus_message_get_args (msg, &error,
			 DBUS_TYPE_STRING, &path,
			 DBUS_TYPE_INVALID);

  if (dbus_error_is_set (&error)) {
    ULOG_CRIT (error.message);
    _error_reply (manager->connection, msg, "com.osso.InvalidArguments", error.message);
    dbus_error_free (&error);
    return DBUS_HANDLER_RESULT_HANDLED;
  }

  proxy = pb_manager_lookup_proxy (manager, sender, path);
  if (proxy == NULL) {
    _error_reply (manager->connection, msg, "com.osso.InvalidArguments", "Unregistered sender/path playback object");
    return DBUS_HANDLER_RESULT_HANDLED;
  }

  reply = dbus_message_new_method_return (msg);
  if (reply == NULL) {
    return DBUS_HANDLER_RESULT_NEED_MEMORY;
  }

  n_allowed_state = 0;
  for (i = 0; i < PB_STATE_LAST; ++i) {
    if (proxy->allowed_state[i] == TRUE) {
      ++n_allowed_state;
    }
  }

  allowed_state_list = (n_allowed_state == 0 ? NULL : g_newa (const char *, n_allowed_state));
  if (n_allowed_state != 0 && allowed_state_list == NULL) {
    return DBUS_HANDLER_RESULT_NEED_MEMORY;
  }

  for (i = 0, j = 0; i < PB_STATE_LAST; ++i) {
    if (proxy->allowed_state[i] == TRUE) {
      allowed_state_list[j++] = pb_state_to_string (i);
    }
  }

  dbus_message_append_args (reply,
			    DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, &allowed_state_list, n_allowed_state,
			    DBUS_TYPE_INVALID);

  _send_message_and_unref (manager->connection, reply);

  return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
_message_func	(DBusConnection *	connection,
	     	 DBusMessage *		msg,
	     	 gpointer		user_data)
{
  DBusError			error;
  pb_manager_t			*pb_manager = user_data;
  const char 			*interface = dbus_message_get_interface (msg);
  const char 			*member = dbus_message_get_member (msg);

  dbus_error_init (&error);
  if (dbus_set_error_from_message (&error, msg)) {
    ULOG_CRIT(error.message);
    dbus_message_unref (msg);
    dbus_error_free (&error);
    return DBUS_HANDLER_RESULT_HANDLED;
  }
  if (g_str_equal (DBUS_INTERFACE_INTROSPECTABLE, interface) &&
      g_str_equal ("Introspect", member)) {
    return _manager_introspect (pb_manager, msg);
  }

  if (!g_str_equal ("com.osso.Playback.Manager", interface))
    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

  if (g_str_equal ("RequestState", member)) {
    return _manager_request_state (pb_manager, msg);
  } else if (g_str_equal ("GetAllowedState", member)) {
    return _manager_get_allowed_state (pb_manager, msg);
  }

  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static pb_playback_proxy_t *
pb_manager_lookup_proxy (pb_manager_t *pb_manager,  const gchar *sender, const gchar *path)
{
  pb_playback_proxy_t	*proxy;
  gchar			*key;

  assert (sender != NULL && path != NULL);

  key = g_strconcat (sender, path, NULL);
  if (key == NULL) {
    ULOG_CRIT ("cannot allocate a key");
    return NULL;
  }

  proxy = g_hash_table_lookup (pb_manager->playback_proxy, key);

  g_free (key);

  return proxy;
}


static void
_manager_select			(gchar			*key,
				 pb_playback_proxy_t	*proxy,
				 pb_select_t		*select)
{
  assert (key != NULL && proxy != NULL && select != NULL);

  if (select->predicate (key, proxy, select->user_data)) {
    select->selection = g_slist_append (select->selection, proxy);
  }
}

static GSList *
pb_manager_select		(pb_manager_t	*pb_manager,
				 GHRFunc	predicate,
				 gpointer	user_data)
{
  pb_select_t	select;

  assert (pb_manager != NULL && predicate);

  select.selection = NULL;
  select.user_data = user_data;
  select.predicate = predicate;

  g_hash_table_foreach (pb_manager->playback_proxy, (GHFunc)_manager_select, &select);

  return select.selection;
}

static gboolean
_pb_manager_add_playback_proxy (pb_manager_t *pb_manager, const gchar *sender, const gchar *path)
{
  gchar			*key;
  pb_playback_proxy_t	*proxy;

  g_return_val_if_fail (pb_manager_lookup_proxy (pb_manager, sender, path) == NULL, FALSE);

  proxy = pb_playback_proxy_new (pb_manager, sender, path);
  if (proxy == NULL) {
    ULOG_CRIT ("cannot allocate a proxy");
    return FALSE;
  }

  key = g_strconcat (sender, path, NULL);
  g_hash_table_insert (pb_manager->playback_proxy, key, proxy);

  pb_playback_proxy_update (proxy);

  return TRUE;
}

static void
_start_element  (GMarkupParseContext *context,
		 const gchar         *element_name,
		 const gchar        **attribute_names,
		 const gchar        **attribute_values,
		 gpointer             user_data,
		 GError             **error)
{
  guint				i;
  pb_pending_introspect_t	*pintrospect = user_data;
  pb_manager_t			*pb_manager = pintrospect->manager;
  const char			*sender = pintrospect->sender;

  if (error != NULL && *error != NULL) {
    ULOG_CRIT ((*error)->message);
    g_error_free (*error); /* FIXME */
    return;
  }

  /* insert proxy in the hashtable */
  if (!g_str_equal (element_name, "node")) {
    return;
  }

  for (i = 0; attribute_names[i] != NULL && attribute_values[i] != NULL; ++i) {
    if (g_str_equal (attribute_names[i], "name")) {
      if (!attribute_values[i] || attribute_values[i][0] != '/')
	continue;
      _pb_manager_add_playback_proxy (pb_manager, sender, attribute_values[i]);
    }
  }
}

static void
_introspect_return	(DBusPendingCall	*pending,
			 void			*user_data)
{
  pb_pending_introspect_t	*pintrospect = user_data;
  DBusMessage			*reply;
  DBusError			error;
  const char			*introspect;
  GMarkupParseContext		*parse_ctxt;
  const GMarkupParser		parser = {
    .start_element = _start_element,
    NULL
  };

  reply = dbus_pending_call_steal_reply (pending);
  if (reply == NULL) {
    dbus_pending_call_unref (pending);
    ULOG_CRIT("no reply...");
    return;
  }

  dbus_error_init (&error);
  if (dbus_set_error_from_message (&error, reply)) {
    goto error;
  }

  dbus_message_get_args (reply, &error, DBUS_TYPE_STRING, &introspect, DBUS_TYPE_INVALID);

  if (dbus_error_is_set (&error)) {
    goto error;
  }

  if (introspect == NULL) {
    ULOG_CRIT ("introspect result is null");
    dbus_message_unref (reply);
    dbus_pending_call_unref (pending);
    return;
  }

  parse_ctxt = g_markup_parse_context_new (&parser, 0, pintrospect, NULL);
#if FIXME_INSTROSPECT
  g_markup_parse_context_parse (parse_ctxt, introspect, strlen (introspect), &gerror);
  if (gerror != NULL) {
    ULOG_CRIT (gerror->message);
    g_error_free (gerror);
  }
#endif
  g_markup_parse_context_free (parse_ctxt);
  dbus_message_unref (reply);
  dbus_pending_call_unref (pending);

  return;

 error:
  ULOG_CRIT(error.message);
  dbus_error_free (&error);
  dbus_message_unref (reply);
  dbus_pending_call_unref (pending);
}

static DBusHandlerResult
_dbus_signal_handler (DBusConnection  *connection,
		      DBusMessage     *message,
		      void            *user_data)
{
  DBusError		derror;
  pb_manager_t		*pb_manager = user_data;
  pb_playback_proxy_t	*proxy;
  pb_playback_proxy_query_t query = { 0, };
  const char		*req_name, *name, *old, *new, *path, *iface, *prop, *val;
  const char		*sender = dbus_message_get_sender (message);
  dbus_uint32_t		num;

  assert (pb_manager != NULL);
  dbus_error_init (&derror);
  if (dbus_message_is_method_call (message, "org.freedesktop.DBus", "RequestName")) {
    dbus_message_get_args (message, &derror,
			   DBUS_TYPE_STRING, &req_name,
			   DBUS_TYPE_UINT32, &num,
			   DBUS_TYPE_INVALID);
    if (dbus_error_is_set (&derror)) {
      ULOG_CRIT (derror.message);
      dbus_error_free (&derror);
      return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    pb_manager_refresh_playback_proxy_by_sender (pb_manager, sender);
  } else if (dbus_message_is_signal (message, "org.freedesktop.DBus", "NameOwnerChanged")) {
    dbus_message_get_args (message, &derror,
			   DBUS_TYPE_STRING, &name,
			   DBUS_TYPE_STRING, &old,
			   DBUS_TYPE_STRING, &new,
			   DBUS_TYPE_INVALID);
    if (dbus_error_is_set (&derror)) {
      ULOG_CRIT (derror.message);
      dbus_error_free (&derror);
      return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    query.connection_name = (gchar*)old;
    /* FIXME: make sure no pending call have a reference! use ref count instead of _destroy.. */
    g_hash_table_foreach_remove (pb_manager->playback_proxy, (GHRFunc)_match_proxy, &query);

  } else if (dbus_message_is_signal (message, "com.osso.Playback", "Hello")) {
    path = dbus_message_get_path (message);
    if (path == NULL) {
      ULOG_CRIT ("no path in the signal!");
      return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }
    _pb_manager_add_playback_proxy (pb_manager, sender, path);
    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
  } else if (dbus_message_is_signal (message, DBUS_INTERFACE_PROPERTIES, "Notify")) {
    path = dbus_message_get_path (message);

    if (path == NULL) {
      ULOG_CRIT ("no path in the signal!");
      return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    if ((proxy = pb_manager_lookup_proxy (pb_manager, sender, path)) == NULL) {
      ULOG_CRIT ("this playback is not registered!");
      return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    dbus_message_get_args (message, &derror,
			   DBUS_TYPE_STRING, &iface,
			   DBUS_TYPE_STRING, &prop,
			   DBUS_TYPE_STRING, &val,
			   DBUS_TYPE_INVALID);

    if (dbus_error_is_set (&derror)) {
      ULOG_CRIT (derror.message);
      dbus_error_free (&derror);
      return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    if (!iface || !prop || !val || strcmp (iface, "com.osso.Playback")) {
      ULOG_CRIT ("bad Notify signal");
      return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    if (strcmp (iface, "com.osso.Playback") == 0 &&
	strcmp (prop, "State") == 0) {
      if (pb_string_to_state (val) == PB_STATE_NONE) {
	ULOG_CRIT ("Invalid State value");
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
      }

      proxy->state = pb_string_to_state (val);
    }

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
  }

  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

pb_manager_t *
pb_manager_new		(DBusConnection *connection)
{
  pb_manager_t	*pb_manager;
  gboolean	success;
  DBusError	error;

  assert (connection != NULL);

  pb_manager = g_new (pb_manager_t, 1);
  if (pb_manager == NULL) {
    ULOG_CRIT("cannot allocate a manager");
    return NULL;
  }

  if (!dbus_connection_register_object_path (connection, DBUS_PB_MANAGER_PATH,
					     &manager_vtable, pb_manager)) {
    ULOG_CRIT("HSS: error registering object path");
    g_free (pb_manager);
    return NULL;
  }

  dbus_error_init (&error);
  dbus_bus_add_match (connection, "type='method_call', destination='org.freedesktop.DBus',"
		      "interface='org.freedesktop.DBus', member='RequestName'", &error);
  if (dbus_error_is_set (&error)) goto dbus_error;

  dbus_bus_add_match (connection, "type='signal', interface='org.freedesktop.DBus',"
		      "member='NameOwnerChanged'", &error);
  if (dbus_error_is_set (&error)) goto dbus_error;

  dbus_bus_add_match (connection, "type='signal', interface='com.osso.Playback',"
		      "member='Hello'", &error);
  if (dbus_error_is_set (&error)) goto dbus_error;

  dbus_bus_add_match (connection, "type='signal', interface='" DBUS_INTERFACE_PROPERTIES "',"
		      "member='Notify'", &error);
  if (dbus_error_is_set (&error)) goto dbus_error;

  success = dbus_connection_add_filter (connection, _dbus_signal_handler, pb_manager, NULL);
  if (success == FALSE) {
    ULOG_CRIT ("could not add a signal handler");
  }

  if (dbus_bus_request_name (connection, "com.osso.Playback.Manager", 0, &error)
      != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
    ULOG_CRIT("HSS: error acquiring service");
  }
  if (dbus_error_is_set (&error)) goto dbus_error;

  pb_manager->connection = dbus_connection_ref (connection);
  pb_manager->playback_proxy = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, (GDestroyNotify) pb_playback_proxy_free);

  /* pb_manager_refresh_playback_proxy (pb_manager); */

  return pb_manager;

 dbus_error:
  if (dbus_error_is_set (&error)) {
    ULOG_CRIT (error.message);
    dbus_error_free (&error);
    g_free (pb_manager);
  }
  return NULL;
}

void
pb_manager_refresh_playback_proxy_by_sender (pb_manager_t	*pb_manager,
					     const char		*sender)
{
  DBusMessage		*call;
  DBusPendingCall	*pending;

  assert (pb_manager != NULL && sender != NULL);

  call = dbus_message_new_method_call (sender, "/", "org.freedesktop.DBus.Introspectable", "Introspect");
  dbus_connection_send_with_reply (pb_manager->connection, call, &pending, -1);
  if (pending != NULL) {
    dbus_pending_call_set_notify (pending, _introspect_return,
				  pb_pending_introspect_new (pb_manager, sender), (DBusFreeFunction)pb_pending_introspect_free);
  } else {
    ULOG_CRIT ("no pending call allocation!");
  }

  dbus_message_unref (call);
}

static void
_list_pb_return		(DBusPendingCall	*pending,
			 void			*user_data)
{
  pb_manager_t			*pb_manager = user_data;
  DBusMessage			*reply;
  DBusError			error;
  const char			**owners;
  int				i, num_owners;

  reply = dbus_pending_call_steal_reply (pending);
  if (reply == NULL) {
    dbus_pending_call_unref (pending);
    ULOG_CRIT("no reply...");
    return;
  }

  dbus_error_init (&error);
  if (dbus_set_error_from_message (&error, reply)) {
    goto error;
  }

  dbus_message_get_args (reply, &error,
			 DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, &owners, &num_owners,
			 DBUS_TYPE_INVALID);

  if (dbus_error_is_set (&error)) {
    goto error;
  }

  if (owners == NULL) {
    ULOG_CRIT ("introspect result is null");
    dbus_message_unref (reply);
    dbus_pending_call_unref (pending);
    return;
  }

  for (i = 0; i < num_owners; ++i) {
    pb_manager_refresh_playback_proxy_by_sender (pb_manager, owners[i]);
  }

  dbus_message_unref (reply);
  dbus_pending_call_unref (pending);
  return;

 error:
  ULOG_CRIT (error.message);
  dbus_error_free (&error);
  dbus_message_unref (reply);
  dbus_pending_call_unref (pending);
}

void
pb_manager_refresh_playback_proxy	(pb_manager_t	*pb_manager)
{
  DBusMessage		*call;
  DBusError		error;
  DBusPendingCall	*pending;
  static const char	pb_service[] = "com.osso.Playback";
  const char		*ptr = pb_service;

  assert (pb_manager != NULL);

  call = dbus_message_new_method_call ("org.freedesktop.DBus", "/", "org.freedesktop.DBus", "ListQueuedOwners");

  dbus_error_init (&error);
  dbus_message_append_args (call,
			    DBUS_TYPE_STRING, &ptr,
			    DBUS_TYPE_INVALID);

  dbus_connection_send_with_reply (pb_manager->connection, call, &pending, -1);

  if (pending != NULL) {
    dbus_pending_call_set_notify (pending, _list_pb_return, pb_manager, NULL);
  } else {
    ULOG_CRIT ("no pending call allocation!");
  }

  dbus_message_unref (call);
}

void
pb_manager_free		(pb_manager_t	*pb_manager)
{
  assert (pb_manager != NULL);

  g_hash_table_destroy (pb_manager->playback_proxy);

  dbus_connection_unregister_object_path (pb_manager->connection, DBUS_PB_MANAGER_PATH);
  dbus_connection_unref (pb_manager->connection);

  g_free (pb_manager);
}
