#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <unistd.h>
#include <alsa/asoundlib.h>

#define HAVE_770

#ifdef HAVE_770

#include <fcntl.h>
#include <math.h>
#include <sched.h>
#include <sys/poll.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <sys/file.h>
#include <sys/stat.h>

#include <dsp/interface_common.h>
#include <dsp/audiopp_dsptask.h>

#define readw_lock(fd, offset, whence, len)  lock_reg(fd, F_SETLKW, F_RDLCK, offset, whence, len)
#define writew_lock(fd, offset, whence, len) lock_reg(fd, F_SETLKW, F_WRLCK, offset, whence, len)
#define un_lock(fd, offset, whence, len)     lock_reg(fd, F_SETLK,  F_UNLCK, offset, whence, len)

#define HSS         0x00
#define MVA         0x01
#define LS_DEVICE_NAME "/sys/devices/platform/gpio-switch/speaker/connection_switch"
#define STATUS_ERROR -1
#define STATUS_OK 0
#define DISABLED 0
#define ENABLED 1

#include "mvi_voltables.h"

#endif


#include "common.h"
#include "mvi.h"

#define VOLMAX 20
#define VOLSTEP (100/VOLMAX)
#define VOLINDEX(v) (((v)+VOLSTEP/2)*VOLMAX/100)

/* CTL_PCM_VOL values for volume steps 0...VOLMAX */
/* loudspeaker: */
static const int tsc_volume_ls[VOLMAX+1] = {
  67, 70, 73, 76, 79, 82, 85, 88, 91, 94, 97, 100, 103, 106, 109, 112, 115, 118, 121, 124, 127
};

/* headset/headphones: */
static const int tsc_volume_hp[VOLMAX+1] = {
  21, 27, 33, 39, 45, 51, 57, 63, 69, 75, 81, 87,  93,  97, 101, 105, 109, 113, 117, 121, 125
};

static const int tlv_volume_ls[VOLMAX+1] = {
  11, 61, 64, 67, 70, 73, 76, 79, 82, 85, 88, 91, 94, 97, 100, 103, 106, 109, 112, 115, 118,
};

static const int tlv_volume_hp[VOLMAX+1] = {
  11, 22, 26, 32, 38, 44, 50, 56, 62, 68, 74, 80, 86, 90, 94, 98, 102, 106, 110, 114, 118
};

const int *volume_ls = tsc_volume_ls;
const int *volume_hp = tsc_volume_hp;

enum {
    CTL_PCM_VOL,
    CTL_PCM_SW,
    CTL_LS_AMP,
    CTL_HS_AMP,
    CTL_CAP_VOL,
    CTL_HS_CAP_VOL,
    CTL_HS_CAP_SW,
    CTL_CAP_SW,
    CTL_RX44_DIGITAL_MIC_SW,
    CTL_PCM_CAP_SW,
    CTL_EACPCM_SW,
    CTL_LAST
};

static char *ctl_names[] = {
    "PCM Playback Volume",
    "PCM Playback Switch",
    "Speaker Amplifier",
    "Headset Amplifier",
    "Capture Volume",
    "Mic Boost",
    "Headset Mic Select",
    "Capture Switch",
    "Digital Mic Switch",
    "PCM Capture Switch",
    "EACPCM Playback Switch",
};

typedef struct {
    snd_hctl_elem_t      *elem;
    snd_ctl_elem_type_t   type;
    int                   min;
    int                   max;
    int                   count;
} control;

typedef struct {
    snd_hctl_t  *hctl;
    control      ctl[CTL_LAST];
} MVI_data;

/* This struct should really be allocated dynamically by MVI_open
 * and passed to all libmvi functions as a parameter. For API
 * compatibility that's not done yet, but all code already uses
 * it through the pointer */
static MVI_data controls = { NULL, };
static MVI_data *mvi = &controls;

#ifdef HAVE_770
#define INIT_APOP

# ifdef INIT_APOP
//Audio Post Processing Parameters
POSTPROC_INIT_PARAMS post_proc;

// MuMDRC parameter table: Sputnik_MID7_16_9_05.txt
IMUMDRC_Status default_tables2 =
  {
    16384,       // linkCoeffSelf
    0,           // linkCoeffOthers
    0x0022,      // lim_attCoeff
    0x0A3B,      // lim_relCoeff
    0x000047FB,  // limiterTreshold
    0x00000000,  // LimGain
    {0x00001C50, 0x00001C50, 0x00001C50, 0x00002448, 0x00002448}, // AttCoeff
    {0x000000A3, 0x000000A3, 0x000000A3, 0x000000A3, 0x000000A3}, // RelCoeff
    {
       {0x00000003, 0x0000040C, 0x00000CCD, 0x0000287A, 0x00008000}, // levelLimits[1]
       {0x00000003, 0x0000040C, 0x00000CCD, 0x0000287A, 0x00008000}, // levelLimits[2]
       {0x00000003, 0x0000040C, 0x00000CCD, 0x0000287A, 0x00008000}, // levelLimits[3]
       {0x00000003, 0x0000040C, 0x00000CCD, 0x0000287A, 0x00008000}, // levelLimits[4]
       {0x00000003, 0x0000040C, 0x00000CCD, 0x0000287A, 0x00008000}  // levelLimits[5]
    },
    {
       {0x0000199A, 0x000011EC, 0xFFFFCCCE, 0xFFFFC001, 0xFFFFE667, 0xFFFF8001}, // K[1]
       {0x00002CCD, 0x0000147B, 0xFFFFCCCE, 0xFFFFB334, 0xFFFFC001, 0xFFFF8001}, // K[2]
       {0x00002CCD, 0x00000A3D, 0x00000000, 0xFFFFCCCE, 0xFFFFA667, 0xFFFF8001}, // K[3]
       {0x00002666, 0x00000000, 0x00000000, 0x00000000, 0xFFFFB334, 0xFFFF8001}, // K[4]
       {0x00000CCD, 0x00000000, 0x00000000, 0x00000000, 0xFFFFE667, 0xFFFF8001}  // K[5]
    },
    {
       {0x3DCCCCCD, 0x3E31F368, 0x3F642905, 0x3F353BEF, 0x3F800000, 0x3F800000}, // A[1]
       {0x3C91AD39, 0x3DD19206, 0x3FB4CE08, 0x3F642905, 0x3F800000, 0x3F800000}, // A[2]
       {0x3C91AD39, 0x3E5AED65, 0x406314A0, 0x3FB4CE08, 0x3F800000, 0x3F800000}, // A[3]
       {0x3D0186E2, 0x3FFF64C1, 0x3FFF64C1, 0x3FFF64C1, 0x3F800000, 0x3F800000}, // A[4]
       {0x3EA1E89B, 0x3FA12478, 0x3FA12478, 0x3FA12478, 0x3F800000, 0x3F800000}  // A[5]
    },
    0x0001 // use_mumdrc
  };
# endif


static int type, state, app_drv;
static int fd_hss=-1, fd_mva=-1, fd_cmd=-1, fd_vol=-1;
static int volume_cur=-102;
static int ls_state = -1;

static char statfname[] = "/tmp/.mvi_status_info";
static char volfname[]  = "/tmp/.mvi_volume_info";
static char taskname[]  = "/dev/dsptask/audiopp";
static char rsysfsname[] = "/sys/devices/platform/audio-i2c/volume_right";
static char lsysfsname[] = "/sys/devices/platform/audio-i2c/volume_left";
static short int app_bridge_buffer[BRIDGE_BUFFER_SIZE];
AUDIO_INIT_STATUS app_init_status;

# ifdef INIT_APOP
/* To enable/disable APOP */
POSTPROCCESSING_BYPASS_DATA audiopp_bypass_data;
# endif

#endif

static void add_control(MVI_data *mvi, snd_hctl_elem_t *elem, int index);
static int  get_control(MVI_data *mvi, int index);
static void set_control(MVI_data *mvi, int index, int value);
static int find_nearest(const int *vtable, int vol);


#ifdef HAVE_770

static int hp_speaker_set_loudspeaker(int is_on);
static int read_block(int fdd);
static int write_volume(int volume_in);
static int set_hpvolume(char* sysfs_filename_in, int volume_in);
static int set_lsvolume(int volume_in);
static int lock_reg(int fd, int cmd, int type, off_t offset, int whence, off_t len);
static int is_write_unlock(int fd, off_t offset, int whence, off_t len);
static int open_770(int type_in);


static int hp_speaker_set_loudspeaker(int is_on)
{
  if (ls_state == is_on) {
    return STATUS_OK;
  }

  int ls_device = open(LS_DEVICE_NAME, O_WRONLY);
  if (ls_device < 0) {
    return STATUS_ERROR;
  }

  //  int volume_tmp = 0;
  //  volume_tmp = MVI_getVolume();
  //  MVI_setVolume (-101);

  int result = STATUS_ERROR;
  char buf;

  if (is_on == ENABLED) {
    buf = '1';
  }
  else {
    buf = '0';
  }

  int status = write(ls_device, &buf, 1);

  if (status > 0) {
    ls_state = is_on;
    result = STATUS_OK;
  }

  close(ls_device);
  //  MVI_setVolume (volume_tmp);

  fprintf(stderr,"\nError opening %s.\n",taskname);
  return result;
}

# ifdef INIT_APOP
# define AUDIO_PP_EQ_IN_USE

int init_audiopp()
{
    AUDIO_STATUS_INFO status_info;
    
    if((app_drv=open(taskname,O_RDWR))<0) {
        fprintf(stderr,"\nError opening %s.\n",taskname);
    }
	
    while(!read_block(app_drv)) read(app_drv,&app_bridge_buffer[0],sizeof(short int));
	
    app_bridge_buffer[0] = DSP_CMD_STATE;
    write(app_drv,app_bridge_buffer,sizeof(short int));
    read(app_drv,&status_info,sizeof(AUDIO_STATUS_INFO));
    if(status_info.status == STATE_UNINITIALISED)
    {
        app_bridge_buffer[0] = DSP_CMD_INIT;
	write(app_drv,app_bridge_buffer,sizeof(short int));
	read(app_drv,&app_init_status, sizeof(app_init_status));
	if(app_init_status.init_status!=DSP_OK)
	{
	    fprintf(stderr, "\nError initialising audiopp socket node. \n"); 
	    switch(app_init_status.init_status)
	    {
	        case DSP_ERROR_GENERAL: printf("\nDSP_ERROR_GENERAL\n\n");break;
	        case DSP_ERROR_MEMORY: printf("\nDSP_ERROR_MEMORY\n\n");break;
	        case DSP_ERROR_STATE: printf("\nDSP_ERROR_STATE\n\n");break;
	        default: printf("\nDSP init: Unknown error\n\n");break;
	    }
	    return -1;
	} 
	else{printf("\n audiopp socket node is successfully initialised.");}
	
	//Sending Post Proc Init Parameters
	post_proc.dsp_cmd=DSP_CMD_SET_POSTPROC_INIT_PARAMS;
        post_proc.audio_params.dsp_cmd= 0x1111;
	post_proc.audio_params.audio_fmt= 0x2222;
	post_proc.audio_params.sample_rate=0x3333;
	post_proc.audio_params.number_channels=0x4444;
	post_proc.audio_params.ds_stream_ID=0x5555;

#  ifdef AUDIO_PP_EQ_IN_USE
	post_proc.EQ_init_params.size=0;
	post_proc.EQ_init_params.max_order= 0x20;
	post_proc.EQ_init_params.max_bands= 0x6;
	post_proc.EQ_init_params.nb_samples= 240;
	
	post_proc.EQ_runtime_params.dsp_cmd=0;
	post_proc.EQ_runtime_params.use_eq=0;
	post_proc.EQ_runtime_params.size=0;
	// one empty 16 byte word
	post_proc.EQ_runtime_params.order = 0x14;
	// one empty 16 byte word
	post_proc.EQ_runtime_params.bands = 0x3;
	post_proc.EQ_runtime_params.band[0]= 0x398;
	post_proc.EQ_runtime_params.band[1]= 0x7d0;
	post_proc.EQ_runtime_params.band[2]= 0xfa0;
	post_proc.EQ_runtime_params.level_db[0] =0x0;
	post_proc.EQ_runtime_params.level_db[1] =0xfff4;
	post_proc.EQ_runtime_params.level_db[2] =0xfff4;
#  endif
	// set drc parameters
        post_proc.MUSICDRC_init_params.maxDelayCompanderMs=4;
        // sample sate
        post_proc.MUSICDRC_init_params.SampleRate=48;
        // for hpfilter
        post_proc.MUSICDRC_init_params.max_nbr_iir_blocks=1;
        //for limiter
        post_proc.MUSICDRC_init_params.maxDelayLimiterMs=1;

	post_proc.MUSICDRC_runtime_params.dsp_cmd=0x8888;

#if 1
	memcpy(&post_proc.MUSICDRC_runtime_params.linkCoeffSelf,
               &default_tables2,
               sizeof(IMUMDRC_Status));
#endif
 	post_proc.hp_filter_coeffs.use_hpfilt=1;
        post_proc.hp_filter_coeffs.coeffs[0]= 0;
        post_proc.hp_filter_coeffs.coeffs[1]= 5;
        post_proc.hp_filter_coeffs.coeffs[2]= -10;
        post_proc.hp_filter_coeffs.coeffs[3]= 1;
        post_proc.hp_filter_coeffs.coeffs[4]= -14941;
        post_proc.hp_filter_coeffs.coeffs[5]= 31259;
        post_proc.hp_filter_coeffs.coeffs[6]= 10;
        post_proc.hp_filter_coeffs.coeffs[7]= 15646;
        post_proc.hp_filter_coeffs.coeffs[8]= -31292;
        post_proc.hp_filter_coeffs.coeffs[9]= 15646;
        
	write(app_drv, &post_proc, sizeof(POSTPROC_INIT_PARAMS));
        read(app_drv, app_bridge_buffer, sizeof(DSP_CMD_STATUS));
	
	if(((DSP_CMD_STATUS *)app_bridge_buffer)->status != DSP_OK)
	{
		switch(((DSP_CMD_STATUS *)app_bridge_buffer)->status)
		{
			case DSP_ERROR_STATE: printf("\nError setting APP parameters: DSP_ERROR_STATE\n\n");break;
			case DSP_ERROR_RATE: printf("\nError setting APP parameters: DSP_ERROR_RATE\n\n");break;
			case DSP_ERROR_CHANNELS: printf("\nError setting APP parameters: DSP_ERROR_CHANNELS\n\n");break;
			case DSP_ERROR_DS_ID: printf("\nError setting APP parameters: DSP_ERROR_DS_ID\n\n");break;
			case DSP_ERROR_GENERAL: printf("\nError setting APP parameters: DSP_ERROR_GENERAL\n\n");break;
			case DSP_ERROR_FMT: printf("\nError setting APP parameters: DSP_ERROR_FMT\n\n");break;
		}
		return -1;
	}
	else printf("\nSetting up parameters for audiopp socket node successful.\n");
    }
    else
    {
        app_init_status.bridge_buffer_size = status_info.bridge_buffer_size;
	app_init_status.mmap_buffer_size = status_info.mmap_buffer_size;
	app_init_status.stream_ID = status_info.stream_ID;
	printf("\naudiopp socket node already initialised.");
    }	
    close(app_drv);
    return 0;
}
# endif

static int read_block(int fdd)
{
    struct pollfd pfd;
	
    pfd.fd = fdd;
    pfd.events = POLLIN;
    
    if(poll(&pfd,1,0)>0) return 0;
    else return 1;
}

static int write_volume(int volume_in)
{
    umask(0000);
    if((fd_vol=open(volfname,O_RDWR|O_CREAT,0666))<0){
        fprintf(stderr,"\nError open the %s.\n", volfname);
	return -1;
    }

    while(1) {
      if(is_write_unlock(fd_vol, 0, SEEK_SET, 0)>0) break;
    }

    writew_lock(fd_vol, 0, SEEK_SET, 0);
    write(fd_vol, &volume_in, sizeof(volume_in));
    un_lock(fd_vol, 0, SEEK_SET, 0);
    close(fd_vol);

    return 0;
}

static int set_hpvolume(char* sysfs_filename_in, int volume_in)
{
    FILE* sysfs_fp;
    int res=0;
    
    if((sysfs_fp=fopen(sysfs_filename_in,"w"))==NULL) { 
        fprintf(stderr, "Error: failed to open %s.\n", sysfs_filename_in);
	return -1;
    }
    	
    if((res=fprintf(sysfs_fp, "%d", hp_volumetable[(int)(volume_in/5)])) < 0) {
        fprintf(stderr,"Error: failed to set HEADPHONE volume: %d\n", res);
	//	fclose(sysfs_fp);
	//	return res;
    }
    fclose(sysfs_fp);
    return res;
}

static int set_lsvolume(int volume_in)
{
    if((app_drv=open(taskname,O_RDWR))<0) 
      fprintf(stderr,"\nError opening %s.\n",taskname);
        
    while(!read_block(app_drv)) 
      read(app_drv,&app_bridge_buffer[0],sizeof(short int));

    VOLUME_DATA mvi_volume_data;
    mvi_volume_data.dsp_cmd = DSP_CMD_SET_VOLUME;
    mvi_volume_data.scale  = 0x7FFF;
    
    if (volume_in >= 0)
	mvi_volume_data.power2 = ls_volumetable[(int)(volume_in/5)];
    else 
	mvi_volume_data.power2 = -ls_volumetable[(int)(-volume_in/5)];
    
    int ret_val;
    if((ret_val=write(app_drv, &mvi_volume_data, sizeof(mvi_volume_data))) != 
       sizeof(mvi_volume_data)) {
        fprintf(stderr,"Error: failed to set LOUDSPEAKER volume. Error: %d\n", ret_val);
	return -1;
    }

    volume_cur = volume_in;
    if(write_volume(volume_in)<0) {
      // Naturally satisfy -101 <= volume_in <= 100
      // error message will be shown in write_volume(volume_in).
      return -1;
    }

    read(app_drv,&app_bridge_buffer[0],sizeof(DSP_CMD_STATUS));
    close(app_drv);
    return 0;
}

static int lock_reg(int fd, int cmd, int type, off_t offset, int whence, off_t len)
{
    struct flock lock;
    
    lock.l_type = type;
    lock.l_start = offset;
    lock.l_whence = whence;
    lock.l_len = len;
    
    return (fcntl(fd, cmd, &lock));
}

static int is_write_unlock(int fd, off_t offset, int whence, off_t len)
{
    struct flock lock;
    
    lock.l_type = F_WRLCK;
    lock.l_start = offset;
    lock.l_whence = whence;
    lock.l_len = len;
    
    fcntl(fd, F_GETLK, &lock);
    if(lock.l_type==F_UNLCK) return 1;
    else return 0;
}


static int open_770(int type_in)
{
    type = type_in;

    volume_cur = route_get_gconf_volume ();
    if (volume_cur == 0) {
        volume_cur = MVI_getVolume();
    }
    if(volume_cur < -101 && volume_cur > 100) {
        // error message will be shown in read_volume().
        return -1;
    }

    if(type_in == HSS) {
# ifdef INIT_APOP
	init_audiopp();
# endif
	if((fd_hss=open(statfname,O_RDWR|O_CREAT,0666))<0) { 
	    fprintf(stderr,"\nError open the file.\n");
	    return -1;
	}

	int state_cur, ret_val=0;

	readw_lock(fd_hss, 0, SEEK_SET, 0);
	lseek(fd_hss,0,SEEK_SET);
	ret_val=read(fd_hss, &state_cur, sizeof(state));
	un_lock(fd_hss, 0, SEEK_SET, 0);
	close(fd_hss);

	if((state_cur != HEADPHONE) && (state_cur != LOUDSPEAKER)) {
	    // set default state, LOUDSPEAKER, when it first opens the file.
	    MVI_setState(LOUDSPEAKER);
	} 

	return fd_hss;

    } else if(type_in == MVA) {

        if((fd_mva=open(statfname,O_RDONLY))<0) {
	    fprintf(stderr,"\nError open the file.\n");
	    return -1;
	}

	close(fd_mva);
	return fd_mva;

    } else {
       fprintf(stderr, "\nUndefined unit.\n");
       return -2;
    }
}
#endif


int MVI_open	(void)
{
#ifdef HAVE_770
    int ret = 0;
    
    fprintf (stderr, "open_770(HSS)");
    if ((ret = open_770(HSS)) < 0) {
      return -1;
    }
    fprintf (stderr, "open_770(MVA)");
    if ((ret = open_770(MVA)) < 0) {
      return -1;
    }

#else
    snd_hctl_elem_t *elem;
    int err;
    int i;
    const char *name;

    if ((name = mvi_get_snd_card_name ()) && g_str_equal (name, "TLV320AIC33")) {
      ctl_names[CTL_LS_AMP] = "Line Playback Switch";
      ctl_names[CTL_HS_AMP] = "HP Playback Switch";
      ctl_names[CTL_HS_CAP_VOL] = "Mic1 Capture Volume";
      ctl_names[CTL_HS_CAP_SW] = "Mic1 Capture Switch";
      volume_hp = tlv_volume_hp;
      volume_ls = tlv_volume_ls;
    }

    err = snd_hctl_open(&(mvi->hctl), "master", 0);
    if(err < 0) {
        fprintf(stderr, "libmvi: snd_hctl_open(master) failed: %s\n", snd_strerror(err));
        return -1;
    }

    err = snd_hctl_load(mvi->hctl);
    if(err < 0) {
        fprintf(stderr, "libmvi: snd_hctl_load failed: %s\n", snd_strerror(err));
        snd_hctl_close(mvi->hctl);
        return -1;
    }

    elem = snd_hctl_first_elem(mvi->hctl);
    while(elem) {
        snd_ctl_elem_id_t *id;
        const char *name;

        snd_ctl_elem_id_alloca(&id);
        snd_hctl_elem_get_id(elem, id);
        name = snd_ctl_elem_id_get_name(id);

        for(i = 0; i < CTL_LAST; i++) {
            if(name && (strcmp(name, ctl_names[i]) == 0)) {
                add_control(mvi, elem, i);
                break;
            }
        }
        elem = snd_hctl_elem_next(elem);
    }

    if ((name = mvi_get_snd_card_name ()) && g_str_equal (name, "TLV320AIC33")) {
      set_control(mvi, CTL_CAP_VOL, mvi->ctl[CTL_CAP_VOL].max * 0.53); /* 35dB??? */
      set_control(mvi, CTL_HS_CAP_VOL, mvi->ctl[CTL_HS_CAP_VOL].max);
    } else {
      set_control(mvi, CTL_CAP_VOL, mvi->ctl[CTL_CAP_VOL].max);
      set_control(mvi, CTL_HS_CAP_VOL, mvi->ctl[CTL_HS_CAP_VOL].max);
    }

#endif
    return 1;
}

/* Sets microphone (i.e. capture) mute. Must only be used for power mgmt,
 * and should only be called by HSS. (Application mute is done in dsp.)
 * Currently this does NOT regard the Capture Source switch, i.e. it will
 * also mute the input if recording from LINEIN. If FM radio recording
 * (not just playback) is implemented, we'll need a function to set the
 * capture source, and the mic state will need to be saved somewhere.
 */
void MVI_setMicMute(int state)
{
#ifdef HAVE_770
  /* If this really is used for power management Hacker edition is not 
   * intrested
   */
#else
  const char *name;

  set_control (mvi, CTL_CAP_SW, !state);
  if (!MVI_getBT())
    set_control (mvi, CTL_PCM_CAP_SW, !state);

  if ((name = mvi_get_snd_card_name ()) && g_str_equal (name, "TLV320AIC33")) {
    if (get_control(mvi, CTL_HS_CAP_SW)) {
      return;
    }
    set_control(mvi, CTL_RX44_DIGITAL_MIC_SW, !state); /* TRUE mean Digital Mic ON */
  }
#endif
}

/* Sets state to either HEADPHONE, HEADSET or LOUDSPEAKER. The unused
 * fd_in parameter is a remnant from the old API.
 */
int MVI_setState(int state_in)
{
#ifdef HAVE_770

  fprintf (stderr,"%s - HEADPHONE=%d LOUDSPEAKER=%d", __FUNCTION__, HEADPHONE, 
	   LOUDSPEAKER);
  fprintf (stderr,"%s - state: %d, state_in %d", __FUNCTION__, state, state_in);

  /* These lines control APOP according to the connection status */
	if (state_in==HEADPHONE) { //APOP has to be disabled
	  hp_speaker_set_loudspeaker (DISABLED);
		if((app_drv=open(taskname,O_RDWR))<0) 
		  fprintf(stderr,"\nError opening %s.\n",taskname);
		audiopp_bypass_data.dsp_cmd = DSP_CMD_BYPASS_POSTPROCCESSING;
		audiopp_bypass_data.audio_pp_bypassed = 1;
		write(app_drv,&audiopp_bypass_data, sizeof(POSTPROCCESSING_BYPASS_DATA));
		read(app_drv,app_bridge_buffer,sizeof(DSP_CMD_STATUS));
		if(((DSP_CMD_STATUS *)app_bridge_buffer)->status != DSP_OK) 
		  printf("\nError disabling audio post-processing.\n");
		else 
		  printf("\n Audio Post processing is disabled  \n");
		close(app_drv);
	}	
	if (state_in==LOUDSPEAKER) { //APOP has to be enabled
	  hp_speaker_set_loudspeaker (ENABLED);
		if((app_drv=open(taskname,O_RDWR))<0) 
		  fprintf(stderr,"\nError opening %s.\n",taskname);
		audiopp_bypass_data.dsp_cmd = DSP_CMD_BYPASS_POSTPROCCESSING;
		audiopp_bypass_data.audio_pp_bypassed = 0;
		write(app_drv,&audiopp_bypass_data, sizeof(POSTPROCCESSING_BYPASS_DATA));
		read(app_drv,app_bridge_buffer,sizeof(DSP_CMD_STATUS));
		if(((DSP_CMD_STATUS *)app_bridge_buffer)->status != DSP_OK) 
		  printf("\nError enabling audio post-processing.\n");
		else 
		  printf("\n Audio Post processing is enabled  \n");
		close(app_drv);
	}
			
        state = state_in;
	if((fd_cmd=open(statfname,O_WRONLY))<0) {
	    fprintf(stderr,"\nError open the file.\n");
	    return -1;  
	}

	while(1) {
	  if(is_write_unlock(fd_cmd, 0, SEEK_SET, 0) > 0)
	    break;
	}
        writew_lock(fd_cmd, 0, SEEK_SET, 0);
	write(fd_cmd, &state, sizeof(state));
        un_lock(fd_cmd, 0, SEEK_SET, 0);
	close(fd_cmd);
	// MVI_setVolume(MVI_getVolume());

#else
    const char *name;

    if (state_in == HEADPHONE || state_in == HEADSET) {
        set_control(mvi, CTL_HS_AMP, 1);
        set_control(mvi, CTL_LS_AMP, 0);
	if ((name = mvi_get_snd_card_name ()) && g_str_equal (name, "TLV320AIC33")) {
	  set_control(mvi, CTL_RX44_DIGITAL_MIC_SW, state_in != HEADSET);
	} else {
	  set_control(mvi, CTL_CAP_SW, (state_in == HEADSET));
	}
	set_control(mvi, CTL_HS_CAP_SW, (state_in == HEADSET));
    }
    else if (state_in == LOUDSPEAKER) {
        set_control(mvi, CTL_LS_AMP, 1);
        set_control(mvi, CTL_HS_AMP, 0);
	set_control(mvi, CTL_HS_CAP_SW, FALSE);
	if ((name = mvi_get_snd_card_name ()) && g_str_equal (name, "TLV320AIC33")) {
	  set_control(mvi, CTL_RX44_DIGITAL_MIC_SW, TRUE); /* Enabled by default? */
	}
    }
    else {
        fprintf(stderr, "libmvi: Invalid state (%d)\n", state_in);
        return -1;
    }

    if ((name = mvi_get_snd_card_name ()) && g_str_equal (name, "TSC2301")) {
      if (state_in == HEADPHONE) {
	set_control(mvi, CTL_CAP_VOL, mvi->ctl[CTL_CAP_VOL].max * 0.79);
      } else if (state_in == LOUDSPEAKER) {
	set_control(mvi, CTL_CAP_VOL, mvi->ctl[CTL_CAP_VOL].max * 0.88);
      } else if (state_in == HEADSET) {
	set_control(mvi, CTL_CAP_VOL, mvi->ctl[CTL_CAP_VOL].max * 0.93);
      }
    }
#endif

    return 0;
}

/* Returns the current state, as determined from the amp and mic controls
 */
int MVI_getState()
{
#ifdef HAVE_770
    int state_cur;
    int ret_val=0;

    fprintf (stderr, __FUNCTION__);

    if((fd_cmd=open(statfname,O_RDONLY))<0){
        fprintf(stderr,"\nError open the file. it will use default value: LOUDSPEAKER.\n");
	return LOUDSPEAKER;
    }
    readw_lock(fd_cmd, 0, SEEK_SET, 0);
    lseek(fd_cmd,0,SEEK_SET);
    ret_val=read(fd_cmd, &state_cur, sizeof(state));
    un_lock(fd_cmd, 0, SEEK_SET, 0);
    close(fd_cmd);
    
    if(ret_val < 0)
    {
        fprintf(stderr,"Error: failed to read state. it will use default value: LOUDSPEAKER.\n");
        return LOUDSPEAKER;
    }

    return state_cur;
#else
    int ls, hs;
    int state = INVALID;

    ls = get_control(mvi, CTL_LS_AMP);
    hs = get_control(mvi, CTL_HS_AMP);

    if(ls && hs) {
        fprintf(stderr, "libmvi: Invalid state (both hs and ls enabled)\n");
    } else if(ls) {
        state = LOUDSPEAKER;
    } else if(get_control(mvi, CTL_HS_CAP_SW)) {
        state = HEADSET;
    } else {
        state = HEADPHONE;
    }

    return state;
#endif
}

/* Sets the master volume. Values [-100..-1] correspond to a muted
 * volume setting and -101 to muted 0 (this comes from the earlier API).
 */
int MVI_setVolume(int volume_in)
{
#ifdef HAVE_770
  fprintf (stderr,"%s - %d", __FUNCTION__, volume_in);
    if(volume_in >= -101 && volume_in <= 100)
    {
        if(MVI_getState()==HEADPHONE)
        {
	    if(volume_in > 0) {
	        set_lsvolume(100);
	        set_hpvolume(rsysfsname, (int) (volume_in));
		set_hpvolume(lsysfsname, (int) (volume_in));
	    }
	    else if (volume_in == 0) {
	      set_hpvolume(rsysfsname, (int)volume_in);
	      set_hpvolume(lsysfsname, (int)volume_in);
	    }
	    else {
	    	// In case of -ve volume level, set nil to sysfs.
	        set_hpvolume(rsysfsname, 0);
		set_hpvolume(lsysfsname, 0);
	    }

	    volume_cur = volume_in;
	    if(write_volume(volume_in)<0)
	    {
	        // Naturally satisfy -101 <= volume_in <= 100
	        // error message will be shown in write_volume(volume_in).
	        return -1;
	    }
	}
	else
	{
	  set_hpvolume(rsysfsname, 0);
	  set_hpvolume(lsysfsname, 0);
	  set_lsvolume(volume_in);
	}
    }
    else // volume_in is out of range
    {
        fprintf(stderr,"Error: wrong volume level: %d. Should be [-101:100]\n", volume_in);
	return -1;
    }
#else
  int state, vol, mute;
  const int *vtable;

  if (!(volume_in >= -101 && volume_in <= 100)) {
    ULOG_ERR ("libmvi: invalid volume %d (should be -101..100)\n", volume_in);
    return -1;
  }

  state = MVI_getState();
  vtable = (state == HEADSET || state == HEADPHONE) ? volume_hp : volume_ls;
  mute = (volume_in <= 0);
  vol = volume_in == -101 ? 0 : ABS (volume_in);

  set_control (mvi, CTL_PCM_SW, !mute);
  set_control (mvi, CTL_PCM_VOL, vtable[VOLINDEX(vol)]);
#endif
  return 0;
}

/* Finds the volume table index that corresponds most closely to vol.
 */
static int find_nearest(const int *vtable, int vol)
{
    int i, vi, vp;

    vi = vtable[VOLMAX];
    if(vol >= vi) return VOLMAX;

    for(i = VOLMAX-1; i >= 0; i--) {
        vp = vi;
        vi = vtable[i];
        if(vol == vi) return i;
        if(vol > vi) {
            if(vp-vol < vol-vi) return i+1;
            else return i;
        }
    }
    return 0;
}

/* Returns the current master volume (<0 if muted, or -101 if muted and
 * zero, as per the earlier API)
 */
int MVI_getVolume()
{
#ifdef HAVE_770
    int ret_val, volume=0;
    umask(0000);
    if((fd_vol=open(volfname,O_RDWR|O_CREAT,0666))<0) {
        fprintf(stderr,"\nError open the %s.\n", volfname);
	return -102; 
    }
    
    readw_lock(fd_vol, 0, SEEK_SET, 0);
    lseek(fd_vol,0,SEEK_SET);
    ret_val=read(fd_vol, &volume, sizeof(volume));
    un_lock(fd_vol, 0, SEEK_SET, 0);
    if(ret_val < 0) { 
      fprintf(stderr,"\nError read the %s.\n", volfname);
      return -102;
    }
    close(fd_vol);
    
    return volume;
#else
    int mute, vol, state;
    const int *vtable;

    state = MVI_getState();

    if(state == HEADSET || state == HEADPHONE) {
        vtable = volume_hp;
    } else {
        vtable = volume_ls;
    }

    vol  =  get_control(mvi, CTL_PCM_VOL);
    mute = !get_control(mvi, CTL_PCM_SW);

    vol = find_nearest(vtable, vol) * VOLSTEP;
    if(mute) {
        if(vol == 0) 
	  vol = -101;
        else 
	  vol = -vol;
    }

    return vol;
#endif
}

/* Closes mvi and frees any allocated resources.
 */
int MVI_close()
{
#ifdef HAVE_770
  /* FIXME: closes ???*/
  return 0;
#else
  if (mvi->hctl != NULL)
    snd_hctl_close(mvi->hctl);

  mvi->hctl = NULL;

  return 0;
#endif
}

/* Copies elem to mvi and fills in some info fields
 */
static void add_control(MVI_data *mvi, snd_hctl_elem_t *elem, int index)
{
    snd_ctl_elem_info_t *info;
    int type = 0;
    int min = 0;
    int max = 0;
    int count = 0;
    int err;

    snd_ctl_elem_info_alloca(&info);
    err = snd_hctl_elem_info(elem, info);

    if(err >= 0) {
        type  = snd_ctl_elem_info_get_type(info);
        count = snd_ctl_elem_info_get_count(info);

        if(type == SND_CTL_ELEM_TYPE_INTEGER) {
            min = snd_ctl_elem_info_get_min(info);
            max = snd_ctl_elem_info_get_max(info);
        }
        else if(type == SND_CTL_ELEM_TYPE_BOOLEAN) {
            min = 0;
            max = 1;
        }
    } else {
        fprintf(stderr, "libmvi: snd_hctl_elem_info failed (%s)\n", snd_strerror(err));
    }

    mvi->ctl[index].elem  = elem;
    mvi->ctl[index].type  = type;
    mvi->ctl[index].min   = min;
    mvi->ctl[index].max   = max;
    mvi->ctl[index].count = count;
}

/* Returns the current value of a control. Only boolean and integer
 * controls are supported, and for stereo controls the value for
 * channel 0 is returned.
 */
static int get_control(MVI_data *mvi, int index)
{
    snd_ctl_elem_value_t *ev;
    snd_hctl_elem_t *elem;
    int type;
    int err;

    elem = mvi->ctl[index].elem;
    type = mvi->ctl[index].type;

    if(!elem) {
        fprintf(stderr, "libmvi: control '%s' not found\n", ctl_names[index]);
        return 0;
    }

    snd_ctl_elem_value_alloca(&ev);
    err = snd_hctl_elem_read(elem, ev);
    if(err < 0) {
        fprintf(stderr, "libmvi: snd_hctl_elem_read failed: %s\n", snd_strerror(err));
        return 0;
    }

    if(type == SND_CTL_ELEM_TYPE_BOOLEAN) {
        return snd_ctl_elem_value_get_boolean(ev, 0);
    }
    else if(type == SND_CTL_ELEM_TYPE_INTEGER) {
        return snd_ctl_elem_value_get_integer(ev, 0);
    }
    else {
        fprintf(stderr, "libmvi: unsupported elem type %d\n", type);
        return 0;
    }
}

/* Changes the value of a control (setting both channels if stereo).
 * Only boolean and integer controls are supported.
 */
static void set_control(MVI_data *mvi, int index, int value)
{
    snd_hctl_elem_t *elem;
    snd_ctl_elem_value_t *ev;
    int count, type, min, max;
    int err;
    int i;

    elem  = mvi->ctl[index].elem;
    type  = mvi->ctl[index].type;
    count = mvi->ctl[index].count;
    min   = mvi->ctl[index].min;
    max   = mvi->ctl[index].max;

    if(!elem) {
        fprintf(stderr, "libmvi: ctl '%s' not found\n", ctl_names[index]);
        return;
    }

    snd_ctl_elem_value_alloca(&ev);

    err = snd_hctl_elem_read(elem, ev);
    if(err < 0) {
        fprintf(stderr, "libmvi: snd_hctl_elem_read failed: %s\n", snd_strerror(err));
        return;
    }

    value = CLAMP (value, min, max);

    for(i = 0; i < count; i++) {

        if(type == SND_CTL_ELEM_TYPE_BOOLEAN) {
            snd_ctl_elem_value_set_boolean(ev, i, value);
        }
        else if(type == SND_CTL_ELEM_TYPE_INTEGER) {
            snd_ctl_elem_value_set_integer(ev, i, value);
        }
        else {
            fprintf(stderr, "libmvi: unsupported elem type %d\n", type);
        }
    }

    err = snd_hctl_elem_write(elem, ev);
    if(err < 0) {
        fprintf(stderr, "libmvi: snd_hctl_elem_write failed: %s\n", snd_strerror(err));
    }
}

int
MVI_setBT		(int	enable)
{
#ifndef HAVE_770
  int				err, i;
  snd_ctl_elem_value_t		*ev = NULL;
  snd_ctl_elem_id_t		*id = NULL;
  snd_hctl_elem_t		*elem;
  const char			*name;
  const char *			ctl[] = {
    "BT Capture Switch",
    "BT Playback Switch",
    "BTM Capture Switch",
    "BTM Playback Switch",
  };

  snd_ctl_elem_id_alloca (&id);
  snd_ctl_elem_value_alloca (&ev);

  for (i = 0; i < sizeof (ctl)/sizeof (ctl[0]); ++i) {
    snd_ctl_elem_id_set_interface (id, SND_CTL_ELEM_IFACE_MIXER);
    snd_ctl_elem_id_set_name (id, ctl[i]);

    elem = snd_hctl_find_elem (mvi->hctl, id);
    if (elem == NULL) {
      fprintf (stderr, "Cannot find %s\n", ctl[i]);
      return 1;
    }

    snd_ctl_elem_value_set_boolean (ev, 0, enable);
    err = snd_hctl_elem_write (elem, ev);
    if (err) {
      fprintf (stderr, "Error while setting %s\n", ctl[i]);
      return 1;
    }
  }

  set_control (mvi, CTL_PCM_CAP_SW, !enable);
  set_control (mvi, CTL_EACPCM_SW, !enable);

  /* FIXME: this is perfectible */
  if (enable) {
    set_control (mvi, CTL_PCM_SW, TRUE);
    set_control(mvi, CTL_LS_AMP, 0);
    set_control(mvi, CTL_HS_AMP, 0);
    if ((name = mvi_get_snd_card_name ()) && g_str_equal (name, "TLV320AIC33")) {
      set_control(mvi, CTL_RX44_DIGITAL_MIC_SW, TRUE);
    } else {
      set_control(mvi, CTL_CAP_SW, FALSE);
    }
  } else {
    set_control(mvi, CTL_LS_AMP, !get_control(mvi, CTL_HS_AMP));
  }
#endif

  return 0;
}

int
MVI_getBT		(void)
{
#ifdef HAVE_770
  return 0;
#else
  snd_ctl_elem_id_t		*id = NULL;
  snd_ctl_elem_value_t		*ev;
  snd_hctl_elem_t		*elem;
  int				err;
  gboolean			rv;

  snd_ctl_elem_id_alloca (&id);
  snd_ctl_elem_value_alloca (&ev);

  /* here we just test one switch. ya-mvi branch check all */
  snd_ctl_elem_id_set_interface (id, SND_CTL_ELEM_IFACE_MIXER);
  snd_ctl_elem_id_set_name (id, "BTM Playback Switch");
  elem = snd_hctl_find_elem (mvi->hctl, id);

  err = snd_hctl_elem_read (elem, ev);
  if (err < 0) {
    fprintf (stderr, "libmvi: snd_hctl_elem_read failed: %s\n", snd_strerror(err));
    return 0;
  }

  rv = snd_ctl_elem_value_get_boolean (ev, 0);
  ULOG_DEBUG ("getbt %s", rv ? "on" : "off");

  return rv;
#endif
}

const char *
mvi_get_snd_card_name (void)
{
  snd_ctl_t		*ctlp;
  snd_ctl_card_info_t	*info;
  static char		card_name[64] = "";

  if (card_name[0] != 0)
    return card_name;

  snd_ctl_card_info_alloca (&info);

  if (snd_ctl_open (&ctlp, "master", 0) < 0) {
    ULOG_ERR ("could not open master ALSA ctl");
    return NULL;
  }

  if (snd_ctl_card_info (ctlp, info) < 0) {
    ULOG_ERR ("could not get info of master ALSA ctl");
    return NULL;
  }

  strncpy (card_name, snd_ctl_card_info_get_mixername (info), sizeof (card_name));

  if (ctlp != NULL)
    snd_ctl_close (ctlp);

  g_debug ("ALSA snd card (*hic*) is: %s", card_name);

  return card_name;
}
