/**
 * Copyright (C) 2005 Nokia Corporation
 * Contact: Makoto Sugano <makoto.sugano@nokia.com>
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#define _INCLUDE_POSIX_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <glib.h>
#include <gconf/gconf-client.h>
#include <signal.h>
#include <sys/wait.h>
#if DEBUG_MTRACE
# include <mcheck.h>
#endif
#include <alsa/asoundlib.h>

#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>

#include "common.h"
#include "hss.h"
#include "route.h"
#include "dsp.h"
#include "mvi.h"
#if ENABLE_INTERACTION
#include "interaction/interaction.h"
#endif
#ifdef HAVE_LIBPLAYBACK
# include "pb_manager.h"
#endif /* HAVE_LIBPLAYBACK */

#include "resource.h"
#include "bluez-audio-proxy.h"

#define HAVE_770

#ifdef HAVE_770
#define SYSFS_GPIO_HP_STATE	"/sys/devices/platform/gpio-switch/headphone/connection_switch"
#define SYSFS_DSP_STATE		"/tmp/.dsp_state"
#else
#define SYSFS_GPIO_HP_STATE	"/sys/devices/platform/gpio-switch/headphone/state"
#define SYSFS_DSP_STATE		"/sys/devices/platform/dsp/state"
#endif

static gboolean		_handle_plug_event			(void);
static gboolean		_handle_audio_enabled_event		(void);

static guint			audio_inactive_id;
static guint			gconf_volume_id;
static gboolean			audio_active = TRUE;
bluez_audio_proxy_t *		bluez_audio_proxy = NULL;
dsp_info_t *			dsp = NULL;


static void
sig_handler	(int sig)
{
#if DEBUG_MTRACE
  muntrace ();
#endif	/* DEBUG_MTRACE */

  route_deinitialize ();
#if HAVE_DSP
  if (dsp != NULL)
    dsp_close (dsp);
#endif	/* HAVE_DSP */

  ULOG_INFO("multimediad exiting in sig_handler");

  exit (EXIT_SUCCESS);
}

static DBusHandlerResult
_bluez_audio_hs_handler			(DBusConnection  *connection,
					 DBusMessage     *message,
					 void            *user_data)
{
  if (dbus_message_is_signal (message, "org.bluez.audio.Headset", "AnswerRequested")) {
    bluez_headset_proxy_t *headset;

    /* Forward signal to keep dbus signal compatibility with VoIP applications (FIXME with HAL or uevent) */

    if ((headset = bluez_audio_headset_get_default (bluez_audio_proxy)) == NULL)
      return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

    if (bluez_headset_get_state (headset) > BT_AUDIO_STATE_CONNECTED) {
      hss_signal_button ("button_pressed");
      hss_signal_button ("button_released");
    }

    bluez_headset_unref (headset);
  }

  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

void
_bluez_set_enable (gboolean enable)
{
  bluez_headset_proxy_t *headset;
  static gint		disable_count = 0;

  disable_count = enable ? disable_count - 1 : disable_count + 1;
  g_return_if_fail (disable_count >= 0);
  enable = (disable_count == 0);

  if (enable) {
    /* add auto-play & power-management */
    bluez_audio_headset_set_default_flags (bluez_audio_proxy, bluez_audio_headset_get_default_flags (bluez_audio_proxy) | BLUEZ_HEADSET_AUTO_PLAY | BLUEZ_HEADSET_PM, TRUE);
  } else {
    /* keep muted only */
    bluez_audio_headset_set_default_flags (bluez_audio_proxy, (bluez_audio_headset_get_default_flags (bluez_audio_proxy) & BLUEZ_HEADSET_MUTED), TRUE);
  }

  headset = bluez_audio_headset_get_default (bluez_audio_proxy);
  if (headset == NULL)
    return;

  if (enable) {
    /* add auto-play & power-management */
    bluez_headset_set_flags (headset, bluez_headset_get_flags (headset) | BLUEZ_HEADSET_AUTO_PLAY | BLUEZ_HEADSET_PM);
    /* if not muted and not playing -> play */
    if ((bluez_headset_get_state (headset) >= BT_AUDIO_STATE_CONNECTED && bluez_headset_get_state (headset) < BT_AUDIO_STATE_PLAYING)) {
      /* hack to not get sound on loudspeaker to long (you miss some audio) */
      route_force_bt (TRUE);
      if (!(bluez_headset_get_flags (headset) & BLUEZ_HEADSET_MUTED))
	bluez_headset_call (headset, "Play");
      else
	bluez_headset_set_state (headset, BT_AUDIO_STATE_PLAYING_PM);
    }
  } else {
    /* keep muted only */
    bluez_headset_set_flags (headset, (bluez_audio_headset_get_default_flags (bluez_audio_proxy) & BLUEZ_HEADSET_MUTED));
    /* if playing -> stop */
    if (bluez_headset_get_state (headset) >= BT_AUDIO_STATE_PLAYING_PM) {
      bluez_headset_call (headset, "Stop");
      bluez_headset_set_state (headset, BT_AUDIO_STATE_STOPPED);
    }
  }

  bluez_headset_unref (headset);
}

static void
_bluez_headset_set_muted (bluez_headset_proxy_t *headset, gboolean muted)
{
  if (headset == NULL)
    return;

  /* if it's not already muted, mute it */
  if (muted &&
      (!(bluez_headset_get_flags (headset) & BLUEZ_HEADSET_MUTED))) {
    bluez_headset_set_flags (headset, bluez_headset_get_flags (headset) | BLUEZ_HEADSET_MUTED); /* add muted */

    if (!MVI_getBT ())
      return;

    if (bluez_headset_get_state (headset) >= BT_AUDIO_STATE_PLAYING || bluez_headset_get_state (headset) == BT_AUDIO_STATE_NONE) {
      bluez_headset_set_state (headset, BT_AUDIO_STATE_PLAYING_PM);
      bluez_headset_call (headset, "Stop");
    }
  } else if (!muted) {
    bluez_headset_set_flags (headset, bluez_audio_headset_get_default_flags (bluez_audio_proxy) & ~BLUEZ_HEADSET_MUTED);
    if (bluez_headset_get_flags (headset) & BLUEZ_HEADSET_AUTO_PLAY) {
      if ((bluez_headset_get_state (headset) >= BT_AUDIO_STATE_CONNECTED && bluez_headset_get_state (headset) < BT_AUDIO_STATE_PLAYING)) {
	bluez_headset_call (headset, "Play");
      }
    }
  }
}

static void
_gconf_master_volume_cb		(GConfClient	*client, guint		cnxn_id,
				 GConfEntry	*entry, gpointer	user_data)
{
  bluez_headset_proxy_t		*headset;
  int				volume;
  gboolean			muted;

  volume = route_get_gconf_volume ();
  muted = volume <= 0;
  volume = volume * 15/100;

  if (muted) {
    bluez_audio_headset_set_default_flags (bluez_audio_proxy, bluez_audio_headset_get_default_flags (bluez_audio_proxy) | BLUEZ_HEADSET_MUTED, FALSE);
  } else {
    bluez_audio_headset_set_default_flags (bluez_audio_proxy, bluez_audio_headset_get_default_flags (bluez_audio_proxy) & ~BLUEZ_HEADSET_MUTED, FALSE);
  }

  if ((headset = bluez_audio_headset_get_default (bluez_audio_proxy))) {
    _bluez_headset_set_muted (headset, muted);
    if (!muted)
      bluez_headset_set_volume (headset, volume);
    bluez_headset_unref (headset);
  }
}

static gboolean
_handle_bluez_audio (DBusConnection *sysbus)
{
  DBusError		error;
  gboolean		success = FALSE;
  GConfClient		*client;
  GError		*err = NULL;

#define MATCH_RULE_BLUEZ_AUDIO_HS				\
  "type='signal',interface='org.bluez.audio.Headset'"

  dbus_error_init (&error);
  dbus_bus_add_match (sysbus, MATCH_RULE_BLUEZ_AUDIO_HS, &error);

  success = dbus_connection_add_filter (sysbus, _bluez_audio_hs_handler, NULL, NULL);
  if (!success) {
    g_warning ("Cannot add BT CONNECTED filter");
    dbus_bus_remove_match (sysbus, MATCH_RULE_BLUEZ_AUDIO_HS, NULL);
    return FALSE;
  }

  client = gconf_client_get_default ();
  g_return_val_if_fail (client != NULL, FALSE);
  gconf_volume_id = gconf_client_notify_add (client, ROUTE_GCONF_MASTER_VOLUME_PATH, _gconf_master_volume_cb, NULL, NULL, &err);
  if (err != NULL) {
    ULOG_CRIT ("Could not register gconf volume update-cb: %s", err->message);
    g_error_free (err);
    return FALSE;
  }
  g_object_unref (client);

  return TRUE;
}

static gboolean
_plug_event_watcher	(GIOChannel	*source,
			 GIOCondition	condition,
			 gpointer	user_data)
{
  GIOStatus		status;
  gchar			*state;
  gsize			length;
  GError		*error = NULL;
  static gboolean	bluez_disabled_by_this = FALSE;
  
  g_debug (__FUNCTION__);
  /* Sysfs files need to be reopened */
  g_io_channel_unref (source);
  source = g_io_channel_new_file (SYSFS_GPIO_HP_STATE, "r", NULL);
  if (source == NULL) {
    ULOG_ERR ("cannot create a source for GPIO");
    return FALSE;
  }

  ULOG_DEBUG ("state changed");
  status = g_io_channel_read_to_end (source, &state, &length, NULL);
  if (error != NULL) {
    ULOG_ERR ("error reading %s", error->message ? error->message : "(null)");
    g_error_free (error);
    error = NULL;
  }

  if (status == G_IO_STATUS_NORMAL && length > 0 && state) {
    if (g_strrstr (state, "dis") != NULL) {
      route_set_hp_plugged (FALSE);
      if (bluez_disabled_by_this) {
	bluez_disabled_by_this = FALSE;
	_bluez_set_enable (TRUE);
      }
    } else if (g_strrstr (state, "con") != NULL) {
      route_force_bt (FALSE);
      if (!bluez_disabled_by_this) {
	bluez_disabled_by_this = TRUE;
	_bluez_set_enable (FALSE);
      }
      route_set_hp_plugged (TRUE);
    } else {
      ULOG_ERR ("Unexpected state: %s", state);
    }
    g_free (state);
  }

  g_io_add_watch (source, G_IO_PRI | G_IO_ERR, _plug_event_watcher, NULL);

  return FALSE;
}

/* Reads the plug state (connect/disconnect), calls the respective
 * route function, and installs a handler to catch the next event */
static gboolean
_handle_plug_event (void)
{
  GIOChannel *ioc;

  ioc = g_io_channel_new_file (SYSFS_GPIO_HP_STATE, "r", NULL);
  if (ioc == NULL) {
    ULOG_ERR ("error opening %s", SYSFS_GPIO_HP_STATE);
    return FALSE;
  }

  _plug_event_watcher (ioc, 0, NULL);

  return TRUE;
}

static gboolean
_audio_inactive_timeout	(gpointer data)
{
  bluez_headset_proxy_t *headset;

  if (bluez_audio_is_active (bluez_audio_proxy) == FALSE)
    goto end;

  if ((headset = bluez_audio_headset_get_default (bluez_audio_proxy)) == NULL)
    goto end;

  if (bluez_headset_get_state (headset) >= BT_AUDIO_STATE_PLAYING_PM || bluez_headset_get_state (headset) == BT_AUDIO_STATE_NONE) {
    headset->state = BT_AUDIO_STATE_PLAYING_PM;
    bluez_headset_call (headset, "Stop");
  }

  bluez_headset_unref (headset);
end:
  audio_inactive_id = 0;
  return FALSE;
}

static void
_general_audio_set_active (gboolean active)
{
  bluez_headset_proxy_t *headset;

  if (audio_inactive_id != 0)
    g_source_remove (audio_inactive_id);
  audio_inactive_id = 0;

  if (active && !audio_active) {
    audio_active = TRUE;

    if (!bluez_audio_is_active (bluez_audio_proxy) || !(headset = bluez_audio_headset_get_default (bluez_audio_proxy)))
      return;

    if (bluez_headset_get_state (headset) >= BT_AUDIO_STATE_PLAYING_PM
	&& (bluez_headset_get_flags (headset) & (BLUEZ_HEADSET_AUTO_PLAY))
	&& !(bluez_headset_get_flags (headset) & (BLUEZ_HEADSET_MUTED))) {
      bluez_headset_call (headset, "Play");
    }

    bluez_headset_unref (headset);
  } else if (!active) {
    audio_active = FALSE;
    audio_inactive_id = g_timeout_add (5 * 1000, _audio_inactive_timeout, NULL);
  }
}

static gboolean
_audio_enabled_watcher	(GIOChannel	*source,
			 GIOCondition	condition,
			 gpointer	user_data)
{
  GIOStatus status;
  gsize length;
  gchar *state;
  GError *error = NULL;

  /* Sysfs files need to be reopened */
  g_io_channel_unref (source);
  source = g_io_channel_new_file (SYSFS_DSP_STATE, "r", NULL);
  if (source == NULL) {
    ULOG_ERR ("cannot create a source for ""audio enabled"" status");
    return FALSE;
  }

  status = g_io_channel_read_to_end (source, &state, &length, &error);
  if (error != NULL) {
    ULOG_ERR ("error reading %s", error->message ? error->message : "(null)");
    g_error_free (error);
    error = NULL;
  }

  if (status == G_IO_STATUS_NORMAL && length >= 2 && state) {
    switch (state[length - 2]) {
    case '0':
    case '2': {
      ULOG_DEBUG ("sysfs: audio disabled");
      _general_audio_set_active (FALSE);
      break;
    }
    case '1':
    case '3': {
      ULOG_DEBUG ("sysfs: audio enabled");
      _general_audio_set_active (TRUE);
      break;
    }
    default:
      ULOG_ERR ("Unexpected state: %s", state);
    }
    g_free (state);
  }

  g_io_add_watch (source, G_IO_PRI | G_IO_ERR, _audio_enabled_watcher, NULL);

  return FALSE;
}

static gboolean
_handle_audio_enabled_event (void)
{
  GIOChannel *ioc;

  ioc = g_io_channel_new_file (SYSFS_DSP_STATE, "r", NULL);
  if (ioc == NULL) {
    ULOG_ERR ("error opening %s", SYSFS_DSP_STATE);
    return FALSE;
  }

  _audio_enabled_watcher (ioc, 0, NULL);

  return TRUE;
}

static void
_bluez_headset_change_cb	(bluez_headset_proxy_t *headset, bluez_headset_notify_t event, gpointer data)
{
  int volume;

  g_return_if_fail (headset != NULL);

  if (event == BLUEZ_HEADSET_NOTIFY_STATE) {
    switch (headset->state) {
    case BT_AUDIO_STATE_PLAYING:
      g_warning ("Playing from bluez audio service, switch BT on");
    case BT_AUDIO_STATE_PLAYING_PM:
      route_force_bt (TRUE);

      if (!(bluez_headset_get_flags (headset) & BLUEZ_HEADSET_MUTED)) {
	/* force unmute */
	volume = route_get_gconf_volume ();
	if (volume < 0) {
	  route_set_gconf_volume (-volume);
	}
      }

      if (audio_active)
	return;

      /* Here we set a timer for audio inactivity if we get a state change
	 to play, no timer is already set, and there is no audio */

      if (audio_inactive_id != 0)
	g_source_remove (audio_inactive_id);
      audio_inactive_id = g_timeout_add (5 * 1000, _audio_inactive_timeout, NULL);

      break;
    case BT_AUDIO_STATE_NONE:
      ULOG_DEBUG ("None from bluez audio service, switch BT off");
      route_force_bt (FALSE);
      break;
    case BT_AUDIO_STATE_DISCONNECTED:
      ULOG_DEBUG ("Disconnected from bluez audio service, switch BT off");
      route_force_bt (FALSE);
      break;
    case BT_AUDIO_STATE_CONNECTED:
      ULOG_DEBUG ("Connected from bluez audio service, switch BT off");
      route_force_bt (FALSE);
      break;
    case BT_AUDIO_STATE_STOPPED:
      ULOG_DEBUG ("Stopped from bluez audio service, switch BT off");
      route_force_bt (FALSE);
      break;
    }
  } else if (event == BLUEZ_HEADSET_NOTIFY_VOLUME) {
    if (MVI_getBT ()) {
      volume = bluez_headset_get_volume (headset);
      g_warning ("bths vol %d", volume);
      gconf_client_notify_remove (gconf_client_get_default(), gconf_volume_id);
      volume = route_get_gconf_volume () < 0 ? -volume : volume;
      route_set_gconf_volume (volume * 100/15 + 1);
      gconf_volume_id = gconf_client_notify_add (gconf_client_get_default(), ROUTE_GCONF_MASTER_VOLUME_PATH, _gconf_master_volume_cb, NULL, NULL, NULL);
    }
  }
}

int
main			(int argc, char *argv[])
{
  GMainLoop		*loop;
#if HAVE_OSSO
static osso_context_t	*osso_context;
#endif
#if HAVE_DSP
  guint			try;
#endif
#if ENABLE_INTERACTION
  int			rv;
  struct sigaction	act;
#endif
  struct sigaction	sa = { .sa_handler = sig_handler };
  DBusConnection	*sysbus;
  DBusConnection	*bus;
  DBusError		derror;


#if DEBUG_MTRACE
  mtrace ();
#endif

  ULOG_OPEN (PACKAGE_NAME);

  sigaction (SIGBUS, &sa, NULL);
  sigaction (SIGTERM, &sa, NULL);
  sigaction (SIGINT, &sa, NULL);
  sigaction (SIGKILL, &sa, NULL);

  g_type_init ();
  loop = g_main_loop_new (NULL, FALSE);

#if HAVE_DSP
  for (try = 0; try < DSP_MAX_INIT_ITER; ++try) {
    g_debug ("try: %i", try+1);
    dsp = dsp_initialize (DSP_TASK_PATH, g_str_equal (mvi_get_snd_card_name () ? mvi_get_snd_card_name () : "", "TLV320AIC33") ? DSP_RX_44 : DSP_RX_NONE);
    if (dsp != NULL)
      break;
    ULOG_ERR ("DSP initialization round %d / %d failed", try + 1, DSP_MAX_INIT_ITER);
    sleep (DSP_INIT_SLEEP);
  }
  g_debug ("4");

  if (dsp == NULL) {
    ULOG_CRIT ("DSP initialization failed, giving up!");
    goto dsp_error;
  }
#endif	/* HAVE_DSP */

#if ENABLE_INTERACTION
  gboolean		fork_interaction = FALSE;

  if (fork_interaction == TRUE) {
    /* FIXME: do not ignore if interaction part terminate */
    act.sa_handler = SIG_IGN;
    sigemptyset (&act.sa_mask);
    act.sa_flags = SA_NOCLDWAIT;
    sigaction (SIGCHLD, &act, NULL);
    if ((rv = fork ()) == 0) {
      rv = interaction_init (argc, argv, loop, fork_interaction);
      exit (rv);
    } else if (rv == -1) {
      ULOG_ERR ("Error while forking the interaction part");
    }
  } else {
    interaction_init (argc, argv, loop, fork_interaction);
  }
#endif

  if (route_initialize (dsp) == FALSE) {
    ULOG_CRIT ("route initialization failed");
  }

#if HAVE_OSSO
  osso_context = osso_initialize (PACKAGE_NAME, PACKAGE_VERSION, TRUE, NULL);
  if (osso_context == NULL) {
    ULOG_CRIT("osso_initialize failed");
    return FALSE;
  }

  bus = osso_get_dbus_connection (osso_context);
  sysbus = osso_get_sys_dbus_connection (osso_context);
#else
  dbus_error_init (&derror);
  bus = dbus_bus_get (DBUS_BUS_SESSION, &derror);

  sysbus = bus;
  if (dbus_error_is_set (&derror)) {
    ULOG_CRIT (error.message);
    dbus_error_free (&derror);
  }
#endif	/* HAVE_OSSO */

  bluez_audio_proxy = bluez_audio (sysbus);
  bluez_audio_headset_set_updated_cb (bluez_audio_proxy, _bluez_headset_change_cb, NULL);
  _handle_bluez_audio (sysbus);
  _handle_plug_event ();
  _handle_audio_enabled_event ();

  resource_init (sysbus);

#ifdef ENABLE_LIBPLAYBACK
  pb_manager_t	*manager;

  manager = pb_manager_new (bus);
#endif /* ENABLE_LIBPLAYBACK */

  if (hss_initialize (bus) == FALSE) {
    ULOG_CRIT("Server initialization failed");
    goto server_error;
  }

  ULOG_INFO("multimediad initialized");
  g_main_loop_run (loop);

  g_main_loop_unref (loop);

  route_deinitialize ();
#if HAVE_DSP
  if (dsp != NULL)
    dsp_close (dsp);
#endif	/* HAVE_DSP */
  ULOG_INFO("multimediad exiting");
  exit (EXIT_SUCCESS);

 server_error:
  route_deinitialize ();
 route_error:
#if HAVE_DSP
  if (dsp != NULL)
    dsp_close (dsp);
#endif	/* HAVE_DSP */
#if HAVE_DSP
 dsp_error:
#endif	/* HAVE_DSP */
  exit (EXIT_FAILURE);
  return 1;
}
