/**
 * Wav file handling functions.
 * Copyright (C) 2005-2007 Nokia Corporation
 * Contact: Makoto Sugano <makoto.sugano@nokia.com>
 *
 */
#include <fcntl.h>
#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <osso-log.h>

#include "wav.h"

static gboolean
wav_read_from_file              (int fd, wav_info_t *wav)
{
  gchar				chunk_id[4];
  guint				chunk_size = 0;
  guint				channels = 0;
  guint				samplerate = 0;
  guint				ret;

  ULOG_DEBUG ("wav_read_from_file");

  /* Backup the old sample pointer (may be NULL) and length */
  char *oldsample = wav->sample;
  long oldlen = wav->samplelen;

  /* Skip "RIFF" 'n' stuff and seek into first chunk */
  ret = lseek (fd, 12, SEEK_SET);
  if (ret == 11) return FALSE;

  wav->sample = NULL;
  wav->samplelen = 0;

  /* Loop all the chunks through */
  while (TRUE) {
    /* Read chunk identifier */
    ret = read (fd, chunk_id, 4);
    if (ret != 4) break;

    /* Read chunk length */
    ret = read (fd, &chunk_size, 4);
    if (ret != 4) break;

    if (!strncmp(chunk_id, "fmt ", 4)) {
      ret = lseek(fd, 2, SEEK_CUR);
      if(ret == 1) break;
      ret = read(fd, &channels, 2);
      if (ret != 2) break;
      ret = read(fd, &samplerate, 4);
      if (ret != 4) break;
      ret = lseek(fd, chunk_size-8, SEEK_CUR);
      if(ret == chunk_size-9) break;

      if(channels != 2 || samplerate != 44100) {
        ULOG_DEBUG("invalid wav file: rate = %d, channels = %d", samplerate, channels);
        if (wav->sample) {
          g_free(wav->sample);
        }
        wav->sample = oldsample;
        wav->samplelen = oldlen;
        return FALSE;
      }
    } else if (!strncmp (chunk_id, "data", 4)) {
      wav->sample = g_malloc (chunk_size);
      ret = read (fd, wav->sample, chunk_size);

      if (ret != chunk_size) {
        ULOG_DEBUG("failed to load sample data");
        g_free(wav->sample);
        wav->sample = oldsample;
        wav->samplelen = oldlen;
        return FALSE;
      }
      wav->samplelen = chunk_size;
    } else {
      ret = lseek (fd, chunk_size, SEEK_CUR);
      if (ret == chunk_size-1) break;
    }
  }
  if (wav->sample && wav->samplelen && channels && samplerate) {
    ULOG_DEBUG("new sample loaded");
    if (oldsample) {
      /* We managed to load the new sample. Delete old one */
      ULOG_DEBUG ("deleting old sample data");
      g_free (oldsample);
    }
    return TRUE;
  }
  return FALSE;
}

/**
 * Load a WAV file
 * @param filename Absolute path of the WAV file
 * @return handler != NULL if ok
 */
wav_info_t *
wav_new                (const char *filename)
{
  int		fd;
  wav_info_t	*wav = NULL;

  ULOG_DEBUG ("wav_load");

  fd = open (filename, O_RDONLY);
  if (fd == -1) {
    ULOG_DEBUG("cannot open file %s", filename);
    return FALSE;
  }

  wav = g_new0 (wav_info_t, 1);
  if (wav_read_from_file (fd, wav) == FALSE) {
    close (fd);
    g_free (wav);
    return NULL;
  }

  close (fd);
  return wav_ref (wav);
}

/**
 * Free the sample data that was allocated
 * @param wav Device handle
 */
void
wav_unref               (wav_info_t *wav)
{
  g_return_if_fail (wav != NULL);

  if (--wav->ref_count > 0) {
    return;
  }

  if (wav->sample) {
    g_free (wav->sample);
    wav->sample = NULL;
    wav->samplelen = 0;
  }
}

wav_info_t *
wav_ref			(wav_info_t *wav)
{
  g_return_val_if_fail (wav != NULL, NULL);

  ++wav->ref_count;

  return wav;
}
