/**
 * DSP emulation functions for OSS.
 * Copyright (C) 2005 Nokia Corporation
 * Contact: Makoto Sugano <makoto.sugano@nokia.com>
 */
#include <fcntl.h>
#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/poll.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/soundcard.h>

#include "dsp.h"
#include "dsp_timer.h"

static void		dsp_info_reset		(dsp_info_s *dspinfo);
static gboolean		dsp_check_read		(dsp_info_s *dspinfo);
static gboolean		dsp_send_init		(dsp_info_s *dsp);
static gboolean		dsp_send_play		(dsp_info_s *dsp);
static gboolean		dsp_wait_access		(dsp_info_s *dsp);
static gboolean		dsp_set_volume		(dsp_info_s *dsp, guint volume);

/**
 * Check if there is some data in DSP node to be read
 */
gboolean 
dsp_check_read		(dsp_info_s *dsp)
{
  struct pollfd pfd;

  pfd.fd = dsp->fd;
  pfd.events = POLLIN;

  if(poll(&pfd,1,0)>0) return TRUE;
  else return FALSE;
}

/**
 * Initializes the dsp info structure into default values
 */
void 
dsp_info_reset		(dsp_info_s *dspinfo)
{
  dspinfo->name = NULL;
  dspinfo->fd = -1;
}

/**
 * Sends "init" command to dsp
 */
gboolean 
dsp_send_init		(dsp_info_s *dsp)
{
  return TRUE;
}

/**
 * Set the volume level for DSP
 */
gboolean 
dsp_set_volume		(dsp_info_s *dsp, guint volume)
{
  return TRUE;
}

/**
 * Start the playback on DSP
 */
gboolean 
dsp_send_play		(dsp_info_s *dsp)
{
  return TRUE;
}

/**
 * Initialize DSP
 * @param devname Path to the device to be opened
 * @return DSP data structure, or NULL if error
 */
dsp_info_s *
ias_dsp_initialize		(gchar *devname)
{
  dsp_info_s *dspinfo = NULL;  
  
  dspinfo = g_new0(dsp_info_s, 1);
  dsp_info_reset(dspinfo);

  dspinfo->fd = open("/dev/dsp", O_RDWR);
  if(dspinfo->fd == -1) {
    return NULL;
  }

  /* 0=mono 1=stereo */
  int channels=0;
  if ( ioctl(dspinfo->fd, SNDCTL_DSP_STEREO, &channels) == -1 ) {
    perror("ioctl stereo");
    return NULL;
  }
  if (channels != 1) {
    printf("Only mono supported\n");
  }

  int format = AFMT_S16_LE;

  if ( ioctl(dspinfo->fd, SNDCTL_DSP_SETFMT, &format) == -1 ) {
    perror("ioctl format");
    return NULL;
  }

  int rate = 16000;
  if ( ioctl(dspinfo->fd, SNDCTL_DSP_SPEED, &rate) == -1 ) {
    perror("ioctl sample rate");
    return NULL;
  }
  printf("DSP opened\n");
    
  dspinfo->name = g_strdup(devname);

  return dspinfo;
}

/**
 * Close DSP and free all reserved data structures.
 * @param dspinfo DSP handle
 */
void 
ias_dsp_close		(dsp_info_s *dspinfo)
{
  g_assert(dspinfo != NULL);
  
  if(dspinfo->fd) {
    close(dspinfo->fd);
    dspinfo->fd = -1;
  }
    
  g_free(dspinfo->name);
  g_free(dspinfo);
}

/**
 * Play the audio data given
 * @param dspinfo DSP handle
 * @param sample Audio data
 * @param sample_len Number of samples in audio data (in bytes)
 * @return TRUE if operation was successful
 */
  
gboolean 
ias_dsp_play_sample		(dsp_info_s *dsp, gchar *sample,
				 gulong sample_len, gint volume)
{
  /* FIXME elmarco (27/05/2007) ias_dsp_timer_handle_timer(dsp); ? */
 
  write(dsp->fd, sample, sample_len);
  ioctl(dsp->fd, SNDCTL_DSP_SYNC, 0);
  return TRUE;
}

/**
 * Recover the DSP from error
 * @param dspinfo DSP handle
 * @return TRUE of operation was successful
 */
gboolean 
ias_dsp_reset		(dsp_info_s *dsp)
{
  return TRUE;  
}

void 
ias_dsp_set_sleep		(dsp_info_s *dsp, gboolean mode)
{
  printf("DSP SET SLEEP: %d\n", mode);
}
