/**
 * Copyright (C) 2005-2007 Nokia Corporation
 * Contact: Makoto Sugano <makoto.sugano@nokia.com>
 */
#include <glib-object.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <tslib.h>
#include <unistd.h>
#include <linux/input.h>
#include <sys/select.h>
#include <osso-log.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "common.h"
#include "dsp.h"
#include "gconf.h"
#include "evdev.h"
#include "ias-dbus.h"
#include "wav.h"

static void             setup_signals           (void);
static void             ias_signal_handler      (int sig);

GMainLoop *loop;
ias_data_s ias = { 
  .dsp  = NULL, 
  .running = TRUE, 
  .locked = FALSE, 
  .muted = FALSE, 
  .ts_volume_gconf_id = -1,
  .kb_volume_gconf_id = -1,
  .master_volume_gconf_id = -1,
  .ts_dev = NULL,
  .ts_dev_pressed = FALSE,
  .evdev_list = NULL,
  .ts_wav = NULL,
  .ts_volume = 1,
  .kb_wav = NULL,
  .kb_volume = 1,
  .touchscreen_io_timeout_id = -1,
};

/**
 * Callback to be invoked when some signal arrives
 */
static void
ias_signal_handler              (int sig)
{
  GSList	*elem;

  ULOG_DEBUG("signal %d received, shutting down", sig);
  ias.running = FALSE;

  /*  When SIGBUS arrives, we might already be stuck. Deinitialize */
  /*    everything here instead and exit(). No need to worry about DSP. */
  if (sig == SIGBUS) {
    gconf_remove_callbacks (&ias);
    ias_dbus_finish ();

    ts_close (ias.ts_dev);

    for (elem = ias.evdev_list; elem != NULL; elem = g_slist_next (elem)) {
      evdev_unref (elem->data);
    }
    g_slist_free (ias.evdev_list);
    exit(0);
  }

  /* Otherwise we can just shutdown the g_main_loop and main() */
  /* will take care of the rest. */
  if (g_main_loop_is_running (loop)) {
    g_main_loop_quit (loop);
  }
}

static void
setup_signals			(void)
{
  struct sigaction	sa;

  sa.sa_handler = ias_signal_handler;
  sigaction (SIGBUS, &sa, NULL);
  sigaction (SIGTERM, &sa, NULL);
  sigaction (SIGINT, &sa, NULL);
  sigaction (SIGKILL, &sa, NULL);
}

void
interaction_finish		(void)
{
  GSList	*elem;

  gconf_remove_callbacks (&ias);
  ias_dbus_finish ();

  if (ias.ts_dev != NULL) {
    ts_close (ias.ts_dev);
  }

  for (elem = ias.evdev_list; elem != NULL; elem = g_slist_next (elem)) {
    if (evdev_has_type (elem->data, EV_KEY)) {
      wav_unref (ias.kb_wav);
    } else if (evdev_has_type (elem->data, EV_ABS)) {
      wav_unref (ias.ts_wav);
    }
    evdev_unref (elem->data);
  }
  g_slist_free (ias.evdev_list);

  ULOG_DEBUG ("input devices closed");

  ias_dsp_close (ias.dsp);
  ULOG_DEBUG ("dsp closed");
}

static gboolean
_play_wav_with_dsp (wav_info_t *wav, gint volume) 
{
  gboolean		ret;

  g_return_val_if_fail (wav != NULL && wav->sample != NULL, FALSE);

  if (ias.locked || ias.muted) {
    return TRUE;
  }

  ret = ias_dsp_play_sample (ias.dsp, wav->sample, wav->samplelen, volume);
  
  if (ret == FALSE) {
    if (ias_dsp_reset (ias.dsp) == FALSE) {
      ULOG_WARN ("Can't play sound, DSP failed.");
      return FALSE;
    }
  }

  return TRUE;
}

static gboolean            
_evdev_kb_watcher	(GIOChannel *source, GIOCondition condition, gpointer data)
{
  struct input_event            ev;
  evdev_t			*evdev = data;
  gsize				br;

  g_return_val_if_fail (evdev != NULL, FALSE);

  if (condition & (G_IO_PRI|G_IO_ERR|G_IO_HUP|G_IO_NVAL)) {
    g_warning ("error getting source event (stopping handler)");
    return FALSE;
  }

  /* This need more investigation:  it always return ERROR */
/*   if (g_io_channel_read_chars (evdev->io_channel, (gchar *)&ev, sizeof (struct input_event), &br, NULL) == G_IO_STATUS_ERROR) { */
/*     g_warning ("error while reading evdev input (continue handler)"); */
/*     return TRUE; */
/*   } */
  if (read (evdev->fd, &ev,sizeof (struct input_event)) < 0) {
    g_warning ("error while reading evdev input (continue handler)");
    return FALSE;
  }
	
  /* It must be either first press, or repeating press for arrow keys */
  if ((ev.value == 1) ||
      (ev.value == 2 && (ev.code == KEY_UP || ev.code == KEY_LEFT || ev.code == KEY_RIGHT || ev.code == KEY_DOWN))) {
    return _play_wav_with_dsp (ias.kb_wav, ias.kb_volume);
  } else if (ev.value == 0) {
    /* button released */
  }

  return TRUE;
}

static gboolean
touchscreen_io_timeout          (void)
{
  ias.touchscreen_io_timeout_id = -1;

  return FALSE;
}

gboolean            
_evdev_ts_watcher	(GIOChannel *source, GIOCondition condition, gpointer data)
{
  struct ts_sample		ev;

  if (ts_read (ias.ts_dev, &ev, 1) <= 0)
    return FALSE;

  if (ev.pressure > 0) {
    if (ias.ts_dev_pressed == TRUE)
      return TRUE;

    ias.ts_dev_pressed = TRUE;

    if (ias.touchscreen_io_timeout_id != -1) {
      return TRUE;
    } 

#define TOUCHSCREEN_IO_DELAY_IN_MS 100
    ias.touchscreen_io_timeout_id = g_timeout_add (TOUCHSCREEN_IO_DELAY_IN_MS, (GSourceFunc)touchscreen_io_timeout, NULL);

    return _play_wav_with_dsp (ias.ts_wav, ias.ts_volume);
  } 

  ias.ts_dev_pressed = FALSE;
    
  return TRUE;
}

static gboolean		
evdev_found	(evdev_t *evdev, gpointer data)
{
  g_return_val_if_fail (evdev != NULL, FALSE);

  if (evdev_open (evdev, O_RDONLY) < 0) {
    return FALSE;
  }
  
  if ((g_strstr_len (evdev_get_name (evdev), EVDEV_MAX_NAME_LEN, "keypad")
       || g_strstr_len (evdev_get_name (evdev), EVDEV_MAX_NAME_LEN, "pwrbutton")
       || g_strstr_len (evdev_get_name (evdev), EVDEV_MAX_NAME_LEN, "keyboard"))
      && evdev_has_type (evdev, EV_KEY)) {

    ias.evdev_list = g_slist_append (ias.evdev_list, evdev_ref (evdev));
    ias.kb_wav = ias.kb_wav == NULL ? wav_new (KB_SAMPLE_PATH) : wav_ref (ias.kb_wav);
    g_io_add_watch (evdev_get_io_channel (evdev), G_IO_IN|G_IO_PRI|G_IO_ERR|G_IO_HUP|G_IO_NVAL, _evdev_kb_watcher, evdev);

  } else if (g_strstr_len (evdev_get_name (evdev), EVDEV_MAX_NAME_LEN, "touchscreen")
	     && evdev_has_type (evdev, EV_ABS)) {
    /* TODO: have a proper implementation of tslib! */
    if ( (ias.ts_dev = ts_open (evdev_get_filename (evdev), 0)) != NULL) { 
      ts_config (ias.ts_dev);
      ias.evdev_list = g_slist_append (ias.evdev_list, evdev_ref (evdev));
      ias.ts_wav = ias.ts_wav == NULL ? wav_new (TS_SAMPLE_PATH) : wav_ref (ias.ts_wav);
      g_io_add_watch (g_io_channel_unix_new (ts_fd (ias.ts_dev)), G_IO_IN, _evdev_ts_watcher, evdev);
    }
  }

  return FALSE;
}

int
interaction_init			(int argc, char **argv, 
					 GMainLoop *main_loop, gboolean forked)
{
  guint turn = 0;

  ULOG_DEBUG ("Starting multimediad audio interaction...");

  for (turn = 0; turn < DSP_MAX_INIT_ITER; turn++) {
    ias.dsp = ias_dsp_initialize (DSP_NODE_NAME);
    if (ias.dsp != NULL) 
      break;

    ULOG_DEBUG ("DSP initialization round %d / %d failed", turn + 1, DSP_MAX_INIT_ITER);
    sleep (DSP_INIT_SLEEP);
  }
  g_return_val_if_fail (ias.dsp != NULL, 1);

  if (gconf_get_int (MASTER_VOLUME_GCONF_PATH) < 0)
    ias.muted = TRUE;

  ias.ts_volume = gconf_get_int (TS_VOLUME_GCONF_PATH);
  ULOG_DEBUG ("GConf volume for touchscreen: %d", ias.ts_volume);
  ias.ts_volume = CLAMP (ias.ts_volume, 0, 2);

  ias.kb_volume = gconf_get_int (KB_VOLUME_GCONF_PATH);
  ULOG_DEBUG ("GConf volume for touchscreen: %d", ias.kb_volume);
  ias.kb_volume = CLAMP (ias.kb_volume, 0, 2);

  evdev_foreach (evdev_found, NULL);

  if (forked) 
    setup_signals ();

  ias_dbus_initialize ();
  gconf_setup_callbacks (&ias);
  ias.locked = ias_dbus_query_lock ();

  if (forked) {
    loop = main_loop;
    g_main_loop_run (loop);
    interaction_finish ();
  }

  return 0;
}

