/**
 * Copyright (C) 2006-2007 Nokia Corporation
 * Contact: Makoto Sugano <makoto.sugano@nokia.com>
 */
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>
#include <signal.h>
#include <stdio.h>
#include <glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <libosso.h>
#include <osso-log.h>
#include <mce/dbus-names.h>
#include <mce/mode-names.h>

#include "common.h"
#include "ias-dbus.h"

extern ias_data_s       ias;
DBusConnection *        sysbus;

static DBusHandlerResult
ias_dbus_signal_handler         (DBusConnection *connection, DBusMessage *signal,
                                 gpointer user_data)
{
  DBusMessageIter		iter;
  const gchar			*data = NULL;

  if (dbus_message_is_signal(signal, MCE_SIGNAL_IF, MCE_TKLOCK_MODE_SIG)) {
    ULOG_DEBUG ("IAS: lock change signal");

    dbus_message_iter_init (signal, &iter);
    if (dbus_message_iter_get_arg_type (&iter) == DBUS_TYPE_STRING) {
      dbus_message_iter_get_basic (&iter, &data);
      if (!strcmp (data, MCE_TK_LOCKED)) {
        ULOG_DEBUG("IAS: Locked");
        ias.locked = TRUE;
      } else if (!strcmp (data, MCE_TK_UNLOCKED)) {
        ULOG_DEBUG("IAS: Unlocked");
        ias.locked = FALSE;
      }
    }
  }

  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

/**
 * ias_dbus_query_lock:
 *
 * Queries the TS/KB lock status
 *
 * Return value: TRUE = locked, FALSE = unlocked
 */
gboolean
ias_dbus_query_lock             (void)
{
  DBusMessage			*message, *reply;
  DBusError			error;
  gchar				*lockstatus;
  gboolean			ret = FALSE;

  dbus_error_init (&error);
  message = dbus_message_new_method_call (MCE_SERVICE, MCE_REQUEST_PATH,
                                          MCE_REQUEST_IF, MCE_TKLOCK_MODE_GET);

  reply = dbus_connection_send_with_reply_and_block (sysbus, message, -1, &error);

  if (dbus_error_is_set (&error)) {
    dbus_error_free (&error);
    dbus_message_unref (message);
    return ret;
  }

  if (!dbus_message_get_args (reply, &error,
                              DBUS_TYPE_STRING, &lockstatus,
                              DBUS_TYPE_INVALID)) {
    dbus_error_free (&error);
    dbus_message_unref (message);
    return ret;
  }

  if (!strncmp (lockstatus, MCE_TK_LOCKED, strlen(MCE_TK_LOCKED))) {
    ret = TRUE;
  }

  dbus_error_free (&error);
  dbus_message_unref (message);

  return ret;
}

/**
 * ias_dbus_initialize:
 *
 * Initialize D-BUS signal listener
 *
 * Return value: 0 for success, 1 for error
 */
int
ias_dbus_initialize     (void)
{
  DBusError		error;

  ULOG_DEBUG("IAS: initializing dbus signal listener");
  dbus_error_init (&error);

  sysbus = dbus_bus_get (DBUS_BUS_SYSTEM, &error);
  if (sysbus == NULL) {
    dbus_error_free (&error);
    ULOG_DEBUG("IAS: cannot get system bus");
    return 1;
  }

  dbus_connection_setup_with_g_main (sysbus, NULL);
  dbus_bus_add_match (sysbus, "type='signal',sender='" MCE_SERVICE "',interface='" MCE_SIGNAL_IF "'", &error);

  if (dbus_error_is_set (&error)) {
    ULOG_DEBUG ("IAS: error adding signal filter\n");
    dbus_error_free (&error);
    return 1;
  } else {
    if (!dbus_connection_add_filter (sysbus, ias_dbus_signal_handler,
                                     NULL, NULL)) {
      ULOG_DEBUG ("IAS: error adding signal handler\n");
      return 1;
    }
  }

  dbus_error_free (&error);
  return 0;
}

/**
 * Shut down D-BUS connection
 */
void
ias_dbus_finish       (void)
{
  DBusError error;

  dbus_error_init (&error);
  ULOG_DEBUG ("ias_dbus_finish");

  dbus_connection_remove_filter (sysbus, ias_dbus_signal_handler, NULL);
  dbus_bus_remove_match (sysbus, "type='signal',sender='" MCE_SERVICE "',interface='" MCE_SIGNAL_IF "'", &error);
  dbus_error_free (&error);
}
