/**
 * Copyright (C) 2007 Nokia Corporation
 * Contact: Marc-Andre Lureau <marc-andre.lureau@nokia.com>
 */
#include <stdio.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>

#include "evdev.h"

evdev_t *
evdev_new		(char *filename)
{
  evdev_t *		evdev;

  evdev = g_new0 (evdev_t, 1);
  g_return_val_if_fail (evdev != NULL, NULL);

  evdev->filename = filename;
  evdev->fd = -1;

  return evdev_ref (evdev);
}

const char *
evdev_get_name		(evdev_t *evdev)
{
  g_return_val_if_fail (evdev != NULL && evdev->fd != -1, NULL);

  if (ioctl (evdev->fd, EVIOCGNAME (sizeof (evdev->name)), evdev->name) >= 0) 
    return evdev->name;

  return NULL;
}

const char *
evdev_get_filename	(evdev_t *evdev)
{
  g_return_val_if_fail (evdev != NULL, NULL);

  return evdev->filename;
}

#define test_bit(bit, array)    (array[bit/8] & (1<<(bit%8)))

gboolean
evdev_has_type		(evdev_t *evdev, const int ev_type)
{
  g_return_val_if_fail (evdev != NULL && evdev->fd != -1, FALSE);

  if (ioctl (evdev->fd, EVIOCGBIT (0, sizeof (evdev->bitmask)), evdev->bitmask) < 0) {
    g_warning ("Unable to read EVIOCGBIT of %s", evdev->filename);
    return FALSE;
  }

  if (test_bit (ev_type, evdev->bitmask)) {
    return TRUE;
  }

  return FALSE;
}

int
evdev_open		(evdev_t *evdev, int flags)
{
  g_return_val_if_fail (evdev != NULL && evdev->filename != NULL, -1);

  if (evdev->fd != -1) {
    return evdev->fd;
  }

  evdev->fd = open (evdev->filename, flags);

  return evdev->fd;
}

GIOChannel *			
evdev_get_io_channel		(evdev_t *evdev)
{
  g_return_val_if_fail (evdev != NULL && evdev->fd != -1, NULL);

  if (evdev->io_channel == NULL) {
    evdev->io_channel = g_io_channel_unix_new (evdev->fd);
  }

  return evdev->io_channel;
}

void				
evdev_foreach		(evdev_func func, gpointer data)
{
  guint			devnum = 0;
  evdev_t		*dev = NULL;

  g_return_if_fail (func != NULL);

  for (devnum = 0; ; ++devnum) {
    dev = evdev_new (g_strdup_printf (EVDEV_BASENAME_FORMAT, devnum));

    if (access (dev->filename, R_OK) == -1) {
      evdev_unref (dev);
      break;
    }

    if ((func) (dev, data) == TRUE) {
      evdev_unref (dev);
      break;
    }

    evdev_unref (dev);
  }
}

evdev_t *			
evdev_ref		(evdev_t *evdev)
{
  g_return_val_if_fail (evdev != NULL, NULL);
  
  ++evdev->ref_count;

  return evdev;
}

void				
evdev_unref		(evdev_t *evdev)
{
  g_return_if_fail (evdev != NULL);

  if (--evdev->ref_count > 0)
    return;

  if (evdev->filename != NULL)
    g_free (evdev->filename);

  if (evdev->io_channel != NULL) {
    g_io_channel_unref (evdev->io_channel);
  }

  if (evdev->fd > -1) {
    close (evdev->fd);
  }
}

gboolean			
evdev_get_absinfo	(evdev_t *evdev, 
			 guint32 *min, guint32 *max, guint32 *fuzz, guint32 *flat)
{
  char			abs_bitmask[ABS_MAX/8 + 1] = {0, };
  struct input_absinfo	abs_info;
  int			yalv;

  g_return_val_if_fail (evdev != NULL, TRUE);

  if (ioctl (evdev->fd, EVIOCGBIT (EV_ABS, sizeof (abs_bitmask)), abs_bitmask) < 0) {
    g_warning ("evdev ioctl");
    return TRUE;
  }

  for (yalv = 0; yalv < ABS_MAX; yalv++) {
    if (test_bit(yalv, abs_bitmask)) {
      /* this means that the bit is set in the axes list */
      if (ioctl (evdev->fd, EVIOCGABS(yalv), &abs_info)) {
	g_warning ("evdev EVIOCGABS ioctl");
	return TRUE;
      }
      min ? (*min = abs_info.minimum) : 0;
      max ? (*max = abs_info.maximum) : 0;
      fuzz ? (*fuzz = abs_info.fuzz) : 0;
      flat ? (*flat = abs_info.flat) : 0;
      return FALSE;
    }
  }
  
  return TRUE;
}
