/**
 * DSP handling functions.
 * Copyright (C) 2005 Nokia Corporation
 * Contact: Makoto Sugano <makoto.sugano@nokia.com>
 */

#include <fcntl.h>
#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/poll.h>
#include <unistd.h>
#include <osso-log.h>

#include <dsp/interface_common.h>
#include <dsp/audio_dsptask.h>

#include "dsp.h"

#define BUF_SIZE 2000
guint volume_table[3] = { 0, 0x2666, 0x4333 };

static void dsp_info_reset(dsp_info_s *dspinfo);
static gboolean dsp_check_read(dsp_info_s *dspinfo);
static gboolean dsp_send_init(dsp_info_s *dsp);
static gboolean dsp_send_play(dsp_info_s *dsp);
static gboolean dsp_wait_access(dsp_info_s *dsp);
static gboolean dsp_set_volume(dsp_info_s *dspinfo, guint volume);

int count = 0;


/**
 * Check if there is some data in DSP node to be read
 */

static gboolean dsp_check_read(dsp_info_s *dsp)
{
  struct pollfd pfd;

  pfd.fd = dsp->fd;
  pfd.events = POLLIN;

  if(poll(&pfd,1,0)>0) return TRUE;
  else return FALSE;
}


/**
 * Initializes the dsp info structure into default values
 */

static void dsp_info_reset(dsp_info_s *dspinfo)
{
  ULOG_DEBUG("dsp_info_reset");
  dspinfo->name = NULL;
  dspinfo->fd = -1;
  dspinfo->curr_volume = 0;
  dspinfo->timeout_pending = FALSE;
  dspinfo->sleeping = FALSE;
  dspinfo->write_pending = 0;
}


/**
 * Sends "init" command to dsp
 */

static gboolean dsp_send_init(dsp_info_s *dsp)
{
  AUDIO_INIT_STATUS init_status;
  AUDIO_STATUS_INFO statInfo;
  AUDIO_PARAMS_DATA init_data;
  DSP_CMD_STATUS cmd;
  gint len;

  ULOG_DEBUG("dsp_send_init");

  // First we try to read all "old stuff"
  while(dsp_check_read(dsp)) {
    read(dsp->fd, &cmd, sizeof(unsigned short int));
  }

  statInfo.dsp_cmd = DSP_CMD_STATE;
  if(write(dsp->fd, &statInfo, sizeof(unsigned short int)) == -1) {
    ULOG_DEBUG("write failed");
    return FALSE;
  }

  if(read(dsp->fd, &statInfo, sizeof(AUDIO_STATUS_INFO)) == -1) {
    ULOG_DEBUG("read failed");
    return FALSE;
  }

  if(statInfo.status == STATE_UNINITIALISED) {
    ULOG_DEBUG("dsp_cmd_init");

    cmd.dsp_cmd = DSP_CMD_INIT;
    if(write(dsp->fd, &cmd, sizeof(unsigned short int)) == -1) {
      ULOG_DEBUG("write failed");
      return FALSE;
    }

    len = read(dsp->fd, &init_status, sizeof(AUDIO_INIT_STATUS));
    if(len == -1 || init_status.init_status != DSP_OK) {
      ULOG_DEBUG("read failed, status: %d", init_status.init_status);
      return FALSE;
    }

    dsp->dsp_sample_buffer_size = init_status.bridge_buffer_size;
    dsp->dsp_mmap_buffer_size = init_status.mmap_buffer_size;
  }
  else {
    dsp->dsp_sample_buffer_size = statInfo.bridge_buffer_size;
    dsp->dsp_mmap_buffer_size = statInfo.mmap_buffer_size;

    if(statInfo.status == STATE_PLAYING ||
       statInfo.status == STATE_PAUSED)
    {
      ULOG_DEBUG("dsp_cmd_stop");

      cmd.dsp_cmd = DSP_CMD_STOP;
      if(write(dsp->fd, &cmd, sizeof(unsigned short int)) == -1) {
        ULOG_DEBUG("write failed");
        return FALSE;
      }
      len = read(dsp->fd, &cmd, sizeof(DSP_CMD_STATUS));
      if(len == -1 || cmd.status != DSP_OK) {
        ULOG_DEBUG("read failed, status: %d", cmd.status);
        return FALSE;
      }
    }
  }

  dsp->mmap_buf_addr = (short int *) mmap((void *)0,
    dsp->dsp_mmap_buffer_size*sizeof(unsigned short int),
    PROT_READ | PROT_WRITE, MAP_SHARED, dsp->fd, 0);

  if(dsp->mmap_buf_addr == NULL) {
    ULOG_DEBUG("mmap failed");
    return FALSE;
  }

  dsp->dsp_write_size = dsp->dsp_mmap_buffer_size * \
      sizeof(unsigned short int);

  // Hack: we "quess" that initial tap is with this volume
  dsp_set_volume(dsp, 1);

  while(dsp_check_read(dsp)) {
    read(dsp->fd, &cmd, sizeof(unsigned short int));
  }

  // Set DSP parameters
  init_data.dsp_cmd = DSP_CMD_SET_PARAMS;
  init_data.audio_fmt = DSP_AFMT_S16_LE;
  init_data.sample_rate = SAMPLE_RATE_44_1KHZ;
  init_data.number_channels = CHANNELS_2;
  init_data.ds_stream_ID = 0;
  init_data.stream_priority = 0;

  ULOG_DEBUG("dsp_cmd_set_params");

  if(write(dsp->fd, &init_data, sizeof(AUDIO_PARAMS_DATA)) == -1) {
    ULOG_DEBUG("write failed");
    return FALSE;
  }

  len = read(dsp->fd, &cmd, sizeof(DSP_CMD_STATUS));

  if(len == -1 || cmd.status != DSP_OK) {
    ULOG_DEBUG("read failed, status: %d", cmd.status);
    return FALSE;
  }
  return TRUE;
}


/**
 * Set the volume level for DSP
 */

static gboolean dsp_set_volume(dsp_info_s *dsp, guint volume)
{
  VOLUME_DATA volume_data;

  ULOG_DEBUG("dsp_set_volume: %d", volume);

  volume_data.dsp_cmd = DSP_CMD_SET_VOLUME;
  volume_data.scale = volume_table[volume];
  volume_data.power2 = -1;

  if(write(dsp->fd, &volume_data, sizeof(volume_data)) == -1) {
    ULOG_DEBUG("write failed");
    return FALSE;
  }

  // The response is read later in "dsp_wait_access()" method

  return TRUE;
}


/**
 * Start the playback on DSP
 */

static gboolean dsp_send_play(dsp_info_s *dsp)
{
  unsigned short int tmp = DSP_CMD_PLAY;

  ULOG_DEBUG("dsp_send_play");

  if(write(dsp->fd, &tmp, sizeof(unsigned short int)) == -1) {
    ULOG_DEBUG("write failed");
    return FALSE;
  }
  return TRUE;
}


/**
 *  Wait for DSP_CMD_DATA_WRITE request
 */

static gboolean dsp_wait_access(dsp_info_s *dsp)
{
  DSP_CMD_STATUS incoming;
  short int tmp;
  gint done;

  while ( TRUE ) {

    done = read(dsp->fd, &incoming, sizeof(DSP_CMD_STATUS));

    if(done == -1 || incoming.status != DSP_OK) {
      ULOG_DEBUG("error occured, status: %d", incoming.status);
      return FALSE;
    }

    if(incoming.dsp_cmd == DSP_CMD_DATA_WRITE) {
      read(dsp->fd, &tmp, sizeof(unsigned short int));
      dsp->write_pending++;
    }

    // If there are more messages waiting, read them first
    if(dsp_check_read(dsp)) {
      continue;
    }

    if( dsp->write_pending ) {
      break;
    }
  }

  dsp->write_pending = 0;
  return TRUE;
}


/**
 * Initialize DSP
 * @param devname Path to the device to be opened
 * @return DSP data structure, or NULL if error
 */

dsp_info_s *ias_dsp_initialize(gchar *devname)
{
  ULOG_DEBUG("ias_dsp_initialize");

  dsp_info_s *dspinfo = g_new0(dsp_info_s, 1);
  dsp_info_reset(dspinfo);

  dspinfo->fd = open(devname, O_RDWR);
  if(dspinfo->fd == -1) {
    return NULL;
  }

  dspinfo->name = g_strdup(devname);

  if(!dsp_send_init(dspinfo)) {
    ias_dsp_close(dspinfo);
    return NULL;
  }

  // Initialize the volume to zero for starters
//   if(!dsp_set_volume(dspinfo, 0)) {
//     dsp_close(dspinfo);
//     return NULL;
//   }

  // Wait 10ms for the volume to take effect
//   usleep(10000);

  if(!dsp_send_play(dspinfo)) {
    ias_dsp_close(dspinfo);
    return NULL;
  }

  ULOG_DEBUG("ias_dsp_initialize ok");
  return dspinfo;
}


/**
 * Set DSP sleep mode on or off
 * @param dsp DSP handle
 * @param mode Sleep mode (TRUE = on, FALSE = off)
 *
 */

void ias_dsp_set_sleep(dsp_info_s *dsp, gboolean mode)
{
  if(dsp->sleeping == mode) {
    ULOG_DEBUG("set_sleep: dsp already at required state");
    return;
  }

  if(mode) {
    ULOG_DEBUG("dsp sleep on");
    dsp->sleeping = TRUE;
  }
  else {
    ULOG_DEBUG("dsp sleep on");
    dsp->sleeping = FALSE;
  }
}



/**
 * Play the audio data given
 * @param dspinfo DSP handle
 * @param sample Audio data
 * @param sample_len Number of samples in audio data (in bytes)
 * @return TRUE if operation was successful
 */

gboolean ias_dsp_play_sample(dsp_info_s *dsp,
			     gchar *sample,
			     gulong sample_len,
			     gint volume)
{
  // We are not using DSP sleep for now...
//  dsp_timer_handle_timer(dsp);

  if(!volume) {
    // Volume == 0 -> no need to play it
    return TRUE;
  }

  unsigned short int outbuf[2] = { DSP_CMD_DATA_WRITE, 0 };
  int tw, done, dataIndex = 0;

  if(dsp->curr_volume != volume) {
    ULOG_DEBUG("Changing volume to %d", volume);
    if(!dsp_set_volume(dsp, volume)) {
      return FALSE;
    }
    dsp->curr_volume = volume;
  }

  do {
    // Make sure that we have got DATA_WRITE!
    if(!dsp_wait_access(dsp)) {
      return FALSE;
    }

    tw = MIN(dsp->dsp_write_size, sample_len);

    memcpy(dsp->mmap_buf_addr, sample+dataIndex, tw);
    dataIndex += tw;
    sample_len -= tw;

    outbuf[1] = tw / sizeof(short int);
    done = write(dsp->fd, outbuf, 2*sizeof(unsigned short int));

    if(done == -1) {
      ULOG_DEBUG("write failed");
      return FALSE;
    }
  }
  while( sample_len > 0 );

  return TRUE;
}


/**
 * Recover the DSP from error
 * @param dspinfo DSP handle
 * @return TRUE of operation was successful
 */

gboolean ias_dsp_reset(dsp_info_s *dsp)
{
  DSP_CMD_STATUS cmd;
  short int tmp, len;

  ULOG_DEBUG("ias_dsp_reset");
  guint try = 0;

  munmap(dsp->mmap_buf_addr,
         dsp->dsp_mmap_buffer_size * sizeof(unsigned short int));

  while(dsp_check_read(dsp)) {
    read(dsp->fd, &tmp, sizeof(short int));
  }

  cmd.dsp_cmd = DSP_CMD_CLOSE;
  len = write(dsp->fd, &cmd, sizeof(short int));
  if(len != -1) {
    read(dsp->fd, &cmd, sizeof(DSP_CMD_STATUS));
  }

  close(dsp->fd);

  while( TRUE ) {
    sleep(1);  // Sleep for 1sec

    dsp->fd = open(dsp->name, O_RDWR);
    try++;

    if(dsp->fd != -1) {
      break;
    }

    // If DSP won't open within 10 seconds, give up
    if(try == 10) {
      return FALSE;
    }
  }

  if(!dsp_send_init(dsp)) {
    ias_dsp_close(dsp);
    return FALSE;
  }

  if(!dsp_send_play(dsp)) {
    ias_dsp_close(dsp);
    return FALSE;
  }

  dsp->curr_volume = 0;
  return TRUE;
}


/**
 * Close DSP and free all reserved data structures.
 * @param dspinfo DSP handle
 */

void ias_dsp_close(dsp_info_s *dsp)
{
  DSP_CMD_STATUS cmd;
  short int tmp, len;

  ULOG_DEBUG("ias_dsp_close");

  munmap(dsp->mmap_buf_addr,
         dsp->dsp_mmap_buffer_size * sizeof(unsigned short int));

  while(dsp_check_read(dsp)) {
    read(dsp->fd, &tmp, sizeof(short int));
  }

  cmd.dsp_cmd = DSP_CMD_CLOSE;
  len = write(dsp->fd, &cmd, sizeof(short int));
  if(len != -1) {
    read(dsp->fd, &cmd, sizeof(DSP_CMD_STATUS));
  }

  if(dsp->fd) {
    close(dsp->fd);
    dsp->fd = -1;
  }

  if(dsp->name) {
    g_free(dsp->name);
    dsp->name = NULL;
  }
  g_free(dsp);
}

static gboolean 
dsp_timer_timeout_func		(gpointer data)
{
  ULOG_DEBUG("dsp_timeout_func");
  dsp_info_s *dsp = (dsp_info_s *) data;

  ias_dsp_set_sleep (dsp, TRUE);

  dsp->timeout_id = 0;
  return FALSE;
}

void 
ias_dsp_timer_handle_timer		(dsp_info_s *dsp)
{
  /* Wake up DSP */
  ias_dsp_set_sleep (dsp, FALSE);
  
  /* Remove the old timeout function */
  if (dsp->timeout_id != 0) {
    g_source_remove (dsp->timeout_id);
    dsp->timeout_id = 0;
  }
      
  dsp->timeout_id = g_timeout_add (DSP_TIMER_TIMEOUT, dsp_timer_timeout_func, dsp);
}
