/**
 * Copyright (C) 2005-2007 Nokia Corporation
 * Contact: Makoto Sugano <makoto.sugano@nokia.com>
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <assert.h>

#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include "common.h"
#include "hss.h"
#include "route.h"
#include "resource.h"

#define HAVE_770

#ifdef HAVE_770
#include "bluez-audio-proxy.h"
#endif


static DBusHandlerResult _dbus_message_func	(DBusConnection *conn, DBusMessage *msg, gpointer user_data);

static const DBusObjectPathVTable server_vtable = {
  NULL,
  _dbus_message_func,
};

static DBusConnection *conn = NULL;

#define HSS_DBUS_OBJECT_PATH "/com/nokia/osso_hp_ls_controller"

#ifdef HAVE_770
# define SERVER_DBUS_KERNEL_SERVICE_NAME "org.kernel"
# define PLAYER_DBUS_KERNEL_INTERFACE_NAME "org.kernel.kevent"



#define HP_DEVICE_NAME "/sys/devices/platform/gpio-switch/headphone/connection_switch"

#define SERVER_DBUS_SIGNAL_LISTENER_FMT "type='signal',sender='%s',"\
"interface='%s'"

#define LS_DEVICE_NAME "/sys/devices/platform/gpio-switch/speaker/connection_switch"
#define HP_LS_SERVICE_NAME "com.nokia.osso_hp_ls_controller"

#define HP_STATE_ERROR 0
#define HP_STATE_UNINITIALIZED 1
#define HP_STATE_CONNECT 2
#define HP_STATE_DISCONNECT 3

#define STATUS_ERROR -1
#define STATUS_OK 0


static int hp_state = HP_STATE_UNINITIALIZED;
static int current_state = HP_STATE_UNINITIALIZED;
static int force_mode = 0;
static DBusConnection* sysbus;

static int hp_speaker_read_hp_status()
{
  int hp_device = open(HP_DEVICE_NAME, O_RDONLY);
  if (hp_device < 0) {
    return STATUS_ERROR;
  }

  char status[13];
  int num_read = -1;

  num_read = read(hp_device, (void*)&status, 13);

  int result;

  if (strncmp(status, "disconnected", 12)==0) {
    result = HP_STATE_DISCONNECT;
  }
  else if (strncmp(status, "connected", 9)==0) {
    result = HP_STATE_CONNECT;
  }
  else {
    result = HP_STATE_ERROR;
  }

  close(hp_device);
  return result;
}

static void hp_speaker_handle_change()
{
  hp_state = hp_speaker_read_hp_status();

  if (hp_state != HP_STATE_ERROR &&
      hp_state != current_state &&
      force_mode == 0)
  {
    if(hp_state == HP_STATE_CONNECT) {
      //g_debug("hp_speaker_set_hp_on();");
      route_force_bt (FALSE);
      //      if (!bluez_disabled_by_this) {
      //	bluez_disabled_by_this = TRUE;
	_bluez_set_enable (FALSE);
      //      }
      route_set_hp_plugged (TRUE);
    }
    else {
      //g_debug ("hp_speaker_set_hp_off();");
      route_set_hp_plugged (FALSE);
      //      if (bluez_disabled_by_this) {
	//	bluez_disabled_by_this = FALSE;
	_bluez_set_enable (TRUE);
	//      }
    }
    current_state = hp_state;
  }

}


/**
 * Catch kernel dbus message
 */
static DBusHandlerResult 
server_signal_handler(
    DBusConnection *connection,
    DBusMessage *signal,
    gpointer user_data)
{
  DBusHandlerResult msg_handled = DBUS_HANDLER_RESULT_HANDLED;

  if(dbus_message_is_signal(signal,
     PLAYER_DBUS_KERNEL_INTERFACE_NAME,
     "change") )
  {
    ULOG_DEBUG("HSS: change signal");
    hp_speaker_handle_change();

  }

  return msg_handled;
}



/**
 * server_initialize:
 * @argc: Number of command line arguments.
 * @argv: Command line arguments.
 *
 * Initialize D-BUS, register object paths and services.
 *
 * Return value: Return value for main program.
 */
static int initialize770(DBusConnection *bus)
{
    DBusError error;

    g_debug (__FUNCTION__);

    conn = bus;

    dbus_error_init(&error);

    // Add signal handler for HP status signal
    gchar *signal_rule = g_strdup_printf(
        SERVER_DBUS_SIGNAL_LISTENER_FMT,
        SERVER_DBUS_KERNEL_SERVICE_NAME,
        PLAYER_DBUS_KERNEL_INTERFACE_NAME );

    sysbus = dbus_bus_get(DBUS_BUS_SYSTEM, &error);
    if(sysbus == NULL) {
      g_debug ("HSS: error gettin sysbus");
      dbus_error_free(&error);
      return 1;
    }

    dbus_bus_add_match( sysbus, signal_rule, &error );
    g_free( signal_rule );

    if( dbus_error_is_set( &error ) ) {
      g_debug("HSS: error adding signal filter\n");
      dbus_error_free(&error);
      return 1;
    }
    else {
      if(!dbus_connection_add_filter( sysbus,
                                      server_signal_handler,
                                      NULL, NULL ))
      {
        g_debug("HSS: error adding signal handler\n");
        return 1;
      }
    }

    return 0;
}

#endif


static DBusMessage *
_force_loudspeaker	(DBusMessage	*msg, gboolean	force)
{
  static gboolean	bluez_disabled_by_this = FALSE;

  if (!route_force_loudspeaker (force)) {
    return dbus_message_new_error (msg, "com.nokia.osso_hp_ls_controller", "Forcing loudspeaker failed");
  }

  /* readibility count */
  if (force && !bluez_disabled_by_this) {
    bluez_disabled_by_this = TRUE;
    _bluez_set_enable (FALSE);
  }

  if (!force && bluez_disabled_by_this) {
    bluez_disabled_by_this = FALSE;
    _bluez_set_enable (TRUE);
  }

  return dbus_message_new_method_return (msg);
}

static DBusHandlerResult
_dbus_message_func		(DBusConnection *	conn,
				 DBusMessage *		msg,
				 gpointer		user_data)
{
  DBusMessage *reply = NULL;
  const char *sender = dbus_message_get_sender(msg);

  if (dbus_message_is_method_call (msg, "com.nokia.osso_hp_ls_controller.loudspeaker", "force_loudspeaker_on")) {
    ULOG_DEBUG("force_loudspeaker_on");
    reply = _force_loudspeaker (msg, TRUE);
  } else if (dbus_message_is_method_call (msg, "com.nokia.osso_hp_ls_controller.loudspeaker", "force_loudspeaker_off")) {
    ULOG_DEBUG("force_loudspeaker_off");
    reply = _force_loudspeaker(msg, FALSE);
  } else if (dbus_message_is_method_call(msg, "com.nokia.osso_hp_ls_controller.headset", "listen_button")) {
    ULOG_CRIT ("DEPRECATED: listen_button from %s", sender);
    resource_request_wrapper ("/com/nokia/osso/headset/button", sender, button_resource_handler);
    reply = dbus_message_new_method_return (msg);
  } else if (dbus_message_is_method_call (msg, "com.nokia.osso_hp_ls_controller.headset", "ignore_button")) {
    ULOG_CRIT ("DEPRECATED: ignore_button from %s", sender);
    resource_release_wrapper("/com/nokia/osso/headset/button", sender);
    reply = dbus_message_new_method_return(msg);
  }

  if (reply) {
    dbus_connection_send (conn, reply, NULL);
    dbus_message_unref (reply);
    return DBUS_HANDLER_RESULT_HANDLED;
  }
  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

/**
 * hss_signal_button:
 * bug NB#59313
 */
void
hss_signal_button			(const char *signal)
{
  DBusMessage *msg;

  msg = dbus_message_new_signal (HSS_DBUS_OBJECT_PATH, "com.nokia.osso_hp_ls_controller.headset", signal);
  dbus_connection_send (conn, msg, NULL);
  dbus_message_unref (msg);
}

/**
 * hss_initialize:
 * Return: TRUE on success
 */
int
hss_initialize			(DBusConnection *bus)
{
  const char	*server_object_path = HSS_DBUS_OBJECT_PATH;
  DBusError	error;

  g_return_val_if_fail (bus != NULL, FALSE);

  conn = bus;

  dbus_error_init (&error);
  if (!dbus_connection_register_object_path (bus, server_object_path, &server_vtable, NULL)) {
    ULOG_CRIT ("HSS: error registering object path");
    return FALSE;
  }

  if (dbus_bus_request_name (conn, "com.nokia.osso_hp_ls_controller", 0, &error) != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
    ULOG_CRIT ("HSS: error acquiring service");
    if (dbus_error_is_set (&error)) {
      ULOG_CRIT (error.message);
      dbus_error_free (&error);
    }
    return FALSE;
  }

#ifdef HAVE_770
  if(initialize770 (conn) != 0) {
    g_debug ("Problem: initialize770");
    return FALSE;
  }
#endif

  return TRUE;
}

