/**
 * DSP handling functions.
 * Copyright (C) 2005-2007 Nokia Corporation
 * Contact: Makoto Sugano <makoto.sugano@nokia.com>
 */

#include <fcntl.h>
#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>    /* for bzero() */
#include <sys/mman.h>
#include <sys/poll.h>
#include <unistd.h>

#include <dsp/interface_common.h>
#include <dsp/audio_dsptask.h>
#include <dsp/audiopp_dsptask.h>

#include "common.h"
#include "dsp.h"

#define HAVE_770

#ifndef HAVE_770
# ifdef HOST_ARM
#  if DSP_API_VERSION != 10
#   error Expecting DSP_API_VERSION == 10
#  endif
# endif
#endif

static void		dsp_info_reset		(dsp_info_t *dspinfo);
static gboolean		dsp_check_read		(dsp_info_t *dspinfo);
static gboolean		dsp_send_init		(dsp_info_t *dsp);

extern MUSICDRC_RUNTIME_PARAMS			musicdrc_ihf;
extern const MUSICDRC_RUNTIME_PARAMS		musicdrc_hp;
extern const MUSICDRC_RUNTIME_PARAMS		musicdrc_hs48;
extern const MUSICDRC_RUNTIME_PARAMS		musicdrc_bt;
extern const MUSICDRC_HPFILTER_COEFFS		hpfilter_ihf;
extern const MUSICDRC_HPFILTER_COEFFS		hpfilter_hp;
extern const MUSICDRC_HPFILTER_COEFFS		hpfilter_hs48;

/**
 * dsp_check_read:
 * Check if there is some data in DSP node to be read
 */
static gboolean
dsp_check_read(dsp_info_t *dsp)
{
  struct pollfd pfd;

  pfd.fd = dsp->fd;
  pfd.events = POLLIN;

  if (poll (&pfd,1,0) > 0)
    return TRUE;
  else
    return FALSE;
}

/**
 * dsp_info_reset:
 * Initializes the dsp info structure into default values
 */
static void
dsp_info_reset(dsp_info_t *dspinfo)
{
  ULOG_DEBUG("dsp_info_reset");
  dspinfo->name = NULL;
  dspinfo->fd = -1;
}

/**
 * dsp_send_init:
 * Sends "init" command to dsp
 */
static gboolean
dsp_send_init(dsp_info_t *dsp)
{
  /* APOP initialization code */
  AUDIO_STATUS_INFO status_info;
  DSP_CMD_STATUS cmd;

  short int app_bridge_buffer[BRIDGE_BUFFER_SIZE];
  AUDIO_INIT_STATUS app_init_status;

  /*   Audio Post Processing Parameters */
  POSTPROC_INIT_PARAMS post_proc;

  /* Clear the initialization struct just in case */
  bzero(&post_proc, sizeof(POSTPROC_INIT_PARAMS));

  ULOG_DEBUG("dsp_send_init");

  /* First we try to read all "old stuff" */
  while(dsp_check_read(dsp)) {
    read(dsp->fd, &cmd, sizeof(unsigned short int));
  }

  app_bridge_buffer[0] = DSP_CMD_STATE;
  if (write(dsp->fd,app_bridge_buffer,sizeof(short int)) == -1) {
    ULOG_ERR("write failed");
        return FALSE;
  }

#ifdef HOST_ARM
  if (read(dsp->fd,&status_info,sizeof(AUDIO_STATUS_INFO)) == -1) {
    ULOG_ERR("read failed");
    return FALSE;
  }
#endif

  if(status_info.status == STATE_UNINITIALISED)
  {
    app_bridge_buffer[0] = DSP_CMD_INIT;
    if (write(dsp->fd,app_bridge_buffer,sizeof(short int)) == -1) {
      ULOG_ERR("write failed");
      return FALSE;
    }

#ifdef HOST_ARM
    if (read(dsp->fd,&app_init_status, sizeof(app_init_status)) == -1) {
      ULOG_ERR("read failed");
      return FALSE;
    }
#endif
    if(app_init_status.init_status!=DSP_OK)
    {
      char *error;
      switch(app_init_status.init_status)
      {
        case DSP_ERROR_GENERAL:
          error = "DSP_ERROR_GENERAL";
          break;
        case DSP_ERROR_MEMORY:
          error = "DSP_ERROR_MEMORY";
          break;
        case DSP_ERROR_STATE:
          error = "DSP_ERROR_STATE";
          break;
        default:
          error = "unknown error";
          break;
      }
      ULOG_ERR("Error initialising audiopp socket node: %s", error);
      return FALSE;
    }
    else {
      ULOG_DEBUG("audiopp socket node is successfully initialised");
    }
  }

  return TRUE; /* FIXME: lots of code has been removed here, see svn history */
}


/**
 * dsp_initialize:
 * Initialize DSP
 * @param devname Path to the device to be opened
 * @return DSP data structure, or NULL if error
 */
dsp_info_t *
dsp_initialize			(const gchar *path, enum e_dsp_rx rx)
{
  ULOG_DEBUG("dsp_initialize");
# if defined MUSICDRC_MODE_IHF_RX44
  switch (rx) {
  case DSP_RX_44:
    ((MUSICDRC_RUNTIME_PARAMS *) (&musicdrc_ihf))->mode = MUSICDRC_MODE_IHF_RX44;
    break;
  case DSP_RX_34:
    ((MUSICDRC_RUNTIME_PARAMS *) (&musicdrc_ihf))->mode = MUSICDRC_MODE_IHF_RX34;
    break;
  default:
    break;
  }
# endif

  dsp_info_t *dspinfo = g_new0(dsp_info_t, 1);
  dsp_info_reset(dspinfo);

  dspinfo->fd = open(path, O_RDWR);
  if(dspinfo->fd == -1) {
    dsp_close(dspinfo);
    return NULL;
  }

  dspinfo->name = g_strdup(path);

  if(!dsp_send_init(dspinfo)) {
    dsp_close(dspinfo);
    return NULL;
  }

  ULOG_DEBUG("dsp_initialize ok");

#ifdef HAVE_770
  MVI_setVolume (90);
#endif
  return dspinfo;

}


/**
 * dsp_close:
 * Close DSP and free all reserved data structures.
 * @param dspinfo DSP handle
 */
void
dsp_close(dsp_info_t *dspinfo)
{
  ULOG_DEBUG("dsp_close");

  g_return_if_fail (dspinfo != NULL);

  if(dspinfo->fd > 0) {
    close(dspinfo->fd);
    dspinfo->fd = -1;
  }

  if(dspinfo->name) {
    g_free(dspinfo->name);
    dspinfo->name = NULL;
  }
  g_free(dspinfo);
}


/**
 * dsp_set_mode:
 * Sets musicdrc runtime params and hpfilter coeffs according to mode
 */
void dsp_set_mode(dsp_info_t *dspinfo, enum e_dsp_mode dspmode)
{
    const MUSICDRC_RUNTIME_PARAMS  *runtime = NULL;
    const MUSICDRC_HPFILTER_COEFFS *hpfilter = NULL;
    DSP_CMD_STATUS                 cmd_status;

    int fd, r, w;
    gboolean ok = TRUE;

    static enum e_dsp_mode current = -1;
    if (dspmode == current) return;

#ifndef ENABLE_DSP_SET_MODE
    ULOG_DEBUG ("dsp_set_mode disabled(%d)", dspmode);
    return;
#endif

#ifdef HOST_ARM
    ULOG_DEBUG("setting mode to %d", dspmode);

    fd = open("/dev/dsptask/audiopp", O_RDWR);
    if(fd < 0) {
        ULOG_ERR("Error opening %s. DSP mode not changed", "/dev/dsptask/audiopp");
        return;
    }
#ifdef HAVE_770
     switch(dspmode) {
    case DSP_MODE_IHF:
        runtime = &musicdrc_ihf;
        hpfilter = &hpfilter_ihf;
        break;
    case DSP_MODE_HP:
        runtime = &musicdrc_hp;
        hpfilter = &hpfilter_hp;
        break;
    case DSP_MODE_HS:
        runtime = &musicdrc_hs48;
        hpfilter = &hpfilter_hs48;
        break;
    case DSP_MODE_BT:
        runtime = &musicdrc_bt;
        hpfilter = NULL;
        break;
    default:
        ULOG_ERR("Invalid dsp mode: %d", dspmode);
        return;
    }
#else
    switch(dspmode) {
    case DSP_MODE_IHF:
        runtime = &musicdrc_ihf;
        hpfilter = &hpfilter_ihf;
        break;
    case DSP_MODE_HP:
        runtime = &musicdrc_hp;
        hpfilter = &hpfilter_hp;
        break;
    case DSP_MODE_HS:
        runtime = &musicdrc_hs48;
        hpfilter = &hpfilter_hs48;
        break;
    case DSP_MODE_BT:
        runtime = &musicdrc_bt;
        hpfilter = NULL;
        break;
    default:
        ULOG_ERR("Invalid dsp mode: %d", dspmode);
        return;
    }

    g_warning("=======================+> %d", runtime->mode);
    w = write(fd, runtime, sizeof(MUSICDRC_RUNTIME_PARAMS));
    r = read(fd, &cmd_status, sizeof(DSP_CMD_STATUS));

    if (w != sizeof(MUSICDRC_RUNTIME_PARAMS) ||
        r != sizeof(DSP_CMD_STATUS)          ||
        cmd_status.status != DSP_OK) {
        ULOG_ERR("Error changing musicdrc runtime params");
        ok = FALSE;
    }

    if (hpfilter == NULL)
      goto end;

    w = write(fd, hpfilter, sizeof(MUSICDRC_HPFILTER_COEFFS));
    r = read(fd, &cmd_status, sizeof(DSP_CMD_STATUS));

    if (w != sizeof(MUSICDRC_HPFILTER_COEFFS) ||
        r != sizeof(DSP_CMD_STATUS)           ||
        cmd_status.status != DSP_OK) {
        ULOG_ERR("Error changing musicdrc hpfilter coeffs");
        ok = FALSE;
    }
#endif

 end:
    close(fd);
    if(ok) current = dspmode;

#else
        ULOG_ERR("Invalid dsp mode: %d", dspmode);
        return;
#endif
}
