
/*
** Bluez audio/headset proxy
** (c) Nokia MM 2007
*/
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include "bluez-audio-proxy.h"

#define BLUEZ_AUDIO_PROXY_PATH "/com/nokia/bluez_audio_proxy"

static char *		_bluez_manager_activate_audio_service	(DBusConnection *dbus_connection);
static DBusHandlerResult _headset_proxy_handler			(DBusConnection *	conn,
								 DBusMessage *		msg,
								 gpointer		user_data);
static DBusHandlerResult _audio_proxy_handler			(DBusConnection *	conn,
								 DBusMessage *		msg,
								 gpointer		user_data);

typedef struct foreach_headset_data_s {
  headset_func_t	func;
  gpointer		user_data;
  GDestroyNotify	destroy_notify;
} foreach_headset_data_t;

struct s_state_string {
  bt_audio_state_t	state;
  char			*string;
} _bt_audio_state_string_table[] = {
  { BT_AUDIO_STATE_NONE,		"None" },
  { BT_AUDIO_STATE_DISCONNECTED,	"Disconnected" },
  { BT_AUDIO_STATE_CONNECTED,		"Connected" },
  { BT_AUDIO_STATE_STOPPED,		"Stopped" },
  { BT_AUDIO_STATE_PLAYING_PM,		"PlayingPowerManaged" },
  { BT_AUDIO_STATE_PLAYING,		"Playing" },
};

static const DBusObjectPathVTable _audio_vtable = {
  NULL,
  _audio_proxy_handler,
};

static const DBusObjectPathVTable _headset_vtable = {
  NULL,
  _headset_proxy_handler,
};

static bluez_audio_proxy_t	_audio_proxy = {
  0,
  NULL,
};

static inline DBusHandlerResult
_send_message_and_unref	(DBusConnection	*conn, DBusMessage	*msg)
{
  if (msg) {
    dbus_connection_send (conn, msg, NULL);
    dbus_message_unref (msg);
  }

  return DBUS_HANDLER_RESULT_HANDLED;
}

static const char *
_bt_audio_state_to_string (bt_audio_state_t state)
{
  int i;

  for (i = 0; i < G_N_ELEMENTS (_bt_audio_state_string_table); ++i) {
    if (state == _bt_audio_state_string_table[i].state)
      return _bt_audio_state_string_table[i].string;
  }

  g_warning ("could not convert state to string");
  return NULL;
}

static bt_audio_state_t
_bt_audio_string_to_state (const char *state)
{
  int i;

  for (i = 0; i < G_N_ELEMENTS (_bt_audio_state_string_table); ++i) {
    if (g_str_equal (state, _bt_audio_state_string_table[i].string))
      return _bt_audio_state_string_table[i].state;
  }

  g_warning ("could not convert string to state: %s", state);
  return BT_AUDIO_STATE_NONE;
}

void
bluez_headset_call		(bluez_headset_proxy_t *headset, const gchar *method)
{
  DBusError		derror;
  DBusMessage		*call;
  bluez_audio_proxy_t	*proxy = headset->audio_proxy;

  if (proxy->audio_service == NULL) {
    proxy->audio_service = _bluez_manager_activate_audio_service (proxy->sysbus);
  }

  g_return_if_fail (proxy->sysbus != NULL && proxy->audio_service != NULL);

  dbus_error_init (&derror);
  call = dbus_message_new_method_call (proxy->audio_service, headset->path, "org.bluez.audio.Headset", method);

  _send_message_and_unref (proxy->sysbus, call);
}

static char *
_bluez_manager_activate_audio_service (DBusConnection *dbus_connection)
{
  DBusError     derror;
  DBusMessage   *call, *reply;
  const char    *bt_audio_service = "audio";
  char		*dbus_connection_name;

  g_return_val_if_fail (dbus_connection != NULL, NULL);

  dbus_error_init (&derror);
  if (dbus_bus_name_has_owner (dbus_connection, "org.bluez", &derror) == FALSE)
    return NULL;

  call = dbus_message_new_method_call ("org.bluez", "/org/bluez", "org.bluez.Manager", "ActivateService");
  g_return_val_if_fail (call != NULL, NULL);

  dbus_message_append_args (call, DBUS_TYPE_STRING, &bt_audio_service, DBUS_TYPE_INVALID);
  reply = dbus_connection_send_with_reply_and_block (dbus_connection, call, -1, &derror);
  dbus_message_unref (call);
  if (dbus_error_is_set (&derror)) {
    g_warning ("Activate service returned %s", derror.message);
    dbus_error_free (&derror);
    return NULL;
  }
  g_return_val_if_fail (reply != NULL, NULL);

  dbus_message_get_args (reply, &derror, DBUS_TYPE_STRING, &dbus_connection_name, DBUS_TYPE_INVALID);
  if (dbus_error_is_set (&derror)) {
    g_warning ("Activate service get-args returned %s", derror.message);
    dbus_error_free (&derror);
    return NULL;
  }
  dbus_connection_name = dbus_connection_name ? g_strdup (dbus_connection_name) : NULL;
  dbus_message_unref (reply);

  /* on activated event, should be handled elsewhere */
  _audio_proxy.audio_service = dbus_connection_name;
  bluez_audio_headset_foreach (&_audio_proxy, NULL, bluez_audio_ref (&_audio_proxy), (GDestroyNotify)bluez_audio_unref);

  return dbus_connection_name;
}

static void
_hfunc_headset_set_state	(const gchar *path, bluez_headset_proxy_t *headset, gpointer s)
{
  bt_audio_state_t	state = (bt_audio_state_t)s;

  g_return_if_fail (path != NULL);
  g_return_if_fail (headset != NULL);

  bluez_headset_set_state (headset, state);
}

static DBusHandlerResult
_dbus_signal_handler (DBusConnection  *connection,
		      DBusMessage     *message,
		      void            *user_data)
{
  DBusError		derror;
  const char		*name, *old, *new;
  bluez_audio_proxy_t	*proxy = user_data;

  g_return_val_if_fail (proxy != NULL, DBUS_HANDLER_RESULT_NOT_YET_HANDLED);

  dbus_error_init (&derror);

  if (dbus_message_is_signal (message, "org.freedesktop.DBus", "NameOwnerChanged")) {
    dbus_message_get_args (message, &derror,
			   DBUS_TYPE_STRING, &name,
			   DBUS_TYPE_STRING, &old,
			   DBUS_TYPE_STRING, &new,
			   DBUS_TYPE_INVALID);

    if (dbus_error_is_set (&derror)) {
      g_warning (derror.message);
      dbus_error_free (&derror);
      return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    /* audio service disappears */
    if (proxy->audio_service &&
	g_str_equal (name, proxy->audio_service) && (g_str_equal (new, ""))) {
      g_free (proxy->audio_service);
      proxy->audio_service = NULL;
      g_hash_table_foreach (proxy->headsets, (GHFunc)_hfunc_headset_set_state, (gpointer)BT_AUDIO_STATE_NONE);
    }

    /* hcid appears */
    if ((proxy->audio_service == NULL) && g_str_equal (name, "org.bluez") && (g_str_equal (new, "") == FALSE)) {
      _audio_proxy.audio_service = _bluez_manager_activate_audio_service (_audio_proxy.sysbus);
    }
  }

  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static DBusHandlerResult
_audio_proxy_handler			(DBusConnection *	conn,
					 DBusMessage *		msg,
					 gpointer		user_data)
{
  DBusMessage			*msg_return;
  const char			*iface, *prop;
  DBusError			error;
  bluez_audio_proxy_t		*proxy = user_data;
  bluez_headset_proxy_t		*headset = NULL;

  g_return_val_if_fail (proxy != NULL, DBUS_HANDLER_RESULT_NOT_YET_HANDLED);

  dbus_error_init (&error);
  if (dbus_set_error_from_message (&error, msg)) {
    g_warning (error.message);
    dbus_message_unref (msg);
    dbus_error_free (&error);
    return DBUS_HANDLER_RESULT_HANDLED;
  }

  if (dbus_message_is_method_call (msg, DBUS_INTERFACE_PROPERTIES, "Get")) {
    dbus_message_get_args (msg, &error,
			   DBUS_TYPE_STRING, &iface,
			   DBUS_TYPE_STRING, &prop,
			   DBUS_TYPE_INVALID);
    if (dbus_error_is_set (&error)) {
      g_warning ("Error with msg properties: %s", error.message);
      dbus_error_free (&error);
      return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }
    if (g_str_equal (prop, "DefaultHeadset")) {
      if ((headset = bluez_audio_headset_get_default (proxy)) == NULL) {
	msg_return = dbus_message_new_error (msg, "com.nokia.FIXME", "No default headset");
	g_return_val_if_fail (msg_return != NULL, DBUS_HANDLER_RESULT_NOT_YET_HANDLED);

	_send_message_and_unref (_audio_proxy.sysbus, msg_return);
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
      }

      msg_return = dbus_message_new_method_return (msg);
      g_return_val_if_fail (msg_return != NULL, DBUS_HANDLER_RESULT_NOT_YET_HANDLED);

      dbus_message_append_args (msg_return,
				DBUS_TYPE_STRING, &headset->path,
				DBUS_TYPE_INVALID);
      _send_message_and_unref (headset->audio_proxy->sysbus, msg_return);
      bluez_headset_unref (headset);
      return DBUS_HANDLER_RESULT_HANDLED;
    } else {
      msg_return = dbus_message_new_error (msg, DBUS_ERROR_INVALID_ARGS, "Invalid property name");
      g_return_val_if_fail (msg_return != NULL, DBUS_HANDLER_RESULT_NOT_YET_HANDLED);

      _send_message_and_unref (headset->audio_proxy->sysbus, msg_return);
      return DBUS_HANDLER_RESULT_HANDLED;
    }
  }

  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static DBusHandlerResult
_bluez_headset_introspect	(bluez_audio_proxy_t	*proxy,
				 DBusMessage		*msg)
{
  DBusMessage	*reply;
  const char	*path;
  const char	introspect[] = \
    "<!DOCTYPE node PUBLIC \"-//freedesktop//DTD D-BUS Object Introspection 1.0//EN\""
    " \"http://www.freedesktop.org/standards/dbus/1.0/introspect.dtd\">\n"
    "<node name=""/"">\n"
    "<interface name=\"com.nokia.bluez_headset_proxy\">\n"
    " <property name=\"State\" type=\"s\" access=\"read\">\n"
    "</interface>\n"
    "</node>";
  const char	*ptr = introspect;

  path = dbus_message_get_path (msg);
  g_message ("Introspect path: %s", path);

  if (!dbus_message_has_signature (msg, DBUS_TYPE_INVALID_AS_STRING)) {
    g_warning ("Unexpected signature to introspect call");
    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
  }

  reply = dbus_message_new_method_return (msg);
  if (reply == NULL)
    return DBUS_HANDLER_RESULT_NEED_MEMORY;

  dbus_message_append_args (reply,
			    DBUS_TYPE_STRING, &ptr,
			    DBUS_TYPE_INVALID);

  return _send_message_and_unref (proxy->sysbus, reply);
}

static DBusHandlerResult
_headset_proxy_handler			(DBusConnection *	conn,
					 DBusMessage *		msg,
					 gpointer		user_data)
{
  bluez_headset_proxy_t		*headset = user_data;
  DBusMessage			*msg_return;
  DBusError			error;
  const char			*iface, *prop, *val;

  g_return_val_if_fail (headset != NULL, DBUS_HANDLER_RESULT_NOT_YET_HANDLED);

  dbus_error_init (&error);
  if (dbus_set_error_from_message (&error, msg)) {
    g_warning (error.message);
    dbus_message_unref (msg);
    dbus_error_free (&error);
    return DBUS_HANDLER_RESULT_HANDLED;
  }

  if (dbus_message_is_method_call (msg, DBUS_INTERFACE_PROPERTIES, "Get")) {
    dbus_message_get_args (msg, &error,
			   DBUS_TYPE_STRING, &iface,
			   DBUS_TYPE_STRING, &prop,
			   DBUS_TYPE_INVALID);
    if (dbus_error_is_set (&error)) {
      g_warning ("Error with msg properties: %s", error.message);
      dbus_error_free (&error);
      return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }
    if (g_str_equal (prop, "State")) {
      msg_return = dbus_message_new_method_return (msg);
      g_return_val_if_fail (msg_return != NULL, DBUS_HANDLER_RESULT_NOT_YET_HANDLED);

      const char * state = _bt_audio_state_to_string (bluez_headset_get_state (headset));
      dbus_message_append_args (msg_return,
				DBUS_TYPE_STRING, &state,
				DBUS_TYPE_INVALID);
      _send_message_and_unref (headset->audio_proxy->sysbus, msg_return);
      return DBUS_HANDLER_RESULT_HANDLED;
    } else {
      msg_return = dbus_message_new_error (msg, DBUS_ERROR_INVALID_ARGS, "Invalid property name");
      g_return_val_if_fail (msg_return != NULL, DBUS_HANDLER_RESULT_NOT_YET_HANDLED);

      _send_message_and_unref (headset->audio_proxy->sysbus, msg_return);
      return DBUS_HANDLER_RESULT_HANDLED;
    }
  } else if (dbus_message_is_method_call (msg, DBUS_INTERFACE_PROPERTIES, "Set")) {
    dbus_message_get_args (msg, &error,
			   DBUS_TYPE_STRING, &iface,
			   DBUS_TYPE_STRING, &prop,
			   DBUS_TYPE_STRING, &val,
			   DBUS_TYPE_INVALID);
    if (dbus_error_is_set (&error)) {
      g_warning ("Error with msg properties: %s", error.message);
      dbus_error_free (&error);
      return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }
    if (g_str_equal (prop, "State")) {
      /* refactor this */
      if (g_str_equal (val, "Playing")) {
	bluez_headset_call (headset, "Play");
	/* bluez_headset_set_state (headset, _bt_audio_string_to_state (val)); */
      } else if (g_str_equal (val, "Disconnected")) {
	bluez_headset_call (headset, "Disconnect");
	bluez_headset_set_state (headset, _bt_audio_string_to_state (val));
      }
      return DBUS_HANDLER_RESULT_HANDLED;
    }
  }

  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

bluez_headset_proxy_t *
bluez_audio_headset_get			(bluez_audio_proxy_t *proxy, const char *path)
{
  gboolean			success;
  bluez_headset_proxy_t *	headset;

  g_return_val_if_fail (proxy != NULL && path != NULL, NULL);

  if ((headset = g_hash_table_lookup (proxy->headsets, path)) != NULL) {
    return bluez_headset_ref (headset);
  }

  headset = g_new0 (bluez_headset_proxy_t, 1);
  headset->path = g_strdup (path);
  headset->audio_proxy = bluez_audio_ref (proxy);
  headset->refcount = 1;
  headset->req_volume = -1;

  success = dbus_connection_register_object_path (proxy->sysbus, path, &_headset_vtable, headset);
  if (success == FALSE) {
    g_warning ("could not create a bluez audio proxy object");
  }

  g_hash_table_insert (proxy->headsets, (gpointer)headset->path, headset);

  bluez_headset_set_flags (headset, proxy->headset_default_flags);

  return bluez_headset_ref (headset);
}

static void
_find_default				(const gchar *key, bluez_headset_proxy_t *proxy, gpointer user_data)
{
  bluez_headset_proxy_t **	headset = user_data;

  /* FIXME: what is a default?? audio service default ? */
  if (*headset == NULL || proxy->state > (*headset)->state)
    *headset = proxy;
}

bluez_headset_proxy_t *
bluez_audio_headset_get_default		(bluez_audio_proxy_t *proxy)
{
  bluez_headset_proxy_t *	headset = NULL;

  g_hash_table_foreach (proxy->headsets, (GHFunc)_find_default, &headset);

  return headset != NULL ? bluez_headset_ref (headset) : NULL;
}

static void
_list_headsets_returned			(DBusPendingCall	*pending,
					 void			*user_data)
{
  foreach_headset_data_t	*data = user_data;
  DBusMessage			*reply;
  DBusError			error;
  const char			**headsets;
  int				i, num_headsets;
  bluez_headset_proxy_t		*headset;

  reply = dbus_pending_call_steal_reply (pending);
  if (reply == NULL) {
    dbus_pending_call_unref (pending);
    g_warning ("no reply...");
    return;
  }

  dbus_error_init (&error);
  if (dbus_set_error_from_message (&error, reply)) {
    g_warning (error.message);
    goto error;
  }

  dbus_message_get_args (reply, &error,
			 DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, &headsets, &num_headsets, DBUS_TYPE_INVALID);

  if (dbus_error_is_set (&error)) {
    goto error;
  }

  if (headsets == NULL) {
    g_warning ("introspect result is null");
    goto error;
  }

  for (i = 0; i < num_headsets; ++i) {
    if ((headset = bluez_audio_headset_get (&_audio_proxy, headsets[i])) != NULL) {
      if (data->func != NULL)
	data->func (headset, data->user_data);
      bluez_headset_unref (headset);
    }
  }

 error:
  if (data->destroy_notify != NULL)
    data->destroy_notify (data->user_data);
  dbus_message_unref (reply);
  dbus_pending_call_unref (pending);
  return;
}

void
bluez_audio_headset_foreach		(bluez_audio_proxy_t *proxy, headset_func_t func,
					 gpointer user_data, GDestroyNotify destroy_notify)
{
  DBusError			derror;
  DBusMessage			*call = NULL;
  DBusPendingCall		*pending = NULL;
  foreach_headset_data_t	*data;

  if (proxy->audio_service == NULL) {
    proxy->audio_service = _bluez_manager_activate_audio_service (proxy->sysbus);
  }

  if (proxy->sysbus == NULL || proxy->audio_service == NULL) {
    g_debug ("bluez audio service is not yet started");
    return;
  }

  data = g_new0 (foreach_headset_data_t, 1);
  g_return_if_fail (data != NULL);
  data->func = func;
  data->user_data = user_data;
  data->destroy_notify = destroy_notify;

  dbus_error_init (&derror);
  call = dbus_message_new_method_call (proxy->audio_service, "/org/bluez/audio", "org.bluez.audio.Manager", "ListHeadsets");
  if (call == NULL) {
    if (destroy_notify != NULL)
      destroy_notify (user_data);
    return;
  }

  dbus_connection_send_with_reply (proxy->sysbus, call, &pending, -1);

  if ((pending == NULL) ||
      (dbus_pending_call_set_notify (pending, _list_headsets_returned, data, g_free) == FALSE)) {
    g_warning ("error with pending call allocation!");
    if (destroy_notify != NULL)
      destroy_notify (user_data);
  }

  dbus_message_unref (call);
}

void
bluez_headset_set_state		(bluez_headset_proxy_t *headset, bt_audio_state_t state)
{
  const char	*iface = "com.nokia.bluez_headset_proxy";
  const char	*prop = "State";
  const char	*val;
  DBusMessage	*signal;

  g_return_if_fail (headset != NULL);

  headset->state = state;
  signal = dbus_message_new_signal (headset->path, DBUS_INTERFACE_PROPERTIES, "Notify");
  if (signal == NULL) {
    g_warning ("connot allocate a notify signal");
    return;
  }

  val = _bt_audio_state_to_string (state);

  dbus_message_append_args (signal,
			    DBUS_TYPE_STRING, &iface,
			    DBUS_TYPE_STRING, &prop,
			    DBUS_TYPE_STRING, &val,
			    DBUS_TYPE_INVALID);

  _send_message_and_unref (headset->audio_proxy->sysbus, signal);

  if (headset->audio_proxy->headset_updated_cb == NULL)
    return;

  headset->audio_proxy->headset_updated_cb (headset, BLUEZ_HEADSET_NOTIFY_STATE, headset->audio_proxy->headset_state_data);
}

void
bluez_headset_set_volume	(bluez_headset_proxy_t *headset, unsigned char volume)
{
  DBusMessage			*call;
  DBusError			derror;
  dbus_uint16_t			v = volume;

  g_return_if_fail (headset != NULL);

  if ((headset->req_volume != -1 && headset->req_volume != volume) ||
      (headset->req_volume == -1 && headset->volume != volume)) {
    headset->req_volume = volume;
    dbus_error_init (&derror);

    call = dbus_message_new_method_call (headset->audio_proxy->audio_service, headset->path, "org.bluez.audio.Headset", "SetSpeakerGain");
    g_return_if_fail (call != NULL);

    dbus_message_append_args (call,
			      DBUS_TYPE_UINT16, &v,
			      DBUS_TYPE_INVALID);

    _send_message_and_unref (headset->audio_proxy->sysbus, call);

    if (headset->audio_proxy->headset_updated_cb == NULL)
      return;

    /* FIXME */
    /* headset->audio_proxy->headset_updated_cb (headset, BLUEZ_HEADSET_NOTIFY_VOLUME, headset->audio_proxy->headset_state_data); */
  }
}

unsigned char
bluez_headset_get_volume	(bluez_headset_proxy_t *headset)
{
  g_return_val_if_fail (headset != NULL, 0);

  return headset->req_volume != -1 ? headset->req_volume : headset->volume;
}

bt_audio_state_t
bluez_headset_get_state		(bluez_headset_proxy_t *headset)
{
  g_return_val_if_fail (headset != NULL, BT_AUDIO_STATE_NONE);

  return headset->state;
}

static void
_headset_func_set_flags (bluez_headset_proxy_t *headset, gpointer data)
{
  bluez_headset_flags_t flags = (bluez_headset_flags_t)data;

  g_return_if_fail (headset != NULL);

  headset->flags = flags;
}

void
bluez_audio_headset_set_default_flags	(bluez_audio_proxy_t *proxy, bluez_headset_flags_t flags, gboolean override_all)
{
  g_return_if_fail (proxy != NULL);

  proxy->headset_default_flags = flags;

  if (override_all == TRUE) {
    bluez_audio_headset_foreach (proxy, _headset_func_set_flags, (gpointer)flags, NULL);
  }
}

bluez_headset_flags_t
bluez_audio_headset_get_default_flags	(bluez_audio_proxy_t *proxy)
{
  g_return_val_if_fail (proxy != NULL, 0);

  return proxy->headset_default_flags;
}

void
bluez_headset_set_flags			(bluez_headset_proxy_t *headset,
					 bluez_headset_flags_t flags)
{
  g_return_if_fail (headset != NULL);

  headset->flags = flags;
}

bluez_headset_flags_t
bluez_headset_get_flags			(bluez_headset_proxy_t *headset)
{
  g_return_val_if_fail (headset != NULL, 0);

  return headset->flags;
}

static gboolean
_send_dbus_message_on_sysbus	(gpointer data)
{
  DBusMessage		*call = data;

  g_return_val_if_fail (call != NULL, FALSE);

  dbus_connection_send (_audio_proxy.sysbus, call, NULL);
  dbus_message_unref (call);
  return FALSE;
}

static DBusHandlerResult
_bluez_audio_signal_handler			(DBusConnection  *connection,
						 DBusMessage     *message,
						 void            *user_data)
{
  const gchar		*path = dbus_message_get_path (message);
  const gchar		*sender = dbus_message_get_sender (message);
  bluez_audio_proxy_t	*proxy = user_data;
  bluez_headset_proxy_t *headset;
  DBusMessage		*msg;
  DBusError		error;
  dbus_uint16_t		v;

  if (dbus_message_is_signal (message, "org.bluez.audio.Headset", "Connected")) {
    headset = bluez_audio_headset_get (proxy, path);
    g_return_val_if_fail (headset != NULL, DBUS_HANDLER_RESULT_NOT_YET_HANDLED);
    bluez_headset_set_state (headset, BT_AUDIO_STATE_CONNECTED);

    if (!(headset->flags & BLUEZ_HEADSET_AUTO_PLAY)) {
      bluez_headset_unref (headset);
      return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    if (headset->flags & BLUEZ_HEADSET_MUTED) {
      bluez_headset_set_state (headset, BT_AUDIO_STATE_PLAYING_PM);
      bluez_headset_unref (headset);
      return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    bluez_headset_unref (headset);
    msg = dbus_message_new_method_call (sender, path, "org.bluez.audio.Headset", "Play");
    g_timeout_add (1 * 1000, _send_dbus_message_on_sysbus, msg); /* FIXME: workaround for bluez/kernel.. */
  }

  if (dbus_message_is_signal (message, "org.bluez.audio.Headset", "Disconnected")) {
    headset = bluez_audio_headset_get (proxy, path);
    g_return_val_if_fail (headset != NULL, DBUS_HANDLER_RESULT_NOT_YET_HANDLED);
    bluez_headset_set_state (headset, BT_AUDIO_STATE_DISCONNECTED);
    bluez_headset_unref (headset);
  }

  if (dbus_message_is_signal (message, "org.bluez.audio.Headset", "Playing")) {
    headset = bluez_audio_headset_get (proxy, path);
    g_return_val_if_fail (headset != NULL, DBUS_HANDLER_RESULT_NOT_YET_HANDLED);
    headset->flags &= ~BLUEZ_HEADSET_MUTED;
    bluez_headset_set_state (headset, BT_AUDIO_STATE_PLAYING);
    bluez_headset_unref (headset);
  }

  if (dbus_message_is_signal (message, "org.bluez.audio.Headset", "Stopped")) {
    headset = bluez_audio_headset_get (proxy, path);
    g_return_val_if_fail (headset != NULL, DBUS_HANDLER_RESULT_NOT_YET_HANDLED);
    if (headset->flags & BLUEZ_HEADSET_PM || headset->flags & BLUEZ_HEADSET_MUTED) {
      bluez_headset_set_state (headset, BT_AUDIO_STATE_PLAYING_PM);
    } else {
      bluez_headset_set_state (headset, BT_AUDIO_STATE_STOPPED);
    }
    bluez_headset_unref (headset);
  }

  if (dbus_message_is_signal (message, "org.bluez.audio.Headset", "SpeakerGainChanged")) {
    headset = bluez_audio_headset_get (proxy, path);
    g_return_val_if_fail (headset != NULL, DBUS_HANDLER_RESULT_NOT_YET_HANDLED);
    dbus_error_init (&error);
    dbus_message_get_args (message, &error,
			   DBUS_TYPE_UINT16, &v,
			   DBUS_TYPE_INVALID);
    if (dbus_error_is_set (&error)) {
      g_warning ("error while speakergainchanged %s:", error.message);
      dbus_error_free (&error);
      return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }
    headset->volume = v;
    /* I am unsure about the req == new volume -> no relation between those 2 */
    if (headset->req_volume != -1 && headset->req_volume == v)
      headset->req_volume = -1;
    if (headset->audio_proxy->headset_updated_cb != NULL && headset->req_volume == -1)
      headset->audio_proxy->headset_updated_cb (headset, BLUEZ_HEADSET_NOTIFY_VOLUME, headset->audio_proxy->headset_state_data);
    bluez_headset_unref (headset);
  }

  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

/* Fixme: gsignal-like would REALLLLLLY make sense... Gobject should still be avoided, what about GHook? */
void
bluez_audio_headset_set_updated_cb	(bluez_audio_proxy_t *proxy, headset_updated_func_t cb, gpointer data)
{
  g_return_if_fail (proxy != NULL);

  proxy->headset_updated_cb = cb;
  proxy->headset_state_data = data;
}

gboolean
bluez_audio_is_active		(bluez_audio_proxy_t *proxy)
{
  g_return_val_if_fail (proxy != NULL, FALSE);

  return (proxy->audio_service != NULL) ? TRUE : FALSE;
}

bluez_audio_proxy_t *
bluez_audio			(DBusConnection *system_bus)
{
  DBusError	error;
  gboolean	success;

  g_return_val_if_fail (system_bus != NULL, NULL);

  if (_audio_proxy.sysbus != NULL) {
    return &_audio_proxy;
  }

  _audio_proxy.headsets = g_hash_table_new_full (g_str_hash, g_str_equal, NULL, (GDestroyNotify)bluez_headset_unref);
  _audio_proxy.sysbus = system_bus;

  if (_audio_proxy.audio_service == NULL) {
    _audio_proxy.audio_service = _bluez_manager_activate_audio_service (_audio_proxy.sysbus);
  }

  success = dbus_connection_register_object_path (_audio_proxy.sysbus, BLUEZ_AUDIO_PROXY_PATH, &_audio_vtable, &_audio_proxy);
  if (success == FALSE) {
    g_warning ("could not create a bluez audio proxy object");
  }

#define MATCH_RULE_DBUS_NAMEOWNER				\
  "type='signal',interface='org.freedesktop.DBus',member='NameOwnerChanged'"
  dbus_error_init (&error);
  dbus_bus_add_match (_audio_proxy.sysbus, MATCH_RULE_DBUS_NAMEOWNER, &error);

  success = dbus_connection_add_filter (_audio_proxy.sysbus, _dbus_signal_handler, &_audio_proxy, NULL);
  if (success == FALSE) {
    g_warning ("could not filter the NameOwnerChanged signal");
  }

#define MATCH_RULE_BLUEZ_AUDIO_HS				\
  "type='signal',interface='org.bluez.audio.Headset'"
  dbus_error_init (&error);
  dbus_bus_add_match (_audio_proxy.sysbus, MATCH_RULE_BLUEZ_AUDIO_HS, &error);

  success = dbus_connection_add_filter (_audio_proxy.sysbus, _bluez_audio_signal_handler, &_audio_proxy, NULL);
  if (!success) {
    g_warning ("Cannot add BT CONNECTED filter");
  }
  if (dbus_error_is_set (&error)) {
    g_warning ("dbus error: %s", error.message);
    dbus_error_free (&error);
  }

  if (dbus_bus_request_name (_audio_proxy.sysbus, "com.nokia.bluez_audio_proxy", 0, &error) != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
    g_warning ("There is already a bluez_audio_proxy running");
  }
  if (dbus_error_is_set (&error)) {
    g_warning ("dbus error: %s", error.message);
    dbus_error_free (&error);
  }

  bluez_audio_headset_set_default_flags (&_audio_proxy, BLUEZ_HEADSET_PM | BLUEZ_HEADSET_AUTO_PLAY, TRUE);

  return bluez_audio_ref (&_audio_proxy);
}

bluez_headset_proxy_t *
bluez_headset_ref			(bluez_headset_proxy_t *headset)
{
  g_return_val_if_fail (headset != NULL, NULL);

  ++headset->refcount;

  return headset;
}

bluez_audio_proxy_t *
bluez_audio_ref			(bluez_audio_proxy_t *proxy)
{
  g_return_val_if_fail (proxy != NULL, NULL);

  ++proxy->refcount;

  return proxy;
}

void
bluez_headset_unref			(bluez_headset_proxy_t *headset)
{
  if ((--headset->refcount) > 0) {
    return;
  }

  if (headset->audio_proxy != NULL) {
    bluez_audio_unref (headset->audio_proxy);
    headset->audio_proxy = NULL;
  }

  if (headset->path != NULL) {
    g_free (headset->path);
    headset->path = NULL;
  }

  g_free (headset);
}

void
bluez_audio_unref			(bluez_audio_proxy_t *proxy)
{
  g_return_if_fail (proxy != NULL && proxy->refcount > 0);

  if ((--proxy->refcount) > 0) {
    return;
  }

  g_warning ("unref audio proxy is not fully implemented");

  if (proxy->audio_service != NULL) {
    g_free (proxy->audio_service);
    proxy->audio_service = NULL;
  }
}
