#define _GNU_SOURCE
#include <signal.h>
#include <dbus/dbus.h>
#include <hildon/hildon.h>
#include "swap_ui.h"
#include "swap_mgr.h"
#include "i18n.h"

typedef enum
{
  SWAP_CREATE,
  SWAP_DELETE,
  SWAP_RESIZE,
  SWAP_NOOP
} SwapOp;

#define ONE_MEGABYTE 1048576

static GtkListStore *make_vmem_sizes_model(void);
static void select_swap_size(GtkWidget *combo_box, GtkTreeModel *tm, guint64 swap_value);
static gboolean get_swap_size(GtkWidget *cb, guint64 *new_size);
static gboolean change_swap(MemoryApplet *memory_applet, guint64 old_swap_size, guint64 new_swap_size);
static guint check_virtual_size(guint size, guint granularity);

static GtkWidget *progress_bar = NULL;
static gboolean cancel_progress_dialog = FALSE;

static gboolean
delete_event(GtkWidget *widget, GdkEvent *event, gpointer null)
{
  /* GRRRRR! WTF must the dialog be destroyed on delete_event? I /am/ getting a "response" signal, you know! */
  return TRUE;
}

static int
update_progress_bar(unsigned n, unsigned m)
{
  g_return_val_if_fail(0 != m, 1);
  g_return_val_if_fail(NULL != progress_bar, 1);

  gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progress_bar), (((double)n) / ((double)m)));
  while (g_main_context_iteration(NULL, FALSE));

  return cancel_progress_dialog ? 1 : 0;
}

void
swap_page_update(MemoryApplet *applet, StorageDeviceStatus status)
{
  if(swap_permitted()) {
    GtkTreeModel *tm = NULL;
    guint64 max_swap = check_virtual_size(swap_maximal_size(), swap_size_granularity()), available_swap = swap_available();

    if (STORAGE_DEVICE_AVAILABLE == status) {
      char *label = NULL;

      g_object_set(G_OBJECT(applet->swap_settings_alignment), "xalign", 0.0, "yalign", 1.0, NULL);
      g_object_set(G_OBJECT(applet->swap_settings_status_label), "visible", FALSE, NULL);
      g_object_set(G_OBJECT(applet->swap_settings_vbox), "visible", TRUE, NULL);

      tm = GTK_TREE_MODEL(make_vmem_sizes_model());
      g_object_set(G_OBJECT(applet->size_combo), "model", tm, NULL);
      select_swap_size(applet->size_combo, tm, available_swap ? available_swap : swap_automatic_size());

      g_object_set(G_OBJECT(applet->extend_checkbox), "active", swap_enabled(), "sensitive", NULL != tm, NULL);
      label = g_strdup_printf(_("memo_fi_virtual_abbr_size_mb"), (long int)(((STORAGE_DEVICE_AVAILABLE == status) ? max_swap : 0) / ONE_MEGABYTE));
      g_object_set(G_OBJECT(applet->available_label), "label", label, NULL);

      g_free(label);
    } else {
      g_object_set(G_OBJECT(applet->swap_settings_alignment), "xalign", 0.5, "yalign", 0.5, NULL);
      g_object_set(G_OBJECT(applet->swap_settings_vbox), "visible", FALSE, NULL);
      g_object_set(G_OBJECT(applet->swap_settings_status_label), "visible", TRUE, "label", 
        STORAGE_DEVICE_UNAVAILABLE  == status ? _("memo_ia_virtual_memory_card_card_not_inserted") :
        STORAGE_DEVICE_USB_OBSCURED == status ? _("memo_ia_virtual_memory_card_used_by_usb") :
        STORAGE_DEVICE_CORRUPTED    == status ? _("memo_ia_virtual_memory_corrupt_card") : "???", NULL);
    }
  }
  else
    gtk_notebook_remove_page(GTK_NOTEBOOK(applet->notebook), MEM_APPLET_VIRTUAL_PAGE);
}

static void select_swap_size(GtkWidget *combo_box, GtkTreeModel *tm, guint64 swap_value)
{
  GtkTreeIter itr;
  guint64 diff = G_MAXUINT64, new_diff;
  guint64 model_swap = 0, closest_swap_value = swap_value;


  if (tm) {
    if (0 == swap_value)
      g_object_set(G_OBJECT(combo_box), "active", -1, NULL);
    else {
      if (gtk_tree_model_get_iter_first(tm, &itr))
        do {
          gtk_tree_model_get(tm, &itr, 1, &model_swap, -1);
          new_diff = MAX(model_swap, swap_value) - MIN(model_swap, swap_value);
          if (new_diff < diff) {
            closest_swap_value = model_swap;
            diff = new_diff;
          }
        } while (gtk_tree_model_iter_next(tm, &itr));

      swap_value = closest_swap_value;

      if (gtk_tree_model_get_iter_first(tm, &itr))
        do {
          gtk_tree_model_get(tm, &itr, 1, &model_swap, -1);
          if (model_swap == swap_value) {
            gtk_combo_box_set_active_iter(GTK_COMBO_BOX(combo_box), &itr);
            break;
          }
        } while (gtk_tree_model_iter_next(tm, &itr));
    }
  }
}

gboolean
maybe_update_swap(MemoryApplet *memory_applet)
{
  StorageDeviceStatus internal_mmc_status;
  guint64 old_swap_size, new_swap_size;
  gboolean swap_changed, want_swap;

  g_object_get(G_OBJECT(memory_applet->extend_checkbox), "sensitive", &swap_changed, "active", &want_swap, NULL);
  g_object_get(G_OBJECT(memory_applet->internal_mmc), "status", &internal_mmc_status, NULL);

  if (internal_mmc_status != STORAGE_DEVICE_AVAILABLE) return TRUE;
  if (!swap_changed) return TRUE;
  if (want_swap) {
    if (!get_swap_size(memory_applet->size_combo, &new_swap_size)) return TRUE;
  } else {
    new_swap_size = 0;
  }
  old_swap_size = swap_available();

  return change_swap(memory_applet, old_swap_size, new_swap_size);
}

static gboolean
get_swap_size(GtkWidget *cb, guint64 *new_size)
{
  GtkTreeModel *tm = NULL;
  GtkTreeIter itr;

  g_object_get(G_OBJECT(cb), "model", &tm, NULL);
  if (NULL != tm)
    if (gtk_combo_box_get_active_iter(GTK_COMBO_BOX(cb), &itr)) {
      gtk_tree_model_get(tm, &itr, 1, new_size, -1);
      return TRUE;
    }

  return FALSE;
}

/* This function is used because we want to reduce the amount of
 options shown to the user. It filters out the first four ones and
 after that only every second. So with 8M granularity the options are
 8,16,24,32 and then 48,64,80... The returned value is the biggest
 allowed option which is at most size */
static guint
check_virtual_size(guint size, guint granularity)
{
  guint factor = size/granularity;

  if (factor > 4 && factor % 2 != 0)
    factor = (factor >> 1) << 1;

  return factor * granularity;
}

static GtkListStore *
make_vmem_sizes_model(void)
{
  guint64
    min_size = swap_minimal_size(),
    max_size = swap_maximal_size(),
    granularity = swap_size_granularity();
  GString *str = NULL;
  GtkListStore *ls = NULL;
  GtkTreeIter itr;
  guint64 size = 0, validated_size = 0;
/*  gboolean need_to_add_available = TRUE; */

  if (min_size && max_size && granularity) {
    if ((ls = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_UINT64)) != NULL) {
      if ((str = g_string_new("")) != NULL) {
        for (size = min_size ; size <= max_size ; size += granularity) {
          if (size == check_virtual_size(size, granularity)) {
            if ((validated_size = swap_validate_size(size)) != 0) {
              /*
              if (validated_size == available_swap)
                available_swap = 0;
              else
              if (available_swap) {
                if (validated_size > available_swap && need_to_add_available) {
                  gtk_list_store_append(ls, &itr);
                  g_string_printf(str, _("memo_fi_virtual_abbr_size_mb"), (long int)(available_swap / ONE_MEGABYTE));
                  gtk_list_store_set(ls, &itr, 0, str->str, 1, available_swap, -1);
                  need_to_add_available = FALSE;
                }
              }
              */
              gtk_list_store_append(ls, &itr);
              g_string_printf(str, _("memo_fi_virtual_abbr_size_mb"), (long int)(validated_size / ONE_MEGABYTE));
              gtk_list_store_set(ls, &itr, 0, str->str, 1, validated_size, -1);
            }
          }
        }
        g_string_free(str, TRUE);
      }
    }
  }
  return ls;
}

static SwapOp
swap_op_from_size_change(guint64 old, guint64 new, char **confirmation_message, gboolean *kill_programs)
{
  (*confirmation_message) = NULL;
  (*kill_programs) = FALSE;

  if (0 == old) {
    if (0 == new)
      return SWAP_NOOP;
    else {
      (*confirmation_message) = _("memr_no_turn_mmc_swap_on");
      return SWAP_CREATE;
    }
  } else {
    if (0 == new) {
      if (((*kill_programs) = !swap_can_switch_off()))
        (*confirmation_message) = _("memr_no_turn_mmc_swap_off");
      return SWAP_DELETE;
    } else
    if (new == old)
      return SWAP_NOOP;
    else {
      if (((*kill_programs) = !swap_can_switch_off()))
        (*confirmation_message) = _("memo_no_resize_mmc_swap");
      return SWAP_RESIZE;
    }
  }
  g_return_val_if_reached(SWAP_NOOP);
}

static void
progress_note_response(GtkWidget *note, gint response, gpointer null)
{
  cancel_progress_dialog = TRUE;
}

static gboolean
restore_sigterm(sighandler_t old_term_handler)
{
  g_print("Restoring sigterm handler\n");
  signal(SIGTERM, old_term_handler);
  return FALSE;
}

/* com.nokia.ke_recv /com/nokia/ke_recv/internal_mmc_swap_off com.nokia.ke_recv.internal_mmc_swap_off internal_mmc_swap_off */
static gboolean
turn_swap_off(gboolean kill_programs)
{
  gboolean retval = FALSE;
  DBusMessage *message = NULL;
  DBusConnection *conn = NULL;
  sighandler_t old_term_handler;

  if ((conn = dbus_bus_get(DBUS_BUS_SYSTEM, NULL)) != NULL) {
    if ((message = dbus_message_new_method_call(
      "com.nokia.ke_recv",
      "/com/nokia/ke_recv/mmc_swap_off",
      "com.nokia.ke_recv.mmc_swap_off",
      "mmc_swap_off")) != NULL) {
      if (dbus_message_append_args(message, DBUS_TYPE_BOOLEAN, &kill_programs, DBUS_TYPE_INVALID)) {
        DBusMessage *reply = NULL;
        old_term_handler = signal(SIGTERM, SIG_IGN);
        if ((reply = dbus_connection_send_with_reply_and_block(conn, message, 5000, NULL)) != NULL) {
          dbus_message_unref(reply);
          retval = TRUE;
        }
        g_timeout_add(5000, (GSourceFunc)restore_sigterm, old_term_handler);
      }
      dbus_message_unref(message);
    }
  }

  return retval;
}

static gboolean
turn_swap_on()
{
  gboolean retval = FALSE;
  DBusMessage *message = NULL;
  DBusConnection *conn = NULL;

  g_debug ("1 Plaa plaa plaa plaa plaa plaap lapalpaaplapla");
  if ((conn = dbus_bus_get(DBUS_BUS_SYSTEM, NULL)) != NULL) {
    g_debug ("2 Plaa plaa plaa plaa plaa plaap lapalpaaplapla");
    if ((message = dbus_message_new_method_call(
      "com.nokia.ke_recv",
      "/com/nokia/ke_recv/mmc_swap_on",
      "com.nokia.ke_recv.mmc_swap_on",
      "mmc_swap_on")) != NULL) {
      DBusMessage *reply = NULL;

      if ((reply = dbus_connection_send_with_reply_and_block(conn, message, 5000, NULL)) != NULL) {
        dbus_message_unref(reply);
        retval = TRUE;
      }
      dbus_message_unref(message);
    }
  }

  return retval;
}

static gboolean
change_swap(MemoryApplet *memory_applet, guint64 old_swap_size, guint64 new_swap_size)
{
  gboolean kill_programs = FALSE;
  char *confirmation_message = NULL;
  SwapOp op = swap_op_from_size_change(old_swap_size, new_swap_size, &confirmation_message, &kill_programs);
  GtkWidget *note = NULL;
  gboolean error = FALSE;

  /* St00pid global variables */
  cancel_progress_dialog = FALSE;

  if (SWAP_NOOP == op) {
    if (new_swap_size > 0)
      turn_swap_on();
    return TRUE;
  }

  if (confirmation_message) {
    guint response = GTK_RESPONSE_OK;
    GtkWidget *msg = hildon_note_new_confirmation(GTK_WINDOW(memory_applet->dialog), confirmation_message);

    if (msg) {
      g_signal_connect(G_OBJECT(msg), "delete-event", (GCallback)delete_event, NULL);
      response = gtk_dialog_run(GTK_DIALOG(msg));
      gtk_widget_destroy(msg);
      msg = NULL;
    }

  if (GTK_RESPONSE_OK != response)
    return FALSE;
  }

  note = hildon_note_new_cancel_with_progress_bar(GTK_WINDOW(memory_applet->dialog),
    SWAP_DELETE == op ? _("memo_nw_turnoff_virtual_memory") :
    SWAP_CREATE == op ? _("memo_nw_creating_virtual_memory") : _("memo_nw_resizing"),
    GTK_PROGRESS_BAR(progress_bar = g_object_new(GTK_TYPE_PROGRESS_BAR, "visible", TRUE, NULL)));
  g_signal_connect(G_OBJECT(note), "delete-event", (GCallback)delete_event, NULL);
  g_object_set(G_OBJECT(note), "modal", TRUE, "transient-for", memory_applet->dialog, NULL);
  g_signal_connect(G_OBJECT(note), "response", (GCallback)progress_note_response, NULL);
  gtk_widget_show(note);

  if (SWAP_DELETE == op || SWAP_RESIZE == op) {
    error = !turn_swap_off(kill_programs);
    if (SWAP_DELETE == op)
      update_progress_bar(1, 2);
    if (!error && !cancel_progress_dialog) {
      error = (swap_delete() != 0);
      if (!error && SWAP_DELETE == op)
        update_progress_bar(2, 2);
    }
  }

  while (g_main_context_iteration(NULL, FALSE));

  if (!error && !cancel_progress_dialog && (SWAP_CREATE == op || SWAP_RESIZE == op)) {
    error = (swap_create(new_swap_size, update_progress_bar) != 0);
    while (g_main_context_iteration(NULL, FALSE));
 
    if (!error && !cancel_progress_dialog) {
      error = !turn_swap_on();
      while (g_main_context_iteration(NULL, FALSE));
    }

    if (error || cancel_progress_dialog)
      swap_delete();
  }
 
  gtk_widget_destroy(note);

  /* St00pid global variables */
  progress_bar = NULL;

  hildon_banner_show_information(memory_applet->dialog_parent, NULL,
    cancel_progress_dialog ? _("memo_ib_cancelled") :
    error ?
      (SWAP_DELETE == op ? _("memo_no_unable_remove") : 
       SWAP_CREATE == op ? _("memo_no_unable_create") : 
       SWAP_RESIZE == op ? _("memo_no_unable_resize") : "???") : 
      (SWAP_DELETE == op ? _("memo_ib_removed") :
       SWAP_CREATE == op ? _("memo_ib_created") :
       SWAP_RESIZE == op ? _("memo_ib_resized") : "???"));
  while (g_main_context_iteration(NULL, FALSE));

  return !cancel_progress_dialog && !error;
}
