#include <stdlib.h>
#include <string.h>
#include <libgnomevfs/gnome-vfs-volume-monitor.h>
#include <gconf/gconf-client.h>
#include <sys/vfs.h>
#include "i18n.h"
#include "storagedevice.h"

#define FLASH_RESERVED_SPACE 2465792 + /* 2408 kB */ \
                              294912 + /*  288 kB extra observed */ \
                                4096   /*    4 kB to make sure it comes out negative and gets clamped */

#define NULL_SAFE_STRCMP(s1,s2) \
  ((NULL == (s1)) \
    ? (NULL == (s2)) ? 0 : -1 \
    : (NULL == (s2)) ? 1 : strcmp((s1),(s2)))

typedef struct
{
  StorageDeviceKey key;
  guint64 total_size;
  guint64 free_size;
  StorageDeviceStatus status;
  char *name;
  gboolean swap_enabled;

  guint volume_mounted_signal_id;
  guint volume_unmounted_signal_id;
  guint cover_open_signal_id;
  guint device_present_signal_id;
  guint usb_obscured_signal_id;
  guint corrupted_signal_id;
  guint swap_enabled_signal_id;
  guint update_sizes_timeout_id;

  guint thaw_notify_timeout_id;
} StorageDevicePrivate;

#define STORAGE_DEVICE_GET_PRIVATE(object) \
  (G_TYPE_INSTANCE_GET_PRIVATE((object), STORAGE_DEVICE_TYPE, StorageDevicePrivate))

#define GCONF_PATH_SYSTEM_OSSO_AF    "/system/osso/af"
#define GCONF_KEY_MMC_USED_OVER_USB  "mmc-used-over-usb"
#define GCONF_KEY_MMC_DEVICE_PRESENT "mmc-device-present"
#define GCONF_KEY_MMC_CORRUPTED      "mmc-corrupted"
#define GCONF_KEY_MMC_COVER_OPEN     "mmc-cover-open"
#define GCONF_KEY_MMC_SWAP_ENABLED   "mmc-swap"

enum
{
  STORAGE_DEVICE_FIRST_PROPERTY = 0,
  STORAGE_DEVICE_KEY_PROPERTY,
  STORAGE_DEVICE_TOTAL_SIZE_PROPERTY,
  STORAGE_DEVICE_FREE_SIZE_PROPERTY,
  STORAGE_DEVICE_STATUS_PROPERTY,
  STORAGE_DEVICE_NAME_PROPERTY,
  STORAGE_DEVICE_SWAP_ENABLED_PROPERTY,
  STORAGE_DEVICE_LAST_PROPERTY
};

static void storage_device_class_init(gpointer g_class, gpointer null);
static void storage_device_init(GTypeInstance *instance, gpointer g_class);

static void finalize(GObject *object);
static void set_property(GObject *object, guint property_id, const GValue *val, GParamSpec *pspec);
static void get_property(GObject *object, guint property_id,       GValue *val, GParamSpec *pspec);

static void storage_device_set_key(StorageDevice *storage_device, StorageDeviceKey key);
static void storage_device_update(StorageDevice *storage_device, StorageDevicePrivate *private, StorageDeviceKey key, gboolean immediately);
static void storage_device_determine_sizes(StorageDevice *storage_device, StorageDevicePrivate *private, StorageDeviceKey key);
static void storage_device_determine_status(StorageDevice *storage_device, StorageDevicePrivate *private, StorageDeviceKey key);
static void listen_to_gconf(StorageDevice *storage_device, StorageDevicePrivate *private, StorageDeviceKey key);
static StorageDeviceKey storage_device_key_from_uri(char *uri);
static const char *storage_device_uri_from_key(StorageDeviceKey key);
static void storage_device_set_name_from_vfs_volume(StorageDevice *storage_device, GnomeVFSVolume *volume);

static void storage_device_status_changed(GConfClient *client, guint cnxn_id, GConfEntry *entry, StorageDevice *storage_device);
static void volume_mounted_or_unmounted(GnomeVFSVolumeMonitor *monitor, GnomeVFSVolume *volume, StorageDevice *storage_device);

GType
storage_device_key_get_type(void)
{
  static GType the_type = 0;

  if (0 == the_type) {
    static GEnumValue enum_vals[] = {
      {STORAGE_DEVICE_NONE,         "STORAGE_DEVICE_NONE",         "Storage Device Not Set"},
      {STORAGE_DEVICE_FLASH,        "STORAGE_DEVICE_FLASH",        "Device Flash"},
      {STORAGE_DEVICE_INTERNAL_MMC, "STORAGE_DEVICE_INTERNAL_MMC", "Internal MMC"},
      {STORAGE_DEVICE_EXTERNAL_MMC, "STORAGE_DEVICE_EXTERNAL_MMC", "External MMC"},
      {0, NULL, NULL}
    };

    the_type = g_enum_register_static(STORAGE_DEVICE_KEY_TYPE_STRING, enum_vals);
  }

  return the_type;
}

GType
storage_device_status_get_type(void)
{
  static GType the_type = 0;

  if (0 == the_type) {
    static GEnumValue enum_vals[] = {
      {STORAGE_DEVICE_AVAILABLE,    "STORAGE_DEVICE_AVAILABLE",    "Storage Device Available"},
      {STORAGE_DEVICE_UNAVAILABLE,  "STORAGE_DEVICE_UNAVAILABLE",  "Storage Device Unavailable"},
      {STORAGE_DEVICE_USB_OBSCURED, "STORAGE_DEVICE_USB_OBSCURED", "USB Cable Plugged In"},
      {STORAGE_DEVICE_COVER_OPEN,   "STORAGE_DEVICE_COVER_OPEN",   "Storage Device Cover Open"},
      {STORAGE_DEVICE_CORRUPTED,    "STORAGE_DEVICE_CORRUPTED",    "Storage Device Corrupted"},
      {0, NULL, NULL}
    };

    the_type = g_enum_register_static(STORAGE_DEVICE_STATUS_TYPE_STRING, enum_vals);
  }

  return the_type;
}

GType storage_device_get_type(void)
{
  static GType the_type = 0;

  if (0 == the_type) {
    static GTypeInfo the_type_info =
    {
      .class_size     = sizeof(StorageDeviceClass),
      .base_init      = NULL,
      .base_finalize  = NULL,
      .class_init     = storage_device_class_init,
      .class_finalize = NULL,
      .class_data     = NULL,
      .instance_size  = sizeof(StorageDevice),
      .n_preallocs    = 0,
      .instance_init  = storage_device_init,
      .value_table    = NULL
    } ;

    if ((the_type = g_type_register_static(G_TYPE_OBJECT, STORAGE_DEVICE_TYPE_STRING, &the_type_info, 0)) != 0)
      g_type_class_ref(the_type);
  } ;

  return the_type;
}

static void
storage_device_class_init(gpointer g_class, gpointer null)
{
  G_OBJECT_CLASS(g_class)->finalize = finalize;
  G_OBJECT_CLASS(g_class)->set_property = set_property;
  G_OBJECT_CLASS(g_class)->get_property = get_property;

  g_object_class_install_property(g_class, STORAGE_DEVICE_TOTAL_SIZE_PROPERTY,
    g_param_spec_uint64("total-size", "Total Size", "Total size in bytes of storage device",
      0, G_MAXUINT64, 0, G_PARAM_READABLE));

  g_object_class_install_property(g_class, STORAGE_DEVICE_FREE_SIZE_PROPERTY,
    g_param_spec_uint64("free-size", "Free Size", "Amount of free space in bytes on storage device",
      0, G_MAXUINT64, 0, G_PARAM_READABLE));

  g_object_class_install_property(g_class, STORAGE_DEVICE_KEY_PROPERTY,
    g_param_spec_enum("key", "Storage Device Type", "Type of the storage device whose details to display",
      STORAGE_DEVICE_KEY_TYPE, STORAGE_DEVICE_FLASH, G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property(g_class, STORAGE_DEVICE_STATUS_PROPERTY,
    g_param_spec_enum("status", "Storage Device Status", "Status of the storage device",
      STORAGE_DEVICE_STATUS_TYPE, STORAGE_DEVICE_FLASH, G_PARAM_READABLE));

  g_object_class_install_property(g_class, STORAGE_DEVICE_NAME_PROPERTY,
    g_param_spec_string("name", "Storage Device Name", "Name of the storage device",
      "Storage", G_PARAM_READABLE));

  g_object_class_install_property(g_class, STORAGE_DEVICE_SWAP_ENABLED_PROPERTY,
    g_param_spec_boolean("swap-enabled", "Swap Enabled", "Whether the storage device is being used for swapping",
      FALSE, G_PARAM_READABLE));

  g_type_class_add_private(g_class, sizeof(StorageDevicePrivate));
}

static void
storage_device_init(GTypeInstance *instance, gpointer g_class)
{
  StorageDevicePrivate *private = STORAGE_DEVICE_GET_PRIVATE(instance);

  private->key = STORAGE_DEVICE_NONE;
  private->total_size = 0;
  private->free_size = 0;
  private->status = STORAGE_DEVICE_UNAVAILABLE;

  private->volume_mounted_signal_id = 0;
  private->volume_unmounted_signal_id = 0;
  private->cover_open_signal_id = 0;
  private->device_present_signal_id = 0;
  private->usb_obscured_signal_id = 0;
  private->corrupted_signal_id = 0;
  private->swap_enabled_signal_id = 0;
  private->update_sizes_timeout_id = 0;
  private->thaw_notify_timeout_id = 0;
}

static void
finalize(GObject *object)
{
  StorageDevicePrivate *private = STORAGE_DEVICE_GET_PRIVATE(object);
  GnomeVFSVolumeMonitor *monitor = NULL;
  GConfClient *client = NULL;

  if ((monitor = gnome_vfs_get_volume_monitor()) != NULL) {
    if (private->volume_mounted_signal_id)
      g_signal_handler_disconnect(G_OBJECT(monitor), private->volume_mounted_signal_id);
    if (private->volume_unmounted_signal_id)
      g_signal_handler_disconnect(G_OBJECT(monitor), private->volume_unmounted_signal_id);
  }

  if ((client = gconf_client_get_default()) != NULL) {
    if (private->cover_open_signal_id)
      gconf_client_notify_remove(client, private->cover_open_signal_id);
    if (private->device_present_signal_id)
      gconf_client_notify_remove(client, private->device_present_signal_id);
    if (private->usb_obscured_signal_id)
      gconf_client_notify_remove(client, private->usb_obscured_signal_id);
    if (private->corrupted_signal_id)
      gconf_client_notify_remove(client, private->corrupted_signal_id);
    if (private->swap_enabled_signal_id)
      gconf_client_notify_remove(client, private->swap_enabled_signal_id);

    gconf_client_remove_dir(client, GCONF_PATH_SYSTEM_OSSO_AF, NULL);
  }

  if (private->update_sizes_timeout_id)
    g_source_remove(private->update_sizes_timeout_id);
  if (private->thaw_notify_timeout_id)
    g_source_remove(private->thaw_notify_timeout_id);
}

static void
set_property(GObject *object, guint property_id, const GValue *val, GParamSpec *pspec)
{
  if (STORAGE_DEVICE_KEY_PROPERTY == property_id)
    storage_device_set_key(STORAGE_DEVICE(object), (StorageDeviceKey)g_value_get_enum(val));
  else
    G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
}

static void
get_property(GObject *object, guint property_id, GValue *val, GParamSpec *pspec)
{
  StorageDevicePrivate *private = STORAGE_DEVICE_GET_PRIVATE(object);

  if (STORAGE_DEVICE_KEY_PROPERTY == property_id)
    g_value_set_enum(val, private->key);
  else
  if (STORAGE_DEVICE_STATUS_PROPERTY == property_id)
    g_value_set_enum(val, private->status);
  else
  if (STORAGE_DEVICE_TOTAL_SIZE_PROPERTY == property_id)
    g_value_set_uint64(val, private->total_size);
  else
  if (STORAGE_DEVICE_FREE_SIZE_PROPERTY == property_id)
    g_value_set_uint64(val, private->free_size);
  else
  if(STORAGE_DEVICE_NAME_PROPERTY == property_id)
    g_value_set_string(val, private->name);
  else
    G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
}

static gboolean update_sizes_timeout (StorageDevice *storage_device)
{
  StorageDevicePrivate *private = STORAGE_DEVICE_GET_PRIVATE(storage_device);

  storage_device_determine_sizes(storage_device, private, private->key);

  return TRUE;
}

static void
storage_device_set_key(StorageDevice *storage_device, StorageDeviceKey key)
{
  GnomeVFSVolumeMonitor *monitor = NULL;
  StorageDevicePrivate *private = STORAGE_DEVICE_GET_PRIVATE(storage_device);

  if (key == private->key) return;

  private->key = key;
  private->update_sizes_timeout_id = g_timeout_add(1000, (GSourceFunc)update_sizes_timeout, storage_device);
  listen_to_gconf(storage_device, private, private->key);

  if ((monitor = gnome_vfs_get_volume_monitor()) != NULL) {
    private->volume_mounted_signal_id = g_signal_connect(G_OBJECT(monitor), "volume-mounted", (GCallback)volume_mounted_or_unmounted, storage_device);
    private->volume_unmounted_signal_id = g_signal_connect(G_OBJECT(monitor), "volume-unmounted", (GCallback)volume_mounted_or_unmounted, storage_device);
  }

  storage_device_update(storage_device, private, key, TRUE);

  g_object_notify(G_OBJECT(storage_device), "key");
}

static gboolean
thaw_notify_timeout(StorageDevice *storage_device)
{
  g_object_thaw_notify(G_OBJECT(storage_device));
  STORAGE_DEVICE_GET_PRIVATE(storage_device)->thaw_notify_timeout_id = 0;
  return FALSE;
}

static void
storage_device_update(StorageDevice *storage_device, StorageDevicePrivate *private, StorageDeviceKey key, gboolean immediately)
{
  if (!immediately) {
    if (0 == private->thaw_notify_timeout_id) {
      g_object_freeze_notify(G_OBJECT(storage_device));
    } else {
      g_source_remove(private->thaw_notify_timeout_id);
    }
    private->thaw_notify_timeout_id = g_timeout_add(1500, (GSourceFunc)thaw_notify_timeout, storage_device);
  }
  storage_device_determine_status(storage_device, private, key);
  storage_device_determine_sizes(storage_device, private, key);
}

static void
volume_mounted_or_unmounted(GnomeVFSVolumeMonitor *monitor, GnomeVFSVolume *volume, StorageDevice *storage_device)
{
  StorageDevicePrivate *private = STORAGE_DEVICE_GET_PRIVATE(storage_device);

  storage_device_update(storage_device, private, private->key, FALSE);
}

static void
storage_device_set_name_from_vfs_volume(StorageDevice *storage_device, GnomeVFSVolume *volume)
{
  StorageDevicePrivate *private = STORAGE_DEVICE_GET_PRIVATE(storage_device);
  char *uri = NULL;
  StorageDeviceKey key;

  if ((uri = gnome_vfs_volume_get_activation_uri(volume)) != NULL) {
    if ((key = storage_device_key_from_uri(uri)) != STORAGE_DEVICE_NONE)
      if (key == private->key) {
        char *new_name = NULL;

        if ((new_name = gnome_vfs_volume_get_display_name(volume)) == NULL) {
          new_name = g_strdup(STORAGE_DEVICE_INTERNAL_MMC == key ? _("memo_ti_card_internal")  :
                              STORAGE_DEVICE_EXTERNAL_MMC == key ? _("memo_ti_card_removable") : NULL);
        } else if (STORAGE_DEVICE_FLASH == key) {
          /* HACK */
          g_free(new_name);
          new_name = g_strdup(_("memo_ti_memorynote_device"));
        }
        else if (!strncmp(new_name, "mmc-undefined-name", 18)) {
          /* ANOTHER HACK */
          g_free(new_name);
          new_name = g_strdup(STORAGE_DEVICE_INTERNAL_MMC == key ? _("memo_ti_card_internal")  :
                              STORAGE_DEVICE_EXTERNAL_MMC == key ? _("memo_ti_card_removable") : NULL);
        }

        if (NULL_SAFE_STRCMP(new_name, private->name)) {
          g_free(private->name);
          private->name = new_name;
          g_object_notify(G_OBJECT(storage_device), "name");
        } else {
          g_free(new_name);
        }
      }
    g_free(uri);
  }
}

static gboolean
gconf_get_bool(char *key)
{
  gboolean ret = FALSE;
  GConfClient *client = gconf_client_get_default();
  GConfValue *val = NULL;

  g_return_val_if_fail(client != NULL, FALSE);

  val = gconf_client_get(client, key, NULL);

  g_return_val_if_fail(val != NULL, FALSE);

  if (val->type != GCONF_VALUE_BOOL) {
    g_warning("gconf_get_bool: \"%s\" is not boolean\n", key);
  }

  ret = gconf_value_get_bool(val);

  gconf_value_free(val);

  return ret;
}

static void
storage_device_determine_status(StorageDevice *storage_device, StorageDevicePrivate *private, StorageDeviceKey key)
{
  GList *mounted_volumes = NULL, *llItr = NULL;
  const char *uri = storage_device_uri_from_key(key) ;
  char *volume_uri = NULL;
  GnomeVFSVolume *vol = NULL;
  StorageDeviceStatus new_status = STORAGE_DEVICE_UNAVAILABLE;
  gboolean usb_obscured = FALSE, cover_open = FALSE, device_present = FALSE, corrupted = FALSE, swap_enabled = FALSE;
  GString *str = NULL;
  char *internal_prefix = NULL;

#if (0)
  g_print("storage_device_determine_status: Current status is %s\n",
    STORAGE_DEVICE_AVAILABLE == private->status ? "STORAGE_DEVICE_AVAILABLE" :
    STORAGE_DEVICE_UNAVAILABLE == private->status ? "STORAGE_DEVICE_UNAVAILABLE" :
    STORAGE_DEVICE_USB_OBSCURED == private->status ? "STORAGE_DEVICE_USB_OBSCURED" :
    STORAGE_DEVICE_COVER_OPEN == private->status ? "STORAGE_DEVICE_COVER_OPEN" : "STORAGE_DEVICE_CORRUPTED");
#endif /* (0) */

  if (STORAGE_DEVICE_FLASH == key) {
    new_status = STORAGE_DEVICE_AVAILABLE;
  } else if ((str = g_string_new("")) != NULL) {

    if (STORAGE_DEVICE_INTERNAL_MMC == key) {
      internal_prefix = "internal-";
    } else {
      internal_prefix = "";
      cover_open = gconf_get_bool(GCONF_PATH_SYSTEM_OSSO_AF "/" GCONF_KEY_MMC_COVER_OPEN);
    }

    g_string_printf(str, GCONF_PATH_SYSTEM_OSSO_AF "/%s" GCONF_KEY_MMC_USED_OVER_USB, internal_prefix);
    usb_obscured = gconf_get_bool(str->str);

    g_string_printf(str, GCONF_PATH_SYSTEM_OSSO_AF "/%s" GCONF_KEY_MMC_DEVICE_PRESENT, internal_prefix);
    device_present = gconf_get_bool(str->str);

    g_string_printf(str, GCONF_PATH_SYSTEM_OSSO_AF "/mmc/%s" GCONF_KEY_MMC_CORRUPTED, internal_prefix);
    corrupted = gconf_get_bool(str->str);

    g_string_printf(str, GCONF_PATH_SYSTEM_OSSO_AF "/%s" GCONF_KEY_MMC_SWAP_ENABLED, internal_prefix);
    swap_enabled = gconf_get_bool(str->str);

    device_present = device_present && !cover_open;

    new_status = 
      device_present ? 
        usb_obscured ? STORAGE_DEVICE_USB_OBSCURED : 
            corrupted ? STORAGE_DEVICE_CORRUPTED : STORAGE_DEVICE_AVAILABLE
        : STORAGE_DEVICE_UNAVAILABLE;
    g_string_free(str, TRUE);
#if (0)
    g_print("storage_device_determine_status: Initial new status is %s\n",
      STORAGE_DEVICE_AVAILABLE == new_status ? "STORAGE_DEVICE_AVAILABLE" :
      STORAGE_DEVICE_UNAVAILABLE == new_status ? "STORAGE_DEVICE_UNAVAILABLE" :
      STORAGE_DEVICE_USB_OBSCURED == new_status ? "STORAGE_DEVICE_USB_OBSCURED" :
      STORAGE_DEVICE_COVER_OPEN == new_status ? "STORAGE_DEVICE_COVER_OPEN" : "STORAGE_DEVICE_CORRUPTED");
#endif /* (0) */
  }

  /* Set device name from GNOMEVFS volume */
  for (llItr = mounted_volumes = gnome_vfs_volume_monitor_get_mounted_volumes(gnome_vfs_get_volume_monitor()) ; llItr ; llItr = llItr->next) {
    if ((vol = GNOME_VFS_VOLUME(llItr->data)) != NULL) {
      volume_uri = gnome_vfs_volume_get_activation_uri(vol);
      if(!NULL_SAFE_STRCMP(volume_uri,uri)) {
        storage_device_set_name_from_vfs_volume(storage_device, vol);
        break;
      }
      g_free(volume_uri);
    }
  }
  g_list_foreach(mounted_volumes, (GFunc)gnome_vfs_volume_unref, NULL);
  g_list_free(mounted_volumes);

  if (!llItr && STORAGE_DEVICE_UNAVAILABLE != new_status && STORAGE_DEVICE_USB_OBSCURED != new_status)
    new_status = STORAGE_DEVICE_UNAVAILABLE;

  /* If the device is unavailable, force default name */
  if (STORAGE_DEVICE_AVAILABLE != new_status)
    if (NULL != private->name) {
      g_free(private->name);
      private->name = NULL;
    }

  if (NULL == private->name) {
    if ((private->name = g_strdup(STORAGE_DEVICE_FLASH        == private->key ? _("memo_ti_memorynote_device") :
                                  STORAGE_DEVICE_INTERNAL_MMC == private->key ? _("memo_ti_card_internal")     :
                                  STORAGE_DEVICE_EXTERNAL_MMC == private->key ? _("memo_ti_card_removable")    : NULL)) != NULL)
      g_object_notify(G_OBJECT(storage_device), "name");
  }

  if (new_status != private->status) {
    private->status = new_status;
    g_object_notify(G_OBJECT(storage_device), "status");
  }

  if (new_status == STORAGE_DEVICE_AVAILABLE && swap_enabled != private->swap_enabled) {
    private->swap_enabled = swap_enabled;
    g_object_notify(G_OBJECT(storage_device), "swap-enabled");
  }
#if (0)
  g_print("storage_device_determine_status: New status is %s\n",
    STORAGE_DEVICE_AVAILABLE == new_status ? "STORAGE_DEVICE_AVAILABLE" :
    STORAGE_DEVICE_UNAVAILABLE == new_status ? "STORAGE_DEVICE_UNAVAILABLE" :
    STORAGE_DEVICE_USB_OBSCURED == new_status ? "STORAGE_DEVICE_USB_OBSCURED" :
    STORAGE_DEVICE_COVER_OPEN == new_status ? "STORAGE_DEVICE_COVER_OPEN" : "STORAGE_DEVICE_CORRUPTED");
#endif /* (0) */
}

static void
listen_to_gconf(StorageDevice *storage_device, StorageDevicePrivate *private, StorageDeviceKey key)
{
  GConfClient *gconf_client;

  if (STORAGE_DEVICE_EXTERNAL_MMC == key) {
    if ((gconf_client = gconf_client_get_default()) == NULL) return;
    gconf_client_add_dir(gconf_client, GCONF_PATH_SYSTEM_OSSO_AF, GCONF_CLIENT_PRELOAD_NONE, NULL);

    private->usb_obscured_signal_id = gconf_client_notify_add(gconf_client, GCONF_PATH_SYSTEM_OSSO_AF "/" GCONF_KEY_MMC_USED_OVER_USB,
      (GConfClientNotifyFunc)storage_device_status_changed, storage_device, NULL, NULL);
    private->device_present_signal_id = gconf_client_notify_add(gconf_client, GCONF_PATH_SYSTEM_OSSO_AF "/" GCONF_KEY_MMC_DEVICE_PRESENT,
      (GConfClientNotifyFunc)storage_device_status_changed, storage_device, NULL, NULL);
    private->corrupted_signal_id = gconf_client_notify_add(gconf_client, GCONF_PATH_SYSTEM_OSSO_AF "/mmc/" GCONF_KEY_MMC_CORRUPTED,
      (GConfClientNotifyFunc)storage_device_status_changed, storage_device, NULL, NULL);
    private->swap_enabled_signal_id = gconf_client_notify_add(gconf_client, GCONF_PATH_SYSTEM_OSSO_AF "/" GCONF_KEY_MMC_SWAP_ENABLED,
      (GConfClientNotifyFunc)storage_device_status_changed, storage_device, NULL, NULL);
    private->cover_open_signal_id = gconf_client_notify_add(gconf_client, GCONF_PATH_SYSTEM_OSSO_AF "/" GCONF_KEY_MMC_COVER_OPEN,
      (GConfClientNotifyFunc)storage_device_status_changed, storage_device, NULL, NULL);
  } else if (STORAGE_DEVICE_INTERNAL_MMC == key) {
    if ((gconf_client = gconf_client_get_default()) == NULL) return;
    gconf_client_add_dir(gconf_client, GCONF_PATH_SYSTEM_OSSO_AF, GCONF_CLIENT_PRELOAD_NONE, NULL);

    private->usb_obscured_signal_id = gconf_client_notify_add(gconf_client, GCONF_PATH_SYSTEM_OSSO_AF "/internal-" GCONF_KEY_MMC_USED_OVER_USB,
      (GConfClientNotifyFunc)storage_device_status_changed, storage_device, NULL, NULL);
    private->device_present_signal_id = gconf_client_notify_add(gconf_client, GCONF_PATH_SYSTEM_OSSO_AF "/internal-" GCONF_KEY_MMC_DEVICE_PRESENT,
      (GConfClientNotifyFunc)storage_device_status_changed, storage_device, NULL, NULL);
    private->corrupted_signal_id = gconf_client_notify_add(gconf_client, GCONF_PATH_SYSTEM_OSSO_AF "/mmc/internal-" GCONF_KEY_MMC_CORRUPTED,
      (GConfClientNotifyFunc)storage_device_status_changed, storage_device, NULL, NULL);
    private->swap_enabled_signal_id = gconf_client_notify_add(gconf_client, GCONF_PATH_SYSTEM_OSSO_AF "/mmc/internal-" GCONF_KEY_MMC_SWAP_ENABLED,
      (GConfClientNotifyFunc)storage_device_status_changed, storage_device, NULL, NULL);
  }
}

static void
storage_device_determine_sizes(StorageDevice *storage_device, StorageDevicePrivate *private, StorageDeviceKey key)
{
  gint64 new_total_size = 0, new_free_size = 0;
  struct statfs stat_fs;
  char *path = NULL;

  if (STORAGE_DEVICE_AVAILABLE != private->status) return;

  path = (char *)storage_device_uri_from_key(key);

  g_return_if_fail(NULL != path);

  if (!strncmp(path, "file://", 7))
    path = &path[7];

  if (-1 == (statfs(path, &stat_fs))) return;

  new_total_size = ((gint64)(stat_fs.f_blocks)) * ((gint64)(stat_fs.f_bsize));
  new_free_size = ((gint64)(stat_fs.f_bfree)) * ((gint64)(stat_fs.f_bsize));

  if (STORAGE_DEVICE_FLASH == private->key) {
    new_free_size -= FLASH_RESERVED_SPACE;
    new_free_size = MAX(new_free_size, 0);
  }

  if (new_total_size != private->total_size) {
    private->total_size = new_total_size;
    g_object_notify(G_OBJECT(storage_device), "total-size");
  }

  if (new_free_size != private->free_size) {
    private->free_size = new_free_size;
    g_object_notify(G_OBJECT(storage_device), "free-size");
  }
}

static void
storage_device_status_changed(GConfClient *client, guint cnxn_id, GConfEntry *entry, StorageDevice *storage_device)
{
  StorageDevicePrivate *private = STORAGE_DEVICE_GET_PRIVATE(storage_device);

  storage_device_update(storage_device, private, private->key, FALSE);
}

static const char *
storage_device_uri_from_key(StorageDeviceKey key)
{
  return
    STORAGE_DEVICE_INTERNAL_MMC == key ? "file:///media/mmc1" :
    STORAGE_DEVICE_EXTERNAL_MMC == key ? "file:///media/mmc1" : "file:///";
}

static StorageDeviceKey
storage_device_key_from_uri(char *uri)
{
  return
    !strcmp(uri, "file:///media/mmc1") ? STORAGE_DEVICE_INTERNAL_MMC :
    !strcmp(uri, "file:///media/mmc1") ? STORAGE_DEVICE_EXTERNAL_MMC : 
    !strcmp(uri, "file:///") ? STORAGE_DEVICE_FLASH : STORAGE_DEVICE_NONE;
}

#ifdef RUN_STANDALONE_SD
int main (int argc, char **argv)
{
  StorageDevice *storage_device = NULL;

  g_type_init();

  storage_device = g_object_new(STORAGE_DEVICE_TYPE, "key", STORAGE_DEVICE_INTERNAL_MMC, NULL);

  g_main_loop_run(g_main_loop_new(NULL, FALSE));

  return 0;
}
#endif /* RUN_STANDALONE_SD */
