#include <stdlib.h>
#include <libosso.h>
#include <hildon/hildon.h>
#include <hildon/hildon-help.h>
#include <libgnomevfs/gnome-vfs-init.h>
#include <hildon/hildon-file-system-storage-dialog.h>
#include "i18n.h"
#include "deviceinfo.h"
#include "storagedevice.h"

#define DETAILS_HELP_KEY "utilities_controlpanelappletmemory_memory_details"

#define ICON_PATH "/usr/share/icons/hicolor/26x26/hildon/"

#define DEVICE_INFO_GET_PRIVATE(object) \
  (G_TYPE_INSTANCE_GET_PRIVATE((object), DEVICE_INFO_TYPE, DeviceInfoPrivate))

enum
{
  DEVICE_INFO_FIRST_PROPERTY = 0,
  DEVICE_INFO_STORAGE_DEVICE_PROPERTY,
  DEVICE_INFO_OSSO_CONTEXT_PROPERTY,
  DEVICE_INFO_LAST_PROPERTY
} ;

typedef struct
{
  GtkWidget *image;
  GtkWidget *name_label;
  GtkWidget *progress_bar;
  GtkWidget *details_button;
  GtkWidget *status_label;
  GtkWidget *details_dialog;
  StorageDevice *storage_device;
  osso_context_t *osso;
} DeviceInfoPrivate ;

static void device_info_init(GTypeInstance *instance, gpointer g_class);
static void device_info_class_init(gpointer g_class, gpointer null);

static void finalize(GObject *obj);
static void set_property(GObject *object, guint property_id, const GValue *val, GParamSpec *pspec);
static void get_property(GObject *object, guint property_id,       GValue *val, GParamSpec *pspec);
static void show(GtkWidget *widget);

static void storage_device_sizes_changed(GObject *storage_device, GParamSpec *pspec, DeviceInfoPrivate *priv);
static void storage_device_status_changed(GObject *storage_device, GParamSpec *pspec, DeviceInfo *devinfo);
static void storage_device_name_changed(GObject *storage_device, GParamSpec *pspec, DeviceInfo *devinfo);
static void details_button_clicked(GtkWidget *btn, DeviceInfo *device_info);

static void device_info_set_storage_device(DeviceInfo *devinfo, StorageDevice *storage_device);

GType
device_info_get_type(void)
{
  static GType the_type = 0;

  if (0 == the_type) {
    static GTypeInfo the_type_info = {
      .class_size     = sizeof(DeviceInfoClass),
      .base_init      = NULL,
      .base_finalize  = NULL,
      .class_init     = device_info_class_init,
      .class_finalize = NULL,
      .class_data     = NULL,
      .instance_size  = sizeof(DeviceInfo),
      .n_preallocs    = 0,
      .instance_init  = device_info_init,
      .value_table    = NULL
    } ;

    if ((the_type = g_type_register_static(GTK_TYPE_VBOX, DEVICE_INFO_TYPE_STRING, &the_type_info, 0)) != 0)
      g_type_class_ref(the_type);
  }

  return the_type;
}

static void
device_info_class_init(gpointer g_class, gpointer null)
{
  /* This should not be here */
  gnome_vfs_init();

  G_OBJECT_CLASS(g_class)->finalize = finalize;
  G_OBJECT_CLASS(g_class)->set_property = set_property;
  G_OBJECT_CLASS(g_class)->get_property = get_property;
  GTK_WIDGET_CLASS(g_class)->show = show;

  g_object_class_install_property(g_class, DEVICE_INFO_STORAGE_DEVICE_PROPERTY,
    g_param_spec_object("storage-device", "Storage Device", "Storage device whose properties to showcase", STORAGE_DEVICE_TYPE, G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property(g_class, DEVICE_INFO_OSSO_CONTEXT_PROPERTY,
    g_param_spec_pointer("osso-context", "OSSO Context", "OSSO context to use for context-sensitive help", G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
  g_type_class_add_private(g_class, sizeof(DeviceInfoPrivate));
}

static void
device_info_init(GTypeInstance *instance, gpointer g_class)
{
  GtkWidget *hbox = NULL;
  DeviceInfoPrivate *private = DEVICE_INFO_GET_PRIVATE(instance);

  private->osso = NULL;
  /* Top row hbox =================*/
  gtk_container_add_with_properties(GTK_CONTAINER(instance),
    hbox = g_object_new(GTK_TYPE_HBOX, "visible", TRUE, "spacing", HILDON_MARGIN_DEFAULT, NULL),
    "expand", FALSE, /*"padding", HILDON_MARGIN_DOUBLE, */NULL);

    /* Image ========================*/
    gtk_container_add_with_properties(GTK_CONTAINER(hbox),
      private->image = g_object_new(GTK_TYPE_IMAGE, "visible", TRUE, "stock", GTK_STOCK_OK, NULL),
      "expand", FALSE, NULL);
    /* Image ========================*/

    /* Device name ==================*/
    gtk_container_add_with_properties(GTK_CONTAINER(hbox),
      private->name_label = g_object_new(GTK_TYPE_LABEL, "visible", TRUE, "label", "The Label", "xalign", 0.0, "yalign", 0.5, "justify", GTK_JUSTIFY_LEFT, NULL),
      "expand", FALSE, NULL);
    /* Device name ==================*/
  /* Top row hbox =================*/

  /* Bottom row hbox ==============*/
  gtk_container_add_with_properties(GTK_CONTAINER(instance),
    hbox = g_object_new(GTK_TYPE_HBOX, "visible", TRUE, "spacing", HILDON_MARGIN_DOUBLE, NULL),
    "expand", FALSE, NULL);

    /* Progress bar =================*/
    gtk_container_add_with_properties(GTK_CONTAINER(hbox),
      private->progress_bar = g_object_new(GTK_TYPE_PROGRESS_BAR, "visible", TRUE, NULL),
      "expand", TRUE, NULL);
    /* Progress bar =================*/

    /* status label =================*/
    gtk_container_add_with_properties(GTK_CONTAINER(hbox),
      private->status_label = g_object_new(GTK_TYPE_LABEL, "visible", TRUE, "label", "", "xalign", 0.0, "yalign", 0.5, "justify", GTK_JUSTIFY_LEFT, NULL),
      "expand", TRUE, NULL);
    /* status label =================*/

    /* Details button ===============*/
    gtk_container_add_with_properties(GTK_CONTAINER(hbox),
      private->details_button = g_object_new(GTK_TYPE_BUTTON, "visible", TRUE, "label", _("memo_bd_details"), NULL),
      "expand", FALSE, NULL);
    /* Details button ===============*/
  /* Bottom row hbox ==============*/

  g_signal_connect(G_OBJECT(private->details_button), "clicked", (GCallback)details_button_clicked, instance);
}

static void
finalize(GObject *obj)
{
  DeviceInfoPrivate *private = DEVICE_INFO_GET_PRIVATE(obj);

  g_object_unref(private->storage_device);
}

static void
show(GtkWidget *widget)
{
  void (*parent_show) (GtkWidget *widget) = NULL;
  DeviceInfoPrivate *private = DEVICE_INFO_GET_PRIVATE(widget);
  StorageDeviceStatus status;

  g_object_get(G_OBJECT(private->storage_device), "status", &status, NULL);

  if (STORAGE_DEVICE_UNAVAILABLE == status) {
    gtk_widget_hide(widget);
  } else if ((parent_show = GTK_WIDGET_CLASS(g_type_class_peek(g_type_parent(DEVICE_INFO_TYPE)))->show) != NULL) {
    parent_show(widget);
  }
}

static void
get_property(GObject *object, guint property_id, GValue *val, GParamSpec *pspec)
{
  if (DEVICE_INFO_STORAGE_DEVICE_PROPERTY == property_id)
    g_value_set_object(val, DEVICE_INFO_GET_PRIVATE(object)->storage_device);
  else
  if (DEVICE_INFO_OSSO_CONTEXT_PROPERTY == property_id)
    g_value_set_pointer(val, DEVICE_INFO_GET_PRIVATE(object)->osso);
  else
    G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
}

static void
set_property(GObject *object, guint property_id, const GValue *val, GParamSpec *pspec)
{
  if (DEVICE_INFO_STORAGE_DEVICE_PROPERTY == property_id)
    device_info_set_storage_device(DEVICE_INFO(object), STORAGE_DEVICE(g_value_get_object(val)));
  else
  if (DEVICE_INFO_OSSO_CONTEXT_PROPERTY == property_id)
    DEVICE_INFO_GET_PRIVATE(object)->osso = g_value_get_pointer(val);
  else
    G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
}

static void
device_info_set_storage_device(DeviceInfo *devinfo, StorageDevice *storage_device)
{
  StorageDeviceKey key = STORAGE_DEVICE_NONE;
  DeviceInfoPrivate *private = DEVICE_INFO_GET_PRIVATE(devinfo);
  char *file_name = NULL;

  g_object_get(G_OBJECT(storage_device), "key", &key, NULL);
  private->storage_device = storage_device;

  file_name =
    STORAGE_DEVICE_FLASH        == key ? ICON_PATH "qgn_list_filesys_divc_cls.png" :
    STORAGE_DEVICE_INTERNAL_MMC == key ? ICON_PATH "qgn_list_gene_internal_memory_card.png" :
    STORAGE_DEVICE_EXTERNAL_MMC == key ? ICON_PATH "qgn_list_gene_removable_memory_card.png" : NULL;

  if (file_name)
    g_object_set(G_OBJECT(private->image), "file", file_name, NULL);
  else
    g_object_set(G_OBJECT(private->image), "stock", GTK_STOCK_MISSING_IMAGE, NULL);

  g_signal_connect(G_OBJECT(private->storage_device), "notify::total-size", (GCallback)storage_device_sizes_changed, private);
  g_signal_connect(G_OBJECT(private->storage_device), "notify::free-size", (GCallback)storage_device_sizes_changed, private);
  g_signal_connect(G_OBJECT(private->storage_device), "notify::status", (GCallback)storage_device_status_changed, devinfo);
  g_signal_connect(G_OBJECT(private->storage_device), "notify::name", (GCallback)storage_device_name_changed, devinfo);

  storage_device_sizes_changed(G_OBJECT(private->storage_device), NULL, private);
  storage_device_status_changed(G_OBJECT(private->storage_device), NULL, devinfo);
  storage_device_name_changed(G_OBJECT(private->storage_device), NULL, devinfo);
}

static char *
get_size_in_bytes_user_string(guint64 size_in_bytes)
{
  char *size_str = NULL;
  if (size_in_bytes < 1024) size_in_bytes = 1024 ;
  if (size_in_bytes <= 101376) /* 99 kB */
    size_str = g_strdup_printf (_("memo_li_properties_size_1kb_99kb"), (gint)(((double)size_in_bytes) / 1024.0)) ;
  else
  if (size_in_bytes <= 10485760) /* 10 MB */
    size_str = g_strdup_printf (_("memo_li_properties_size_1mb_10mb"), (gfloat)(((double)size_in_bytes) / 1048576.0)) ;
  else
  if (size_in_bytes <= 1073741824) /* 1 GB */
    size_str = g_strdup_printf (_("memo_li_properties_size_10mb_1gb"), (gfloat)(((double)size_in_bytes) / 1048576.0)) ;
  else /* > 1 GB */
    size_str = g_strdup_printf (_("memo_li_properties_size_1gb_or_greater"), (gfloat)(((double)size_in_bytes) / 1073741824.0)) ;

  return size_str;
}

static void
storage_device_sizes_changed(GObject *storage_device, GParamSpec *pspec, DeviceInfoPrivate *priv)
{
  StorageDeviceKey key;
  guint64 total_size, free_size;
  char *str = NULL, *str_avail = NULL;
  double fraction = -1.0;

  g_object_get(G_OBJECT(storage_device), "total-size", &total_size, "free-size", &free_size, "key", &key, NULL);

  if (0 == total_size) return;

  /* Show at least 1KB ... *shrug* ... */
  free_size = MAX(free_size, 1024);

  g_object_set(G_OBJECT(priv->progress_bar),
    "text-xalign", 0.0, "text-yalign", 0.5,
    NULL);

  gtk_progress_bar_set_text(GTK_PROGRESS_BAR(priv->progress_bar), str_avail = g_strdup_printf (_("memo_li_available_space"), str = get_size_in_bytes_user_string(free_size)));
  fraction = ((double)(total_size - free_size)) /  ((double)total_size);
  fraction = CLAMP(fraction, 0.0, 1.0);
  gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(priv->progress_bar), fraction);
  g_free(str);
  g_free(str_avail);
}

static void
storage_device_name_changed(GObject *storage_device, GParamSpec *pspec, DeviceInfo *devinfo)
{
  DeviceInfoPrivate *priv = DEVICE_INFO_GET_PRIVATE(devinfo);
  char *new_name = NULL;

  g_object_get(G_OBJECT(storage_device), "name", &new_name, NULL);
  if (new_name)
    g_object_set(G_OBJECT(priv->name_label), "label", new_name, NULL);
}

static void
storage_device_status_changed(GObject *storage_device, GParamSpec *pspec, DeviceInfo *devinfo)
{
  DeviceInfoPrivate *priv = DEVICE_INFO_GET_PRIVATE(devinfo);
  StorageDeviceStatus status;

  g_object_get(G_OBJECT(storage_device), "status", &status, NULL);

  if (status == STORAGE_DEVICE_AVAILABLE) {
    gtk_widget_show(GTK_WIDGET(devinfo));
    gtk_widget_show(priv->progress_bar);
    gtk_widget_show(priv->details_button);
    gtk_widget_hide(priv->status_label);
    if (priv->details_dialog)
      gtk_widget_show(priv->details_dialog);
  } else if (status == STORAGE_DEVICE_UNAVAILABLE) {
    gtk_widget_hide(GTK_WIDGET(devinfo));
    if (priv->details_dialog)
      gtk_widget_hide(priv->details_dialog);
  } else {
    gtk_widget_show(GTK_WIDGET(devinfo));
    gtk_widget_hide(priv->progress_bar);
    gtk_widget_hide(priv->details_button);
    gtk_widget_show(priv->status_label);
    if (priv->details_dialog)
      gtk_widget_hide(priv->details_dialog);
  }

  g_object_set(G_OBJECT(priv->status_label), "label", 
    STORAGE_DEVICE_USB_OBSCURED == status ? _("memo_ia_pc_connected") :
    STORAGE_DEVICE_CORRUPTED    == status ? _("memo_ia_corrupt_card") : "", NULL);
}

static void
details_button_clicked(GtkWidget *btn, DeviceInfo *device_info)
{
  DeviceInfoPrivate *private = DEVICE_INFO_GET_PRIVATE(device_info);
  char *details_path = NULL;
  char *mydocs_uri = getenv("MYDOCSDIR");
  StorageDeviceKey key = STORAGE_DEVICE_NONE;

  g_debug("details_button_clicked: value of MYDOCSDIR is %s\n", mydocs_uri);

  g_object_get (private->storage_device, "key", &key, NULL);

  if (STORAGE_DEVICE_NONE == key) return;

  mydocs_uri = g_strdup_printf("file://%s", mydocs_uri);

  details_path = 
    STORAGE_DEVICE_FLASH        == key ? (NULL == mydocs_uri ? "file:///home/user/MyDocs" : mydocs_uri) :
    STORAGE_DEVICE_INTERNAL_MMC == key ? "file:///media/mmc1" :
    STORAGE_DEVICE_EXTERNAL_MMC == key ? "file:///media/mmc1" : NULL;

  g_debug("details_button_clicked: Showing details dialog for path %s\n", details_path);

  private->details_dialog = hildon_file_system_storage_dialog_new(GTK_WINDOW(gtk_widget_get_toplevel(btn)), details_path);
  if (private->osso)
    hildon_help_dialog_help_enable(GTK_DIALOG(private->details_dialog), DETAILS_HELP_KEY, private->osso);
  gtk_dialog_run(GTK_DIALOG(private->details_dialog));
  gtk_widget_destroy(private->details_dialog);
  private->details_dialog = NULL;
  g_free(mydocs_uri);
}
