/**
 * @file mcetool.c
 * Tool to test and remote control the Mode Control Entity
 * <p>
 * Copyright © 2005-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>

#include <errno.h>			/* errno,
					 * ENOMEM
					 */
#include <stdio.h>			/* fprintf() */
#define _GNU_SOURCE
#include <getopt.h>			/* getopt_long(),
					 * struct option
					 */
#include <stdlib.h>			/* exit(), EXIT_FAILURE */
#include <string.h>			/* strcmp(), strlen(), strdup() */
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>	/* dbus_connection_setup_with_g_main */
#include <gconf/gconf-client.h>

#include <cal.h>
#include <mce/dbus-names.h>
#include <mce/mode-names.h>
#include <systemui/dbus-names.h>
#include <systemui/tklock-dbus-names.h>
#include <systemui/devlock-dbus-names.h>
#include <systemui/actingdead-dbus-names.h>
#include <systemui/modechange-dbus-names.h>
#include <systemui/alarm_dialog-dbus-names.h>
#include <systemui/powerkeymenu-dbus-names.h>
#include <systemui/splashscreen-dbus-names.h>

#include "mcetool.h"

#include "mce-log.h"			/* mce_log_open(), mce_log_close(),
					 * mce_log(), mce_log_set_verbosity(),
					 * LL_*
					 */

#include "tklock.h"			/* For GConf paths */
#include "devlock.h"			/* For GConf paths */
#include "mce-dsme.h"			/* For GConf paths */

/** Name shown by --help etc. */
#define PRG_NAME			"mcetool"

/** Open Acting Dead UI string; used for arg-parsing */
#define OPEN_ACTDEAD_STR		"open"
/** Close Acting Dead UI string; used for arg-parsing */
#define CLOSE_ACTDEAD_STR		"close"

/** Open Startup Splashscreen UI string; used for arg-parsing */
#define STARTUP_SPLASH_STR		"startup"
/** Open Shutdown Splashscreen UI string; used for arg-parsing */
#define SHUTDOWN_SPLASH_STR		"shutdown"
/** Open Silent Shutdown Splashscreen UI string; used for arg-parsing */
#define SILENT_SHUTDOWN_SPLASH_STR	"silent-shutdown"
/** Close Splashscreen UI string; used for arg-parsing */
#define CLOSE_SPLASH_STR		"close"

/** Open Normal Modechange UI string; used for arg-parsing */
#define NORMAL_MODECHANGE_STR		"normal"
/** Open Flight Modechange UI string; used for arg-parsing */
#define FLIGHT_MODECHANGE_STR		"flight"
/** Close Modechange UI string; used for arg-parsing */
#define CLOSE_MODECHANGE_STR		"close"

/** Open Normal Powerkey Menu UI string; used for arg-parsing */
#define NORMAL_POWERKEYMENU_STR		"normal"
/** Open Flight Powerkey Menu UI string; used for arg-parsing */
#define FLIGHT_POWERKEYMENU_STR		"flight"
/** Close Flight Powerkey Menu UI string; used for arg-parsing */
#define CLOSE_POWERKEYMENU_STR		"close"

/** Open Normal Device Lock UI string; used for arg-parsing */
#define NORMAL_DEVICELOCK_STR		"normal"
/** Open No-input Device Lock UI string; used for arg-parsing */
#define NOINPUT_DEVICELOCK_STR		"noinput"
/** Open Shutdown Confirmation Device Lock UI string; used for arg-parsing */
#define CONFIRM_DEVICELOCK_STR		"confirm"
/** Close Device Lock UI string; used for arg-parsing */
#define CLOSE_DEVICELOCK_STR		"close"

/** Open Normal Touchscreen/Keypad Lock UI string; used for arg-parsing */
#define OPEN_TKLOCK_STR			"open"
/** Open Silent Touchscreen/Keypad Lock UI string; used for arg-parsing */
#define SOPEN_TKLOCK_STR		"silent-open"
/** Open Helptext Touchscreen/Keypad Lock UI string; used for arg-parsing */
#define HELP_TKLOCK_STR			"help"
/** Open Press select Touchscreen/Keypad Lock UI string; used for arg-parsing */
#define SELECT_TKLOCK_STR		"select"
/** Open Event-eater UI string; used for arg-parsing */
#define ONEINPUT_TKLOCK_STR		"oneinput"
/** Close Touchscreen/Keypad Lock UI string; used for arg-parsing */
#define CLOSE_TKLOCK_STR		"close"
/** Close (Silent) Touchscreen/Keypad Lock UI string; used for arg-parsing */
#define SCLOSE_TKLOCK_STR		"silent-close"

/** Open Normal Alarm UI string; used for arg-parsing */
#define NORMAL_ALARM_STR		"normal"
/** Open No-snooze Alarm UI string; used for arg-parsing */
#define NOSNOOZE_ALARM_STR		"nosnooze"
/** Open Switch on device Alarm UI string; used for arg-parsing */
#define SWITCHON_ALARM_STR		"switchon"
/** Close Alarm UI string; used for arg-parsing */
#define CLOSE_ALARM_STR			"close"

/** Short powerkey event string; used for arg-parsing */
#define SHORT_EVENT_STR			"short"
/** Long powerkey event string; used for arg-parsing */
#define LONG_EVENT_STR			"long"

/** Enums for actingdead UI modes */
enum {
	INVALID_ACTDEAD = -1,	/**< Acting Dead value not set */
	OPEN_ACTDEAD,		/**< Show Acting Dead UI */
	CLOSE_ACTDEAD		/**< Close Acting Dead UI */
};

/** Enums for Splashscreen UI modes */
enum {
	INVALID_SPLASH = -1,	/**< Splashscreen value not set */
	STARTUP_SPLASH,		/**< Show startup Splashscreen UI */
	SHUTDOWN_SPLASH,	/**< Show shutdown Splashscreen UI */
	SILENT_SHUTDOWN_SPLASH,	/**< Show silent shutdown Splashscreen UI */
	CLOSE_SPLASH		/**< Close Splashscreen UI */
};

/** Enums for Modechange UI modes */
enum {
	INVALID_MODECHANGE = -1,	/**< Modechange value not set */
	NORMAL_MODECHANGE,		/**< Show Normal Modechange UI */
	FLIGHT_MODECHANGE,		/**< Show Flightmode Modechange UI */
	CLOSE_MODECHANGE		/**< Close Modechange UI */
};

/** Enums for Powerkey Menu UI modes */
enum {
	INVALID_POWERKEYMENU = -1,	/**< Powerkey Menu value not set */
	NORMAL_POWERKEYMENU,		/**< Show Normal Powerkey Menu UI */
	FLIGHT_POWERKEYMENU,		/**< Show Flightmode Powerkey Menu UI */
	CLOSE_POWERKEYMENU		/**< Close Powerkey Menu UI */
};

/** Enums for Device Lock UI modes */
enum {
	INVALID_DEVICELOCK = -1,	/**< Devicelock value not set */
	NORMAL_DEVICELOCK,		/**< Show Normal Devicelock UI */
	NOINPUT_DEVICELOCK,		/**< Show No-input Devicelock UI */
	CONFIRM_DEVICELOCK,		/**< Show Confirm Devicelock UI */
	CLOSE_DEVICELOCK		/**< Close Devicelock UI */
};

/** Enums for Touchscreen/Keypad Lock UI modes */
enum {
	INVALID_TKLOCK = -1,		/**< T/K Lock value not set */
	OPEN_TKLOCK,			/**< Show T/K Lock UI */
	HELP_TKLOCK,			/**< Show Help T/K Lock UI */
	SELECT_TKLOCK,			/**< Show Select T/K Lock UI */
	ONEINPUT_TKLOCK,		/**< Show Event-eater T/K Lock UI */
	CLOSE_TKLOCK			/**< Close T/K Lock UI */
};

/** Enums for Touchscreen/Keypad Lock modes */
enum {
	INVALID_TKLOCK_MODE = -1,	/**< T/K Lock Mode value not set */
	LOCKED_TKLOCK_MODE,		/**< Locked T/K Lock Mode */
	LOCKED_DIM_TKLOCK_MODE,		/**< Locked, dimmed T/K Lock Mode */
	SILENT_LOCKED_TKLOCK_MODE,	/**< Locked T/K Lock Mode;
					 *   no feedback
					 */
	SILENT_LOCKED_DIM_TKLOCK_MODE,	/**< Locked T/K Lock Mode;
					 *   no feedback
					 */
	UNLOCKED_TKLOCK_MODE,		/**< Unlocked T/K Lock Mode  */
	SILENT_UNLOCKED_TKLOCK_MODE	/**< Unlocked T/K Lock Mode;
					 *   no feedback
					 */
};

/** Enums for Alarm UI modes */
enum {
	INVALID_ALARM = -1,		/**< Alarm value not set */
	NORMAL_ALARM,			/**< Open Normal Alarm UI */
	NOSNOOZE_ALARM,			/**< Open No-snooze Alarm UI */
	SWITCHON_ALARM,			/**< Open Switch-on Alarm UI */
	CLOSE_ALARM			/**< Close T/K Lock UI */
};

/** Enums for Alarm time settings */
enum {
	INVALID_TIME = 99,		/**< Alarm time not set */
	REMOVE_TIME = 100		/**< Remove alarm time */
};

/** Enums for Alarm time settings */
enum {
	INVALID_EVENT = -1,		/**< Event not set */
	SHORT_EVENT = 0,		/**< Short powerkey press event */
	LONG_EVENT = 1			/**< Long powerkey press event */
};

extern int optind;			/**< Used by getopt */
extern char *optarg;			/**< Used by getopt */

static const gchar *progname;	/**< Used to store the name of the program */

static DBusConnection *dbus_connection;	/**< D-Bus connection */

static GConfClient *gconf_client = NULL;	/**< GConf client */

static GMainLoop *mainloop = NULL;	/**< The GMainLoop used by mcetool */

static gboolean powerkey_menu(const gboolean normal,
			      const gboolean open);
static gboolean devicelock_dialog(const dbus_uint32_t mode,
				  const gboolean open);
static gboolean tklock_dialog(const dbus_uint32_t mode,
			      const dbus_bool_t silent,
			      const gboolean open);
static gboolean alarm_dialog(const dbus_uint32_t mode,
			     const gboolean open,
			     const gchar *const alarmtext,
			     const gchar *const alarmsound,
			     const gchar *const alarmicon);

/**
 * Display usage information
 */
static void usage(void)
{
	fprintf(stdout,
		_("Usage: %s [OPTION]\n"
		  "Mode Control Entity tool\n"
		  "\n"
		  "      --reset-passwd            reset the device lock "
		  "password\n"
		  "      --blank-prevent           send blank prevent request "
		  "to MCE\n"
		  "      --unblank-screen          send unblank request to "
		  "MCE\n"
		  "      --dim-screen              send dim request to MCE\n"
		  "      --blank-screen            send blank request to MCE\n"
		  "      --set-mode=MODE           set the mode to MODE; "
		  "valid modes are:\n"
		  "                                  ``normal'', ``voip'', "
		  "and ``flight''\n"
		  "      --powerup                 send powerup request to "
		  "MCE\n"
		  "      --reboot                  send reboot request to "
		  "MCE\n"
		  "      --shutdown                send shutdown request "
		  "to MCE\n"
		  "      --actdead=MODE            show/hide acting dead UI, "
		  "valid modes are:\n"
		  "                                  ``open'', and ``close'',\n"
		  "      --splashscreen=MODE       show/hide splashscreen, "
		  "valid modes are:\n"
		  "                                  ``startup'', "
		  "``shutdown'',\n"
		  "                                  ``silent-shutdown'', "
		  "and ``close''\n"
		  "      --modechange=MODE         show/hide modechange "
		  "dialog, valid modes are:\n"
		  "                                  ``normal''(*), "
		  "``flight''(*), and ``close''\n"
		  "      --powerkeymenu=MODE       show/hide the powerkey "
		  "menu, valid modes are:\n"
		  "                                  ``normal''(*), "
		  "``flight''(*), and ``close''\n"
		  "      --devicelock=MODE         show/hide the devicelock "
		  "dialog, valid modes are:\n"
		  "                                  ``normal''(*), "
		  "``noinput''(*),\n"
		  "                                  ``confirm''(*), "
		  "and ``close''\n"
		  "      --set-tklock-mode=MODE    set the touchscreen/"
		  "keypad lock mode;\n"
		  "                                  valid modes are:\n"
		  "                                  ``locked'', "
		  "``locked-dim'', ``silent-locked'',\n"
		  "                                  ``silent-locked-dim'', "
		  "``unlocked'', and ``silent-unlocked''\n"
		  "      --tklock=MODE             show/hide the touchscreen/"
		  "keypad lock,\n"
		  "                                  valid modes are:\n"
		  "                                  ``open'', "
		  "``silent-open'',\n"
		  "                                  ``help'', ``select''(*), "
		  "``oneinput''(*),\n"
		  "                                  ``close'', and "
		  "``silent-close''\n"
		  "      --alarm=MODE              show/hide the alarm "
		  "dialog, valid modes are:\n"
		  "                                  ``normal''(*), "
		  "``nosnooze''(*),\n"
		  "                                  ``switchon''(*), "
		  "and ``close''\n"
		  "      --alarm-text=STRING       alarm message\n"
		  "      --alarm-sound=PATH        alarm sound\n"
		  "      --alarm-icon=PATH         alarm icon to use\n"
		  "      --set-alarm=HH:MM         set the alarm clock to "
		  "HH:MM\n"
		  "      --disable-alarm           disable the alarm\n"
		  "      --enable-led              enable LED framework\n"
		  "      --disable-led             disable LED framework\n"
		  "      --activate-led-pattern    activate a LED pattern\n"
		  "      --deactivate-led-pattern  deactivate a LED pattern\n"
		  "      --powerkey-event=TYPE     trigger a powerkey event;\n"
		  "                                  valid types are:\n"
		  "                                  ``short'', ``long''\n"
		  "      --dont-block              return immediately even "
		  "from commands that\n"
		  "                                  would normally block\n"
		  "      --no-status               don't output MCE status\n"
		  "  -S, --session                 use the session bus "
		  "instead of the system bus\n"
		  "                                  for D-Bus\n"
		  "      --verbose                 increase debug message "
		  "verbosity\n"
		  "      --quiet                   decrease debug message "
		  "verbosity\n"
		  "      --help                    display this help and exit\n"
		  "      --version                 output version information "
		  "and exit\n"
		  "\n"
		  "Note: commands/modes marked (*) will block until there "
		  "is user input,\n"
		  "unless --dont-block has been specified\n"
		  "\n"
		  "Report bugs to <david.weinehall@nokia.com>\n"),
		progname);
}

/**
 * Display version information
 */
static void version(void)
{
	fprintf(stdout, _("%s v%s\n%s"),
		progname,
		G_STRINGIFY(PRG_VERSION),
		_("Written by David Weinehall.\n"
		  "\n"
		  "Copyright (C) 2005-2007 Nokia Corporation.  "
		  "All rights reserved.\n"));
}

/**
 * Initialise locale support
 *
 * @param name The program name to output in usage/version information
 * @return 0 on success, non-zero on failure
 */
static gint init_locales(const gchar *const name)
{
	gint status = 0;

#ifdef ENABLE_NLS
	setlocale(LC_ALL, "");

	if ((bindtextdomain(name, LOCALEDIR) == 0) && (errno == ENOMEM)) {
		status = errno;
		goto EXIT;
	}

	if ((textdomain(name) == 0) && (errno == ENOMEM)) {
		status = errno;
		return 0;
	}

EXIT:
	/* In this error-message we don't use _(), since we don't
	 * know where the locales failed, and we probably won't
	 * get a reasonable result if we try to use them.
	 */
	if (status != 0) {
		fprintf(stderr,
			"%s: `%s' failed; %s. Aborting.\n",
			name, "init_locales", g_strerror(errno));
	} else {
		progname = name;
		errno = 0;
	}
#else
	progname = name;
#endif /* ENABLE_NLS */

	return status;
}

/**
 * Create a new D-Bus signal, with proper error checking
 * will exit the mainloop if an error occurs
 *
 * @param path The signal path
 * @param interface The signal interface
 * @param name The name of the signal to send
 * @return A new DBusMessage
 */
static DBusMessage *dbus_new_signal(const gchar *const path,
				    const gchar *const interface,
				    const gchar *const name)
{
	DBusMessage *msg;

	if ((msg = dbus_message_new_signal(path, interface, name)) == NULL) {
		mce_log(LL_CRIT, "No memory for new signal!");
		exit(EXIT_FAILURE);
	}

	return msg;
}

/**
 * Create a new D-Bus method call, with proper error checking
 * will exit the mainloop if an error occurs
 *
 * @param service The method call service
 * @param path The method call path
 * @param interface The method call interface
 * @param name The name of the method to call
 * @return A new DBusMessage
 */
static DBusMessage *dbus_new_method_call(const gchar *const service,
					 const gchar *const path,
					 const gchar *const interface,
					 const gchar *const name)
{
	DBusMessage *msg;

	if ((msg = dbus_message_new_method_call(service, path,
						interface, name)) == NULL) {
		mce_log(LL_CRIT,
			"Cannot allocate memory for D-Bus method call!");
		exit(EXIT_FAILURE);
	}

	return msg;
}

/**
 * Create a new D-Bus method call reply, with proper error checking
 * will exit the mainloop if an error occurs
 *
 * @param message The DBusMessage to reply to
 * @return A new DBusMessage
 */
static DBusMessage *dbus_new_method_reply(DBusMessage *const message)
{
	DBusMessage *msg;

	if ((msg = dbus_message_new_method_return(message)) == NULL) {
		mce_log(LL_CRIT, "No memory for new reply!");
		g_main_loop_quit(mainloop);
		exit(EXIT_FAILURE);
	}

	return msg;
}

/**
 * Send a D-Bus message
 * Side effects: frees msg
 *
 * @param msg The D-Bus message to send
 * @return TRUE on success, FALSE on out of memory
 */
static gboolean dbus_send_message(DBusMessage *const msg)
{
	if (dbus_connection_send(dbus_connection, msg, NULL) == FALSE) {
		dbus_message_unref(msg);
		return FALSE;
	}

	dbus_connection_flush(dbus_connection);
	dbus_message_unref(msg);

	return TRUE;
}

/**
 * Call a D-Bus method
 *
 * @param method The method to call
 * @param arg An pointer to the string to append;
 *            if a reply is expected, a string with the reply will be
 *            returned through this pointer
 * @param no_reply TRUE if no reply is expected, FALSE if a reply is expected
 * @return 0 on success, EXIT_FAILURE on failure
 */
static gint mcetool_dbus_call(const gchar *const method,
			      gchar **arg, const gboolean no_reply)
{
	DBusMessage *reply = NULL;
	DBusMessage *msg;
	gint status = 0;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	if ((msg = dbus_message_new_method_call(MCE_SERVICE,
						MCE_REQUEST_PATH,
						MCE_REQUEST_IF,
						method)) == NULL) {
		mce_log(LL_CRIT,
			"Cannot allocate memory for D-Bus method call!");
		status = EXIT_FAILURE;
		goto EXIT;
	}

	/* Is there an argument to append? */
	if (arg != NULL && *arg != NULL) {
		if (dbus_message_append_args(msg,
					     DBUS_TYPE_STRING, arg,
					     DBUS_TYPE_INVALID) != TRUE) {
			dbus_message_unref(msg);
			mce_log(LL_CRIT,
				"Failed to append argument to D-Bus message "
				"for %s",
				method);
			status = EXIT_FAILURE;
			goto EXIT;
		}
	}

	if (no_reply == TRUE) {
		dbus_message_set_no_reply(msg, TRUE);

		if (dbus_connection_send(dbus_connection, msg, NULL) == FALSE) {
			dbus_message_unref(msg);
			goto EXIT;
		}
	} else {
		reply = dbus_connection_send_with_reply_and_block(dbus_connection, msg, -1, &error);
	}

	dbus_message_unref(msg);

	if (arg && *arg) {
		free(*arg);
		*arg = NULL;
	}

	if (dbus_error_is_set(&error) == TRUE) {
		mce_log(LL_CRIT,
			"Could not call method %s: %s; exiting",
			method, error.message);
		dbus_error_free(&error);
		status = EXIT_FAILURE;
		goto EXIT;
	} else if (reply) {
		gchar *tmp = NULL;

		if (dbus_message_get_args(reply, &error,
					  DBUS_TYPE_STRING, &tmp,
					  DBUS_TYPE_INVALID) == FALSE) {
			mce_log(LL_ERR,
				"Failed to get reply argument from %s: "
				"%s; exiting",
				method, error.message);
			dbus_message_unref(reply);
			dbus_error_free(&error);
			status = EXIT_FAILURE;
			goto EXIT;
		}

		*arg = strdup(tmp);
		dbus_message_unref(reply);
	}

EXIT:
	return status;
}

/**
 * Generic function to send D-Bus messages and signals
 * to send a signal, call dbus_send with service == NULL
 *
 * @param service D-Bus service; for signals, set to NULL
 * @param path D-Bus path
 * @param interface D-Bus interface
 * @param name D-Bus method or signal name to send to
 * @param no_reply FALSE if a reply is expected, TRUE if reply is ignored;
 *                 for signals this is ignored, but for consistency,
 *                 please use FALSE
 * @param first_arg_type The DBUS_TYPE of the first argument in the list
 * @param ... The arguments to append to the D-Bus message; terminate with NULL
 *            Note: the arguments MUST be passed by reference
 * @return TRUE on success, FALSE on failure
 */
static gboolean dbus_send(const gchar *const service, const gchar *const path,
			  const gchar *const interface, const gchar *const name,
			  const gboolean no_reply, int first_arg_type, ...)
{
	DBusMessage *msg;
	gboolean status = FALSE;
	va_list var_args;

	if (service != NULL) {
		msg = dbus_new_method_call(service, path, interface, name);

		if (no_reply == TRUE)
			dbus_message_set_no_reply(msg, TRUE);
	} else {
		msg = dbus_new_signal(path, interface, name);
	}

	/* Append the arguments, if any */
	va_start(var_args, first_arg_type);

	if (first_arg_type != DBUS_TYPE_INVALID) {
		if (dbus_message_append_args_valist(msg,
						    first_arg_type,
						    var_args) == FALSE) {
			mce_log(LL_CRIT,
				"Failed to append arguments to D-Bus message");
			dbus_message_unref(msg);
			goto EXIT;
		}
	}

	/* Send the signal / call the method */
	if (dbus_send_message(msg) == FALSE) {
		if (service != NULL)
			mce_log(LL_CRIT, "Cannot call method %s", name);
		else
			mce_log(LL_CRIT, "Cannot send signal %s", name);

		goto EXIT;
	}

	status = TRUE;

EXIT:
	va_end(var_args);

	return status;
}

/**
 * Get inactivity status
 *
 * @param[out] inactive TRUE if the device is inactive,
 *                      FALSE if the device is active or if the call fails
 * @return 0 on success, EXIT_FAILURE on failure
 */
static gint get_inactivity_status(gboolean *inactive)
{
	DBusMessage *reply = NULL;
	DBusMessage *msg;
	gint status = 0;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	if ((msg = dbus_message_new_method_call(MCE_SERVICE,
						MCE_REQUEST_PATH,
						MCE_REQUEST_IF,
						MCE_INACTIVITY_STATUS_GET)) == NULL) {
		mce_log(LL_CRIT,
			"Cannot allocate memory for D-Bus method call!");
		status = EXIT_FAILURE;
		goto EXIT;
	}

	reply = dbus_connection_send_with_reply_and_block(dbus_connection, msg, -1, &error);

	dbus_message_unref(msg);

	if (dbus_error_is_set(&error) == TRUE) {
		mce_log(LL_CRIT,
			"Could not call method %s: %s; exiting",
			MCE_INACTIVITY_STATUS_GET, error.message);
		dbus_error_free(&error);
		status = EXIT_FAILURE;
		goto EXIT;
	}

	if (dbus_message_get_args(reply, &error,
				  DBUS_TYPE_BOOLEAN, inactive,
				  DBUS_TYPE_INVALID) == FALSE) {
		mce_log(LL_ERR,
			"Failed to get reply argument from %s: %s; exiting",
			MCE_INACTIVITY_STATUS_GET, error.message);
		dbus_message_unref(reply);
		dbus_error_free(&error);
		status = EXIT_FAILURE;
		goto EXIT;
	}

	dbus_message_unref(reply);

EXIT:
	return status;
}

/**
 * Enable/disable the powerup splashscreen
 *
 * @param enable TRUE to enable splashscreen, FALSE to disable splashscreen
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_powerup_splash(const gboolean enable)
{
	const dbus_uint32_t splashtype = SPLASHSCREEN_ENABLE_BOOTUP;

	mce_log(LL_DEBUG, "Calling bootup splashscreen (%d)", enable);

	/* com.nokia.system_ui.request.splashscreen_{open,close} */
	return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
			 SYSTEMUI_REQUEST_IF,
			 enable ? SYSTEMUI_SPLASHSCREEN_OPEN_REQ :
				  SYSTEMUI_SPLASHSCREEN_CLOSE_REQ, TRUE,
			 DBUS_TYPE_UINT32, &splashtype,
			 DBUS_TYPE_INVALID);
}

/**
 * Enable/disable the shutdown splashscreen
 *
 * @param enable TRUE to enable splashscreen, FALSE to disable splashscreen
 * @param sound TRUE to play splashsound, FALSE to inhibit splashsound
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_shutdown_splash(const gboolean enable, const gboolean sound)
{
	const dbus_uint32_t splashtype = SPLASHSCREEN_ENABLE_SHUTDOWN;

	mce_log(LL_DEBUG, "Calling shutdown splashscreen (%d)", enable);

	/* com.nokia.system_ui.request.splashscreen_{open,close} */
	return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
			 SYSTEMUI_REQUEST_IF,
			 enable ? SYSTEMUI_SPLASHSCREEN_OPEN_REQ :
				  SYSTEMUI_SPLASHSCREEN_CLOSE_REQ, TRUE,
			 DBUS_TYPE_UINT32, &splashtype,
			 DBUS_TYPE_BOOLEAN, &sound,
			 DBUS_TYPE_INVALID);
}

/**
 * Set the alarm time
 *
 * @param hour Alarm time; hour
 * @param hour Alarm time; minute
 * @return TRUE on success, FALSE on failure
 */
static gboolean set_alarm(const guint hour, const guint minute)
{
	DBusError error;
	gint status = 0;

	/* Register error channel */
	dbus_error_init(&error);

	mce_log(LL_DEBUG, "Establishing temporary D-Bus connection");

	/* Establish D-Bus connection */
	if ((dbus_connection = dbus_bus_get(DBUS_BUS_SESSION,
					    &error)) == NULL) {
		mce_log(LL_CRIT,
			"Failed to open connection to message bus; %s",
			error.message);
		dbus_error_free(&error);
		status = EXIT_FAILURE;
		goto EXIT;
	}

	if (hour == REMOVE_TIME) {
		mce_log(LL_DEBUG, "Sending set_alarm request");
		if (dbus_send(ALARMDAEMON_DBUS_SERVICE, ALARMDAEMON_DBUS_PATH,
			      ALARMDAEMON_DBUS_IFACE,
			      SET_ALARM_REQ, TRUE,
			      DBUS_TYPE_UINT32, &hour,
			      DBUS_TYPE_UINT32, &minute,
			      DBUS_TYPE_INVALID) == FALSE) {
			status = EXIT_FAILURE;
			goto EXIT;
		}
	} else {
		mce_log(LL_DEBUG, "Sending remove_alarm request");
		if (dbus_send(ALARMDAEMON_DBUS_SERVICE, ALARMDAEMON_DBUS_PATH,
			      ALARMDAEMON_DBUS_IFACE,
			      REMOVE_ALARM_REQ, TRUE,
			      DBUS_TYPE_INVALID) == FALSE) {
			status = EXIT_FAILURE;
			goto EXIT;
		}
	}

EXIT:
	/* If there is an established D-Bus connection, unreference it */
	if (dbus_connection != NULL) {
		mce_log(LL_DEBUG, "Unreferencing temporary D-Bus connection");
		dbus_connection_unref(dbus_connection);
		dbus_connection = NULL;
	}

	return status;
}

/**
 * D-Bus message handler
 *
 * @todo Error message should be sent on failure instead of
 *       returning DBUS_HANDLER_RESULT_NOT_YET_HANDLED
 *
 * @param connection Unused
 * @param msg The D-Bus message received
 * @param user_data Unused
 * @return DBUS_HANDLER_RESULT_HANDLED if message was handled
 *         DBUS_HANDLER_RESULT_NOT_HANDLED if message was not for us or
 *         if some error occured
 */
static DBusHandlerResult msg_handler(DBusConnection *const connection,
				     DBusMessage *const msg,
				     gpointer const user_data)
{
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);

	(void)connection;
	(void)user_data;

/* Callbacks from SystemUI */

	/* Powerkey menu callback */
	if (dbus_message_is_method_call(msg, MCETOOL_REQUEST_IF,
					MCETOOL_POWERKEY_CB_REQ) == TRUE) {
		gint32 result = INT_MAX;

		mce_log(LL_DEBUG, "Received powerkey callback");

		dbus_message_get_args(msg, NULL, DBUS_TYPE_INT32, &result, DBUS_TYPE_INVALID);

		/* Close the powerkey menu */
		powerkey_menu(FALSE, FALSE);

		fprintf(stdout, "powerkeymenu return value: ");

		switch (result) {
		case POWER_KEY_MENU_RESPONSE_TKLOCK:
			fprintf(stdout, "Lock touch screen and keys");
			break;

		case POWER_KEY_MENU_RESPONSE_DEVICELOCK:
			fprintf(stdout, "Lock device");
			break;

		case POWER_KEY_MENU_RESPONSE_NORMALMODE:
			fprintf(stdout, "Normal mode");
			break;

		case POWER_KEY_MENU_RESPONSE_FLIGHTMODE:
			fprintf(stdout, "Offline mode");
			break;

#ifdef POWER_KEY_MENU_RESPONSE_REBOOT
		case POWER_KEY_MENU_RESPONSE_REBOOT:
			fprintf(stdout, "Reboot");
			break;
#endif /* POWER_KEY_MENU_RESPONSE_REBOOT */

		case POWER_KEY_MENU_RESPONSE_POWEROFF:
			fprintf(stdout, "Switch off!");
			break;

#ifdef POWER_KEY_MENU_RESPONSE_CANCEL
		case POWER_KEY_MENU_RESPONSE_CANCEL:
#else
		case -6:
#endif /* POWER_KEY_MENU_RESPONSE_CANCEL */
			fprintf(stdout, "Cancel");
			break;

		default:
			fprintf(stdout, "Unknown return value %d", result);
			break;
		}

		fprintf(stdout, "\n\n");
	/* Device lock menu callback */
	} else if (dbus_message_is_method_call(msg, MCETOOL_REQUEST_IF,
					       MCETOOL_DEVLOCK_CB_REQ) == TRUE) {
		gint32 result = INT_MAX;

		mce_log(LL_DEBUG, "Received devlock callback");

		dbus_message_get_args(msg, NULL, DBUS_TYPE_INT32, &result, DBUS_TYPE_INVALID);

		/* Close the device lock menu */
		devicelock_dialog(0, FALSE);

		fprintf(stdout, "devlock return value: ");

		switch (result) {
		case DEVLOCK_RESPONSE_SHUTDOWN:
			fprintf(stdout, "Shutdown requested");
			break;

		case DEVLOCK_RESPONSE_NOSHUTDOWN:
			fprintf(stdout, "Shutdown cancelled");
			break;

		case DEVLOCK_RESPONSE_CORRECT:
			fprintf(stdout, "Correct code");
			break;

		case DEVLOCK_RESPONSE_INCORRECT:
			fprintf(stdout, "Incorrect code");
			break;

		case DEVLOCK_RESPONSE_CANCEL:
			fprintf(stdout, "Cancel");
			break;

		default:
			fprintf(stdout, "Unknown return value %d", result);
			break;
		}

		fprintf(stdout, "\n\n");
	/* Touchscreen/keypad lock menu callback */
	} else if (dbus_message_is_method_call(msg, MCETOOL_REQUEST_IF,
					       MCETOOL_TKLOCK_CB_REQ) == TRUE) {
		gint32 result = INT_MAX;

		mce_log(LL_DEBUG, "Received tklock callback");

		dbus_message_get_args(msg, NULL, DBUS_TYPE_INT32, &result, DBUS_TYPE_INVALID);

		/* Close the tklock menu */
		tklock_dialog(0, TRUE, FALSE);

		fprintf(stdout, "tklock return value: ");

		switch (result) {
		case TKLOCK_UNLOCK:
			fprintf(stdout, "TKlock unlocked");
			break;

		case TKLOCK_RETRY:
			fprintf(stdout, "Incorrect combination");
			break;

		case TKLOCK_TIMEOUT:
			fprintf(stdout, "Timeout");
			break;

		default:
			fprintf(stdout, "Unknown return value %d", result);
			break;
		}

		fprintf(stdout, "\n\n");
	/* Modechange callback */
	} else if (dbus_message_is_method_call(msg, MCETOOL_REQUEST_IF,
					       MCETOOL_MODECHG_CB_REQ) == TRUE) {
		dbus_int32_t result = INT_MAX;

		mce_log(LL_DEBUG, "Received modechange callback");

		dbus_message_get_args(msg, NULL, DBUS_TYPE_INT32, &result, DBUS_TYPE_INVALID);

		fprintf(stdout,
			"modechange return value: %s\n\n",
			result == MODECHANGE_RESPONSE_OK ? "OK" : "Cancel");
	/* Alarm dialog callback */
	} else if (dbus_message_is_method_call(msg, MCETOOL_REQUEST_IF,
					       MCETOOL_ALARM_CB_REQ) == TRUE) {
		gint32 result = INT_MAX;

		mce_log(LL_DEBUG, "Received alarm callback");

		dbus_message_get_args(msg, NULL, DBUS_TYPE_INT32, &result, DBUS_TYPE_INVALID);

		/* Close the alarm menu */
		alarm_dialog(0, FALSE, NULL, NULL, NULL);

		fprintf(stdout, "alarm return value: ");

		switch (result) {
		case ALARM_DIALOG_RESPONSE_SNOOZE:
			fprintf(stdout, "Alarm snoozed");
			break;

		case ALARM_DIALOG_RESPONSE_DISMISS:
			fprintf(stdout, "Alarm dismissed");
			break;

		case ALARM_DIALOG_RESPONSE_POWERUP:
			fprintf(stdout, "Powerup requested");
			break;

		case ALARM_DIALOG_RESPONSE_POWERDOWN:
			fprintf(stdout, "Powerdown requested");
			break;

		default:
			fprintf(stdout, "Unknown return value %d", result);
			break;
		}

		fprintf(stdout, "\n\n");
	} else {
		/* Nothing was meant for us */
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		if (dbus_send_message(reply) == FALSE)
			mce_log(LL_CRIT, "Cannot send reply");
	}

	/* We got our callback, so we're done */
	g_main_loop_quit(mainloop);

	return DBUS_HANDLER_RESULT_HANDLED;
}

/**
 * Acquire D-Bus services
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean dbus_acquire_services(void)
{
	gboolean status = FALSE;
	int ret;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	ret = dbus_bus_request_name(dbus_connection, MCETOOL_SERVICE, 0,
				    &error);

	if (ret == DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
		mce_log(LL_DEBUG,
			"Service %s acquired",
			MCETOOL_SERVICE);
	} else {
		mce_log(LL_CRIT,
			"Cannot acquire service: %s",
			error.message);
		dbus_error_free(&error);
		goto EXIT;
	}

	status = TRUE;

EXIT:
	return status;
}

/**
 * Initialise the message handler(s) used by mcetool
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean dbus_init_message_handlers(void)
{
	gboolean status = FALSE;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	/* Listen to all method calls meant for us */
	dbus_bus_add_match(dbus_connection,
			   "type='method_call',"
			   "interface='" MCETOOL_REQUEST_IF "'",
			   &error);

	if (dbus_error_is_set(&error) == TRUE) {
		mce_log(LL_CRIT,
			"Failed to add D-Bus match for '" MCETOOL_SERVICE "'"
			"; %s",
			error.message);
		dbus_error_free(&error);
		goto EXIT;
	}

	if (dbus_connection_add_filter(dbus_connection, msg_handler,
				       NULL, NULL) == FALSE) {
		mce_log(LL_CRIT, "Failed to add D-Bus filter");
		goto EXIT;
	}

	status = TRUE;

EXIT:
	return status;
}

/**
 * D-Bus initialisation
 *
 * @param bus_type "system" or "session" bus
 * @return 0 on success, non-zero on failure
 */
static gint mcetool_dbus_init(DBusBusType bus_type)
{
	gint status = 0;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	mce_log(LL_DEBUG, "Establishing D-Bus connection");

	/* Establish D-Bus connection */
	if ((dbus_connection = dbus_bus_get(bus_type, &error)) == 0) {
		mce_log(LL_CRIT,
			"Failed to open connection to message bus; %s",
			error.message);
		dbus_error_free(&error);
		status = EXIT_FAILURE;
		goto EXIT;
	}

	mce_log(LL_DEBUG, "Connecting D-Bus to the mainloop");

	/* Connect D-Bus to the mainloop */
	dbus_connection_setup_with_g_main(dbus_connection, NULL);

	mce_log(LL_DEBUG, "Acquiring D-Bus service");

	/* Acquire D-Bus service */
	if (dbus_acquire_services() == FALSE) {
		status = EXIT_FAILURE;
		goto EXIT;
	}

	/* Initialise message handlers */
	if (dbus_init_message_handlers() == FALSE) {
		status = EXIT_FAILURE;
		goto EXIT;
	}

EXIT:
	return status;
}

/**
 * D-Bus exit
 */
static void mcetool_dbus_exit(void)
{
	/* If there is an established D-Bus connection, unreference it */
	if (dbus_connection != NULL) {
		mce_log(LL_DEBUG, "Unreferencing D-Bus connection");
		dbus_connection_unref(dbus_connection);
		dbus_connection = NULL;
	}
}

/**
 * Enable/disable the acting dead UI
 *
 * @param enable TRUE to enable acting dead UI, FALSE to disable acting dead UI
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_actdead(const gboolean enable)
{
	/* com.nokia.system_ui.request.acting_dead_{open,close} */
	return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
			 SYSTEMUI_REQUEST_IF,
			 enable ? SYSTEMUI_ACTINGDEAD_OPEN_REQ :
				  SYSTEMUI_ACTINGDEAD_CLOSE_REQ, TRUE,
			 DBUS_TYPE_INVALID);
}

/**
 * Open/close the powerkey menu
 *
 * @param normal TRUE to open in NORMAL mode, FALSE to open in FLIGHT mode
 * @param open TRUE to open the powerkey menu, FALSE to close it
 * @return TRUE on success, FALSE on failure
 */
static gboolean powerkey_menu(const gboolean normal, const gboolean open)
{
	const gchar *const cb_service = MCETOOL_SERVICE;
	const gchar *const cb_path = MCETOOL_REQUEST_PATH;
	const gchar *const cb_interface = MCETOOL_REQUEST_IF;
	const gchar *const cb_method = MCETOOL_POWERKEY_CB_REQ;
	const dbus_uint32_t mode = (normal == FALSE) ? MODE_FLIGHT :
						       MODE_NORMAL;

	if (open == TRUE) {
		return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
				 SYSTEMUI_REQUEST_IF,
				 SYSTEMUI_POWERKEYMENU_OPEN_REQ, TRUE,
				 DBUS_TYPE_STRING, &cb_service,
				 DBUS_TYPE_STRING, &cb_path,
				 DBUS_TYPE_STRING, &cb_interface,
				 DBUS_TYPE_STRING, &cb_method,
				 DBUS_TYPE_UINT32, &mode,
				 DBUS_TYPE_INVALID);
	} else {
		return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
				 SYSTEMUI_REQUEST_IF,
				 SYSTEMUI_POWERKEYMENU_CLOSE_REQ, TRUE,
				 DBUS_TYPE_INVALID);
	}
}

/**
 * Show/hide the modetransition confirmation dialog
 *
 * @param mode The current mode
 * @param open TRUE to open the dialog, FALSE to close the dialog
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_modechange(const dbus_uint32_t mode, const gboolean open)
{
	const gchar *const cb_service = MCETOOL_SERVICE;
	const gchar *const cb_path = MCETOOL_REQUEST_PATH;
	const gchar *const cb_interface = MCETOOL_REQUEST_IF;
	const gchar *const cb_method = MCETOOL_MODECHG_CB_REQ;

	/* com.nokia.system_ui.request.modechange_{open,close} */
	if (open == TRUE) {
		return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
				 SYSTEMUI_REQUEST_IF,
				 SYSTEMUI_MODECHANGE_OPEN_REQ, TRUE,
				 DBUS_TYPE_STRING, &cb_service,
				 DBUS_TYPE_STRING, &cb_path,
				 DBUS_TYPE_STRING, &cb_interface,
				 DBUS_TYPE_STRING, &cb_method,
				 DBUS_TYPE_UINT32, &mode,
				 DBUS_TYPE_INVALID);
	} else {
		return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
				 SYSTEMUI_REQUEST_IF,
				 SYSTEMUI_MODECHANGE_CLOSE_REQ, TRUE,
				 DBUS_TYPE_INVALID);
	}
}

/**
 * Enable/disable the devicelock
 *
 * @param mode The mode to open in; valid modes:
 *             DEVLOCK_QUERY_ENABLE (open device lock dialog in normal mode)
 *             DEVLOCK_QUERY_OPEN (open device lock dialog in no input mode)
 *             DEVLOCK_QUERY_NOTE (open device lock confirm cancel dialog)
 * @param open TRUE to open the dialog, FALSE to close the dialog
 * @return TRUE on success, FALSE on failure
 */
static gboolean devicelock_dialog(const dbus_uint32_t mode,
				  const gboolean open)
{
	const gchar *const cb_service = MCETOOL_SERVICE;
	const gchar *const cb_path = MCETOOL_REQUEST_PATH;
	const gchar *const cb_interface = MCETOOL_REQUEST_IF;
	const gchar *const cb_method = MCETOOL_DEVLOCK_CB_REQ;

	/* com.nokia.system_ui.request.devlock_{open,close} */
	if (open == TRUE) {
		return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
				 SYSTEMUI_REQUEST_IF,
				 SYSTEMUI_DEVLOCK_OPEN_REQ, TRUE,
				 DBUS_TYPE_STRING, &cb_service,
				 DBUS_TYPE_STRING, &cb_path,
				 DBUS_TYPE_STRING, &cb_interface,
				 DBUS_TYPE_STRING, &cb_method,
				 DBUS_TYPE_UINT32, &mode,
				 DBUS_TYPE_INVALID);
	} else {
		return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
				 SYSTEMUI_REQUEST_IF,
				 SYSTEMUI_DEVLOCK_CLOSE_REQ, TRUE,
				 DBUS_TYPE_INVALID);
	}
}

/**
 * Show/hide the tklock dialog
 *
 * @param mode The mode to open in; valid modes:
 *             TKLOCK_ENABLE (open the tklock in normal mode)
 *             TKLOCK_HELP (show the tklock help infoprint)
 *             TKLOCK_SELECT (show the press select infoprint)
 *             TKLOCK_ONEINPUT (open the tklock in event eater mode)
 * @param silent TRUE to disable infoprints, FALSE to enable infoprints
 * @param open TRUE to open the dialog, FALSE to close the dialog
 * @return TRUE on success, FALSE on FAILURE
 */
static gboolean tklock_dialog(const dbus_uint32_t mode,
			      const dbus_bool_t silent,
			      const gboolean open)
{
	const gchar *const cb_service = MCETOOL_SERVICE;
	const gchar *const cb_path = MCETOOL_REQUEST_PATH;
	const gchar *const cb_interface = MCETOOL_REQUEST_IF;
	const gchar *const cb_method = MCETOOL_TKLOCK_CB_REQ;

	/* com.nokia.system_ui.request.tklock_{open,close} */
	if (open == TRUE) {
		return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
				 SYSTEMUI_REQUEST_IF,
				 SYSTEMUI_TKLOCK_OPEN_REQ, TRUE,
				 DBUS_TYPE_STRING, &cb_service,
				 DBUS_TYPE_STRING, &cb_path,
				 DBUS_TYPE_STRING, &cb_interface,
				 DBUS_TYPE_STRING, &cb_method,
				 DBUS_TYPE_UINT32, &mode,
				 DBUS_TYPE_BOOLEAN, &silent,
				 DBUS_TYPE_INVALID);
	} else {
		return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
				 SYSTEMUI_REQUEST_IF,
				 SYSTEMUI_TKLOCK_CLOSE_REQ, TRUE,
				 DBUS_TYPE_BOOLEAN, &silent,
				 DBUS_TYPE_INVALID);
	}
}

/**
 * Enable/disable the tklock
 *
 * @param mode The mode to change to; valid modes:
 *             "locked", "locked-dim", "silent-locked", "silent-locked-dim",
 *             "unlocked", "silent-unlocked"
 * @return TRUE on success, FALSE on FAILURE
 */
static gboolean set_tklock_mode(gchar **mode)
{
	/* com.nokia.mce.request.req_tklock_mode_change */
	return mcetool_dbus_call(MCE_TKLOCK_MODE_CHANGE_REQ, mode, TRUE);
}

/**
 * Enable/disable the alarm dialog
 *
 * @param mode The mode to open in; valid modes:
 *             ALARM_MODE_NORMAL (open alarm dialog in normal mode)
 *             ALARM_MODE_NOSNOOZE (open alarm dialog in no snooze mode)
 *             ALARM_MODE_SWITCHON (open alarm dialog in switch on mode)
 * @param open TRUE to open the dialog, FALSE to close the dialog
 * @param alarmtext Message to show in the alarm dialog
 * @param alarmicon Icon to use in the alarm dialog
 * @return TRUE on success, FALSE on failure
 */
static gboolean alarm_dialog(const dbus_uint32_t mode,
			     const gboolean open,
			     const gchar *const alarmtext,
			     const gchar *const alarmsound,
			     const gchar *const alarmicon)
{
	const gchar *const cb_service = MCETOOL_SERVICE;
	const gchar *const cb_path = MCETOOL_REQUEST_PATH;
	const gchar *const cb_interface = MCETOOL_REQUEST_IF;
	const gchar *const cb_method = MCETOOL_ALARM_CB_REQ;

	/* com.nokia.system_ui.request.alarm_{open,close} */
	if (open == TRUE) {
		return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
				 SYSTEMUI_REQUEST_IF,
				 SYSTEMUI_ALARM_OPEN_REQ, TRUE,
				 DBUS_TYPE_STRING, &cb_service,
				 DBUS_TYPE_STRING, &cb_path,
				 DBUS_TYPE_STRING, &cb_interface,
				 DBUS_TYPE_STRING, &cb_method,
				 DBUS_TYPE_UINT32, &mode,
				 DBUS_TYPE_STRING, &alarmtext,
				 DBUS_TYPE_STRING, &alarmsound,
				 alarmicon ? DBUS_TYPE_STRING :
					     DBUS_TYPE_INVALID,
				 alarmicon ? &alarmicon :
					     DBUS_TYPE_INVALID,
				 DBUS_TYPE_INVALID);
	} else {
		return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
				 SYSTEMUI_REQUEST_IF,
				 SYSTEMUI_ALARM_CLOSE_REQ, TRUE,
				 DBUS_TYPE_INVALID);
	}
}

/**
 * Trigger a powerkey event
 *
 * @param event The type of event to trigger; valid types:
 *             "short", "long"
 * @return TRUE on success, FALSE on FAILURE
 */
static gboolean trigger_powerkey_event(const gint type)
{
	gint booltype = (type == LONG_EVENT) ? TRUE : FALSE;

	/* com.nokia.mce.request.req_trigger_powerkey_event */
	return dbus_send(MCE_SERVICE, MCE_REQUEST_PATH, MCE_REQUEST_IF,
			 MCE_TRIGGER_POWERKEY_EVENT_REQ, TRUE,
			 DBUS_TYPE_BOOLEAN, &booltype,
			 DBUS_TYPE_INVALID);
}

/**
 * Enable/Disable the LED
 *
 * @param enable TRUE to enable the LED, FALSE to disable the LED
 * @return TRUE on success, FALSE on FAILURE
 */
static gboolean set_led_state(const gboolean enable)
{
	/* com.nokia.mce.request.{req_led_enable,req_led_disable} */
	return dbus_send(MCE_SERVICE, MCE_REQUEST_PATH, MCE_REQUEST_IF,
			 enable ? MCE_ENABLE_LED : MCE_DISABLE_LED, TRUE,
			 DBUS_TYPE_INVALID);
}

/**
 * Activate/Deactivate a LED pattern
 *
 * @param pattern The name of the pattern to activate/deactivate
 * @param activate TRUE to activate pattern, FALSE to deactivate pattern
 * @return TRUE on success, FALSE on FAILURE
 */
static gboolean set_led_pattern_state(const gchar *const pattern,
				      const gboolean activate)
{
	/* com.nokia.mce.request.{req_led_enable,req_led_disable} */
	return dbus_send(MCE_SERVICE, MCE_REQUEST_PATH, MCE_REQUEST_IF,
			 activate ? MCE_ACTIVATE_LED_PATTERN :
				    MCE_DEACTIVATE_LED_PATTERN, TRUE,
			 DBUS_TYPE_STRING, &pattern,
			 DBUS_TYPE_INVALID);
}

/**
 * Init function for the mcetool GConf handling
 *
 * @return TRUE on success, FALSE on failure
 */
static gint mcetool_gconf_init(void)
{
	gint status = 0;

	/* Init GType */
	g_type_init();

	mce_log(LL_DEBUG, "Setting up GConf client");

	gconf_client = gconf_client_get_default();

	if (gconf_client == NULL) {
		mce_log(LL_CRIT, "Could not get default GConf client");
		status = EXIT_FAILURE;
		goto EXIT;
	}

EXIT:
	return status;
}

/**
 * Exit function for the mcetool GConf handling
 */
static void mcetool_gconf_exit(void)
{
	mce_log(LL_DEBUG, "Unreferencing GConf client");

	/* Unreference GConf */
	if (gconf_client != NULL)
		g_object_unref(gconf_client);
}

/**
 * Return a boolean from the specified GConf key
 *
 * @param key The GConf key to get the value from
 * @param value Will contain the value on return
 * @return TRUE on success, FALSE on failure
 */
static gboolean mcetool_gconf_get_bool(const gchar *const key, gboolean *value)
{
	gboolean status = FALSE;
	GError *error = NULL;
	GConfValue *gcv;

	gcv = gconf_client_get(gconf_client, key, &error);

	if (gcv == NULL) {
		mce_log(LL_WARN,
			"Could not retrieve %s from GConf; %s",
			key, (error != NULL) ? error->message : "Key not set");
		goto EXIT;
	}

	if (gcv->type != GCONF_VALUE_BOOL) {
		mce_log(LL_ERR,
			"GConf key %s should have type: %d, but has type: %d",
			key, GCONF_VALUE_BOOL, gcv->type);
		goto EXIT;
	}

	*value = gconf_value_get_bool(gcv);

	status = TRUE;

EXIT:
	return status;
}

/**
 * Return an integer from the specified GConf key
 *
 * @param key The GConf key to get the value from
 * @param value Will contain the value on return
 * @return TRUE on success, FALSE on failure
 */
static gboolean mcetool_gconf_get_int(const gchar *const key, gint *value)
{
	gboolean status = FALSE;
	GError *error = NULL;
	GConfValue *gcv;

	gcv = gconf_client_get(gconf_client, key, &error);

	if (gcv == NULL) {
		mce_log(LL_WARN,
			"Could not retrieve %s from GConf; %s",
			key, (error != NULL) ? error->message : "Key not set");
		goto EXIT;
	}

	if (gcv->type != GCONF_VALUE_INT) {
		mce_log(LL_ERR,
			"GConf key %s should have type: %d, but has type: %d",
			key, GCONF_VALUE_INT, gcv->type);
		goto EXIT;
	}

	*value = gconf_value_get_int(gcv);

	status = TRUE;

EXIT:
	g_clear_error(&error);

	return status;
}

/**
 * Reset the device lock code
 *
 * @return 0 on success, EXIT_FAILURE on failure
 */
static gint mcetool_reset_passwd(void)
{
	gint status = EXIT_FAILURE;
	struct cal *cal_data;
	char new_entry;

	if (cal_init(&cal_data) < 0) {
		mce_log(LL_ERR, "Failed to init CAL");
		goto EXIT;
	}

	new_entry = 0;

	if (cal_write_block(cal_data, "lock_code",
			    &new_entry, 0, CAL_FLAG_USER) < 0) {
		mce_log(LL_ERR, "Failed to write to CAL");
		cal_finish(cal_data);
		goto EXIT;
	}

	cal_finish(cal_data);

	if (gconf_client_set_bool(gconf_client,
				  SYSTEMUI_GCONF_DEVICE_AUTOLOCK_ENABLED_PATH,
				  FALSE, NULL) == FALSE) {
		mce_log(LL_WARN, "Failed to write %s to GConf",
			SYSTEMUI_GCONF_DEVICE_AUTOLOCK_ENABLED_PATH);
		goto EXIT;
	}

	status = 0;

EXIT:
	return status;
}

/**
 * Print mode related information
 *
 * @return 0 on success, EXIT_FAILURE on failure
 */
static gint mcetool_get_status(void)
{
	gint status = 0;
	gchar *mode = NULL;
	gchar *tklock = NULL;
	gchar *devlock = NULL;
	gchar *display = NULL;
	gchar *mce_version = NULL;
	gint brightness = DEFAULT_DISP_BRIGHTNESS;
	gint dim_timeout = DEFAULT_DIM_TIMEOUT;
	gint blank_timeout = DEFAULT_BLANK_TIMEOUT;
	gboolean inactive;
	gboolean tklock_autolock = DEFAULT_TK_AUTOLOCK;
	gboolean devlock_autolock = DEFAULT_DEVICE_AUTOLOCK_ENABLED;
	gint devlock_autolock_timeout = DEFAULT_DEVICE_AUTOLOCK_TIMEOUT;
	gint devlock_failed = DEFAULT_DEVICE_LOCK_FAILED;
	gint devlock_total_failed = DEFAULT_DEVICE_LOCK_TOTAL_FAILED;

	/* Get the mode */
	status = mcetool_dbus_call(MCE_DEVICE_MODE_GET, &mode, FALSE);

	if (status != 0)
		goto EXIT;

	fprintf(stdout,
		_("\n"
		  "MCE status:\n"
		  "-----------\n"));

	/* Get the version; just ignore if no reply */
	mcetool_dbus_call(MCE_VERSION_GET, &mce_version, FALSE);

	fprintf(stdout,
		" %-40s %s\n",
		_("MCE version:"),
		(mce_version == NULL) ? "unknown" : mce_version);

	fprintf(stdout,
		" %-40s %s\n",
		_("Mode:"),
		  mode);
	free(mode);

	/* Display brightness */
	mcetool_gconf_get_int(MCE_GCONF_DISPLAY_BRIGHTNESS_PATH,
			      &brightness);

	fprintf(stdout,
		" %-40s %d %s\n",
		_("Brightness:"),
		brightness,
		_("(1-9)"));

	mcetool_gconf_get_int(MCE_GCONF_DISPLAY_DIM_TIMEOUT_PATH,
			      &dim_timeout);

	fprintf(stdout,
		" %-40s %d %s\n",
		_("Dim timeout:"),
		dim_timeout,
		_("seconds"));

	mcetool_gconf_get_int(MCE_GCONF_DISPLAY_BLANK_TIMEOUT_PATH,
			      &blank_timeout);

	fprintf(stdout,
		" %-40s %d %s\n",
		_("Blank timeout:"),
		blank_timeout,
		_("seconds"));

	/* Get inactivity status */
	status = get_inactivity_status(&inactive);

	if (status != 0)
		goto EXIT;

	fprintf(stdout,
		" %-40s %s\n",
		_("Inactivity status:"),
		inactive ? "Inactive" : "Active");

	/* Get display state */
	status = mcetool_dbus_call(MCE_DISPLAY_STATUS_GET, &display, FALSE);

	if (status != 0)
		goto EXIT;

	fprintf(stdout,
		" %-40s %s\n",
		_("Display state:"),
		display);

	/* Get touchscreen/keypad lock mode */
	status = mcetool_dbus_call(MCE_TKLOCK_MODE_GET, &tklock, FALSE);

	if (status != 0)
		goto EXIT;

	fprintf(stdout,
		" %-40s %s\n",
		_("Touchscreen/Keypad lock:"),
		tklock);

	mcetool_gconf_get_bool(MCE_GCONF_TK_AUTOLOCK_ENABLED_PATH,
			       &tklock_autolock);

	fprintf(stdout,
		" %-40s %s\n",
		_("Touchscreen/Keypad autolock:"),
		tklock_autolock ? _("enabled") : _("disabled"));

	mcetool_gconf_get_bool(MCE_GCONF_DEVICE_AUTOLOCK_ENABLED_PATH,
			       &devlock_autolock);

	/* Get device lock mode */
	status = mcetool_dbus_call(MCE_DEVLOCK_MODE_GET, &devlock, FALSE);

	if (status != 0)
		goto EXIT;

	fprintf(stdout,
		" %-40s %s\n",
		_("Device lock:"),
		devlock);
	free(devlock);

	fprintf(stdout,
		" %-40s %s\n",
		_("Device autolock:"),
		devlock_autolock ? _("enabled") : _("disabled"));

	mcetool_gconf_get_int(MCE_GCONF_DEVICE_AUTOLOCK_TIMEOUT_PATH,
			      &devlock_autolock_timeout);

	fprintf(stdout,
		" %-40s %d %s\n",
		_("Device autolock timeout:"),
		devlock_autolock_timeout,
		_("minutes"));

	mcetool_gconf_get_int(MCE_GCONF_DEVICE_LOCK_FAILED_PATH,
			      &devlock_failed);

	fprintf(stdout,
		" %-40s %d\n",
		_("Device unlocking failures (current):"),
		devlock_failed);

	mcetool_gconf_get_int(MCE_GCONF_DEVICE_LOCK_TOTAL_FAILED_PATH,
			      &devlock_total_failed);

	fprintf(stdout,
		" %-40s %d\n",
		_("Device unlocking failures (total):"),
		devlock_total_failed);

EXIT:
	fprintf(stdout, "\n");

	return status;
}

/**
 * Main
 *
 * @param argc Number of command line arguments
 * @param argv Array with command line arguments
 * @return 0 on success, non-zero on failure
 */
int main(int argc, char **argv)
{
	int optc;
	int opt_index;

	int verbosity = LL_DEFAULT;
	int consolelog = MCE_LOG_SYSLOG;

	int status = 0;

	gint actdead = INVALID_ACTDEAD;
	gint splashscreen = INVALID_SPLASH;
	gint modechange = INVALID_MODECHANGE;
	gint powerkeymenu = INVALID_POWERKEYMENU;
	gint devlockdialog = INVALID_DEVICELOCK;
	gint tklockdialog = INVALID_TKLOCK;
	gint alarmdialog = INVALID_ALARM;
	gint powerkeyevent = INVALID_EVENT;
	guint alarmhour = INVALID_TIME;
	guint alarmminute = INVALID_TIME;
	gchar *alarmtext = NULL;
	gchar *alarmsound = NULL;
	gchar *alarmicon = NULL;
	gchar *newmode = NULL;
	gchar *tklockmode = NULL;
	gchar *ledpattern = NULL;
	gint led_enable = -1;
	gboolean pattern_activate = TRUE;
	gboolean get_mce_status = TRUE;
	gboolean block = FALSE;
	gboolean dont_block = FALSE;
	gboolean tksilent = FALSE;
	gboolean reset_password = FALSE;
	gboolean send_prevent = FALSE;
	gboolean send_unblank = FALSE;
	gboolean send_dim = FALSE;
	gboolean send_blank = FALSE;
	gboolean send_powerup = FALSE;
	gboolean send_reboot = FALSE;
	gboolean send_shutdown = FALSE;

	DBusBusType bus_type = DBUS_BUS_SYSTEM;

	const char optline[] = "S";

	struct option const options[] = {
		{ "reset-passwd", no_argument, 0, 'R' },
		{ "blank-prevent", no_argument, 0, 'P' },
		{ "unblank-screen", no_argument, 0, 'U' },
		{ "dim-screen", no_argument, 0, 'd' },
		{ "blank-screen", no_argument, 0, 'B' },
		{ "powerup", no_argument, 0, 'p' },
		{ "reboot", no_argument, 0, 'r' },
		{ "shutdown", no_argument, 0, 'H' },
		{ "set-mode", required_argument, 0, 's' },
		{ "actdead", required_argument, 0, 'A' },
		{ "splashscreen", required_argument, 0, 'a' },
		{ "modechange", required_argument, 0, 'M' },
		{ "powerkeymenu", required_argument, 0, 'W' },
		{ "devicelock", required_argument, 0, 'D' },
		{ "tklock", required_argument, 0, 'K' },
		{ "set-tklock-mode", required_argument, 0, 'k' },
		{ "alarm", required_argument, 0, 'C' },
		{ "alarm-icon", required_argument, 0, 'I' },
		{ "alarm-sound", required_argument, 0, 'u' },
		{ "alarm-text", required_argument, 0, 't' },
		{ "set-alarm", required_argument, 0, 'T' },
		{ "disable-alarm", no_argument, 0, 'E' },
		{ "enable-led", no_argument, 0, 'l' },
		{ "disable-led", no_argument, 0, 'L' },
		{ "activate-led-pattern", required_argument, 0, 'y' },
		{ "deactivate-led-pattern", required_argument, 0, 'Y' },
		{ "powerkey-event", required_argument, 0, 'e' },
		{ "dont-block", no_argument, 0, 'b' },
		{ "no-status", no_argument, 0, 'N' },
		{ "session", no_argument, 0, 'S' },
		{ "quiet", no_argument, 0, 'q' },
		{ "verbose", no_argument, 0, 'v' },
		{ "help", no_argument, 0, 'h' },
		{ "version", no_argument, 0, 'V' },
		{ 0, 0, 0, 0 }
	};

	/* Initialise support for locales, and set the program-name */
	if (init_locales(PRG_NAME) != 0)
		goto EXIT;

	/* Parse the command-line options */
	while ((optc = getopt_long(argc, argv, optline,
				   options, &opt_index)) != -1) {
		switch (optc) {
		case 'R':
			reset_password = TRUE;
			break;

		case 'P':
			send_prevent = TRUE;
			break;

		case 'U':
			send_unblank = TRUE;
			break;

		case 'd':
			send_dim = TRUE;
			break;

		case 'B':
			send_blank = TRUE;
			break;

		case 'p':
			send_powerup = TRUE;
			break;

		case 'r':
			send_reboot = TRUE;
			break;

		case 'H':
			send_shutdown = TRUE;
			break;

		case 's':
			newmode = strdup(optarg);
			break;

		case 'A':
			if (strcmp(optarg, OPEN_ACTDEAD_STR) == 0) {
				actdead = OPEN_ACTDEAD;
			} else if (strcmp(optarg, CLOSE_ACTDEAD_STR) == 0) {
				actdead = CLOSE_ACTDEAD;
			} else {
				usage();
				status = EINVAL;
				goto EXIT;
			}

			break;

		case 'a':
			if (strcmp(optarg, STARTUP_SPLASH_STR) == 0) {
				splashscreen = STARTUP_SPLASH;
			} else if (strcmp(optarg, SHUTDOWN_SPLASH_STR) == 0) {
				splashscreen = SHUTDOWN_SPLASH;
			} else if (strcmp(optarg,
					  SILENT_SHUTDOWN_SPLASH_STR) == 0) {
				splashscreen = SILENT_SHUTDOWN_SPLASH;
			} else if (strcmp(optarg, CLOSE_SPLASH_STR) == 0) {
				splashscreen = CLOSE_SPLASH;
			} else {
				usage();
				status = EINVAL;
				goto EXIT;
			}

			break;

		case 'M':
			if (strcmp(optarg, NORMAL_MODECHANGE_STR) == 0) {
				if (block == TRUE) {
					usage();
					status = EINVAL;
					goto EXIT;
				}

				block = TRUE;
				modechange = NORMAL_MODECHANGE;
			} else if (strcmp(optarg, FLIGHT_MODECHANGE_STR) == 0) {
				if (block == TRUE) {
					usage();
					status = EINVAL;
					goto EXIT;
				}

				block = TRUE;
				modechange = FLIGHT_MODECHANGE;
			} else if (strcmp(optarg, CLOSE_MODECHANGE_STR) == 0) {
				modechange = CLOSE_MODECHANGE;
			} else {
				usage();
				status = EINVAL;
				goto EXIT;
			}

			break;

		case 'W':
			if (strcmp(optarg, NORMAL_POWERKEYMENU_STR) == 0) {
				if (block == TRUE) {
					usage();
					status = EINVAL;
					goto EXIT;
				}

				block = TRUE;
				powerkeymenu = NORMAL_POWERKEYMENU;
			} else if (strcmp(optarg,
					  FLIGHT_POWERKEYMENU_STR) == 0) {
				if (block == TRUE) {
					usage();
					status = EINVAL;
					goto EXIT;
				}

				block = TRUE;
				powerkeymenu = FLIGHT_POWERKEYMENU;
			} else if (strcmp(optarg,
					  CLOSE_POWERKEYMENU_STR) == 0) {
				powerkeymenu = CLOSE_POWERKEYMENU;
			} else {
				usage();
				status = EINVAL;
				goto EXIT;
			}

			break;

		case 'D':
			if (strcmp(optarg, NORMAL_DEVICELOCK_STR) == 0) {
				if (block == TRUE) {
					usage();
					status = EINVAL;
					goto EXIT;
				}

				block = TRUE;
				devlockdialog = NORMAL_DEVICELOCK;
			} else if (strcmp(optarg,
					  NOINPUT_DEVICELOCK_STR) == 0) {
				if (block == TRUE) {
					usage();
					status = EINVAL;
					goto EXIT;
				}

				block = TRUE;
				devlockdialog = NOINPUT_DEVICELOCK;
			} else if (strcmp(optarg,
					  CONFIRM_DEVICELOCK_STR) == 0) {
				if (block == TRUE) {
					usage();
					status = EINVAL;
					goto EXIT;
				}

				block = TRUE;
				devlockdialog = CONFIRM_DEVICELOCK;
			} else if (strcmp(optarg,
					  CLOSE_DEVICELOCK_STR) == 0) {
				devlockdialog = CLOSE_DEVICELOCK;
			} else {
				usage();
				status = EINVAL;
				goto EXIT;
			}

			break;

		case 'K':
			if (strcmp(optarg, OPEN_TKLOCK_STR) == 0) {
				tklockdialog = OPEN_TKLOCK;
			} else if (strcmp(optarg, SOPEN_TKLOCK_STR) == 0) {
				tklockdialog = OPEN_TKLOCK;
				tksilent = TRUE;
			} else if (strcmp(optarg, HELP_TKLOCK_STR) == 0) {
				tklockdialog = HELP_TKLOCK;
			} else if (strcmp(optarg, SELECT_TKLOCK_STR) == 0) {
				if (block == TRUE) {
					usage();
					status = EINVAL;
					goto EXIT;
				}

				block = TRUE;
				tklockdialog = SELECT_TKLOCK;
			} else if (strcmp(optarg, ONEINPUT_TKLOCK_STR) == 0) {
				if (block == TRUE) {
					usage();
					status = EINVAL;
					goto EXIT;
				}

				block = TRUE;
				tklockdialog = ONEINPUT_TKLOCK;
			} else if (strcmp(optarg, CLOSE_TKLOCK_STR) == 0) {
				tklockdialog = CLOSE_TKLOCK;
			} else if (strcmp(optarg, SCLOSE_TKLOCK_STR) == 0) {
				tklockdialog = CLOSE_TKLOCK;
				tksilent = TRUE;
			} else {
				usage();
				status = EINVAL;
				goto EXIT;
			}

			break;

		case 'k':
			tklockmode = strdup(optarg);
			break;

		case 'C':
			if (strcmp(optarg, NORMAL_ALARM_STR) == 0) {
				if (block == TRUE) {
					usage();
					status = EINVAL;
					goto EXIT;
				}

				block = TRUE;
				alarmdialog = NORMAL_ALARM;
			} else if (strcmp(optarg,
					  NOSNOOZE_ALARM_STR) == 0) {
				if (block == TRUE) {
					usage();
					status = EINVAL;
					goto EXIT;
				}

				block = TRUE;
				alarmdialog = NOSNOOZE_ALARM;
			} else if (strcmp(optarg,
					  SWITCHON_ALARM_STR) == 0) {
				if (block == TRUE) {
					usage();
					status = EINVAL;
					goto EXIT;
				}

				block = TRUE;
				alarmdialog = SWITCHON_ALARM;
			} else if (strcmp(optarg,
					  CLOSE_ALARM_STR) == 0) {
				alarmdialog = CLOSE_ALARM;
			} else {
				usage();
				status = EINVAL;
				goto EXIT;
			}

			break;

		case 'I':
			alarmicon = strdup(optarg);
			break;

		case 'u':
			alarmsound = strdup(optarg);
			break;

		case 't':
			alarmtext = strdup(optarg);
			break;

		case 'T':
			if ((sscanf(optarg,
				    "%u:%u",
				    &alarmhour, &alarmminute) != 2) ||
			    (alarmhour > 23) || (alarmminute > 59)) {
				usage();
				status = EINVAL;
				goto EXIT;
			}

			break;

		case 'E':
			alarmhour = REMOVE_TIME;
			alarmminute = REMOVE_TIME;
			break;

		case 'l':
			if (led_enable != -1) {
				usage();
				status = EINVAL;
				goto EXIT;
			}

			led_enable = 1;
			break;

		case 'L':
			if (led_enable != -1) {
				usage();
				status = EINVAL;
				goto EXIT;
			}

			led_enable = 0;
			break;

		case 'y':
			if (ledpattern != NULL) {
				usage();
				status = EINVAL;
				goto EXIT;
			}

			ledpattern = strdup(optarg);
			pattern_activate = TRUE;
			break;

		case 'Y':
			if (ledpattern != NULL) {
				usage();
				status = EINVAL;
				goto EXIT;
			}

			ledpattern = strdup(optarg);
			pattern_activate = FALSE;
			break;

		case 'e':
			if (!strcmp(optarg, LONG_EVENT_STR)) {
				powerkeyevent = LONG_EVENT;
			} else if (!strcmp(optarg, SHORT_EVENT_STR)) {
				powerkeyevent = SHORT_EVENT;
			} else {
				usage();
				status = EINVAL;
				goto EXIT;
			}

			break;

		case 'b':
			dont_block = TRUE;
			break;

		case 'N':
			get_mce_status = FALSE;
			break;

		case 'S':
			bus_type = DBUS_BUS_SESSION;
			break;

		case 'q':
			if (verbosity > LL_CRIT)
				verbosity--;
			break;

		case 'v':
			if (verbosity < LL_DEBUG)
				verbosity++;
			break;

		case 'h':
			usage();
			goto EXIT;

		case 'V':
			version();
			goto EXIT;

		default:
			usage();
			status = EINVAL;
			goto EXIT;
		}
	}

	/* Any non-flag arguments is an error */
	if ((argc - optind) > 0) {
		usage();
		status = EINVAL;
		goto EXIT;
	}

	/* alarmtext and alarmsound must not be NULL */
	if (alarmtext == NULL)
		alarmtext = strdup("");

	if (alarmsound == NULL)
		alarmsound = strdup("");

	mce_log_open(PRG_NAME, LOG_USER,
		     (consolelog == TRUE) ? MCE_LOG_SYSLOG : MCE_LOG_STDERR);
	mce_log_set_verbosity(verbosity);

	/* Register a mainloop */
	mainloop = g_main_loop_new(NULL, FALSE);

	/* The alarm time is set using a D-Bus call that is always
	 * sent on the session bus, so do this before dbus_init
	 */
	if (alarmhour != INVALID_TIME)
		set_alarm(alarmhour, alarmminute);

	/* Initialise D-Bus */
	if ((status = mcetool_dbus_init(bus_type)) != 0)
		goto EXIT;

	/* Init GConf */
	if ((status = mcetool_gconf_init()) != 0)
		goto EXIT;

	if (reset_password == TRUE) {
		if ((status = mcetool_reset_passwd()) != 0)
			goto EXIT;

		fprintf(stdout, "Device lock password reset\n");
	}

	if (send_prevent == TRUE) {
		if ((status = mcetool_dbus_call(MCE_PREVENT_BLANK_REQ,
						NULL, TRUE)) != 0)
			goto EXIT;

		fprintf(stdout, "Blank prevent requested\n");
	}

	if (send_unblank == TRUE) {
		if ((status = mcetool_dbus_call(MCE_DISPLAY_ON_REQ,
						NULL, TRUE)) != 0)
			goto EXIT;

		fprintf(stdout, "Display on requested\n");
	}

	if (send_dim == TRUE) {
		if ((status = mcetool_dbus_call(MCE_DISPLAY_DIM_REQ,
						NULL, TRUE)) != 0)
			goto EXIT;

		fprintf(stdout, "Display dim requested\n");
	}

	if (send_blank == TRUE) {
		if ((status = mcetool_dbus_call(MCE_DISPLAY_OFF_REQ,
						NULL, TRUE)) != 0)
			goto EXIT;

		fprintf(stdout, "Display off requested\n");
	}

	if (send_powerup == TRUE) {
		if ((status = mcetool_dbus_call(MCE_POWERUP_REQ,
						NULL, TRUE)) != 0)
			goto EXIT;

		fprintf(stdout, "Powerup requested\n");
	}

	if (send_reboot == TRUE) {
		if ((status = mcetool_dbus_call(MCE_REBOOT_REQ,
						NULL, TRUE)) != 0)
			goto EXIT;

		fprintf(stdout, "Reboot requested\n");
	}

	if (send_shutdown == TRUE) {
		if ((status = mcetool_dbus_call(MCE_SHUTDOWN_REQ,
						NULL, TRUE)) != 0)
			goto EXIT;

		fprintf(stdout, "Shutdown requested\n");
	}

	switch (actdead) {
	case OPEN_ACTDEAD:
		mce_actdead(TRUE);
		break;

	case CLOSE_ACTDEAD:
		mce_actdead(FALSE);
		break;

	default:
		break;
	}

	switch (splashscreen) {
	case STARTUP_SPLASH:
		mce_powerup_splash(TRUE);
		break;

	case SHUTDOWN_SPLASH:
		mce_shutdown_splash(TRUE, TRUE);
		break;

	case SILENT_SHUTDOWN_SPLASH:
		mce_shutdown_splash(TRUE, FALSE);
		break;

	case CLOSE_SPLASH:
		mce_powerup_splash(FALSE);
		mce_shutdown_splash(FALSE, FALSE);
		break;

	default:
		break;
	}

	switch (modechange) {
	case NORMAL_MODECHANGE:
		mce_modechange(MODECHANGE_TO_FLIGHTMODE, TRUE);
		break;

	case FLIGHT_MODECHANGE:
		mce_modechange(MODECHANGE_TO_NORMALMODE, TRUE);
		break;

	case CLOSE_MODECHANGE:
		mce_modechange(0, FALSE);
		break;

	default:
		break;
	}

	switch (powerkeymenu) {
	case NORMAL_POWERKEYMENU:
		powerkey_menu(TRUE, TRUE);
		break;

	case FLIGHT_POWERKEYMENU:
		powerkey_menu(FALSE, TRUE);
		break;

	case CLOSE_POWERKEYMENU:
		powerkey_menu(FALSE, FALSE);
		break;

	default:
		break;
	}

	switch (devlockdialog) {
	case NORMAL_DEVICELOCK:
		devicelock_dialog(DEVLOCK_QUERY_ENABLE, TRUE);
		break;

	case NOINPUT_DEVICELOCK:
		devicelock_dialog(DEVLOCK_QUERY_OPEN, TRUE);
		break;

	case CONFIRM_DEVICELOCK:
		devicelock_dialog(DEVLOCK_QUERY_NOTE, TRUE);
		break;

	case CLOSE_DEVICELOCK:
		devicelock_dialog(0, FALSE);
		break;

	default:
		break;
	}

	switch (tklockdialog) {
	case OPEN_TKLOCK:
		tklock_dialog(TKLOCK_ENABLE, tksilent, TRUE);
		break;

	case HELP_TKLOCK:
		tklock_dialog(TKLOCK_HELP, tksilent, TRUE);
		break;

	case SELECT_TKLOCK:
		tklock_dialog(TKLOCK_SELECT, tksilent, TRUE);
		break;

	case ONEINPUT_TKLOCK:
		tklock_dialog(TKLOCK_ONEINPUT, tksilent, TRUE);
		break;

	case CLOSE_TKLOCK:
		tklock_dialog(0, tksilent, FALSE);
		break;

	default:
		break;
	}

	/* Change the tklock mode */
	if (tklockmode != NULL) {
		set_tklock_mode(&tklockmode);
	}

	switch (alarmdialog) {
	case NORMAL_ALARM:
		alarm_dialog(ALARM_MODE_NORMAL, TRUE,
			     alarmtext, alarmsound, alarmicon);
		break;

	case NOSNOOZE_ALARM:
		alarm_dialog(ALARM_MODE_NOSNOOZE, TRUE,
			     alarmtext, alarmsound, alarmicon);
		break;

	case SWITCHON_ALARM:
		alarm_dialog(ALARM_MODE_SWITCHON, TRUE,
			     alarmtext, alarmsound, NULL);
		break;

	case CLOSE_ALARM:
		alarm_dialog(0, FALSE, NULL, NULL, NULL);
		break;

	default:
		break;
	}

	if (powerkeyevent != INVALID_EVENT) {
		trigger_powerkey_event(powerkeyevent);
	}

	if (led_enable != -1) {
		set_led_state(led_enable);
	}

	if (ledpattern != NULL) {
		set_led_pattern_state(ledpattern, pattern_activate);
	}

	/* If --dont-block was specified, force-disable blocking */
	if (dont_block == TRUE) {
		block = FALSE;
	}

	if (block == TRUE) {
		/* Connect D-Bus to the mainloop */
		dbus_connection_setup_with_g_main(dbus_connection, NULL);

		/* Run the mainloop */
		g_main_loop_run(mainloop);

		/* If we get here, the main loop has terminated;
		 * either because we requested or because of an error
		 */
		g_main_loop_unref(mainloop);
	}

	if (newmode != NULL) {
		/* Change the mode */
		status = mcetool_dbus_call(MCE_DEVICE_MODE_CHANGE_REQ,
					   &newmode, TRUE);
		free(newmode);

		if (status != 0)
			goto EXIT;
	}

	if (get_mce_status == TRUE)
		mcetool_get_status();

EXIT:
	mcetool_gconf_exit();
	mcetool_dbus_exit();
	free(alarmicon);
	free(alarmtext);
	free(tklockmode);
	mce_log_close();

	return status;
}
