/**
 * @file tklock.c
 * This file implements the touchscreen/keypad lock component
 * of the Mode Control Entity
 * <p>
 * Copyright © 2004-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>
#include <glib/gstdio.h>		/* g_access */

#include <string.h>			/* strcmp() */
#include <unistd.h>			/* W_OK */
#include <linux/input.h>		/* struct input_event */

#include <mce/mode-names.h>
#include <systemui/dbus-names.h>
#include <systemui/tklock-dbus-names.h>

#include "mce.h"
#include "tklock.h"

#include "mce-io.h"			/* mce_write_number_string_to_file */
#include "mce-log.h"			/* mce_log(), LL_* */
#include "datapipe.h"			/* execute_datapipe(),
					 * datapipe_get_gint(),
					 * datapipe_get_old_gint(),
					 * append_input_trigger_to_datapipe(),
					 * append_output_trigger_to_datapipe(),
					 * remove_input_trigger_from_datapipe(),
					 * remove_output_trigger_from_datapipe()
					 */
#include "mce-conf.h"			/* mce_conf_get_bool() */
#include "mce-dbus.h"			/* mce_dbus_handler_add(),
					 * dbus_send(),
					 * dbus_send_message(),
					 * dbus_new_method_reply(),
					 * dbus_new_signal(),
					 * dbus_message_get_no_reply(),
					 * dbus_message_get_args(),
					 * dbus_message_append_args(),
					 * dbus_message_unref(),
					 * dbus_error_init(),
					 * dbus_error_free(),
					 * DBUS_MESSAGE_TYPE_METHOD_CALL,
					 * DBUS_TYPE_BOOLEAN,
					 * DBUS_TYPE_UINT32, DBUS_TYPE_INT32,
					 * DBUS_TYPE_STRING,
					 * DBUS_TYPE_INVALID
					 * DBusMessage, DBusError,
					 * dbus_bool_t,
					 * dbus_uint32_t, dbus_int32_t
					 */
#include "mce-gconf.h"			/* mce_gconf_notifier_add(),
					 * mce_gconf_get_bool()
					 */

/**
 * TRUE if the touchscreen/keypad autolock is enabled,
 * FALSE if the touchscreen/keypad autolock is disabled
 */
static gboolean tk_autolock_enabled = DEFAULT_TK_AUTOLOCK;

/** GConf callback ID for the autolock entry */
static guint tk_autolock_enabled_cb_id = 0;

/** Dimming timeout ID for the tklock */
static guint tklock_dim_timeout_cb_id = 0;

/** ID for touchscreen/keypad unlock source */
static guint tklock_unlock_timeout_cb_id = 0;

/** Blank immediately on tklock instead of dim/blank */
static gboolean blank_immediately = DEFAULT_BLANK_IMMEDIATELY;

/** Dim immediately on tklock instead of timeout */
static gboolean dim_immediately = DEFAULT_DIM_IMMEDIATELY;

/** Touchscreen/keypad dim timeout */
static gint dim_delay = DEFAULT_DIM_DELAY;

/** Disable touchscreen immediately on tklock instead of at blank */
static gboolean disable_ts_immediately = DEFAULT_TS_OFF_IMMEDIATELY;

/** Disable keypad immediately on tklock instead of at blank */
static gboolean disable_kp_immediately = DEFAULT_KP_OFF_IMMEDIATELY;

/** SysFS path to touchscreen event disable */
static const gchar *mce_touchscreen_sysfs_disable_path = NULL;

/** SysFS path to keypad event disable */
static const gchar *mce_keypad_sysfs_disable_path = NULL;

/**
 * Automagically re-lock the device when keyboard is closed,
 * if and only if no key has been pressed in-between
 */
static gboolean autorelock = FALSE;

static void touchscreen_trigger(gconstpointer const data);
static void remove_tklock_unlock_timeout(void);

/**
 * Query the event eater status
 *
 * @return TRUE if the event eater is enabled,
 *         FALSE if the event eater is disabled
 */
static gboolean is_eveater_enabled(void)
{
	return ((mce_get_submode_int32() & MCE_EVEATER_SUBMODE) != 0);
}

/**
 * Query the touchscreen/keypad lock status
 *
 * @return TRUE if the touchscreen/keypad lock is enabled,
 *         FALSE if the touchscreen/keypad lock is disabled
 */
static gboolean is_tklock_enabled(void)
{
	return ((mce_get_submode_int32() & MCE_TKLOCK_SUBMODE) != 0);
}

/**
 * Query the autorelock status
 *
 * @return TRUE if the autorelock is enabled,
 *         FALSE if the autorelock is disabled
 */
static gboolean is_autorelock_enabled(void)
{
	return autorelock;
}

/**
 * Enable auto-relock
 */
static void enable_autorelock(void)
{
	/* Touchscreen monitoring is only needed for the auto-relock */
	append_input_trigger_to_datapipe(&touchscreen_pipe,
					 touchscreen_trigger);
	autorelock = TRUE;
}

/**
 * Disable auto-relock
 */
static void disable_autorelock(void)
{
	/* Touchscreen monitoring is only needed for the auto-relock */
	remove_input_trigger_from_datapipe(&touchscreen_pipe,
					   touchscreen_trigger);
	autorelock = FALSE;
}

/**
 * Enable/disable touchscreen/keypad events
 *
 * @param file Path to enable/disable file
 * @param enable TRUE enable events, FALSE disable events
 * @return TRUE on success, FALSE on failure
 */
static gboolean generic_event_control(const gchar *const file,
				      const gboolean enable)
{
	gboolean status = FALSE;

	/* If no filename is specified, there is no interface
	 * for event control available; just smile and be happy
	 */
	if (file == NULL) {
		mce_log(LL_DEBUG,
			"No event control interface available; "
			"request ignored");
		status = TRUE;
		goto EXIT;
	}

	if (mce_write_number_string_to_file(file, !enable ? 1 : 0) == FALSE) {
		mce_log(LL_ERR,
			"%s: Event status *not* modified",
			file);
		goto EXIT;
	}

	mce_log(LL_DEBUG,
		"%s: events %s\n",
		file, enable ? "enabled" : "disabled");
	status = TRUE;

EXIT:
	return status;
}

/**
 * Enable/disable touchscreen events
 *
 * @param enable TRUE enable events, FALSE disable events
 * @return TRUE on success, FALSE on failure
 */
static gboolean ts_event_control(gboolean enable)
{
	return generic_event_control(mce_touchscreen_sysfs_disable_path,
				     enable);
}

/**
 * Enable/disable keypad events
 *
 * @param enable TRUE enable events, FALSE disable events
 * @return TRUE on success, FALSE on failure
 */
static gboolean kp_event_control(gboolean enable)
{
	return generic_event_control(mce_keypad_sysfs_disable_path, enable);
}

/**
 * Enable touchscreen (events will be generated by kernel)
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean ts_enable(void)
{
	return ts_event_control(TRUE);
}

/**
 * Disable touchscreen (no events will be generated by kernel)
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean ts_disable(void)
{
	return ts_event_control(FALSE);
}

/**
 * Enable keypad (events will be generated by kernel)
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean kp_enable(void)
{
	return kp_event_control(TRUE);
}

/**
 * Disable keypad (no events will be generated by kernel)
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean kp_disable(void)
{
	return kp_event_control(FALSE);
}

/**
 * Enable touchscreen and keypad
 *
 * @return TRUE on success, FALSE on failure or partial failure
 */
static gboolean ts_kp_enable(void)
{
	gboolean status = TRUE;

	if (kp_enable() == FALSE)
		status = FALSE;

	if (ts_enable() == FALSE)
		status = FALSE;

	return status;
}

/**
 * Disable touchscreen and keypad
 *
 * @return TRUE on success, FALSE on failure or partial failure
 */
static gboolean ts_kp_disable(void)
{
	gboolean status = TRUE;

	if (kp_disable() == FALSE)
		status = FALSE;

	if (ts_disable() == FALSE)
		status = FALSE;

	return status;
}

/**
 * Policy based enabling of touchscreen and keypad
 *
 * @return TRUE on success, FALSE on failure or partial failure
 */
static gboolean ts_kp_enable_policy(void)
{
	cover_state_t lid_cover_state = datapipe_get_gint(lid_cover_pipe);
	system_state_t system_state = datapipe_get_gint(system_state_pipe);
	alarm_ui_state_t alarm_ui_state = datapipe_get_gint(alarm_ui_state_pipe);
	gboolean status = FALSE;

	/* If the cover is closed, don't bother */
	if (lid_cover_state == COVER_CLOSED)
		goto EXIT2;

	if ((system_state == MCE_STATE_USER) ||
	    (alarm_ui_state == MCE_ALARM_UI_VISIBLE_INT32)) {
		if (ts_kp_enable() == FALSE)
			goto EXIT;
	}

EXIT2:
	status = TRUE;

EXIT:
	return status;
}

/**
 * Policy based disabling of touchscreen and keypad
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean ts_kp_disable_policy(void)
{
	display_state_t display_state = datapipe_get_gint(display_state_pipe);
	system_state_t system_state = datapipe_get_gint(system_state_pipe);
	alarm_ui_state_t alarm_ui_state = datapipe_get_gint(alarm_ui_state_pipe);
	submode_t submode = mce_get_submode_int32();
	gboolean status = FALSE;

	/* If we're in softoff submode, always disable */
	if ((submode & MCE_SOFTOFF_SUBMODE) == MCE_SOFTOFF_SUBMODE) {
		if (ts_kp_disable() == FALSE)
			goto EXIT;

		goto EXIT2;
	}

	/* If the Alarm UI is visible, don't disable */
	if (alarm_ui_state == MCE_ALARM_UI_VISIBLE_INT32) {
		mce_log(LL_DEBUG,
			"Alarm UI visible; refusing to disable touchscreen "
			"and keypad events");
		goto EXIT2;
	}

	if (system_state != MCE_STATE_USER) {
		if (ts_kp_disable() == FALSE)
			goto EXIT;
	} else if ((display_state == MCE_DISPLAY_OFF) &&
		   (is_tklock_enabled() == TRUE)) {
		if (ts_kp_disable() == FALSE)
			goto EXIT;
	} else if (is_tklock_enabled() == TRUE) {
		gboolean status2 = TRUE;

		if (disable_kp_immediately == TRUE)
			if (kp_disable() == FALSE)
				status2 = FALSE;

		if (disable_ts_immediately == TRUE)
			if (ts_disable() == FALSE)
				status2 = FALSE;

		if (status2 == FALSE)
			goto EXIT;
	}

EXIT2:
	status = TRUE;

EXIT:
	if (status == FALSE) {
		mce_log(LL_ERR, "Failed to disable ts/kp events!");
	}

	return status;
}

/**
 * Send the touchscreen/keypad lock mode
 *
 * @param method_call A DBusMessage to reply to;
 *                    pass NULL to send a tklock mode signal instead
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_send_tklock_mode(DBusMessage *const method_call)
{
	DBusMessage *msg = NULL;
	const gchar *modestring;
	gboolean status = FALSE;

	if (is_tklock_enabled() == TRUE)
		modestring = MCE_TK_LOCKED;
	else
		modestring = MCE_TK_UNLOCKED;

	/* If method_call is set, send a reply,
	 * otherwise, send a signal
	 */
	if (method_call != NULL) {
		msg = dbus_new_method_reply(method_call);
	} else {
		/* tklock_mode_ind */
		msg = dbus_new_signal(MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
				      MCE_TKLOCK_MODE_SIG);
	}

	/* Append the new mode */
	if (dbus_message_append_args(msg,
				     DBUS_TYPE_STRING, &modestring,
				     DBUS_TYPE_INVALID) == FALSE) {
		mce_log(LL_CRIT,
			"Failed to append %sargument to D-Bus message for %s",
			method_call ? "reply " : "",
			method_call ? MCE_TKLOCK_MODE_GET :
				      MCE_TKLOCK_MODE_SIG);
		dbus_message_unref(msg);
		goto EXIT;
	}

	/* Send the message */
	status = dbus_send_message(msg);

EXIT:
	return status;
}

/**
 * Show the touchscreen/keypad lock UI
 *
 * @param mode The mode to open in; valid modes:
 *             TKLOCK_ENABLE (open the tklock in normal mode)
 *             TKLOCK_HELP (show the tklock help infoprint)
 *             TKLOCK_SELECT (show the press select infoprint)
 *             TKLOCK_ONEINPUT (open the tklock in event eater mode)
 * @param silent TRUE to disable infoprints,
 *		 FALSE to enable infoprints
 * @return TRUE on success, FALSE on FAILURE
 */
static gboolean show_tklock_ui(const dbus_uint32_t mode,
			       const dbus_bool_t silent)
{
	const gchar *const cb_service = MCE_SERVICE;
	const gchar *const cb_path = MCE_REQUEST_PATH;
	const gchar *const cb_interface = MCE_REQUEST_IF;
	const gchar *const cb_method = MCE_TKLOCK_CB_REQ;

	/* com.nokia.system_ui.request.tklock_open */
	return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
			 SYSTEMUI_REQUEST_IF, SYSTEMUI_TKLOCK_OPEN_REQ, NULL,
			 DBUS_TYPE_STRING, &cb_service,
			 DBUS_TYPE_STRING, &cb_path,
			 DBUS_TYPE_STRING, &cb_interface,
			 DBUS_TYPE_STRING, &cb_method,
			 DBUS_TYPE_UINT32, &mode,
			 DBUS_TYPE_BOOLEAN, &silent,
			 DBUS_TYPE_BOOLEAN, (dbus_bool_t *)&has_flicker_key,
			 DBUS_TYPE_INVALID);
}

/**
 * Hide the touchscreen/keypad lock UI
 *
 * @param silent TRUE to disable infoprints,
 *		 FALSE to enable infoprints
 * @return TRUE on success, FALSE on failure
 */
static gboolean hide_tklock_ui(const dbus_bool_t silent)
{
	/* com.nokia.system_ui.request.tklock_close */
	return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
			 SYSTEMUI_REQUEST_IF, SYSTEMUI_TKLOCK_CLOSE_REQ, NULL,
			 DBUS_TYPE_BOOLEAN, (dbus_bool_t *)&silent,
			 DBUS_TYPE_INVALID);
}


/**
 * Enable the touchscreen/keypad lock
 *
 * If the internal state indicates that the tklock is already enabled,
 * silent mode will always be used
 *
 * @param silent TRUE to disable infoprints, FALSE to enable infoprints
 * @return TRUE on success, FALSE on failure
 */
static gboolean enable_tklock(gboolean silent)
{
	gboolean status = FALSE;

	if (is_tklock_enabled() == TRUE) {
		mce_log(LL_DEBUG,
			"Touchscreen/keypad lock enabled "
			"when already enabled");
		silent = TRUE;
	}

	if (show_tklock_ui(TKLOCK_ENABLE, silent) == FALSE)
		goto EXIT;

	mce_add_submode_int32(MCE_TKLOCK_SUBMODE);
	(void)mce_send_tklock_mode(NULL);

	/* Enable automagic relock */
	enable_autorelock();

	status = TRUE;

EXIT:
	return status;
}

/**
 * Timeout callback for touchscreen/keypad lock dim
 *
 * @param data Unused
 * @return Always returns FALSE, to disable the timeout
 */
static gboolean tklock_dim_timeout_cb(gpointer data)
{
	(void)data;

	tklock_dim_timeout_cb_id = 0;

	execute_datapipe(&display_state_pipe,
			 GINT_TO_POINTER(MCE_DISPLAY_DIM),
			 FALSE, TRUE);

	/* FALSE here is just to disable the timeout */
	return FALSE;
}

/**
 * Setup a timeout for tklock dimming
 */
static void setup_tklock_dim_timeout(void)
{
	tklock_dim_timeout_cb_id = g_timeout_add(dim_delay,
						 tklock_dim_timeout_cb,
						 NULL);
}

/**
 * Remove timeout for tklock dimming
 */
static void remove_tklock_dim_timeout(void)
{
	if (tklock_dim_timeout_cb_id != 0) {
		g_source_remove(tklock_dim_timeout_cb_id);
		tklock_dim_timeout_cb_id = 0;
	}
}

/**
 * Enable the touchscreen/keypad lock with policy
 *
 * If the internal state indicates that the tklock is already enabled,
 * silent mode will always be used
 *
 * @param force_blank Force immediate blanking
 * @return TRUE on success, FALSE on failure
 */
static gboolean enable_tklock_policy(gboolean force_blank)
{
	display_state_t display_state = datapipe_get_gint(display_state_pipe);
	system_state_t system_state = datapipe_get_gint(system_state_pipe);
	gboolean status = FALSE;

	/* If we're in any other state than USER, don't enable tklock */
	if (system_state != MCE_STATE_USER) {
		status = TRUE;
		goto EXIT;
	}

	/* Enable lock */
	if (enable_tklock(force_blank |
			  dim_immediately |
			  blank_immediately) == FALSE)
		goto EXIT;

	if ((blank_immediately == FALSE) &&
	    (force_blank == FALSE) &&
	    (display_state == MCE_DISPLAY_ON)) {
		if (dim_immediately == FALSE) {
			setup_tklock_dim_timeout();
		} else {
			execute_datapipe(&display_state_pipe,
					 GINT_TO_POINTER(MCE_DISPLAY_DIM),
					 FALSE, TRUE);
		}
	}

	if (blank_immediately == TRUE || force_blank == TRUE)
		execute_datapipe(&display_state_pipe,
				 GINT_TO_POINTER(MCE_DISPLAY_OFF),
				 FALSE, TRUE);

	/* Disable touchscreen and keypad */
	(void)ts_kp_disable_policy();

	status = TRUE;

EXIT:
	return status;
}

/**
 * Disable the touchscreen/keypad lock
 *
 * If the internal state indicates that the tklock is already disabled,
 * silent mode will always be used
 *
 * @param silent Enable without infoprint
 * @return TRUE on success, FALSE on failure
 */
static gboolean disable_tklock(gboolean silent)
{
	gboolean status = FALSE;

	/* Disable timeouts, just to be sure */
	remove_tklock_unlock_timeout();
	remove_tklock_dim_timeout();

	/* On startup of MCE, we always disable
	 * the touchscreen/keypad lock and single event eater
	 */
	if (is_tklock_enabled() == FALSE) {
		mce_log(LL_DEBUG,
			"Touchscreen/keypad lock disabled "
			"when already disabled");
		silent = TRUE;
	}

	if (is_eveater_enabled() == FALSE) {
		mce_log(LL_DEBUG,
			"Touchscreen/keypad single event eater disabled "
			"when already disabled");
	}

	if (hide_tklock_ui(silent) == FALSE)
		goto EXIT;

	mce_rem_submode_int32(MCE_TKLOCK_SUBMODE | MCE_EVEATER_SUBMODE);
	(void)mce_send_tklock_mode(NULL);
	(void)ts_kp_enable();
	status = TRUE;

EXIT:
	return status;
}

/**
 * Timeout callback for tklock unlock
 *
 * @param data Unused
 * @return Always returns FALSE, to disable the timeout
 */
static gboolean tklock_unlock_timeout_cb(gpointer data)
{
	(void)data;

	tklock_unlock_timeout_cb_id = 0;

	// FIXME: error handling?
	(void)disable_tklock(FALSE);
	disable_autorelock();

	/* FALSE here is just to disable the timeout */
	return FALSE;
}

/**
 * Setup a timeout for delayed unlocking of touchscreen/keypad lock
 */
static void setup_tklock_unlock_timeout(void)
{
	tklock_unlock_timeout_cb_id = g_timeout_add(MCE_TKLOCK_UNLOCK_DELAY,
						    tklock_unlock_timeout_cb,
						    NULL);
}

/**
 * Remove timeout for delayed unlocking of touchscreen/keypad lock
 */
static void remove_tklock_unlock_timeout(void)
{
	if (tklock_unlock_timeout_cb_id != 0) {
		g_source_remove(tklock_unlock_timeout_cb_id);
		tklock_unlock_timeout_cb_id = 0;
	}
}

/**
 * Enable the touchscreen/keypad single event eater
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean enable_eveater(void)
{
	if (show_tklock_ui(TKLOCK_ONEINPUT, TRUE) == TRUE)
		mce_add_submode_int32(MCE_EVEATER_SUBMODE);
	else
		return FALSE;

	return TRUE;
}

/**
 * Enable the touchscreen/keypad autolock
 *
 * Will enable touchscreen/keypad lock if tk_autolock_enabled is TRUE,
 * and enable the touchscreen/keypad single event eater if FALSE
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean enable_autokeylock(void)
{
	if (tk_autolock_enabled &&
	    (mce_get_mode_int32() != MCE_VOIP_MODE_INT32)) {
		(void)ts_kp_disable();
		return enable_tklock(TRUE);
	} else {
		return enable_eveater();
	}
}

/**
 * State machine for lock change requests
 *
 * @param state The requested touchscreen/keypad lock state
 */
static void set_tklock_state(lock_state_t lock_state)
{
	switch (lock_state) {
	case LOCK_OFF:
		(void)disable_tklock(FALSE);
		disable_autorelock();
		break;

	case LOCK_OFF_SILENT:
		(void)disable_tklock(TRUE);
		disable_autorelock();
		break;

	case LOCK_OFF_DELAYED:
		setup_tklock_unlock_timeout();
		break;

	case LOCK_ON:
		(void)enable_tklock_policy(FALSE);
		break;

	case LOCK_ON_DIMMED:
		(void)enable_tklock(FALSE);
		setup_tklock_dim_timeout();
		break;

	case LOCK_ON_SILENT:
		(void)enable_tklock(TRUE);
		break;

	case LOCK_ON_SILENT_DIMMED:
		(void)enable_tklock(TRUE);
		setup_tklock_dim_timeout();
		break;

	case LOCK_TOGGLE:
		/* Touchscreen/keypad lock */
		if (is_tklock_enabled() == FALSE) {
			(void)enable_tklock_policy(FALSE);
		} else {
			(void)disable_tklock(FALSE);
			disable_autorelock();

			/* Unblank screen */
			execute_datapipe(&display_state_pipe,
					 GINT_TO_POINTER(MCE_DISPLAY_ON),
					 FALSE, TRUE);
		}
		break;

	default:
		break;
	}
}

/**
 * D-Bus callback for the get tklock mode method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean tklock_mode_get_req_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	mce_log(LL_DEBUG, "Received tklock mode get request");

	/* Try to send a reply that contains the current tklock mode */
	if (mce_send_tklock_mode(msg) == FALSE)
		goto EXIT;

	status = TRUE;

EXIT:
	return status;
}

/**
 * D-Bus callback for the tklock mode change method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean tklock_mode_change_req_dbus_cb(DBusMessage *const msg)
{
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	gboolean status = FALSE;
	gchar *mode = NULL;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	mce_log(LL_DEBUG, "Received tklock mode change request");

	if (dbus_message_get_args(msg, &error,
				  DBUS_TYPE_STRING, &mode,
				  DBUS_TYPE_INVALID) == FALSE) {
		// XXX: should we return an error instead?
		mce_log(LL_CRIT,
			"Failed to get argument from %s: %s",
			MCE_TKLOCK_MODE_CHANGE_REQ,
			error.message);
		dbus_error_free(&error);
		goto EXIT;
	}

	/* Try to change to the requested tklock mode
	 * XXX: right now we silently ignore invalid modes;
	 * should we return an error?
	 */
	if (strcmp(MCE_TK_LOCKED, mode) == 0) {
		set_tklock_state(LOCK_ON);
	} else if (strcmp(MCE_TK_LOCKED_DIM, mode) == 0) {
		set_tklock_state(LOCK_ON_DIMMED);
	} else if (strcmp(MCE_TK_SILENT_LOCKED, mode) == 0) {
		set_tklock_state(LOCK_ON_SILENT);
	} else if (strcmp(MCE_TK_SILENT_LOCKED_DIM, mode) == 0) {
		set_tklock_state(LOCK_ON_SILENT_DIMMED);
	} else if (strcmp(MCE_TK_UNLOCKED, mode) == 0) {
		set_tklock_state(LOCK_OFF);
	} else if (strcmp(MCE_TK_SILENT_UNLOCKED, mode) == 0) {
		set_tklock_state(LOCK_OFF_SILENT);
	} else {
		mce_log(LL_ERR,
			"Received an invalid tklock mode; ignoring");
	}

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

EXIT:
	return status;
}

/**
 * D-Bus callback from SystemUI touchscreen/keypad lock
 *
 * @todo the calls to disable_tklock/show_tklock_ui need error handling
 *
 * @param msg D-Bus message with the lock status
 */
static gboolean systemui_tklock_dbus_cb(DBusMessage *const msg)
{
	dbus_int32_t result = INT_MAX;
	gboolean status = FALSE;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	mce_log(LL_DEBUG, "Received tklock callback");

	if (dbus_message_get_args(msg, &error,
				  DBUS_TYPE_INT32, &result,
				  DBUS_TYPE_INVALID) == FALSE) {
		// XXX: should we return an error instead?
		mce_log(LL_CRIT,
			"Failed to get argument from %s: %s",
			MCE_TKLOCK_CB_REQ,
			error.message);
		dbus_error_free(&error);
		goto EXIT;
	}

	mce_log(LL_DEBUG, "tklock callback value: %d", result);

	switch (result) {
	case TKLOCK_UNLOCK:
		(void)execute_datapipe(&tk_lock_pipe,
				       GINT_TO_POINTER(LOCK_OFF),
				       FALSE, TRUE);
		break;

	case TKLOCK_RETRY:
		/* disable keypad again */
		// FIXME: error handling?
		(void)show_tklock_ui(TKLOCK_HELP, FALSE);
		break;

	case TKLOCK_TIMEOUT:
	default:
		break;
	}

	status = TRUE;

EXIT:
	return status;
}

/**
 * GConf callback for touchscreen/keypad lock related settings
 *
 * @param gce Unused
 * @param id Connection ID from gconf_client_notify_add()
 * @param entry The modified GConf entry
 * @param data Unused
 */
static void tklock_gconf_cb(GConfClient *const gce, const guint id,
			    GConfEntry *const entry, gpointer const data)
{
	GConfValue *gcv = gconf_entry_get_value(entry);

	(void)gce;
	(void)data;

	if (id == tk_autolock_enabled_cb_id)
		tk_autolock_enabled = gconf_value_get_bool(gcv) ? 1 : 0;
	else
		mce_log(LL_WARN, "Spurious GConf value received; confused!");
}

/**
 * Datapipe trigger for the keyboard slide
 *
 * @param data 1 if the keyboard is open, 0 if the keyboard is closed
 */
static void kbd_slide_trigger(gconstpointer const data)
{
	if (GPOINTER_TO_INT(data) == 1) {
		if (is_tklock_enabled() == TRUE) {
			(void)disable_tklock(FALSE);
		}
	} else if (is_autorelock_enabled() == TRUE) {
		(void)enable_tklock_policy(FALSE);
	}
}

/**
 * Datapipe trigger for the [lock] flicker key
 *
 * @param data 1 if the key was pressed, 0 if the key was released
 */
static void lockkey_trigger(gconstpointer const data)
{
	system_state_t system_state = datapipe_get_gint(system_state_pipe);

	/* Only react on the [lock] flicker key in USER state */
	if ((GPOINTER_TO_INT(data) == 1) && (system_state == MCE_STATE_USER)) {
		(void)execute_datapipe(&tk_lock_pipe,
				       GINT_TO_POINTER(LOCK_TOGGLE),
				       FALSE, TRUE);
	}
}

/**
 * Datapipe trigger for keypresses
 *
 * @param data Unused
 */
static void keypress_trigger(gconstpointer const data)
{
	struct input_event const *const *evp = data;
	struct input_event const *ev = *evp;
	submode_t submode = mce_get_submode_int32();

	(void)data;

	/* If the tklock is disabled, disable automagic relocking */
	if (is_tklock_enabled() == FALSE) {
		disable_autorelock();
	}

	/* If the keypress is [power], disable the event eater submode,
	 * and, if the tklock is enabled, refresh the tklock UI
	 */
	if ((ev->code == power_keycode) && (ev->value == 1)) {
		/* If set, the [power] key was pressed */

		mce_rem_submode_int32(MCE_EVEATER_SUBMODE);

		if (is_tklock_enabled() == TRUE) {
			/* Enable keypad, restart timeout
			 * If alarm UI is visible, ignore the keypress
			 */
			// FIXME: error handling?
			if ((submode & MCE_ALARM_SUBMODE) == 0) {
				(void)kp_enable();
				(void)show_tklock_ui(TKLOCK_SELECT, FALSE);
			}
		}
	}
}

/**
 * Datapipe trigger for touchscreen events
 *
 * @param data Unused
 */
static void touchscreen_trigger(gconstpointer const data)
{
	(void)data;

	/* If the tklock is disabled, disable automagic relocking */
	if (is_tklock_enabled() == FALSE) {
		disable_autorelock();
	}
}

/**
 * Handle system state change
 *
 * @param data The system state stored in a pointer
 */
static void system_state_trigger(gconstpointer data)
{
	system_state_t system_state = GPOINTER_TO_INT(data);

	switch (system_state) {
	case MCE_STATE_SHUTDOWN:
	case MCE_STATE_REBOOT:
	case MCE_STATE_ACTDEAD:
		(void)ts_kp_disable_policy();
		break;

	case MCE_STATE_USER:
	default:
		(void)ts_kp_enable_policy();
		break;
	}
}

/**
 * Handle display state change
 *
 * @param data The display state stored in a pointer
 */
static void display_state_trigger(gconstpointer data)
{
	display_state_t old_display_state = datapipe_get_old_gint(display_state_pipe);
	display_state_t display_state = GPOINTER_TO_INT(data);
	system_state_t system_state = datapipe_get_gint(system_state_pipe);

	switch (display_state) {
	case MCE_DISPLAY_OFF:
		if (is_tklock_enabled() == TRUE) {
			(void)ts_kp_disable_policy();
		} else {
			if (system_state == MCE_STATE_USER)
				enable_autokeylock();
		}

		break;

	case MCE_DISPLAY_DIM:
	case MCE_DISPLAY_ON:
	default:
		/* If the display transitions from OFF or UNDEF,
		 * to DIM or ON, do policy based enable
		 */
		if ((old_display_state == MCE_DISPLAY_UNDEF) ||
		    (old_display_state == MCE_DISPLAY_OFF)) {
			(void)ts_kp_enable_policy();

			if (is_eveater_enabled() == TRUE)
				disable_tklock(TRUE);
		}

		break;
	}
}

/**
 * Handle alarm UI state change
 *
 * @param data The alarm state stored in a pointer
 */
static void alarm_ui_state_trigger(gconstpointer data)
{
	alarm_ui_state_t alarm_ui_state = GPOINTER_TO_INT(data);

	switch (alarm_ui_state) {
	case MCE_ALARM_UI_VISIBLE_INT32:
		ts_kp_enable_policy();
		break;

	case MCE_ALARM_UI_SNOOZED_INT32:
	case MCE_ALARM_UI_OFF_INT32:
		ts_kp_disable_policy();
		break;

	default:
		break;
	}
}

/**
 * Handle lid cover state change
 *
 * @param data The lid cover state stored in a pointer
 */
static void lid_cover_trigger(gconstpointer data)
{
	cover_state_t lid_cover_state = GPOINTER_TO_INT(data);
	system_state_t system_state = datapipe_get_gint(system_state_pipe);

	switch (lid_cover_state) {
	case COVER_OPEN:
		/* XXX: should we cover alarm state too? */
		if (system_state == MCE_STATE_USER)
			setup_tklock_unlock_timeout();
		break;

	case COVER_CLOSED:
		if (system_state == MCE_STATE_USER)
			(void)enable_tklock_policy(FALSE);
		break;

	default:
		break;
	}
}

/**
 * Handle touchscreen/keypad lock state
 *
 * @param data The touchscreen/keypad lock state stored in a pointer
 */
static void tk_lock_trigger(gconstpointer data)
{
	lock_state_t tk_lock_state = GPOINTER_TO_INT(data);

	set_tklock_state(tk_lock_state);
}

/**
 * Init function for the touchscreen/keypad lock component
 *
 * @todo the call to disable_tklock needs error handling
 *
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_tklock_init(void)
{
	gboolean status = FALSE;

	/* Init event control files */
	if (g_access(MCE_KEYBOARD_SYSFS_DISABLE_PATH, W_OK) == 0) {
		mce_keypad_sysfs_disable_path = MCE_KEYBOARD_SYSFS_DISABLE_PATH;
	} else if (g_access(MCE_KEYPAD_SYSFS_DISABLE_PATH, W_OK) == 0) {
		mce_keypad_sysfs_disable_path = MCE_KEYPAD_SYSFS_DISABLE_PATH;
	}

	if (g_access(MCE_TOUCHSCREEN_SYSFS_DISABLE_PATH, W_OK) == 0) {
		mce_touchscreen_sysfs_disable_path = MCE_TOUCHSCREEN_SYSFS_DISABLE_PATH;
	}

	/* Close the touchscreen/keypad lock UI to make sure MCE
	 * doesn't end up in a confused mode if it's restarted
	 */
	// FIXME: error handling?
	(void)disable_tklock(TRUE);
	disable_autorelock();

	/* Append triggers/filters to datapipes */
	append_input_trigger_to_datapipe(&keypress_pipe,
					 keypress_trigger);
	append_input_trigger_to_datapipe(&lockkey_pipe,
					 lockkey_trigger);
	append_input_trigger_to_datapipe(&keyboard_slide_pipe,
					 kbd_slide_trigger);
	append_output_trigger_to_datapipe(&system_state_pipe,
					  system_state_trigger);
	append_output_trigger_to_datapipe(&display_state_pipe,
					  display_state_trigger);
	append_output_trigger_to_datapipe(&alarm_ui_state_pipe,
					  alarm_ui_state_trigger);
	append_output_trigger_to_datapipe(&lid_cover_pipe,
					  lid_cover_trigger);
	append_output_trigger_to_datapipe(&tk_lock_pipe,
					  tk_lock_trigger);

	/* Touchscreen/keypad autolock */
	/* Since we've set a default, error handling is unnecessary */
	(void)mce_gconf_get_bool(MCE_GCONF_TK_AUTOLOCK_ENABLED_PATH,
				 &tk_autolock_enabled);

	/* Touchscreen/keypad autolock enabled/disabled */
	if (mce_gconf_notifier_add(MCE_GCONF_LOCK_PATH,
				   MCE_GCONF_TK_AUTOLOCK_ENABLED_PATH,
				   tklock_gconf_cb,
				   &tk_autolock_enabled_cb_id) == FALSE)
		goto EXIT;

	/* get_tklock_mode */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_TKLOCK_MODE_GET,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 tklock_mode_get_req_dbus_cb) == FALSE)
		goto EXIT;

	/* rq_tklock_mode_change */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_TKLOCK_MODE_CHANGE_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 tklock_mode_change_req_dbus_cb) == FALSE)
		goto EXIT;

	/* tklock_callback */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_TKLOCK_CB_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 systemui_tklock_dbus_cb) == FALSE)
		goto EXIT;

	/* Get configuration options */
	blank_immediately = mce_conf_get_bool(MCE_CONF_TKLOCK_GROUP,
					      MCE_CONF_BLANK_IMMEDIATELY,
					      DEFAULT_BLANK_IMMEDIATELY,
					      NULL);

	dim_immediately = mce_conf_get_bool(MCE_CONF_TKLOCK_GROUP,
					    MCE_CONF_DIM_IMMEDIATELY,
					    DEFAULT_DIM_IMMEDIATELY,
					    NULL);

	dim_delay = mce_conf_get_int(MCE_CONF_TKLOCK_GROUP,
				     MCE_CONF_DIM_DELAY,
				     DEFAULT_DIM_DELAY,
				     NULL);

	disable_ts_immediately = mce_conf_get_bool(MCE_CONF_TKLOCK_GROUP,
						   MCE_CONF_TS_OFF_IMMEDIATELY,
						   DEFAULT_TS_OFF_IMMEDIATELY,
						   NULL);

	disable_kp_immediately = mce_conf_get_bool(MCE_CONF_TKLOCK_GROUP,
						   MCE_CONF_KP_OFF_IMMEDIATELY,
						   DEFAULT_KP_OFF_IMMEDIATELY,
						   NULL);

	status = TRUE;

EXIT:
	return status;
}

/**
 * Exit function for the touchscreen/keypad lock component
 *
 * @todo D-Bus unregistration
 */
void mce_tklock_exit(void)
{
	/* Remove triggers/filters from datapipes */
	remove_output_trigger_from_datapipe(&tk_lock_pipe,
					    tk_lock_trigger);
	remove_output_trigger_from_datapipe(&lid_cover_pipe,
					    lid_cover_trigger);
	remove_output_trigger_from_datapipe(&alarm_ui_state_pipe,
					    alarm_ui_state_trigger);
	remove_output_trigger_from_datapipe(&display_state_pipe,
					    display_state_trigger);
	remove_output_trigger_from_datapipe(&system_state_pipe,
					    system_state_trigger);
	remove_input_trigger_from_datapipe(&keyboard_slide_pipe,
					   kbd_slide_trigger);
	remove_input_trigger_from_datapipe(&lockkey_pipe,
					   lockkey_trigger);
	remove_input_trigger_from_datapipe(&keypress_pipe,
					   keypress_trigger);

	/* This trigger is conditional; attempt to remove it anyway */
	remove_input_trigger_from_datapipe(&touchscreen_pipe,
					   touchscreen_trigger);

	/* Remove the timeout source for the touchscreen/keypad unlock */
	remove_tklock_unlock_timeout();

	/* Remove the timeout source for the touchscreen/keypad lock dim */
	remove_tklock_dim_timeout();

	return;
}
