/**
 * @file powerkey.c
 * Power key logic for the Mode Control Entity
 * <p>
 * Copyright © 2004-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>

#include <stdlib.h>			/* exit(), EXIT_FAILURE */
#include <string.h>			/* strcmp() */
#include <linux/input.h>		/* struct input_event */

#include <mce/mode-names.h>
#include <systemui/dbus-names.h>
#include <systemui/modechange-dbus-names.h>
#include <systemui/powerkeymenu-dbus-names.h>

#include "mce.h"			/* mce_set_mode_int32(),
					 * mce_get_submode_int32(),
					 * mce_add_submode_int32(),
					 * mce_rem_submode_int32(),
					 * mainloop,
					 * mode_pipe,
					 * submode_pipe,
					 * system_state_pipe,
					 * tk_lock_pipe,
					 * device_lock_pipe,
					 * keypress_pipe,
					 * system_state_t,
					 * submode_t
					 */
#include "powerkey.h"

#include "mce-log.h"			/* mce_log(), LL_* */
#include "mce-conf.h"			/* mce_conf_get_int(),
					 * mce_conf_get_string()
					 */
#include "mce-dbus.h"			/* mce_dbus_handler_add(),
					 * dbus_send(),
					 * dbus_send_message(),
					 * dbus_new_method_reply(),
					 * dbus_message_get_no_reply(),
					 * dbus_message_get_args(),
					 * dbus_error_init(),
					 * dbus_error_free(),
					 * DBUS_MESSAGE_TYPE_METHOD_CALL,
					 * DBUS_TYPE_BOOLEAN,
					 * DBUS_TYPE_UINT32, DBUS_TYPE_INT32,
					 * DBUS_TYPE_STRING,
					 * DBUS_TYPE_INVALID,
					 * DBusMessage, DBusError,
					 * dbus_bool_t,
					 * dbus_uint32_t, dbus_int32_t
					 */
#include "mce-dsme.h"			/* request_normal_shutdown(),
					 * request_soft_poweron(),
					 * request_soft_poweroff(),
					 * request_powerup(),
					 * request_reboot(),
					 */
#include "datapipe.h"			/* execute_datapipe(),
					 * execute_datapipe_filters(),
					 * datapipe_get_gint(),
					 * append_output_trigger_to_datapipe(),
					 * remove_output_trigger_from_datapipe()
					 */
#include "connectivity.h"		/* get_connectivity_status() */

/**
 * The ID of the timeout used when determining
 * whether the key press was short or long
 */
static guint powerkey_timeout_cb_id = 0;

/**
 * The ID of the timeout used when determining
 * whether the key press was a double press
 */
static guint doublepress_timeout_cb_id = 0;

/**
 * Used to keep track of whether the powerkeymenu component has been
 * initialised or not
 */
static gboolean initialised = FALSE;

/** Time in milliseconds before the key press is considered medium */
static gint mediumdelay = DEFAULT_POWER_MEDIUM_DELAY;
/** Time in milliseconds before the key press is considered long */
static gint longdelay = DEFAULT_POWER_LONG_DELAY;
/** Timeout in milliseconds during which key press is considered double */
static gint doublepressdelay = DEFAULT_POWER_DOUBLE_DELAY;
/** Action to perform on a short key press */
static poweraction_t shortpressaction = DEFAULT_POWERKEY_SHORT_ACTION;
/** Action to perform on a long key press */
static poweraction_t longpressaction = DEFAULT_POWERKEY_LONG_ACTION;
/** Action to perform on a double key press */
static poweraction_t doublepressaction = DEFAULT_POWERKEY_DOUBLE_ACTION;

static guint32 modetransition = 0;	/**< Modetransition taking place */

/**
 * D-Bus reply handler for device lock UI enabling
 *
 * @param pending_call The DBusPendingCall
 * @param data Unused
 * @return TRUE on success, FALSE on failure
 */
static void device_menu_reply_dbus_cb(DBusPendingCall *pending_call,
				      void *data)
{
	DBusMessage *reply;
	dbus_int32_t retval;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	(void)data;

	mce_log(LL_DEBUG, "Received device menu UI reply");

	if ((reply = dbus_pending_call_steal_reply(pending_call)) == NULL) {
		mce_log(LL_ERR,
			"Device menu reply callback invoked, "
			"but no pending call available");
		goto EXIT;
	}

	/* Make sure we didn't get an error message */
	if (dbus_message_is_error(reply, DBUS_ERROR_UNKNOWN_METHOD) == TRUE) {
		mce_log(LL_ERR,
			"Device menu D-Bus method not handled");
		goto EXIT2;
	} else if (dbus_message_is_error(reply,
					 DBUS_ERROR_INVALID_ARGS) == TRUE) {
		mce_log(LL_ERR,
			"Incorrect arguments passed to "
			"device menu D-Bus method");
		goto EXIT2;
	}

	/* Extract reply */
	if (dbus_message_get_args(reply, &error,
				  DBUS_TYPE_INT32, &retval,
				  DBUS_TYPE_INVALID) == FALSE) {
		mce_log(LL_CRIT,
			"Failed to get reply from %s/%s: %s",
			SYSTEMUI_POWERKEYMENU_OPEN_REQ,
			SYSTEMUI_POWERKEYMENU_CLOSE_REQ,
			error.message);
		dbus_error_free(&error);
		goto EXIT2;
	}

	/* XXX: Iiiieeek, hardcoded values! */
	switch (retval) {
	case -3:
		/* Menu opened as it should */
		mce_add_submode_int32(MCE_DEVMENU_SUBMODE);
		break;

	case -2:
		/* Some other process has opened the menu */
		mce_log(LL_ERR,
			"Device menu already opened another by other process");
		break;

	case 0:
		/* The menu closed as it should */
		mce_rem_submode_int32(MCE_DEVMENU_SUBMODE);
		break;

	default:
		/* Unknown reply */
		mce_log(LL_ERR,
			"Unknown return value received from the device menu");
		break;
	}

EXIT2:
	dbus_message_unref(reply);

EXIT:
	dbus_pending_call_unref(pending_call);

	return;
}

/**
 * Open/close the powerkey menu
 *
 * @param enable TRUE to open the powerkey menu, FALSE to close it
 * @return TRUE on success, FALSE on failure
 */
static gboolean device_menu(const gboolean enable)
{
	const gchar *const cb_service = MCE_SERVICE;
	const gchar *const cb_path = MCE_REQUEST_PATH;
	const gchar *const cb_interface = MCE_REQUEST_IF;
	const gchar *const cb_method = MCE_POWERKEY_CB_REQ;
	gconstpointer cookedmode = execute_datapipe_filters(&mode_pipe,
							    NULL, TRUE);
	dbus_uint32_t mode;

	mode = (GPOINTER_TO_INT(cookedmode) ==
		MCE_FLIGHT_MODE_INT32) ? MODE_FLIGHT : MODE_NORMAL;

	mce_log(LL_DEBUG,
		"UI called with cooked mode %d (raw mode %d)",
		mode, datapipe_get_gint(mode_pipe));

	/* com.nokia.system_ui.request.powerkeymenu_{open,close} */
	return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
			 SYSTEMUI_REQUEST_IF,
			 enable ? SYSTEMUI_POWERKEYMENU_OPEN_REQ :
				  SYSTEMUI_POWERKEYMENU_CLOSE_REQ,
			 device_menu_reply_dbus_cb,
			 DBUS_TYPE_STRING, &cb_service,
			 DBUS_TYPE_STRING, &cb_path,
			 DBUS_TYPE_STRING, &cb_interface,
			 DBUS_TYPE_STRING, &cb_method,
			 DBUS_TYPE_UINT32, &mode,
			 DBUS_TYPE_INVALID);
}

/**
 * Show/hide the modetransition confirmation dialog
 *
 * @param mode The current mode
 * @param open_dialog TRUE to open the dialog, FALSE to close the dialog
 * @return TRUE on success, FALSE on failure
 */
static gboolean call_confirm_mode(const dbus_uint32_t mode,
				  const gboolean open_dialog)
{
	const gchar *const cb_service = MCE_SERVICE;
	const gchar *const cb_path = MCE_REQUEST_PATH;
	const gchar *const cb_interface = MCE_REQUEST_IF;
	const gchar *const cb_method = MCE_MODECHG_CB_REQ;

	/* com.nokia.system_ui.request.modechange_{open,close} */
	if (open_dialog == TRUE) {
		return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
				 SYSTEMUI_REQUEST_IF,
				 SYSTEMUI_MODECHANGE_OPEN_REQ, NULL,
				 DBUS_TYPE_STRING, &cb_service,
				 DBUS_TYPE_STRING, &cb_path,
				 DBUS_TYPE_STRING, &cb_interface,
				 DBUS_TYPE_STRING, &cb_method,
				 DBUS_TYPE_UINT32, &mode,
				 DBUS_TYPE_INVALID);
	} else {
		return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
				 SYSTEMUI_REQUEST_IF,
				 SYSTEMUI_MODECHANGE_CLOSE_REQ, NULL,
				 DBUS_TYPE_INVALID);
	}
}

/**
 * Flightmode confirmation logic
 *
 * @todo Decide on error handling policy
 * @todo Error handling for mce_set_mode_int32
 *
 * @return TRUE on success, FALSE on FAILURE
 */
static gboolean confirm_flightmode(void)
{
	gboolean connected;

	/* D-Bus call failed */
	if (get_connectivity_status(&connected) == FALSE) {
		// XXX: Is this a reasonable policy?
		g_main_loop_quit(mainloop);
		exit(EXIT_FAILURE);
	}

	if (connected == TRUE) {
		modetransition = MODECHANGE_TO_FLIGHTMODE;
		mce_log(LL_DEBUG, "Opening mode confirmation dialog (%d)",
			modetransition);
		mce_add_submode_int32(MCE_MODECHG_SUBMODE);
		return call_confirm_mode(modetransition, TRUE);
	} else {
		// FIXME: error handling?
		(void)mce_set_mode_int32(MCE_FLIGHT_MODE_INT32);
		return TRUE;
	}
}

/**
 * Normalmode confirmation logic
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean confirm_normalmode(void)
{
	modetransition = MODECHANGE_TO_NORMALMODE;
	mce_add_submode_int32(MCE_MODECHG_SUBMODE);
	mce_log(LL_DEBUG,
		"Opening mode confirmation dialog (%d)", modetransition);
	return call_confirm_mode(modetransition, TRUE);
}

/**
 * Close mode confirmation dialog
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean close_confirmmode(void)
{
	mce_log(LL_DEBUG, "Closing mode confirmation dialog");
	mce_rem_submode_int32(MCE_MODECHG_SUBMODE);
	return call_confirm_mode(0, FALSE);
}

/**
 * Timeout callback for double key press
 *
 * @param data Unused
 * @return Always returns FALSE, to disable the timeout
 */
static gboolean doublepress_timeout_cb(gpointer data)
{
	(void)data;

	doublepress_timeout_cb_id = 0;

	/* FALSE here is just to disable the timeout */
	return FALSE;
}

/**
 * Generic logic for key presses
 *
 * @param action The action to take
 */
static void generic_powerkey_handler(poweraction_t action)
{
	submode_t submode = mce_get_submode_int32();

	switch (action) {
	case POWER_DISABLED:
		break;

	case POWER_MENU:
		if (submode == MCE_NORMAL_SUBMODE) {
			/* Show the device menu */
			// FIXME: error handling?
			(void)device_menu(TRUE);
		}

		break;

	case POWER_POWEROFF:
	default:
		/* If the device mode menu is open, close it */
		// FIXME: error handling?
		(void)device_menu(FALSE);

		/* Only shutdown if the tklock isn't active */
		if ((submode & MCE_TKLOCK_SUBMODE) == 0) {
			request_normal_shutdown();
		}

		break;

	case POWER_SOFT_POWEROFF:
		/* If the device mode menu is open, close it */
		// FIXME: error handling?
		(void)device_menu(FALSE);

		/* Only soft poweroff if the tklock isn't active */
		if ((submode & MCE_TKLOCK_SUBMODE) == 0) {
			request_soft_poweroff();
		}

		break;
	}
}

/**
 * Logic for short key press
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean handle_shortpress(void)
{
	system_state_t system_state = datapipe_get_gint(system_state_pipe);
	gboolean status = FALSE;

	if (powerkey_timeout_cb_id != 0) {
		g_source_remove(powerkey_timeout_cb_id);
		powerkey_timeout_cb_id = 0;
	}

	if (doublepress_timeout_cb_id == 0) {
		doublepress_timeout_cb_id =
			g_timeout_add(doublepressdelay,
				      doublepress_timeout_cb,
				      NULL);

		if (system_state == MCE_STATE_USER) {
			generic_powerkey_handler(shortpressaction);
		}
	} else {
		g_source_remove(doublepress_timeout_cb_id);
		doublepress_timeout_cb_id = 0;
		generic_powerkey_handler(doublepressaction);
	}

	status = TRUE;

//EXIT:
	return status;
}

/**
 * Logic for long key press
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean handle_longpress(void)
{
	system_state_t state = datapipe_get_gint(system_state_pipe);
	gboolean status = FALSE;

	powerkey_timeout_cb_id = 0;

	/* If we're in transition submode, ignore [power],
	 * unless the device lock is active
	 */
	if (((mce_get_submode_int32() &
	      MCE_DEVLOCK_SUBMODE) != MCE_DEVLOCK_SUBMODE) &&
	    (mce_get_submode_int32() & MCE_TRANSITION_SUBMODE)) {
		status = FALSE;
		goto EXIT;
	}

	/* Long key press */
	if (state == MCE_STATE_ACTDEAD) {
		/* If the alarm UI is visible, close it */
		if ((mce_get_submode_int32() & MCE_ALARM_SUBMODE)) {
			request_normal_shutdown();
		} else {
			request_powerup();
		}
	} else if (state == MCE_STATE_USER) {
		/* If softoff is enabled, wake up
		 * otherwise, perform long press action
		 */
		if ((mce_get_submode_int32() & MCE_SOFTOFF_SUBMODE)) {
			request_soft_poweron();
		} else {
			generic_powerkey_handler(longpressaction);
		}
	}

	status = TRUE;

EXIT:
	return status;
}

/**
 * Timeout callback for long key press
 *
 * @param data Unused
 * @return Always returns FALSE, to disable the timeout
 */
static gboolean powerkey_timeout_cb(gpointer data)
{
	(void)data;

	handle_longpress();

	/* FALSE here is just to disable the timeout */
	return FALSE;
}

/**
 * D-Bus callback for powerkey event triggering
 *
 * @param msg D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean trigger_powerkey_event_req_dbus_cb(DBusMessage *const msg)
{
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	dbus_bool_t result;
	gboolean status = FALSE;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	mce_log(LL_DEBUG, "Received [power] button trigger request");

	if (dbus_message_get_args(msg, &error,
				  DBUS_TYPE_BOOLEAN, &result,
				  DBUS_TYPE_INVALID) == FALSE) {
		// XXX: should we return an error instead?
		mce_log(LL_CRIT,
			"Failed to get argument from %s: %s",
			MCE_TRIGGER_POWERKEY_EVENT_REQ,
			error.message);
		dbus_error_free(&error);
		goto EXIT;
	}

	mce_log(LL_DEBUG, "[power] button event trigger value: %d", result);

	/* TRUE == long key press, FALSE == short key press */
	if (result == TRUE) {
		handle_longpress();
	} else {
		handle_shortpress();
	}

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

EXIT:
	return status;
}

/**
 * D-Bus callback from SystemUI powerkeymenu
 *
 * @todo the calls to confirm_{normal,flight}mode need error handling
 *
 * @param msg D-Bus message with the return value
 * @return TRUE on success, FALSE on failure
 */
static gboolean systemui_device_menu_dbus_cb(DBusMessage *const msg)
{
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	dbus_int32_t result = INT_MAX;
	gboolean status = FALSE;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	mce_log(LL_DEBUG, "Received device menu callback");

	if (dbus_message_get_args(msg, &error,
				  DBUS_TYPE_INT32, &result,
				  DBUS_TYPE_INVALID) == FALSE) {
		// XXX: should we return an error instead?
		mce_log(LL_CRIT,
			"Failed to get argument from %s: %s",
			MCE_POWERKEY_CB_REQ,
			error.message);
		dbus_error_free(&error);
		goto EXIT;
	}

	mce_log(LL_DEBUG, "Device menu callback value: %d", result);

	/* Close the powerkey menu */
	device_menu(FALSE);

	switch (result) {
	case POWER_KEY_MENU_RESPONSE_TKLOCK:
		/* Request enabling of touchscreen/keypad lock */
		execute_datapipe(&tk_lock_pipe,
				 GINT_TO_POINTER(LOCK_ON),
				 FALSE, TRUE);
		break;

	case POWER_KEY_MENU_RESPONSE_DEVICELOCK:
		/* Request enabling of device lock */
		execute_datapipe(&device_lock_pipe,
				 GINT_TO_POINTER(LOCK_ON),
				 FALSE, TRUE);
		break;

	case POWER_KEY_MENU_RESPONSE_NORMALMODE:
		/* normal mode */
		// FIXME: error handling?
		confirm_normalmode();
		break;

	case POWER_KEY_MENU_RESPONSE_FLIGHTMODE:
		/* flight mode */
		// FIXME: error handling?
		confirm_flightmode();
		break;

	case POWER_KEY_MENU_RESPONSE_REBOOT:
		/* reboot */
		request_reboot();
		break;

	case POWER_KEY_MENU_RESPONSE_SOFT_POWEROFF:
		/* soft poweroff */
		request_soft_poweroff();
		break;

	case POWER_KEY_MENU_RESPONSE_POWEROFF:
		/* shutdown */
		request_normal_shutdown();
		break;

	case 0:
	default:
		/* aborted menu; no action necessary */
		break;
	}

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

EXIT:
	return status;
}

/**
 * D-Bus callback from SystemUI modechange dialog
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean systemui_modechange_dbus_cb(DBusMessage *const msg)
{
	dbus_int32_t result = INT_MAX;
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	gboolean status = FALSE;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	mce_log(LL_DEBUG, "Received modechange callback");

	if (dbus_message_get_args(msg, &error,
				  DBUS_TYPE_INT32, &result,
				  DBUS_TYPE_INVALID) == FALSE) {
		// XXX: should we return an error instead?
		mce_log(LL_CRIT,
			"Failed to get argument from %s: %s",
			MCE_MODECHG_CB_REQ,
			error.message);
		dbus_error_free(&error);
		goto EXIT;
	}

	mce_log(LL_DEBUG, "Modechange callback value: %d", result);

	switch (result) {
	case MODECHANGE_RESPONSE_OK:
		if (modetransition == MODECHANGE_TO_FLIGHTMODE)
			mce_set_mode_int32(MCE_FLIGHT_MODE_INT32);
		else
			mce_set_mode_int32(MCE_NORMAL_MODE_INT32);

		break;

	case MODECHANGE_RESPONSE_CANCEL:
	default:
		/* Don't change the mode */
		break;
	}

	mce_rem_submode_int32(MCE_MODECHG_SUBMODE);

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

EXIT:
	return status;
}

/**
 * Handle system state change
 *
 * @param data The system state stored in a pointer
 */
static void system_state_trigger(gconstpointer data)
{
	system_state_t system_state = GPOINTER_TO_INT(data);

	switch (system_state) {
	case MCE_STATE_SHUTDOWN:
	case MCE_STATE_ACTDEAD:
	case MCE_STATE_REBOOT:
		close_confirmmode();

	default:
		break;
	}
}

/**
 * Handle system mode change
 *
 * @param data Unused
 */
static void system_mode_trigger(gconstpointer data)
{
	submode_t submode = datapipe_get_gint(submode_pipe);

	(void)data;

	/* If the device menu is open and the mode changes,
	 * we need to reopen the menu to make sure that the mode
	 * is shown correctly
	 */
	if ((submode & MCE_DEVMENU_SUBMODE) == MCE_DEVMENU_SUBMODE) {
		/* Reopen the device menu
		 * XXX: handle errors!
		 */
		(void)device_menu(TRUE);
	}
}

/**
 * Datapipe trigger for the [power] key
 *
 * @param data A pointer to the input_event struct
 */
static void powerkey_trigger(gconstpointer data)
{
        system_state_t system_state = datapipe_get_gint(system_state_pipe);
	struct input_event const *const *evp = data;
	struct input_event const *ev = *evp;

	if (ev->code == power_keycode) {
		/* If set, the [power] key was pressed */
		if (ev->value == 1) {
			mce_log(LL_DEBUG, "[power] pressed");

			/* Remove old timeout */
			if (powerkey_timeout_cb_id != 0)
				g_source_remove(powerkey_timeout_cb_id);

			/* Setup new timeout */
			if ((system_state == MCE_STATE_ACTDEAD) ||
			    (mce_get_submode_int32() & MCE_SOFTOFF_SUBMODE)) {
				/* Shorter delay for startup
				 * than for shutdown
				 */
				powerkey_timeout_cb_id =
					g_timeout_add(mediumdelay,
						      powerkey_timeout_cb,
						      NULL);
			} else {
				powerkey_timeout_cb_id =
					g_timeout_add(longdelay,
						      powerkey_timeout_cb,
						      NULL);
			}
		} else if (ev->value == 0) {
			/* Short key press */
			if (powerkey_timeout_cb_id != 0) {
				handle_shortpress();
			}
		}
	}
}

/**
 * Parse the [power] action string
 *
 * @param string The string to parse
 * @param action A pointer to the variable to store the action in
 * @return TRUE if the string contained a valid action,
 *         FALSE if the action was invalid
 */
static gboolean parse_action(char *string, poweraction_t *action)
{
	gboolean status = FALSE;

	/** @todo This should probably not be hardcoded */
	if (!strcmp(string, "disabled")) {
		*action = POWER_DISABLED;
	} else if (!strcmp(string, "menu")) {
		*action = POWER_MENU;
	} else if (!strcmp(string, "poweroff")) {
		*action = POWER_POWEROFF;
	} else if (!strcmp(string, "softpoweroff")) {
		*action = POWER_SOFT_POWEROFF;
	} else {
		mce_log(LL_WARN,
			"Unknown [power] action");
		goto EXIT;
	}

	status = TRUE;

EXIT:
	return status;
}

/**
 * Init function for the powerkey component
 *
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_powerkey_init(void)
{
	gboolean status = FALSE;
	gchar *tmp = NULL;

	/* close down the powerkey menu on init to make sure MCE
	 * doesn't end up in a confused mode if it's restarted
	 */
	device_menu(FALSE);

	/* Append triggers/filters to datapipes */
	append_input_trigger_to_datapipe(&keypress_pipe,
					 powerkey_trigger);
	append_output_trigger_to_datapipe(&system_state_pipe,
					  system_state_trigger);
	append_output_trigger_to_datapipe(&mode_pipe,
					  system_mode_trigger);

	initialised = 1;

	/* req_trigger_powerkey_event */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_TRIGGER_POWERKEY_EVENT_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 trigger_powerkey_event_req_dbus_cb) == FALSE)
		goto EXIT;

	/* powerkey_callback */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_POWERKEY_CB_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 systemui_device_menu_dbus_cb) == FALSE)
		goto EXIT;

	/* modechange_callback */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_MODECHG_CB_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 systemui_modechange_dbus_cb) == FALSE)
		goto EXIT;

	/* Get configuration options */
	longdelay = mce_conf_get_int(MCE_CONF_POWERKEY_GROUP,
				     MCE_CONF_POWERKEY_LONG_DELAY,
				     DEFAULT_POWER_LONG_DELAY,
				     NULL);
	mediumdelay = mce_conf_get_int(MCE_CONF_POWERKEY_GROUP,
				       MCE_CONF_POWERKEY_MEDIUM_DELAY,
				       DEFAULT_POWER_MEDIUM_DELAY,
				       NULL);
	tmp = mce_conf_get_string(MCE_CONF_POWERKEY_GROUP,
				  MCE_CONF_POWERKEY_SHORT_ACTION,
				  "",
				  NULL);

	/* Since we've set a default, error handling is unnecessary */
	(void)parse_action(tmp, &shortpressaction);
	g_free(tmp);

	tmp = mce_conf_get_string(MCE_CONF_POWERKEY_GROUP,
				  MCE_CONF_POWERKEY_LONG_ACTION,
				  "",
				  NULL);

	/* Since we've set a default, error handling is unnecessary */
	(void)parse_action(tmp, &longpressaction);
	g_free(tmp);

	doublepressdelay = mce_conf_get_int(MCE_CONF_POWERKEY_GROUP,
					    MCE_CONF_POWERKEY_DOUBLE_DELAY,
					    DEFAULT_POWER_DOUBLE_DELAY,
					    NULL);
	tmp = mce_conf_get_string(MCE_CONF_POWERKEY_GROUP,
				  MCE_CONF_POWERKEY_DOUBLE_ACTION,
				  "",
				  NULL);

	/* Since we've set a default, error handling is unnecessary */
	(void)parse_action(tmp, &doublepressaction);
	g_free(tmp);

	status = TRUE;

EXIT:
	return status;
}

/**
 * Exit function for the powerkey component
 *
 * @todo D-Bus unregistration
 */
void mce_powerkey_exit(void)
{
	/* Append triggers/filters to datapipes */
	remove_output_trigger_from_datapipe(&mode_pipe,
					    system_mode_trigger);
	remove_output_trigger_from_datapipe(&system_state_pipe,
					    system_state_trigger);
	remove_input_trigger_from_datapipe(&keypress_pipe,
					   powerkey_trigger);

	// FIXME: there are probably better ways to do this...
	if (initialised == TRUE)
		device_menu(FALSE);

	/* Remove the timeout source for the [power] long key press handler */
	if (powerkey_timeout_cb_id != 0) {
		g_source_remove(powerkey_timeout_cb_id);
		powerkey_timeout_cb_id = 0;
	}

	/* Remove the timeout source for the [power] double key press handler */
	if (doublepress_timeout_cb_id != 0) {
		g_source_remove(doublepress_timeout_cb_id);
		doublepress_timeout_cb_id = 0;
	}
}
