/**
 * @file inactivity.c
 * Inactivity module -- this implements inactivity logic for MCE
 * <p>
 * Copyright © 2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>
#include <gmodule.h>

#include "mce.h"

#include "mce-log.h"		/* mce_log(), LL_* */
#include "mce-dbus.h"		/* mce_dbus_handler_add(),
				 * dbus_send_message(),
				 * dbus_new_method_reply(),
				 * dbus_new_signal(),
				 * dbus_message_append_args(),
				 * dbus_message_unref(),
				 * DBusMessage,
				 * DBUS_MESSAGE_TYPE_METHOD_CALL,
				 * DBUS_TYPE_BOOLEAN,
				 * DBUS_TYPE_INVALID,
				 * dbus_bool_t
				 */
#include "datapipe.h"		/* datapipe_get_gbool(),
				 * append_output_trigger_to_datapipe(),
				 * remove_output_trigger_from_datapipe()
				 */

/** Module name */
#define MODULE_NAME		"inactivity"

static const gchar *const provides[] = { MODULE_NAME, NULL };

/** Module information */
G_MODULE_EXPORT module_info_struct module_info = {
	/** Name of the module */
	.name = MODULE_NAME,
	/** Module provides */
	.provides = provides,
	/** Module priority */
	.priority = 250
};

/**
 * Send an inactivity status reply or signal
 *
 * @param method_call A DBusMessage to reply to;
 *                    pass NULL to send an inactivity status signal instead
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_send_inactivity_status(DBusMessage *const method_call)
{
	DBusMessage *msg = NULL;
	dbus_bool_t device_inactive = datapipe_get_gbool(device_inactive_pipe);
	gboolean status = FALSE;

	mce_log(LL_DEBUG,
		"Sending inactivity status: %s",
		device_inactive ? "inactive" : "active");

	/* If method_call is set, send a reply,
	 * otherwise, send a signal
	 */
	if (method_call != NULL) {
		msg = dbus_new_method_reply(method_call);
	} else {
		/* system_inactivity_ind */
		msg = dbus_new_signal(MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
				      MCE_INACTIVITY_SIG);
	}

	/* Append the inactivity status */
	if (dbus_message_append_args(msg,
				     DBUS_TYPE_BOOLEAN, &device_inactive,
				     DBUS_TYPE_INVALID) == FALSE) {
		mce_log(LL_CRIT,
			"Failed to append %sargument to D-Bus message for %s",
			method_call ? "reply " : "",
			method_call ? MCE_INACTIVITY_STATUS_GET :
				      MCE_INACTIVITY_SIG);
		dbus_message_unref(msg);
		goto EXIT;
	}

	/* Send the message */
	status = dbus_send_message(msg);

EXIT:
	return status;
}

/**
 * D-Bus callback for the get inactivity status method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean inactivity_status_get_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	mce_log(LL_DEBUG, "Received inactivity status get request");

	/* Try to send a reply that contains the current inactivity status */
	if (mce_send_inactivity_status(msg) == FALSE)
		goto EXIT;

	status = TRUE;

EXIT:
	return status;
}

/**
 * Inactivity output trigger
 *
 * @param data inactivity signal
 */
static void inactivity_output_trigger(gconstpointer data)
{
	(void)data;

	mce_send_inactivity_status(NULL);
}

/**
 * Init function for the inactivity module
 *
 * @param module unused
 * @return NULL on success, a string with an error message on failure
 */
G_MODULE_EXPORT const gchar *g_module_check_init(GModule *module);
const gchar *g_module_check_init(GModule *module)
{
	(void)module;

	/* Append triggers/filters to datapipes */
	append_output_trigger_to_datapipe(&device_inactive_pipe,
					  inactivity_output_trigger);

	/* get_inactivity_status */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_INACTIVITY_STATUS_GET,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 inactivity_status_get_dbus_cb) == FALSE)
		goto EXIT;

EXIT:
	return NULL;
}

/**
 * Exit function for the inactivity module
 *
 * @todo D-Bus unregistration
 * @param module unused
 */
G_MODULE_EXPORT void g_module_unload(GModule *module);
void g_module_unload(GModule *module)
{
	(void)module;

	/* Remove triggers/filters from datapipes */
	remove_output_trigger_from_datapipe(&device_inactive_pipe,
					    inactivity_output_trigger);
}
