/**
 * @file homekey.c
 * [home] key module for the Mode Control Entity
 * <p>
 * Copyright © 2004-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>
#include <gmodule.h>

#include <string.h>			/* strcmp() */
#include <linux/input.h>		/* struct input_event, KEY_F5 */

#include "mce.h"
#include "homekey.h"

#include "mce-log.h"			/* mce_log(), LL_* */
#include "mce-conf.h"			/* mce_conf_get_int(),
					 * mce_conf_get_string()
					 */
#include "mce-dbus.h"			/* dbus_send(),
					 * DBUS_TYPE_INVALID
					 */
#include "datapipe.h"			/* datapipe_get_gint(),
					 * append_output_trigger_to_datapipe(),
					 * remove_output_trigger_from_datapipe()
					 */

/** Module name */
#define MODULE_NAME		"homekey"

static const gchar *const provides[] = { MODULE_NAME, NULL };

/** Module information */
G_MODULE_EXPORT module_info_struct module_info = {
	/** Name of the module */
	.name = MODULE_NAME,
	/** Module provides */
	.provides = provides,
	/** Module priority */
	.priority = 250
};

/** KEY_F5 == [home] key on the Nokia 770, N800, and N810 */
static guint16 home_keycode = KEY_F5;

/** Time in milliseconds before the key press is considered long */
static gint longdelay = DEFAULT_HOME_LONG_DELAY;
/** Action to perform on a short key press */
static homeaction_t shortpressaction = DEFAULT_HOMEKEY_SHORT_ACTION;
/** Action to perform on a long key press */
static homeaction_t longpressaction = DEFAULT_HOMEKEY_LONG_ACTION;

/**
 * The ID of the timeout used when determining
 * whether the keypress was short or long
 */
static guint homekey_timeout_cb_id = 0;

/**
 * Send the [home] key signal
 * The signal will only be sent in normal submode/user state
 *
 * @param longpress TRUE if the keypress was long,
 *                  FALSE if the keypress was short
 */
static void send_home_key_signal(const gboolean longpress)
{
	system_state_t system_state = datapipe_get_gint(system_state_pipe);

	if ((system_state == MCE_STATE_USER) &&
	    (mce_get_submode_int32() == MCE_NORMAL_SUBMODE)) {
		// FIXME: error handling?
		dbus_send(NULL, MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
			  longpress == TRUE ? MCE_HOME_KEY_LONG_SIG :
					      MCE_HOME_KEY_SIG,
			  NULL,
			  DBUS_TYPE_INVALID);
	}

	// XXX: Is this necessary?
	mce_rem_submode_int32(MCE_EVEATER_SUBMODE);
}

/**
 * Timeout callback for long keypress
 *
 * @param data Unused
 * @return Always returns FALSE, to disable the timeout
 */
static gboolean homekey_timeout_cb(gpointer data)
{
	(void)data;

	homekey_timeout_cb_id = 0;

	switch (longpressaction) {
	case HOME_DISABLED:
		break;

	case HOME_SHORTSIGNAL:
		send_home_key_signal(FALSE);
		break;

	case HOME_LONGSIGNAL:
	default:
		send_home_key_signal(TRUE);
		break;
	}

	return FALSE;
}

/**
 * Datapipe trigger for the [home] key
 *
 * @param data A pointer to the input_event struct
 */
static void homekey_trigger(gconstpointer data)
{
	struct input_event const *const *evp = data;
	struct input_event const *ev = *evp;

	if (ev->code == home_keycode) {
		/* If set, the [home] key was pressed */
		if (ev->value == 1) {
			mce_log(LL_DEBUG, "[home] pressed");

			/* Remove old timeout */
			if (homekey_timeout_cb_id != 0)
				g_source_remove(homekey_timeout_cb_id);

			/* Setup new timeout */
			homekey_timeout_cb_id =
				g_timeout_add(longdelay,
					      homekey_timeout_cb,
					      NULL);
		} else if (ev->value == 0) {
			/* Short keypress */

			/* Only send a signal if there's a pending timer;
			 * if there's no pending timer, it has expired and
			 * triggered a long keypress signal to be sent
			 */
			if (homekey_timeout_cb_id != 0) {
				g_source_remove(homekey_timeout_cb_id);
				homekey_timeout_cb_id = 0;

				switch (shortpressaction) {
				case HOME_DISABLED:
					break;

				case HOME_SHORTSIGNAL:
				default:
					send_home_key_signal(FALSE);
					break;

				case HOME_LONGSIGNAL:
					send_home_key_signal(TRUE);
					break;
				}
			}
		}
	}
}

/**
 * Parse the [home] action string
 *
 * @param string The string to parse
 * @param action A pointer to the variable to store the action in
 * @return TRUE if the string contained a valid action,
 *         FALSE if the action was invalid
 */
static gboolean parse_action(char *string, homeaction_t *action)
{
	gboolean status = FALSE;

	/** @todo This should probably not be hardcoded */
	if (!strcmp(string, "disabled")) {
		*action = HOME_DISABLED;
	} else if (!strcmp(string, "shortsignal")) {
		*action = HOME_SHORTSIGNAL;
	} else if (!strcmp(string, "longsignal")) {
		*action = HOME_LONGSIGNAL;
	} else {
		mce_log(LL_WARN,
			"Unknown [home] action");
		goto EXIT;
	}

	status = TRUE;

EXIT:
	return status;
}

/**
 * Init function for the [home] key module
 *
 * @param module unused
 * @return NULL on success, a string with an error message on failure
 */
G_MODULE_EXPORT const gchar *g_module_check_init(GModule *module);
const gchar *g_module_check_init(GModule *module)
{
	gchar *tmp = NULL;

	(void)module;

	/* Append triggers/filters to datapipes */
	append_input_trigger_to_datapipe(&keypress_pipe,
					 homekey_trigger);

	/* Get configuration options */
	longdelay = mce_conf_get_int(MCE_CONF_HOMEKEY_GROUP,
				     MCE_CONF_HOMEKEY_LONG_DELAY,
				     DEFAULT_HOME_LONG_DELAY,
				     NULL);
	tmp = mce_conf_get_string(MCE_CONF_HOMEKEY_GROUP,
				  MCE_CONF_HOMEKEY_SHORT_ACTION,
				  "",
				  NULL);

	/* Since we've set a default, error handling is unnecessary */
	(void)parse_action(tmp, &shortpressaction);
	g_free(tmp);

	tmp = mce_conf_get_string(MCE_CONF_HOMEKEY_GROUP,
				  MCE_CONF_HOMEKEY_LONG_ACTION,
				  "",
				  NULL);

	/* Since we've set a default, error handling is unnecessary */
	(void)parse_action(tmp, &longpressaction);
	g_free(tmp);

	return NULL;
}

/**
 * Exit function for the [home] key module
 *
 * @param module unused
 */
G_MODULE_EXPORT void g_module_unload(GModule *module);
void g_module_unload(GModule *module)
{
	(void)module;

	/* Remove triggers/filters from datapipes */
	remove_input_trigger_from_datapipe(&keypress_pipe,
					   homekey_trigger);

	/* Remove timeout sources */
	if (homekey_timeout_cb_id != 0) {
		g_source_remove(homekey_timeout_cb_id);
		homekey_timeout_cb_id = 0;
	}
}
