/**
 * @file filter-brightness-simple.c
 * Simple level adjusting brightness filter module
 * for display backlight brightness
 * This file implements a filter module for MCE
 * <p>
 * Copyright © 2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>
#include <gmodule.h>

#include "mce.h"
#include "filter-brightness-simple.h"

#include "datapipe.h"			/* append_filter_to_datapipe(),
					 * remove_filter_from_datapipe(),
					 * execute_datapipe()
					 */

/** Module name */
#define MODULE_NAME		"filter-brightness-simple"

static const gchar *const enhances[] = {
	"display-brightness",
	NULL
};
static const gchar *const provides[] = {
	"display-brightness-filter",
	NULL
};

/** Module information */
G_MODULE_EXPORT module_info_struct module_info = {
	/** Name of the module */
	.name = MODULE_NAME,
	/** Module dependencies */
	.enhances = enhances,
	/** Module provides */
	.provides = provides,
	/** Module priority */
	.priority = 250
};

/** Display state */
static display_state_t display_state = MCE_DISPLAY_UNDEF;

/**
 * Simple level adjustment filter for display brightness
 *
 * @param data The un-processed brightness value (1-5)
 * @return The processed brightness value (0-255)
 */
static gpointer display_brightness_filter(gpointer data)
{
	/** DSME imposed limit */
	static gint maximum_display_brightness = 255;
	gint raw = GPOINTER_TO_INT(data);
	gpointer retval;

	/* If the display is off, don't update its brightness */
	if (display_state == MCE_DISPLAY_OFF) {
		raw = 0;
		goto EXIT;
	}

	/* Safety net */
	if (raw < DISPLAY_BRIGHTNESS_MINIMUM)
		raw = DISPLAY_BRIGHTNESS_MINIMUM;
	else if (raw > DISPLAY_BRIGHTNESS_MAXIMUM)
		raw = DISPLAY_BRIGHTNESS_MAXIMUM;

	raw = (maximum_display_brightness * raw) / 5;

EXIT:
	retval = GINT_TO_POINTER(raw);

	return retval;
}

/**
 * Init function for the simple level-adjusting brightness filter
 *
 * @param module unused
 * @return NULL on success, a string with an error message on failure
 */
G_MODULE_EXPORT const gchar *g_module_check_init(GModule *module);
const gchar *g_module_check_init(GModule *module)
{
	(void)module;

	/* Append triggers/filters to datapipes */
	append_filter_to_datapipe(&display_brightness_pipe,
				  display_brightness_filter);

	/* Re-filter the brightness */
	(void)execute_datapipe(&display_brightness_pipe, NULL, TRUE, FALSE);

	return NULL;
}

/**
 * Exit function for the simple level-adjusting brightness filter
 *
 * @param module unused
 */
G_MODULE_EXPORT void g_module_unload(GModule *module);
void g_module_unload(GModule *module)
{
	(void)module;

	/* Remove triggers/filters from datapipes */
	remove_filter_from_datapipe(&display_brightness_pipe,
				    display_brightness_filter);
}
