/**
 * @file filter-brightness-als.c
 * Ambient Light Sensor level adjusting filter module
 * for display backlight, key backlight, and LED brightness
 * This file implements a filter module for MCE
 * <p>
 * Copyright © 2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 * @author Semi Malinen <semi.malinen@nokia.com>
 */
#include <glib.h>
#include <gmodule.h>

#include "mce.h"
#include "filter-brightness-als.h"

#include "mce-io.h"			/* mce_read_number_string_from_file() */
#include "datapipe.h"			/* execute_datapipe(),
					 * append_output_trigger_to_datapipe(),
					 * append_filter_to_datapipe(),
					 * remove_filter_from_datapipe(),
					 * remove_output_trigger_from_datapipe()
					 */

/** Module name */
#define MODULE_NAME		"filter-brightness-als"

static const gchar *const enhances[] = {
	"display-brightness",
	"led-brightness",
	"key-backlight-brightness",
	NULL
};
static const gchar *const provides[] = {
	"display-brightness-filter",
	"led-brightness-filter",
	"key-backlight-brightness-filter",
	NULL
};

/** Module information */
G_MODULE_EXPORT module_info_struct module_info = {
	/** Name of the module */
	.name = MODULE_NAME,
	/** Module dependencies */
	.enhances = enhances,
	/** Module provides */
	.provides = provides,
	/** Module priority */
	.priority = 100
};

static gboolean als_enabled = TRUE;	/**< Filter things through ALS? */
static gint als_lux = -1;		/**< Lux reading from the ALS */

/** Display state */
static display_state_t display_state = MCE_DISPLAY_UNDEF;

/** Median filter */
static median_filter_struct median_filter;

/**
 * Poll every 2s for changes when display is on,
 * and every 60s when display is off
 */
static gint als_poll_timeout = 2;

/** ID for ALS poll timer source */
static guint als_poll_timeout_cb_id = 0;

/**
 * Initialise median filter

 * @param filter The median filter to initialise
 * @param window_size The window size to use
 */
static void median_filter_init(median_filter_struct *filter,
			       gsize window_size)
{
	guint i;

	filter->window_size = window_size;

	for (i = 0; i < filter->window_size; i++) {
		filter->window[i] = 0;
		filter->ordered_window[i] = 0;
	}

	filter->samples = 0;
	filter->oldest = 0;
}

/**
 * Insert a new sample into the median filter
 *
 * @param filter The median filter to insert the value into
 * @param value The value to insert
 * @param oldest The oldest value
 * @return The filtered value
 */
static gint insert_ordered(median_filter_struct *filter,
			   gint value, gint oldest)
{
	guint i;

	/* If the filter window hasn't been filled yet, insert the new value */
	if (filter->samples < filter->window_size) {
		/* Find insertion point */
		for (i = 0; i < filter->samples; i++) {
			if (filter->ordered_window[i] >= value) {
				/* Found the insertion point */
				for ( ; i < filter->samples; ++i) {
					gint tmp;

					/* Swap the value at insertion point
					 * with the new value
					 */
					tmp = filter->ordered_window[i];
					filter->ordered_window[i] = value;
					value = tmp;
				}

				break;
			}
		}

		/* Do the insertion */
		filter->ordered_window[i] = value;
		filter->samples++;

		goto EXIT;
	} else {
		/* The filter window is full;
		 * remove the oldest value and insert new
		 */
		if (value == oldest) {
			/* Do nothing */
			goto EXIT;
		}

		/* Find either the insertion point
		 * and/or the deletion point
		 */
		for (i = 0; i < filter->window_size; i++) {
			if (filter->ordered_window[i] >= value) {
				/* Found the insertion point
				 * (it might be the deletion point
				 * as well!)
				 */
				for ( ; i < filter->window_size; i++) {
					int tmp;

					/* Swap value at insertion
					 * point and the new value
					 */
					tmp = filter->ordered_window[i];
					filter->ordered_window[i] = value;
					value = tmp;

					if (value == oldest) {
						/* Found the deletion point */
						goto EXIT;
					}
				}

				goto EXIT;
			} else if (filter->ordered_window[i] == oldest) {
				/* Found the deletion point */
				for ( ; i < filter->window_size - 1; i++) {
					if (filter->ordered_window[i + 1] >= value) {
						/* Found the insertion point */
						break;
					}
					/* Shift the window,
					 * overwriting the value to delete
					 */
					filter->ordered_window[i] = filter->ordered_window[i + 1];
				}
				/* Insert */
				filter->ordered_window[i] = value;
				goto EXIT;
			}
		}
	}

EXIT:
	/* For odd number of samples return the middle one
	 * For even number of samples return the average
	 * of the two middle ones
	 */
	return (filter->ordered_window[(filter->samples - 1) / 2] +
		filter->ordered_window[filter->samples / 2]) / 2;
}

/**
 * Do a complete insertion of a sample into the median filter
 *
 * @param filter The median filter to insert the value into
 * @param value The value to insert
 * @return The filtered value
 */
static gint median_filter_map(median_filter_struct *filter, gint value)
{
  gint filtered_value;

  /* Insert into the ordered array (deleting the oldest value) */
  filtered_value = insert_ordered(filter, value,
				  filter->window[filter->oldest]);

  /* Insert into the ring buffer (overwriting the oldest value) */
  filter->window[filter->oldest] = value;
  filter->oldest = (filter->oldest + 1) % filter->window_size;

  return filtered_value;
}

/**
 * Use the ALS profiles to calculate proper ALS modified values
 *
 * @param profiles The profile struct to use for calculations
 * @param profile The profile to use
 * @param lux The lux value
 * @param[in,out] level The old level; will be replaced by the new level
 * @return The brightness in % of maximum
 */
static gint filter_data(als_profile_struct *profiles, als_profile_t profile,
			gint lux, gint *level)
{
	gint tmp = *level;

	if (tmp == -1)
		tmp = 0;
	else if (tmp > 5)
		tmp = 5;

	if ((profiles[profile].range[4][0] != -1) &&
	    (lux > profiles[profile].range[4][((5 - tmp) > 0) ? 1 : 0]))
		*level = 5;
	else if ((profiles[profile].range[3][0] != -1) &&
		 (lux > profiles[profile].range[3][((4 - tmp) > 0) ? 1 : 0]))
		*level = 4;
	else if ((profiles[profile].range[2][0] != -1) &&
		 (lux > profiles[profile].range[2][((3 - tmp) > 0) ? 1 : 0]))
		*level = 3;
	else if ((profiles[profile].range[1][0] != -1) &&
		 (lux > profiles[profile].range[1][((2 - tmp) > 0) ? 1 : 0]))
		*level = 2;
	else if ((profiles[profile].range[0][0] != -1) &&
		 (lux > profiles[profile].range[0][((1 - tmp) > 0) ? 1 : 0]))
		*level = 1;
	else
		*level = 0;

	return profiles[profile].value[*level];
}

/**
 * Ambient Light Sensor filter for display brightness
 *
 * @param data The un-processed brightness value (1-5)
 * @return The processed brightness value (0-255)
 */
static gpointer display_brightness_filter(gpointer data)
{
	/** DSME imposed limit */
	static gint maximum_display_brightness = 255;
	/** Display ALS level */
	static gint display_als_level = -1;
	gint raw = GPOINTER_TO_INT(data) - 1;
	gpointer retval;

	/* If the display is off, don't update its brightness */
	if (display_state == MCE_DISPLAY_OFF) {
		raw = 0;
		goto EXIT;
	}

	/* Safety net */
	if (raw < ALS_PROFILE_MINIMUM)
		raw = ALS_PROFILE_MINIMUM;
	else if (raw > ALS_PROFILE_MAXIMUM)
		raw = ALS_PROFILE_MAXIMUM;

	if (als_enabled == TRUE) {
		gint percentage = filter_data(display_als_profiles, raw,
					      als_lux, &display_als_level);

		raw = (maximum_display_brightness * percentage) / 100;
	} else {
		raw = (maximum_display_brightness * (raw + 1)) / 5;
	}

EXIT:
	retval = GINT_TO_POINTER(raw);

	return retval;
}

/**
 * Ambient Light Sensor filter for LED brightness
 *
 * @param data The un-processed brightness value
 * @return The processed brightness value
 */
static gpointer led_brightness_filter(gpointer data)
{
	// XXX: Hardcoded for now, since there's nowhere to read this setting
	static gint normal_led_backlight_brightness = 2;
	/** LED ALS level */
	static gint led_als_level = -1;
	gint raw = GPOINTER_TO_INT(data);
	gpointer retval;

	/* If the led is disabled, don't modify the value */
	if (raw == -1)
		goto EXIT;

	if (als_enabled == TRUE) {
		gint percentage = filter_data(led_als_profiles,
					      ALS_PROFILE_NORMAL,
					      als_lux, &led_als_level);
		raw = (normal_led_backlight_brightness * percentage) / 100;
	} else {
		raw = normal_led_backlight_brightness;
	}

EXIT:
	retval = GINT_TO_POINTER(raw);
	return retval;
}

/**
 * Ambient Light Sensor filter for keyboard backlight brightness
 *
 * @param data The un-processed brightness value
 * @return The processed brightness value
 */
static gpointer key_backlight_filter(gpointer data)
{
	// XXX: Hardcoded for now, since there's nowhere to read this setting
	static gint maximum_key_backlight_brightness = 255;
	/** Keyboard ALS level */
	static gint kbd_als_level = -1;
	gint raw = GPOINTER_TO_INT(data);
	gpointer retval;

	/* If the backlight is disabled, don't modify the value */
	if (raw == 0)
		goto EXIT;

	if (als_enabled == TRUE) {
		gint percentage = filter_data(kbd_als_profiles,
					      ALS_PROFILE_NORMAL,
					      als_lux, &kbd_als_level);
		raw = (maximum_key_backlight_brightness * percentage) / 100;
	} else {
		raw = maximum_key_backlight_brightness;
	}

EXIT:
	retval = GINT_TO_POINTER(raw);

	return retval;
}

/**
 * Read a value from the ALS and update the median filter
 *
 * @return the filtered result of the read,
 *         -1 on failure,
 *         -2 if the ALS is disabled
 */
static gint als_read_value_filtered(void)
{
	gulong tmp;
	gint filtered_read = -2;

	if (als_enabled == FALSE)
		goto EXIT;

	/* Read lux value from ALS */
	if (mce_read_number_string_from_file(ALS_LUX_PATH, &tmp) == TRUE) {
		filtered_read = median_filter_map(&median_filter, tmp);
	} else {
		filtered_read = -1;
	}

EXIT:
	return filtered_read;
}

/**
 * Poll the ALS
 *
 * @param data Unused
 * @return Always returns TRUE, for continuous polling,
           unless the ALS is disabled
 */
static gboolean als_poll_timeout_cb(gpointer data)
{
	gboolean status = FALSE;
	gint old_lux;
	(void)data;

	old_lux = als_lux;

	/* Read lux value from ALS */
	if ((als_lux = als_read_value_filtered()) == -2)
		goto EXIT;

	/* There's no point in readjusting the brightness
	 * if the ambient light did not change or the read failed
	 */
	if ((als_lux == -1) || (als_lux == old_lux))
		goto EXIT2;

	/* Re-filter the brightness */
	(void)execute_datapipe(&display_brightness_pipe, NULL, TRUE, FALSE);
	(void)execute_datapipe(&led_brightness_pipe, NULL, TRUE, FALSE);
	(void)execute_datapipe(&key_backlight_pipe, NULL, TRUE, FALSE);

EXIT2:
	status = TRUE;

EXIT:
	return status;
}

/**
 * Handle display state change
 *
 * @param data The display stated stored in a pointer
 */
static void display_state_cb(gconstpointer data)
{
	display_state_t old_display_state =
				datapipe_get_old_gint(display_state_pipe);
	display_state = GPOINTER_TO_INT(data);

	/* Disable old timer */
	if (als_poll_timeout_cb_id != 0) {
		g_source_remove(als_poll_timeout_cb_id);
		als_poll_timeout_cb_id = 0;
	}

	if (als_enabled == FALSE)
		goto EXIT;

	/* Update poll timeout */
	switch (display_state) {
	case MCE_DISPLAY_OFF:
		als_poll_timeout = ALS_DISPLAY_OFF_POLL_FREQ;
		break;

	case MCE_DISPLAY_DIM:
		als_poll_timeout = ALS_DISPLAY_DIM_POLL_FREQ;
		break;

	case MCE_DISPLAY_UNDEF:
	case MCE_DISPLAY_ON:
	default:
		als_poll_timeout = ALS_DISPLAY_ON_POLL_FREQ;
		break;
	}

#ifdef ALS_DISPLAY_OFF_FLUSH_FILTER
	/* Re-fill the median filter */
	if (((old_display_state == MCE_DISPLAY_OFF) ||
	     (old_display_state == MCE_DISPLAY_UNDEF)) &&
	    ((display_state == MCE_DISPLAY_ON) ||
	     (display_state == MCE_DISPLAY_DIM))) {
		/* Re-initialise the median filter */
		median_filter_init(&median_filter, MEDIAN_FILTER_WINDOW_SIZE);

		/* Read lux value from ALS */
		if ((als_lux = als_read_value_filtered()) >= 0) {
			/* Re-filter the brightness */
			(void)execute_datapipe(&display_brightness_pipe,
					       NULL, TRUE, FALSE);
			(void)execute_datapipe(&led_brightness_pipe,
					       NULL, TRUE, FALSE);
			(void)execute_datapipe(&key_backlight_pipe,
					       NULL, TRUE, FALSE);
		}
	}
#endif /* ALS_DISPLAY_OFF_FLUSH_FILTER */

	/* Setup new timer */
	if (als_poll_timeout != 0) {
		als_poll_timeout_cb_id = g_timeout_add(als_poll_timeout,
						       als_poll_timeout_cb,
						       NULL);
	}

EXIT:
	return;
}

/**
 * Init function for the ALS filter
 *
 * @param module unused
 * @return NULL on success, a string with an error message on failure
 */
G_MODULE_EXPORT const gchar *g_module_check_init(GModule *module);
const gchar *g_module_check_init(GModule *module)
{
	(void)module;

	/* Use 1:1 mappings until we have registered a handler
	 * and made the first reading
	 */
	als_enabled = FALSE;

	/* Append triggers/filters to datapipes */
	append_filter_to_datapipe(&display_brightness_pipe,
				  display_brightness_filter);
	append_filter_to_datapipe(&led_brightness_pipe,
				  led_brightness_filter);
	append_filter_to_datapipe(&key_backlight_pipe,
				  key_backlight_filter);
	append_output_trigger_to_datapipe(&display_state_pipe,
					  display_state_cb);

	/* Initialise the median filter */
	median_filter_init(&median_filter, MEDIAN_FILTER_WINDOW_SIZE);

	/* Enable reads */
	als_enabled = TRUE;

	/* Initial read of lux value from ALS */
	if ((als_lux = als_read_value_filtered()) >= 0) {
		/* Setup ALS polling */
		als_poll_timeout_cb_id =
			g_timeout_add(als_poll_timeout * 1000,
				      als_poll_timeout_cb, NULL);
	} else {
		/* We don't have an ALS */
		als_lux = -1;
		als_enabled = FALSE;
	}

	/* Re-filter the brightness */
	(void)execute_datapipe(&display_brightness_pipe, NULL, TRUE, FALSE);
	(void)execute_datapipe(&led_brightness_pipe, NULL, TRUE, FALSE);
	(void)execute_datapipe(&key_backlight_pipe, NULL, TRUE, FALSE);

	return NULL;
}

/**
 * Exit function for the ALS filter
 *
 * @param module unused
 */
G_MODULE_EXPORT void g_module_unload(GModule *module);
void g_module_unload(GModule *module)
{
	(void)module;

	als_enabled = FALSE;

	/* Remove the timeout source for the ALS poling */
	if (als_poll_timeout_cb_id != 0) {
		g_source_remove(als_poll_timeout_cb_id);
		als_poll_timeout_cb_id = 0;
	}

	/* Remove triggers/filters from datapipes */
	remove_output_trigger_from_datapipe(&display_state_pipe,
					    display_state_cb);
	remove_filter_from_datapipe(&key_backlight_pipe,
				    key_backlight_filter);
	remove_filter_from_datapipe(&led_brightness_pipe,
				    led_brightness_filter);
	remove_filter_from_datapipe(&display_brightness_pipe,
				    display_brightness_filter);
}
