/**
 * @file camera.c
 * Camera module -- this handles the camera LED-indicator for MCE
 * <p>
 * Copyright © 2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>
#include <gmodule.h>

#include <string.h>			/* strncmp(), strlen() */

#include "mce.h"
#include "camera.h"

#include "mce-io.h"			/* mce_register_io_monitor_string(),
					 * mce_unregister_io_monitor()
					 */
#include "mce-conf.h"			/* mce_conf_get_bool() */
#include "datapipe.h"			/* execute_datapipe_output_triggers() */

/** Unlock the tklock if the camera is popped out? */
static gboolean popout_unlock = DEFAULT_CAMERA_POPOUT_UNLOCK;

/* Used by camera pop-out */
G_MODULE_IMPORT void unlock_tklock_delayed(void);

/** Module name */
#define MODULE_NAME		"camera"

static const gchar *const depends[] = { "tklock", NULL };
static const gchar *const recommends[] = { "led", NULL };
static const gchar *const provides[] = { MODULE_NAME, NULL };

/** Module information */
G_MODULE_EXPORT module_info_struct module_info = {
	/** Name of the module */
	.name = MODULE_NAME,
	/** Module dependencies */
	.depends = depends,
	/** Module provides */
	.provides = provides,
	/** Module priority */
	.priority = 250
};

/** ID for the camera active state I/O monitor */
static gconstpointer camera_active_state_iomon_id = NULL;
/** ID for the camera pop-out state I/O monitor */
static gconstpointer camera_popout_state_iomon_id = NULL;

/**
 * I/O monitor callback for the camera active state
 *
 * @param data The new data
 * @param bytes_read Unused
 */
static void camera_active_state_cb(gpointer data, gsize bytes_read)
{
	(void)bytes_read;

	if (!strncmp(data, MCE_CAMERA_ACTIVE, strlen(MCE_CAMERA_ACTIVE))) {
		execute_datapipe_output_triggers(&led_pattern_activate_pipe,
						 MCE_LED_PATTERN_CAMERA, FALSE);
	} else {
		execute_datapipe_output_triggers(&led_pattern_deactivate_pipe,
						 MCE_LED_PATTERN_CAMERA, FALSE);
	}
}

/**
 * I/O monitor callback for the camera pop-out state
 *
 * @param data The new data
 * @param bytes_read Unused
 */
static void camera_popout_state_cb(gpointer data, gsize bytes_read)
{
	(void)bytes_read;

	if (popout_unlock == FALSE)
		goto EXIT;

	/* Unlock tklock if camera is popped out */
	if (!strncmp(data, MCE_CAMERA_POPPED_OUT,
		     strlen(MCE_CAMERA_POPPED_OUT))) {
		/* Request delayed unlock of touchscreen/keypad lock */
		execute_datapipe(&tk_lock_pipe,
				 GINT_TO_POINTER(LOCK_OFF_DELAYED),
				 FALSE, TRUE);
	}

EXIT:
	return;
}

/**
 * Init function for the camera module
 *
 * @param module unused
 * @return NULL on success, a string with an error message on failure
 */
G_MODULE_EXPORT const gchar *g_module_check_init(GModule *module);
const gchar *g_module_check_init(GModule *module)
{
	(void)module;

	/* Get configuration options */
	popout_unlock = mce_conf_get_bool(MCE_CONF_TKLOCK_GROUP,
					  MCE_CONF_CAMERA_POPOUT_UNLOCK,
					  DEFAULT_CAMERA_POPOUT_UNLOCK,
					  NULL);

	/* Register I/O monitors */
	// FIXME: error handling?
	camera_active_state_iomon_id =
		mce_register_io_monitor_string(-1, CAMERA_ACTIVE_STATE_PATH,
					       FALSE, TRUE,
					       camera_active_state_cb);

	camera_popout_state_iomon_id =
		mce_register_io_monitor_string(-1, CAMERA_POPOUT_STATE_PATH,
					       FALSE, TRUE,
					       camera_popout_state_cb);

//EXIT:
	return NULL;
}

/**
 * Exit function for the inactivity module
 *
 * @param module unused
 */
G_MODULE_EXPORT void g_module_unload(GModule *module);
void g_module_unload(GModule *module)
{
	(void)module;

	/* Unregister I/O monitors */
	mce_unregister_io_monitor(camera_popout_state_iomon_id);
	mce_unregister_io_monitor(camera_active_state_iomon_id);
}
