/**
 * @file alarm.c
 * Alarm interface module for the Mode Control Entity
 * <p>
 * Copyright © 2005-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>
#include <gmodule.h>

#include <string.h>			/* strcmp() */

#include <mce/mode-names.h>		/* MCE_ALARM_VISIBLE,
					 * MCE_ALARM_SNOOZED,
					 * MCE_ALARM_OFF
					 */

#include "mce.h"

#include "mce-log.h"			/* mce_log(), LL_* */
#include "mce-dbus.h"			/* mce_dbus_handler_add(),
					 * dbus_send_message(),
					 * dbus_new_method_call(),
					 * dbus_new_method_reply(),
					 * dbus_message_get_args(),
					 * dbus_message_append_args(),
					 * dbus_message_unref(),
					 * dbus_error_init(),
					 * dbus_error_free(),
					 * DBUS_MESSAGE_TYPE_METHOD_CALL,
					 * DBUS_TYPE_BOOLEAN,
					 * DBUS_TYPE_INT32,
					 * DBUS_TYPE_UINT64,
					 * DBUS_TYPE_ARRAY,
					 * DBUS_TYPE_INVALID,
					 * DBusMessage, DBusError,
					 * dbus_int32_t, dbus_int64_t
					 */
#include "datapipe.h"			/* execute_datapipe() */

/** Module name */
#define MODULE_NAME		"alarm"

static const gchar *const provides[] = { MODULE_NAME, NULL };

/** Module information */
G_MODULE_EXPORT module_info_struct module_info = {
	/** Name of the module */
	.name = MODULE_NAME,
	/** Module provides */
	.provides = provides,
	/** Module priority */
	.priority = 250
};

/**
 * D-Bus callback for the alarm ui state method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean set_alarm_ui_state_dbus_cb(DBusMessage *const msg)
{
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	alarm_ui_state_t alarm_ui_state = MCE_ALARM_UI_INVALID_INT32;
	gboolean status = FALSE;
	gchar *mode = NULL;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	mce_log(LL_DEBUG, "Received set alarm mode request");

	if (dbus_message_get_args(msg, &error,
				  DBUS_TYPE_STRING, &mode,
				  DBUS_TYPE_INVALID) == FALSE) {
		// XXX: should we return an error instead?
		mce_log(LL_CRIT,
			"Failed to get argument from %s: %s",
			MCE_ALARM_MODE_CHANGE_REQ,
			error.message);
		dbus_error_free(&error);
		goto EXIT;
	}

	/* Convert alarm state to enum */
	if (strcmp(MCE_ALARM_VISIBLE, mode) == 0) {
		alarm_ui_state = MCE_ALARM_UI_VISIBLE_INT32;
	} else if (strcmp(MCE_ALARM_SNOOZED, mode) == 0) {
		alarm_ui_state = MCE_ALARM_UI_SNOOZED_INT32;
	} else if (strcmp(MCE_ALARM_OFF, mode) == 0) {
		alarm_ui_state = MCE_ALARM_UI_OFF_INT32;
	} else {
		mce_log(LL_DEBUG, "Invalid alarm state received");
		goto EXIT;
	}

	(void)execute_datapipe(&alarm_ui_state_pipe,
			       GINT_TO_POINTER(alarm_ui_state), FALSE, TRUE);

	/* If alarm is visible, set alarm submode */
	if (alarm_ui_state == MCE_ALARM_UI_VISIBLE_INT32) {
		mce_add_submode_int32(MCE_ALARM_SUBMODE);
	} else {
		mce_rem_submode_int32(MCE_ALARM_SUBMODE);
	}

	status = TRUE;

EXIT:
	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

	return status;
}

/**
 * Init function for the alarm interface module
 *
 * @param module unused
 * @return NULL on success, a string with an error message on failure
 */
G_MODULE_EXPORT const gchar *g_module_check_init(GModule *module);
const gchar *g_module_check_init(GModule *module)
{
	(void)module;

	/* set_alarm_mode */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_ALARM_MODE_CHANGE_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 set_alarm_ui_state_dbus_cb) == FALSE)
		goto EXIT;

EXIT:
	return NULL;
}

/**
 * Exit function for the alarm interface module
 *
 * @param module unused
 */
G_MODULE_EXPORT void g_module_unload(GModule *module);
void g_module_unload(GModule *module)
{
	(void)module;
}
