/**
 * @file modetransition.c
 * This file implements the mode transition component
 * (normal/flight mode/(suspend/)off)
 * of the Mode Control Entity
 * <p>
 * Copyright © 2004-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>

#include <errno.h>			/* errno, ENOENT */
#include <stdlib.h>			/* exit(), EXIT_FAILURE */
#include <string.h>			/* strcmp() */
#include <unistd.h>			/* access(), F_OK */

#include <mce/mode-names.h>
#include <systemui/dbus-names.h>
#include <systemui/actingdead-dbus-names.h>
#include <systemui/splashscreen-dbus-names.h>

#include "mce.h"
#include "modetransition.h"

#include "mce-io.h"			/* mce_read_string_from_file(),
					 * mce_write_string_to_file()
					 */
#include "mce-log.h"			/* mce_log(), LL_* */
#include "mce-dbus.h"			/* mce_dbus_handler_add(),
					 * dbus_send(),
					 * dbus_send_message(),
					 * dbus_new_signal(),
					 * dbus_new_method_reply(),
					 * dbus_message_get_no_reply(),
					 * dbus_message_get_args(),
					 * dbus_message_append_args(),
					 * dbus_message_unref(),
					 * dbus_error_init(),
					 * dbus_error_free(),
					 * DBUS_MESSAGE_TYPE_METHOD_CALL,
					 * DBUS_MESSAGE_TYPE_SIGNAL,
					 * DBUS_TYPE_BOOLEAN,
					 * DBUS_TYPE_UINT32,
					 * DBUS_TYPE_STRING,
					 * DBUS_TYPE_INVALID,
					 * DBusMessage, DBusError,
					 * dbus_bool_t,
					 * dbus_uint32_t
					 */
#include "datapipe.h"			/* execute_datapipe(),
					 * execute_datapipe_output_triggers(),
					 * datapipe_get_gint(),
					 * append_filter_to_datapipe(),
					 * append_output_trigger_to_datapipe(),
					 * remove_filter_from_datapipe(),
					 * remove_output_trigger_from_datapipe()
					 */

/** Mode mapping structure */
typedef struct {
	const system_mode_t number;	/**< Mode ID */
	const gchar *const name;	/**< Mode name */
} mce_mode_t;

/** Mapping of mode ID <-> mode name */
static const mce_mode_t mce_mode_names[] = {
	{
		.number = MCE_NORMAL_MODE_INT32,
		.name = MCE_NORMAL_MODE
	}, {
		.number = MCE_FLIGHT_MODE_INT32,
		.name = MCE_FLIGHT_MODE
	}, {
		.number = MCE_OFFLINE_MODE_INT32,
		.name = MCE_OFFLINE_MODE
	}, {
		.number = MCE_VOIP_MODE_INT32,
		.name = MCE_VOIP_MODE
	}, { /* MCE_INVALID_MODE_INT32 also marks the end of this array */
		.number = MCE_INVALID_MODE_INT32,
		.name = MCE_INVALID_MODE
	}
};

static system_mode_t mce_mode = MCE_NORMAL_MODE_INT32;	/**< System mode */

static guint powerup_timeout_cb_id = 0;	/**< Timeout for powerup splash */
static guint splash_timeout_cb_id = 0;	/**< Timeout for splash screen */
static guint actdead_timeout_cb_id = 0;	/**< Timeout for acting dead screen */

/**
 * Enable/disable the powerup splashscreen
 *
 * @param enable TRUE to enable splashscreen, FALSE to disable splashscreen
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_powerup_splash(const gboolean enable)
{
	const dbus_uint32_t splashtype = SPLASHSCREEN_ENABLE_BOOTUP;

	mce_log(LL_DEBUG, "Calling bootup splashscreen (%d)", enable);

	/* com.nokia.system_ui.request.splashscreen_{open,close} */
	return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
			 SYSTEMUI_REQUEST_IF,
			 enable ? SYSTEMUI_SPLASHSCREEN_OPEN_REQ :
				  SYSTEMUI_SPLASHSCREEN_CLOSE_REQ, NULL,
			 DBUS_TYPE_UINT32, &splashtype,
			 DBUS_TYPE_INVALID);
}

/**
 * Enable/disable the shutdown splashscreen
 *
 * @param enable TRUE to enable splashscreen, FALSE to disable splashscreen
 * @param sound TRUE to play splashsound, FALSE to inhibit splashsound
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_shutdown_splash(const gboolean enable,
				    const dbus_bool_t sound)
{
	const dbus_uint32_t splashtype = SPLASHSCREEN_ENABLE_SHUTDOWN;

	mce_log(LL_DEBUG, "Calling shutdown splashscreen (%d)", enable);

	/* com.nokia.system_ui.request.splashscreen_{open,close} */
	return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
			 SYSTEMUI_REQUEST_IF,
			 enable ? SYSTEMUI_SPLASHSCREEN_OPEN_REQ :
				  SYSTEMUI_SPLASHSCREEN_CLOSE_REQ, NULL,
			 DBUS_TYPE_UINT32, &splashtype,
			 DBUS_TYPE_BOOLEAN, &sound,
			 DBUS_TYPE_INVALID);
}

/**
 * Enable/disable the acting dead UI
 *
 * @param enable TRUE to enable acting dead UI, FALSE to disable acting dead UI
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_actdead(const gboolean enable)
{
	mce_log(LL_DEBUG, "Calling acting dead UI (%d)", enable);

	/* com.nokia.system_ui.request.acting_dead_{open,close} */
	return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
			 SYSTEMUI_REQUEST_IF,
			 enable ? SYSTEMUI_ACTINGDEAD_OPEN_REQ :
				  SYSTEMUI_ACTINGDEAD_CLOSE_REQ, NULL,
			 DBUS_TYPE_INVALID);
}

/**
 * Timeout callback for powerup splashscreen
 *
 * @param data Unused
 * @return Always returns FALSE, to disable the timeout
 */
static gboolean powerup_timeout_cb(gpointer data)
{
	(void)data;

	powerup_timeout_cb_id = 0;

	if (mce_powerup_splash(FALSE) == FALSE) {
		g_main_loop_quit(mainloop);
		exit(EXIT_FAILURE);
	}

	execute_datapipe_output_triggers(&led_pattern_deactivate_pipe, MCE_LED_PATTERN_POWER_ON, FALSE);

	/* FALSE here is just to disable the timeout */
	return FALSE;
}

/**
 * Timeout callback for splashscreen disabling
 *
 * @param data Unused
 * @return Always returns FALSE, to disable the timeout
 */
static gboolean splash_timeout_cb(gpointer data)
{
	(void)data;

	splash_timeout_cb_id = 0;

	if (mce_shutdown_splash(FALSE, FALSE) == FALSE) {
		g_main_loop_quit(mainloop);
		exit(EXIT_FAILURE);
	}

	execute_datapipe_output_triggers(&led_pattern_deactivate_pipe, MCE_LED_PATTERN_POWER_OFF, FALSE);

	/* FALSE here is just to disable the timeout */
	return FALSE;
}

/**
 * Timeout callback for acting dead UI enabling;
 * sets a timeout for the shutdown splashscreen
 *
 * @param data Unused
 * @return Always returns FALSE, to disable the timeout
 */
static gboolean actdead_timeout_cb(gpointer data)
{
	(void)data;

	actdead_timeout_cb_id = 0;

	if (mce_actdead(TRUE) == FALSE) {
		g_main_loop_quit(mainloop);
		exit(EXIT_FAILURE);
	}

	splash_timeout_cb_id = g_timeout_add(ACTDEAD_DELAY,
					     splash_timeout_cb, NULL);

	/* FALSE here is just to disable the timeout */
	return FALSE;
}

/**
 * Set the MCE submode flags
 *
 * @param submode All submodes to set OR:ed together
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_set_submode_int32(const submode_t submode)
{
	execute_datapipe(&submode_pipe, GINT_TO_POINTER(submode), FALSE, TRUE);
	mce_log(LL_DEBUG, "Submode changed to %d", submode);

	return TRUE;
}

/**
 * Add flags to the MCE submode
 *
 * @param submode submode(s) to add OR:ed together
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_add_submode_int32(const submode_t submode)
{
	submode_t old_submode = datapipe_get_gint(submode_pipe);

	return mce_set_submode_int32(old_submode | submode);
}

/**
 * Remove flags from the MCE submode
 *
 * @param submode submode(s) to remove OR:ed together
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_rem_submode_int32(const submode_t submode)
{
	submode_t old_submode = datapipe_get_gint(submode_pipe);

	return mce_set_submode_int32(old_submode & ~submode);
}

/**
 * Return all set MCE submode flags
 *
 * @return All set submode flags OR:ed together
 */
submode_t mce_get_submode_int32(void)
{
	submode_t submode = datapipe_get_gint(submode_pipe);

	return submode;
}

/**
 * Return the integer representation of the MCE device mode
 *
 * @return The integer representation of the MCE device mode
 */
system_mode_t mce_get_mode_int32(void)
{
	return mce_mode;
}

/**
 * Return the string representation of the MCE device mode
 *
 * @return A string representation of the MCE device mode
 */
static const gchar *mce_get_mode_string(void)
{
	const gchar *mode;
	gint i = 0;

	/* This might seem awkward, but it's made to allow sparse
	 * number spaces
	 */
	do {
		mode = mce_mode_names[i].name;
	} while (mce_mode_names[i].number != MCE_INVALID_MODE_INT32 &&
		 mce_mode_names[i++].number != mce_mode);

	return mode;
}

/**
 * Send the MCE device mode
 *
 * @param method_call A DBusMessage to reply to;
 *                    pass NULL to send a device mode signal instead
 * @param mode A string representation of an alternate mode to send instead
 *             of the real device mode
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_send_mode(DBusMessage *const method_call, const gchar *const mode)
{
	DBusMessage *msg = NULL;
	const gchar *smode;
	gboolean status = FALSE;

	/* Allow mode spoofing, and send "normal" instead of "voip" */
	if (mode != NULL)
		smode = mode;
	else if (mce_get_mode_int32() == MCE_VOIP_MODE_INT32)
		smode = MCE_NORMAL_MODE;
	else
		smode = mce_get_mode_string();

	/* If method_call is set, send a reply,
	 * otherwise, send a signal
	 */
	if (method_call != NULL) {
		msg = dbus_new_method_reply(method_call);
	} else {
		/* sig_device_mode_ind */
		msg = dbus_new_signal(MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
				      MCE_DEVICE_MODE_SIG);
	}

	/* Append the mode */
	if (dbus_message_append_args(msg,
				     DBUS_TYPE_STRING, &smode,
				     DBUS_TYPE_INVALID) == FALSE) {
		mce_log(LL_CRIT,
			"Failed to append %sargument to D-Bus message for %s",
			method_call ? "reply " : "",
			method_call ? MCE_DEVICE_MODE_GET :
				      MCE_DEVICE_MODE_SIG);
		dbus_message_unref(msg);
		goto EXIT;
	}

	/* Send the message */
	status = dbus_send_message(msg);

EXIT:
	return status;
}

/**
 * Save the MCE device mode to persistant storage
 *
 * @param mode A string representation of the MCE device mode
 *             to save to persistant storage
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_save_mode(const gchar *const mode)
{
	return mce_write_string_to_file(MCE_MODE_FILENAME, mode);
}

/**
 * Set the MCE device mode
 *
 * @param mode The integer representation of the MCE device mode
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_set_mode_int32(const system_mode_t mode)
{
	gboolean status = FALSE;

	if (mode == MCE_INVALID_MODE_INT32) {
		mce_log(LL_INFO, "mce_set_mode_int32 called with invalid mode");
		goto EXIT;
	}

	if (mode == MCE_VOIP_MODE_INT32) {
		mce_send_mode(NULL, MCE_NORMAL_MODE);
	} else {
		mce_send_mode(NULL, mce_mode_names[mode].name);
	}

	mce_log(LL_INFO, "Mode changed to %s", mce_mode_names[mode].name);

	/* Post mode set policies:
	 * o Store the mode for persistance over reboot;
	 *   however, VoIP-mode is replaced by normal mode
	 * o For VoIP-mode, disable the device autolock
	 */
	if (mode == MCE_VOIP_MODE_INT32) {
		/* Disable the device lock option in device menu;
		 * the autolock signal from dsme will be ignored
		 * by the mce-dsme statemachine, and does not need
		 * to be disabled here
		 */
		(void)mce_write_string_to_file(MCE_DEVLOCK_FILENAME,
					       DISABLED_STRING);

		if (mce_mode != mode && mce_mode != MCE_NORMAL_MODE_INT32)
			status = mce_save_mode(MCE_NORMAL_MODE);
		else
			status = TRUE;
	} else {
		(void)mce_write_string_to_file(MCE_DEVLOCK_FILENAME,
					       ENABLED_STRING);

		if (mce_mode != mode)
			status = mce_save_mode(mce_mode_names[mode].name);
		else
			status = TRUE;
	}

	/* Keep this last; if we fail to inform others about
	 * the mode change, don't perform the mode change at all
	 */
	mce_mode = mode;
	execute_datapipe(&mode_pipe, GINT_TO_POINTER(mode), FALSE, TRUE);

EXIT:
	return status;
}

/**
 * Set the MCE device mode
 *
 * @param mode A string representation of a MCE device mode
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_set_mode(const gchar *const mode)
{
	system_mode_t newmode = MCE_INVALID_MODE_INT32;
	gint i = 0;

	while (mce_mode_names[i].number != MCE_INVALID_MODE_INT32) {
		/* If the modename matches, set newmode and stop searching */
		if (strcmp(mce_mode_names[i].name, mode) == 0) {
			newmode = mce_mode_names[i].number;
			break;
		}

		i++;
	}

	return mce_set_mode_int32(newmode);
}

/**
 * Start SystemUI
 */
void mce_startup_ui(void)
{
	system_state_t system_state = datapipe_get_gint(system_state_pipe);

	if (system_state == MCE_STATE_ACTDEAD) {
		if (mce_actdead(TRUE) == FALSE) {
			g_main_loop_quit(mainloop);
			exit(EXIT_FAILURE);
		}
	}

	execute_datapipe(&system_state_pipe, NULL, TRUE, TRUE);
}

/**
 * D-Bus callback for device mode change method call
 *
 * @todo Decide on error handling policy
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean mode_change_req_dbus_cb(DBusMessage *const msg)
{
	system_state_t system_state = datapipe_get_gint(system_state_pipe);
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	gboolean status = FALSE;
	gchar *mode = NULL;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	mce_log(LL_DEBUG, "Received mode change request");

	if (dbus_message_get_args(msg, &error,
				  DBUS_TYPE_STRING, &mode,
				  DBUS_TYPE_INVALID) == FALSE) {
		// XXX: should we return an error instead?
		mce_log(LL_CRIT,
			"Failed to get argument from %s: %s",
			MCE_DEVICE_MODE_CHANGE_REQ,
			error.message);
		dbus_error_free(&error);
		goto EXIT;
	}

	/* Try to change to the requested device mode,
	 * but only if we're in USER-state
	 * XXX: right now we silently ignore such
	 * requests; should we return an error?
	 */
	if (system_state == MCE_STATE_USER) {
		if (mce_set_mode(mode) == FALSE)
			goto EXIT;
	}

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

EXIT:
	return status;
}

/**
 * D-Bus callback for the get device mode method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean mode_get_req_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	mce_log(LL_DEBUG, "Received mode get request");

	/* Try to send a reply that contains the current mode */
	if (mce_send_mode(msg, NULL) == FALSE)
		goto EXIT;

	status = TRUE;

EXIT:
	return status;
}

/**
 * D-Bus callback for the SystemUI started signal
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean systemui_started_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	(void)msg;

	mce_log(LL_DEBUG, "Received SystemUI startup indication");

	mce_startup_ui();

	status = TRUE;

//EXIT:
	return status;
}

/**
 * Restore the MCE device mode from persistant storage
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_restore_mode(void)
{
	gchar *mode = NULL;
	gboolean status;

	if (mce_read_string_from_file(MCE_MODE_FILENAME, &mode) == FALSE) {
		/* On error, default to flight mode */
		status = mce_set_mode_int32(MCE_FLIGHT_MODE_INT32);
	} else {
		int i = 0;

		/* Terminate the string at the first non alnum */
		while (g_ascii_isalnum(mode[i]) != 0)
			i++;

		mode[i] = '\0';

		status = mce_set_mode(mode);
		g_free(mode);
	}

	return status;
}

/**
 * Filter out VOIP-mode, since it is a transparent mode
 *
 * @param data The unfiltered mode
 * @return The filtered mode
 */
static gpointer mode_filter(gpointer data)
{
	gint raw = GPOINTER_TO_INT(data);
	gpointer retval = data;

	switch (raw) {
	case MCE_INVALID_MODE_INT32:
	/* MCE_OFFLINE_MODE_INT32: */
	case MCE_FLIGHT_MODE_INT32:
		retval = GINT_TO_POINTER(MCE_FLIGHT_MODE_INT32);
		break;

	case MCE_NORMAL_MODE_INT32:
	case MCE_VOIP_MODE_INT32:
	default:
		retval = GINT_TO_POINTER(MCE_NORMAL_MODE_INT32);
		break;
	}

	return retval;
}

/**
 * Handle system state change
 *
 * @param data The system state stored in a pointer
 */
static void system_state_trigger(gconstpointer data)
{
	system_state_t old_system_state = datapipe_get_old_gint(system_state_pipe);
	system_state_t system_state = GPOINTER_TO_INT(data);

	switch (system_state) {
	case MCE_STATE_USER:
		if (old_system_state == MCE_STATE_ACTDEAD) {
			/* If the device isn't locked, show splashscreen */
			if ((mce_get_submode_int32() &
			     MCE_DEVLOCK_SUBMODE) == 0) {
				if (mce_powerup_splash(TRUE) == FALSE) {
					g_main_loop_quit(mainloop);
					exit(EXIT_FAILURE);
				}

				execute_datapipe_output_triggers(&led_pattern_activate_pipe, MCE_LED_PATTERN_POWER_ON, FALSE);

				powerup_timeout_cb_id =
					g_timeout_add(POWERUP_DELAY,
						      powerup_timeout_cb,
						      NULL);
			}

			/* Disable acting dead UI */
			if (mce_actdead(FALSE) == FALSE) {
				g_main_loop_quit(mainloop);
				exit(EXIT_FAILURE);
			}
		}

		mce_rem_submode_int32(MCE_BOOTUP_SUBMODE);
		break;

	case MCE_STATE_SHUTDOWN:
	case MCE_STATE_ACTDEAD:
	case MCE_STATE_REBOOT:
		if (old_system_state == MCE_STATE_USER) {
			if (mce_shutdown_splash(TRUE, TRUE) == FALSE) {
				g_main_loop_quit(mainloop);
				exit(EXIT_FAILURE);
			}

			execute_datapipe_output_triggers(&led_pattern_deactivate_pipe, MCE_LED_PATTERN_DEVICE_ON, FALSE);
			execute_datapipe_output_triggers(&led_pattern_activate_pipe, MCE_LED_PATTERN_POWER_OFF, FALSE);
		}

		/* If we're shutting down or rebooting from acting dead,
		 * blank the screen
		 */
		if ((old_system_state == MCE_STATE_ACTDEAD) &&
		    ((system_state == MCE_STATE_SHUTDOWN) ||
		     (system_state == MCE_STATE_REBOOT))) {
			execute_datapipe(&display_state_pipe,
					 GINT_TO_POINTER(MCE_DISPLAY_OFF),
					 FALSE, TRUE);
		}

		break;

	case MCE_STATE_UNDEF:
		mce_log(LL_ERR, "Invalid state received from DSME");
		goto EXIT;

	default:
		break;
	}

	if (system_state == MCE_STATE_ACTDEAD) {
		actdead_timeout_cb_id = g_timeout_add(ACTDEAD_DELAY,
						      actdead_timeout_cb,
						      NULL);
	}

	mce_log(LL_DEBUG,
		"dsmestate set to: %d (old: %d)",
		system_state, old_system_state);

EXIT:
	return;
}

/**
 * Init function for the modetransition component
 *
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_mode_init(void)
{
	gboolean status = FALSE;

	/* Append triggers/filters to datapipes */
	append_filter_to_datapipe(&mode_pipe, mode_filter);
	append_output_trigger_to_datapipe(&system_state_pipe,
					  system_state_trigger);

	/* If the call file exists, we have crashed / restarted;
	 * since it exists in /var/run it will be removed when we reboot
	 */
	if (access(MCE_DEVLOCK_FILENAME, F_OK) == -1) {
		if (errno == ENOENT) {
			mce_log(LL_DEBUG, "Bootup mode enabled");
			mce_add_submode_int32(MCE_BOOTUP_SUBMODE);
			mce_add_submode_int32(MCE_TRANSITION_SUBMODE);
			errno = 0;
		} else {
			mce_log(LL_CRIT,
				"access() failed: %s. Exiting.",
				g_strerror(errno));
			goto EXIT;
		}
	}

	mce_restore_mode();

	/* req_device_mode_change */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_DEVICE_MODE_CHANGE_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 mode_change_req_dbus_cb) == FALSE)
		goto EXIT;

	/* get_device_mode */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_DEVICE_MODE_GET,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 mode_get_req_dbus_cb) == FALSE)
		goto EXIT;

	/* system_ui_started */
	if (mce_dbus_handler_add(SYSTEMUI_SIGNAL_IF,
				 SYSTEMUI_STARTED_SIG,
				 NULL,
				 DBUS_MESSAGE_TYPE_SIGNAL,
				 systemui_started_dbus_cb) == FALSE)
		goto EXIT;

	status = TRUE;

EXIT:
	return status;
}

/**
 * Exit function for the modetransition component
 *
 * @todo D-Bus unregistration
 */
void mce_mode_exit(void)
{
	/* Remove triggers/filters from datapipes */
	remove_output_trigger_from_datapipe(&system_state_pipe,
					    system_state_trigger);
	remove_filter_from_datapipe(&mode_pipe, mode_filter);

	/* Remove the timeout source for the powerup splashscreen enabling */
	if (powerup_timeout_cb_id != 0) {
		g_source_remove(powerup_timeout_cb_id);
		powerup_timeout_cb_id = 0;
	}

	/* Remove the timeout source for the splashscreen disabling */
	if (splash_timeout_cb_id != 0) {
		g_source_remove(splash_timeout_cb_id);
		splash_timeout_cb_id = 0;
	}

	/* Remove the timeout source for the acting dead UI enabling */
	if (actdead_timeout_cb_id != 0) {
		g_source_remove(actdead_timeout_cb_id);
		actdead_timeout_cb_id = 0;
	}
}
