/**
 * @file mce.h
 * Generic headers for Mode Control Entity
 * <p>
 * Copyright © 2004-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#ifndef _MCE_H_
#define _MCE_H_

#include <glib.h>
#include <locale.h>

#include "datapipe.h"

#ifdef ENABLE_NLS
#include <libintl.h>
/** _() to use when NLS is enabled */
#define _(__str)		gettext(__str)
#else
#undef bindtextdomain
/** Dummy bindtextdomain to use when NLS is disabled */
#define bindtextdomain(__domain, __directory)
#undef textdomain
/** Dummy textdomain to use when NLS is disabled */
#define textdomain(__domain)
/** Dummy _() to use when NLS is disabled */
#define _(__str)		__str
#endif /* ENABLE_NLS */

/** Names of LED patterns */
#define MCE_LED_PATTERN_POWER_ON		"PatternPowerOn"
#define MCE_LED_PATTERN_POWER_OFF		"PatternPowerOff"
#define MCE_LED_PATTERN_CAMERA			"PatternWebcamActive"
#define MCE_LED_PATTERN_DEVICE_ON		"PatternDeviceOn"
#define MCE_LED_PATTERN_BATTERY_CHARGING	"PatternBatteryCharging"
#define MCE_LED_PATTERN_BATTERY_FULL		"PatternBatteryFull"
#define MCE_LED_PATTERN_BATTERY_LOW		"PatternBatteryLow"

/** Module information */
typedef struct {
/** Name of the module */
	const gchar *const name;
/** Module dependencies */
	const gchar *const *const depends;
/** Module recommends */
	const gchar *const *const recommends;
/** Module provides */
	const gchar *const *const provides;
/** Module provides */
	const gchar *const *const enhances;
/** Module conflicts */
	const gchar *const *const conflicts;
/** Module replaces */
	const gchar *const *const replaces;
/** Module priority:
 * lower value == higher priority
 * This value is only used when modules conflict
 */
	const gint priority;
} module_info_struct;

/** The GMainLoop used by MCE */
GMainLoop *mainloop;

/** System Modes; integer representations
 *
 * Note:
 *  - Offline mode is treated as an alias for flight mode
 *  - VoIP mode is transparent; the system will officially
 *    remain in normal mode.  If you need the raw value,
 *    you need to connect to attach an input trigger instead
 *    of an output trigger
 */
typedef enum {
	/** Mode not valid */
	MCE_INVALID_MODE_INT32 = -1,
	/** Normal mode */
	MCE_NORMAL_MODE_INT32 = 0,
	/** Flight mode */
	MCE_FLIGHT_MODE_INT32 = 1,
	/** Offline mode; alias for flight */
	MCE_OFFLINE_MODE_INT32 = 1,
	/** VoIP mode */
	MCE_VOIP_MODE_INT32 = 3
} system_mode_t;

/** Alarm UI states; integer representations
 */
typedef enum {
	MCE_ALARM_UI_INVALID_INT32 = -1,	/**< Alarm UI state not valid */
	MCE_ALARM_UI_OFF_INT32 = 0,		/**< Alarm UI not visible */
	MCE_ALARM_UI_SNOOZED_INT32 = 1,		/**< Alarm UI snoozed */
	MCE_ALARM_UI_VISIBLE_INT32 = 2,		/**< Alarm UI visible */
} alarm_ui_state_t;

/* System sub-modes; several of these can be active at once */
typedef gint submode_t;

/** No submodes enabled */
#define MCE_NORMAL_SUBMODE	0
/** Touchscreen/Keypad lock enabled */
#define MCE_TKLOCK_SUBMODE	(1 << 0)
/** Device lock enabled */
#define MCE_DEVLOCK_SUBMODE	(1 << 1)
/** Bootup in progress */
#define MCE_BOOTUP_SUBMODE	(1 << 2)
/** Modechange dialog shown */
#define MCE_MODECHG_SUBMODE	(1 << 3)
/** Device menu shown */
#define MCE_DEVMENU_SUBMODE	(1 << 4)
/** Event eater enabled */
#define MCE_EVEATER_SUBMODE	(1 << 5)
/** Device emulates soft poweroff */
#define MCE_SOFTOFF_SUBMODE	(1 << 6)
/** Alarm UI is visible */
#define MCE_ALARM_SUBMODE	(1 << 7)
/** State transition in progress */
#define MCE_TRANSITION_SUBMODE	(1 << 8)
/** Devlock verification in progress */
#define MCE_VERIFY_SUBMODE	(1 << 9)

typedef enum {
	MCE_STATE_UNDEF = -1,		/**< System state not set */
	MCE_STATE_SHUTDOWN = 0,		/**< System is in shutdown state */
	MCE_STATE_USER = 2,		/**< System is in user state */
	MCE_STATE_ACTDEAD = 5,		/**< System is in acting dead state */
	MCE_STATE_REBOOT = 6,		/**< System is in reboot state */
	MCE_STATE_BOOT = 9		/**< System is in bootup state */
} system_state_t;

typedef enum {
	MCE_DISPLAY_UNDEF = -1,		/**< Display state not set */
	MCE_DISPLAY_OFF	= 0,		/**< Display is off */
	MCE_DISPLAY_DIM = 1,		/**< Display is dimmed */
	MCE_DISPLAY_ON = 2		/**< Display is on */
} display_state_t;

typedef enum {
	COVER_UNDEF = -1,		/**< Cover state not set */
	COVER_CLOSED = 0,		/**< Cover is closed */
	COVER_OPEN = 1			/**< Cover is open */
} cover_state_t;

typedef enum {
	LOCK_UNDEF = -1,		/**< Lock state not set */
	LOCK_OFF = 0,			/**< Lock is disabled */
	LOCK_OFF_DELAYED = 1,		/**< Delayed unlock; write only */
	LOCK_OFF_SILENT = 2,		/**< Silent unlock; write only */
	LOCK_ON = 3,			/**< Lock is enabled */
	LOCK_ON_DIMMED = 4,		/**< Dimmed lock; write only */
	LOCK_ON_SILENT = 5,		/**< Silent lock; write only */
	LOCK_ON_SILENT_DIMMED = 6,	/**< Silent dimmed lock; write only */
	LOCK_TOGGLE = 7			/**< Toggle lock state; write only */
} lock_state_t;

datapipe_struct led_brightness_pipe;	/**< LED brightness */
datapipe_struct device_inactive_pipe;	/**< State of device; read only */
/** LED pattern to activate; read only */
datapipe_struct led_pattern_activate_pipe;
/** LED pattern to deactivate; read only */
datapipe_struct led_pattern_deactivate_pipe;
datapipe_struct display_state_pipe;	/**< State of display; read only */
/** Display brightness */
datapipe_struct display_brightness_pipe;
datapipe_struct blanking_inhibit_pipe;	/**< Blanking inhibit */
datapipe_struct key_backlight_pipe;	/**< Key backlight */
datapipe_struct keypress_pipe;		/**< A key has been pressed */
datapipe_struct touchscreen_pipe;	/**< Touchscreen activity took place */
datapipe_struct lockkey_pipe;		/**< The lock-key has been pressed;
					     read only */
datapipe_struct keyboard_slide_pipe;	/**< Keyboard open/closed; read only */
datapipe_struct lid_cover_pipe;		/**< Lid cover open/closed; read only */
datapipe_struct alarm_ui_state_pipe;	/**< The alarm UI state */
datapipe_struct system_state_pipe;	/**< The device state */
datapipe_struct mode_pipe;		/**< The device mode */
datapipe_struct submode_pipe;		/**< The device submode */
datapipe_struct device_lock_pipe;	/**< The device lock state */
/** The touchscreen/keypad lock state */
datapipe_struct tk_lock_pipe;

extern gint has_keyboard;		/**< Yuck! XXX: use HAL */
extern gboolean has_flicker_key;	/**< Yuck! XXX: use HAL */
extern guint16 power_keycode;		/**< Yuck! But we cannot use HAL */

system_mode_t mce_get_mode_int32(void);
gboolean mce_set_mode_int32(const system_mode_t mode);
submode_t mce_get_submode_int32(void);
gboolean mce_add_submode_int32(const submode_t submode);
gboolean mce_rem_submode_int32(const submode_t submode);

/* XXX: yuck */
void mce_startup_ui(void);

#endif /* _MCE_H_ */
