/**
 * @file mce-modules.c
 * Module handling for MCE
 * <p>
 * Copyright © 2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>
#include <gmodule.h>

#include "mce-modules.h"

#include "mce-log.h"			/* mce_log(), LL_* */
#include "mce-conf.h"			/* mce_conf_get_string(),
					 * mce_conf_get_string_list()
					 */

/** List of all loaded modules */
static GSList *modules = NULL;

/**
 * Init function for the mce-modules component
 *
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_modules_init(void)
{
	gchar **modlist = NULL;
	gsize length;
	GModule *module;
	gchar *path = NULL;

	/* Get the module path */
	path = mce_conf_get_string(MCE_CONF_MODULES_GROUP,
				   MCE_CONF_MODULES_PATH,
				   DEFAULT_MCE_MODULE_PATH,
				   NULL);

	/* Get the list modules to load */
	modlist = mce_conf_get_string_list(MCE_CONF_MODULES_GROUP,
					   MCE_CONF_MODULES_MODULES,
					   &length,
					   NULL);

	if (modlist != NULL) {
		gint i;

		for (i = 0; modlist[i]; i++) {
			char *tmp;

			tmp = g_module_build_path(path, modlist[i]);

			mce_log(LL_DEBUG,
				"Loading module: %s from %s",
				modlist[i], path);

			if ((module = g_module_open(tmp, 0)) != NULL) {
				modules = g_slist_append(modules, module);
			} else {
				mce_log(LL_DEBUG,
					"Failed to load module: %s; skipping",
					modlist[i]);
			}

			g_free(tmp);
		}

		g_strfreev(modlist);
	}

	g_free(path);

	return TRUE;
}

/**
 * Exit function for the mce-modules component
 */
void mce_modules_exit(void)
{
	GModule *module;
	gint i;

	if (modules != NULL) {
		for (i = 0; (module = g_slist_nth_data(modules, i)) != NULL; i++) {
			g_module_close(module);
		}

		g_slist_free(modules);
		modules = NULL;
	}
}
