/**
 * @file mce-io.c
 * Generic I/O functionality for the Mode Control Entity
 * <p>
 * Copyright © 2006-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>

#include <errno.h>			/* errno, ERANGE */
#include <stdlib.h>			/* exit(), strtoul(), EXIT_FAILURE */
#include <string.h>			/* strlen() */

#include "mce.h"
#include "mce-io.h"

#include "mce-log.h"			/* mce_log(), LL_* */

/** List of all file monitors */
static GSList *file_monitors = NULL;

/** I/O monitor structure */
typedef struct {
	gchar *file;				/**< Monitored file */
	GIOChannel *iochan;			/**< I/O channel */
	iomon_cb callback;			/**< Callback */
	gulong chunk_size;			/**< Read-chunk size */
	gint fd;				/**< File Descriptor */
	gboolean error_policy;			/**< Error policy */
	gboolean rewind;			/**< Rewind policy */
} iomon_struct;

/**
 * Read a string from a file
 *
 * @param file Path to the file
 * @param string A newly allocated string with the first line of the file
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_read_string_from_file(const gchar *const file, gchar **string)
{
	GError *error = NULL;
	gboolean status = FALSE;

	if (g_file_get_contents(file, string, NULL, &error) == FALSE) {
		mce_log(LL_ERR,
			"Cannot open `%s' for reading; %s",
			file, error->message);
		goto EXIT;
	}

	status = TRUE;

EXIT:
	g_clear_error(&error);

	return status;
}

/**
 * Read a number representation of a string from a file
 *
 * @param file Path to the file
 * @param string A number representation of the first line of the file
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_read_number_string_from_file(const gchar *const file,
					  gulong *number)
{
	gchar *tmp;
	gboolean status;

	if ((status = mce_read_string_from_file(file, &tmp)) == TRUE) {
		gulong tmpnum = strtoul(tmp, NULL, 10);

		g_free(tmp);

		/* Error check for the results of strtoul() */
		if ((errno == EINVAL) ||
		    ((tmpnum == ULONG_MAX) && (errno == ERANGE))) {
			status = FALSE;

			/* Reset errno,
			 * to avoid false positives down the line
			 */
			errno = 0;
		} else {
			*number = tmpnum;
		}
	}

	return status;
}

/**
 * Write a string to a file
 *
 * @param file Path to the file
 * @param string The string to write
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_write_string_to_file(const gchar *const file,
				  const gchar *const string)
{
	GIOChannel *iochan = NULL;
	GIOStatus iostatus;
	GError *error = NULL;
	gboolean status = TRUE;

	if ((iochan = g_io_channel_new_file(file, "w", &error)) == NULL) {
		mce_log(LL_ERR,
			"Cannot open %s for writing; %s",
			file, error->message);
		status = FALSE;
		goto EXIT;
	}

	iostatus = g_io_channel_write_chars(iochan, string,
					    -1, NULL, &error);

	if (iostatus != G_IO_STATUS_NORMAL) {
		mce_log(LL_ERR,
			"Cannot modify %s; %s",
			file, error->message);
		status = FALSE;
		g_clear_error(&error);
	}

	iostatus = g_io_channel_shutdown(iochan, TRUE, &error);

	if (iostatus != G_IO_STATUS_NORMAL) {
		mce_log(LL_ERR,
			"Cannot close %s; %s",
			file, error->message);
		status = FALSE;
	}

	g_io_channel_unref(iochan);

EXIT:
	g_clear_error(&error);

	return status;
}

/**
 * Write a string representation of a number to a file
 *
 * @param file Path to the file
 * @param number The number to write
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_write_number_string_to_file(const gchar *const file,
					 const gulong number)
{
	gchar *tmp;
	gboolean status;

	tmp = g_strdup_printf("%lu", number);
	status = mce_write_string_to_file(file, tmp);
	g_free(tmp);

	return status;
}

/**
 * Callback for I/O activity monitoring
 *
 * @param source Unused
 * @param condition Unused
 * @param data The iomon structure
 * @return Depending on error policy this function either exits
 *         or returns TRUE
 */
static gboolean io_activity_cb(GIOChannel *source,
			       GIOCondition condition,
			       gpointer data)
{
	const iomon_struct *iomon = data;

	/* Silence warnings */
	(void)source;
	(void)condition;
	gboolean status = TRUE;

	if (iomon == NULL) {
		mce_log(LL_CRIT,
			"io_activity_cb called with NULL iomon struct!");
		status = FALSE;
		goto EXIT;
	}

	iomon->callback(NULL, 0);

EXIT:
	if ((status == FALSE) &&
	    (iomon != NULL) && (iomon->error_policy == TRUE)) {
		g_main_loop_quit(mainloop);
		exit(EXIT_FAILURE);
	}

	return TRUE;
}

/**
 * Callback for successful string I/O
 *
 * @param source The source of the activity
 * @param condition Unused
 * @param data The iomon structure
 * @return Depending on error policy this function either exits
 *         or returns TRUE
 */
static gboolean io_string_cb(GIOChannel *source,
			     GIOCondition condition,
			     gpointer data)
{
	const iomon_struct *iomon = data;
	gchar *str = NULL;
	gsize bytes_read;
	GError *error = NULL;
	gboolean status = TRUE;

	/* Silence warnings */
	(void)condition;

	if (iomon == NULL) {
		mce_log(LL_CRIT,
			"io_string_cb called with NULL iomon struct!");
		status = FALSE;
		goto EXIT;
	}

	/* Seek to the beginning of the file before reading if needed */
	if (iomon->rewind == TRUE) {
		g_io_channel_seek_position(source, 0, G_SEEK_SET, &error);
		g_clear_error(&error);
	}

	g_io_channel_read_line(source, &str, &bytes_read, NULL, &error);

	/* Errors and empty reads are nasty */
	if (error != NULL) {
		mce_log(LL_ERR,
			"Error when reading from %s: %s",
			iomon->file, error->message);
	} else if ((bytes_read == 0) || (str == NULL) || (strlen(str) == 0)) {
		mce_log(LL_ERR,
			"Empty read from %s",
			iomon->file);
	} else {
		iomon->callback(str, bytes_read);
		g_free(str);
	}

	g_clear_error(&error);

EXIT:
	if ((status == FALSE) &&
	    (iomon != NULL) && (iomon->error_policy == TRUE)) {
		g_main_loop_quit(mainloop);
		exit(EXIT_FAILURE);
	}

	return TRUE;
}

/**
 * Callback for successful chunk I/O
 *
 * @param source The source of the activity
 * @param condition Unused
 * @param data The iomon structure
 * @return Depending on error policy this function either exits
 *         or returns TRUE
 */
static gboolean io_chunk_cb(GIOChannel *source,
			    GIOCondition condition,
			    gpointer data)
{
	const iomon_struct *iomon = data;
	gchar *chunk = NULL;
	gsize bytes_read;
	GError *error = NULL;
	gboolean status = TRUE;

	/* Silence warnings */
	(void)condition;

	if (iomon == NULL) {
		mce_log(LL_CRIT,
			"io_chunk_cb called with NULL iomon struct!");
		status = FALSE;
		goto EXIT;
	}

	/* Seek to the beginning of the file before reading if needed */
	if (iomon->rewind == TRUE) {
		g_io_channel_seek_position(source, 0, G_SEEK_SET, &error);
		g_clear_error(&error);
	}

	chunk = g_malloc(iomon->chunk_size);

	g_io_channel_read_chars(source, chunk, iomon->chunk_size,
				&bytes_read, &error);

	/* Errors and empty reads are nasty */
	if (error != NULL) {
		mce_log(LL_ERR,
			"Error when reading from %s: %s",
			iomon->file, error->message);
	} else if (bytes_read == 0) {
		mce_log(LL_ERR,
			"Empty read from %s",
			iomon->file);
	} else {
		iomon->callback(chunk, bytes_read);
	}

	/* Free the chunk memory we allocated */
	g_free(chunk);

	g_clear_error(&error);

EXIT:
	if ((status == FALSE) &&
	    (iomon != NULL) && (iomon->error_policy == TRUE)) {
		g_main_loop_quit(mainloop);
		exit(EXIT_FAILURE);
	}

	return TRUE;
}

/**
 * Callback for I/O-errors
 *
 * @param source Unused
 * @param condition Unused
 * @param data The iomon structure
 * @return Depending on error policy this function either exits
 *         or returns TRUE
 */
static gboolean io_error_cb(GIOChannel *source,
			    GIOCondition condition,
			    gpointer data)
{
	const iomon_struct *iomon = data;

	/* Silence warnings */
	(void)source;
	(void)condition;

	if (iomon == NULL) {
		mce_log(LL_CRIT,
			"io_error_cb called with NULL iomon struct!");
		goto EXIT;
	}

	mce_log(LL_CRIT,
		"Error accessing %s, %s",
		iomon->file ? iomon->file : "<filename not set>",
		(iomon->error_policy == TRUE) ? "Exiting" : "Ignoring");

EXIT:
	if ((iomon != NULL) && (iomon->error_policy == TRUE)) {
		g_main_loop_quit(mainloop);
		exit(EXIT_FAILURE);
	}

	return TRUE;
}

/**
 * Register an I/O monitor; reads and returns data
 *
 * @param fd File Descriptor; this takes priority over file; -1 if not used
 * @param file Path to the file
 * @param error_policy TRUE to exit on error, FALSE to ignore errors
 * @param callback Function to call with result
 * @return An I/O monitor pointer on success, NULL on failure
 */
static iomon_struct *mce_register_io_monitor(const gint fd,
					     const gchar *const file,
					     gboolean error_policy,
					     iomon_cb callback)
{
	GIOChannel *iochan = NULL;
	GError *error = NULL;
	iomon_struct *iomon = NULL;

	if (file == NULL) {
		mce_log(LL_CRIT,
			"mce_register_io_monitor called with file unset");
		goto EXIT;
	}

	if (callback == NULL) {
		mce_log(LL_CRIT,
			"mce_register_io_monitor called with callback unset");
		goto EXIT;
	}

	if ((iomon = g_slice_new(iomon_struct)) == NULL) {
		mce_log(LL_CRIT,
			"Failed to allocate memory for iomon_struct");
		goto EXIT;
	}

	if (fd != -1) {
		if ((iochan = g_io_channel_unix_new(fd)) == NULL) {
			mce_log(LL_ERR,
				"Failed to open `%s'", file);
			g_slice_free(iomon_struct, iomon);
			iomon = NULL;
			goto EXIT;
		}
	} else {
		if ((iochan = g_io_channel_new_file(file, "r",
						   &error)) == NULL) {
			mce_log(LL_ERR,
				"Failed to open `%s'; %s",
				file, error->message);
			g_slice_free(iomon_struct, iomon);
			iomon = NULL;
			goto EXIT;
		}
	}

	iomon->fd = fd;
	iomon->file = g_strdup(file);
	iomon->iochan = iochan;
	iomon->callback = callback;
	iomon->error_policy = error_policy;
	iomon->chunk_size = 0;

	file_monitors = g_slist_prepend(file_monitors, iomon);

	(void)g_io_add_watch(iomon->iochan, G_IO_ERR | G_IO_HUP,
			     io_error_cb, iomon);

EXIT:
	g_clear_error(&error);

	return iomon;
}

/**
 * Register an I/O activity monitor
 *
 * @param fd File Descriptor; this takes priority over file; -1 if not used
 * @param file Path to the file
 * @param error_policy TRUE to exit on error, FALSE to ignore errors
 * @param callback Function to call when activity takes place
 * @return An I/O monitor pointer on success, NULL on failure
 */
gconstpointer mce_register_io_monitor_activity(const gint fd,
					       const gchar *const file,
					       gboolean error_policy,
					       iomon_cb callback)
{
	iomon_struct *iomon = NULL;

	iomon = mce_register_io_monitor(fd, file, error_policy, callback);

	if (iomon == NULL)
		goto EXIT;

	(void)g_io_add_watch(iomon->iochan, G_IO_IN | G_IO_PRI,
			     io_activity_cb, iomon);

EXIT:
	return iomon;
}

/**
 * Register an I/O monitor; reads and returns a string
 *
 * @param fd File Descriptor; this takes priority over file; -1 if not used
 * @param file Path to the file
 * @param error_policy TRUE to exit on error, FALSE to ignore errors
 * @param rewind_policy TRUE to seek to the beginning,
 *                      FALSE to stay at current position
 * @param callback Function to call with result
 * @return An I/O monitor pointer on success, NULL on failure
 */
gconstpointer mce_register_io_monitor_string(const gint fd,
					     const gchar *const file,
					     gboolean error_policy,
					     gboolean rewind_policy,
					     iomon_cb callback)
{
	iomon_struct *iomon = NULL;

	iomon = mce_register_io_monitor(fd, file, error_policy, callback);

	if (iomon == NULL)
		goto EXIT;

	/* Set the rewind policy */
	iomon->rewind = rewind_policy;

	(void)g_io_add_watch(iomon->iochan, G_IO_IN | G_IO_PRI,
			     io_string_cb, iomon);

EXIT:
	return iomon;
}

/**
 * Register an I/O monitor; reads and returns a chunk of specified size
 *
 * @param fd File Descriptor; this takes priority over file; -1 if not used
 * @param file Path to the file
 * @param error_policy TRUE to exit on error, FALSE to ignore errors
 * @param rewind_policy TRUE to seek to the beginning,
 *                      FALSE to stay at current position
 * @param callback Function to call with result
 * @param chunk_size The number of bytes to read in each chunk
 * @return An I/O monitor pointer on success, NULL on failure
 */
gconstpointer mce_register_io_monitor_chunk(const gint fd,
					    const gchar *const file,
					    gboolean error_policy,
					    gboolean rewind_policy,
					    iomon_cb callback,
					    gulong chunk_size)
{
	iomon_struct *iomon = NULL;
	GError *error = NULL;

	iomon = mce_register_io_monitor(fd, file, error_policy, callback);

	if (iomon == NULL)
		goto EXIT;

	/* Set the read chunk size */
	iomon->chunk_size = chunk_size;

	/* Set the rewind policy */
	iomon->rewind = rewind_policy;

	/* We only read this file in binary form */
	(void)g_io_channel_set_encoding(iomon->iochan, NULL, &error);

	g_clear_error(&error);

	/* Don't block */
	(void)g_io_channel_set_flags(iomon->iochan, G_IO_FLAG_NONBLOCK, &error);

	g_clear_error(&error);

	(void)g_io_add_watch(iomon->iochan, G_IO_IN | G_IO_PRI,
			     io_chunk_cb, iomon);

EXIT:
	return iomon;
}

/**
 * Unregister an I/O monitor
 *
 * @param io_monitor A pointer to the I/O monitor to unregister
 */
void mce_unregister_io_monitor(gconstpointer io_monitor)
{
	iomon_struct *iomon = (iomon_struct *)io_monitor;

	if (iomon != NULL) {
		if (file_monitors != NULL)
			file_monitors = g_slist_remove(file_monitors, iomon);

		g_slice_free(iomon_struct, iomon);
	}
}
