/**
 * @file mce-dsme.c
 * Interface code and logic between
 * DSME (the Device State Management Entity)
 * and MCE (the Mode Control Entity)
 * <p>
 * Copyright © 2004-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 * @author Ismo Laitinen <ismo.laitinen@nokia.com>
 */
#include <glib.h>

#include <time.h>			/* time(), time_t */
#include <errno.h>			/* errno */
#include <stdlib.h>			/* exit(), free(), EXIT_FAILURE */
#include <unistd.h>			/* getpid() */

#include <dsme/state.h>			/* dsme_state_t */
#include <dsme/messages.h>		/* DSM_MSGTYPE_* */
#include <dsme/protocol.h>		/* dsmesock_send(),
					 * dsmesock_receive(),
					 * dsmesock_connect(),
					 * dsmesock_close(),
					 * dsmemsg_state_change_t,
					 * dsmemsg_timeout_change_t,
					 * dsmemsg_generic_t,
					 * dsmemsg_swwd_t,
					 * dsmesock_connection_t
					 */
#include <dsme/inactivity-blank.h>	/* DSME_DISPLAY_ON,
					 * DSME_DISPLAY_DIM,
					 * DSME_DISPLAY_OFF
					 */
#include <mce/mode-names.h>
#include <alarmd/alarm_dbus.h>		/* ALARMD_SERVICE, ALARMD_PATH,
					 * ALARMD_INTERFACE, ALARM_EVENT_QUERY
					 */
#include <alarmd/alarm_event.h>		/* ALARM_EVENT_BOOT */
#include <bme-dbus-proxy/dbus-names.h>	/* BME_SERVICE, BME_REQUEST_PATH,
					 * BME_REQUEST_IF, BME_SIGNAL_IF,
					 * BME_STATUS_INFO_REQ,
					 * BME_BATTERY_LOW, BME_BATTERY_EMPTY,
					 * BME_CHARGER_CHARGING_ON,
					 * BME_CHARGER_CHARGING_OFF
					 */

#include "mce.h"			/* mce_set_mode_int32(),
					 * mce_get_mode_int32(),
					 * mce_add_submode_int32(),
					 * mce_rem_submode_int32(),
					 * mce_get_submode_int32(),
					 * MCE_LED_PATTERN_DEVICE_ON,
					 * MCE_LED_PATTERN_POWER_ON,
					 * MCE_LED_PATTERN_BATTERY_LOW,
					 * MCE_LED_PATTERN_BATTERY_CHARGING,
					 * MCE_LED_PATTERN_BATTERY_FULL,
					 * system_mode_t,
					 * system_state_t,
					 * display_state_t,
					 * alarm_ui_state_t,
					 * MCE_ALARM_SUBMODE,
					 * MCE_SOFTOFF_SUBMODE,
					 * MCE_TRANSITION_SUBMODE,
					 * MCE_MODECHG_SUBMODE,
					 * mainloop,
					 * alarm_ui_state_pipe,
					 * device_lock_pipe,
					 * device_inactive_pipe,
					 * display_state_pipe,
					 * display_brightness_pipe,
					 * system_state_pipe,
					 * keyboard_slide_pipe,
					 * led_pattern_activate_pipe,
					 * led_pattern_deactivate_pipe
					 */
#include "mce-dsme.h"

#include "mce-log.h"			/* mce_log(), LL_* */
#include "mce-conf.h"			/* mce_conf_get_string() */
#include "mce-dbus.h"			/* mce_dbus_handler_add(),
					 * dbus_send(),
					 * dbus_send_message(),
					 * dbus_send_message_with_reply_and_block(),
					 * dbus_new_method_call(),
					 * dbus_new_method_reply(),
					 * dbus_new_signal(),
					 * dbus_message_get_no_reply(),
					 * dbus_message_get_args(),
					 * dbus_message_append_args(),
					 * dbus_message_unref(),
					 * dbus_error_init(),
					 * dbus_error_free(),
					 * DBUS_MESSAGE_TYPE_METHOD_CALL,
					 * DBUS_MESSAGE_TYPE_SIGNAL,
					 * DBUS_TYPE_INT32,
					 * DBUS_TYPE_UINT64,
					 * DBUS_TYPE_STRING, DBUS_TYPE_ARRAY,
					 * DBUS_TYPE_INVALID,
					 * DBusMessage, DBusError,
					 * dbus_bool_t,
					 * dbus_int32_t,
					 * dbus_int64_t
					 */
#include "mce-gconf.h"			/* mce_gconf_get_bool(),
					 * mce_gconf_get_int(),
					 * mce_gconf_notifier_add(),
					 * gconf_value_get_bool(),
					 * gconf_value_get_int(),
					 * GConfClient, GConfEntry,
					 * GConfValue
					 */
#include "datapipe.h"			/* execute_datapipe(),
					 * execute_datapippe_output_triggers(),
					 * datapipe_get_gint(),
					 * append_output_trigger_to_datapipe(),
					 * remove_output_trigger_from_datapipe()
					 */
#include "connectivity.h"		/* get_connectivity_status() */

/** Cached display state */
static display_state_t cached_display_state = MCE_DISPLAY_UNDEF;

/** GConf callback ID for display brightness setting */
static guint disp_brightness_gconf_cb_id = 0;

/** Display dimming timeout setting */
static gint disp_dim_timeout = DEFAULT_DIM_TIMEOUT;
/** GConf callback ID for display dimming timeout setting */
static guint disp_dim_timeout_gconf_cb_id = 0;

/** Display blanking timeout setting */
static gint disp_blank_timeout = DEFAULT_BLANK_TIMEOUT;
/** GConf callback ID for display blanking timeout setting */
static guint disp_blank_timeout_gconf_cb_id = 0;

/** Don't blank on charger connected */
static gboolean disp_on_with_charger = DEFAULT_DISPLAY_ON_WITH_CHARGER;
/** GConf callback ID for don't blank on charger connected setting */
static guint disp_on_with_charger_gconf_cb_id = 0;

/** ID for display blank prevention timer source */
static guint display_blank_timeout_cb_id = 0;

/** Display blank prevention timer */
static gint display_blank_timeout = 60;

/** Charger state */
static gboolean charger_connected = FALSE;

/** Pointer to the dsmesock connection */
static dsmesock_connection_t *dsme_conn = NULL;
/** TRUE if dsme is disabled (for debugging), TRUE if dsme is enabled */
static gboolean dsme_disabled = FALSE;

/** ID for battery low shutdown source */
static guint lowbat_shutdown_timeout_cb_id = 0;

/** ID for state transition timer source */
static guint transition_timeout_cb_id = 0;

/** Soft poweroff connectivity policy when connected to charger */
static gint softoff_connectivity_policy_charger =
					DEFAULT_SOFTOFF_CONNECTIVITY_CHARGER;

/** Soft poweroff connectivity policy when running on battery */
static gint softoff_connectivity_policy_battery =
					DEFAULT_SOFTOFF_CONNECTIVITY_BATTERY;

/** Soft poweroff connectivity policy on poweron */
static gint softoff_connectivity_policy_poweron =
					DEFAULT_SOFTOFF_CONNECTIVITY_POWERON;

static system_mode_t previous_mode = MCE_INVALID_MODE_INT32;

static gboolean init_dsmesock(void);

/**
 * Send a data save request signal
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean send_data_save(void)
{
	/* save_unsaved_data_ind */
	return dbus_send(NULL, MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
			 MCE_DATA_SAVE_SIG, NULL, DBUS_TYPE_INVALID);
}

/**
 * Send a shutdown notification signal
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean send_shutdown(void)
{
	/* shutdown_ind */
	return dbus_send(NULL, MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
			 MCE_SHUTDOWN_SIG, NULL, DBUS_TYPE_INVALID);
}

/**
 * Send a thermal shutdown notification signal
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean send_thermal_shutdown(void)
{
	/* thermal_shutdown_ind */
	return dbus_send(NULL, MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
			 MCE_THERMAL_SHUTDOWN_SIG, NULL, DBUS_TYPE_INVALID);
}

/**
 * Send a display status reply or signal
 *
 * @param method_call A DBusMessage to reply to;
 *                    pass NULL to send a display status signal instead
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_send_display_status(DBusMessage *const method_call)
{
	display_state_t display_state = datapipe_get_gint(display_state_pipe);
	DBusMessage *msg = NULL;
	const gchar *state = NULL;
	gboolean status = FALSE;

	switch (display_state) {
	case MCE_DISPLAY_OFF:
		state = MCE_DISPLAY_OFF_STRING;
		break;

	case MCE_DISPLAY_DIM:
		state = MCE_DISPLAY_DIM_STRING;
		break;

	case MCE_DISPLAY_ON:
	default:
		state = MCE_DISPLAY_ON_STRING;
		break;
	}

	mce_log(LL_DEBUG,
		"Sending display status: %s",
		state);

	/* If method_call is set, send a reply,
	 * otherwise, send a signal
	 */
	if (method_call != NULL) {
		msg = dbus_new_method_reply(method_call);
	} else {
		/* display_status_ind */
		msg = dbus_new_signal(MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
				      MCE_DISPLAY_SIG);
	}

	/* Append the display status */
	if (dbus_message_append_args(msg,
				     DBUS_TYPE_STRING, &state,
				     DBUS_TYPE_INVALID) == FALSE) {
		mce_log(LL_CRIT,
			"Failed to append %sargument to D-Bus message for %s",
			method_call ? "reply " : "",
			method_call ? MCE_DISPLAY_STATUS_GET :
				      MCE_DISPLAY_SIG);
		dbus_message_unref(msg);
		goto EXIT;
	}

	/* Send the message */
	status = dbus_send_message(msg);

EXIT:
	return status;
}

/**
 * Generic send function for dsmesock messages
 * XXX: How should we handle sending failures?
 *
 * @param msg A pointer to the message to send
 */
static void mce_dsme_send(gpointer msg)
{
	if (dsme_disabled == TRUE)
		goto EXIT;

	if (dsme_conn == NULL) {
		mce_log(LL_CRIT,
			"Attempt to use dsme_conn uninitialised; aborting!");
		g_main_loop_quit(mainloop);
		exit(EXIT_FAILURE);
	}

	if ((dsmesock_send(dsme_conn, msg)) == -1) {
		mce_log(LL_CRIT,
			"dsmesock_send error: %s",
			g_strerror(errno));
#ifdef MCE_DSME_ERROR_POLICY
		g_main_loop_quit(mainloop);
		exit(EXIT_FAILURE);
#endif /* MCE_DSME_ERROR_POLICY */
	}

EXIT:
	return;
}

/**
 * Set display brightness
 *
 * @param brightness Screen brightness
 */
static void dsme_set_disp_brightness(gint brightness)
{
	dsmemsg_state_change_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_LCD_BRIGHTNESS_SET;
	msg.size = sizeof msg;
	msg.state = brightness;

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_LCD_BRIGHTNESS_SET (%i) sent to DSME",
		msg.state);
}

/**
 * Display brightness trigger
 *
 * @note A brightness request is only sent if the value changed
 * @param brightness Screen brightness
 */
static void display_brightness_trigger(gconstpointer brightness)
{
	/** DSME imposed limit */
	static gint maximum_display_brightness = 255;
	static gint cached_brightness = -1;
	gint new_brightness = GPOINTER_TO_INT(brightness);

	/* If the pipe is choked, ignore the value */
	if (new_brightness == 0)
		goto EXIT;

	/* If the brightness value is uncooked, adjust it */
	if ((new_brightness > 0) && (new_brightness < 10)) {
		if (new_brightness > 5)
			new_brightness = 5;

		new_brightness = (maximum_display_brightness *
				  new_brightness) / 5;
	}

	/* If we're just rehashing the same brightness value, don't bother */
	if ((new_brightness == cached_brightness) && (cached_brightness != -1))
		goto EXIT;

	cached_brightness = new_brightness;

	dsme_set_disp_brightness(new_brightness);

EXIT:
	return;
}

/**
 * Set blank timeout
 *
 * @param timeout Time of inactivity, in seconds, until screen blanks
 */
static void dsme_set_blank_timeout(const gint timeout)
{
	dsmemsg_timeout_change_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_SET_BLANK_TIMEOUT;
	msg.size = sizeof msg;
	msg.timeout = timeout;

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_SET_BLANK_TIMEOUT (%i) sent to DSME",
		msg.timeout);
}

/**
 * Set dim timeout
 *
 * @param timeout Time of inactivity, in seconds, until screen dims
 */
static void dsme_set_dim_timeout(const gint timeout)
{
	dsmemsg_timeout_change_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_SET_DIM_TIMEOUT;
	msg.size = sizeof msg;
	msg.timeout = timeout;

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_SET_DIM_TIMEOUT (%i) sent to DSME",
		msg.timeout);
}

/**
 * Set alarm state
 *
 * @param state Alarm state
 */
static void dsme_set_alarm_state(const dsme_alarm_state_t state)
{
	dsmemsg_state_change_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_SET_ALARM_STATE;
	msg.size = sizeof msg;
	msg.state = state;

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_SET_ALARM_STATE (%i) sent to DSME",
		msg.state);
}

/**
 * Unblank display
 */
static void request_display_on(void)
{
	dsmemsg_state_change_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_LCD_STATE_REQ;
	msg.state = DSME_DISPLAY_ON;
	msg.size = sizeof msg;

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_LCD_STATE_REQ (%i) sent to DSME",
		msg.state);
}

/**
 * Dim display
 */
static void request_display_dim(void)
{
	dsmemsg_state_change_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_LCD_STATE_REQ;
	msg.state = DSME_DISPLAY_DIM;
	msg.size = sizeof msg;

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_LCD_STATE_REQ (%i) sent to DSME",
		msg.state);
}

/**
 * Blank display
 */
static void request_display_off(void)
{
	dsmemsg_state_change_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_LCD_STATE_REQ;
	msg.state = DSME_DISPLAY_OFF;
	msg.size = sizeof msg;

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_LCD_STATE_REQ (%i) sent to DSME",
		msg.state);
}

/**
 * Allow screen blanking
 */
static void request_display_blanking_allow(void)
{
	dsmemsg_generic_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_BLANK_ALLOW;
	msg.size = sizeof msg;

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_BLANK_ALLOW sent");
}

/**
 * Prevent screen blanking
 */
static void request_display_blanking_prevent(void)
{
	dsmemsg_generic_t msg;

	/* Remove the timeout source for the display blank prevention */
	if (display_blank_timeout_cb_id != 0) {
		g_source_remove(display_blank_timeout_cb_id);
		display_blank_timeout_cb_id = 0;
	}

	/* Set up the message */
	msg.type = DSM_MSGTYPE_BLANK_PREVENT;
	msg.size = sizeof msg;

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_BLANK_PREVENT sent");
}

/**
 * Enable/Disable blanking inhibit based on charger status
 */
static void update_blanking_inhibit(void)
{
	if ((charger_connected == TRUE) && (disp_on_with_charger == TRUE))
		request_display_blanking_prevent();
	else if (display_blank_timeout_cb_id == 0)
		request_display_blanking_allow();
}

/**
 * Display blanking pause timeout function
 *
 * @param data Unused
 * @return Always returns FALSE, to disable the timeout
 */
static gboolean display_blank_timeout_cb(gpointer data)
{
	(void)data;

	display_blank_timeout_cb_id = 0;

	update_blanking_inhibit();

	return FALSE;
}

/**
 * Prevent screen blanking for display_timeout seconds
 */
static void request_display_blanking_pause(void)
{
	request_display_blanking_prevent();

	/* Setup new timeout */
	display_blank_timeout_cb_id =
		g_timeout_add(display_blank_timeout * 1000,
			      display_blank_timeout_cb, NULL);
}

/**
 * Send pong message to the DSME process watchdog
 */
static void dsme_send_pong(void)
{
	dsmemsg_swwd_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_PROCESSWD_PONG;
	msg.size = sizeof msg;
	msg.pid = getpid();

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_PROCESSWD_PONG sent to DSME");
}

/**
 * Register to DSME process watchdog
 */
static void dsme_init_processwd(void)
{
	dsmemsg_swwd_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_PROCESSWD_CREATE;
	msg.size = sizeof msg;
	msg.pid = getpid();

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_PROCESSWD_CREATE sent to DSME");
}

/**
 * Unregister from DSME process watchdog
 */
static void dsme_exit_processwd(void)
{
	dsmemsg_swwd_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_PROCESSWD_DELETE;
	msg.size = sizeof msg;
	msg.pid = getpid();

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_PROCESSWD_DELETE sent to DSME");
}

/**
 * Send system state inquiry
 */
static void query_system_state(void)
{
	dsmemsg_generic_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_STATE_QUERY;
	msg.size = sizeof msg;

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_STATE_QUERY sent to DSME");
}

/**
 * Request powerup
 */
void request_powerup(void)
{
	dsmemsg_generic_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_POWERUP_REQ;
	msg.size = sizeof msg;

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_POWERUP_REQ sent to DSME");
}

/**
 * Request reboot
 */
void request_reboot(void)
{
	dsmemsg_generic_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_REBOOT_REQ;
	msg.size = sizeof msg;

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_REBOOT_REQ sent to DSME");
}

/**
 * Request soft poweron
 */
void request_soft_poweron(void)
{
	dsmemsg_generic_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_COVER_OPENED;
	msg.size = sizeof msg;

	mce_rem_submode_int32(MCE_SOFTOFF_SUBMODE);
	execute_datapipe(&display_state_pipe,
			 GINT_TO_POINTER(MCE_DISPLAY_ON),
			 FALSE, TRUE);

	/* Send the message */
	mce_dsme_send(&msg);

	/* Connectivity policy */
	switch (softoff_connectivity_policy_poweron) {
	case SOFTOFF_CONNECTIVITY_FORCE_OFFLINE:
		/* Restore previous mode */
		mce_set_mode_int32(previous_mode);
		break;

	case SOFTOFF_CONNECTIVITY_OFFLINE:
	default:
		/* Do nothing */
		break;
	}
}

/**
 * Request soft poweroff
 */
void request_soft_poweroff(void)
{
	dsmemsg_generic_t msg;
	gboolean connected;
	gint policy;

	if (charger_connected == TRUE)
		policy = softoff_connectivity_policy_charger;
	else
		policy = softoff_connectivity_policy_battery;

	/** Ignore errors */
	(void)get_connectivity_status(&connected);

	/* Connectivity policy */
	switch (policy) {
	case SOFTOFF_CONNECTIVITY_SOFT_OFFLINE:
		/* If there are open connections, abort */
		if (connected == TRUE)
			break;

		/* Fall-through */
	case SOFTOFF_CONNECTIVITY_FORCE_OFFLINE:
		/* Store current mode for restore on soft poweron */
		previous_mode = mce_get_mode_int32();

		/* Go offline */
		mce_set_mode_int32(MCE_FLIGHT_MODE_INT32);
		break;

	case SOFTOFF_CONNECTIVITY_RETAIN:
	default:
		break;
	}

	mce_add_submode_int32(MCE_SOFTOFF_SUBMODE);
	execute_datapipe(&display_state_pipe,
			 GINT_TO_POINTER(MCE_DISPLAY_OFF),
			 FALSE, TRUE);

	/* Set up the message */
	msg.type = DSM_MSGTYPE_COVER_CLOSED;
	msg.size = sizeof msg;

	/* Send the message */
	mce_dsme_send(&msg);
}

/**
 * Timeout callback for transition
 *
 * @param data Unused
 * @return Always returns FALSE, to disable the timeout
 */
static gboolean transition_timeout_cb(gpointer data)
{
	(void)data;

	transition_timeout_cb_id = 0;

	mce_rem_submode_int32(MCE_TRANSITION_SUBMODE);

	/* FALSE here is just to disable the timeout */
	return FALSE;
}

/**
 * Setup state transition timeout
 */
static void setup_transition_timeout(void)
{
	transition_timeout_cb_id = g_timeout_add(TRANSITION_DELAY,
						 transition_timeout_cb, NULL);
}

/**
 * Request shutdown
 */
static void request_shutdown(void)
{
	dsmemsg_state_change_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_SHUTDOWN_REQ;
	msg.size = sizeof msg;
	msg.state = DSME_NORMAL_SHUTDOWN;

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_SHUTDOWN_REQ (DSME_NORMAL_SHUTDOWN) "
		"sent to DSME");
}

/**
 * Request lowbat shutdown
 */
static void request_lowbat_shutdown(void)
{
	dsmemsg_state_change_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_SHUTDOWN_REQ;
	msg.size = sizeof msg;
	msg.state = DSME_BATTERY_EMPTY_SHUTDOWN;

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_SHUTDOWN_REQ (DSME_BATTERY_EMPTY_SHUTDOWN) "
		"sent to DSME");
}

/**
 * Timeout callback for emergency shutdown
 *
 * @param data Unused
 * @return Always returns FALSE, to disable the timeout
 */
static gboolean lowbat_shutdown_timeout_cb(gpointer data)
{
	(void)data;

	lowbat_shutdown_timeout_cb_id = 0;

	request_lowbat_shutdown();

	/* FALSE here is just to disable the timeout */
	return FALSE;
}

#if 0
/**
 * Cancel lowbat shutdown
 */
static void cancel_lowbat_shutdown(void)
{
	if (lowbat_shutdown_timeout_cb_id != 0) {
		g_source_remove(lowbat_shutdown_timeout_cb_id);
		lowbat_shutdown_timeout_cb_id = 0;
	}
}
#endif

/**
 * GConf callback for DSME related settings
 *
 * @param gce Unused
 * @param id Connection ID from gconf_client_notify_add()
 * @param entry The modified GConf entry
 * @param data Unused
 */
static void dsme_gconf_cb(GConfClient *const gce, const guint id,
			  GConfEntry *const entry, gpointer const data)
{
	GConfValue *gcv = gconf_entry_get_value(entry);

	(void)gce;
	(void)data;

	if (id == disp_brightness_gconf_cb_id) {
		gint tmp = gconf_value_get_int(gcv);

		(void)execute_datapipe(&display_brightness_pipe,
				       GINT_TO_POINTER(tmp), FALSE, TRUE);
	} else if (id == disp_blank_timeout_gconf_cb_id) {
		disp_blank_timeout = gconf_value_get_int(gcv);

		if (disp_dim_timeout > disp_blank_timeout) {
			disp_dim_timeout = disp_blank_timeout;
			dsme_set_dim_timeout(disp_dim_timeout);
		}

		dsme_set_blank_timeout(disp_blank_timeout);
	} else if (id == disp_dim_timeout_gconf_cb_id) {
		disp_dim_timeout = gconf_value_get_int(gcv);

		if (disp_dim_timeout > disp_blank_timeout) {
			disp_blank_timeout = disp_dim_timeout;
			dsme_set_blank_timeout(disp_blank_timeout);
		}

		dsme_set_dim_timeout(disp_dim_timeout);
	} else if (id == disp_on_with_charger_gconf_cb_id) {
		disp_on_with_charger = gconf_value_get_bool(gcv);
		update_blanking_inhibit();
	} else {
		mce_log(LL_WARN,
			"Spurious GConf value received; confused!");
	}
}

/**
 * Convert DSME dsme display state
 * to a state enum that we can export to datapipes
 *
 * @param dsmestate The DSME dsme_state_t with the value to convert
 * @return the converted value
 */
static display_state_t normalise_dsme_display_state(dsme_state_t dsmestate)
{
	display_state_t state;

	switch (dsmestate) {
	case -1:
		state = MCE_DISPLAY_UNDEF;
		break;

	case DSME_DISPLAY_OFF:
		state = MCE_DISPLAY_OFF;
		break;

	case DSME_DISPLAY_DIM:
		state = MCE_DISPLAY_DIM;
		break;

	case DSME_DISPLAY_ON:
	default:
		state = MCE_DISPLAY_ON;
		break;
	}

	return state;
}

/**
 * Convert DSME dsme state
 * to a state enum that we can export to datapipes
 *
 * @param dsmestate The DSME dsme_state_t with the value to convert
 * @return the converted value
 */
static system_state_t normalise_dsme_state(dsme_state_t dsmestate)
{
	system_state_t state;

	switch (dsmestate) {
	case DSME_STATE_SHUTDOWN:
		state = MCE_STATE_SHUTDOWN;
		break;

	case DSME_STATE_USER:
		state = MCE_STATE_USER;
		break;

	case DSME_STATE_ACTDEAD:
		state = MCE_STATE_ACTDEAD;
		break;

	case DSME_STATE_REBOOT:
		state = MCE_STATE_REBOOT;
		break;

	case DSME_STATE_BOOT:
		state = MCE_STATE_BOOT;
		break;

	case DSME_STATE_NOT_SET:
	default:
		state = MCE_STATE_UNDEF;
		break;
	}

	return state;
}

/**
 * Callback for pending I/O from dsmesock
 *
 * XXX: is the error policy reasonable?
 *
 * @param source Unused
 * @param condition Unused
 * @param data Unused
 * @return TRUE on success, FALSE on failure
 */
static gboolean io_data_ready_cb(GIOChannel *source,
				 GIOCondition condition,
				 gpointer data)
{
	dsmemsg_generic_t *msg;
	dsmemsg_state_change_t *msg2;
	system_state_t oldstate = datapipe_get_gint(system_state_pipe);
	system_state_t newstate = MCE_STATE_UNDEF;
	display_state_t display_state;
	display_state_t new_display_state = MCE_DISPLAY_UNDEF;

	display_state = datapipe_get_gint(display_state_pipe);

	(void)source;
	(void)condition;
	(void)data;

	if (dsme_disabled == TRUE)
		goto EXIT;

	if ((msg = dsmesock_receive(dsme_conn)) == NULL)
		goto EXIT;

	switch (msg->type) {
	case DSM_MSGTYPE_CLOSE:
		/* DSME socket closed: try once to reopen;
		 * if that fails, exit
		 */
		mce_log(LL_ERR,
			"DSME socket closed; trying to reopen");

		if ((init_dsmesock()) == FALSE) {
			g_main_loop_quit(mainloop);
			exit(EXIT_FAILURE);
		}

		break;

	case DSM_MSGTYPE_PROCESSWD_PING:
		dsme_send_pong();
		break;

	case DSM_MSGTYPE_STATE_CHANGE_IND:
		msg2 = (dsmemsg_state_change_t *)msg;

		if (msg2->size != sizeof (dsmemsg_state_change_t))
			break;

		newstate = normalise_dsme_state(msg2->state);
		mce_log(LL_DEBUG,
			"DSME device state change: %d",
			newstate);

		switch (newstate) {
		case MCE_STATE_USER:
			if (display_state == MCE_DISPLAY_OFF)
				execute_datapipe_output_triggers(&led_pattern_activate_pipe, MCE_LED_PATTERN_DEVICE_ON, FALSE);
			break;

		case MCE_STATE_ACTDEAD:
			if (oldstate == MCE_STATE_USER)
				send_shutdown();
			/* fall-through */

		case MCE_STATE_BOOT:
		case MCE_STATE_UNDEF:
			mce_add_submode_int32(MCE_TRANSITION_SUBMODE);
			mce_rem_submode_int32(MCE_MODECHG_SUBMODE);
			setup_display_settings();
			break;

		case MCE_STATE_SHUTDOWN:
		case MCE_STATE_REBOOT:
			mce_rem_submode_int32(MCE_MODECHG_SUBMODE);
			send_shutdown();
			break;

		default:
			break;
		}

		execute_datapipe(&system_state_pipe,
				 GINT_TO_POINTER(newstate),
				 FALSE, TRUE);
		break;

	case DSM_MSGTYPE_THERMAL_SHUTDOWN_IND:
		mce_log(LL_DEBUG,
			"thermal_shutdown event received from DSME!");
		send_thermal_shutdown();
		break;

	case DSM_MSGTYPE_SAVE_DATA_IND:
		mce_log(LL_DEBUG,
			"data_save event received from DSME!");
		send_data_save();
		break;

	case DSM_MSGTYPE_STATE_ACTIVE_IND:
		execute_datapipe(&device_inactive_pipe,
				 GINT_TO_POINTER(FALSE), FALSE, TRUE);
		break;

	case DSM_MSGTYPE_STATE_INACTIVE_IND:
		execute_datapipe(&device_inactive_pipe,
				 GINT_TO_POINTER(TRUE), FALSE, TRUE);
		break;

	case DSM_MSGTYPE_LCD_STATUS:
		msg2 = (dsmemsg_state_change_t *)msg;

		if (msg2->size != sizeof (dsmemsg_state_change_t))
			break;

		new_display_state = normalise_dsme_display_state(msg2->state);

		mce_log(LL_DEBUG,
			"lcd_status event received from DSME!");

		/* Note: this won't trigger on transitions from
		 * DIM to ON or from ON to DIM
		 */
		if (new_display_state == MCE_DISPLAY_OFF) {
			if (oldstate == MCE_STATE_USER) {
				execute_datapipe_output_triggers(&led_pattern_activate_pipe, MCE_LED_PATTERN_DEVICE_ON, FALSE);
			}
		} else if ((display_state == MCE_DISPLAY_OFF ||
			    display_state == -1) &&
			   (new_display_state == MCE_DISPLAY_ON ||
			    new_display_state == MCE_DISPLAY_DIM)) {
			execute_datapipe_output_triggers(&led_pattern_deactivate_pipe, MCE_LED_PATTERN_DEVICE_ON, FALSE);
		}

		/* Update cached value first, to avoid requesting
		 * a display state from dsme that we already have
		 */
		cached_display_state = new_display_state;
		execute_datapipe(&display_state_pipe,
				 GINT_TO_POINTER(new_display_state),
				 FALSE, TRUE);
		break;

	case DSM_MSGTYPE_DEVICELOCK_AUTOLOCK_EXPIRED:
		mce_log(LL_DEBUG,
			"autolock_expired event received from DSME!");

		if (oldstate != MCE_STATE_USER)
			break;

		/* If we're in VoIP-mode, don't lock */
		if (mce_get_mode_int32() == MCE_VOIP_MODE_INT32)
			break;

		/* Request enabling of device lock */
		execute_datapipe(&device_lock_pipe,
				 GINT_TO_POINTER(LOCK_ON),
				 FALSE, TRUE);
		break;

	default:
		mce_log(LL_DEBUG,
			"Unknown message type (%x) received from DSME!",
			msg->type);
		break;
	}

	free(msg);

EXIT:
	return TRUE;
}

/**
 * Callback for I/O-errors from dsmesock
 *
 * @param source Unused
 * @param condition Unused
 * @param data Unused
 * @return Will never return; if there is an I/O-error we exit the mainloop
 */
G_GNUC_NORETURN static gboolean io_error_cb(GIOChannel *source,
					    GIOCondition condition,
					    gpointer data)
{
	/* Silence warnings */
	(void)source;
	(void)condition;
	(void)data;

	/* DSME socket closed/error */
	mce_log(LL_CRIT,
		"DSME socket closed/error, exiting...");
	g_main_loop_quit(mainloop);
	exit(EXIT_FAILURE);
}

/**
 * D-Bus callback for the init done notification signal
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean init_done_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	(void)msg;

	mce_log(LL_DEBUG,
		"Received init done notification");

	if ((mce_get_submode_int32() & MCE_TRANSITION_SUBMODE)) {
		setup_transition_timeout();
	}

	status = TRUE;

//EXIT:
	return status;
}

/**
 * D-Bus callback for the get display status method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean display_status_get_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	mce_log(LL_DEBUG,
		"Received display status get request");

	/* Try to send a reply that contains the current display status */
	if (mce_send_display_status(msg) == FALSE)
		goto EXIT;

	status = TRUE;

EXIT:
	return status;
}

/**
 * D-Bus callback for the display on method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean display_on_req_dbus_cb(DBusMessage *const msg)
{
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	gboolean status = FALSE;

	mce_log(LL_DEBUG,
		"Received display on request");

	execute_datapipe(&display_state_pipe,
			 GINT_TO_POINTER(MCE_DISPLAY_ON),
			 FALSE, TRUE);

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

//EXIT:
	return status;
}

/**
 * D-Bus callback for the display dim method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean display_dim_req_dbus_cb(DBusMessage *const msg)
{
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	gboolean status = FALSE;

	mce_log(LL_DEBUG,
		"Received display dim request");

	execute_datapipe(&display_state_pipe,
			 GINT_TO_POINTER(MCE_DISPLAY_DIM),
			 FALSE, TRUE);

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

//EXIT:
	return status;
}

/**
 * D-Bus callback for the display off method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean display_off_req_dbus_cb(DBusMessage *const msg)
{
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	gboolean status = FALSE;

	mce_log(LL_DEBUG,
		"Received display off request");

	execute_datapipe(&display_state_pipe,
			 GINT_TO_POINTER(MCE_DISPLAY_OFF),
			 FALSE, TRUE);

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

//EXIT:
	return status;
}

/**
 * D-Bus callback for display blanking prevent request method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean display_blanking_pause_req_dbus_cb(DBusMessage *const msg)
{
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	gboolean status = FALSE;

	mce_log(LL_DEBUG,
		"Received blanking pause request");

	request_display_blanking_pause();

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

//EXIT:
	return status;
}

/**
 * D-Bus callback for the powerup req method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean powerup_req_dbus_cb(DBusMessage *const msg)
{
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	gboolean status = FALSE;

	mce_log(LL_DEBUG,
		"Received powerup request");

	request_powerup();

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

//EXIT:
	return status;
}

/**
 * D-Bus callback for the reboot req method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean reboot_req_dbus_cb(DBusMessage *const msg)
{
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	gboolean status = FALSE;

	mce_log(LL_DEBUG,
		"Received reboot request");

	request_reboot();

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

//EXIT:
	return status;
}

/**
 * D-Bus callback for the shutdown req method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean shutdown_req_dbus_cb(DBusMessage *const msg)
{
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	gboolean status = FALSE;

	mce_log(LL_DEBUG,
		"Received shutdown request");

	request_normal_shutdown();

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

//EXIT:
	return status;
}

/**
 * Handle alarm UI state change
 *
 * @param data The alarm state stored in a pointer
 */
static void alarm_ui_state_trigger(gconstpointer data)
{
	alarm_ui_state_t alarm_ui_state = GPOINTER_TO_INT(data);

	switch (alarm_ui_state) {
	case MCE_ALARM_UI_VISIBLE_INT32:
		mce_log(LL_DEBUG,
			"Setting DSME alarm state to SET");
		dsme_set_alarm_state(DSME_ALARM_SET);
		break;

	case MCE_ALARM_UI_SNOOZED_INT32:
		mce_log(LL_DEBUG,
			"Setting DSME alarm state to SNOOZED");
		dsme_set_alarm_state(DSME_ALARM_SNOOZED);
		break;

	case MCE_ALARM_UI_OFF_INT32:
		mce_log(LL_DEBUG,
			"Setting DSME alarm state to OFF");
		dsme_set_alarm_state(DSME_ALARM_NOT_SET);
		break;

	default:
		break;
	}
}

/**
 * D-Bus callback for the battery low signal
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean battery_low_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	(void)msg;

	mce_log(LL_DEBUG,
		"Received battery low signal");
	// FIXME: error handling?
	(void)send_data_save();
//	execute_datapipe_output_triggers(&led_pattern_activate_pipe, MCE_LED_PATTERN_BATTERY_LOW, FALSE);

	status = TRUE;

//EXIT:
	return status;
}

/**
 * D-Bus callback for the battery empty signal
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean battery_empty_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	(void)msg;

	mce_log(LL_DEBUG,
		"Received battery empty signal");
	lowbat_shutdown_timeout_cb_id =
		g_timeout_add(LOWBAT_SHUTDOWN_DELAY,
			      lowbat_shutdown_timeout_cb, NULL);
	// FIXME: error handling?
	(void)send_data_save();

	status = TRUE;

//EXIT:
	return status;
}

/**
 * D-Bus callback for the battery full signal
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean battery_full_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	(void)msg;

	mce_log(LL_DEBUG,
		"Received battery full signal");
	execute_datapipe_output_triggers(&led_pattern_deactivate_pipe, MCE_LED_PATTERN_BATTERY_CHARGING, FALSE);
	execute_datapipe_output_triggers(&led_pattern_activate_pipe, MCE_LED_PATTERN_BATTERY_FULL, FALSE);

	status = TRUE;

//EXIT:
	return status;
}

/**
 * D-Bus callback for the charger_charging_on signal
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean charger_charging_on_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	(void)msg;

	mce_log(LL_DEBUG,
		"Received charger_charging_on signal");
	/* In case these are active; there's no harm to call them anyway */
	execute_datapipe_output_triggers(&led_pattern_deactivate_pipe, MCE_LED_PATTERN_BATTERY_FULL, FALSE);
	execute_datapipe_output_triggers(&led_pattern_deactivate_pipe, MCE_LED_PATTERN_BATTERY_LOW, FALSE);

	execute_datapipe_output_triggers(&led_pattern_activate_pipe, MCE_LED_PATTERN_BATTERY_CHARGING, FALSE);
#if 0
	/* We cancel the low-battery shutdown here rather than
	 * when the charger_connected signal arrives, since we need to
	 * know that charging actually takes place before we can
	 * be sure that we do not need to shutdown
	 */
	if (lowbat_shutdown_timeout_cb_id != 0) {
		g_source_remove(lowbat_shutdown_timeout_cb_id);
		lowbat_shutdown_timeout_cb_id = 0;
	}
#endif

	status = TRUE;

//EXIT:
	return status;
}

/**
 * D-Bus callback for the charger_charging_off signal
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean charger_charging_off_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	(void)msg;

	mce_log(LL_DEBUG,
		"Received charger_charging_off signal");
	/* In case these are active; there's no harm to call them anyway */
	execute_datapipe_output_triggers(&led_pattern_deactivate_pipe, MCE_LED_PATTERN_BATTERY_FULL, FALSE);
	execute_datapipe_output_triggers(&led_pattern_deactivate_pipe, MCE_LED_PATTERN_BATTERY_CHARGING, FALSE);

	status = TRUE;

//EXIT:
	return status;
}

/**
 * D-Bus callback for the charger_connected signal
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean charger_connected_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	(void)msg;

	mce_log(LL_DEBUG,
		"Received charger_connected signal");

	charger_connected = TRUE;
	update_blanking_inhibit();

	status = TRUE;

//EXIT:
	return status;
}

/**
 * D-Bus callback for the charger_disconnected signal
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean charger_disconnected_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	(void)msg;

	mce_log(LL_DEBUG,
		"Received charger_disconnected signal");

	charger_connected = FALSE;
	update_blanking_inhibit();

	status = TRUE;

//EXIT:
	return status;
}

#if 0
/**
 * D-Bus callback for the cancel lowbat shutdown signal
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean cancel_lowbat_shutdown_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	(void)msg;

	mce_log(LL_DEBUG,
		"Received charging on signal");
	cancel_lowbat_shutdown();

	status = TRUE;

//EXIT:
	return status;
}
#endif /* XXX: charger replugging */

/**
 * Setup display settings
 */
void setup_display_settings(void)
{
	mce_log(LL_DEBUG,
		"Setting display settings");

	/* Set blank timeout twice, since we don't know the defaults
	 * that DSME uses, and since DSME has checks that deny changes
	 * that make dim timeout > blank timeout, we need both variations
	 */
	dsme_set_blank_timeout(disp_blank_timeout);
	dsme_set_dim_timeout(disp_dim_timeout);
	dsme_set_blank_timeout(disp_blank_timeout);
}

/**
 * D-Bus callback for the desktop startup notification signal
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean desktop_startup_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	(void)msg;

	mce_log(LL_DEBUG,
		"Received desktop startup notification");

	setup_display_settings();
	execute_datapipe_output_triggers(&led_pattern_deactivate_pipe, MCE_LED_PATTERN_POWER_ON, FALSE);

	status = TRUE;

//EXIT:
	return status;
}

/**
 * Set device autolock timeout and enable/disable it
 *
 * @param timeout New device autolock timeout
 * @param enabled TRUE to enable autolock, FALSE to disable autolock
 */
void set_device_autolock(const gint timeout, const gboolean enabled)
{
	dsmemsg_timeout_change_t msg;

	/* Set up the message */
	msg.type = DSM_MSGTYPE_DEVICELOCK_AUTOLOCK_SET_TIMEOUT;
	msg.size = sizeof msg;
	msg.timeout = (enabled == TRUE) ? timeout : -1;

	/* Send the message */
	mce_dsme_send(&msg);
	mce_log(LL_DEBUG,
		"DSM_MSGTYPE_DEVICELOCK_AUTOLOCK_SET_TIMEOUT (%i) "
		"sent to DSME",
		msg.timeout);
}

/**
 * Request update of Charger status
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean request_charger_status(void)
{
	return dbus_send(BME_SERVICE, BME_REQUEST_PATH, BME_REQUEST_IF,
			 BME_STATUS_INFO_REQ, NULL, DBUS_TYPE_INVALID);
}

/**
 * Check for pending alarms;
 * are there any pending alarms in the specified interval?
 *
 * @param first Start of interval
 * @param last End of interval
 * @return TRUE if there are alarms pending,
 *         FALSE if no alarms are pending
 */
static gboolean check_pending_alarm(time_t first, time_t last)
{
	dbus_int32_t flag_mask = ALARM_EVENT_BOOT;
	dbus_int32_t flags = ALARM_EVENT_BOOT;
	gboolean status = FALSE;
	DBusMessage *msg = NULL;
	DBusMessage *reply = NULL;
	dbus_int64_t first_64 = first;
	dbus_int64_t last_64 = last;
	dbus_int64_t **array;
	dbus_int32_t length;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	/* Query next pending alarm */
	msg = dbus_new_method_call(ALARMD_SERVICE,
				   ALARMD_PATH,
				   ALARMD_INTERFACE,
				   ALARM_EVENT_QUERY);

	/* Append the parameters */
	if (dbus_message_append_args(msg,
				     DBUS_TYPE_UINT64, &first_64,
				     DBUS_TYPE_UINT64, &last_64,
				     DBUS_TYPE_INT32, &flag_mask,
				     DBUS_TYPE_INT32, &flags,
				     DBUS_TYPE_INVALID) == FALSE) {
		mce_log(LL_CRIT,
			"Failed to append argument to D-Bus message for %s",
			ALARM_EVENT_QUERY);
		dbus_message_unref(msg);
		goto EXIT;
	}

	if ((reply = dbus_send_message_with_reply_and_block(msg)) == NULL) {
		mce_log(LL_ERR,
			"Cannot call method %s with reply; "
			"assuming no alarms pending",
			ALARM_EVENT_QUERY);
		goto EXIT;
	}

	if (dbus_message_get_args(reply, &error,
				  DBUS_TYPE_ARRAY, DBUS_TYPE_INT32,
				  &array, &length,
				  DBUS_TYPE_INVALID) == FALSE) {
		mce_log(LL_ERR,
			"Failed to get reply argument from %s: %s; %s",
			"alarmd",
			error.message,
			"assuming no alarms pending");
		dbus_message_unref(reply);
		dbus_error_free(&error);
		goto EXIT;
	}

	/* If the array is not empty, there is at least one pending alarm */
	if (length != 0)
		status = TRUE;

	dbus_message_unref(reply);

EXIT:
	return status;
}

/**
 * Request normal shutdown with alarm policy
 */
void request_normal_shutdown(void)
{
	int olderrno = errno;
	gboolean pending;
	time_t now;

	/* If the alarm UI is visible, close it */
	if ((mce_get_submode_int32() & MCE_ALARM_SUBMODE)) {
		(void)execute_datapipe(&alarm_ui_state_pipe,
				       MCE_ALARM_UI_OFF_INT32,
				       FALSE, TRUE);
	}

	if ((now = time(NULL)) == -1) {
		now = 0;
		errno = olderrno;
	}

	pending = check_pending_alarm(now, now + SNOOZE_TIMEOUT);

	/* If there's a pending alarm the next SNOOZE_TIMEOUT
	 * seconds, set the DSME alarm state to DSME_ALARM_SNOOZED
	 * to make sure we don't miss alarms during a shutdown
	 */
	if (pending == TRUE) {
		mce_log(LL_DEBUG,
			"We have a pending alarm within the next "
			"%d seconds, alert DSME",
			SNOOZE_TIMEOUT);
		dsme_set_alarm_state(DSME_ALARM_SNOOZED);
	}

	request_shutdown();
}

/**
 * Handle display state change
 *
 * @param data The display state stored in a pointer
 */
static void display_state_trigger(gconstpointer data)
{
	display_state_t display_state = GPOINTER_TO_INT(data);

	/* This will send the correct state
	 * since the pipe contains the new value
	 */
	mce_send_display_status(NULL);

	/* If we already have the right state, don't send another request */
	if (cached_display_state == display_state)
		goto EXIT;

	switch (display_state) {
	case MCE_DISPLAY_OFF:
		request_display_off();
		break;

	case MCE_DISPLAY_DIM:
		request_display_dim();
		break;

	case MCE_DISPLAY_ON:
	default:
		request_display_on();
		break;
	}

	/* If the cache isn't up to date, update it */
	cached_display_state = display_state;

EXIT:
	return;
}

/**
 * Datapipe trigger for the keyboard slide
 *
 * @param data 1 if the keyboard is open, 0 if the keyboard is closed
 */
static void keyboard_slide_trigger(gconstpointer const data)
{
	(void)data;

	/* Unblank the screen if we're in USER state */
	if (datapipe_get_gint(system_state_pipe) == MCE_STATE_USER) {
		execute_datapipe(&display_state_pipe,
				 GINT_TO_POINTER(MCE_DISPLAY_ON),
				 FALSE, TRUE);
	}
}

/**
 * Initialise dsmesock connection
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean init_dsmesock(void)
{
	GIOChannel *iochan;
	gboolean status = FALSE;

	if (dsme_conn == NULL) {
		if ((dsme_conn = dsmesock_connect()) == NULL) {
			mce_log(LL_CRIT,
				"Failed to open DSME socket");
			goto EXIT;
		}
	}

	if ((iochan = g_io_channel_unix_new(dsme_conn->fd)) == NULL) {
		mce_log(LL_CRIT,
			"Failed to set up I/O channel for DSME socket");
		goto EXIT;
	}

	(void)g_io_add_watch(iochan, G_IO_IN | G_IO_PRI,
			     io_data_ready_cb, NULL);
	(void)g_io_add_watch(iochan, G_IO_ERR | G_IO_HUP,
			     io_error_cb, NULL);

	/* Query the current system state */
	query_system_state();

	status = TRUE;

EXIT:
	return status;
}

/**
 * Parse the soft poweroff connectivity policy
 *
 * @param string The string to parse
 * @param action A pointer to the variable to store the policy in
 * @return TRUE if the string contained a valid policy,
 *         FALSE if the policy was invalid
 */
static gboolean parse_policy(char *string, gint *action)
{
	gboolean status = FALSE;

	/** @todo This should probably not be hardcoded */
	if (!strcmp(string, "forceoffline")) {
		*action = SOFTOFF_CONNECTIVITY_FORCE_OFFLINE;
	} else if (!strcmp(string, "softoffline")) {
		*action = SOFTOFF_CONNECTIVITY_SOFT_OFFLINE;
	} else if (!strcmp(string, "retain")) {
		*action = SOFTOFF_CONNECTIVITY_RETAIN;
	} else if (!strcmp(string, "offline")) {
		*action = SOFTOFF_CONNECTIVITY_OFFLINE;
	} else if (!strcmp(string, "restore")) {
		*action = SOFTOFF_CONNECTIVITY_RESTORE;
	} else {
		mce_log(LL_WARN,
			"Unknown Soft Offline connectivity policy");
		goto EXIT;
	}

	status = TRUE;

EXIT:
	return status;
}

/**
 * Init function for the mce-dsme component
 *
 * @param debug_mode TRUE - do not exit if dsme fails
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_dsme_init(gboolean debug_mode)
{
	gboolean status = FALSE;
	gint disp_brightness = DEFAULT_DISP_BRIGHTNESS;
	gchar *tmp = NULL;

	/* Append triggers/filters to datapipes */
	append_output_trigger_to_datapipe(&display_brightness_pipe,
					  display_brightness_trigger);
	append_output_trigger_to_datapipe(&keyboard_slide_pipe,
					  keyboard_slide_trigger);
	append_output_trigger_to_datapipe(&display_state_pipe,
					  display_state_trigger);
	append_output_trigger_to_datapipe(&alarm_ui_state_pipe,
					  alarm_ui_state_trigger);

	mce_log(LL_DEBUG,
		"Connecting to DSME sock");

	if (init_dsmesock() == FALSE) {
		if (debug_mode == TRUE) {
			dsme_disabled = TRUE;
		} else {
			goto EXIT;
		}
	}

	/* Display brightness */
	/* Since we've set a default, error handling is unnecessary */
	(void)mce_gconf_get_int(MCE_GCONF_DISPLAY_BRIGHTNESS_PATH,
				&disp_brightness);

	(void)execute_datapipe(&display_brightness_pipe,
			       GINT_TO_POINTER(disp_brightness),
			       FALSE, TRUE);

	if (mce_gconf_notifier_add(MCE_GCONF_DISPLAY_PATH,
				   MCE_GCONF_DISPLAY_BRIGHTNESS_PATH,
				   dsme_gconf_cb,
				   &disp_brightness_gconf_cb_id) == FALSE)
		goto EXIT;

	/* Display blank */
	/* Since we've set a default, error handling is unnecessary */
	(void)mce_gconf_get_int(MCE_GCONF_DISPLAY_BLANK_TIMEOUT_PATH,
				&disp_blank_timeout);

	if (mce_gconf_notifier_add(MCE_GCONF_DISPLAY_PATH,
				   MCE_GCONF_DISPLAY_BLANK_TIMEOUT_PATH,
				   dsme_gconf_cb,
				   &disp_blank_timeout_gconf_cb_id) == FALSE)
		goto EXIT;

	/* Display dim */
	/* Since we've set a default, error handling is unnecessary */
	(void)mce_gconf_get_int(MCE_GCONF_DISPLAY_DIM_TIMEOUT_PATH,
				&disp_dim_timeout);

	if (mce_gconf_notifier_add(MCE_GCONF_DISPLAY_PATH,
				   MCE_GCONF_DISPLAY_DIM_TIMEOUT_PATH,
				   dsme_gconf_cb,
				   &disp_dim_timeout_gconf_cb_id) == FALSE)
		goto EXIT;

	dsme_set_blank_timeout(DEFAULT_BOOT_BLANK_TIMEOUT);
	dsme_set_dim_timeout(DEFAULT_BOOT_DIM_TIMEOUT);
	dsme_set_blank_timeout(DEFAULT_BOOT_BLANK_TIMEOUT);

	/* Don't blank on charger */
	/* Since we've set a default, error handling is unnecessary */
	(void)mce_gconf_get_bool(MCE_GCONF_DISPLAY_ON_WITH_CHARGER_PATH,
				 &disp_on_with_charger);

	if (mce_gconf_notifier_add(MCE_GCONF_DISPLAY_PATH,
				   MCE_GCONF_DISPLAY_ON_WITH_CHARGER_PATH,
				   dsme_gconf_cb,
				   &disp_on_with_charger_gconf_cb_id) == FALSE)
		goto EXIT;

	/* Update charger status */
	request_charger_status();

	/* Register with DSME's process watchdog */
	dsme_init_processwd();

	/* get_display_status */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_DISPLAY_STATUS_GET,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 display_status_get_dbus_cb) == FALSE)
		goto EXIT;

	/* req_display_state_on */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_DISPLAY_ON_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 display_on_req_dbus_cb) == FALSE)
		goto EXIT;

	/* req_display_state_dim */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_DISPLAY_DIM_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 display_dim_req_dbus_cb) == FALSE)
		goto EXIT;

	/* req_display_state_off */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_DISPLAY_OFF_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 display_off_req_dbus_cb) == FALSE)
		goto EXIT;

	/* req_display_blanking_pause */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_PREVENT_BLANK_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 display_blanking_pause_req_dbus_cb) == FALSE)
		goto EXIT;

	/* req_powerup */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_POWERUP_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 powerup_req_dbus_cb) == FALSE)
		goto EXIT;

	/* req_reboot */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_REBOOT_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 reboot_req_dbus_cb) == FALSE)
		goto EXIT;

	/* req_shutdown */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_SHUTDOWN_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 shutdown_req_dbus_cb) == FALSE)
		goto EXIT;

	/* battery_low */
	if (mce_dbus_handler_add(BME_SIGNAL_IF,
				 BME_BATTERY_LOW,
				 NULL,
				 DBUS_MESSAGE_TYPE_SIGNAL,
				 battery_low_dbus_cb) == FALSE)
		goto EXIT;

	/* battery_empty */
	if (mce_dbus_handler_add(BME_SIGNAL_IF,
				 BME_BATTERY_EMPTY,
				 NULL,
				 DBUS_MESSAGE_TYPE_SIGNAL,
				 battery_empty_dbus_cb) == FALSE)
		goto EXIT;

	/* battery_full */
	if (mce_dbus_handler_add(BME_SIGNAL_IF,
				 BME_BATTERY_FULL,
				 NULL,
				 DBUS_MESSAGE_TYPE_SIGNAL,
				 battery_full_dbus_cb) == FALSE)
		goto EXIT;

	/* charger_charging_on */
	if (mce_dbus_handler_add(BME_SIGNAL_IF,
				 BME_CHARGER_CHARGING_ON,
				 NULL,
				 DBUS_MESSAGE_TYPE_SIGNAL,
				 charger_charging_on_dbus_cb) == FALSE)
		goto EXIT;

	/* charger_charging_off */
	if (mce_dbus_handler_add(BME_SIGNAL_IF,
				 BME_CHARGER_CHARGING_OFF,
				 NULL,
				 DBUS_MESSAGE_TYPE_SIGNAL,
				 charger_charging_off_dbus_cb) == FALSE)
		goto EXIT;

	/* charger_connected */
	if (mce_dbus_handler_add(BME_SIGNAL_IF,
				 BME_CHARGER_CONNECTED,
				 NULL,
				 DBUS_MESSAGE_TYPE_SIGNAL,
				 charger_connected_dbus_cb) == FALSE)
		goto EXIT;

	/* charger_disconnected */
	if (mce_dbus_handler_add(BME_SIGNAL_IF,
				 BME_CHARGER_DISCONNECTED,
				 NULL,
				 DBUS_MESSAGE_TYPE_SIGNAL,
				 charger_disconnected_dbus_cb) == FALSE)
		goto EXIT;

	/* NameOwnerChanged: hildon-desktop */
	if (mce_dbus_handler_add("org.freedesktop.DBus",
				 "NameOwnerChanged",
				 "arg0='com.nokia.hildon-desktop'",
				 DBUS_MESSAGE_TYPE_SIGNAL,
				 desktop_startup_dbus_cb) == FALSE)
		goto EXIT;

	/* init_done */
	if (mce_dbus_handler_add("com.nokia.startup.signal",
				 "init_done",
				 NULL,
				 DBUS_MESSAGE_TYPE_SIGNAL,
				 init_done_dbus_cb) == FALSE)
		goto EXIT;

	/* Get configuration options */
	tmp = mce_conf_get_string(MCE_CONF_SOFTPOWEROFF_GROUP,
				  MCE_CONF_SOFTPOWEROFF_CONNECTIVITY_POLICY_CHARGER,
				  "",
				  NULL);

	/* Since we've set a default, error handling is unnecessary */
	(void)parse_policy(tmp, &softoff_connectivity_policy_charger);
	g_free(tmp);

	tmp = mce_conf_get_string(MCE_CONF_SOFTPOWEROFF_GROUP,
				  MCE_CONF_SOFTPOWEROFF_CONNECTIVITY_POLICY_BATTERY,
				  "",
				  NULL);

	/* Since we've set a default, error handling is unnecessary */
	(void)parse_policy(tmp, &softoff_connectivity_policy_battery);
	g_free(tmp);

	tmp = mce_conf_get_string(MCE_CONF_SOFTPOWEROFF_GROUP,
				  MCE_CONF_SOFTPOWEROFF_CONNECTIVITY_POLICY_POWERON,
				  "",
				  NULL);

	/* Since we've set a default, error handling is unnecessary */
	(void)parse_policy(tmp, &softoff_connectivity_policy_poweron);
	g_free(tmp);

	/* Request display on to get the state machine in sync */
	execute_datapipe(&display_state_pipe,
			 GINT_TO_POINTER(MCE_DISPLAY_ON),
			 FALSE, TRUE);

	/* Query the current system state */
	query_system_state();

	status = TRUE;

EXIT:
	return status;
}

/**
 * Exit function for the mce-dsme component
 *
 * @todo D-Bus unregistration
 * @todo trigger unregistratioin
 */
void mce_dsme_exit(void)
{
	if (dsme_conn != NULL) {
		mce_log(LL_DEBUG,
			"Disabling DSME process watchdog");
		dsme_exit_processwd();

		mce_log(LL_DEBUG,
			"Closing DSME sock");
		dsmesock_close(dsme_conn);
	}

	/* Remove triggers/filters from datapipes */
	remove_output_trigger_from_datapipe(&alarm_ui_state_pipe,
					    alarm_ui_state_trigger);
	remove_output_trigger_from_datapipe(&display_state_pipe,
					    display_state_trigger);
	remove_output_trigger_from_datapipe(&keyboard_slide_pipe,
					    keyboard_slide_trigger);
	remove_output_trigger_from_datapipe(&display_brightness_pipe,
					    display_brightness_trigger);

	/* Remove the timeout source for the battery low shutdown */
	if (lowbat_shutdown_timeout_cb_id != 0) {
		g_source_remove(lowbat_shutdown_timeout_cb_id);
		lowbat_shutdown_timeout_cb_id = 0;
	}

	/* Remove the timeout source for the display blank prevention */
	if (display_blank_timeout_cb_id != 0) {
		g_source_remove(display_blank_timeout_cb_id);
		display_blank_timeout_cb_id = 0;
	}

	/* Remove the timeout source for state transitions */
	if (transition_timeout_cb_id != 0) {
		g_source_remove(transition_timeout_cb_id);
		transition_timeout_cb_id = 0;
	}
}
