/**
 * @file mce-dbus.c
 * D-Bus handling code for the Mode Control Entity
 * <p>
 * Copyright © 2004-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 * @author Ismo Laitinen <ismo.laitinen@nokia.com>
 */
#include <glib.h>

#include <stdarg.h>			/* va_start(), va_end() */
#include <stdlib.h>			/* exit(), EXIT_FAILURE */
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>	/* dbus_connection_setup_with_g_main */

#include "mce.h"
#include "mce-dbus.h"

#include "mce-log.h"			/* mce_log(), LL_* */

/** List of all D-Bus handlers */
static GSList *dbus_handlers = NULL;

/** D-Bus handler structure */
typedef struct {
	gboolean (*callback)(DBusMessage *const msg);	/**< Handler callback */
	gchar *interface;		/**< The interface to listen on */
	gchar *rules;			/**< Additional matching rules */
	gchar *name;			/**< Method call or signal name */
	guint type;			/**< DBUS_MESSAGE_TYPE */
} handler_struct;

/** Pointer to the DBusConnection */
static DBusConnection *dbus_connection = NULL;

/**
 * Create a new D-Bus signal, with proper error checking
 * will exit the mainloop if an error occurs
 *
 * @param path The signal path
 * @param interface The signal interface
 * @param name The name of the signal to send
 * @return A new DBusMessage
 */
DBusMessage *dbus_new_signal(const gchar *const path,
			     const gchar *const interface,
			     const gchar *const name)
{
	DBusMessage *msg;

	if ((msg = dbus_message_new_signal(path, interface, name)) == NULL) {
		mce_log(LL_CRIT, "No memory for new signal!");
		g_main_loop_quit(mainloop);
		exit(EXIT_FAILURE);
	}

	return msg;
}

#if 0
/**
 * Create a new D-Bus error message, with proper error checking
 * will exit the mainloop if an error occurs
 *
 * @param message The DBusMessage that caused the error message to be sent
 * @param error The message to send
 * @return A new DBusMessage
 */
static DBusMessage *dbus_new_error(DBusMessage *const message,
				   const gchar *const error)
{
	DBusMessage *error_msg;

	if ((error_msg = dbus_message_new_error(message, error,
						NULL)) == NULL) {
		mce_log(LL_CRIT, "No memory for new D-Bus error message!");
		g_main_loop_quit(mainloop);
		exit(EXIT_FAILURE);
	}

	return error_msg;
}
#endif

/**
 * Create a new D-Bus method call, with proper error checking
 * will exit the mainloop if an error occurs
 *
 * @param service The method call service
 * @param path The method call path
 * @param interface The method call interface
 * @param name The name of the method to call
 * @return A new DBusMessage
 */
DBusMessage *dbus_new_method_call(const gchar *const service,
				  const gchar *const path,
				  const gchar *const interface,
				  const gchar *const name)
{
	DBusMessage *msg;

	if ((msg = dbus_message_new_method_call(service, path,
						interface, name)) == NULL) {
		mce_log(LL_CRIT,
			"Cannot allocate memory for D-Bus method call!");
		g_main_loop_quit(mainloop);
		exit(EXIT_FAILURE);
	}

	return msg;
}

/**
 * Create a new D-Bus method call reply, with proper error checking
 * will exit the mainloop if an error occurs
 *
 * @param message The DBusMessage to reply to
 * @return A new DBusMessage
 */
DBusMessage *dbus_new_method_reply(DBusMessage *const message)
{
	DBusMessage *msg;

	if ((msg = dbus_message_new_method_return(message)) == NULL) {
		mce_log(LL_CRIT, "No memory for new reply!");
		g_main_loop_quit(mainloop);
		exit(EXIT_FAILURE);
	}

	return msg;
}

/**
 * Send a D-Bus message
 * Side-effects: frees msg
 *
 * @param msg The D-Bus message to send
 * @return TRUE on success, FALSE on out of memory
 */
gboolean dbus_send_message(DBusMessage *const msg)
{
	gboolean status = FALSE;

	if (dbus_connection_send(dbus_connection, msg, NULL) == FALSE) {
		mce_log(LL_CRIT,
			"Out of memory when sending D-Bus message");
		goto EXIT;
	}

	dbus_connection_flush(dbus_connection);
	status = TRUE;

EXIT:
	dbus_message_unref(msg);

	return status;
}

/**
 * Send a D-Bus message and setup a reply callback
 * Side-effects: frees msg
 *
 * @param msg The D-Bus message to send
 * @param callback The reply callback
 * @return TRUE on success, FALSE on failure
 */
gboolean dbus_send_message_with_reply_handler(DBusMessage *const msg,
					      DBusPendingCallNotifyFunction callback)
{
	DBusPendingCall *pending_call;
	gboolean status = FALSE;

	if (dbus_connection_send_with_reply(dbus_connection, msg,
					    &pending_call, -1) == FALSE) {
		mce_log(LL_CRIT,
			"Out of memory when sending D-Bus message");
		goto EXIT;
	} else if (pending_call == NULL) {
		mce_log(LL_ERR,
			"D-Bus connection disconnected");
		goto EXIT;
	}

	dbus_connection_flush(dbus_connection);

	if (dbus_pending_call_set_notify(pending_call, callback, NULL, NULL) == FALSE) {
		mce_log(LL_CRIT,
			"Out of memory when sending D-Bus message");
		goto EXIT;
	}

	status = TRUE;

EXIT:
	dbus_message_unref(msg);

	return status;
}

/**
 * Send a D-Bus message and wait for a reply
 * Side-effects: frees msg
 *
 * @param msg The D-Bus message to send
 * @return A DBusMessage with the reply, or NULL on failure
 */
DBusMessage *dbus_send_message_with_reply_and_block(DBusMessage *const msg)
{
	DBusMessage *reply;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	reply = dbus_connection_send_with_reply_and_block(dbus_connection, msg,								  -1, &error);

	dbus_message_unref(msg);

	if (dbus_error_is_set(&error) == TRUE) {
		mce_log(LL_ERR, "Error sending with reply: %s", error.message);
		dbus_error_free(&error);
		reply = NULL;
	}

	return reply;
}

/**
 * Generic function to send D-Bus messages and signals
 * to send a signal, call dbus_send with service == NULL
 *
 * @todo Make it possible to send D-Bus replies as well
 *
 * @param service D-Bus service; for signals, set to NULL
 * @param path D-Bus path
 * @param interface D-Bus interface
 * @param name The D-Bus method or signal name to send to
 * @param callback A reply callback, or NULL to set no reply;
 *                 for signals, this is unused, but please use NULL
 *                 for consistency
 * @param first_arg_type The DBUS_TYPE of the first argument in the list
 * @param ... The arguments to append to the D-Bus message;
 *            terminate with DBUS_TYPE_INVALID
 *            Note: the arguments MUST be passed by reference
 * @return TRUE on success, FALSE on failure
 */
gboolean dbus_send(const gchar *const service, const gchar *const path,
		   const gchar *const interface, const gchar *const name,
		   DBusPendingCallNotifyFunction callback,
		   int first_arg_type, ...)
{
	DBusMessage *msg;
	gboolean status = FALSE;
	va_list var_args;

	if (service != NULL) {
		msg = dbus_new_method_call(service, path, interface, name);

		if (callback == NULL)
			dbus_message_set_no_reply(msg, TRUE);
	} else {
		if (callback != NULL) {
			mce_log(LL_ERR,
				"Programmer snafu! "
				"dbus_send() called with a DBusPending "
				"callback for a signal.  Whoopsie!");
			callback = NULL;
		}

		msg = dbus_new_signal(path, interface, name);
	}

	/* Append the arguments, if any */
	va_start(var_args, first_arg_type);

	if (first_arg_type != DBUS_TYPE_INVALID) {
		if (dbus_message_append_args_valist(msg,
						    first_arg_type,
						    var_args) == FALSE) {
			mce_log(LL_CRIT,
				"Failed to append arguments to D-Bus message "
				"for %s",
				name);
			dbus_message_unref(msg);
			goto EXIT;
		}
	}

	/* Send the signal / call the method */
	if (callback == NULL) {
		status = dbus_send_message(msg);
	} else {
		status = dbus_send_message_with_reply_handler(msg, callback);
	}

EXIT:
	va_end(var_args);

	return status;
}

/**
 * D-Bus callback for the version get method call
 *
 * @param msg The D-Bus message to reply to
 * @return TRUE on success, FALSE on failure
 */
static gboolean version_get_dbus_cb(DBusMessage *const msg)
{
	static const gchar *const versionstring = G_STRINGIFY(PRG_VERSION);
	DBusMessage *reply = NULL;
	gboolean status = FALSE;

	mce_log(LL_DEBUG, "Received version information request");

	/* Create a reply */
	reply = dbus_new_method_reply(msg);

	/* Append the version information */
	if (dbus_message_append_args(reply,
				     DBUS_TYPE_STRING, &versionstring,
				     DBUS_TYPE_INVALID) == FALSE) {
		mce_log(LL_CRIT,
			"Failed to append reply argument to D-Bus message "
			"for %s",
			MCE_VERSION_GET);
		dbus_message_unref(reply);
		goto EXIT;
	}

	/* Send the message */
	status = dbus_send_message(reply);

EXIT:
	return status;
}

/**
 * D-Bus message handler
 *
 * @param connection Unused
 * @param msg The D-Bus message received
 * @param user_data Unused
 * @return DBUS_HANDLER_RESULT_HANDLED for handled messages
 *         DBUS_HANDLER_RESULT_NOT_HANDLED for unhandled messages
 */
static DBusHandlerResult msg_handler(DBusConnection *const connection,
				     DBusMessage *const msg,
				     gpointer const user_data)
{
	guint status = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	handler_struct *handler = NULL;
	guint i;

	(void)connection;
	(void)user_data;

	for (i = 0; (handler = g_slist_nth_data(dbus_handlers,
						i)) != NULL; i++) {
		switch (handler->type) {
		case DBUS_MESSAGE_TYPE_METHOD_CALL:
			if (dbus_message_is_method_call(msg,
							handler->interface,
							handler->name) == TRUE) {
				handler->callback(msg);
				status = DBUS_HANDLER_RESULT_HANDLED;
				goto EXIT;
			}

			break;

		case DBUS_MESSAGE_TYPE_ERROR:
			if (dbus_message_is_error(msg,
						  handler->name) == TRUE) {
				handler->callback(msg);
				status = DBUS_HANDLER_RESULT_HANDLED;
				goto EXIT;
			}

			break;

		case DBUS_MESSAGE_TYPE_SIGNAL:
			if (dbus_message_is_signal(msg,
						   handler->interface,
						   handler->name) == TRUE) {
				handler->callback(msg);
				status = DBUS_HANDLER_RESULT_HANDLED;
			}

			break;

		default:
			mce_log(LL_ERR,
				"There's a bug somewhere in MCE; something "
				"has registered an invalid D-Bus handler");
			break;
		}
	}

EXIT:
	return status;
}

/**
 * Register a D-Bus signal or method handler
 *
 * @param interface The interface to listen on
 * @param name The signal/method call to listen for
 * @param rules Additional matching rules
 * @param type DBUS_MESSAGE_TYPE
 * @param callback The callback function
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_dbus_handler_add(const gchar *const interface,
			      const gchar *const name,
			      const gchar *const rules,
			      const guint type,
			      gboolean (*callback)(DBusMessage *const msg))
{
	gboolean status = FALSE;
	gchar *match = NULL;
	handler_struct *h = NULL;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	if (type == DBUS_MESSAGE_TYPE_SIGNAL) {
		match = g_strdup_printf("type='signal'"
					"%s%s%s"
					", member='%s'"
					"%s%s",
					interface ? ", interface='" : "",
					interface ? interface : "",
					interface ? "'" : "",
					name,
					rules ? ", " : "",
					rules ? rules : "");
	} else if (type == DBUS_MESSAGE_TYPE_METHOD_CALL) {
		match = g_strdup_printf("type='method_call'"
					"%s%s%s"
					", member='%s'"
					"%s%s",
					interface ? ", interface='" : "",
					interface ? interface : "",
					interface ? "'" : "",
					name,
					rules ? ", " : "",
					rules ? rules : "");
	} else {
		mce_log(LL_CRIT,
			"There's definitely a programming error somewhere; "
			"MCE is trying to register an invalid message type");
		goto EXIT;
	}

	if (match == NULL) {
		mce_log(LL_CRIT, "Failed to allocate memory for match");
		goto EXIT;
	}

	if ((h = g_try_malloc(sizeof (handler_struct))) == NULL) {
		mce_log(LL_CRIT, "Failed to allocate memory for h");
		goto EXIT;
	}

	h->interface = NULL;

	if (interface && (h->interface = g_strdup(interface)) == NULL) {
		mce_log(LL_CRIT, "Failed to allocate memory for h->interface");
		g_free(h);
		goto EXIT;
	}

	h->rules = NULL;

	if (rules && (h->rules = g_strdup(rules)) == NULL) {
		mce_log(LL_CRIT, "Failed to allocate memory for h->rules");
		g_free(h->interface);
		g_free(h);
		goto EXIT;
	}

	if ((h->name = g_strdup(name)) == NULL) {
		mce_log(LL_CRIT, "Failed to allocate memory for h->name");
		g_free(h->interface);
		g_free(h->rules);
		g_free(h);
		goto EXIT;
	}

	h->type = type;
	h->callback = callback;

	dbus_bus_add_match(dbus_connection, match, &error);

	if (dbus_error_is_set(&error) == TRUE) {
		mce_log(LL_CRIT, "Failed to add D-Bus match '%s' for '%s'; %s",
			match, h->interface, error.message);
		dbus_error_free(&error);
		g_free(h->interface);
		g_free(h->rules);
		g_free(h);
		goto EXIT;
	}

	dbus_handlers = g_slist_prepend(dbus_handlers, h);

	status = TRUE;

EXIT:
	g_free(match);

	return status;
}

/**
 * Unregister a D-Bus signal or method handler
 * XXX: this probably isn't the ideal interface if we want to call this
 *      "manually"
 *
 * @param handler A pointer to the handler struct that should be removed
 * @param user_data Unused
 */
void mce_dbus_handler_remove(gpointer handler, gpointer user_data)
{
	gchar *match = NULL;
	handler_struct *h = (handler_struct *)handler;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	(void)user_data;

	if (h->type == DBUS_MESSAGE_TYPE_SIGNAL) {
		match = g_strdup_printf("type='signal'"
					"%s%s%s"
					", member='%s'"
					"%s%s",
					h->interface ? ", interface='" : "",
					h->interface ? h->interface : "",
					h->interface ? "'" : "",
					h->name,
					h->rules ? ", " : "",
					h->rules ? h->rules : "");
	} else if (h->type == DBUS_MESSAGE_TYPE_METHOD_CALL) {
		match = g_strdup_printf("type='method_call'"
					"%s%s%s"
					", member='%s'"
					"%s%s",
					h->interface ? ", interface='" : "",
					h->interface ? h->interface : "",
					h->interface ? "'" : "",
					h->name,
					h->rules ? ", " : "",
					h->rules ? h->rules : "");
	} else {
		mce_log(LL_ERR,
			"There's definitely a programming error somewhere; "
			"MCE is trying to unregister an invalid message type");
		/* Don't abort here, since we want to unregister it anyway */
	}

	if (match != NULL) {
		dbus_bus_remove_match(dbus_connection, match, &error);

		if (dbus_error_is_set(&error) == TRUE) {
			mce_log(LL_CRIT, "Failed to remove D-Bus match "
				"'%s' for '%s': %s",
				match, h->interface, error.message);
			dbus_error_free(&error);
		}
	} else {
		mce_log(LL_CRIT, "Failed to allocate memory for match");
	}

	g_free(h->interface);
	g_free(h->rules);
	g_free(h->name);
	g_free(h);

	dbus_handlers = g_slist_remove(dbus_handlers, handler);

//EXIT:
	g_free(match);
}

/**
 * Acquire D-Bus services
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean dbus_acquire_services(void)
{
	gboolean status = FALSE;
	int ret;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	ret = dbus_bus_request_name(dbus_connection, MCE_SERVICE, 0, &error);

	if (ret == DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
		mce_log(LL_DEBUG, "Service %s acquired", MCE_SERVICE);
	} else {
		mce_log(LL_CRIT, "Cannot acquire service: %s", error.message);
		dbus_error_free(&error);
		goto EXIT;
	}

	status = TRUE;

EXIT:
	return status;
}

/**
 * Initialise the message handler used by MCE
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean dbus_init_message_handler(void)
{
	gboolean status = FALSE;

	if (dbus_connection_add_filter(dbus_connection, msg_handler,
				       NULL, NULL) == FALSE) {
		mce_log(LL_CRIT, "Failed to add D-Bus filter");
		goto EXIT;
	}

	status = TRUE;

EXIT:
	return status;
}

/**
 * Init function for the mce-dbus component
 * Pre-requisites: glib mainloop registered
 *
 * @param systembus TRUE to use system bus, FALSE to use session bus
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_dbus_init(const gboolean systembus)
{
	DBusBusType bus_type = DBUS_BUS_SYSTEM;
	gboolean status = FALSE;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	if (systembus == FALSE)
		bus_type = DBUS_BUS_SESSION;

	mce_log(LL_DEBUG, "Establishing D-Bus connection");

	/* Establish D-Bus connection */
	if ((dbus_connection = dbus_bus_get(bus_type,
					    &error)) == NULL) {
		mce_log(LL_CRIT, "Failed to open connection to message bus; %s",
			  error.message);
		dbus_error_free(&error);
		goto EXIT;
	}

	mce_log(LL_DEBUG, "Connecting D-Bus to the mainloop");

	/* Connect D-Bus to the mainloop */
	dbus_connection_setup_with_g_main(dbus_connection, NULL);

	mce_log(LL_DEBUG, "Acquiring D-Bus service");

	/* Acquire D-Bus service */
	if (dbus_acquire_services() == FALSE)
		goto EXIT;

	/* Initialise message handlers */
	if (dbus_init_message_handler() == FALSE)
		goto EXIT;

	/* Register callbacks that are handled inside mce-dbus.c */

	/* get_version */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_VERSION_GET,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 version_get_dbus_cb) == FALSE)
		goto EXIT;

	status = TRUE;

EXIT:
	return status;
}

/**
 * Exit function for the mce-dbus component
 */
void mce_dbus_exit(void)
{
	/* Unregister D-Bus handlers */
	if (dbus_handlers != NULL) {
		g_slist_foreach(dbus_handlers,
				(GFunc)mce_dbus_handler_remove, NULL);
		g_slist_free(dbus_handlers);
		dbus_handlers = NULL;
	}

	/* If there is an established D-Bus connection, unreference it */
	if (dbus_connection != NULL) {
		mce_log(LL_DEBUG, "Unreferencing D-Bus connection");
		dbus_connection_unref(dbus_connection);
		dbus_connection = NULL;
	}
}
