/**
 * @file led.c
 * LED logic for the Mode Control Entity
 * <p>
 * Copyright © 2006-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>
#include <glib/gstdio.h>		/* g_access */

#include <stdlib.h>			/* strtoul() */
#include <string.h>			/* strcmp(), strcpy(), strdup() */
#include <unistd.h>			/* W_OK */

#include "mce.h"
#include "led.h"

#include "mce-io.h"			/* mce_write_string_to_file(),
					 * mce_write_number_string_to_file()
					 */
#include "mce-log.h"			/* mce_log(), LL_* */
#include "mce-conf.h"			/* mce_conf_get_string_list() */
#include "mce-dbus.h"			/* mce_dbus_handler_add(),
					 * dbus_send_message(),
					 * dbus_new_method_reply(),
					 * dbus_message_get_no_reply(),
					 * dbus_message_get_args(),
					 * dbus_error_init(),
					 * dbus_error_free(),
					 * DBUS_MESSAGE_TYPE_METHOD_CALL,
					 * DBUS_TYPE_STRING,
					 * DBUS_TYPE_INVALID,
					 * DBusMessage, DBusError,
					 * dbus_bool_t
					 */
#include "mce-gconf.h"			/* mce_gconf_notifier_add(),
					 * mce_gconf_notifier_remove(),
					 * mce_gconf_get_bool(),
					 * gconf_entry_get_value(),
					 * gconf_value_get_bool(),
					 * gconf_concat_dir_and_key(),
					 * GConfClient, GConfValue, GConfEntry
					 */
#include "datapipe.h"			/* execute_datapipe(),
					 * datapipe_get_gint(),
					 * append_output_trigger_to_datapipe(),
					 * remove_output_trigger_from_datapipe()
					 */

/** The pattern queue */
static GQueue *patternstack = NULL;
/** The D-Bus controlled LED switch */
static gboolean ledenable = FALSE;

/** Fields in the patterns */
typedef enum {
	PATTERN_PRIO_FIELD = 0,		/**< Priority of the pattern */
	PATTERN_SCREEN_ON_FIELD = 1,	/**< Show pattern when screen is on? */
	PATTERN_TIMEOUT_FIELD = 2,	/**< Pattern timeout */
	PATTERN_ON_PERIOD_FIELD = 3,	/**< On-period field of pattern */
	PATTERN_R_CHANNEL_FIELD = 3,	/**< R-channel of pattern */
	PATTERN_OFF_PERIOD_FIELD = 4,	/**< Off-period field of pattern */
	PATTERN_G_CHANNEL_FIELD = 4,	/**< G-channel of pattern */
	PATTERN_BRIGHTNESS_FIELD = 5,	/**< Pattern brightness */
	PATTERN_B_CHANNEL_FIELD = 5,	/**< B-channel of pattern */
	NUMBER_OF_PATTERN_FIELDS
} pattern_field;

#define CHANNEL_SIZE		32

/** Structure holding LED patterns */
typedef struct {
	gchar *name;			/**< Pattern name */
	gint priority;			/**< Pattern priority */
	gint screen_on;			/**< Show pattern when screen is on? */
	gint timeout;			/**< Timeout in seconds */
	gint on_period;			/**< Pattern on-period in ms  */
	gint off_period;		/**< Pattern off-period in ms  */
	gint brightness;		/**< Pattern brightness */
	gboolean active;		/**< Is the pattern active? */
	gboolean enabled;		/**< Is the pattern enabled? */
	/** Pattern for the R-channel */
	gchar rchannel[CHANNEL_SIZE + 1];
	/** Pattern for the G-channel */
	gchar gchannel[CHANNEL_SIZE + 1];
	/** Pattern for the B-channel */
	gchar bchannel[CHANNEL_SIZE + 1];
	guint gconf_cb_id;		/**< Callback ID for GConf entry */
} pattern_struct;

/** Pointer to the top pattern */
static pattern_struct *activepattern = NULL;
/** Timeout for the active pattern */
static glong patterntimeout = -1;
/** The active brightness */
static gint activebrightness = -1;

/** Brightness levels for the mono-LED */
static const gchar *const brightness_map[] = {
	BRIGHTNESS_LEVEL_0,
	BRIGHTNESS_LEVEL_1,
	BRIGHTNESS_LEVEL_2,
	BRIGHTNESS_LEVEL_3,
	BRIGHTNESS_LEVEL_4,
	BRIGHTNESS_LEVEL_5,
	BRIGHTNESS_LEVEL_6,
	BRIGHTNESS_LEVEL_7,
	BRIGHTNESS_LEVEL_8,
	BRIGHTNESS_LEVEL_9,
	BRIGHTNESS_LEVEL_10,
	BRIGHTNESS_LEVEL_11,
	BRIGHTNESS_LEVEL_12,
	BRIGHTNESS_LEVEL_13,
	BRIGHTNESS_LEVEL_14,
	BRIGHTNESS_LEVEL_15
};

typedef enum {
	LED_TYPE_UNSET = -1,
	LED_TYPE_NONE = 0,
	LED_TYPE_NJOY = 1,
	LED_TYPE_MONO = 2
} led_type_t;

/**
 * The ID of the LED timer
 */
static guint led_pattern_timer_cb_id = 0;

/**
 * Get the LED type
 *
 * @return The LED-type
 */
static led_type_t get_led_type(void)
{
	static led_type_t led_type = LED_TYPE_UNSET;

	if (led_type != LED_TYPE_UNSET)
		goto EXIT;

	if (g_access(MCE_LED_MODE_PATH, W_OK) == 0) {
		led_type = LED_TYPE_NJOY;
	} else if (g_access(MCE_LED_TRIGGER_PATH, W_OK) == 0) {
		led_type = LED_TYPE_MONO;
	} else {
		led_type = LED_TYPE_NONE;
	}

	mce_log(LL_DEBUG, "LED-type: %d", led_type);

EXIT:
	return led_type;
}

/**
 * Custom find function to get a particular entry in the patternstack
 *
 * @param data The pattern_struct entry
 * @param userdata The pattern name
 */
static gint queue_find(gconstpointer data, gconstpointer userdata)
{
	pattern_struct *psp;

	if (data == NULL || userdata == NULL)
		return -1;

	psp = (pattern_struct *)data;

	if (psp->name == NULL)
		return -1;

	return strcmp(psp->name, (gchar *)userdata);
}

/**
 * Custom compare function used for priority insertions
 *
 * @param entry1 Queue entry 1
 * @param entry2 Queue entry 2
 * @param userdata The pattern name
 */
static gint queue_prio_compare(gconstpointer entry1,
			       gconstpointer entry2,
			       gpointer userdata)
{
	pattern_struct *psp1 = (pattern_struct *)entry1;
	pattern_struct *psp2 = (pattern_struct *)entry2;

	(void)userdata;

	return psp1->priority - psp2->priority;
}

/**
 * Set njoy-LED brightness
 *
 * @param brightness The brightness of the LED (0-3)
 */
static void mce_njoy_set_brightness(gint brightness)
{
	if (brightness < 0 || brightness > 3) {
		mce_log(LL_WARN, "Invalid brightness value %d", brightness);
		return;
	}

	if (activebrightness == brightness)
		return;

	activebrightness = brightness;
	(void)mce_write_number_string_to_file(MCE_NJOY_LED_BRIGHTNESS_PATH,
					      brightness);

	mce_log(LL_DEBUG, "Brightness set to %d", brightness);
}

/**
 * Set mono-LED brightness
 *
 * @param brightness The brightness of the LED (0-15)
 */
static void mce_mono_set_brightness(gint brightness)
{
	if (brightness < 0 || brightness > 15) {
		mce_log(LL_WARN, "Invalid brightness value %d", brightness);
		return;
	}

	if (activebrightness == brightness)
		return;

	activebrightness = brightness;
	(void)mce_write_string_to_file(MCE_MONO_LED_BRIGHTNESS_PATH,
				       brightness_map[brightness]);

	mce_log(LL_DEBUG, "Brightness set to %d", brightness);
}

/**
 * Disable the NJoy-LED
 */
static void mce_njoy_disable_led(void)
{
	(void)mce_write_string_to_file(MCE_LED_MODE_PATH, MCE_LED_LOAD_MODE);
	(void)mce_write_string_to_file(MCE_LED_CHANNELS_PATH, "rgb");
	(void)mce_write_string_to_file(MCE_LED_LOAD_PATH, "0000");
	(void)mce_write_string_to_file(MCE_LED_CHANNELS_PATH, "");
	(void)mce_write_string_to_file(MCE_LED_MODE_PATH, MCE_LED_RUN_MODE);
}

/**
 * Disable the mono-LED
 */
static void mce_mono_disable_led(void)
{
	(void)mce_write_string_to_file(MCE_LED_TRIGGER_PATH,
				       MCE_LED_TRIGGER_NONE);
	mce_mono_set_brightness(0);
}

/**
 * Disable the LED
 */
static void mce_disable_led(void)
{
	/* Disable old timeouts */
	if (led_pattern_timer_cb_id != 0) {
		g_source_remove(led_pattern_timer_cb_id);
		led_pattern_timer_cb_id = 0;
	}

	switch (get_led_type()) {
	case LED_TYPE_NJOY:
		mce_njoy_disable_led();
		break;

	case LED_TYPE_MONO:
		mce_mono_disable_led();
		break;

	default:
		break;
	}
}

/**
 * Setup and activate a new NJoy-LED pattern
 *
 * @param pattern A pointer to a pattern_struct with the new pattern
 */
static void mce_njoy_program_led(const pattern_struct *const pattern)
{
	/* Load a new pattern, one channel at a time */
	(void)mce_write_string_to_file(MCE_LED_MODE_PATH, MCE_LED_LOAD_MODE);
	(void)mce_write_string_to_file(MCE_LED_CHANNELS_PATH, "r");
	(void)mce_write_string_to_file(MCE_LED_LOAD_PATH, pattern->rchannel);
	(void)mce_write_string_to_file(MCE_LED_CHANNELS_PATH, "g");
	(void)mce_write_string_to_file(MCE_LED_LOAD_PATH, pattern->gchannel);
	(void)mce_write_string_to_file(MCE_LED_CHANNELS_PATH, "b");
	(void)mce_write_string_to_file(MCE_LED_LOAD_PATH, pattern->bchannel);

	/* Run the new pattern */
	(void)mce_write_string_to_file(MCE_LED_CHANNELS_PATH, "rgb");
	(void)mce_write_string_to_file(MCE_LED_MODE_PATH, MCE_LED_RUN_MODE);
}

/**
 * Setup and activate a new mono-LED pattern
 *
 * @param pattern A pointer to a pattern_struct with the new pattern
 */
static void mce_mono_program_led(const pattern_struct *const pattern)
{
	/* This shouldn't happen; disable the LED instead */
	if (pattern->on_period == 0) {
		mce_mono_disable_led();
		goto EXIT;
	}

	/* If we have a normal, on/off pattern,
	 * use a timer trigger, otherwise disable the trigger
	 */
	if (pattern->off_period != 0) {
		(void)mce_write_string_to_file(MCE_LED_TRIGGER_PATH,
					       MCE_LED_TRIGGER_TIMER);
		(void)mce_write_number_string_to_file(MCE_LED_OFF_PERIOD_PATH,
						      pattern->off_period);
		(void)mce_write_number_string_to_file(MCE_LED_ON_PERIOD_PATH,
						      pattern->on_period);
	} else {
		(void)mce_write_string_to_file(MCE_LED_TRIGGER_PATH,
					       MCE_LED_TRIGGER_NONE);
	}

	mce_mono_set_brightness(pattern->brightness);

EXIT:
	return;
}

/**
 * Setup and activate a new LED pattern
 *
 * @param pattern A pointer to a pattern_struct with the new pattern
 */
static void mce_program_led(const pattern_struct *const pattern)
{
	switch (get_led_type()) {
	case LED_TYPE_NJOY:
		mce_njoy_program_led(pattern);
		break;

	case LED_TYPE_MONO:
		mce_mono_program_led(pattern);
		break;

	default:
		break;
	}
}

/**
 * Timeout callback for LED patterns
 *
 * @param data Unused
 * @return Always returns FALSE to disable pattern
 */
static gboolean led_pattern_timeout_cb(gpointer data)
{
	(void)data;

	mce_disable_led();
	led_pattern_timer_cb_id = 0;

	return FALSE;
}

/**
 * Recalculate active pattern and update the pattern timer
 */
static void mce_led_update_active_pattern(void)
{
	display_state_t display_state = datapipe_get_gint(display_state_pipe);
	system_state_t system_state = datapipe_get_gint(system_state_pipe);
	pattern_struct *newactive;
	gint i = 0;

	if (g_queue_is_empty(patternstack) == TRUE) {
		mce_disable_led();
		goto EXIT;
	}

	while ((newactive = g_queue_peek_nth(patternstack, i++)) != NULL) {
		mce_log(LL_DEBUG,
			"pattern: %s, active: %d, enabled: %d",
			newactive->name, newactive->active,
			newactive->enabled);

		/* If the pattern is deactivated, ignore */
		if (newactive->active == FALSE)
			continue;

		/* If the pattern is disabled through GConf, ignore */
		if (newactive->enabled == FALSE)
			continue;

		/* Always show pattern with visibility 3 */
		if (newactive->screen_on == 3)
			break;

		/* Acting dead behaviour */
		if (system_state == MCE_STATE_ACTDEAD) {
			/* If we're in acting dead,
			 * show patterns with visibility 4
			 */
			if (newactive->screen_on == 4)
				break;

			if ((display_state == MCE_DISPLAY_OFF) &&
			    (newactive->screen_on == 2))
				break;

			/* If the display is on and visibility is 2,
			 * or if visibility is 1/0, ignore pattern
			 */
			continue;
		}

		/* If the display is off, we can use any active pattern */
		if (display_state == MCE_DISPLAY_OFF)
			break;

		/* If the pattern should be shown with screen on, use it */
		if (newactive->screen_on == 1)
			break;
	}

	if (ledenable == FALSE || newactive == NULL) {
		activepattern = NULL;
		mce_disable_led();
		goto EXIT;
	}

	/* Only reprogram the pattern and timer if the pattern changed */
	if (newactive != activepattern) {
		mce_disable_led();

		if (newactive->timeout == -1) {
			patterntimeout = -1;
		} else {
			patterntimeout = newactive->timeout * 1000;
			led_pattern_timer_cb_id = g_timeout_add(patterntimeout, led_pattern_timeout_cb, NULL);
		}

		mce_program_led(newactive);
	}

	activepattern = newactive;

EXIT:
	return;
}

/**
 * Activate a pattern in the LED-stack
 *
 * @param name The name of the pattern to activate
 */
static void mce_led_activate_pattern(const gchar *const name)
{
	pattern_struct *psp;
	GList *glp;

	if ((glp = g_queue_find_custom(patternstack,
				       name, queue_find)) != NULL) {
		psp = (pattern_struct *)glp->data;
		psp->active = TRUE;
		mce_led_update_active_pattern();
		mce_log(LL_DEBUG,
			"LED pattern %s activated",
			name);
	} else {
		mce_log(LL_DEBUG,
			"Received request to activate "
			"a non-existing LED pattern");
	}
}

/**
 * Deactivate a pattern in the LED-stack
 *
 * @param name The name of the pattern to deactivate
 */
static void mce_led_deactivate_pattern(const gchar *const name)
{
	pattern_struct *psp;
	GList *glp;

	if ((glp = g_queue_find_custom(patternstack,
				       name, queue_find)) != NULL) {
		psp = (pattern_struct *)glp->data;
		psp->active = FALSE;
		mce_led_update_active_pattern();
		mce_log(LL_DEBUG,
			"LED pattern %s deactivated",
			name);
	} else {
		mce_log(LL_DEBUG,
			"Received request to deactivate "
			"a non-existing LED pattern");
	}
}

/**
 * Enable the LED
 */
static void mce_led_enable(void)
{
	mce_led_update_active_pattern();
	ledenable = TRUE;
}

/**
 * Disable the LED
 */
static void mce_led_disable(void)
{
	ledenable = FALSE;
	mce_disable_led();
}

/**
 * Handle system state change
 *
 * @param data The system state stored in a pointer
 */
static void system_state_trigger(gconstpointer data)
{
	(void)data;

	mce_led_update_active_pattern();
}

/**
 * Handle display state change
 *
 * @param data The display state stored in a pointer
 */
static void display_state_trigger(gconstpointer data)
{
	(void)data;

	mce_led_update_active_pattern();
}

/**
 * Handle led brightness change
 *
 * @param data The LED brightness stored in a pointer
 */
static void led_brightness_trigger(gconstpointer data)
{
	gint led_brightness = GPOINTER_TO_INT(data);

	switch (get_led_type()) {
	case LED_TYPE_NJOY:
		mce_njoy_set_brightness(led_brightness);
		break;

	case LED_TYPE_MONO:
	case LED_TYPE_UNSET:
	case LED_TYPE_NONE:
	default:
		break;
	}
}

/**
 * Handle LED pattern activate requests
 *
 * @param data The pattern name
 */
static void led_pattern_activate_trigger(gconstpointer data)
{
	mce_led_activate_pattern((gchar *)data);
}

/**
 * Handle LED pattern deactivate requests
 *
 * @param data The pattern name
 */
static void led_pattern_deactivate_trigger(gconstpointer data)
{
	mce_led_deactivate_pattern((gchar *)data);
}

/**
 * Custom find function to get a GConf callback ID in the patternstack
 *
 * @param data The pattern_struct entry
 * @param userdata The pattern name
 */
static gint gconf_cb_find(gconstpointer data, gconstpointer userdata)
{
	pattern_struct *psp;

	if (data == NULL || userdata == NULL)
		return -1;

	psp = (pattern_struct *)data;

	return psp->gconf_cb_id != *(guint *)userdata;
}

/**
 * GConf callback for LED related settings
 *
 * @param gce Unused
 * @param id Connection ID from gconf_client_notify_add()
 * @param entry The modified GConf entry
 * @param data Unused
 */
static void led_gconf_cb(GConfClient *const gce, const guint id,
			 GConfEntry *const entry, gpointer const data)
{
	GConfValue *gcv = gconf_entry_get_value(entry);
	pattern_struct *psp;
	GList *glp = NULL;

	(void)gce;
	(void)data;

	if ((glp = g_queue_find_custom(patternstack,
				       &id, gconf_cb_find)) != NULL) {
		psp = (pattern_struct *)glp->data;
		psp->enabled = gconf_value_get_bool(gcv);
		mce_led_update_active_pattern();
	} else {
		mce_log(LL_WARN, "Spurious GConf value received; confused!");
	}
}

/**
 * Get the enabled/disabled value from GConf and set up a notifier
 */
static gboolean pattern_get_enabled(const gchar *const patternname,
				    guint *gconf_cb_id)
{
	gboolean retval = DEFAULT_PATTERN_ENABLED;
	gchar *path = gconf_concat_dir_and_key(MCE_GCONF_LED_PATH,
					       patternname);

	/* Since we've set a default, error handling is unnecessary */
	(void)mce_gconf_get_bool(path, &retval);

	if (mce_gconf_notifier_add(MCE_GCONF_LED_PATH, path,
				   led_gconf_cb, gconf_cb_id) == FALSE)
		goto EXIT;

EXIT:
	g_free(path);

	return retval;
}

/**
 * D-Bus callback for the activate LED pattern method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_led_activate_pattern_dbus_cb(DBusMessage *const msg)
{
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	gboolean status = FALSE;
	gchar *pattern = NULL;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	mce_log(LL_DEBUG, "Received activate LED pattern request");

	if (dbus_message_get_args(msg, &error,
				  DBUS_TYPE_STRING, &pattern,
				  DBUS_TYPE_INVALID) == FALSE) {
		// XXX: should we return an error instead?
		mce_log(LL_CRIT,
			"Failed to get argument from %s: %s",
			MCE_ACTIVATE_LED_PATTERN,
			error.message);
		dbus_error_free(&error);
		goto EXIT;
	}

	mce_led_activate_pattern(pattern);

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

EXIT:
	return status;
}

/**
 * D-Bus callback for the deactivate LED pattern method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_led_deactivate_pattern_dbus_cb(DBusMessage *const msg)
{
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	gboolean status = FALSE;
	gchar *pattern = NULL;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	mce_log(LL_DEBUG, "Received deactivate LED pattern request");

	if (dbus_message_get_args(msg, &error,
				  DBUS_TYPE_STRING, &pattern,
				  DBUS_TYPE_INVALID) == FALSE) {
		// XXX: should we return an error instead?
		mce_log(LL_CRIT,
			"Failed to get argument from %s: %s",
			MCE_DEACTIVATE_LED_PATTERN,
			error.message);
		dbus_error_free(&error);
		goto EXIT;
	}

	mce_led_deactivate_pattern(pattern);

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

EXIT:
	return status;
}

/**
 * D-Bus callback for the enable LED method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_led_enable_dbus_cb(DBusMessage *const msg)
{
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	gboolean status = FALSE;

	mce_log(LL_DEBUG, "Received LED enable request");

	mce_led_enable();

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

//EXIT:
	return status;
}

/**
 * D-Bus callback for the disable LED method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_led_disable_dbus_cb(DBusMessage *const msg)
{
	dbus_bool_t no_reply = dbus_message_get_no_reply(msg);
	gboolean status = FALSE;

	mce_log(LL_DEBUG, "Received LED disable request");

	mce_led_disable();

	if (no_reply == FALSE) {
		DBusMessage *reply = dbus_new_method_reply(msg);

		status = dbus_send_message(reply);
	} else {
		status = TRUE;
	}

//EXIT:
	return status;
}

/**
 * Init patterns for the NJoy LED
 *
 * @param TRUE on success, FALSE on failure
 */
static gboolean init_njoy_patterns(void)
{
	gchar **patternlist = NULL;
	gboolean status = FALSE;
	gsize length;
	gint i;

	/* Get the list of valid LED patterns */
	patternlist = mce_conf_get_string_list(MCE_CONF_LED_GROUP,
					       MCE_CONF_LED_PATTERNS,
					       &length,
					       NULL);

	if (patternlist == NULL)
		goto EXIT;

	for (i = 0; patternlist[i]; i++) {
		gchar **tmp;

		mce_log(LL_DEBUG,
			"Getting LED pattern for: %s",
			patternlist[i]);

		tmp = mce_conf_get_string_list(MCE_CONF_LED_PATTERN_NJOY_GROUP,
					       patternlist[i],
					       &length,
					       NULL);

		if (tmp != NULL) {
			pattern_struct *psp;

			if ((length != NUMBER_OF_PATTERN_FIELDS) ||
			    (strlen(tmp[PATTERN_R_CHANNEL_FIELD]) >
			     CHANNEL_SIZE) ||
			    (strlen(tmp[PATTERN_G_CHANNEL_FIELD]) >
			     CHANNEL_SIZE) ||
			    (strlen(tmp[PATTERN_B_CHANNEL_FIELD]) >
			     CHANNEL_SIZE)) {
				mce_log(LL_ERR,
					"Skipping invalid LED-pattern");
				g_free(tmp);
				continue;
			}

			psp = g_slice_new(pattern_struct);

			if (!psp) {
				g_strfreev(tmp);
				goto EXIT2;
			}

			psp->name = strdup(patternlist[i]);
			/* XXX:
			 * These strtoul's should probably have
			 * some sort of error checking
			 */
			psp->priority = strtoul(tmp[PATTERN_PRIO_FIELD],
						NULL, 10);
			psp->screen_on = strtoul(tmp[PATTERN_SCREEN_ON_FIELD],
						 NULL, 10);

			if ((psp->timeout = strtoul(tmp[PATTERN_TIMEOUT_FIELD],
						    NULL, 10)) == 0)
				psp->timeout = -1;

			strncpy(psp->rchannel,
			       tmp[PATTERN_R_CHANNEL_FIELD],
			       CHANNEL_SIZE + 1);
			strncpy(psp->gchannel,
			       tmp[PATTERN_G_CHANNEL_FIELD],
			       CHANNEL_SIZE + 1);
			strncpy(psp->bchannel,
			       tmp[PATTERN_B_CHANNEL_FIELD],
			       CHANNEL_SIZE + 1);

			psp->active = FALSE;

			psp->enabled = pattern_get_enabled(patternlist[i],
							   &(psp->gconf_cb_id));

			g_strfreev(tmp);

			g_queue_insert_sorted(patternstack, psp,
					      queue_prio_compare,
					      NULL);
		}
	}

	/* Set the LED brightness */
	execute_datapipe(&led_brightness_pipe,
			 GINT_TO_POINTER(DEFAULT_NJOY_LED_BRIGHTNESS),
			 FALSE, TRUE);

	status = TRUE;

EXIT2:
	g_strfreev(patternlist);

EXIT:
	return status;
}

/**
 * Init patterns for the Mono LED
 *
 * @param TRUE on success, FALSE on failure
 */
static gboolean init_mono_patterns(void)
{
	gchar **patternlist = NULL;
	gboolean status = FALSE;
	gsize length;
	gint i;

	/* Get the list of valid LED patterns */
	patternlist = mce_conf_get_string_list(MCE_CONF_LED_GROUP,
					       MCE_CONF_LED_PATTERNS,
					       &length,
					       NULL);

	if (patternlist == NULL)
		goto EXIT;

	/* Used for single-colour LED patterns */
	for (i = 0; patternlist[i]; i++) {
		gint *tmp;

		mce_log(LL_DEBUG,
			"Getting LED pattern for: %s",
			patternlist[i]);

		tmp = mce_conf_get_int_list(MCE_CONF_LED_PATTERN_MONO_GROUP,
					    patternlist[i],
					    &length,
					    NULL);

		if (tmp != NULL) {
			pattern_struct *psp;

			if (length != NUMBER_OF_PATTERN_FIELDS) {
				mce_log(LL_ERR,
					"Skipping invalid LED-pattern");
				g_free(tmp);
				continue;
			}

			psp = g_slice_new(pattern_struct);

			if (!psp) {
				g_free(tmp);
				goto EXIT2;
			}

			psp->name = strdup(patternlist[i]);
			psp->priority = tmp[PATTERN_PRIO_FIELD];
			psp->screen_on = tmp[PATTERN_SCREEN_ON_FIELD];
			psp->timeout = tmp[PATTERN_TIMEOUT_FIELD] ? tmp[PATTERN_TIMEOUT_FIELD] : -1;
			psp->on_period = tmp[PATTERN_ON_PERIOD_FIELD];
			psp->off_period = tmp[PATTERN_OFF_PERIOD_FIELD];
			psp->brightness = tmp[PATTERN_BRIGHTNESS_FIELD];
			psp->active = FALSE;

			psp->enabled = pattern_get_enabled(patternlist[i],
							   &(psp->gconf_cb_id));

			g_free(tmp);

			g_queue_insert_sorted(patternstack, psp,
					      queue_prio_compare,
					      NULL);
		}
	}

	status = TRUE;

EXIT2:
	g_strfreev(patternlist);

EXIT:
	return status;
}

/**
 * Init patterns for the LED
 *
 * @param TRUE on success, FALSE on failure
 */
static gboolean init_patterns(void)
{
	gboolean status;

	switch (get_led_type()) {
	case LED_TYPE_NJOY:
		status = init_njoy_patterns();
		break;

	case LED_TYPE_MONO:
		status = init_mono_patterns();
		break;

	default:
		status = TRUE;
		break;
	}

	return status;
}

/**
 * Init function for the LED component
 *
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_led_init(void)
{
	gboolean status = FALSE;

	/* Setup a pattern stack and initialise the patterns */
	patternstack = g_queue_new();

	if (init_patterns() == FALSE)
		goto EXIT;

	/* Append triggers/filters to datapipes */
	append_output_trigger_to_datapipe(&system_state_pipe,
					  system_state_trigger);
	append_output_trigger_to_datapipe(&display_state_pipe,
					  display_state_trigger);
	append_output_trigger_to_datapipe(&led_brightness_pipe,
					  led_brightness_trigger);
	append_output_trigger_to_datapipe(&led_pattern_activate_pipe,
					  led_pattern_activate_trigger);
	append_output_trigger_to_datapipe(&led_pattern_deactivate_pipe,
					  led_pattern_deactivate_trigger);

	/* req_led_pattern_activate */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_ACTIVATE_LED_PATTERN,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 mce_led_activate_pattern_dbus_cb) == FALSE)
		goto EXIT;

	/* req_led_pattern_deactivate */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_DEACTIVATE_LED_PATTERN,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 mce_led_deactivate_pattern_dbus_cb) == FALSE)
		goto EXIT;

	/* req_led_enable */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_ENABLE_LED,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 mce_led_enable_dbus_cb) == FALSE)
		goto EXIT;

	/* req_led_disable */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_DISABLE_LED,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 mce_led_disable_dbus_cb) == FALSE)
		goto EXIT;

	status = TRUE;

	mce_led_enable();

EXIT:
	return status;
}

/**
 * Exit function for the LED component
 *
 * @todo D-Bus unregistration
 */
void mce_led_exit(void)
{
	system_state_t system_state = datapipe_get_gint(system_state_pipe);

	/* Remove triggers/filters from datapipes */
	remove_output_trigger_from_datapipe(&led_pattern_deactivate_pipe,
					    led_pattern_deactivate_trigger);
	remove_output_trigger_from_datapipe(&led_pattern_activate_pipe,
					    led_pattern_activate_trigger);
	remove_output_trigger_from_datapipe(&led_brightness_pipe,
					    led_brightness_trigger);
	remove_output_trigger_from_datapipe(&display_state_pipe,
					    display_state_trigger);
	remove_output_trigger_from_datapipe(&system_state_pipe,
					    system_state_trigger);

	/* Don't disable the LED on shutdown/reboot/acting dead */
	if ((system_state != MCE_STATE_ACTDEAD) &&
	    (system_state != MCE_STATE_SHUTDOWN) &&
	    (system_state != MCE_STATE_REBOOT)) {
		mce_led_disable();
	}

	if (patternstack != NULL) {
		pattern_struct *psp;

		while ((psp = g_queue_pop_head(patternstack)) != NULL) {
			mce_gconf_notifier_remove(GINT_TO_POINTER(psp->gconf_cb_id), NULL);
			g_free(psp->name);
			g_slice_free(pattern_struct, psp);
		}

		g_queue_free(patternstack);
	}
}
