/**
 * @file keypad.c
 * Key pad logic for the Mode Control Entity
 * <p>
 * Copyright © 2004-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>

#include <stdlib.h>			/* exit(), EXIT_FAILURE */
#include <linux/input.h>		/* struct input_event, KEY_F5 */

#include "mce.h"
#include "keypad.h"

#include "mce-io.h"			/* mce_write_number_string_to_file() */
#include "mce-log.h"			/* mce_log(), LL_* */
#include "datapipe.h"			/* execute_datapipe(),
					 * datapipe_get_gbool(),
					 * datapipe_get_guint(),
					 * datapipe_get_old_gint(),
					 * append_output_trigger_to_datapipe(),
					 * remove_output_trigger_from_datapipe()
					 */

gint has_keyboard = FALSE;

/**
 * The ID of the timeout used for the key backlight
 */
static guint key_backlight_timeout_cb_id = 0;

/** Default backlight brightness */
static gint key_backlight_timeout = DEFAULT_KEY_BACKLIGHT_TIMEOUT;

/**
 * Set keypad/keyboard backlight brightness
 *
 * @param brightness Backlight brightness
 */
static void set_backlight_brightness(gconstpointer data)
{
	static gint cached_brightness = -1;
	gint new_brightness = GPOINTER_TO_INT(data);

	/* XXX: if we're running on the N800, just exit */
	if (has_keyboard == FALSE)
		goto EXIT;

	/* If we're just rehashing the same brightness value, don't bother */
	if ((new_brightness == cached_brightness) && (cached_brightness != -1))
		goto EXIT;

	cached_brightness = new_brightness;

	/* Set fadetime */
	if (new_brightness == 0) {
		(void)mce_write_number_string_to_file(MCE_KEYPAD_BACKLIGHT_FADETIME_SYS_PATH, DEFAULT_KEY_BACKLIGHT_FADETIME);
		(void)mce_write_number_string_to_file(MCE_KEYBOARD_BACKLIGHT_FADETIME_SYS_PATH, DEFAULT_KEY_BACKLIGHT_FADETIME);
	} else {
		(void)mce_write_number_string_to_file(MCE_KEYPAD_BACKLIGHT_FADETIME_SYS_PATH, 0);
		(void)mce_write_number_string_to_file(MCE_KEYBOARD_BACKLIGHT_FADETIME_SYS_PATH, 0);
	}

	(void)mce_write_number_string_to_file(MCE_KEYPAD_BACKLIGHT_BRIGHTNESS_SYS_PATH, new_brightness);
	(void)mce_write_number_string_to_file(MCE_KEYBOARD_BACKLIGHT_BRIGHTNESS_SYS_PATH, new_brightness);

EXIT:
	return;
}

/**
 * Disable key backlight
 */
static void disable_key_backlight(void)
{
	/* Remove the timeout source for the keyboard/keypad
	 * backlight timeout
	 */
	if (key_backlight_timeout_cb_id != 0) {
		g_source_remove(key_backlight_timeout_cb_id);
		key_backlight_timeout_cb_id = 0;
	}

	execute_datapipe(&key_backlight_pipe, GINT_TO_POINTER(0), FALSE, TRUE);
}

/**
 * Key backlight timeout function
 *
 * @param data Unused
 * @return Always returns FALSE, to disable the timeout
 */
static gboolean key_backlight_timeout_cb(gpointer data)
{
	(void)data;

	disable_key_backlight();

	return FALSE;
}

/**
 * Enable key backlight
 */
static void enable_key_backlight(void)
{
	/* Remove the timeout source for the keyboard/keypad
	 * backlight timeout
	 */
	if (key_backlight_timeout_cb_id != 0) {
		g_source_remove(key_backlight_timeout_cb_id);
		key_backlight_timeout_cb_id = 0;
	}

	/* Only enable the key backlight if the slide is open */
	if (datapipe_get_gbool(keyboard_slide_pipe) == FALSE)
		goto EXIT;

	/* Add a new timeout */
	key_backlight_timeout_cb_id =
		g_timeout_add(key_backlight_timeout * 1000,
			      key_backlight_timeout_cb, NULL);

	/* If the backlight is off, turn it on */
	if (datapipe_get_guint(key_backlight_pipe) == 0) {
		execute_datapipe(&key_backlight_pipe,
				 GINT_TO_POINTER(DEFAULT_KEY_BACKLIGHT_LEVEL),
				 FALSE, TRUE);
	}

EXIT:
	return;
}

/**
 * Policy based enabling of key backlight
 */
static void enable_key_backlight_policy(void)
{
	system_state_t system_state = datapipe_get_gint(system_state_pipe);

	/* Only enable the backlight in USER state
	 * and when the alarm dialog is visible
	 */
	if ((system_state == MCE_STATE_USER) ||
	    (mce_get_submode_int32() & MCE_ALARM_SUBMODE))
		enable_key_backlight();
}

/**
 * Datapipe trigger for keypresses
 *
 * @param data Unused
 */
static void keypress_trigger(gconstpointer data)
{
	/* We're not interested in the keypress,
	 * just that it has taken place
	 */
	(void)data;

	enable_key_backlight_policy();
}

/**
 * Datapipe trigger for the keyboard slide
 *
 * @param data 1 if the keyboard is open, 0 if the keyboard is closed
 */
static void kbd_slide_trigger(gconstpointer const data)
{
	if (GPOINTER_TO_INT(data) == 1) {
		enable_key_backlight_policy();
	} else {
		disable_key_backlight();
	}
}

/**
 * Handle display state change
 *
 * @param data The display stated stored in a pointer
 */
static void display_state_cb(gconstpointer data)
{
	display_state_t display_state = GPOINTER_TO_INT(data);
	display_state_t old_display_state =
				datapipe_get_old_gint(display_state_pipe);

	/* Disable the keyboard backlight if the display dims */
	switch (display_state) {
	case MCE_DISPLAY_OFF:
	case MCE_DISPLAY_DIM:
		disable_key_backlight();
		break;

	case MCE_DISPLAY_ON:
		if (old_display_state == MCE_DISPLAY_OFF)
			enable_key_backlight_policy();

		break;

	case MCE_DISPLAY_UNDEF:
	default:
		break;
	}
}

/**
 * Handle system state change
 *
 * @param data The system state stored in a pointer
 */
static void system_state_trigger(gconstpointer data)
{
	system_state_t system_state = GPOINTER_TO_INT(data);

	/* If we're changing to another state than USER,
	 * disable the key backlight
	 */
	if (system_state != MCE_STATE_USER)
		disable_key_backlight();
}

/**
 * Init function for the keypad component
 *
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_keypad_init(void)
{
	gboolean status = FALSE;

	/* Append triggers/filters to datapipes */
	append_output_trigger_to_datapipe(&system_state_pipe,
					  system_state_trigger);
	append_output_trigger_to_datapipe(&key_backlight_pipe,
					  set_backlight_brightness);
	append_output_trigger_to_datapipe(&keypress_pipe,
					  keypress_trigger);
	append_output_trigger_to_datapipe(&keyboard_slide_pipe,
					  kbd_slide_trigger);
	append_output_trigger_to_datapipe(&display_state_pipe,
					  display_state_cb);

	status = TRUE;

	return status;
}

/**
 * Exit function for the keypad component
 */
void mce_keypad_exit(void)
{
	/* Remove triggers/filters from datapipes */
	remove_output_trigger_from_datapipe(&display_state_pipe,
					    display_state_cb);
	remove_output_trigger_from_datapipe(&keyboard_slide_pipe,
					    kbd_slide_trigger);
	remove_output_trigger_from_datapipe(&keypress_pipe,
					    keypress_trigger);
	remove_output_trigger_from_datapipe(&key_backlight_pipe,
					    set_backlight_brightness);
	remove_output_trigger_from_datapipe(&system_state_pipe,
					    system_state_trigger);

	return;
}
