/**
 * @file input_event.c
 * Common input event functionality for the Mode Control Entity
 * <p>
 * Copyright © 2004-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <errno.h>			/* errno */
#include <fcntl.h>			/* open() */
#include <string.h>			/* strcmp() */
#include <unistd.h>			/* close() */
#include <sys/ioctl.h>			/* ioctl() */
#include <linux/input.h>		/* EVIOCGNAME */

#include "mce.h"
#include "input_event.h"

#include "mce-log.h"			/* mce_log(), LL_* */

/**
 * Return the input event filedescriptor for a given driver
 *
 * @param drivername A string containing the name of the driver
 * @return An open file descriptor on success, -1 on failure
 */
int get_event_fd(const char *const drivername)
{
	/* Semi-ugly hack....      012345678901234567 */
	static char filename[] = "/dev/input/event0\0";
	static char name[256];
	int fd = -1;
	int i;

	mce_log(LL_DEBUG, "Scanning for event device for `%s'",
		drivername);

	for (i = 0; i <= 99; i++) {
		if (i <= 9) {
			filename[16] = '0' + (i % 10);
			filename[17] = '\0';
		} else {
			filename[16] = '0' + (i / 10);
			filename[17] = '0' + (i % 10);
		}

		/* If we cannot open the file, try the next one */
		if ((fd = open(filename, O_NONBLOCK | O_RDONLY)) == -1) {
			mce_log(LL_DEBUG, "Failed to open `%s', skipping",
				filename);

			/* Ignore error */
			errno = 0;
			continue;
		}

		if (ioctl(fd, EVIOCGNAME(sizeof name), name) >= 0) {
			mce_log(LL_DEBUG, "%s is `%s'", filename, name);

			if (strcmp(name, drivername) == 0) {
				/* We found our event file */
				break;
			}
		} else {
			mce_log(LL_WARN, "ioctl() failed for `%s'",
				filename);
		}

		close(fd);
	}

	if (fd != -1) {
		mce_log(LL_INFO, "Event device for `%s' found: `%s'",
			drivername, filename);
	} else {
		mce_log(LL_WARN, "Failed to find an event device for `%s'",
			drivername);
	}

	return fd;
}
