/**
 * @file event-touchscreen.c
 * Touchscreen event provider for the Mode Control Entity
 * <p>
 * Copyright © 2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>

#include <stdlib.h>			/* exit(), EXIT_FAILURE */
#include <unistd.h>			/* close() */
#include <linux/input.h>		/* struct input_event,
					 * EV_ABS, ABS_PRESSURE
					 */

#include "mce.h"
#include "event-touchscreen.h"

#include "mce-io.h"			/* mce_register_io_monitor_chunk(),
					 * mce_unregister_io_monitor()
					 */
#include "mce-log.h"			/* mce_log(), LL_* */
#include "datapipe.h"			/* execute_datapipe() */
#include "input_event.h"		/* get_event_fd */

/** ID for the touchscreen I/O monitor */
static gconstpointer touchscreen_iomon_id = NULL;

/** File descriptor for touchscreen */
static int eventfd = -1;

/**
 * I/O monitor callback for the touchscreen
 *
 * @param data The new data
 * @param bytes_read The number of bytes read
 */
static void touchscreen_cb(gpointer data, gsize bytes_read)
{
	struct input_event *ev;

	ev = data;

	if (bytes_read != sizeof (struct input_event)) {
		goto EXIT;
	}

	/* Ignore non-touchscreen events */
	if (ev->type != EV_ABS) {
		goto EXIT;
	}

	/* Ignore non-pressure events */
	if (ev->code != ABS_PRESSURE) {
		goto EXIT;
	}

	/* For now there's no reason to cache the value,
	 * or indeed to send any kind of real value at all
	 */
	(void)execute_datapipe(&touchscreen_pipe, NULL, FALSE, FALSE);

EXIT:
	return;
}

/**
 * Init function for the touchscreen component
 *
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_touchscreen_init(void)
{
	gboolean status = FALSE;
	const gchar *event = NULL;

	/* Find out what event handler handles the touchscreen */
	if ((eventfd = get_event_fd(OMAP_TOUCHSCREEN_N810)) != -1) {
		event = OMAP_TOUCHSCREEN_N810;
	} else if ((eventfd = get_event_fd(OMAP_TOUCHSCREEN_N800)) != -1) {
		event = OMAP_TOUCHSCREEN_N800;
	} else if ((eventfd = get_event_fd(OMAP_TOUCHSCREEN_770)) != -1) {
		event = OMAP_TOUCHSCREEN_770;
	}

	if (eventfd != -1) {
		mce_log(LL_DEBUG,
			"Got event file descriptor for `%s'",
			event);
	} else {
		mce_log(LL_ERR,
			"Failed to get a touchscreen event file descriptor");
	}

	/* If we fail to get a file descriptor,
	 * don't attempt to register an I/O-channel for it
	 */
	if (eventfd != -1) {
		/* Register I/O monitors */
		// FIXME: error handling?
		touchscreen_iomon_id =
			mce_register_io_monitor_chunk(eventfd, event, FALSE, FALSE, touchscreen_cb, sizeof (struct input_event));
	}

	status = TRUE;

	return status;
}

/**
 * Exit function for the touchscreen component
 */
void mce_touchscreen_exit(void)
{
	/* Unregister I/O monitors */
	mce_unregister_io_monitor(touchscreen_iomon_id);

	/* Close file descriptors */
	if (eventfd != -1) {
		close(eventfd);
		eventfd = -1;
	}

	return;
}
